#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#ifdef HAVE_IMLIB
# include <Imlib.h>
#endif
#include <gtk/gtk.h>

#include "../include/string.h"
#include "../include/disk.h"
#include "../include/tga.h"

#include "guiutils.h"
#include "cdialog.h"
#include "progressdialog.h"
#include "fprompt.h"

#include "edvtypes.h"
#include "edvdate.h"
#include "edvcfg.h"
#include "edvid.h"
#include "edvobj.h"
#include "edvfop.h"
#include "edvmimetypes.h"
#include "imbr.h"
#include "imbrcb.h"
#include "imbrtlist.h"
#include "endeavour.h"
#include "edvopen.h"
#include "edvcb.h"
#include "edvutils.h"
#include "edvutilsgtk.h"
#include "edvcfglist.h"
#include "config.h"


static void EDVImbrTListFPromptCancelCB(gpointer data);

static guint8 *EDVImbrTListLoadImageRGBATarga(
        edv_core_struct *core_ptr,
        const gchar *path, gint *width, gint *height, gint *bpl,
        GdkWindow *window       /* Reference window. */
);
guint8 *EDVImbrTListLoadImageRGBA(
        edv_core_struct *core_ptr,
        const gchar *path, gint *width, gint *height, gint *bpl,
        GdkWindow *window       /* Reference window. */
);

static void EDVImbrTListSetThumbName(
        edv_core_struct *core_ptr, edv_imbr_struct *imbr,
	tlist_struct *tlist, edv_object_struct *object,
        gint thumb_num
);

static gint EDVImbrTListAppendObject(
        edv_core_struct *core_ptr, edv_imbr_struct *imbr,
	tlist_struct *tlist, edv_object_struct *object
);

gint EDVImbrTListFindThumbByPath(
        edv_imbr_struct *imbr, const gchar *path
);

void EDVImbrTListResetThumbs(edv_imbr_struct *imbr);
void EDVImbrTListDoUpdate(
        edv_imbr_struct *imbr, const gchar *path,
        gbool update_status_bar
);

gint EDVImbrTListLoadIterate(
	edv_imbr_struct *imbr, gbool update_status_bar,
	gbool no_enlarge
);

void EDVImbrTListDoOpenObject(
        edv_imbr_struct *imbr, gint thumb_num, guint state
);
void EDVImbrTListDoOpenWithObject(
        edv_imbr_struct *imbr, gint thumb_num
);

static void EDVImbrTListFPromptRenameApplyCB(
        gpointer data, const gchar *value
);
void EDVImbrTListDoFPromptRename(
        edv_imbr_struct *imbr, gint thumb_num
);

void EDVImbrTListObjectAddedNotify(
        edv_imbr_struct *imbr, const gchar *path,
        const struct stat *lstat_buf
);
void EDVImbrTListObjectModifiedNotify(
        edv_imbr_struct *imbr, const gchar *path,
        const gchar *new_path,
        const struct stat *lstat_buf
);
void EDVImbrTListObjectRemovedNotify(
        edv_imbr_struct *imbr, const gchar *path
);

void EDVImbrTListMountNotify(
        edv_imbr_struct *imbr, edv_device_struct *dev_ptr,
        gbool is_mounted
);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))


/*
 *      All purpose fprompt cancel callback.
 */
static void EDVImbrTListFPromptCancelCB(gpointer data)
{
        gpointer *cb_data = (gpointer *)data;
        if(cb_data == NULL)
            return;

        /* Deallocate callback data. */
        g_free(cb_data);
        cb_data = NULL;
}


/*
 *	Called by EDVImbrTListLoadImageRGBA() to load a targa image.
 */
static guint8 *EDVImbrTListLoadImageRGBATarga(
	edv_core_struct *core_ptr,
        const gchar *path, gint *width, gint *height, gint *bpl,
        GdkWindow *window       /* Reference window. */
)
{
	return(TgaReadFromFileFastRGBA(
	    path, width, height,
	    0xff000000	/* Transparent pixel rgba. */
	));
}

/* 
 *	Returns a dynamically allocated image buffer loaded from the image
 *	file specified by path, the calling function must deallocate this
 *	buffer.
 *
 *	Can return NULL on error.
 */
guint8 *EDVImbrTListLoadImageRGBA(
	edv_core_struct *core_ptr,
        const gchar *path, gint *width, gint *height, gint *bpl,
        GdkWindow *window       /* Reference window. */
)
{
#ifdef HAVE_IMLIB
        gint swidth, sheight, sbpp, sbpl, tbpp, tbpl;
        const guint8 *src_rgb, *src_alpha;
        guint8 *tar_rgba;
        ImlibImage *imlib_image = NULL;


	/* Reset returns. */
	if(width != NULL)
	    *width = 0;
	if(height != NULL)
	    *height = 0;
	if(bpl != NULL)
	    *bpl = 0;

        if(path == NULL)
            return(NULL);

	/* Begin checking extension to see if we should use a native
	 * image format loader instead of Imlib, since native loaders
	 * may be more efficient or provide more features.
	 */

	/* Targa. */
	if(EDVIsExtension(path, ".tga") ||
	   EDVIsExtension(path, ".targa")
	)
	{
	    return(EDVImbrTListLoadImageRGBATarga(
		core_ptr, path, width, height, bpl,
		window
	    ));
	}
/* Add other image types that we should load alternatively (without
 * Imlib) here and make sure they return() the RGBA data pointer and do
 * not continue.
 */

	/* Load Imlib image. */
	if(imlib_handle != NULL)
	{
	    gchar *dpath = g_strdup(path);
	    imlib_image = Imlib_load_image(imlib_handle, dpath);
	    g_free(dpath);
	}

	/* Unable to load Imlib image? */
        if(imlib_image == NULL)
            return(NULL);

        /* Need to realize changes. */
        Imlib_changed_image(imlib_handle, imlib_image);

        src_rgb = (const guint8 *)imlib_image->rgb_data;
        src_alpha = (const guint8 *)imlib_image->alpha_data;

        swidth = imlib_image->rgb_width;
        sheight = imlib_image->rgb_height;
        sbpp = 3;
        sbpl = swidth * sbpp;

        tbpp = 4;
        tbpl = swidth * tbpp;

        if((src_rgb == NULL) || (swidth <= 0) || (sheight <= 0))
        {
            Imlib_destroy_image(imlib_handle, imlib_image);
            return(NULL);
        }

        /* Allocate return image data buffer. */
        tar_rgba = (guint8 *)g_malloc(tbpl * sheight);

	/* Copy source data (with alpha channel) to target data? */
        if((tar_rgba != NULL) && (src_alpha != NULL))
        {
            gint x, y;
            guint8 *tar_ptr;
            const guint8 *src_ptr;

            /* Iterate through each line. */
            for(y = 0; y < sheight; y++)
            {
                /* Iterate through current line. */
                for(x = 0; x < swidth; x++)
                {
                    tar_ptr = &tar_rgba[
                        (y * tbpl) + (x * tbpp)
                    ];
                    src_ptr = &src_rgb[
                        (y * sbpl) + (x * sbpp)
                    ];

                    /* Copy rgba of this pixel (3 bytes plus one alpha
		     * byte).
		     */
                    *tar_ptr++ = *src_ptr++;
                    *tar_ptr++ = *src_ptr++;
                    *tar_ptr++ = *src_ptr;
		    *tar_ptr = src_alpha[(y * swidth) + x];
                }
            }
        }
	/* Copy source data to target data? */
	else if(tar_rgba != NULL)
	{
            gint x, y;
            guint8 *tar_ptr;
            const guint8 *src_ptr;

            /* Iterate through each line. */
            for(y = 0; y < sheight; y++)
            {
                /* Iterate through current line. */
                for(x = 0; x < swidth; x++)
                {
                    tar_ptr = &tar_rgba[
                        (y * tbpl) + (x * tbpp)
                    ];
                    src_ptr = &src_rgb[
                        (y * sbpl) + (x * sbpp)
                    ];

                    /* Copy rgba of this pixel (3 bytes plus one alpha
                     * byte), the alpha byte will be calculated if
                     * the red and blue bytes are 0xff and green byte is
                     * 0x00 (because this is Imlib's way of using an
                     * alpha channel).
                     */
                    *tar_ptr++ = src_ptr[0];
                    *tar_ptr++ = src_ptr[1];
                    *tar_ptr++ = src_ptr[2];

		    if((src_ptr[0] == 0xff) && (src_ptr[1] == 0x00) &&
		       (src_ptr[2] == 0xff)
		    )
			*tar_ptr = 0x00;
		    else
			*tar_ptr = 0xff;
                }
            }
	}


        /* Destroy Imlib image, it is no longer needed. */
        Imlib_destroy_image(imlib_handle, imlib_image);
        imlib_image = NULL;
        src_rgb = NULL;
        src_alpha = NULL;

        if(width != NULL)
            *width = swidth;
        if(height != NULL)
            *height = sheight;
        if(bpl != NULL)
            *bpl = tbpl;

        return(tar_rgba);
#else
	return(NULL);
#endif	/* HAVE_IMLIB */
}



/*
 *      Sets the text for the given thumb.
 */
static void EDVImbrTListSetThumbName(
        edv_core_struct *core_ptr, edv_imbr_struct *imbr,
        tlist_struct *tlist, edv_object_struct *object,
        gint thumb_num
)
{
	const gchar *name;


        if((core_ptr == NULL) || (imbr == NULL) || (tlist == NULL) ||
           (object == NULL)
        )
            return;

	/* Get object's name. */
	name = object->name;
	if(name == NULL)
	    name = "(null)";

	/* Check if this not an image file supported by Imlib. */
	if(!EDVCheckImlibImage(core_ptr, object->full_path))
	{
	    /* Probably not an image, so set pixmap based on MIME Type
	     * icon.
	     */
	    GdkPixmap *pixmap;
	    GdkBitmap *mask;

	    EDVMatchObjectIcon(
                core_ptr->device, core_ptr->total_devices,
                core_ptr->mimetype, core_ptr->total_mimetypes,
                object->type,
                object->full_path,
                object->link_valid, object->permissions,
                1,                  /* Medium icons. */
                &pixmap, &mask,
                NULL, NULL, NULL, NULL
	    );
	    TListSetPixmap(tlist, thumb_num, pixmap, mask);

	    /* Mark it as loaded so no pixmap gets loaded over it during
	     * any incidental calls to the loading process.
	     */
	    TListSetLoadState(tlist, thumb_num, TLIST_LOAD_STATE_LOADED);
	}
	else
	{
	    /* Is an image, so do not set any pixmap on it and leave its
	     * load state as is.
	     */
	}

	TListSetText(tlist, thumb_num, name);
}

/*
 *	Appends the disk object to the thumbs list. The given object will
 *	be transfered to the new thumb and should not be referenced again
 *	after this call.
 *
 *	Returns the thumb number that it was appended to or -1 on failure.
 */
static gint EDVImbrTListAppendObject(
        edv_core_struct *core_ptr, edv_imbr_struct *imbr,
        tlist_struct *tlist, edv_object_struct *object
)
{
	gint thumb_num;


	if((core_ptr == NULL) || (imbr == NULL) ||
           (tlist == NULL) || (object == NULL)
	)
	{
            EDVObjectDelete(object);
            return(-1);
	}

	/* Append a new thumb. */
	thumb_num = TListAppend(
	    tlist, "",
	    object, EDVImbrTListItemDestroyCB
	);
	/* Error appending new thumb? */
	if(thumb_num < 0)
	{
	    EDVObjectDelete(object);
            return(-1);
	}

	/* Reset load state. */
	TListSetLoadState(tlist, thumb_num, TLIST_LOAD_STATE_NOT_LOADED);

	/* Set initial thumb text, any default pixmap, and the load
	 * state.
	 */
	EDVImbrTListSetThumbName(
	    core_ptr, imbr, tlist, object, thumb_num
	);

        /* Object pointer has now been transfered to the new thumb's
         * data. Mark it NULL so we do not reference it again.
         */
        object = NULL;

	return(thumb_num);
}

/*
 *	Returns the thumb index number who's disk object data structure's
 *	full path matches the given path.
 *
 *	Can return -1 on error or failed match.
 */
gint EDVImbrTListFindThumbByPath(
        edv_imbr_struct *imbr, const gchar *path
)
{
	gint i;
	tlist_struct *tlist;
	edv_object_struct *object;


        if((imbr == NULL) || (path == NULL))
            return(-1);

	tlist = imbr->tlist;
        if(tlist == NULL)
            return(-1);

        /* Iterate through all thumbs. */
        for(i = 0; i < tlist->total_thumbs; i++)
        {
            object = (edv_object_struct *)TListGetThumbData(tlist, i);
            if(object == NULL)
                continue;

            /* Full path not specified on disk object structure? */
            if(object->full_path == NULL)
                continue;

            /* Full paths match? */
            if(!strcmp(object->full_path, path))
                return(i);
        }

        return(-1);
}

/*
 *      Updates all thumbs on the thumbs list by getting the thumb data
 *      and updating the thumb list's thumbs.
 *
 *      This is designed to be called whenever the displayed values
 *      of each thumb need to be set again from internal data, usually
 *      when a MIME Type has been added/modified/removed. This function
 *      should not be used to `refresh' the list (get new values of
 *      disk object structures), use EDVImbrTListDoUpdate()
 *      instead.
 */
void EDVImbrTListResetThumbs(edv_imbr_struct *imbr)
{
        gint i;
        tlist_struct *tlist;
        edv_core_struct *core_ptr;
	tlist_thumb_struct *thumb_ptr;
        edv_object_struct *object;


        if(imbr == NULL)
            return;

        core_ptr = (edv_core_struct *)imbr->core_ptr;
        if(core_ptr == NULL)
            return;

        tlist = imbr->tlist;
        if(tlist == NULL)
            return;

        for(i = 0; i < tlist->total_thumbs; i++)
        {
	    thumb_ptr = tlist->thumb[i];
	    if(thumb_ptr == NULL)
		continue;

            object = (edv_object_struct *)thumb_ptr->client_data;
            if(object != NULL)
		EDVImbrTListSetThumbName(
		    core_ptr, imbr, tlist, object, i
                );
        }
}


/*
 *	Deallocates all thumbs currently in the thumbs list and loads
 *	the thumbs (but not each image) from the specified directory
 *	path.
 */
void EDVImbrTListDoUpdate(
        edv_imbr_struct *imbr, const gchar *path,
        gbool update_status_bar
)
{
        gbool	hide_object_hidden, hide_object_noaccess,
		hide_object_nonimages;
	gint strc;
	gchar **strv;
        tlist_struct *tlist;
	edv_core_struct *core_ptr;


        if((imbr == NULL) || (path == NULL))
            return;

	core_ptr = (edv_core_struct *)imbr->core_ptr;
	if(core_ptr == NULL)
	    return;

        tlist = imbr->tlist;
        if(tlist == NULL)
            return;

        hide_object_hidden = !EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_IMBR_SHOW_OBJECT_HIDDEN
        );
        hide_object_noaccess = !EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_IMBR_SHOW_OBJECT_NOACCESS
        );
	hide_object_nonimages = !EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_IMBR_SHOW_OBJECT_NONIMAGE
        );


	/* Begin clearing the thumbs list and getting new listing. */

	TListFreeze(tlist);

	TListClear(tlist);

        /* Get listing of contents in the directory specified by path. */
        strv = GetDirEntNames2(path, &strc);
        if(strv != NULL)
        {
            gint i, objects_loaded = 0;
            gint last_progress_percent, progress_percent;
	    const gchar *cstrptr, *cstrptr2;
	    gchar *obj_path;
	    edv_object_struct *object;
	    gint stat_result;
	    struct stat stat_buf, lstat_buf;


            /* Sort strings. */
            strv = StringQSort(strv, strc);
            if(strv != NULL)
            {
		/* Iterate through directory entry names and pick out
		 * just the directories for the first itteration.
		 */
                last_progress_percent = -1;     /* None/undefined. */
                progress_percent = 0;
		obj_path = NULL;
		for(i = 0; i < strc; i++)
                {
#define FREE_AND_CONTINUE       \
{ \
 g_free(obj_path); \
 obj_path = NULL; \
\
 /* Do not deallocate strv[i] on the first itteration pass. */ \
\
 continue; \
}

		    cstrptr = strv[i];
		    if(cstrptr == NULL)
			FREE_AND_CONTINUE

		    /* Skip special dir notations. */
		    if(!strcmp(cstrptr, "."))
			FREE_AND_CONTINUE

		    /* Allocate full path to object as obj_path. */
                    cstrptr2 = PrefixPaths(path, cstrptr);
                    obj_path = (cstrptr2 != NULL) ?
                        g_strdup(cstrptr2) : NULL;
                    if(obj_path == NULL)
			FREE_AND_CONTINUE

		    /* Get destination object statistics. */
		    if(stat(obj_path, &stat_buf))
			FREE_AND_CONTINUE
		    /* Skip if object's destination is not a directory. */
		    if(!S_ISDIR(stat_buf.st_mode))
			FREE_AND_CONTINUE

		    /* Object's destination is a directory, now get
		     * local stats.
		     */
		    lstat(obj_path, &lstat_buf);

		    /* Create a new disk object structure, set it up, and
                     * add it to the list.
                     */
		    object = EDVObjectNew();
		    if(object != NULL)
		    {
			gint thumb_num;


			/* Set disk object structure. */
                        EDVObjectSetPath(object, obj_path);
			EDVObjectSetStat(object, &lstat_buf);
			object->link_valid = TRUE;

                        /* Begin filter checks. */
                        if((hide_object_hidden ?
                            EDVCheckObjectHidden(core_ptr, object) : FALSE) ||
                           (hide_object_noaccess ?
                            !EDVCheckObjectAccessable(core_ptr, object) : FALSE)
                        )
                        {
                            /* Filter check failed, delete disk object
                             * structure instead of appending it.
                             */
                            EDVObjectDelete(object);
                            object = NULL;
                            thumb_num = -1;
                        }
                        else
                        {
                            /* Append thumb using this disk object structure,
                             * the object structure will be taken by this
                             * function and should not be referenced afterwards.
                             */
			    thumb_num = EDVImbrTListAppendObject(
				core_ptr, imbr, tlist, object
			    );
			    object = NULL;

			    objects_loaded++;	/* Count this as an object loaded. */
			}
		    }

#if 0
/* Do not update progress when getting listing of thumbs as disk objects,
 * since progress should only be updated when each thumb's image is loaded.
 */
                    /* Update progress? */
                    if(update_status_bar && (strc > 0))
                    {
                        progress_percent = objects_loaded * 100 / strc;
                        if(progress_percent > last_progress_percent)
                        {
                            EDVStatusBarProgress(
                                imbr->status_bar,
                                (gfloat)objects_loaded / (gfloat)strc,
                                TRUE
                            );
                            progress_percent = last_progress_percent;
                        }
                    }
#endif

		    FREE_AND_CONTINUE
#undef FREE_AND_CONTINUE
		}

		/* Now iterate through all other objects, skipping
		 * directories. Note that this itteration will also delete
		 * all strings in strv.
		 */
		obj_path = NULL;
		for(i = 0; i < strc; i++)
		{
#define FREE_AND_CONTINUE       \
{ \
 g_free(obj_path); \
 obj_path = NULL; \
\
 g_free(strv[i]); \
 strv[i] = NULL; \
\
 continue; \
}
                    cstrptr = strv[i];
		    if(cstrptr == NULL)
			FREE_AND_CONTINUE

                    /* Skip special dir notations. */
                    if(!strcmp(cstrptr, "."))
                        FREE_AND_CONTINUE

                    /* Allocate full path to object as obj_path. */
                    cstrptr2 = PrefixPaths(path, cstrptr);
                    obj_path = (cstrptr2 != NULL) ?
                        g_strdup(cstrptr2) : NULL;
                    if(obj_path == NULL)
                        FREE_AND_CONTINUE

                    /* Get local stats of the object. */
                    if(lstat(obj_path, &lstat_buf))
			FREE_AND_CONTINUE

		    /* Get destination stats of the object. */
		    stat_result = stat(obj_path, &stat_buf);
		    if(!stat_result)
		    {
			/* Check if object destination is a directory. */
			if(S_ISDIR(stat_buf.st_mode))
			    FREE_AND_CONTINUE
		    }

		    /* Create a new disk object structure, set it up, and
		     * add it to the list.
		     */
                    object = EDVObjectNew();
                    if(object != NULL)
                    {
			gint thumb_num;


			/* Set disk object structure. */
                        EDVObjectSetPath(object, obj_path);
                        EDVObjectSetStat(object, &lstat_buf);
                        object->link_valid = stat_result ? FALSE : TRUE;

                        /* Begin filter checks. */
                        if((hide_object_hidden ?
                            EDVCheckObjectHidden(core_ptr, object) : FALSE) ||
/*
                           (hide_object_noaccess ?
                            !EDVCheckObjectAccessable(core_ptr, object) : FALSE) ||
 */
                           (hide_object_nonimages ?
                            !EDVCheckImlibImage(core_ptr, obj_path) : FALSE)
                        )
                        {
                            /* Filter check failed, delete disk object
                             * structure instead of appending it.
                             */
                            EDVObjectDelete(object);
                            object = NULL;
                            thumb_num = -1;
                        }
                        else
                        {
                            /* Append thumb using this disk object structure,
                             * the object structure will be taken by this
                             * function and should not be referenced afterwards.
                             */
                            thumb_num = EDVImbrTListAppendObject(
                                core_ptr, imbr, tlist, object
                            );
                            object = NULL;

			    objects_loaded++;	/* Count this as an object loaded. */
                        }
                    }

#if 0
/* Do not update progress when getting listing of thumbs as disk objects,
 * since progress should only be updated when each thumb's image is loaded.
 */
                    /* Update progress? */
                    if(update_status_bar && (strc > 0))
                    {
                        progress_percent = objects_loaded * 100 / strc;
                        if(progress_percent > last_progress_percent)
                        {
                            EDVStatusBarProgress(
                                imbr->status_bar,
                                (gfloat)objects_loaded / (gfloat)strc,
                                TRUE
                            );
                            progress_percent = last_progress_percent;
                        }
                    }
#endif

		    FREE_AND_CONTINUE
#undef FREE_AND_CONTINUE
		}

		/* At this point all strings in the strv pointer array have
		 * been deallocated.
		 */
	    }

	    /* Deallocate the string pointer array, each string should
	     * already be deallocated.
	     */
	    g_free(strv);
	    strv = NULL;
	}

	TListThaw(tlist);
}

/*
 *	Loads the next unloaded thumb (if any).
 *
 *	The next unloaded thumb will be the next thumb from the
 *	current scroll position. If there are no unloaded thumbs there
 *	then a second pass will check if there is an unloaded thumb starting
 *	from the top.
 *
 *	Returns:
 *
 *	0	All thumbs loaded
 *	1	Thumb loaded successfully but still might be more thumbs
 *		that need to be loaded
 *	-1	General error
 *	-2	Ambiguous, corrupt image, or unsupported format
 *	-3	Systems error
 */
gint EDVImbrTListLoadIterate(
        edv_imbr_struct *imbr, gbool update_status_bar,
	gbool no_enlarge
)
{
	gint starting_thumb_num, thumb_num, thumb_to_load_num = -1;
	gint thumbs_already_loaded = 0;
	tlist_struct *tlist;
	tlist_thumb_struct *thumb_ptr, *thumb_to_load = NULL;
	edv_object_struct *obj;

#define DO_RESET_PROGRESS	\
{ \
 if(update_status_bar) \
 { \
  EDVStatusBarProgress(imbr->status_bar, 0.0, FALSE); \
  EDVStatusBarMessage(imbr->status_bar, NULL, FALSE); \
 } \
}

	if(imbr == NULL)
	{
	    DO_RESET_PROGRESS
	    return(-1);
	}

	tlist = imbr->tlist;
	if(tlist == NULL)
	{
	    DO_RESET_PROGRESS
	    return(-1);
	}

	/* Count number of thumbs already loaded. */
	if(update_status_bar)
	{
            for(thumb_num = 0;
                thumb_num < tlist->total_thumbs;
                thumb_num++
            )
            {
                thumb_ptr = tlist->thumb[thumb_num];
                if(thumb_ptr == NULL)
		{
		    thumbs_already_loaded++;	/* Count NULL's as loaded. */
                    continue;
		}

		/* This thumb loaded? */
                if(thumb_ptr->load_state != TLIST_LOAD_STATE_NOT_LOADED)
		    thumbs_already_loaded++;
	    }
	}

	/* Calculate first starting thumb at scroll position. */
	TListGetSelection(tlist, 0, 0, &starting_thumb_num, NULL, NULL);

	/* Look for thumb to load starting from the scroll position to the
	 * end of the list.
	 */
	if(starting_thumb_num > -1)
	{
	    for(thumb_num = starting_thumb_num;
		thumb_num < tlist->total_thumbs;
		thumb_num++
	    )
	    {
		thumb_ptr = tlist->thumb[thumb_num];
		if(thumb_ptr == NULL)
		    continue;

		if(thumb_ptr->load_state == TLIST_LOAD_STATE_NOT_LOADED)
		{
		    thumb_to_load = thumb_ptr;
		    thumb_to_load_num = thumb_num;
		    break;
		}
	    }
	}
	/* No thumb to load found? Then iterate from start to
	 * starting_thumb_num.
	 */
	if((thumb_to_load == NULL) &&
	   (starting_thumb_num <= tlist->total_thumbs)
	)
	{
	    for(thumb_num = 0; thumb_num < starting_thumb_num; thumb_num++)
	    {
                thumb_ptr = tlist->thumb[thumb_num];
                if(thumb_ptr == NULL)
                    continue;

                if(thumb_ptr->load_state == TLIST_LOAD_STATE_NOT_LOADED)
                {
                    thumb_to_load = thumb_ptr;
                    thumb_to_load_num = thumb_num;
                    break;
                }
            }
	}

	/* No more thumbs need to be loaded? */
	if((thumb_to_load == NULL) || (thumb_to_load_num < 0))
	{
	    if(update_status_bar)
	    {
		EDVStatusBarProgress(imbr->status_bar, 0.0, FALSE);
		EDVStatusBarMessage(
		    imbr->status_bar,
		    "Loading done",
		    FALSE
		);
	    }
	    return(0);
	}

	/* Get disk object structure from thumb data. */
	obj = (edv_object_struct *)thumb_to_load->client_data;
	if((obj != NULL) ? (obj->full_path == NULL) : TRUE)
	{
	    thumb_to_load->load_state = TLIST_LOAD_STATE_FAILED;
	    DO_RESET_PROGRESS
	    return(-1);
	}

	if(1)
	{
	    gint width, height, bpl;
	    guint8 *data_rgba;


	    if(update_status_bar)
	    {
		/* Format status bar message. */
		gchar *buf = g_strdup_printf(
		    "Loading \"%s\" (%ld bytes)",
		    obj->name, obj->size
		);
		/* Set status bar message. */
		EDVStatusBarMessage(
		    imbr->status_bar, buf, FALSE
		);
		/* Deallocate status bar message. */
		g_free(buf);

		/* Set new progress. */
		if(tlist->total_thumbs > 0)
		    EDVStatusBarProgress(
			imbr->status_bar,
			(gfloat)thumbs_already_loaded /
			    (gfloat)tlist->total_thumbs,
			FALSE
		    );
	    }

	    /* Load image. */
	    data_rgba = EDVImbrTListLoadImageRGBA(
		(edv_core_struct *)imbr->core_ptr,
		obj->full_path, &width, &height, &bpl,
		(tlist->toplevel != NULL) ?
		    tlist->toplevel->window : NULL
	    );
	    if(data_rgba == NULL)
	    {
                thumb_to_load->load_state = TLIST_LOAD_STATE_FAILED;
		DO_RESET_PROGRESS
                return(-1);
            }

	    /* Update load state on thumb. */
            thumb_to_load->load_state = TLIST_LOAD_STATE_LOADED;

	    /* Set image to thumb. */
	    TListSetRGBA(
		tlist, thumb_to_load_num,
		width, height, bpl,
		GDK_RGB_DITHER_NORMAL,
		data_rgba,
		no_enlarge
	    );

	    /* Deallocate loaded image buffer. */
	    g_free(data_rgba);
	    data_rgba = NULL;

	    /* Report one thumb loaded. */
	    return(1);
	}

	return(0);
}


/*
 *	Procedure to open the object specified by thumb_num
 */
void EDVImbrTListDoOpenObject(
        edv_imbr_struct *imbr, gint thumb_num, guint state
)
{
	tlist_struct *tlist;
	edv_object_struct *object;
	const gchar *name;
	gchar *full_path;
	struct stat stat_buf;


	if(imbr == NULL)
	    return;

	tlist = imbr->tlist;
	if(tlist == NULL)
	    return;


	/* Get object from row data. */
	object = (edv_object_struct *)TListGetThumbData(tlist, thumb_num);
	if(object == NULL)
	    return;

	/* Check name of object for special notations. */
	name = object->name;
	if(name != NULL)
	{
	    if(!strcmp(name, "."))
	    {
		full_path = NULL;
	    }
	    else if(!strcmp(name, ".."))
	    {
		const gchar *cstrptr = GetParentDir(
		    EDVImbrCurrentLocation(imbr)
		);
		full_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
	    }
	    else
	    {
		full_path = (object->full_path != NULL) ?
		    g_strdup(object->full_path) : NULL;
	    }
	}
	else
	{
	    /* No name available, then just get a copy of the full path. */
	    full_path = (object->full_path != NULL) ?
		g_strdup(object->full_path) : NULL;
	}
	if(full_path == NULL)
	    return;

	/* Get stats of destination object, the object structure may
	 * have that information already but we want to ensure we have
	 * the most up to date information.
	 */
	if(stat(full_path, &stat_buf))
	{
	    g_free(full_path);
	    return;
	}

	/* Is destination a directory? */
	if(S_ISDIR(stat_buf.st_mode))
	{
	    /* Change directory. */
	    EDVImbrSetBusy(imbr, TRUE);
	    EDVImbrSelectPath(imbr, full_path);
	    EDVImbrUpdateMenus(imbr);
	    EDVImbrSetBusy(imbr, FALSE);
	}
	else
	{
	    gchar	*stdout_path_rtn = NULL,
			*stderr_path_rtn = NULL;
	    const gchar *command_name = NULL;

	    if(state & GDK_CONTROL_MASK)
		command_name = "edit";
	    else if(state & GDK_SHIFT_MASK)
		command_name = "edit";

            EDVImbrSetBusy(imbr, TRUE);
	    EDVOpenObjectPath(
		(edv_core_struct *)imbr->core_ptr, full_path,
		command_name,		/* Command name. */
		imbr->toplevel, TRUE,
		&stdout_path_rtn, &stderr_path_rtn
	    );
            EDVImbrSetBusy(imbr, FALSE);

	    g_free(stdout_path_rtn);
	    stdout_path_rtn = NULL;
	    g_free(stderr_path_rtn);
	    stderr_path_rtn = NULL;
	}

	/* Deallocate copy of full path. */
	g_free(full_path);
	full_path = NULL;
}

/*
 *      Procedure to `open with' the object specified by thumb_num
 */
void EDVImbrTListDoOpenWithObject(
        edv_imbr_struct *imbr, gint thumb_num
)
{
        tlist_struct *tlist;
        edv_object_struct *object;
        const gchar *name;
        gchar *full_path;
        struct stat stat_buf;


        if(imbr == NULL)
            return;

        tlist = imbr->tlist;
        if(tlist == NULL)
            return;


        /* Get object from row data. */
        object = (edv_object_struct *)TListGetThumbData(tlist, thumb_num);
        if(object == NULL)
            return;

        /* Check name of object for special notations. */
        name = object->name;
        if(name != NULL)
        {
            if(!strcmp(name, "."))
	    {
                full_path = NULL;
            }
            else if(!strcmp(name, ".."))
            {
                const gchar *cstrptr = GetParentDir(
                    EDVImbrCurrentLocation(imbr)
                );
                full_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
            }
            else
            {
                full_path = (object->full_path != NULL) ?
                    g_strdup(object->full_path) : NULL;
            }
        }
        else
        {
            /* No name available, then just get a copy of the full path. */
            full_path = (object->full_path != NULL) ?
                g_strdup(object->full_path) : NULL;
        }
        if(full_path == NULL)
            return;

        /* Get stats of destination object, the object structure may
         * have that information already but we want to ensure we have
         * the most up to date information.
         */
        if(stat(full_path, &stat_buf))
        {
            g_free(full_path);
            return;
        }

	if(1)
	{
            gchar       *stdout_path_rtn = NULL,
                        *stderr_path_rtn = NULL;

            EDVOpenWithObjectPath(
                (edv_core_struct *)imbr->core_ptr, full_path,
                NULL,                   /* Command name. */
                imbr->toplevel, TRUE,
                &stdout_path_rtn, &stderr_path_rtn
            );

            g_free(stdout_path_rtn);
            stdout_path_rtn = NULL;
            g_free(stderr_path_rtn);
            stderr_path_rtn = NULL;
        }

        /* Deallocate copy of full path. */
        g_free(full_path);
        full_path = NULL;
}


/*
 *      FPrompt apply callback, set in EDVImbrTListDoFPromptRename().
 */
static void EDVImbrTListFPromptRenameApplyCB(
        gpointer data, const gchar *value
)
{
        edv_imbr_struct *imbr;
        tlist_struct *tlist;
	gint thumb_num;
        gpointer *cb_data = (gpointer *)data;
        if(cb_data == NULL)
            return;

        /* Get callback data. */
        imbr = (edv_imbr_struct *)cb_data[0];
        tlist = (tlist_struct *)cb_data[1];
	thumb_num = (gint)cb_data[2];

        /* Inputs valid? */
        if((imbr != NULL) && (tlist != NULL) && (thumb_num >= 0) &&
           (value != NULL)
        )
        {
	    edv_core_struct *core_ptr = (edv_core_struct *)imbr->core_ptr;
            edv_object_struct *object = (edv_object_struct *)TListGetThumbData(tlist, thumb_num);


	    /* Check if the selected object's disk object structure is
	     * valid.
	     */
            if((object != NULL) ? (object->full_path != NULL) : FALSE)
            {
		gchar *old_full_path = g_strdup(object->full_path);
		gchar *new_obj = NULL;
		const gchar *error_mesg;
		gbool yes_to_all = FALSE;
		gint status;


		/* Perform rename. */
		status = EDVFOPRename(
		    core_ptr, old_full_path, value,
		    &new_obj, imbr->toplevel,
		    FALSE, TRUE,
		    &yes_to_all
		);

                /* Unmap progress dialog, it may have been mapped in the
                 * above operation.
                 */
                ProgressDialogBreakQuery(FALSE);
                ProgressDialogSetTransientFor(NULL);

		/* Get error message if any that might have occured in the
		 * above operation.
		 */
                error_mesg = EDVFOPGetError();
                if(error_mesg != NULL)
                {
                    CDialogSetTransientFor(imbr->toplevel);
                    CDialogGetResponse(
                        "Operation Error",
                        error_mesg,
                        NULL,
                        CDIALOG_ICON_ERROR,
                        CDIALOG_BTNFLAG_OK,
                        CDIALOG_BTNFLAG_OK
                    );
                    CDialogSetTransientFor(NULL);
                }

		/* Got new object full path name (implying success)? */
		if((new_obj != NULL) && (old_full_path != NULL))
		{
		    struct stat lstat_buf;
		    const gchar *new_child, *old_child;


		    /* Get child object names. */
		    new_child = strrchr(new_obj, DIR_DELIMINATOR);
		    if(new_child == NULL)
			new_child = new_obj;
		    else
			new_child++;

		    old_child = strrchr(old_full_path, DIR_DELIMINATOR);
                    if(old_child == NULL)
                        old_child = old_full_path;
                    else
                        old_child++;

		    /* Get new local statistics for the renamed object. */
		    if(!lstat(new_obj, &lstat_buf))
		    {
                        gchar *buf = g_strdup_printf(
			    "Object \"%s\" renamed to \"%s\"",
			    old_child, new_child
			);
                        EDVStatusBarMessage(
                            imbr->status_bar, buf, FALSE
                        );
                        g_free(buf);

                        /* Emit object modified signal to all windows. */
			EDVObjectModifiedEmit(
			    core_ptr, old_full_path,
			    new_obj, &lstat_buf
			);
		    }
		}
                else
                {
                    /* Did not get new object path new_obj, implying failed. */
                    EDVStatusBarMessage(
                        imbr->status_bar, "Rename object failed", FALSE
                    );
                }

		/* The disk object structure may now be invalid if the
		 * object modified signal was emitted.
		 */
		object = NULL;


		/* Deallocate coppies of paths. */
		g_free(new_obj);
		g_free(old_full_path);
            }
        }

        /* Deallocate callback data, it is no longer needed. */
        g_free(cb_data);
        cb_data = NULL;
}

/*
 *	Procedure to map the floating prompt for renaming of the row
 *	specified by row.
 *
 *      If the column is -1 then it implies any column is valid.
 */
void EDVImbrTListDoFPromptRename(
        edv_imbr_struct *imbr, gint thumb_num
)
{
	gint cx, cy, px, py, pwidth, pheight;
        edv_object_struct *object;
	edv_core_struct *core_ptr;
	tlist_struct *tlist;


        if((imbr == NULL) || (thumb_num < 0))
            return;

        if(FPromptIsQuery())
            return;

	core_ptr = (edv_core_struct *)imbr->core_ptr;
	if(core_ptr == NULL)
	    return;

        /* Check and warn if write protect is enabled. */
        if(EDVCheckWriteProtect(core_ptr, TRUE))
            return;

	/* Get thumbs list. */
	tlist = imbr->tlist;
	if(tlist == NULL)
	    return;


        /* Sync data on image browser so as to ensure we have the most up
	 * to date information to send out.
         */
        EDVImbrSyncData(imbr);


	/* Make sure given thumb index is in bounds. */
	if((thumb_num < 0) || (thumb_num >= tlist->total_thumbs))
	    return;

        /* Get position of given thumb. */
        if(!TListGetPosition(tlist, thumb_num, &cx, &cy))
	    return;

	pwidth = tlist->thumb_width;
	pheight = 20;

        /* Get root window relative coordinates. */
        px = 0;
        py = 0;
	if(tlist->list_da != NULL)
	    gdk_window_get_deskrelative_origin(
		tlist->list_da->window, &px, &py
	    );
        px += cx;
        py += cy + tlist->thumb_height - pheight - tlist->thumb_border - 4;

        /* Get disk object structure thumb data. */
        object = (edv_object_struct *)TListGetThumbData(tlist, thumb_num);
        if(object == NULL)
            return;

	/* Check if object name is a special notation that should not be
	 * allowed to be renamed.
	 */
	if(object->name != NULL)
	{
	    const gchar *name = object->name;
	    if(!strcmp(name, ".") || !strcmp(name, "..") ||
               !strcmp(name, "/")
	    )
		return;
	}

	if(1)
	{
            gpointer *cb_data = (gpointer *)g_malloc0(
                3 * sizeof(gpointer)
            );
            gchar *value = (object->name != NULL) ?
                g_strdup(object->name) : NULL;

            /* Set up callback data. */
            if(cb_data != NULL)
            {
                cb_data[0] = imbr;
                cb_data[1] = tlist;
                cb_data[2] = (gpointer)thumb_num;
            }

            /* Map floating prompt to change values. */
            FPromptSetTransientFor(imbr->toplevel);
            FPromptSetPosition(px, py);
            FPromptMapQuery(
                NULL,                   /* No label. */
                value,
                NULL,                   /* No tooltip message. */
                FPROMPT_MAP_NO_MOVE,    /* Map code. */
                pwidth, -1,             /* Width and height. */
/*              FPROMPT_FLAG_OK_BTN | FPROMPT_FLAG_CANCEL_BTN, */
                0,      /* No buttons. */
                (gpointer)cb_data,      /* Callback data. */
                NULL,                   /* No browse callback. */
                EDVImbrTListFPromptRenameApplyCB,
                EDVImbrTListFPromptCancelCB
            );

	    /* Do not reference cb_data after this call, it will be passed
	     * to the callbacks where it will be deallocated.
	     */
	    cb_data = NULL;

            /* Deallocate original value. */
            g_free(value);
	    value = NULL;
        }
}


/*
 *      This should be called whenever a new object has been added, it
 *      will add a new row as needed to represent the new object.
 *
 *      The given path must be an absolute path to the object.
 */
void EDVImbrTListObjectAddedNotify(
        edv_imbr_struct *imbr, const gchar *path,
        const struct stat *lstat_buf
)
{
	const gchar *cstrptr;
	gchar *cur_path = NULL, *parent_path = NULL;
	gint thumb_num;
	tlist_struct *tlist;
        edv_core_struct *core_ptr;
        edv_object_struct *object;
	gint stat_result;
	struct stat stat_buf;


        if((imbr == NULL) || (path == NULL) || (lstat_buf == NULL))
            return;

        core_ptr = (edv_core_struct *)imbr->core_ptr;
        if(core_ptr == NULL)
            return;

	tlist = imbr->tlist;
        if(tlist == NULL)
            return;

#define DO_FREE_LOCALS	\
{ \
 g_free(cur_path); \
 cur_path = NULL; \
 g_free(parent_path); \
 parent_path = NULL; \
}
        /* Get parent of the given path and current location. */
        cstrptr = EDVImbrCurrentLocation(imbr);
        cur_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
        cstrptr = GetParentDir(path);
	parent_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
	if((cur_path == NULL) || (parent_path == NULL))
	{
	    DO_FREE_LOCALS
	    return;
	}

	/* Is location of what contents list is displaying is different
	 * from the location of the new object to add?
	 */
	if(strcmp(cur_path, parent_path))
        {
            DO_FREE_LOCALS
            return;
        }


	/* Get destination stats. */
	stat_result = stat(path, &stat_buf);


	/* Check if the new path of the object to add reffers to an object
	 * that already exists in the list.
	 */
	thumb_num = EDVImbrTListFindThumbByPath(imbr, path);
	if((thumb_num >= 0) && (thumb_num < tlist->total_thumbs))
	{
	    /* The new object to add already exists in the list, so just
	     * update the row.
	     */

	    /* Get disk object structure from matched thumb. */
	    object = (edv_object_struct *)TListGetThumbData(tlist, thumb_num);
	    if(object != NULL)
	    {
		/* Update disk object structure. */
		EDVObjectSetPath(object, path);
                EDVObjectSetStat(object, lstat_buf);
		object->link_valid = stat_result ? FALSE : TRUE;

		/* Update the thumb's displayed name. */
		EDVImbrTListSetThumbName(
		    core_ptr, imbr, tlist, object, thumb_num
		);
	    }
	}
	else
	{
	    /* Create a new disk object structure. */
	    object = EDVObjectNew();
	    if(object != NULL)
	    {
		/* Set disk object structure. */
		EDVObjectSetPath(object, path);
		EDVObjectSetStat(object, lstat_buf);
                object->link_valid = stat_result ? FALSE : TRUE;

		TListFreeze(tlist);

                /* Append thumb using this disk object structure. */
                thumb_num = EDVImbrTListAppendObject(
                    core_ptr, imbr, tlist, object
                );
                /* The object structure is now invalid after passing
                 * it to EDVImbrTListAppendObject().
                 */
                object = NULL;

		TListThaw(tlist);
	    }

	    /* Need to (re)queue loading process so that any newly added
	     * thumb gets it's image loaded.
	     */
	    EDVImbrQueueLoadingProcess(imbr);
        }

	DO_FREE_LOCALS
#undef DO_FREE_LOCALS
}

/*
 *      This should be called whenever a object has been modified, it will
 *      search for the object and then reupdate the matching thumb.
 *
 *      The given path must be an absolute path to the object and must be
 *      the path of the object's original name. The new_path must be an
 *      absolute path to the object at its new name, the new_path may be
 *      NULL if there was no name change.
 */
void EDVImbrTListObjectModifiedNotify(
        edv_imbr_struct *imbr, const gchar *path,
        const gchar *new_path,
        const struct stat *lstat_buf
)
{
        const gchar *cstrptr;
        gchar *cur_path = NULL, *parent_path = NULL;
        gint thumb_num;
	tlist_struct *tlist;
	edv_core_struct *core_ptr;
	edv_object_struct *object;
	gint stat_result;
	struct stat stat_buf;


        if((imbr == NULL) || (path == NULL) || (lstat_buf == NULL))
            return;

	if(new_path == NULL)
	    new_path = path;

	core_ptr = (edv_core_struct *)imbr->core_ptr;
	if(core_ptr == NULL)
	    return;

        tlist = imbr->tlist;
        if(tlist == NULL)
            return;

#define DO_FREE_LOCALS  \
{ \
 g_free(cur_path); \
 cur_path = NULL; \
 g_free(parent_path); \
 parent_path = NULL; \
}
        /* Get parent of the given path and current location. */
        cstrptr = EDVImbrCurrentLocation(imbr);
        cur_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
        cstrptr = GetParentDir(new_path);
        parent_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
        if((cur_path == NULL) || (parent_path == NULL))
        {
            DO_FREE_LOCALS
            return;
        }


        /* Get destination stats. */
        stat_result = stat(new_path, &stat_buf);


        /* Is given path that has been modified the same as the current
         * location?
         */
        if(!strcmp(new_path, cur_path))
        {
	    /* Reget listing. */
	    EDVImbrTListDoUpdate(imbr, new_path, TRUE);
	}
	else
	{
	    /* Look for a thumb who's disk object full path matches the
	     * old path.
	     */
	    thumb_num = EDVImbrTListFindThumbByPath(imbr, path);
	    if((thumb_num >= 0) && (thumb_num < tlist->total_thumbs))
	    {
		/* Get disk object structure from matched thumb. */
		object = (edv_object_struct *)TListGetThumbData(tlist, thumb_num);
		if(object != NULL)
		{
		    /* Update disk object structure. */
                    EDVObjectSetPath(object, new_path);		/* Use new_path. */
		    EDVObjectSetStat(object, lstat_buf);
		    object->link_valid = stat_result ? FALSE : TRUE;

		    /* Update the thumb's displayed name. */
		    EDVImbrTListSetThumbName(
			core_ptr, imbr, tlist, object, thumb_num
		    );
		}
	    }
        }

	DO_FREE_LOCALS
#undef DO_FREE_LOCALS
}

/*
 *      This should be called whenever a object has been removed, it will
 *      search for the object and then remove the matching thumb.
 *
 *      The given path must be an absolute path to the object.
 */
void EDVImbrTListObjectRemovedNotify(
        edv_imbr_struct *imbr, const gchar *path
)
{
        const gchar *cstrptr;
        gchar *cur_path = NULL, *parent_path = NULL;
	gint thumb_num;
	tlist_struct *tlist;


        if((imbr == NULL) || (path == NULL))
            return;

        tlist = imbr->tlist;
        if(tlist == NULL)
            return;

#define DO_FREE_LOCALS  \
{ \
 g_free(cur_path); \
 cur_path = NULL; \
 g_free(parent_path); \
 parent_path = NULL; \
}
        /* Get parent of the given path and current location. */
        cstrptr = EDVImbrCurrentLocation(imbr);
        cur_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
        cstrptr = GetParentDir(path);
        parent_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
        if((cur_path == NULL) || (parent_path == NULL))
        {
            DO_FREE_LOCALS
            return;
        }

	/* Is given path that has been removed the same as the current
	 * location?
	 */
        if(!strcmp(path, cur_path))
	{
	    /* Clear thumbs list. */
	    TListFreeze(tlist);
	    TListClear(tlist);
	    TListThaw(tlist);
	}
	else
	{
	    /* Look for a thumb that matches the given path. */
            thumb_num = EDVImbrTListFindThumbByPath(imbr, path);
            if((thumb_num >= 0) && (thumb_num < tlist->total_thumbs))
	    {
		/* Remove matched thumb. */
		TListFreeze(tlist);
                TListRemove(tlist, thumb_num);
                TListThaw(tlist);;
	    }
	}

	DO_FREE_LOCALS
#undef DO_FREE_LOCALS
}


/*
 *      This should be called whenever a device has been mounted or
 *	unmounted.
 */
void EDVImbrTListMountNotify(
        edv_imbr_struct *imbr, edv_device_struct *dev_ptr,
	gbool is_mounted
)
{
        const gchar *cstrptr;
        gchar *cur_path = NULL, *mount_path = NULL;
	tlist_struct *tlist;


        if((imbr == NULL) || (dev_ptr == NULL))
            return;

        tlist = imbr->tlist;
        if(tlist == NULL)
            return;

#define DO_FREE_LOCALS  \
{ \
 g_free(cur_path); \
 cur_path = NULL; \
 g_free(mount_path); \
 mount_path = NULL; \
}
        /* Get coppies of current location and mount paths. */
        cstrptr = EDVImbrCurrentLocation(imbr);
        cur_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
	cstrptr = dev_ptr->mount_path;
	mount_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
        if((cur_path == NULL) || (mount_path == NULL))
        {
            DO_FREE_LOCALS
            return;
        }

	/* Need to simplify coppies of current and mount paths. */
	EDVSimplifyPath(cur_path);
	EDVSimplifyPath(mount_path);


	/* Check if mount path was the current location, if it was then
	 * the contents clist needs to be updated.
	 */
        if(!strcmp(cur_path, mount_path))
        {
            /* Reget listing. */
            EDVImbrTListDoUpdate(imbr, cur_path, TRUE);
	}

        DO_FREE_LOCALS
#undef DO_FREE_LOCALS
}
