/*
 *	Endeavour Mark II program IO context
 *
 *	The context is considered the "connection" to the Endeavour
 *	program (regardless if it is running or not).  You need just one
 *	context and you will need to pass it to subsequent Endeavour API
 *	functions.
 */

#ifndef EDVCONTEXT_H
#define EDVCONTEXT_H

#include <glib.h>

/*
 *	Endeavour context structure:
 *
 *	This structure is the fundimental "connection" to Endeavour,
 *	it is used by most functions that interact with Endeavour.
 */
typedef struct {

	/* Configuration list, list of edv_cfg_item_struct's. */
	gpointer cfg_list;

	/* List of queued interprocess commands to be sent, these
	 * will be sent when EDVNotifyFlush() is called.
	 */
	gchar **queued_command;
	gint total_queued_commands;

	/* Quick references. */
	gchar *recycled_index_file;

} edv_context_struct;

/*
 *      Allocates a new Endeavour context structure.
 *
 *	The returned pointer should be deallocated by passing it to
 *	EDVContextDelete().
 */
extern edv_context_struct *EDVContextNew(void);

/*
 *      Loads the configuration from the specified Endeavour
 *      configuration file path.  If path is NULL then the default
 *      configuration file found in the user's home directory
 *      will be used.
 *
 *	This should be called right after EDVContextNew() to ensure
 *	that the ctx is set up properly before passing to any other
 *	function.
 */
extern void EDVContextLoadConfigurationFile(
	edv_context_struct *ctx,
	const gchar *path		/* Can be NULL for default file. */
);

/*
 *	Returns the number of queued interprocess commands pending.
 *
 *	Can return 0 on error, so error is not distinguished.
 */
extern gint EDVContextCommandsPending(edv_context_struct *ctx);

/*
 *	Processes and flushes all pending operations and resources on
 *	the given ctx to Endeavour.
 *
 *	This call will not block/wait for Endeavour to acknowlage
 *	the request, for that use EDVContextWait();
 */
extern void EDVContextFlush(edv_context_struct *ctx);

/*
 *	Waits for any pending operations that were sent to Endeavour
 *	to be processed. This will cause the operation to block until
 *	the pending operations (if any) are completed.
 */
extern void EDVContextWait(edv_context_struct *ctx);

/*
 *	Macro to flush all pending operations and wait for them to
 *	be completed.
 */
#define EDVContextSync(c)	\
{ \
 EDVContextFlush(c); \
 EDVContextWait(c); \
}

/*
 *      Deallocates the Endeavour context and all it's resources.
 */
extern void EDVContextDelete(edv_context_struct *ctx);

#endif	/* EDVCONTEXT_H */
