#include <string.h>
#include <limits.h>
#include <sys/stat.h>
#include <unistd.h>
#include <glib.h>
#include "../../include/disk.h"
#include "edvtypes.h"
#include "../edvobj.h"
#include "../edvrecbin.h"
#include "../edvrecbinfio.h"
#include "edvnotify.h"
#include "edvrecycle.h"
#include "edvutils.h"
#include "config.h"

const gchar *EDVRecycleGetError(edv_context_struct *ctx);
static void EDVCopyRecycledObjectToStatBuf(
        struct stat *stat_buf,
        edv_recbin_object_struct *obj
);
gint EDVRecycledObjectStat(
        edv_context_struct *ctx, guint index,
        gchar **path_rtn,
        struct stat *stat_buf
);
gint EDVRecycledObjectStatAll(
        edv_context_struct *ctx,
        gchar ***path_rtn,
	guint **index_rtn,
        struct stat ***stat_buf,
        gint *total
);
guint EDVRecycle(
        edv_context_struct *ctx, const gchar *path,
        gboolean notify,
        gint (*progress_cb)(
                gpointer client_data, gulong pos, gulong total
        ),
        gpointer client_data
);
gint EDVRecover(
        edv_context_struct *ctx, guint index, const gchar *path,
        gboolean notify,
        gint (*progress_cb)(
                gpointer client_data, gulong pos, gulong total
        ),
        gpointer client_data
);
gint EDVPurge(
        edv_context_struct *ctx, guint index,
        gboolean notify,
        gint (*progress_cb)(
                gpointer client_data, gulong pos, gulong total
        ),
        gpointer client_data
);


/*
 *	Returns a statically allocated string describing the last error
 *	that occured when calling EDVRecycle(), EDVRecover(), or
 *	EDVPurge().  Can return NULL if there was no error.
 */
const gchar *EDVRecycleGetError(edv_context_struct *ctx)
{
	if(ctx == NULL)
	    return(NULL);
	else
	    return(EDVRecBinFIOGetError());
}

/*
 *	Coppies values from the recycled objects structure to the
 *	struct stat stat_buf structure.
 */
static void EDVCopyRecycledObjectToStatBuf(
	struct stat *stat_buf,
	edv_recbin_object_struct *obj
)
{
	if((stat_buf == NULL) || (obj == NULL))
	    return;

	if(1)
	{
	    /* Begin setting mode. */
            stat_buf->st_mode = 0;

            /* Object type. */
            switch(obj->type)
            {
              case EDV_OBJECT_TYPE_FILE:
                stat_buf->st_mode |= S_IFREG;
                break;
              case EDV_OBJECT_TYPE_DIRECTORY:
                stat_buf->st_mode |= S_IFDIR;
                break;
              case EDV_OBJECT_TYPE_LINK:
                stat_buf->st_mode |= S_IFLNK;
                break;
              case EDV_OBJECT_TYPE_DEVICE_BLOCK:
                stat_buf->st_mode |= S_IFBLK;
                break;
              case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
                stat_buf->st_mode |= S_IFCHR;
                break;
              case EDV_OBJECT_TYPE_FIFO:
                stat_buf->st_mode |= S_IFIFO;
                break;
              case EDV_OBJECT_TYPE_SOCKET:
                stat_buf->st_mode |= S_IFSOCK;
                break;
            }

            /* Permissions. */
            if(obj->permissions & EDV_PERMISSION_UEXECUTE)
                stat_buf->st_mode |= S_IXUSR;
            if(obj->permissions & EDV_PERMISSION_UREAD)
                stat_buf->st_mode |= S_IRUSR;
            if(obj->permissions & EDV_PERMISSION_UWRITE)
                stat_buf->st_mode |= S_IWUSR;
            if(obj->permissions & EDV_PERMISSION_GEXECUTE)
                stat_buf->st_mode |= S_IXGRP;
            if(obj->permissions & EDV_PERMISSION_GREAD)
                stat_buf->st_mode |= S_IRGRP;
            if(obj->permissions & EDV_PERMISSION_GWRITE)
                stat_buf->st_mode |= S_IWGRP;
            if(obj->permissions & EDV_PERMISSION_AEXECUTE)
                stat_buf->st_mode |= S_IXOTH;
            if(obj->permissions & EDV_PERMISSION_AREAD)
                stat_buf->st_mode |= S_IROTH;
            if(obj->permissions & EDV_PERMISSION_AWRITE)
                stat_buf->st_mode |= S_IWOTH;
            if(obj->permissions & EDV_PERMISSION_SETUID)
                stat_buf->st_mode |= S_ISUID;
            if(obj->permissions & EDV_PERMISSION_SETGID)
                stat_buf->st_mode |= S_ISGID;
            if(obj->permissions & EDV_PERMISSION_STICKY)
                stat_buf->st_mode |= S_ISVTX;

            /* Time stamps, in systime seconds. */
            stat_buf->st_atime = obj->access_time;
            stat_buf->st_mtime = obj->modify_time;
            stat_buf->st_ctime = obj->change_time;

            /* Owner and group ids. */
            stat_buf->st_uid = obj->owner_id;
            stat_buf->st_gid = obj->group_id;

            stat_buf->st_size = obj->size;
        }
}

/*
 *	Get statistics of a recycled object by its index.
 *
 *	If the path_rtn is not NULL then an allocated string containing
 *	the full path of the original object will be returned.  The
 *	calling function must deallocate this string.
 *
 *	Inputs path_rtn and stat_buf may be NULL, indicating that the
 *	return is not wanted.
 *
 *	Returns 0 on success or -1 on error.
 */
gint EDVRecycledObjectStat(
	edv_context_struct *ctx, guint index,
	gchar **path_rtn,
	struct stat *stat_buf
)
{
	const gchar *recycled_index_file;
	edv_recbin_object_struct *obj;


	if(path_rtn != NULL)
	    *path_rtn = NULL;
	if(stat_buf != NULL)
	    memset(stat_buf, 0x00, sizeof(stat_buf));

        if(ctx == NULL)
            return(-1);

        recycled_index_file = ctx->recycled_index_file;
        if(recycled_index_file == NULL)
            return(-1);

	/* Get recycled object statistics. */
	obj = EDVRecBinObjectStat(recycled_index_file, index);
	if(obj == NULL)
	    return(-1);

	/* Transfer values from recycled object structure to path_rtn
	 * and stat_buf.
	 */
	if(path_rtn != NULL)
	{
	    /* Allocate a string containing the full path to the
	     * original object. Note that original_path only contains
	     * the original location and no the original object's name
	     * so original_path and name need to be put togeather.
	     */
	    *path_rtn = g_strdup_printf(
		"%s%c%s",
		obj->original_path,
		DIR_DELIMINATOR,
		obj->name
	    );
	}
	if(stat_buf != NULL)
	{
	    /* Copy recycled object structure values to stat_buf. */
	    EDVCopyRecycledObjectToStatBuf(stat_buf, obj);
	}

	/* Deallocate recycled object structure. */
	EDVRecBinObjectDelete(obj);

	return(0);
}

/*
 *	Same as EDVRecycledObjectStat() except that a list of all
 *	recycled objects are returned.  The returned pointer arrays
 *	and each pointed to location must be deallocated.by the calling
 *	function.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint EDVRecycledObjectStatAll(
        edv_context_struct *ctx,
        gchar ***path_rtn,
        guint **index_rtn,
        struct stat ***stat_buf,
	gint *total
)
{
	gint i;
	struct stat *stat_buf_ptr;
        const gchar *recycled_index_file;
	edv_recbin_object_struct *obj;
        edv_recbin_index_struct *rbi_ptr;


        if(path_rtn != NULL)
            *path_rtn = NULL;
	if(index_rtn != NULL)
	    *index_rtn = NULL;
        if(stat_buf != NULL)
            *stat_buf = NULL;

        if((ctx == NULL) || (total == NULL))
            return(-1);

        recycled_index_file = ctx->recycled_index_file;
        if(recycled_index_file == NULL)
            return(-1);

	/* Open recycled objects index file for subsequent reading
	 * of each recycled object's stats.
	 */
        rbi_ptr = EDVRecBinIndexOpen(recycled_index_file);
        if(rbi_ptr == NULL)
            return(-1);

	/* Read each recycled object. */
        while(!EDVRecBinIndexNext(rbi_ptr))
        {
	    obj = rbi_ptr->obj;
	    if(obj != NULL)
	    {
		/* Increase total and reallocate each pointer array. */
		i = *total;
		*total = i + 1;

		/* Allocate a new path return on the return pointer
		 * array.
		 */
		if(path_rtn != NULL)
		{
		    *path_rtn = (gchar **)g_realloc(
			*path_rtn,
			(*total) * sizeof(gchar *)
		    );
		    (*path_rtn)[i] = g_strdup_printf(
			"%s%c%s",
			obj->original_path,
			DIR_DELIMINATOR,
			obj->name
		    );
		}

                /* Allocate a new index return on the return pointer
                 * array.
                 */
                if(index_rtn != NULL)
                {
                    *index_rtn = (guint *)g_realloc(
                        *index_rtn,
                        (*total) * sizeof(guint)
                    );
		    (*index_rtn)[i] = obj->index;
                }

		/* Allocate a new stat_buf structure on the return
		 * pointer array.
		 */
		if(stat_buf != NULL)
		{
		    *stat_buf = (struct stat **)g_realloc(
			*stat_buf,
			(*total) * sizeof(struct stat *)
		    );
		    (*stat_buf)[i] = stat_buf_ptr = g_malloc0(sizeof(struct stat));
		    EDVCopyRecycledObjectToStatBuf(stat_buf_ptr, obj);
		}
	    }
        }	/* Read each recycled object. */

        EDVRecBinIndexClose(rbi_ptr);
	rbi_ptr = NULL;

	return(0);
}

/*
 *	Deletes an actual disk object and places it into the recycle
 *	bin.
 *
 *	If notify is set to TRUE then a "object_removed_notify" and
 *	"recycled_object_added_notify" will be queued on the ctx
 *	(only on success).
 *
 *	Returns the recycled object index number or 0 on error.
 */
guint EDVRecycle(
        edv_context_struct *ctx, const gchar *path,
        gboolean notify,
        gint (*progress_cb)(
                gpointer client_data, gulong pos, gulong total
        ),
        gpointer client_data
)
{
        const gchar *cstrptr, *recycled_index_file;
	gchar *dpath;
	gchar cwd[PATH_MAX];
        guint index;
        edv_recbin_object_struct *obj;
        mode_t m;
        struct stat lstat_buf;


	if((ctx == NULL) || (path == NULL))
	    return(0);

	recycled_index_file = ctx->recycled_index_file;
	if(recycled_index_file == NULL)
	    return(0);

        dpath = EDVCopyEvaluateInputPath(
            getcwd(cwd, PATH_MAX), path
        );

	if(lstat(dpath, &lstat_buf))
	{
	    g_free(dpath);
	    return(0);
	}

	m = lstat_buf.st_mode;


	/* Create a new tempory recycled object structure. */
	obj = EDVRecBinObjectNew();

	/* Begin setting values from actual object's lstat_buf structure
	 * to the newly allocated recycled object structure.
	 */

	/* Name. */
	cstrptr = strrchr(dpath, DIR_DELIMINATOR);
	if(cstrptr == NULL)
	    cstrptr = dpath;
	else
	    cstrptr++;
	g_free(obj->name);
	obj->name = g_strdup(cstrptr);

	/* Location of object (path without object's name). */
	cstrptr = GetParentDir(dpath);
	if(cstrptr == NULL)
	    cstrptr = "/";
	g_free(obj->original_path);
        obj->original_path = g_strdup(cstrptr);

	if(S_ISREG(m))
	    obj->type = EDV_OBJECT_TYPE_FILE;
        else if(S_ISDIR(m))
            obj->type = EDV_OBJECT_TYPE_DIRECTORY;
        else if(S_ISLNK(m))
            obj->type = EDV_OBJECT_TYPE_LINK;
        else if(S_ISBLK(m))
            obj->type = EDV_OBJECT_TYPE_DEVICE_BLOCK;
        else if(S_ISCHR(m))
            obj->type = EDV_OBJECT_TYPE_DEVICE_CHARACTER;
        else if(S_ISFIFO(m))
            obj->type = EDV_OBJECT_TYPE_FIFO;
	else if(S_ISSOCK(m))
            obj->type = EDV_OBJECT_TYPE_SOCKET;

	obj->permissions = 0;
	if(m & S_IXUSR)
	    obj->permissions |= EDV_PERMISSION_UEXECUTE;
        if(m & S_IRUSR)
            obj->permissions |= EDV_PERMISSION_UREAD;
        if(m & S_IWUSR)
            obj->permissions |= EDV_PERMISSION_UWRITE;
        if(m & S_IXGRP)
            obj->permissions |= EDV_PERMISSION_GEXECUTE;
        if(m & S_IRGRP)
            obj->permissions |= EDV_PERMISSION_GREAD;
        if(m & S_IWGRP)
            obj->permissions |= EDV_PERMISSION_GWRITE;
        if(m & S_IXOTH)
            obj->permissions |= EDV_PERMISSION_AEXECUTE;
        if(m & S_IROTH)
            obj->permissions |= EDV_PERMISSION_AREAD;
        if(m & S_IWOTH)
            obj->permissions |= EDV_PERMISSION_AWRITE;
        if(m & S_ISUID)
            obj->permissions |= EDV_PERMISSION_SETUID;
        if(m & S_ISGID)
            obj->permissions |= EDV_PERMISSION_SETGID;
        if(m & S_ISVTX)
            obj->permissions |= EDV_PERMISSION_STICKY;

	obj->access_time = lstat_buf.st_atime;
        obj->modify_time = lstat_buf.st_mtime;
        obj->change_time = lstat_buf.st_ctime;

        obj->owner_id = lstat_buf.st_uid;
        obj->group_id = lstat_buf.st_gid;

	obj->size = lstat_buf.st_size;



	/* Record this object to recycled objects index file. */
	index = EDVRecBinIndexAdd(
	     recycled_index_file, obj
	);
	if(index > 0)
	{
	    /* Remove actual disk object and place it into the recycled
	     * objects directory.
	     */
	    gint status = EDVRecBinDiskObjectDelete(
		recycled_index_file,
		index,
		dpath,
		progress_cb, client_data
	    );
	    if(status)
	    {
		/* Error. */
		index = 0;
	    }
	    else if(notify)
	    {
		/* Success and need notify. */
		EDVNotifyQueueObjectRemoved(ctx, dpath);
		EDVNotifyQueueRecycledObjectAdded(ctx, index);
	    }
	}

	/* Delete tempory recycled object structure, it is no longer
	 * needed.
	 */
	EDVRecBinObjectDelete(obj);
	obj = NULL;

	g_free(dpath);

	return(index);
}

/*
 *	Recovers an object from the recycled objects directory.
 *
 *	If path is not NULL, then path will be used as the alternate
 *	location to recover the object.  It is recommended that you
 *	always give a path to recover the object.  The object that path
 *	reffers to may not already exist.
 *
 *	If notify is set to TRUE then a "object_added_notify" and
 *	"recycled_object_removed_notify" will be queued on the ctx
 *	(only on success).
 *
 *	Returns non-zero on error.
 */
gint EDVRecover(
	edv_context_struct *ctx, guint index, const gchar *path,
	gboolean notify,
        gint (*progress_cb)(
                gpointer client_data, gulong pos, gulong total
        ),
        gpointer client_data
)
{
	gint status;
	const gchar *recycled_index_file;
        gchar *dpath;
        gchar cwd[PATH_MAX];


        if(ctx == NULL)
            return(-1);

        recycled_index_file = ctx->recycled_index_file;
        if(recycled_index_file == NULL)
            return(-1);

	dpath = EDVCopyEvaluateInputPath(
	    getcwd(cwd, PATH_MAX), path
	);

	status = EDVRecBinDiskObjectRecover(
	    recycled_index_file,
	    index,
	    dpath,
	    progress_cb, client_data
	);
	if(status)
	{
	    /* Error. */
	}
	else
	{
	    /* Success, now remove the recycled object entry from the
	     * recycled objects index file.
	     */
	    EDVRecBinIndexRemove(
		recycled_index_file, index
	    );

	    /* Need notify? */
            if(notify)
            {
                EDVNotifyQueueObjectAdded(ctx, dpath);
                EDVNotifyQueueRecycledObjectRemoved(ctx, index);
            }
	}

	g_free(dpath);

	return(status);
}

/*
 *	Permanently removes an object from the recycled objects directory.
 *
 *	If notify is set to TRUE then a "recycled_object_removed_notify"
 *	will be queued on the ctx (only on success).
 *
 *	Returns non-zero on error.
 */
gint EDVPurge(
        edv_context_struct *ctx, guint index,
        gboolean notify,
        gint (*progress_cb)(
                gpointer client_data, gulong pos, gulong total
        ),
        gpointer client_data
)
{
        gint status;
        const gchar *recycled_index_file;


        if(ctx == NULL)
            return(-1);

        recycled_index_file = ctx->recycled_index_file;
        if(recycled_index_file == NULL)
            return(-1);

	/* Remove recycled object from the recycled objects directory. */
        status = EDVRecBinDiskObjectPurge(
            recycled_index_file,
            index,
            progress_cb, client_data
        );
        if(status)
	{
	    /* Error. */
	}
	else
        {
            /* Success, now remove the recycled object entry from the
	     * recycled objects index file.
             */
            EDVRecBinIndexRemove(
                recycled_index_file, index
            );

            /* Need notify? */
            if(notify)
            {
                EDVNotifyQueueRecycledObjectRemoved(ctx, index);
            }
        }

        return(status);
}

