#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>

#include "guiutils.h"
#include "cdialog.h"
#include "progressdialog.h"

#include "edvtypes.h"
#include "edvcfg.h"
#include "edvconfirm.h"
#include "edvobj.h"
#include "edvrecbin.h"
#include "edvrecbinfio.h"
#include "edvrecbinfop.h"
#include "recbin.h"
#include "recbincb.h"
#include "recbinopcb.h"
#include "recbincontents.h"
#include "endeavour.h"
#include "edvcb.h"
#include "edvop.h"
#include "edvutils.h"
#include "edvutilsgtk.h"
#include "edvcfglist.h"
#include "config.h"


void EDVRecBinMenuItemCB(GtkWidget *widget, gpointer data);
gint EDVRecBinMenuItemEnterCB(
        GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);
gint EDVRecBinMenuItemLeaveCB(
        GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);
void EDVRecBinOPCB(gpointer data);
void EDVRecBinOPEnterCB(gpointer data);
void EDVRecBinOPLeaveCB(gpointer data);

void EDVRecBinFindBarStartCB(gpointer fb, gpointer data);
void EDVRecBinFindBarEndCB(
        gpointer fb, gint total_matches, gpointer data
);
void EDVRecBinFindBarMatchCB(
        const gchar *path, const struct stat *lstat_buf, const gchar *excerpt,
        gpointer data
);

void EDVRecBinBarStatusMessageCB(
        gpointer bar, const gchar *message, gpointer data
);
void EDVRecBinBarStatusProgressCB(
        gpointer bar, gfloat progress, gpointer data
);

void EDVRecBinOPSyncDisks(edv_recbin_struct *recbin);
void EDVRecBinOPWriteProtect(edv_recbin_struct *recbin);

void EDVRecBinOPClose(edv_recbin_struct *recbin);
void EDVRecBinOPExit(edv_recbin_struct *recbin);

void EDVRecBinOPRecover(edv_recbin_struct *recbin);
void EDVRecBinOPPurge(edv_recbin_struct *recbin);
void EDVRecBinOPPurgeAll(edv_recbin_struct *recbin);
void EDVRecBinOPSelectAll(edv_recbin_struct *recbin);
void EDVRecBinOPUnselectAll(edv_recbin_struct *recbin);
void EDVRecBinOPInvertSelection(edv_recbin_struct *recbin);

void EDVRecBinOPRefresh(edv_recbin_struct *recbin);
void EDVRecBinOPRefreshAll(edv_recbin_struct *recbin);




/*
 *      Menu item activate callback.
 *
 *      The given client data must be a edv_recbin_opid_struct *.
 */
void EDVRecBinMenuItemCB(GtkWidget *widget, gpointer data)
{
        EDVRecBinOPCB(data);
}

/*
 *      Menu item "enter_notify_event" signal callback.
 *
 *      The given client data must be a edv_recbin_opid_struct *.
 */
gint EDVRecBinMenuItemEnterCB(
        GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
        EDVRecBinOPEnterCB(data);
        return(TRUE);
}

/*
 *      Menu item "leave_notify_event" signal callback.
 *
 *      The given client data must be a edv_recbin_opid_struct *.
 */
gint EDVRecBinMenuItemLeaveCB(
        GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
        EDVRecBinOPLeaveCB(data);
        return(TRUE);
}

/*
 *      Operation id callback nexus.
 *
 *      The given client data must be a edv_recbin_opid_struct *.
 */
void EDVRecBinOPCB(gpointer data)
{
        static gbool reenterent = FALSE;
        edv_recbin_struct *recbin;
        edv_core_struct *core_ptr;
        edv_recbin_opid_struct *opid = (edv_recbin_opid_struct *)data;
        if(opid == NULL)
            return;

        recbin = (edv_recbin_struct *)opid->recbin;
        if(recbin == NULL)
            return;

        if(recbin->processing)
            return;

        core_ptr = (edv_core_struct *)recbin->core_ptr;
        if(core_ptr == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Handle by operation id code. */
        switch(opid->op)
        {
	  case EDV_RECBIN_OP_CLOSE:
	    EDVRecBinOPClose(recbin);
	    break;

	  case EDV_RECBIN_OP_EXIT:
	    EDVRecBinOPExit(recbin);
            break;


          case EDV_RECBIN_OP_RECOVER:
	    EDVRecBinOPRecover(recbin);
	    break;

          case EDV_RECBIN_OP_PURGE:
	    EDVRecBinOPPurge(recbin);
	    break;

	  case EDV_RECBIN_OP_PURGE_ALL:
	    EDVRecBinOPPurgeAll(recbin);
	    break;

	  case EDV_RECBIN_OP_SELECT_ALL:
	    EDVRecBinOPSelectAll(recbin);
	    break;

          case EDV_RECBIN_OP_UNSELECT_ALL:
            EDVRecBinOPUnselectAll(recbin);
            break;

          case EDV_RECBIN_OP_INVERT_SELECTION:
            EDVRecBinOPInvertSelection(recbin);
            break;

          case EDV_RECBIN_OP_FIND:
            EDVDoMapRecBinFindWin(core_ptr, recbin);
            break;

          case EDV_RECBIN_OP_HISTORY:
            EDVDoMapHistoryListWin(core_ptr, recbin->toplevel);
            break;

	  case EDV_RECBIN_OP_SYNC_DISKS:
	    EDVRecBinOPSyncDisks(recbin);
	    break;

          case EDV_RECBIN_OP_WRITE_PROTECT:
            EDVRecBinOPWriteProtect(recbin);
            break;

          case EDV_RECBIN_OP_RUN:
            EDVDoMapRunDlg(core_ptr);
            break;

          case EDV_RECBIN_OP_RUN_TERMINAL:
            EDVDoRunTerminal(core_ptr, NULL);
            break;


          case EDV_RECBIN_OP_REFRESH:
	    EDVRecBinOPRefresh(recbin);
	    break;

          case EDV_RECBIN_OP_REFRESH_ALL:
            EDVRecBinOPRefreshAll(recbin);
            break;

          case EDV_RECBIN_OP_SHOW_TOOL_BAR:
            if(core_ptr != NULL)
            {
                gbool state = !EDVCFGItemListGetValueI(
                    core_ptr->cfg_list, EDV_CFG_PARM_RECBIN_SHOW_TOOL_BAR
                );
                EDVCFGItemListSetValueI(
                    core_ptr->cfg_list, EDV_CFG_PARM_RECBIN_SHOW_TOOL_BAR,
                    state, FALSE
                );
                EDVReconfiguredEmit(core_ptr);
            }
            break;

          case EDV_RECBIN_OP_SHOW_FIND_BAR:
            if(core_ptr != NULL)
            {
                gbool state = !EDVCFGItemListGetValueI(
                    core_ptr->cfg_list, EDV_CFG_PARM_RECBIN_SHOW_FIND_BAR
                );
                EDVCFGItemListSetValueI(
                    core_ptr->cfg_list, EDV_CFG_PARM_RECBIN_SHOW_FIND_BAR,
                    state, FALSE
                );
                EDVReconfiguredEmit(core_ptr);
            }
            break;

          case EDV_RECBIN_OP_SHOW_STATUS_BAR:
            if(core_ptr != NULL)
            {
                gbool state = !EDVCFGItemListGetValueI(
                    core_ptr->cfg_list, EDV_CFG_PARM_RECBIN_SHOW_STATUS_BAR
                );
                EDVCFGItemListSetValueI(
                    core_ptr->cfg_list, EDV_CFG_PARM_RECBIN_SHOW_STATUS_BAR,
                    state, FALSE
                );
                EDVReconfiguredEmit(core_ptr);
            }
            break;


          case EDV_RECBIN_OP_MIME_TYPES:
            EDVDoMapMIMETypesListWin(core_ptr, recbin->toplevel);
            break;


          case EDV_RECBIN_OP_NEW_BROWSER:
            EDVDoNewBrowser(core_ptr);
            break;

          case EDV_RECBIN_OP_NEW_IMBR:
            EDVDoNewImbr(core_ptr);
            break;

          case EDV_RECBIN_OP_NEW_ARCHIVER:
            EDVDoNewArchiver(core_ptr);
            break;


          case EDV_RECBIN_OP_OPTIONS:
            EDVDoMapOptionsWin(core_ptr, recbin->toplevel);
            break;

          case EDV_RECBIN_OP_CUSTOMIZE:
            EDVDoMapCustomizeWin(core_ptr, recbin->toplevel);
            break;


          case EDV_RECBIN_OP_HELP_ABOUT:
            EDVDoHelpAbout(core_ptr, recbin->toplevel);
            break;
          case EDV_RECBIN_OP_HELP_CONTENTS:
            EDVDoHelp(core_ptr, "Contents", recbin->toplevel);
            break;
          case EDV_RECBIN_OP_HELP_FILE_BROWSER:
            EDVDoHelp(core_ptr, "File Browser", recbin->toplevel);
            break;
          case EDV_RECBIN_OP_HELP_IMAGE_BROWSER:
            EDVDoHelp(core_ptr, "Image Browser", recbin->toplevel);
            break;
          case EDV_RECBIN_OP_HELP_ARCHIVER:
            EDVDoHelp(core_ptr, "Archiver", recbin->toplevel);
            break;
          case EDV_RECBIN_OP_HELP_RECYCLE_BIN:
            EDVDoHelp(core_ptr, "Recycle Bin", recbin->toplevel);
            break;
          case EDV_RECBIN_OP_HELP_KEYS_LIST:
            EDVDoHelp(core_ptr, "Keys List", recbin->toplevel);
            break;
          case EDV_RECBIN_OP_HELP_COMMON_OPERATIONS:
            EDVDoHelp(core_ptr, "Common Operations", recbin->toplevel);
            break;


          default:
            if(1)
            {
                gchar text[256];

                sprintf(
                    text,
                    "Unsupported recycle bin operation `%i'\n",
                    opid->op
                );

                CDialogSetTransientFor(recbin->toplevel);
                CDialogGetResponse(
                    "Operation Error",
                    text,
                    NULL,
                    CDIALOG_ICON_ERROR,
                    CDIALOG_BTNFLAG_OK,
                    CDIALOG_BTNFLAG_OK
                );
                CDialogSetTransientFor(NULL);
            }
            break;
	}

        reenterent = FALSE;
}

/*
 *      Operation id enter notify callback nexus.
 *
 *      The given client data must be a edv_recbin_opid_struct *.
 */
void EDVRecBinOPEnterCB(gpointer data)
{
        const gchar *tooltip;
        edv_recbin_struct *recbin;
        edv_recbin_opid_struct *opid = (edv_recbin_opid_struct *)data;
        if(opid == NULL)
            return;

        recbin = (edv_recbin_struct *)opid->recbin;
        if(recbin == NULL)
            return;

        tooltip = opid->tooltip;
        if(tooltip != NULL)
            EDVStatusBarMessage(recbin->status_bar, tooltip, FALSE);
}

/*
 *      Operation id leave notify callback nexus.
 */
void EDVRecBinOPLeaveCB(gpointer data)
{
        edv_recbin_struct *recbin;
        edv_recbin_opid_struct *opid = (edv_recbin_opid_struct *)data;
        if(opid == NULL)
            return;

        recbin = (edv_recbin_struct *)opid->recbin;
        if(recbin == NULL)
            return;

        EDVStatusBarMessage(recbin->status_bar, NULL, FALSE);
}


/*
 *      Find bar start find callback.
 */
void EDVRecBinFindBarStartCB(gpointer fb, gpointer data)
{
        GtkCList *clist;
        edv_recbin_struct *recbin = (edv_recbin_struct *)data;
        if(recbin == NULL)
            return;

        clist = (GtkCList *)recbin->contents_clist;
        if(clist == NULL)
            return;

        EDVRecBinSetBusy(recbin, TRUE);
/*      GUIBlockInput(recbin->toplevel, TRUE); */

        gtk_clist_freeze(clist);
        gtk_clist_unselect_all(clist);
        gtk_clist_thaw(clist);

        EDVRecBinUpdateMenus(recbin);
}

/*
 *      Find bar end find callback.
 */
void EDVRecBinFindBarEndCB(
        gpointer fb, gint total_matches, gpointer data
)
{
        edv_recbin_struct *recbin = (edv_recbin_struct *)data;
        if(recbin == NULL)
            return;

/*      GUIBlockInput(recbin->toplevel, FALSE); */
        EDVRecBinSetBusy(recbin, FALSE);
}


/*
 *      Find bar match callback.
 */
void EDVRecBinFindBarMatchCB(
        const gchar *path, const struct stat *lstat_buf, const gchar *excerpt,
        gpointer data
)
{
        gint row;
        GtkCList *clist;
        edv_recbin_struct *recbin = (edv_recbin_struct *)data;
        if((path == NULL) || (lstat_buf == NULL) || (recbin == NULL))
            return;

        clist = (GtkCList *)recbin->contents_clist;
        if(clist == NULL)
            return;

        row = EDVRecBinContentsFindRowByIndex(recbin, (guint)atoi(path));
        if((row >= 0) && (row < clist->rows))
            gtk_clist_select_row(clist, row, 0);
}


/*
 *      Opaque status bar message update callback, the first argument
 *      bar is completely ignored.
 */
void EDVRecBinBarStatusMessageCB(
        gpointer bar, const gchar *message, gpointer data
)
{
        static gbool reenterent = FALSE;
        edv_recbin_struct *recbin = (edv_recbin_struct *)data;
        if(recbin == NULL)
            return;

        if(recbin->processing)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        EDVStatusBarMessage(recbin->status_bar, message, FALSE);

        reenterent = FALSE;
}

/*
 *      Opaque status bar progress update callback, the first argument
 *      bar is completely ignored.
 */
void EDVRecBinBarStatusProgressCB(
        gpointer bar, gfloat progress, gpointer data
)
{
        static gbool reenterent = FALSE;
        edv_recbin_struct *recbin = (edv_recbin_struct *)data;
        if(recbin == NULL)
            return;

        if(recbin->processing)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        EDVStatusBarProgress(recbin->status_bar, progress, FALSE);

        reenterent = FALSE;
}


/*
 *      Flush data transfer output to disks.
 */
void EDVRecBinOPSyncDisks(edv_recbin_struct *recbin)
{
        if(recbin == NULL)
            return;

        EDVStatusBarMessage(
            recbin->status_bar,
            "Flushing all write buffers to disk...",
            TRUE
        );

        EDVRecBinSetBusy(recbin, TRUE);

        EDVDoSyncDisks((edv_core_struct *)recbin->core_ptr);

        EDVRecBinSetBusy(recbin, FALSE);

        EDVStatusBarMessage(
            recbin->status_bar,
            "Disk sync done",
            FALSE
        );
        EDVStatusBarProgress(recbin->status_bar, 0.0, FALSE);
}

/*
 *      Write protect toggle.
 */
void EDVRecBinOPWriteProtect(edv_recbin_struct *recbin)
{
        gbool write_protect;
        edv_core_struct *core_ptr;


        if(recbin == NULL)
            return;

        core_ptr = (edv_core_struct *)recbin->core_ptr;
        if(core_ptr == NULL)
            return;

        /* Get new write protect state, changing from the previous one. */
        write_protect = EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_WRITE_PROTECT
        ) ? FALSE : TRUE;

        /* Update write protect in the configuration list. */
        EDVCFGItemListSetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_WRITE_PROTECT,
            write_protect, FALSE
        );

        /* Notify all resources about change in write protect. */
        EDVWriteProtectChangedEmit(core_ptr, write_protect);
}


/*
 *      Close recycle bin.
 */
void EDVRecBinOPClose(edv_recbin_struct *recbin)
{
        if(recbin == NULL)
            return;

        /* Set current properties of the recycle bin to the configuration
         * list.
         */
        EDVRecBinSyncConfiguration(recbin);

        /* Unmap the recbin. */
        EDVRecBinUnmap(recbin);
}

/*
 *      Close all windows.
 */
void EDVRecBinOPExit(edv_recbin_struct *recbin)
{
        edv_core_struct *core_ptr;


        if(recbin == NULL)
            return;

        core_ptr = (edv_core_struct *)recbin->core_ptr;
        if(core_ptr == NULL)
            return;

        /* Set current properties of the recycle bin to the configuration
         * list.
         */
        EDVRecBinSyncConfiguration(recbin);

        /* Unmap this recycle bin. */
        EDVRecBinUnmap(recbin);

        /* Mark need_close_all_windows on the core structure to TRUE, this
         * will be checked during the management timeout. In which case
         * all windows will be deleted.
         */
        core_ptr->need_close_all_windows = TRUE;
}


/*
 *	Recover callback.
 */
void EDVRecBinOPRecover(edv_recbin_struct *recbin)
{
        gbool yes_to_all = FALSE;
        guint *index = NULL;
        gint total_indexes = 0;
        gint recbin_objects_recovered = 0;
        gint i, row, status;
	gchar *new_path;
        const gchar *error_mesg;
        GList *glist;
        GtkCList *clist;
        edv_recbin_object_struct *obj = NULL;
        edv_core_struct *core_ptr;
	gchar obj_index_str[80];


        if(recbin == NULL)
            return;

        core_ptr = (edv_core_struct *)recbin->core_ptr;
        if(core_ptr == NULL)
            return;

        clist = (GtkCList *)recbin->contents_clist;
        if(clist == NULL)
            return;

        /* Check and warn if write protect is enabled. */
        if(EDVCheckWriteProtect(core_ptr, TRUE))
            return;

#define DO_FREE_LOCALS  \
{ \
 g_free(index); \
 index = NULL; \
 total_indexes = 0; \
}

        /* Get a list of selected recycled objects indexes. */
        glist = clist->selection;
        while(glist != NULL)
        {
            row = (gint)glist->data;
            obj = (edv_recbin_object_struct *)gtk_clist_get_row_data(
                clist, row
            );
            if(obj != NULL)
            {
                /* Append this recycled object structure's index to the
                 * list of recycled object indexes to purge.
                 */
                i = total_indexes;
                total_indexes = i + 1;
                index = (guint *)g_realloc(
                    index, total_indexes * sizeof(guint)
                );
                if(index == NULL)
                {
                    total_indexes = 0;
                    break;
                }
                else
                {
                   index[i] = obj->index;
                }
            }

            glist = glist->next;
        }

        /* No recycled objects to recover? */
        if(index == NULL)
	{
	    DO_FREE_LOCALS
            return;
	}


	/* Format recycled object index number string based on the last
	 * selected recycled object (if any). This will be used for
	 * passing to the confirmation function.
	 */
	if(obj != NULL)
	    sprintf(obj_index_str, "%i", obj->index);
	else
	    *obj_index_str = '\0';

	/* Get recover confirmation. */
	status = EDVConfirmRecover(
	    core_ptr, recbin->toplevel,
	    obj_index_str,
	    total_indexes,
	    (obj != NULL) ? obj->original_path : NULL
	);
        switch(status)
        {
          case CDIALOG_RESPONSE_YES:
          case CDIALOG_RESPONSE_YES_TO_ALL:
            break;

          default:
            DO_FREE_LOCALS
            return;
            break;
        }

	/* Do not reference last selected recycled object structure after
	 * this point.
	 */
	obj = NULL;


        EDVRecBinSetBusy(recbin, TRUE);


        /* Iterate through list of recycled object indexes, purging each
         * one.
         */
	status = 0;
        for(i = 0; i < total_indexes; i++)
        {
            /* Recover recycled object. */
	    new_path = NULL;
            status = EDVRecBinFOPRecover(
                core_ptr, index[i], NULL,
		&new_path,
		recbin->toplevel,
		TRUE, TRUE, &yes_to_all
            );

            /* Get error message (if any) that might have occured
             * in the above operation.
             */
            error_mesg = EDVRecBinFOPGetError();
            if(error_mesg != NULL)
            {
                CDialogSetTransientFor(recbin->toplevel);
                CDialogGetResponse(
                    "Operation Error",
                    error_mesg,
                    NULL,
                    CDIALOG_ICON_ERROR,
                    CDIALOG_BTNFLAG_OK,
                    CDIALOG_BTNFLAG_OK
                );
                CDialogSetTransientFor(NULL);
            }


            /* Object was recovered successfully? */
            if(!status)
            {
		struct stat lstat_buf;

		/* Mark this as an object recovered. */
		recbin_objects_recovered++;

		/* Report new disk object added. */
		if((new_path != NULL) ? !lstat(new_path, &lstat_buf) : FALSE)
		    EDVObjectAddedEmit(
			core_ptr, new_path, &lstat_buf
		    );

		/* Report recycled object removed. */
                EDVRecycledObjectRemovedEmit(core_ptr, index[i]);
            }

	    /* Deallocate new path of recovered object. */
	    g_free(new_path);
	    new_path = NULL;

            /* Error occured (not 0) and it was not a user `no' or
             * `not available' (-5) response?
             */
            if(status && (status != -5))
            {
                /* Skip rest of the indexes due to error. */
                break;
            }
        }


        /* Unmap progress dialog, it may have been mapped if any
         * operations occured in the above loop.
         */
        ProgressDialogBreakQuery(TRUE);
        ProgressDialogSetTransientFor(NULL);


        EDVRecBinSetBusy(recbin, FALSE);


        if(1)
        {
            gchar *buf;
            switch(status)
            {
              case 0: case -5:
                buf = g_strdup_printf(
                    "Recovered %i object%s",
                    recbin_objects_recovered,
                    (recbin_objects_recovered == 1) ? "" : "s"
                );
                break;

              case -4:  /* Cancel. */
                buf = g_strdup_printf(
                    "Recover operation canceled"
                );
                break;

              default:  /* Error. */
                buf = g_strdup_printf(
                    "Unable to recover object%s",
                    (total_indexes == 1) ? "" : "s"
                );
                break;
            }
            EDVStatusBarMessage(recbin->status_bar, buf, FALSE);
	    g_free(buf);
        }


	DO_FREE_LOCALS
#undef DO_FREE_LOCALS
}

/*
 *      Purge callback.
 */
void EDVRecBinOPPurge(edv_recbin_struct *recbin)
{
	gbool yes_to_all = FALSE;
	guint *index = NULL;
	gint total_indexes = 0;
	gint recbin_objects_purged = 0;
	gint i, row, status;
	const gchar *error_mesg;
	GList *glist;
	GtkCList *clist;
	edv_recbin_object_struct *obj = NULL;
	edv_core_struct *core_ptr;
	gchar obj_index_str[80];


	if(recbin == NULL)
	    return;

	core_ptr = (edv_core_struct *)recbin->core_ptr;
	if(core_ptr == NULL)
	    return;

	clist = (GtkCList *)recbin->contents_clist;
	if(clist == NULL)
	    return;

        /* Check and warn if write protect is enabled. */
        if(EDVCheckWriteProtect(core_ptr, TRUE))
            return;

#define DO_FREE_LOCALS  \
{ \
 g_free(index); \
 index = NULL; \
 total_indexes = 0; \
}

	/* Get a list of selected recycled objects indexes. */
	glist = clist->selection;
	while(glist != NULL)
	{
	    row = (gint)glist->data;
	    obj = (edv_recbin_object_struct *)gtk_clist_get_row_data(
		clist, row
	    );
	    if(obj != NULL)
	    {
		/* Append this recycled object structure's index to the
		 * list of recycled object indexes to purge.
		 */
		i = total_indexes;
		total_indexes = i + 1;
		index = (guint *)g_realloc(
		    index, total_indexes * sizeof(guint)
		);
		if(index == NULL)
		{
		    total_indexes = 0;
		    break;
		}
		else
		{
		   index[i] = obj->index;
		}
	    }

	    glist = glist->next;
	}

	/* No recycled objects to purge? */
	if(index == NULL)
	{
	    DO_FREE_LOCALS
	    return;
	}


        /* Format recycled object index number string based on the last
         * selected recycled object (if any). This will be used for
         * passing to the confirmation function.
         */
        if(obj != NULL)
            sprintf(obj_index_str, "%i", obj->index);
        else
            *obj_index_str = '\0';

        /* Get purge confirmation. */
        status = EDVConfirmPurge(
            core_ptr, recbin->toplevel,
            obj_index_str,
            total_indexes
        );
        switch(status)
        {
          case CDIALOG_RESPONSE_YES:
          case CDIALOG_RESPONSE_YES_TO_ALL:
            break;

          default:
            DO_FREE_LOCALS
            return;
            break;
        }


        /* Do not reference last selected recycled object structure after
         * this point.
         */
        obj = NULL;


	EDVRecBinSetBusy(recbin, TRUE);

	/* Iterate through list of recycled object indexes, purging each
	 * one.
	 */
	status = 0;
	for(i = 0; i < total_indexes; i++)
	{
	    /* Purge recycled object. */
	    status = EDVRecBinFOPPurge(
		core_ptr, index[i], recbin->toplevel,
                (total_indexes > 0) ?
                    (gfloat)i / (gfloat)total_indexes : -1.0,
		TRUE, TRUE, &yes_to_all
	    );

	    /* Get error message (if any) that might have occured
	     * in the above operation.
	     */
	    error_mesg = EDVRecBinFOPGetError();
	    if(error_mesg != NULL)
	    {
		CDialogSetTransientFor(recbin->toplevel);
		CDialogGetResponse(
		    "Operation Error",
		    error_mesg,
		    NULL,
		    CDIALOG_ICON_ERROR,
		    CDIALOG_BTNFLAG_OK,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
	    }


	    /* Object was purged successfully? */
	    if(!status)
	    {
		recbin_objects_purged++;
		EDVRecycledObjectRemovedEmit(core_ptr, index[i]);
	    }

	    /* Error occured (not 0) and it was not a user `no' or
	     * `not available' (-5) response?
	     */
	    if(status && (status != -5))
	    {
		/* Skip rest of the indexes due to error. */
		break;
	    }
	}


	/* Unmap progress dialog, it may have been mapped if any 
	 * operations occured in the above loop.
	 */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);


	EDVRecBinSetBusy(recbin, FALSE);


	if(1)
	{
	    gchar *buf;
            switch(status)
            {
              case 0: case -5:
                buf = g_strdup_printf(
                    "Purged %i object%s",
                    recbin_objects_purged,
                    (recbin_objects_purged == 1) ? "" : "s"
                );
                break;

              case -4:  /* Cancel. */
                buf = g_strdup_printf(
                    "Purge operation canceled"
                );
                break;

              default:  /* Error. */
                buf = g_strdup_printf(
                    "Unable to purge object%s",
                    (total_indexes == 1) ? "" : "s"
                );
                break;
            }
	    EDVStatusBarMessage(recbin->status_bar, buf, FALSE);
	    g_free(buf);
	}


	DO_FREE_LOCALS
#undef DO_FREE_LOCALS
}

/*
 *	Purge all callback.
 */
void EDVRecBinOPPurgeAll(edv_recbin_struct *recbin)
{
        gbool yes_to_all = FALSE;
        guint *index = NULL;
        gint total_indexes = 0;
        gint recbin_objects_purged = 0;
        gint i, status;
	const gchar *recycled_index_file;
        const gchar *error_mesg;
	edv_recbin_index_struct *rbi_ptr;
        edv_core_struct *core_ptr;


        if(recbin == NULL)
            return;

        core_ptr = (edv_core_struct *)recbin->core_ptr;
        if(core_ptr == NULL)
            return;

        /* Check and warn if write protect is enabled. */
        if(EDVCheckWriteProtect(core_ptr, TRUE))
            return;

        /* Get path to recycled objects index file. */
        recycled_index_file = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_FILE_RECYCLED_INDEX
        );
        if(recycled_index_file == NULL)
            return;

#define DO_FREE_LOCALS  \
{ \
 g_free(index); \
 index = NULL; \
 total_indexes = 0; \
}

	/* Get a list of all recycled object indexes. */
	rbi_ptr = EDVRecBinIndexOpen(recycled_index_file);
        while(!EDVRecBinIndexNext(rbi_ptr))
        {
            if(rbi_ptr->index != 0)
            {
                /* Append this recycled object structure's index to the
                 * list of recycled object indexes to purge.
                 */
                i = total_indexes;
                total_indexes = i + 1;
                index = (guint *)g_realloc(
                    index, total_indexes * sizeof(guint)
                );
                if(index == NULL)
                {
                    total_indexes = 0;
                    break;
                }
                else
                {
                   index[i] = rbi_ptr->index;
                }
            }
        }
	EDVRecBinIndexClose(rbi_ptr);
	rbi_ptr = NULL;

        /* No recycled objects to purge? */
        if(index == NULL)
	{
	    DO_FREE_LOCALS
            return;
	}


        /* Get purge all confirmation. */
        status = EDVConfirmPurge(
            core_ptr, recbin->toplevel,
	    NULL,		/* No object name, force printing of value
				 * of total_indexes.
				 */
            total_indexes
        );
        switch(status)
        {
          case CDIALOG_RESPONSE_YES:
          case CDIALOG_RESPONSE_YES_TO_ALL:
            break;

          default:
            DO_FREE_LOCALS
            return;
            break;
        }


        EDVRecBinSetBusy(recbin, TRUE);

#ifdef EDV_RECBIN_USE_PURGE_ALL
/* Do not define this, this has issues about progress updating,
 * read subsequent comments in this block.
 */
        /* Purge all contents from the recycled objects directory
	 * including the recycled objects index file.
	 */
	status = EDVRecBinFOPPurgeAll(
            core_ptr, recbin->toplevel,
            TRUE, TRUE, &yes_to_all
        );

	/* Get error message (if any) that might have occured in the above
	 * operation.
	 */
	error_mesg = EDVRecBinFOPGetError();
	if(error_mesg != NULL)
	{
	    CDialogSetTransientFor(recbin->toplevel);
	    CDialogGetResponse(
		"Operation Error",
		error_mesg,
		NULL,
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	}

	/* All recycled objects purged successfully? */
	if(!status)
	{
	    recbin_objects_purged = total_indexes;

	    /* Send recycled object purged signal to all of Endeavour's
	     * resources for each recycled object listed in our recorded
	     * recycled objects index array.
	     */
	    for(i = 0; i < total_indexes; i++)
	    {
		/* Update progress bar for notifying of resources as well. */
/* This might appear a bit confusing sinceis not very neat since */
		if(ProgressDialogIsQuery())
		{
		    ProgressDialogUpdate(
			NULL, NULL, NULL, NULL,
			(gfloat)i / (gfloat)total_indexes,
			EDV_DEF_PROGRESS_BAR_TICKS, TRUE
		    );
		    if(ProgressDialogStopCount() > 0)
		    {
			/* If user wants to stop, then end updating of
			 * Endeavour's resources prematurly but do not
			 * change the value of status.
			 */
			break;
		    }
		}
                EDVRecycledObjectRemovedEmit(core_ptr, index[i]);
	    }
        }
#else
        /* Iterate through list of recycled object indexes, purging each
         * one.
         */
	status = 0;
        for(i = 0; i < total_indexes; i++)
        {
            /* Purge recycled object. */
            status = EDVRecBinFOPPurge(
                core_ptr, index[i], recbin->toplevel,
		(total_indexes > 0) ?
		    (gfloat)i / (gfloat)total_indexes : -1.0,
                TRUE, TRUE, &yes_to_all
            );

            /* Get error message (if any) that might have occured
             * in the above operation.
             */
            error_mesg = EDVRecBinFOPGetError();
            if(error_mesg != NULL)
            {
                CDialogSetTransientFor(recbin->toplevel);
                CDialogGetResponse(
                    "Operation Error",
                    error_mesg,
                    NULL,
                    CDIALOG_ICON_ERROR,
                    CDIALOG_BTNFLAG_OK,
                    CDIALOG_BTNFLAG_OK
                );
                CDialogSetTransientFor(NULL);
            }


            /* Object was purged successfully? */
            if(!status)
            {
                recbin_objects_purged++;
                EDVRecycledObjectRemovedEmit(core_ptr, index[i]);
            }

            /* Error occured (not 0) and it was not a user `no' or
             * `not available' (-5) response?
             */
            if(status && (status != -5))
            {
                /* Skip rest of the indexes due to error. */
                break;
            }
        }

	/* If no errors or aborting occured then do one more through
	 * purge all but do not show progress for it. This will also
	 * remove any stray recycled object files and the recycled objects
	 * index file.
	 */
	if(!status)
	    EDVRecBinFOPPurgeAll(
		core_ptr, recbin->toplevel,
		FALSE, TRUE, &yes_to_all
	    );
#endif

        /* Unmap progress dialog, it may have been mapped if any
         * operations occured in the above loop.
         */
        ProgressDialogBreakQuery(TRUE);
        ProgressDialogSetTransientFor(NULL);


        EDVRecBinSetBusy(recbin, FALSE);


        if(1)
        {
            gchar *buf;
            switch(status)
            {
              case 0: case -5:
                buf = g_strdup_printf(
                    "Purged %i object%s",
                    recbin_objects_purged,
                    (recbin_objects_purged == 1) ? "" : "s"
                );
                break;

              case -4:  /* Cancel. */
                buf = g_strdup_printf(
                    "Purge operation canceled"
                );
                break;

              default:  /* Error. */
                buf = g_strdup_printf(
                    "Unable to purge object%s",
                    (total_indexes == 1) ? "" : "s"
                );
                break;
            }
            EDVStatusBarMessage(recbin->status_bar, buf, FALSE);
	    g_free(buf);
        }


	DO_FREE_LOCALS
#undef DO_FREE_LOCALS
}

/*
 *	Select all.
 */
void EDVRecBinOPSelectAll(edv_recbin_struct *recbin)
{
        edv_core_struct *core_ptr;
        GtkCList *clist;


        if(recbin == NULL)
            return;

        core_ptr = (edv_core_struct *)recbin->core_ptr;
        if(core_ptr == NULL)
            return;

        /* Get contents clist. */
        clist = (GtkCList *)recbin->contents_clist;
        if(clist == NULL)
            return;

        EDVRecBinSetBusy(recbin, TRUE);

        /* Select all rows on clist. */
        gtk_clist_freeze(clist);
        gtk_clist_select_all(clist);
        gtk_clist_thaw(clist);

        /* Assume highest row index as the last selected row. */
        recbin->contents_clist_selected_row = clist->rows - 1;

        EDVStatusBarMessage(
            recbin->status_bar, "All objects selected", FALSE
        );

        EDVRecBinSetBusy(recbin, FALSE);
}

/*
 *	Unselect all.
 */
void EDVRecBinOPUnselectAll(edv_recbin_struct *recbin)
{
        edv_core_struct *core_ptr;
        GtkCList *clist;


        if(recbin == NULL)
            return;

        core_ptr = (edv_core_struct *)recbin->core_ptr;
        if(core_ptr == NULL)
            return;

        /* Get contents clist. */
        clist = (GtkCList *)recbin->contents_clist;
        if(clist == NULL)
            return;

        EDVRecBinSetBusy(recbin, TRUE);

        /* Unselect all rows on clist. */
        gtk_clist_freeze(clist);
        gtk_clist_unselect_all(clist);
        gtk_clist_thaw(clist);

        /* Mark contents clist's row as unselected. */
	recbin->contents_clist_selected_row = -1;

        EDVStatusBarMessage(
            recbin->status_bar, "All objects unselected", FALSE
        );

        EDVRecBinSetBusy(recbin, FALSE);
}

/*
 *      Invert selection callback.
 */
void EDVRecBinOPInvertSelection(edv_recbin_struct *recbin)
{
        edv_core_struct *core_ptr;
        GtkCList *clist;
        GList *glist, *selection;
        gint row, total_rows;


        if(recbin == NULL)
            return;

        core_ptr = (edv_core_struct *)recbin->core_ptr;
        if(core_ptr == NULL)
            return;

        /* Get contents clist. */
        clist = (GtkCList *)recbin->contents_clist;
        if(clist == NULL)
            return;

        EDVRecBinSetBusy(recbin, TRUE);
        gtk_clist_freeze(clist);

        /* Get copy of selected rows list from clist. */
        selection = (clist->selection != NULL) ?
            g_list_copy(clist->selection) : NULL;

        for(row = 0, total_rows = clist->rows;
            row < total_rows;
            row++
        )
        {
            for(glist = selection;
                glist != NULL;
               glist = glist->next
            )
            {
                if(row == (gint)glist->data)
                {
                    gtk_clist_unselect_row(clist, row, 0);
                    break;
                }
            }
            /* Row not selected? */
            if(glist == NULL)
                gtk_clist_select_row(clist, row, 0);
        }

        g_list_free(selection);

        gtk_clist_thaw(clist);
        EDVStatusBarMessage(
            recbin->status_bar, "Selection inverted", FALSE
        );
        EDVRecBinSetBusy(recbin, FALSE);
}

/*
 *	Refresh callback.
 */
void EDVRecBinOPRefresh(edv_recbin_struct *recbin)
{
        GtkCList *clist;


        if(recbin == NULL)
            return;

        EDVRecBinSetBusy(recbin, TRUE);
        GUIBlockInput(recbin->toplevel, TRUE);

        /* Update contents clist. */
        clist = (GtkCList *)recbin->contents_clist;
        if(clist != NULL)
        {
            gfloat last_vpos, last_hpos;


            /* Record last scroll position. */
            last_hpos = (clist->hadjustment != NULL) ?
                clist->hadjustment->value : 0.0;
            last_vpos = (clist->vadjustment != NULL) ?
                clist->vadjustment->value : 0.0;

            gtk_clist_freeze(clist);
            EDVRecBinContentsDoUpdate(recbin, TRUE);
            gtk_clist_thaw(clist);

            /* Scroll back to original position. */
            EDVScrollCListToPosition(
                clist, last_hpos, last_vpos
            );
        }

        GUIBlockInput(recbin->toplevel, FALSE);
        EDVRecBinSetBusy(recbin, FALSE);


        EDVRecBinUpdateMenus(recbin);

        EDVStatusBarMessage(
            recbin->status_bar, "Refreshed contents listing", FALSE
        );
}

/*
 *      Refresh all callback.
 */
void EDVRecBinOPRefreshAll(edv_recbin_struct *recbin)
{
	/* Refresh all is the same as a regular refresh for the recycle
	 * bin lists.
	 */
	EDVRecBinOPRefresh(recbin);
}
