#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <gdk/gdkrgb.h>

#include "guiutils.h"

#include "tlist.h"


static void TListDrawCB(
	GtkWidget *widget, GdkRectangle *area, gpointer data
);
static void TListDrawFocusCB(GtkWidget *widget, gpointer data);
static void TListDrawDefaultCB(GtkWidget *widget, gpointer data);
static gint TListButtonEventCB(
	GtkWidget *widget, GdkEventButton *button, gpointer data
);
static gint TListMotionEventCB(
        GtkWidget *widget, GdkEventMotion *motion, gpointer data
);
static gint TListKeyEventCB(
        GtkWidget *widget, GdkEventKey *key, gpointer data
);
static gint TListCrossingEventCB(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);
static gint TListVisibilityEventCB(
	GtkWidget *widget, GdkEventVisibility *visibility, gpointer data
);
static gint TListExposeEventCB(
        GtkWidget *widget, GdkEventExpose *expose, gpointer data
);
static gint TListConfigureEventCB(
        GtkWidget *widget, GdkEventConfigure *configure, gpointer data
);
static gint TListFocusInEventCB(
        GtkWidget *widget, GdkEventFocus *focus, gpointer data
);
static gint TListFocusOutEventCB(
        GtkWidget *widget, GdkEventFocus *focus, gpointer data
);
static void TListAdjustmentValueChangedCB(
	GtkAdjustment *adjustment, gpointer data
);

static tlist_thumb_struct *TListGetThumbPtr(
	tlist_struct *tlist, gint thumb_num
);
static gbool TListIsThumbSelected(
	tlist_struct *tlist, gint thumb_num
);

static tlist_thumb_struct *TListThumbNew(void);
static void TListThumbDelete(tlist_thumb_struct *thumb_ptr);

static void TListDoSelectThumb(
        tlist_struct *tlist, gint thumb_num, GdkEventButton *button,
        gbool append
);
static void TListDoSelectThumbRange(
	tlist_struct *tlist, gint thumb_num, GdkEventButton *button
);
static void TListDoUnselectThumb(
        tlist_struct *tlist, gint thumb_num, GdkEventButton *button
);
static void TListDoSelectAllThumbs(tlist_struct *tlist);
static void TListDoUnselectAllThumbs(tlist_struct *tlist);

void TListDraw(tlist_struct *tlist);
void TListResize(
        tlist_struct *tlist, gint width, gint height,
	gbool allow_gtk_main_iteration
);

void TListFreeze(tlist_struct *tlist);
void TListThaw(tlist_struct *tlist);

gint TListAppend(
	tlist_struct *tlist, const gchar *text,
	gpointer client_data,
        void (*destroy_cb)(gpointer)
);
void TListSetLoadState(
	tlist_struct *tlist, gint thumb_num,
	gint load_state
);
void TListSetText(
        tlist_struct *tlist, gint thumb_num,
        const gchar *text
);
void TListSetPixmap(
        tlist_struct *tlist, gint thumb_num,
        GdkPixmap *pixmap, GdkBitmap *mask
);
void TListSetRGBA(
	tlist_struct *tlist, gint thumb_num,
	gint width, gint height,	/* Of image data. */
        gint bpl,                       /* Bytes per line, can be -1. */
	GdkRgbDither dith,
	const guint8 *rgba_buf,
        gbool no_enlarge                /* Do not upscale if image smaller
                                         * than thumb.
                                         */
);
void TListSetDestroy(
        tlist_struct *tlist, gint thumb_num,
        gpointer client_data,
        void (*destroy_cb)(gpointer)
);
void TListRemove(tlist_struct *tlist, gint thumb_num);
void TListClear(tlist_struct *tlist);

gpointer TListGetThumbData(tlist_struct *tlist, gint thumb_num);

void TListSelectThumb(tlist_struct *tlist, gint thumb_num);
void TListUnselectThumb(tlist_struct *tlist, gint thumb_num);
void TListSelectAll(tlist_struct *tlist);
void TListUnselectAll(tlist_struct *tlist);

gbool TListGetSelection(
        tlist_struct *tlist, gint x, gint y,
	gint *thumb_num, gint *thumb_ix, gint *thumb_iy
);
gbool TListGetPosition(
        tlist_struct *tlist, gint thumb_num,
	gint *x, gint *y
);

gint TListIsThumbVisible(
	tlist_struct *tlist, gint thumb_num
);
void TListMoveTo(
        tlist_struct *tlist, gint thumb_num, gfloat coeff
);

tlist_struct *TListNew(
	gbool horizontal,
	gint thumb_width, gint thumb_height, gint thumb_border,
	gpointer client_data,
	void (*select_cb)(gpointer, GdkEventButton *, gint, gpointer),
        void (*unselect_cb)(gpointer, GdkEventButton *, gint, gpointer)
);
void TListSelectionMode(tlist_struct *tlist, gint selection_mode);
void TListDoubleBuffer(tlist_struct *tlist, gbool double_buffer);
void TListOrientation(tlist_struct *tlist, gbool horizontal); 
void TListMap(tlist_struct *tlist);
void TListUnmap(tlist_struct *tlist);
void TListDelete(tlist_struct *tlist);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))

/* Size of the borders of the drawn GTK frames. */
#define DEF_FRAME_WIDTH		2
#define DEF_FRAME_HEIGHT	2


/*
 *	Drawing area "draw" signal callback.
 */
static void TListDrawCB(GtkWidget *widget, GdkRectangle *area, gpointer data)
{
	TListDraw((tlist_struct *)data);
}

/*
 *	Drawing area "draw_focus" signal callback.
 */
static void TListDrawFocusCB(GtkWidget *widget, gpointer data)
{
        TListDraw((tlist_struct *)data);
}

/* 
 *	Drawing area "draw_default" signal callback.
 */
static void TListDrawDefaultCB(GtkWidget *widget, gpointer data)
{
        TListDraw((tlist_struct *)data);
}

/*
 *	Drawing area "button_press_event" or "button_release_event" signal
 *	callback.
 */
static gint TListButtonEventCB(
        GtkWidget *widget, GdkEventButton *button, gpointer data
)
{
	gint status = FALSE;
	gint etype;
	tlist_struct *tlist = TLIST(data);
	if((button == NULL) || (tlist == NULL))
	    return(status);

	/* Get event type. */
	etype = button->type;

	/* Is the given widget the drawing area widget? */
	if(widget == tlist->list_da)
	{
	    gint thumb_num;


	    switch(etype)
	    {
              case GDK_BUTTON_PRESS:
		switch(button->button)
		{
		  case 1:
		    if(TListGetSelection(
			tlist, button->x, button->y,
			&thumb_num, NULL, NULL
		    ))
		    {
			TListFreeze(tlist);

			tlist->focus_thumb = thumb_num;

			/* Control key held? */
			if(button->state & GDK_CONTROL_MASK)
			{
			    switch(tlist->selection_mode)
			    {
			      case GTK_SELECTION_EXTENDED:
				if(TListIsThumbSelected(tlist, thumb_num))
                                    TListDoUnselectThumb(
                                        tlist, thumb_num, button
                                    );
                                else
                                    TListDoSelectThumb(
                                        tlist, thumb_num, button, TRUE
                                    );
				break;

			      case GTK_SELECTION_BROWSE:
				TListDoSelectThumb(
				    tlist, thumb_num, button, FALSE
				);
				break;

			      default:	/* GTK_SELECTION_SINGLE */
				if(TListIsThumbSelected(tlist, thumb_num))
                                {
                                    TListDoUnselectAllThumbs(tlist);
				}
				else
				{
				    TListDoUnselectAllThumbs(tlist);
				    TListDoSelectThumb(
				        tlist, thumb_num, button, TRUE
				    );
				}
				break;
			    }
			}
			/* Shift key held? */
			else if(button->state & GDK_SHIFT_MASK)
			{
                            switch(tlist->selection_mode)
                            {
                              case GTK_SELECTION_EXTENDED:
				TListDoSelectThumbRange(
				    tlist, thumb_num, button
				);
				break;

                              default:
                                if(TListIsThumbSelected(tlist, thumb_num))
                                {
                                    TListDoUnselectAllThumbs(tlist);
                                }
                                else
                                {
                                    TListDoUnselectAllThumbs(tlist);
                                    TListDoSelectThumb(
                                        tlist, thumb_num, button, TRUE
                                    );
                                }
                                break;
			    }
			}
			/* No important modifier keys held? */
			else
			{
			    /* If there was exactly one thumb selected
			     * and it matches thumb_num then do not
			     * do anything.
			     */
			    GList *selection_base =
			        (tlist->selection != NULL) ?
				    g_list_copy(tlist->selection) : NULL;
			    GList *selection = selection_base;


			    /* Iterate through coppied selections list and
			     * unselect all thumbs (except for thumb_num).
			     */
			    while(selection != NULL)
			    {
				if((gint)selection->data != thumb_num)
				    TListDoUnselectThumb(
					tlist, (gint)selection->data, NULL
				    );
				selection = selection->next;
			    }

			    TListDoSelectThumb(
				tlist, thumb_num, button, TRUE
			    );

			    /* Deallocate coppied selections list. */
			    g_list_free(selection_base);
			}
			TListThaw(tlist);
		    }
		    break;
		}

		gtk_widget_grab_focus(widget);

		break;
	    }
	}

	return(status);
}

/*
 *      Drawing area "motion_notify_event" signal callback.
 */
static gint TListMotionEventCB(
        GtkWidget *widget, GdkEventMotion *motion, gpointer data
)
{
        gint status = FALSE;
        gint etype;
        tlist_struct *tlist = TLIST(data);
        if((motion == NULL) || (tlist == NULL))
            return(status);

        /* Get event type. */
        etype = motion->type;

        /* Is the given widget the drawing area widget? */
        if(widget == tlist->list_da)
        {
	    gint thumb_num;
	    tlist_thumb_struct *thumb_ptr;

	    /* Match thumb index that the pointer has moved over. */
	    if(!TListGetSelection(
		tlist, (gint)motion->x, (gint)motion->y,
		&thumb_num, NULL, NULL
	    ))
		thumb_num = -1;

	    if((thumb_num >= 0) && (thumb_num < tlist->total_thumbs))
		thumb_ptr = tlist->thumb[thumb_num];
	    else
		thumb_ptr = NULL;

	    /* Matched a thumb that the pointer has moved over? */
	    if(thumb_ptr != NULL)
	    {
		/* Got a match, now check if the thumb that the pointer
		 * has moved over is different from the previous
		 * thumb that the pointer used to be over (including
		 * if the pointer was never over a thumb).
		 */
		if(tlist->pointer_over_thumb != thumb_num)
		{
		    /* Update the thumb that the pointer is currently
		     * over.
		     */
		    tlist->pointer_over_thumb = thumb_num;

		    /* Update tip. */
/*
		    GUISetWidgetTip(widget, thumb_ptr->text);
		    GUIShowTipsNow(widget);
 */
		}
	    }
	    else
	    {
		/* No match, check if the pointer was previously over
		 * a thumb.
		 */
		if(tlist->pointer_over_thumb > -1)
		{
/*
		    GUISetWidgetTip(widget, NULL);
 */
		}
	    }
        }

        return(status);
}

/*
 *	Drawing area "key_press_event" or "key_release_event" signal
 *	callback.
 */
static gint TListKeyEventCB(
        GtkWidget *widget, GdkEventKey *key, gpointer data
)
{
        gint status = FALSE;
        gint etype;
        guint keyval, state;
        gbool press;
        tlist_struct *tlist = TLIST(data);
        if((key == NULL) || (tlist == NULL))
            return(status);

        /* Get event type. */
        etype = key->type;

        /* Get other key event values. */
        press = (etype == GDK_KEY_PRESS) ? TRUE : FALSE;
        keyval = key->keyval;
        state = key->state;

/* Macro to emit a signal stop for a key press or release depending
 * on the current event's type.
 */
#define DO_STOP_KEY_SIGNAL_EMIT \
{ \
 gtk_signal_emit_stop_by_name( \
  GTK_OBJECT(widget), \
  press ? "key_press_event" : "key_release_event" \
 ); \
}

/* Macro to clamp the GtkAdjustment adj and emit a "value_changed"
 * signal.
 */
#define DO_ADJ_CLAMP_EMIT       \
{ \
 if(adj->value > (adj->upper - adj->page_size)) \
  adj->value = adj->upper - adj->page_size; \
\
 if(adj->value < adj->lower) \
  adj->value = adj->lower; \
\
 gtk_signal_emit_by_name( \
  GTK_OBJECT(adj), "value_changed" \
 ); \
}

/* Macro to clamp the GtkCList clist's focus_row and queue a redraw
 * for the clist.
 */
#define DO_TLIST_FOCUS_THUMB_CLAMP_DRAW   \
{ \
 if(tlist->focus_thumb >= tlist->total_thumbs) \
  tlist->focus_thumb = tlist->total_thumbs - 1; \
 if(tlist->focus_thumb < 0) \
  tlist->focus_thumb = 0; \
\
 TListDraw(tlist); \
}

        /* Is the given widget the drawing area widget? */
        if(widget == tlist->list_da)
        {
            /* Handle by key value. */
            switch(keyval)
            {
              case GDK_space:
		if(press)
		{
		    gint thumb_num = tlist->focus_thumb;


                    TListFreeze(tlist);

		    switch(tlist->selection_mode)
		    {
		      case GTK_SELECTION_EXTENDED:
                        if(TListIsThumbSelected(tlist, thumb_num))
                            TListDoUnselectThumb(
                                tlist, thumb_num, NULL
                            );
                        else
                            TListDoSelectThumb(
                                tlist, thumb_num, NULL, TRUE
                            );
                        break;

                      case GTK_SELECTION_BROWSE:
                        TListDoSelectThumb(
                            tlist, thumb_num, NULL, FALSE
                        );
                        break;

                      default:  /* GTK_SELECTION_SINGLE */
                        if(TListIsThumbSelected(tlist, thumb_num))
                        {
                            TListDoUnselectAllThumbs(tlist);
                        }
                        else
                        {
                            TListDoUnselectAllThumbs(tlist);
                            TListDoSelectThumb(
                                tlist, thumb_num, NULL, TRUE
                            );
                        }
                        break;
		    }

                    TListThaw(tlist);
		}
		break;


              case GDK_Up:
              case GDK_KP_Up:
                if(state & GDK_CONTROL_MASK)
                {
                    /* Get adjustment and scroll up. */
                    GtkAdjustment *adj = tlist->vadj;
                    if((adj != NULL) && press)
                    {
                        adj->value -= adj->step_increment;
                        DO_ADJ_CLAMP_EMIT
                    }
                }
                else if((state & GDK_SHIFT_MASK) && press)
                {
                    TListFreeze(tlist);

                    /* Move focus thumb up. */
                    tlist->focus_thumb -= tlist->horizontal ?
			1 : tlist->thumbs_per_line;
                    DO_TLIST_FOCUS_THUMB_CLAMP_DRAW

                    switch(tlist->selection_mode)
                    {
                      case GTK_SELECTION_EXTENDED:
                        TListDoSelectThumbRange(
                            tlist, tlist->focus_thumb, NULL
                        );
                        break;

                      default:
                        TListDoUnselectAllThumbs(tlist);
                        TListDoSelectThumb(
                            tlist, tlist->focus_thumb, NULL, TRUE
                        );
                        break;
                    }

                    if(TListIsThumbVisible(tlist, tlist->focus_thumb) !=
                        GTK_VISIBILITY_FULL
                    )
                        TListMoveTo(tlist, tlist->focus_thumb, 0.0);

                    TListThaw(tlist);
                }
                else if(press)
                {
                    TListFreeze(tlist);

                    /* Move focus thumb up. */
                    tlist->focus_thumb -= tlist->horizontal ?
                        1 : tlist->thumbs_per_line;
                    DO_TLIST_FOCUS_THUMB_CLAMP_DRAW

                    if(TListIsThumbVisible(tlist, tlist->focus_thumb) !=
                        GTK_VISIBILITY_FULL
                    )
                        TListMoveTo(tlist, tlist->focus_thumb, 0.0);

                    TListThaw(tlist);
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              case GDK_Down:
              case GDK_KP_Down:
                if(state & GDK_CONTROL_MASK)
                {
                    /* Get adjustment and scroll down. */
                    GtkAdjustment *adj = tlist->vadj;
                    if((adj != NULL) && press)
                    {
                        adj->value += adj->step_increment;
                        DO_ADJ_CLAMP_EMIT
                    }
                }
                else if((state & GDK_SHIFT_MASK) && press)
                {
                    TListFreeze(tlist);

                    /* Move focus thumb down. */
                    tlist->focus_thumb += tlist->horizontal ?
                        1 : tlist->thumbs_per_line;
                    DO_TLIST_FOCUS_THUMB_CLAMP_DRAW

                    switch(tlist->selection_mode)
                    {
                      case GTK_SELECTION_EXTENDED:
                        TListDoSelectThumbRange(
                            tlist, tlist->focus_thumb, NULL
                        );
                        break;

                      default:
                        TListDoUnselectAllThumbs(tlist);
                        TListDoSelectThumb(
                            tlist, tlist->focus_thumb, NULL, TRUE
                        );
                        break;
                    }

                    if(TListIsThumbVisible(tlist, tlist->focus_thumb) !=
                        GTK_VISIBILITY_FULL
                    )
                        TListMoveTo(tlist, tlist->focus_thumb, 1.0);

                    TListThaw(tlist);
                }
                else if(press)
                {
                    TListFreeze(tlist);

                    /* Move focus thumb down. */
                    tlist->focus_thumb += tlist->horizontal ?
                        1 : tlist->thumbs_per_line;
                    DO_TLIST_FOCUS_THUMB_CLAMP_DRAW

                    if(TListIsThumbVisible(tlist, tlist->focus_thumb) !=
                        GTK_VISIBILITY_FULL
                    )
                        TListMoveTo(tlist, tlist->focus_thumb, 1.0);

                    TListThaw(tlist);
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              case GDK_Left:
              case GDK_KP_Left:
                if(state & GDK_CONTROL_MASK)
                {
                    /* Get adjustment and scroll left. */
                    GtkAdjustment *adj = tlist->hadj;
                    if((adj != NULL) && press)
                    {
                        adj->value -= adj->step_increment;
                        DO_ADJ_CLAMP_EMIT
                    }
                }
                else if((state & GDK_SHIFT_MASK) && press)
                {
                    TListFreeze(tlist);

                    /* Move focus thumb left. */
                    tlist->focus_thumb -= tlist->horizontal ?
                        tlist->thumbs_per_line : 1;
                    DO_TLIST_FOCUS_THUMB_CLAMP_DRAW

                    switch(tlist->selection_mode)
                    {
                      case GTK_SELECTION_EXTENDED:
                        TListDoSelectThumbRange(
                            tlist, tlist->focus_thumb, NULL
                        );
                        break;

                      default:
                        TListDoUnselectAllThumbs(tlist);
                        TListDoSelectThumb(
                            tlist, tlist->focus_thumb, NULL, TRUE
                        );
                        break;
                    }

                    if(TListIsThumbVisible(tlist, tlist->focus_thumb) !=
                        GTK_VISIBILITY_FULL
                    )
                        TListMoveTo(tlist, tlist->focus_thumb, 0.0);

                    TListThaw(tlist);
                }
                else if(press)
                {
                    TListFreeze(tlist);

                    /* Move focus thumb left. */
                    tlist->focus_thumb -= tlist->horizontal ?
                        tlist->thumbs_per_line : 1;
                    DO_TLIST_FOCUS_THUMB_CLAMP_DRAW

                    if(TListIsThumbVisible(tlist, tlist->focus_thumb) !=
                        GTK_VISIBILITY_FULL
                    )
                        TListMoveTo(tlist, tlist->focus_thumb, 0.0);

                    TListThaw(tlist);
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              case GDK_Right:
              case GDK_KP_Right:
                if(state & GDK_CONTROL_MASK)
                {
                    /* Get adjustment and scroll right. */
                    GtkAdjustment *adj = tlist->hadj;
                    if((adj != NULL) && press)
                    {
                        adj->value += adj->step_increment;
                        DO_ADJ_CLAMP_EMIT
                    }
                }
                else if((state & GDK_SHIFT_MASK) && press)
                {
                    TListFreeze(tlist);

                    /* Move focus thumb right. */
                    tlist->focus_thumb += tlist->horizontal ?
                        tlist->thumbs_per_line : 1;
                    DO_TLIST_FOCUS_THUMB_CLAMP_DRAW

                    switch(tlist->selection_mode)
                    {
                      case GTK_SELECTION_EXTENDED:
                        TListDoSelectThumbRange(
                            tlist, tlist->focus_thumb, NULL
                        );
                        break;

                      default:
                        TListDoUnselectAllThumbs(tlist);
                        TListDoSelectThumb(
                            tlist, tlist->focus_thumb, NULL, TRUE
                        );
                        break;
                    }

                    if(TListIsThumbVisible(tlist, tlist->focus_thumb) !=
                        GTK_VISIBILITY_FULL
                    )
                        TListMoveTo(tlist, tlist->focus_thumb, 1.0);

                    TListThaw(tlist);
                }
                else if(press)
                {
                    TListFreeze(tlist);

                    /* Move focus thumb right. */
                    tlist->focus_thumb += tlist->horizontal ?
                        tlist->thumbs_per_line : 1;
                    DO_TLIST_FOCUS_THUMB_CLAMP_DRAW

                    if(TListIsThumbVisible(tlist, tlist->focus_thumb) !=
                        GTK_VISIBILITY_FULL
                    )
                        TListMoveTo(tlist, tlist->focus_thumb, 1.0);

                    TListThaw(tlist);
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              case GDK_Page_Up:
              case GDK_KP_Page_Up:
                if(1)
                {
                    /* Get adjustment and scroll up one page. */
                    GtkAdjustment *adj = tlist->horizontal ?
			tlist->hadj : tlist->vadj;
                    if((adj != NULL) && press)
                    {
                        adj->value -= adj->page_increment;
                        DO_ADJ_CLAMP_EMIT
                    }
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              case GDK_Page_Down:
              case GDK_KP_Page_Down:
                if(1)
                {
                    /* Get adjustment and scroll down one page. */
                    GtkAdjustment *adj = tlist->horizontal ?
                        tlist->hadj : tlist->vadj;
                    if((adj != NULL) && press)
                    {
                        adj->value += adj->page_increment;
                        DO_ADJ_CLAMP_EMIT
                    }
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              case GDK_Home:
              case GDK_KP_Home:
                if(1)
                {
                    /* Get adjustment and scroll all the way up. */
                    GtkAdjustment *adj = tlist->horizontal ?
                        tlist->hadj : tlist->vadj;
                    if((adj != NULL) && press)
                    {
                        adj->value = adj->lower;
                        DO_ADJ_CLAMP_EMIT
                    }
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              case GDK_End:
              case GDK_KP_End:
                if(1)
                {
                    /* Get adjustment and scroll all the way down. */
                    GtkAdjustment *adj = tlist->horizontal ?
                        tlist->hadj : tlist->vadj;
                    if((adj != NULL) && press)
                    {
                        adj->value = adj->upper - adj->page_size;
                        DO_ADJ_CLAMP_EMIT
                    }
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

	    }
        }

#undef DO_TLIST_FOCUS_THUMB_CLAMP_DRAW
#undef DO_ADJ_CLAMP_EMIT
#undef DO_STOP_KEY_SIGNAL_EMIT

        return(status);
}

/*
 *      Drawing area "enter_notify_event" or "leave_notify_event" signal
 *	callback.
 */
static gint TListCrossingEventCB(
        GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	gint etype;
	tlist_struct *tlist = TLIST(data);
	if((crossing == NULL) || (data == NULL))
	    return(FALSE);

	/* Get event type. */
	etype = crossing->type;

        /* Is the given widget the drawing area widget? */
        if(widget == tlist->list_da)
        {
	    switch(etype)
	    {
	      case GDK_ENTER_NOTIFY:
		break;

	      case GDK_LEAVE_NOTIFY:
		/* Hide thumb tip. */
/*
                GUISetWidgetTip(widget, NULL);
 */
		/* Mark that the pointer is not over any thumb. */
		tlist->pointer_over_thumb = -1;
		break;
	    }
	}

	return(TRUE);
}

/*
 *	Drawing area "visibility_notify_event" signal callback.
 */
static gint TListVisibilityEventCB(
        GtkWidget *widget, GdkEventVisibility *visibility, gpointer data
)
{
	tlist_struct *tlist = TLIST(data);
	if((visibility == NULL) || (tlist == NULL))
	    return(FALSE);

	tlist->visibility = visibility->state;

	/* Check if the drawing area is now fully obscured, suggesting
	 * that we no longer need the pixmap buffer.
	 *
	 * The pixmap buffer will be recreated on resize or the next
	 * draw.
	 */
	if(visibility->state == GDK_VISIBILITY_FULLY_OBSCURED)
	{
	    if(tlist->list_pm != NULL)
	    {
		gdk_pixmap_unref(tlist->list_pm);
		tlist->list_pm = NULL;
	    }
        }

	return(TRUE);
}

/*
 *	Drawing area "expose_event" signal callback.
 */
static gint TListExposeEventCB(
        GtkWidget *widget, GdkEventExpose *expose, gpointer data
)
{
	TListDraw((tlist_struct *)data);
	return(TRUE);
}

/*
 *      Drawing area "configure_event" signal callback.
 */
static gint TListConfigureEventCB(
        GtkWidget *widget, GdkEventConfigure *configure, gpointer data
)
{
	if(configure != NULL)
	    TListResize(
		(tlist_struct *)data, configure->width, configure->height,
		TRUE
	    );
        return(FALSE);
}

/*
 *	Drawing area "focus_in_event" signal callback.
 */
static gint TListFocusInEventCB(
        GtkWidget *widget, GdkEventFocus *focus, gpointer data
)
{
	if(widget != NULL)
	    GTK_WIDGET_SET_FLAGS(widget, GTK_HAS_FOCUS);
	TListDraw((tlist_struct *)data);
	return(TRUE);
}

/*
 *	Drawing area "focus_out_event" signal callback.
 */
static gint TListFocusOutEventCB(
        GtkWidget *widget, GdkEventFocus *focus, gpointer data
)
{
        if(widget != NULL)
            GTK_WIDGET_UNSET_FLAGS(widget, GTK_HAS_FOCUS);
        TListDraw((tlist_struct *)data);
        return(TRUE);
}

/*
 *	Horizontal or vertical scroll bar GtkAdjustment "value_changed"
 *	signal callback.
 */
static void TListAdjustmentValueChangedCB(
        GtkAdjustment *adjustment, gpointer data
)
{
	TListDraw((tlist_struct *)data);
}

/*
 *	Returns the pointer to the thumb structure specified by thumb_num
 *	on the given thumbs list. Can return NULL on error.
 */
static tlist_thumb_struct *TListGetThumbPtr(
        tlist_struct *tlist, gint thumb_num
)
{
	if(tlist == NULL)
	    return(NULL);

	if((thumb_num < 0) || (thumb_num >= tlist->total_thumbs))
	    return(NULL);
	else
	    return(tlist->thumb[thumb_num]);
}

/*
 *	Returns TRUE if the given thumb is found in the selection list.
 */
static gbool TListIsThumbSelected(
        tlist_struct *tlist, gint thumb_num
)
{
	GList *glist;


	if(tlist == NULL)
	    return(FALSE);

	glist = tlist->selection;
	while(glist != NULL)
	{
	    if((gint)glist->data == thumb_num)
		return(TRUE);

	    glist = glist->next;
	}

	return(FALSE);
}


/*
 *	Allocates a new thumb structure.
 */
static tlist_thumb_struct *TListThumbNew(void)
{
	tlist_thumb_struct *thumb_ptr = TLIST_THUMB(g_malloc0(
	    sizeof(tlist_thumb_struct)
	));
	if(thumb_ptr == NULL)
	    return(thumb_ptr);

        thumb_ptr->load_state = TLIST_LOAD_STATE_NOT_LOADED;
	thumb_ptr->text = NULL;
	thumb_ptr->pixmap = NULL;
	thumb_ptr->mask = NULL;
	thumb_ptr->client_data = NULL;
	thumb_ptr->destroy_cb = NULL;

	return(thumb_ptr);
}

/*
 *	Deallocates the thumb structure and all its resources.
 *
 *	Does not call the thumb's destroy notify function.
 */
static void TListThumbDelete(tlist_thumb_struct *thumb_ptr)
{
	if(thumb_ptr == NULL)
	    return;

        g_free(thumb_ptr->text);

        if(thumb_ptr->pixmap != NULL)
            gdk_pixmap_unref(thumb_ptr->pixmap);

        if(thumb_ptr->mask != NULL)
            gdk_bitmap_unref(thumb_ptr->mask);

        g_free(thumb_ptr);
}


/*
 *	Procedure to add the given thumb to the list of selected
 *	thumbs on the thumbs list. If append is FALSE then all previously
 *	selected thumbs will be unselected.
 */
static void TListDoSelectThumb(
	tlist_struct *tlist, gint thumb_num, GdkEventButton *button,
	gbool append
)
{
        if(tlist == NULL)
            return;

	if((thumb_num < 0) || (thumb_num >= tlist->total_thumbs))
	    return;

        /* Is given thumb already selected? */
        if(g_list_find(tlist->selection, (gpointer)thumb_num) != NULL)
            return;

	/* If not appending a selection then all selected thumbs must be
	 * unselected first.
	 */
	if(!append)
	{
	    /* Report unselect for all selected thumbs? */
	    if(tlist->unselect_cb != NULL)
	    {
		GList *glist = tlist->selection_end;
		while(glist != NULL)
		{
		    tlist->unselect_cb(
			(gpointer)tlist,
			NULL,		/* Do not pass button event. */
			(gint)glist->data,
			tlist->client_data
		    );
		    glist = glist->prev;
		}
	    }

	    /* Deallocate entire selection. */
	    if(tlist->selection != NULL)
		g_list_free(tlist->selection);
	    tlist->selection = NULL;
	    tlist->selection_end = NULL;
	}


	/* Append new selected thumb. */
	tlist->selection = g_list_append(
	    tlist->selection, (gpointer)thumb_num
	);

        /* Update end selection pointer since main selection changed. */
        tlist->selection_end = (tlist->selection != NULL) ?
            g_list_last(tlist->selection) : NULL;

        /* Report select as needed. */
        if(tlist->select_cb != NULL)
            tlist->select_cb(
                (gpointer)tlist,
                button,
                thumb_num,
                tlist->client_data
            );

        TListDraw(tlist);
}

/*
 *	Similar to TListDoSelectThumb() except that it selects all
 *	thumbs in sequential indexes from the last selected thumb
 *	to the given thumb_num.
 */
static void TListDoSelectThumbRange(
        tlist_struct *tlist, gint thumb_num, GdkEventButton *button
)
{
	gint i, last_thumb_num;
	GList *glist;


        if(tlist == NULL)
            return;

	if((thumb_num < 0) || (thumb_num >= tlist->total_thumbs))
	    return;

	/* Get last selected thumb if any. */
	glist = tlist->selection_end;
	last_thumb_num = (glist != NULL) ? (gint)glist->data : -1;
	if((last_thumb_num < 0) || (last_thumb_num >= tlist->total_thumbs))
	{
	    /* No previously selected thumb, so instead call
	     * TListDoSelectThumb().
	     */
	    TListDoSelectThumb(tlist, thumb_num, button, FALSE);
	    return;
	}

	/* Now select all thumbs from last_thumb_num to thumb_num. */
	if(last_thumb_num < thumb_num)
	{
	    /* Select low to high. */
	    for(i = last_thumb_num + 1; i <= thumb_num; i++)
		TListDoSelectThumb(tlist, i, button, TRUE);
	}
	else if(last_thumb_num > thumb_num)
	{
	    /* Select high to low. */
            for(i = last_thumb_num - 1; i >= thumb_num; i--)
                TListDoSelectThumb(tlist, i, button, TRUE);
	}
}

/*
 *	Procedure to unselect the given thumb from the list of selected
 *	thumbs on the thumbs list.
 */
static void TListDoUnselectThumb(
	tlist_struct *tlist, gint thumb_num, GdkEventButton *button
)
{
	if(tlist == NULL)
	    return;

	/* Is given thumb not selected? */
	if(g_list_find(tlist->selection, (gpointer)thumb_num) == NULL)
	    return;

	/* Report unselect as needed. */
	if(tlist->unselect_cb != NULL)
	    tlist->unselect_cb(
		(gpointer)tlist,
		button,
		thumb_num,
		tlist->client_data
	    );

	/* Remove thumb from selection list. */
	if(tlist->selection != NULL)
	    tlist->selection = g_list_remove(
		tlist->selection, (gpointer)thumb_num
	    );

	/* Update end selection pointer since main selection changed. */
	tlist->selection_end = (tlist->selection != NULL) ?
	    g_list_last(tlist->selection) : NULL;

	TListDraw(tlist);
}


/*
 *      Procedure to select all thumbs from the list of selected
 *      thumbs on the thumbs list.
 */
static void TListDoSelectAllThumbs(tlist_struct *tlist)
{
	gint i;


        if(tlist == NULL)
            return;


        /* Deallocate entire selection list first. */
        if(tlist->selection != NULL)
            g_list_free(tlist->selection);
        tlist->selection = NULL;
        tlist->selection_end = NULL;


	/* Create selections list for all thumbs. */
	for(i = 0; i < tlist->total_thumbs; i++)
	    tlist->selection = g_list_append(
		tlist->selection, (gpointer)i
	    );

        /* Update end selection pointer since main selection changed. */
        tlist->selection_end = (tlist->selection != NULL) ?
            g_list_last(tlist->selection) : NULL;


        /* Report select for all selected thumbs? */
        if(tlist->select_cb != NULL)
        {
            GList *glist = tlist->selection;
            while(glist != NULL)
            {
                tlist->select_cb(
                    (gpointer)tlist,
                    NULL,           /* Do not pass button event. */
                    (gint)glist->data,
                    tlist->client_data
                );
                glist = glist->next;
            }
        }

        TListDraw(tlist);
}

/*
 *      Procedure to unselect all thumbs from the list of selected
 *      thumbs on the thumbs list.
 */
static void TListDoUnselectAllThumbs(tlist_struct *tlist)
{
	if(tlist == NULL)
	    return;

	/* Report unselect for all selected thumbs? */
	if(tlist->unselect_cb != NULL)
        {
            GList *glist = tlist->selection_end;
            while(glist != NULL)
            {
                tlist->unselect_cb(
                    (gpointer)tlist,
                    NULL,           /* Do not pass button event. */
                    (gint)glist->data,
                    tlist->client_data
                );
                glist = glist->prev;
            }
        }

        /* Deallocate entire selection. */
        if(tlist->selection != NULL)
            g_list_free(tlist->selection);
        tlist->selection = NULL;
        tlist->selection_end = NULL;

        TListDraw(tlist);
}



/*
 *	Redraws the drawing area on the given thumbs list.
 *
 *	If the tlist is currently frozen or unmapped then it nothing
 *	will be done.
 */
void TListDraw(tlist_struct *tlist)
{
	gboolean has_focus;
	gint state, width, height;
	gint tpl, thumb_width, thumb_height, thumb_border, thumb_pixmap_height;
	gint font_height = 16;
	GdkRectangle text_rect;
	GdkFont *font;
	GdkGC *gc;
	GdkPixmap *bg_pixmap;
	GdkWindow *window;
	GdkDrawable *drawable;
	GtkStyle *style;
	GtkAdjustment *hadj, *vadj;
	GtkWidget *w;


	if(tlist == NULL)
	    return;

	/* Do not draw list if it is frozen, fully obscured, or
	 * unmapped.
	 */
	if((tlist->freeze_count > 0) || (tlist->visibility == GDK_VISIBILITY_FULLY_OBSCURED) ||
	   !tlist->map_state
	)
	    return;

	/* If the list's pixmap buffer has not been created yet then call
	 * the resize function to create it. Note that the resize function
	 * will call this function back again so we can just return in
	 * this case.
	 */
	if((tlist->list_pm == NULL) && tlist->double_buffer)
	{
	    TListResize(tlist, -1, -1, FALSE);
	    return;
	}

	w = tlist->list_da;
	if(w == NULL)
	    return;

	has_focus = GTK_WIDGET_HAS_FOCUS(w);
	state = GTK_WIDGET_STATE(w);
	style = gtk_widget_get_style(w);
	window = w->window;
	if((style == NULL) || (window == NULL))
	    return;

	font = style->font;
	if(font != NULL)
	    font_height = font->ascent + font->descent;

        gdk_window_get_size(window, &width, &height);
	if((width <= 0) || (height <= 0))
	    return;

	/* Get the pointer to the drawable as the pixmap buffer or the
	 * window itself depending on if we are using double buffer or not.
	 */
	if(tlist->double_buffer)
	    drawable = (GdkDrawable *)tlist->list_pm;
	else
	    drawable = (GdkDrawable *)window;
	if(drawable == NULL)
	    return;


	/* Create graphic contexts as needed. */
	gc = tlist->gc;
	if(gc == NULL)
	{
	    tlist->gc = gc = gdk_gc_new(window);
	    gdk_gc_set_function(gc, GDK_COPY);
	    gdk_gc_set_fill(gc, GDK_SOLID);
	    gdk_gc_set_foreground(gc, &style->bg[state]);
	}
	gc = tlist->selection_gc;
	if(gc == NULL)
	{
            tlist->selection_gc = gc = gdk_gc_new(window);
            gdk_gc_set_function(gc, GDK_COPY);
            gdk_gc_set_fill(gc, GDK_SOLID);
            gdk_gc_set_foreground(gc, &style->bg[state]);
	}


	gc = tlist->gc;
	bg_pixmap = style->bg_pixmap[state];


	/* Draw background. */
	if(bg_pixmap != NULL)
	{
	    gdk_gc_set_fill(gc, GDK_TILED);
	    gdk_gc_set_tile(gc, bg_pixmap);
	    gdk_gc_set_ts_origin(gc, 0, 0);
	}
	else
	{
	    gdk_gc_set_fill(gc, GDK_SOLID);
	    gdk_gc_set_foreground(gc, &style->base[state]);
	}
	gdk_draw_rectangle(
	    drawable, gc, TRUE,
	    0, 0, width, height
	);
	gdk_gc_set_fill(gc, GDK_SOLID);
	gdk_gc_set_tile(gc, NULL);


	/* Begin drawing thumbs. */

	hadj = tlist->hadj;
	vadj = tlist->vadj;

	/* Calculate thumbs per line. */
	tpl = MAX(tlist->thumbs_per_line, 1);

	/* Get size of each thumb and sizes with in each thumb. */
	thumb_width = tlist->thumb_width;
	thumb_height = tlist->thumb_height;
	thumb_border = tlist->thumb_border;
        thumb_pixmap_height = thumb_height - (4 * DEF_FRAME_HEIGHT) -
	    (3 * thumb_border) - font_height;

	/* Calculate bounding box for text rectangle. */
	text_rect.x = thumb_border + DEF_FRAME_WIDTH;
	text_rect.y = thumb_height - (3 * DEF_FRAME_HEIGHT) -
	    thumb_border - font_height;
	text_rect.width = MAX(thumb_width - (2 * DEF_FRAME_WIDTH) -
	    (2 * thumb_border), 2);
	text_rect.height = (2 * DEF_FRAME_HEIGHT) + font_height;


	/* Enough information to draw thumbs? */
	if((thumb_width > 0) && (thumb_height > 0) &&
           (hadj != NULL) && (vadj != NULL)
	)
	{
	    gint i, x, y, i_start, x_start, y_start;
	    tlist_thumb_struct *thumb_ptr;

#define DO_DRAW_THUMB_ITERATE	\
{ \
 if(thumb_ptr != NULL) \
 { \
  gbool is_selected = TListIsThumbSelected(tlist, i); \
\
  /* Draw base. */ \
  gdk_gc_set_foreground( \
   gc, \
   is_selected ? &style->bg[GTK_STATE_SELECTED] : &style->bg[state] \
  ); \
  gdk_draw_rectangle( \
   drawable, gc, TRUE, \
   x, y, thumb_width, thumb_height \
  ); \
\
  /* Draw pixmap. */ \
  if(thumb_ptr->pixmap != NULL) \
  { \
   gint pm_x, pm_y; \
   gint pm_width, pm_height; \
   GdkPixmap *pixmap = thumb_ptr->pixmap; \
   GdkBitmap *mask = thumb_ptr->mask; \
\
   gdk_window_get_size((GdkWindow *)pixmap, &pm_width, &pm_height); \
   pm_x = x + (thumb_width / 2) - (pm_width / 2); \
   pm_y = y + DEF_FRAME_HEIGHT + thumb_border + (thumb_pixmap_height / 2) - \
    (pm_height / 2); \
\
   gdk_gc_set_clip_mask(gc, mask); \
   gdk_gc_set_clip_origin(gc, pm_x, pm_y); \
   gdk_draw_pixmap( \
    drawable, gc, pixmap, \
    0, 0, \
    pm_x, pm_y, \
    pm_width, pm_height \
   ); \
   gdk_gc_set_clip_mask(gc, NULL); \
  } \
\
  /* Draw text, text base, and text frame. */ \
  if((font != NULL) && (thumb_ptr->text != NULL)) \
  { \
   gint lbearing, rbearing, swidth, ascent, descent; \
   const gchar *s = thumb_ptr->text; \
   GdkRectangle rect, text_base_rect; \
\
   /* Set up translated text rectangle. */ \
   rect.x = x + text_rect.x; \
   rect.y = y + text_rect.y; \
   rect.width = text_rect.width; \
   rect.height = text_rect.height; \
\
   /* Get geometry of font based on the current text string. */ \
   gdk_string_extents( \
    font, s, \
    &lbearing, &rbearing, &swidth, &ascent, &descent \
   ); \
\
   /* Set up to draw text base. */ \
   text_base_rect.width = MIN((4 * DEF_FRAME_WIDTH) + swidth, rect.width); \
   text_base_rect.height = rect.height; \
   text_base_rect.x = x + MAX((thumb_width / 2) - (text_base_rect.width / 2), \
    thumb_border + DEF_FRAME_WIDTH); \
   text_base_rect.y = rect.y; \
   gdk_gc_set_foreground( \
    gc, \
    is_selected ? &style->bg[GTK_STATE_SELECTED] : &style->base[state] \
   ); \
   /* Draw text base. */ \
   gdk_draw_rectangle( \
    drawable, gc, TRUE, \
    text_base_rect.x, text_base_rect.y, \
    text_base_rect.width, text_base_rect.height \
   ); \
\
   /* Set up to draw text. */ \
   gdk_gc_set_clip_rectangle(gc, &rect); \
   gdk_gc_set_foreground( \
    gc, \
    is_selected ? &style->text[GTK_STATE_SELECTED] : &style->text[state] \
   ); \
   gdk_gc_set_font(gc, font); \
   /* Draw text. */ \
   gdk_draw_string( \
    drawable, font, gc, \
    x + (thumb_width / 2) - (swidth / 2), \
    y + thumb_height - font->descent - thumb_border - \
     (2 * DEF_FRAME_HEIGHT), \
    s \
   ); \
   gdk_gc_set_clip_rectangle(gc, NULL); \
\
   /* Draw text frame shadow. */ \
   gtk_draw_shadow( \
    style, drawable, state, GTK_SHADOW_IN, \
    text_base_rect.x, text_base_rect.y, \
    text_base_rect.width, text_base_rect.height \
   ); \
  } \
\
  /* Draw frame shadow. */ \
  gtk_draw_shadow( \
   style, drawable, state, GTK_SHADOW_OUT, \
   x, y, thumb_width, thumb_height \
  ); \
\
  /* Draw focus rectangle if this thumb is the focus thumb. */ \
  if((tlist->focus_thumb == i) && has_focus) \
  { \
   if(is_selected) \
    gdk_gc_set_function(gc, GDK_INVERT); \
   gdk_draw_rectangle( \
    drawable, gc, FALSE, \
    x + 2, y + 2, thumb_width - 5, thumb_height - 5 \
   ); \
   gdk_gc_set_function(gc, GDK_COPY); \
  } \
\
 } \
}


	    if(tlist->horizontal)
	    {
                /* Calculate starting positions. */
                i_start = (gint)(hadj->value / thumb_width) * tpl;
                x_start = -((gint)hadj->value % (gint)thumb_width);
		y_start = -(gint)vadj->value;


                /* Begin itterating and drawing each thumb. */
                x = x_start;
                y = y_start;
                for(i = MAX(i_start, 0); i < tlist->total_thumbs; i++)
                {
                    /* Drawn past end of page? */
                    if(x > hadj->page_size)
                        break;

                    /* Get this thumb and draw it if it is valid. */
                    thumb_ptr = tlist->thumb[i];
                    DO_DRAW_THUMB_ITERATE

		    /* Increment coordinates. */
		    y += thumb_height;

		    /* Drawn past edge of `row'? */
		    if((y + thumb_height) > vadj->page_size)
		    {
			x += thumb_width;
			y = y_start;
		    }
		}
	    }
	    else
	    {
		/* Calculate starting positions. */
		i_start = (gint)(vadj->value / thumb_height) * tpl;
		x_start = -(gint)hadj->value;
		y_start = -((gint)vadj->value % (gint)thumb_height);


		/* Begin itterating and drawing each thumb. */
		x = x_start;
		y = y_start;
		for(i = MAX(i_start, 0); i < tlist->total_thumbs; i++)
		{
		    /* Drawn past end of page? */
		    if(y > vadj->page_size)
			break;

		    /* Get this thumb and draw it if it is valid. */
		    thumb_ptr = tlist->thumb[i];
		    DO_DRAW_THUMB_ITERATE

                    /* Increment coordinates. */
                    x += thumb_width;

                    /* Drawn past edge of `row'? */
                    if((x + thumb_width) > hadj->page_size)
                    {
                        y += thumb_height;
                        x = x_start;
                    }
		}
	    }
	}

#undef DO_DRAW_THUMB_ITERATE

	/* Put back buffer to window (if using double buffering). */
	if(drawable != (GdkDrawable *)window)
            gdk_window_copy_area(
                window, gc,
                0, 0,
                (GdkWindow *)drawable,
                0, 0,
                width, height
            );
}

/*
 *	Recalculates the geometry of the thumbs list based on the new
 *	width and height.
 *
 *	A "changed" signal will be emitted to the scroll bars and the
 *	tlist will be redrawn.
 */
void TListResize(
	tlist_struct *tlist, gint width, gint height,
	gbool allow_gtk_main_iteration
)
{
	gbool need_queue_resize = FALSE;
	gint tpl;
	GdkWindow *window;
	GtkAdjustment *hadj, *vadj;
	GtkWidget *w;


	if(tlist == NULL)
	    return;

	hadj = tlist->hadj;
	vadj = tlist->vadj;
	if((hadj == NULL) || (vadj == NULL))
	    return;

	w = tlist->list_da;
	if((w != NULL) ? !GTK_WIDGET_REALIZED(w) : TRUE)
	    return;

	window = w->window;
	if(window == NULL)
	    return;

	/* If no width or height is not given then obtain it from the
	 * drawing area widget.
	 */
	if(width < 0)
	    width = w->allocation.width;
        if(height < 0)
            height = w->allocation.height;
/*	gdk_window_get_size(window, &width, &height); */

	if((tlist->thumb_width <= 0) || (tlist->thumb_height <= 0) ||
           (width <= 0) || (height <= 0)
	)
	    return;


	/* Begin recreating list drawing area's pixmap buffer as
	 * needed.
	 */
	if(tlist->list_pm != NULL)
	{
	    gdk_pixmap_unref(tlist->list_pm);
	    tlist->list_pm = NULL;
	}
	if(tlist->double_buffer)
	{
	    /* Using double buffer, so we need to create a new pixmap
	     * buffer.
	     */
	    tlist->list_pm = gdk_pixmap_new(window, width, height, -1);
	    if(tlist->list_pm == NULL)
		return;
	}

	/* Begin updating thumbs per line value and scroll adjustments. */
	if(tlist->horizontal)
	{
            /* Calculate new number of thumbs per line. */
	    tlist->thumbs_per_line = tpl = MAX(
		height / tlist->thumb_height, 1
	    );

	    /* Update adjustment ranges and page sizes. */
            hadj->lower = 0.0;
            hadj->upper = (gfloat)MAX(
                ((gint)(tlist->total_thumbs / tpl) * tlist->thumb_width) +
		(((tlist->total_thumbs % tpl) == 0) ? 0 : tlist->thumb_width),
                width
            );
	    hadj->page_size = (gfloat)width;
	    hadj->page_increment = hadj->page_size / 2;
	    hadj->step_increment = tlist->thumb_width;

	    vadj->lower = 0.0;
	    vadj->upper = (gfloat)MAX(height, tlist->thumb_height);
	    vadj->page_size = (gfloat)height;
            vadj->page_increment = vadj->page_size / 2;
            vadj->step_increment = tlist->thumb_height / 2;
	}
	else
	{
	    /* Calculate new number of thumbs per line. */
            tlist->thumbs_per_line = tpl = MAX(
                width / tlist->thumb_width, 1
            );

            /* Update adjustment ranges and page sizes. */
            hadj->lower = 0.0;
            hadj->upper = (gfloat)MAX(width, tlist->thumb_width);
            hadj->page_size = (gfloat)width;
            hadj->page_increment = hadj->page_size / 2;
            hadj->step_increment = tlist->thumb_width / 2;

            vadj->lower = 0.0;
            vadj->upper = (gfloat)MAX(
                ((gint)(tlist->total_thumbs / tpl) * tlist->thumb_height) +
                (((tlist->total_thumbs % tpl) == 0) ? 0 : tlist->thumb_height),
                height
            );
            vadj->page_size = (gfloat)height;
            vadj->page_increment = vadj->page_size / 2;
            vadj->step_increment = tlist->thumb_height;
	}

	/* Clip scroll adjustment values. */
        if(hadj->value > (hadj->upper - hadj->page_size))
            hadj->value = hadj->upper - hadj->page_size;
	if(hadj->value < hadj->lower)
	    hadj->value = hadj->lower;

        if(vadj->value > (vadj->upper - vadj->page_size))
            vadj->value = vadj->upper - vadj->page_size;
        if(vadj->value < vadj->lower)
            vadj->value = vadj->lower;

	/* Show or hide scroll bars depending on adjustment page size. */
	w = tlist->hsb;
	if(w != NULL)
	{
	    if((hadj->upper - hadj->lower) > hadj->page_size)
	    {
		if(!GTK_WIDGET_MAPPED(w))
		{
		    gtk_widget_show(w);
		    need_queue_resize = TRUE;
		}
	    }
	    else
	    {
		if(GTK_WIDGET_MAPPED(w))
		{
		    gtk_widget_hide(w);
                    need_queue_resize = TRUE;
                }
            }
	}

        w = tlist->vsb;
        if(w != NULL)
        {
            if((vadj->upper - vadj->lower) > vadj->page_size)
            {
                if(!GTK_WIDGET_MAPPED(w))
                {
                    gtk_widget_show(w);
                    need_queue_resize = TRUE;
                }
            }
            else
            {
                if(GTK_WIDGET_MAPPED(w))
                {
                    gtk_widget_hide(w);
                    need_queue_resize = TRUE;
                }
            }
        }


	/* Notify adjustments about change in page size and range. */
	gtk_signal_emit_by_name(GTK_OBJECT(hadj), "changed");
        gtk_signal_emit_by_name(GTK_OBJECT(vadj), "changed");

	/* Notify adjustments about change in value, this will cause a
	 * redraw.
	 */
        gtk_signal_emit_by_name(GTK_OBJECT(hadj), "value_changed");
        gtk_signal_emit_by_name(GTK_OBJECT(vadj), "value_changed");


	/* Check if toplevel needs to be resized again due to
	 * the scroll bar(s) map/unmapping.
	 */
	w = tlist->list_da;
	if(need_queue_resize && (w != NULL))
	{
	    while((gtk_events_pending() > 0) && allow_gtk_main_iteration)
		gtk_main_iteration();

	    gtk_widget_queue_resize(w);
	}
}

/*
 *	Adds a freeze count to the thumbs list. If the previous freeze
 *	count was 0 then the thumbs list will be frozen.
 */
void TListFreeze(tlist_struct *tlist)
{
	if(tlist == NULL)
	    return;

	tlist->freeze_count++;

	/* Previous freeze count was 0? */
	if(tlist->freeze_count == 1)
	{
	    /* Freeze list. */


	}
}

/*
 *	Removes a freeze count from the thumbs list. If the previous freeze
 *      count was positive then the thumbs list will be thawed.
 */
void TListThaw(tlist_struct *tlist)
{
        if(tlist == NULL)
            return;

        tlist->freeze_count--;

        /* Previous freeze count was positive? */
        if(tlist->freeze_count == 0)
        {
            /* Thaw list. */

	    /* Need to redraw once thawed, so that changes that were made
	     * when the list was frozen will be shown.
	     */
	    TListDraw(tlist);
        }

	/* Make sure freeze count stays at 0 or above. */
	if(tlist->freeze_count < 0)
	    tlist->freeze_count = 0;
}


/*
 *	Appends a new thumb to the thumbs list with respect to the given
 *	inputs. Returns the thumb index number or -1 on error.
 */
gint TListAppend(
        tlist_struct *tlist, const gchar *text,
        gpointer client_data,
        void (*destroy_cb)(gpointer)
)
{
	gint thumb_num;
	tlist_thumb_struct *thumb_ptr;


	if(tlist == NULL)
	    return(-1);


	/* Create new thumb and initialize values. */
	thumb_ptr = TListThumbNew();

        thumb_ptr->text = g_strdup((text != NULL) ?
            text : ""
        );

        thumb_ptr->client_data = client_data;
        thumb_ptr->destroy_cb = destroy_cb;


	/* Sanitize total. */
	if(tlist->total_thumbs < 0)
	    tlist->total_thumbs = 0;

	/* Increase total and append new pointer. */
	thumb_num = tlist->total_thumbs;
	tlist->total_thumbs = thumb_num + 1;

	tlist->thumb = (tlist_thumb_struct **)g_realloc(
	    tlist->thumb,
	    tlist->total_thumbs * sizeof(tlist_thumb_struct *)
	);
	if(tlist->thumb == NULL)
	{
	    TListThumbDelete(thumb_ptr);
	    tlist->total_thumbs = 0;
	    return(-1);
	}

	/* Set newly created thumb to the newly allocated pointer. */
	tlist->thumb[thumb_num] = thumb_ptr;

	/* Resize and redraw. */
        TListResize(tlist, -1, -1, FALSE);

	return(thumb_num);
}

/*
 *	Sets the load state of the thumb to load_state.
 */
void TListSetLoadState(
        tlist_struct *tlist, gint thumb_num,
        gint load_state		/* One of TLIST_LOAD_STATE_*. */
)
{
        tlist_thumb_struct *thumb_ptr = TListGetThumbPtr(tlist, thumb_num);
        if(thumb_ptr == NULL)
            return;

	thumb_ptr->load_state = load_state;

        TListDraw(tlist);
}

/*
 *	Sets the text for the thumb specified by thumb_num.
 */
void TListSetText(
        tlist_struct *tlist, gint thumb_num,
        const gchar *text
)
{
	tlist_thumb_struct *thumb_ptr = TListGetThumbPtr(tlist, thumb_num);
	if(thumb_ptr == NULL)
	    return;

	g_free(thumb_ptr->text);
	thumb_ptr->text = g_strdup(
	    (text != NULL) ? text : ""
	);

        TListDraw(tlist);
}

/*
 *	Sets the pixmap for the thumb specified by thumb_num.
 */
void TListSetPixmap(
        tlist_struct *tlist, gint thumb_num,
        GdkPixmap *pixmap, GdkBitmap *mask
)
{
        tlist_thumb_struct *thumb_ptr = TListGetThumbPtr(tlist, thumb_num);
        if(thumb_ptr == NULL)
            return;

	/* If given pixmap is NULL then that implies we should just unref
	 * the currently displayed pixmap.
	 */
	if(pixmap == NULL)
	{
	    /* No given pixmap so just unref the existing pixmap and mask
	     * pair (if any).
	     */
	    if(thumb_ptr->pixmap != NULL)
	    {
		gdk_pixmap_unref(thumb_ptr->pixmap);
		thumb_ptr->pixmap = NULL;
	    }
	    if(thumb_ptr->mask != NULL)
            {
                gdk_bitmap_unref(thumb_ptr->mask);
                thumb_ptr->mask = NULL;
            }
	}
	else
	{
	    GdkPixmap *oldpixmap = thumb_ptr->pixmap;
	    GdkBitmap *oldmask = thumb_ptr->mask;


	    /* Increment a refcount on the given pixmap and mask pair,
	     * then set them as the thumb's current pixmap and mask.
	     *
	     * If either the given pixmap and/or mask is NULL then the
	     * the thumb's current pixmap and/or mask will be set NULL.
	     */
	    gdk_pixmap_ref(pixmap);
	    thumb_ptr->pixmap = pixmap;

	    if(mask != NULL)
		gdk_bitmap_ref(mask);
	    thumb_ptr->mask = mask;


	    /* Unref thumb's previous pixmap and mask (if any). */
	    if(oldpixmap != NULL)
		gdk_pixmap_unref(oldpixmap);
            if(oldmask != NULL)
                gdk_bitmap_unref(oldmask);
	}

        TListDraw(tlist);
}

/*
 *	Sets the pixmap image of the thumb specified by thumb_num.
 */
void TListSetRGBA(
        tlist_struct *tlist, gint thumb_num,
        gint width, gint height,        /* Image size in pixels. */
	gint bpl,			/* Bytes per line, can be -1. */
        GdkRgbDither dith,
	const guint8 *rgba_buf,
        gbool no_enlarge                /* Do not upscale if image smaller
                                         * than thumb.
                                         */
)
{
	gfloat coeff;
	gint thumb_width_clip, thumb_height_clip;
	gint twidth, theight, tbpl;
	guint8 *thumb_rgba;
	gint font_height = 16;
	GdkFont *font;
	GdkWindow *window;
	GtkWidget *w;
	GtkStyle *style;
        tlist_thumb_struct *thumb_ptr = TListGetThumbPtr(tlist, thumb_num);
        if(thumb_ptr == NULL)
            return;

	/* Unref existing pixmap and mask pair (if any). */
	if(thumb_ptr->pixmap != NULL)
	{
	    gdk_pixmap_unref(thumb_ptr->pixmap);
	    thumb_ptr->pixmap = NULL;
	}
        if(thumb_ptr->mask != NULL)
        {
            gdk_bitmap_unref(thumb_ptr->mask);
            thumb_ptr->mask = NULL;
        }

	/* If no image data is given then stop at this point so that
	 * the only operation done was unloading of the thumb's previous
	 * pixmap.
	 */
	if((rgba_buf == NULL) || (width <= 0) || (height <= 0))
	    return;

	/* Need to calculate source bytes per line? */
	if(bpl <= 0)
	    bpl = width * 4 * sizeof(guint8);


	/* Get values from list's drawing area widget. */
	w = tlist->list_da;
	if(w == NULL)
	    return;
	style = gtk_widget_get_style(w);
	window = w->window;
	if((style == NULL) || (window == NULL))
	    return;

	font = style->font;
	if(font != NULL)
	    font_height = font->ascent + font->descent;

	/* Calculate the exact size of the target thumb, note that it
	 * will have to be smaller than the actual thumb size to leave
	 * room for borders and text.
	 */
	thumb_width_clip = tlist->thumb_width - (2 * DEF_FRAME_WIDTH) -
	    (2 * tlist->thumb_border);
	thumb_height_clip = tlist->thumb_height - (4 * DEF_FRAME_HEIGHT) -
	    (3 * tlist->thumb_border) - font_height;
	if((thumb_width_clip <= 0) || (thumb_height_clip <= 0))
	    return;


	/* Begin calculating thumb width and height, mind the aspect
	 * ratio of the given image data.
	 */
	coeff = (gfloat)width / (gfloat)height;

	theight = thumb_height_clip;
	twidth = theight * coeff;

	if(twidth > thumb_width_clip)
	{
	    coeff = (gfloat)height / (gfloat)width;

	    twidth = thumb_width_clip;
	    theight = twidth * coeff;
	}

	/* If the thumb image is not to be enlarged if the given image
	 * data is smaller, then make that check here and update twidth
	 * and theight to match the given image's geometry as needed.
	 */
	if(no_enlarge)
	{
	    if(twidth > width)
		twidth = width;
            if(theight > height)
                theight = height;
	}

	if((twidth <= 0) || (theight <= 0))
	    return;

	/* At this point, twidth and theight are the desired height of
	 * the thumb image data.
	 */

	/* Calculate bytes per line for the thumb image data and
	 * allocate tempory buffer for thumb image.
	 */
	tbpl = twidth * 4 * sizeof(guint8);
	thumb_rgba = (guint8 *)g_malloc(
	    tbpl * theight * sizeof(guint8)
	);
	if(thumb_rgba == NULL)
	    return;

	GUIResizeBuffer(
	    4, 		/* 32 bits. */
	    (const gpointer)rgba_buf, width, height, bpl,
	    thumb_rgba, twidth, theight, tbpl
	);


	/* Create a new thumb pixmap and copy the thumb_rgba
	 * data to it.
	 */
	thumb_ptr->pixmap = gdk_pixmap_new(window, twidth, theight, -1);
	if(thumb_ptr->pixmap != NULL)
	    gdk_draw_rgb_32_image(
		(GdkDrawable *)thumb_ptr->pixmap, style->white_gc,
		0, 0, twidth, theight,
		dith,
		thumb_rgba,
		tbpl
            );
	/* Create new thumb mask from the thumb_rgba data. */
	thumb_ptr->mask = GUICreateBitmapFromDataRGBA(
	    twidth, theight, tbpl,
	    thumb_rgba,
	    0x80,		/* Alpha byte threshold. */
	    window
	);

	/* Deallocate target thumb image data, it is no longer needed
	 * since it has been sent to the server side pixmap.
	 */
	g_free(thumb_rgba);
	thumb_rgba = NULL;

        TListDraw(tlist);
}

/*
 *	Sets the new client data and destroy notify function of the thumb
 *	specified by thumb_num.
 *
 *	The previously set client data will not be notified.
 */
void TListSetDestroy(
        tlist_struct *tlist, gint thumb_num,
        gpointer client_data,
        void (*destroy_cb)(gpointer)
)
{
        tlist_thumb_struct *thumb_ptr = TListGetThumbPtr(tlist, thumb_num);
        if(thumb_ptr == NULL)
            return;

	thumb_ptr->client_data = client_data;
	thumb_ptr->destroy_cb = destroy_cb;
}

/*
 *	Removes the thumb specified by thumb_num from the given thumbs
 *	list.
 */
void TListRemove(tlist_struct *tlist, gint thumb_num)
{
	gint i;
	GList *glist;
        tlist_thumb_struct *thumb_ptr = TListGetThumbPtr(tlist, thumb_num);
        if(thumb_ptr == NULL)
            return;

	/* Unselect this thumb since it is about to be deleted. */
	TListDoUnselectThumb(tlist, thumb_num, NULL);

        /* Call thumb's destroy notify? */
        if(thumb_ptr->destroy_cb != NULL)
            thumb_ptr->destroy_cb(thumb_ptr->client_data);


	/* Need to shift selection data value referencing any thumb index
	 * who's value is greater than this thumb since this thumb is
	 * about to be removed.
	 */
	glist = tlist->selection;
	while(glist != NULL)
	{
	    i = (gint)glist->data;

	    if(i > thumb_num)
		glist->data = (gpointer)(i - 1);

	    glist = glist->next;
	}


	/* Delete thumb. */
	TListThumbDelete(thumb_ptr);
	tlist->thumb[thumb_num] = thumb_ptr = NULL;

	/* Reduce pointers on thumbs list and shift as needed. */
	tlist->total_thumbs--;
	if(tlist->total_thumbs > 0)
	{
	    gint i;

	    /* Shift pointers. */
	    for(i = thumb_num; i < tlist->total_thumbs; i++)
		tlist->thumb[i] = tlist->thumb[i + 1];

	    /* Reduce pointer array allocation. */
	    tlist->thumb = (tlist_thumb_struct **)g_realloc(
		tlist->thumb,
		tlist->total_thumbs * sizeof(tlist_thumb_struct *)
	    );
	    if(tlist->thumb == NULL)
	    {
		tlist->total_thumbs = 0;
	    }
	}
	else
	{
	    /* Total has reached 0, deallocate pointer array. */
	    g_free(tlist->thumb);
	    tlist->thumb = NULL;
	    tlist->total_thumbs = 0;
	}

	/* Clamp focus thumb. */
	if(tlist->focus_thumb >= tlist->total_thumbs)
	    tlist->focus_thumb = tlist->total_thumbs - 1;
	if(tlist->focus_thumb < 0)
	    tlist->focus_thumb = 0;


	TListResize(tlist, -1, -1, FALSE);
}

/*
 *	Removes all thumbs from the given thumbs list.
 */
void TListClear(tlist_struct *tlist)
{
	gint i;
        tlist_thumb_struct *thumb_ptr;


        if(tlist == NULL)
            return;

	/* Report unselect for all selected thumbs? */
	if(tlist->unselect_cb != NULL)
	{
	    GList *glist = tlist->selection_end;
	    while(glist != NULL)
	    {
		tlist->unselect_cb(
		    (gpointer)tlist,
		    NULL,		/* Do not pass button event. */
		    (gint)glist->data,
		    tlist->client_data
		);
		glist = glist->prev;
	    }
	}
	/* Deallocate entire selection list (just in case). */
	if(tlist->selection != NULL)
	    g_list_free(tlist->selection);
	tlist->selection = NULL;
	tlist->selection_end = NULL;

	/* Reset focus and pointer over thumbs. */
	tlist->focus_thumb = 0;
	tlist->pointer_over_thumb = -1;

        /* Iterate through each thumb, calling the destroy notify
	 * and deallocating the thumb.
	 */
	for(i = tlist->total_thumbs - 1; i >= 0; i--)
	{
	    thumb_ptr = tlist->thumb[i];
	    if(thumb_ptr == NULL)
		continue;

	    /* Call destroy notify? */
	    if(thumb_ptr->destroy_cb != NULL)
		thumb_ptr->destroy_cb(thumb_ptr->client_data);

	    /* Delete thumb. */
	    TListThumbDelete(thumb_ptr);
	    tlist->thumb[i] = thumb_ptr = NULL;
	}

	/* Deallocate pointer array. */
	g_free(tlist->thumb);
	tlist->thumb = NULL;
	tlist->total_thumbs = 0;

	TListResize(tlist, -1, -1, FALSE);
}


/*
 *	Returns the client data set for the given thumb.
 */
gpointer TListGetThumbData(tlist_struct *tlist, gint thumb_num)
{
        tlist_thumb_struct *thumb_ptr = TListGetThumbPtr(tlist, thumb_num);
        return((thumb_ptr != NULL) ? thumb_ptr->client_data : NULL);
}


/*
 *	Selects the thumb specified by thumb_num.
 *
 *	If the thumb is already selected or does not exist then nothing
 *	will be done.
 */
void TListSelectThumb(tlist_struct *tlist, gint thumb_num)
{
	TListDoSelectThumb(tlist, thumb_num, NULL, TRUE);
}

/*
 *      Unselects the thumb specified by thumb_num.
 *
 *	If the thumb is not selected or does not exist then nothing will
 *	be done.
 */
void TListUnselectThumb(tlist_struct *tlist, gint thumb_num)
{
        TListDoUnselectThumb(tlist, thumb_num, NULL);
}

/*
 *	Selects all thumbs.
 */
void TListSelectAll(tlist_struct *tlist)
{
	TListDoSelectAllThumbs(tlist);
}

/*
 *	Unselects all thumbs.
 */
void TListUnselectAll(tlist_struct *tlist)
{
	TListDoUnselectAllThumbs(tlist);
}


/*
 *	Returns the thumb number, and its x and y index position based
 *	on the given x and y coordinates relative to the thumb's list
 *	drawing area widget.
 *
 *	Returns TRUE if a selection is made, or FALSE on no match.
 */
gbool TListGetSelection(
        tlist_struct *tlist, gint x, gint y,
        gint *thumb_num, gint *thumb_ix, gint *thumb_iy
)
{
	gint i, ix, iy, tpl;
	GtkAdjustment *hadj, *vadj;


	if(thumb_num != NULL)
	    *thumb_num = -1;
	if(thumb_ix != NULL)
	    *thumb_ix = 0;
	if(thumb_iy != NULL)
	    *thumb_iy = 0;

	if(tlist == NULL)
	    return(FALSE);


	tpl = MAX(tlist->thumbs_per_line, 1);

	hadj = tlist->hadj;
	vadj = tlist->vadj;
	if((hadj == NULL) || (vadj == NULL))
	    return(FALSE);

	if((tlist->thumb_width <= 0) || (tlist->thumb_height <= 0))
	    return(FALSE);

	ix = (gint)(x + hadj->value) / tlist->thumb_width;
	iy = (gint)(y + vadj->value) / tlist->thumb_height;
	if(thumb_ix != NULL)
	    *thumb_ix = ix;
        if(thumb_iy != NULL)
            *thumb_iy = iy;

	if(tlist->horizontal)
            i = (ix * tpl) + CLIP(iy, 0, tpl - 1);
	else
	    i = (iy * tpl) + CLIP(ix, 0, tpl - 1);

	if(thumb_num != NULL)
	    *thumb_num = i;

	if((i < 0) || (i >= tlist->total_thumbs))
	    return(FALSE);
	else
	    return(TRUE);
}

/*
 *	Returns the upper left coordinate position of the given thumb
 *	relative to the thumb's list drawing area widget (with scroll
 *	adjustments applied).
 *
 *	Returns TRUE if thumb_num is valid and a coordinate position is
 *	given. The returned thumb coordinator position may or may not be
 *	visible.
 */
gbool TListGetPosition(
        tlist_struct *tlist, gint thumb_num,
        gint *x, gint *y
)
{
        gint lx, ly, tpl;
        GtkAdjustment *hadj, *vadj;


        if(x != NULL)
            *x = 0;
        if(y != NULL)
            *y = 0;

        if(tlist == NULL)
            return(FALSE);

	if((thumb_num < 0) || (thumb_num >= tlist->total_thumbs))
	    return(FALSE);

        tpl = MAX(tlist->thumbs_per_line, 1);

        hadj = tlist->hadj;
        vadj = tlist->vadj;
        if((hadj == NULL) || (vadj == NULL))
            return(FALSE);

        if((tlist->thumb_width <= 0) || (tlist->thumb_height <= 0))
            return(FALSE);

	if(tlist->horizontal)
	{
            lx = (gint)(thumb_num / tpl) * tlist->thumb_width;
            ly = (thumb_num % tpl) * tlist->thumb_height;
	}
	else
	{
	    lx = (thumb_num % tpl) * tlist->thumb_width;
	    ly = (gint)(thumb_num / tpl) * tlist->thumb_height;
	}
	if(x != NULL)
	    *x = lx - hadj->value;
        if(y != NULL)
            *y = ly - vadj->value;

	return(TRUE);
}


/*
 *	Returns one of GTK_VISIBILITY_*.
 *	If the thumb_num is invalid then GTK_VISIBILITY_NONE is returned.
 */
gint TListIsThumbVisible(
        tlist_struct *tlist, gint thumb_num
)
{
        gint lx, ly, tpl;
        GtkAdjustment *hadj, *vadj;


        if(tlist == NULL)
            return(GTK_VISIBILITY_NONE);

        if((thumb_num < 0) || (thumb_num >= tlist->total_thumbs))
            return(GTK_VISIBILITY_NONE);

        tpl = MAX(tlist->thumbs_per_line, 1);

        hadj = tlist->hadj;
        vadj = tlist->vadj;
        if((hadj == NULL) || (vadj == NULL))
            return(GTK_VISIBILITY_NONE);

        if((tlist->thumb_width <= 0) || (tlist->thumb_height <= 0))
            return(GTK_VISIBILITY_NONE);

        if(tlist->horizontal)
        {
            lx = (gint)(thumb_num / tpl) * tlist->thumb_width;
            ly = (thumb_num % tpl) * tlist->thumb_height;
        }
        else
        {
            lx = (thumb_num % tpl) * tlist->thumb_width;
            ly = (gint)(thumb_num / tpl) * tlist->thumb_height;
        }

	/* lx and ly now contain the absolute coordinate value (without
	 * adjustments applied). Now check if the position is valid
	 * by masking the absolute coordinate values of the adjustments
	 * over it.
	 */

	/* Totally obscured? */
	if(((gint)(lx + tlist->thumb_width) < (gint)hadj->value) ||
           ((gint)lx >= (gint)(hadj->value + hadj->page_size)) ||
           ((gint)(ly + tlist->thumb_height) < (gint)vadj->value) ||
           ((gint)ly >= (gint)(vadj->value + vadj->page_size))
	)
	    return(GTK_VISIBILITY_NONE);
	/* Fully visible? */
	else if(((gint)lx >= (gint)hadj->value) &&
                ((gint)(lx + tlist->thumb_width) <=
                    (gint)(hadj->value + hadj->page_size)) &&
                ((gint)ly >= (gint)vadj->value) &&
                ((gint)(ly + tlist->thumb_height) <=
                    (gint)(vadj->value + vadj->page_size))
	)
	    return(GTK_VISIBILITY_FULL);
	else
	    return(GTK_VISIBILITY_PARTIAL);
}

/* 
 *	Scrolls to the given thumb and offsets relative to the given
 *	coefficient coeff. The coeff must be between 0.0 and 1.0.
 */
void TListMoveTo(
        tlist_struct *tlist, gint thumb_num, gfloat coeff
)
{
        gint lx, ly, tpl;
        GtkAdjustment *hadj, *vadj;


        if(tlist == NULL)
            return;

        if((thumb_num < 0) || (thumb_num >= tlist->total_thumbs))
            return;

	if(coeff < 0.0)
	    coeff = 0.0;
	if(coeff > 1.0)
	    coeff = 1.0;

        tpl = MAX(tlist->thumbs_per_line, 1);

        hadj = tlist->hadj;
        vadj = tlist->vadj;
        if((hadj == NULL) || (vadj == NULL))
            return;

	if(tlist->horizontal)
	{
	    lx = (gint)(thumb_num / tpl) * tlist->thumb_width;
	    lx = lx - (coeff * (hadj->page_size - tlist->thumb_width));
	    hadj->value = (gfloat)lx;
	    if(hadj->value > (hadj->upper - hadj->page_size))
		hadj->value = hadj->upper - hadj->page_size;
	    if(hadj->value < hadj->lower)
		hadj->value = hadj->lower;
	    gtk_signal_emit_by_name(GTK_OBJECT(hadj), "value_changed");
	}
	else
	{
	    ly = (gint)(thumb_num / tpl) * tlist->thumb_height;
            ly = ly - (coeff * (vadj->page_size - tlist->thumb_height));
            vadj->value = (gfloat)ly;
            if(vadj->value > (vadj->upper - vadj->page_size))
                vadj->value = vadj->upper - vadj->page_size;
            if(vadj->value < vadj->lower)
                vadj->value = vadj->lower;
            gtk_signal_emit_by_name(GTK_OBJECT(vadj), "value_changed");
	}
}


/*
 *	Creates a new thumbs list.
 */
tlist_struct *TListNew(
        gbool horizontal,
        gint thumb_width, gint thumb_height, gint thumb_border,
        gpointer client_data,
        void (*select_cb)(gpointer, GdkEventButton *, gint, gpointer),
        void (*unselect_cb)(gpointer, GdkEventButton *, gint, gpointer)
)
{
	gint border_minor = 3;
	GtkAdjustment *adj;
	GtkWidget *w, *parent, *parent2;
	tlist_struct *tlist = TLIST(g_malloc0(
	    sizeof(tlist_struct)
	));
	if(tlist == NULL)
	    return(tlist);

	/* Reset values. */
	tlist->initialized = TRUE;
	tlist->map_state = FALSE;
	tlist->visibility = GDK_VISIBILITY_FULLY_OBSCURED;
	tlist->double_buffer = TRUE;
	tlist->horizontal = horizontal;
	tlist->freeze_count = 0;

	tlist->gc = NULL;
	tlist->selection_gc = NULL;

	tlist->focus_thumb = 0;
	tlist->pointer_over_thumb = -1;
	tlist->selection_mode = GTK_SELECTION_SINGLE;
	tlist->selection = NULL;
	tlist->selection_end = NULL;

	tlist->thumb = NULL;
	tlist->total_thumbs = 0;

	tlist->thumb_width = thumb_width;
	tlist->thumb_height = thumb_height;
	tlist->thumb_border = thumb_border;
	tlist->thumbs_per_line = 1;

	tlist->client_data = client_data;
	tlist->select_cb = select_cb;
	tlist->unselect_cb = unselect_cb;


	/* Create toplevel as a table. */
	tlist->toplevel = w = gtk_table_new(2, 2, FALSE);
	gtk_table_set_row_spacings(GTK_TABLE(w), border_minor);
	gtk_table_set_col_spacings(GTK_TABLE(w), border_minor);
	gtk_container_border_width(GTK_CONTAINER(w), 0);
	parent = w;


	/* Frame that will hold drawing area. */
	w = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
	gtk_table_attach(
	    GTK_TABLE(parent), w,
	    0, 1,
	    0, 1,
	    GTK_SHRINK | GTK_FILL | GTK_EXPAND,
	    GTK_SHRINK | GTK_FILL | GTK_EXPAND,
	    0, 0
	);
	gtk_widget_show(w);
	parent2 = w;

	/* List drawing area. */
	tlist->list_da = w = gtk_drawing_area_new();
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT | GTK_CAN_FOCUS);
        gtk_widget_add_events(
            w,
	    GDK_VISIBILITY_NOTIFY_MASK | GDK_FOCUS_CHANGE |
            GDK_EXPOSURE_MASK | GDK_STRUCTURE_MASK |
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
	    GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK |
            GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
            GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK
        );
        gtk_signal_connect_after(
            GTK_OBJECT(w), "draw",
            GTK_SIGNAL_FUNC(TListDrawCB), tlist
        );
        gtk_signal_connect_after(
            GTK_OBJECT(w), "draw_focus",
            GTK_SIGNAL_FUNC(TListDrawFocusCB), tlist
        );
        gtk_signal_connect_after(
            GTK_OBJECT(w), "draw_default",
            GTK_SIGNAL_FUNC(TListDrawDefaultCB), tlist
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "key_press_event",
            GTK_SIGNAL_FUNC(TListKeyEventCB), tlist
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "key_release_event",
            GTK_SIGNAL_FUNC(TListKeyEventCB), tlist
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "button_press_event",
            GTK_SIGNAL_FUNC(TListButtonEventCB), tlist
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "button_release_event",
            GTK_SIGNAL_FUNC(TListButtonEventCB), tlist
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "motion_notify_event",
            GTK_SIGNAL_FUNC(TListMotionEventCB), tlist
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "enter_notify_event",
            GTK_SIGNAL_FUNC(TListCrossingEventCB), tlist
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "leave_notify_event",
            GTK_SIGNAL_FUNC(TListCrossingEventCB), tlist
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "visibility_notify_event",
            GTK_SIGNAL_FUNC(TListVisibilityEventCB), tlist
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "expose_event",
            GTK_SIGNAL_FUNC(TListExposeEventCB), tlist
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "configure_event",
            GTK_SIGNAL_FUNC(TListConfigureEventCB), tlist
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "focus_in_event",
            GTK_SIGNAL_FUNC(TListFocusInEventCB), tlist
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "focus_out_event",
            GTK_SIGNAL_FUNC(TListFocusOutEventCB), tlist
        );
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);


	/* Vertical adjustment and scroll bar. */
	tlist->vadj = adj = (GtkAdjustment *)gtk_adjustment_new(
	    0.0, 0.0, 10.0, 10.0, 10.0, 10.0
	);
	gtk_object_ref(GTK_OBJECT(adj));
	tlist->vsb = w = gtk_vscrollbar_new(adj);
        gtk_table_attach(
            GTK_TABLE(parent), w,
            1, 2,
	    0, 1,
            GTK_FILL,
            GTK_EXPAND | GTK_SHRINK | GTK_FILL,
            0, 0
        );
        gtk_signal_connect(
            GTK_OBJECT(adj), "value_changed",
            GTK_SIGNAL_FUNC(TListAdjustmentValueChangedCB), tlist
        );
	/* Do not show this now, it will be mapped/unmapped when resizing
	 * of list drawing area occures.
	 */

        /* Horizontal adjustment and scroll bar. */
        tlist->hadj = adj = (GtkAdjustment *)gtk_adjustment_new(
            0.0, 0.0, 10.0, 10.0, 10.0, 10.0
        );
        gtk_object_ref(GTK_OBJECT(adj));
	tlist->hsb = w = gtk_hscrollbar_new(adj);
        gtk_table_attach(
            GTK_TABLE(parent), w,
            0, 1,
            1, 2,
	    GTK_EXPAND | GTK_SHRINK | GTK_FILL,
            GTK_FILL,
            0, 0
        );
        gtk_signal_connect(
            GTK_OBJECT(adj), "value_changed",
            GTK_SIGNAL_FUNC(TListAdjustmentValueChangedCB), tlist
        );
        /* Do not show this now, it will be mapped/unmapped when resizing
         * of list drawing area occures.
         */



	return(tlist);
}

/*
 *	Sets selection mode.
 */
void TListSelectionMode(tlist_struct *tlist, gint selection_mode)
{
	if(tlist == NULL)
	    return;

	tlist->selection_mode = selection_mode;
}

/*
 *	Sets double buffer or single buffer mode.
 */
void TListDoubleBuffer(tlist_struct *tlist, gbool double_buffer)
{
        if(tlist == NULL)
            return;

	tlist->double_buffer = double_buffer;

	if(double_buffer)
	{
	    /* Switching to double buffer. */

	    /* Call resize function to detect change to double buffer
	     * mode and recreate the list_pm.
	     */
	    TListResize(tlist, -1, -1, FALSE);
	}
	else
	{
	    /* Switching to single buffer. */

	    /* Get rid of the list's back buffer (if any). */
	    if(tlist->list_pm != NULL)
	    {
		gdk_pixmap_unref(tlist->list_pm);
		tlist->list_pm = NULL;
	    }
	}
}

/*
 *	Changes the orientation of the thumbs list.
 */
void TListOrientation(tlist_struct *tlist, gbool horizontal)
{
	if(tlist == NULL)
            return;

	if(tlist->horizontal != horizontal)
	{
	    GtkAdjustment	*hadj = tlist->hadj,
				*vadj = tlist->vadj;

	    /* Set new orientation. */
	    tlist->horizontal = horizontal;

	    /* Swap scroll adjustment values. */
	    if((hadj != NULL) && (vadj != NULL))
	    {
		gfloat v = hadj->value;
		hadj->value = vadj->value;
		vadj->value = v;
	    }

	    /* Call resize function to detect change to horizontal or
	     * vertical mode and recreate the list_pm.
	     */
	    TListResize(tlist, -1, -1, FALSE);
	}
}


/*
 *	Maps the given thumbs list.
 */
void TListMap(tlist_struct *tlist)
{
	GtkWidget *w;


	if(tlist == NULL)
	    return;

	w = tlist->toplevel;
	if(w == NULL)
	    return;

	if(!tlist->map_state)
	{
	    gtk_widget_show(w);
	    tlist->map_state = TRUE;
	}
}

/*
 *	Unmaps the given thumbs list.
 */
void TListUnmap(tlist_struct *tlist)
{
        GtkWidget *w;


        if(tlist == NULL)
            return;

        w = tlist->toplevel;
        if(w == NULL)
            return;

        if(tlist->map_state)
        {
            gtk_widget_hide(w);
            tlist->map_state = FALSE;
        }
}

/*
 *	Deallocates the given thumbs list and all its resources.
 */
void TListDelete(tlist_struct *tlist)
{
	GdkGC **gc;
	GdkPixmap **pixmap;
	GtkAdjustment **adj;
	GtkWidget **w;


	if(tlist == NULL)
	    return;

	/* Unselect all thumbs and delete all thumbs. */
	TListClear(tlist);

	if(tlist->initialized)
	{
#define DO_UNREF_GC	\
{ \
 if((*gc) != NULL) \
 { \
  GdkGC *tgc = *gc; \
  (*gc) = NULL; \
  gdk_gc_unref(tgc); \
 } \
}
#define DO_UNREF_PIXMAP	\
{ \
 if((*pixmap) != NULL) \
 { \
  GdkPixmap *tpm = *pixmap; \
  (*pixmap) = NULL; \
  gdk_pixmap_unref(tpm); \
 } \
}
#define DO_UNREF_ADJ	\
{ \
 if((*adj) != NULL) \
 { \
  GtkObject *tobj = GTK_OBJECT(*adj); \
  (*adj) = NULL; \
  gtk_object_unref(tobj); \
 } \
}
#define DO_DESTROY_WIDGET	\
{ \
 if((*w) != NULL) \
 { \
  GtkWidget *tmp_w = *w; \
  (*w) = NULL; \
  gtk_widget_destroy(tmp_w); \
 } \
}


	    w = &tlist->toplevel;
	    DO_DESTROY_WIDGET

	    pixmap = &tlist->list_pm;
	    DO_UNREF_PIXMAP

	    adj = &tlist->vadj;
	    DO_UNREF_ADJ
            adj = &tlist->hadj;
	    DO_UNREF_ADJ

	    gc = &tlist->gc;
	    DO_UNREF_GC
            gc = &tlist->selection_gc;
            DO_UNREF_GC

#undef DO_DESTROY_WIDGET
#undef DO_UNREF_PIXMAP
#undef DO_UNREF_ADJ
#undef DO_UNREF_GC
	}

	/* Deallocate structure itself. */
	g_free(tlist);
}

