#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../../include/fio.h"
#include "../../include/disk.h"
#include "../../include/prochandle.h"

#include "../guiutils.h"
#include "../piechart.h"

#include "../edvtypes.h"
#include "../edvdevices.h"

#include "../toolbar.h"
#include "ziptoolio.h"
#include "ziptool.h"
#include "ziptoolcb.h"

#include "config.h"

#include "../images/icon_drive_zip_48x48.xpm"
#include "../images/icon_mount_20x20.xpm"
#include "../images/icon_unmount_20x20.xpm"
#include "../images/icon_eject_20x20.xpm"
#include "../images/icon_power_20x20.xpm"
#include "../images/icon_security_20x20.xpm"
#include "../images/icon_browse_20x20.xpm"
#include "../images/icon_reload_20x20.xpm"
#include "../images/icon_fsck_20x20.xpm"
#include "../images/icon_exit_20x20.xpm"
#include "../images/icon_close_20x20.xpm"
#include "../images/icon_secure_20x20.xpm"
#include "../images/icon_insecure_20x20.xpm"


zip_tools_opid_struct *ZipToolsOPIDMatchByID(
        zip_tools_opid_struct **opid, gint total,
        gint id
);
zip_tools_opid_struct **ZipToolsOPIDListNew(
        zip_tools_struct *zt, gint *total
);
void ZipToolsOPIDListDelete(
        zip_tools_opid_struct **opid, gint total
);

void ZipToolsSetDevice(
	zip_tools_struct *zt, edv_device_struct *dev_ptr
);
void ZipToolsRefreshDevice(
        zip_tools_struct *zt, edv_device_struct *dev_ptr
);

zip_tools_struct *ZipToolsNew(edv_context_struct *ctx);
void ZipToolsUpdateMenus(zip_tools_struct *zt);
void ZipToolsSetBusy(zip_tools_struct *zt, gboolean is_busy);
void ZipToolsStatusMessage(
        zip_tools_struct *zt, const gchar *mesg,
        gboolean allow_gtk_iteration

);
void ZipToolsMap(zip_tools_struct *zt);
void ZipToolsUnmap(zip_tools_struct *zt);
void ZipToolsDelete(zip_tools_struct *zt);


#define ZIP_TOOLS_WIDTH		550
#define ZIP_TOOLS_HEIGHT	350

#define ZIP_TOOLS_BTN_WIDTH	(100 + (2 * 3))
#define ZIP_TOOLS_BTN_HEIGHT	(30 + (2 * 3))

#define EDV_STATUS_BAR_HEIGHT   26


/*
 *	Matches opid from list by the given id.
 */
zip_tools_opid_struct *ZipToolsOPIDMatchByID(
        zip_tools_opid_struct **opid, gint total,
	gint id
)
{
	gint i;
	zip_tools_opid_struct *opid_ptr;


	for(i = 0; i < total; i++)
	{
	    opid_ptr = opid[i];
	    if(opid_ptr == NULL)
		continue;

	    if(opid_ptr->opid == id)
		return(opid_ptr);
	}

	return(NULL);
}


/*
 *	Creates a new opid list.
 */
zip_tools_opid_struct **ZipToolsOPIDListNew(
        zip_tools_struct *zt, gint *total
)
{
	gint t = 0, id;
	zip_tools_opid_struct **opid = NULL, *opid_ptr;
        void (*func_cb)(GtkWidget *, gpointer);
        guint8 **icon_data;
        const gchar *name;
        const gchar *button_name;
        const gchar *tooltip;

#define DO_APPEND	\
{ \
 t++; \
 opid = (zip_tools_opid_struct **)g_realloc( \
  opid, t * sizeof(zip_tools_opid_struct *) \
 ); \
 opid[t - 1] = opid_ptr = g_malloc0(sizeof(zip_tools_opid_struct)); \
 opid_ptr->opid = id; \
 opid_ptr->zt = zt; \
 opid_ptr->func_cb = func_cb; \
 opid_ptr->func_enter_cb = ZipToolsMenuItemEnterCB; \
 opid_ptr->func_leave_cb = ZipToolsMenuItemLeaveCB; \
 opid_ptr->icon_data = icon_data; \
 opid_ptr->name = (name != NULL) ? g_strdup(name) : NULL; \
 opid_ptr->button_name = (button_name != NULL) ? g_strdup(button_name) : NULL; \
 opid_ptr->tooltip = (tooltip != NULL) ? g_strdup(tooltip) : NULL; \
 \
}

        id = ZIP_TOOLS_OPID_CLOSE;
        func_cb = ZipToolsCloseBtnCB;
        icon_data = (guint8 **)icon_close_20x20_xpm;
        name = "Close";
        button_name = name;
        tooltip = "Close this window";
        DO_APPEND

        id = ZIP_TOOLS_OPID_EXIT;
        func_cb = ZipToolsExitCB;
        icon_data = (guint8 **)icon_exit_20x20_xpm;
        name = "Exit";
        button_name = name;
        tooltip = "Close all windows belonging to this application";
        DO_APPEND

	id = ZIP_TOOLS_OPID_MOUNT;
	func_cb = ZipToolsMountCB;
	icon_data = (guint8 **)icon_mount_20x20_xpm;
	name = "Mount";
	button_name = name;
	tooltip = "Mount the device";
	DO_APPEND

        id = ZIP_TOOLS_OPID_UNMOUNT;
        func_cb = ZipToolsUnmountCB;
        icon_data = (guint8 **)icon_unmount_20x20_xpm;
        name = "Unmount";
        button_name = name;
        tooltip = "Unmount the device";
        DO_APPEND

        id = ZIP_TOOLS_OPID_SPIN_DOWN;
        func_cb = ZipToolsSpinDownCB;
        icon_data = (guint8 **)icon_power_20x20_xpm;
        name = "Spin Down";
        button_name = "SpinDn";
        tooltip = "Spin down the device";
        DO_APPEND

        id = ZIP_TOOLS_OPID_EJECT;
        func_cb = ZipToolsEjectCB;
        icon_data = (guint8 **)icon_eject_20x20_xpm;
        name = "Eject";
        button_name = name;
        tooltip = "Eject the media from the device";
        DO_APPEND

        id = ZIP_TOOLS_OPID_PASSWORD;
        func_cb = ZipToolsPasswordCB;
        icon_data = (guint8 **)icon_security_20x20_xpm;
        name = "Password";
        button_name = name;
        tooltip = "Lock/unlock the media";
        DO_APPEND

        id = ZIP_TOOLS_OPID_BROWSE;
        func_cb = ZipToolsBrowseCB;
        icon_data = (guint8 **)icon_browse_20x20_xpm;
        name = "Browse";
        button_name = name;
        tooltip = "Run file browser to browse media";
        DO_APPEND

        id = ZIP_TOOLS_OPID_REFRESH;
        func_cb = ZipToolsRefreshCB;
        icon_data = (guint8 **)icon_reload_20x20_xpm;
        name = "Refresh";
        button_name = name;
        tooltip = "Refresh device information";
        DO_APPEND

        id = ZIP_TOOLS_OPID_FSCK;
        func_cb = ZipToolsFSCKCB;
        icon_data = (guint8 **)icon_fsck_20x20_xpm;
        name = "File System Check";
        button_name = "FSCK";
        tooltip = "Run fsck to check the media";
        DO_APPEND

#undef DO_APPEND

	if(total != NULL)
	    *total = t;

	return(opid);
}

/*
 *	Deallocates the given list of opids.
 */
void ZipToolsOPIDListDelete(
        zip_tools_opid_struct **opid, gint total
)
{
	gint i;
	zip_tools_opid_struct *opid_ptr;

	for(i = 0; i < total; i++)
	{
	    opid_ptr = opid[i];
	    if(opid_ptr == NULL)
		continue;

            g_free(opid_ptr->name);
            g_free(opid_ptr->button_name);
	    g_free(opid_ptr->tooltip);
	    g_free(opid_ptr);
	}

	g_free(opid);
}

/*
 *	Sets the new device specified by dev_ptr on the zip tools
 *	window. If a device already exists then it will be deallocated
 *	first.
 *
 *	The given device will be taken by this function and should no
 *	longer be referenced by the calling function.
 */
void ZipToolsSetDevice(zip_tools_struct *zt, edv_device_struct *dev_ptr)
{
	GtkWidget *w;

	if(zt == NULL)
	    return;

	/* Deallocate existing device reference (if any) and transfer
	 * the given device reference to our zip tools window structure.
	 */
	EDVDeviceDelete(zt->device);
	zt->device = dev_ptr;

	/* Update title. */
	w = zt->toplevel;
	if(w != NULL)
	{
	    if(dev_ptr != NULL)
	    {
		gchar *buf;

		if((dev_ptr->name != NULL) && (dev_ptr->device_path != NULL))
		    buf = g_strdup_printf(
			"%s: %s (%s)",
			PROG_NAME, dev_ptr->name, dev_ptr->device_path
		    );
		else
		    buf = g_strdup_printf(
                        "%s: %s",
			PROG_NAME,
			(dev_ptr->name != NULL) ? dev_ptr->name :
			    dev_ptr->device_path
		    );
		gtk_window_set_title(GTK_WINDOW(w), buf);
		g_free(buf);
	    }
	    else
	    {
		gtk_window_set_title(GTK_WINDOW(w), PROG_NAME);
	    }
	}
}

/*
 *	Rescans the given device (or the device on the zip tools window
 *	structure if dev_ptr is NULL) and updates values on the zt
 *	structure.
 *
 *	Menus will not be updated.
 */
void ZipToolsRefreshDevice(
        zip_tools_struct *zt, edv_device_struct *dev_ptr
)
{
	gint border_minor = 2, border_major = 5;
	GtkWidget *w, *parent, *parent2, *parent3;
	edv_device_struct *device_list[1];


	if(zt == NULL)
	    return;

	if(dev_ptr == NULL)
	    dev_ptr = zt->device;
	if(dev_ptr == NULL)
	    return;

	/* Check device mount states and capacities. */
	device_list[0] = dev_ptr;
	EDVDevicesListUpdateMountStates(device_list, 1);
	EDVDevicesListUpdateStats(device_list, 1);

	/* Check device lock state. */
	zt->device_locked = ZipToolsDeviceIsProtected(dev_ptr);


	/* Begin updating display widgets by recreating them. */

	/* Destroy existing display widgets. */
	PieChartDelete(zt->capacity_piechart);
	zt->capacity_piechart = NULL;
	w = zt->display_child;
	if(w != NULL)
	    gtk_widget_destroy(w);
	zt->display_child = NULL;

	/* Get display parent and begin recreating display widgets. */
	parent = zt->display_parent;
	if(parent != NULL)
	{
	    gchar *buf, *buf2;
	    pie_chart_struct *pc;
	    GdkColor c;
	    GdkPixmap *pixmap = NULL;
	    GdkBitmap *mask = NULL;
	    GtkAdjustment *adj;
	    gulong adj_division = 10;
	    zip_tools_opid_struct *opid_ptr;


	    /* Child hbox, to separate display into "columns". */
	    zt->display_child = w = gtk_hbox_new(TRUE, 0);
	    gtk_container_border_width(GTK_CONTAINER(w), border_major);
	    gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	    gtk_widget_show(w);
	    parent = w;

/*
	    w = gtk_hbox_new(FALSE, border_major);
            gtk_box_pack_start(GTK_BOX(parent), w, TRUE, FALSE, 0);
            gtk_widget_show(w);
            parent = w;
 */
	    /* Left side column vbox. */
	    w = gtk_vbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent2 = w;

	    /* Hbox for device icon and label. */
	    w = gtk_hbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
            gtk_widget_show(w);
            parent3 = w;

	    /* Get pixmap and mask to display device icon. */
            EDVDeviceRealize(dev_ptr, FALSE);
	    if(dev_ptr->is_mounted &&
	       (EDV_DEVICE_TOTAL_ICON_STATES > EDV_DEVICE_ICON_STATE_STANDARD)
	    )
	    {
		gint i = EDV_DEVICE_ICON_STATE_STANDARD;
		pixmap = dev_ptr->large_pixmap[i];
		mask = dev_ptr->large_mask[i];
		if(pixmap == NULL)
		{
                    pixmap = dev_ptr->medium_pixmap[i];
                    mask = dev_ptr->medium_mask[i];
		}
		if(pixmap == NULL)
		{
                    pixmap = dev_ptr->small_pixmap[i];
                    mask = dev_ptr->small_mask[i];
                }
	    }
	    else if(EDV_DEVICE_TOTAL_ICON_STATES > EDV_DEVICE_ICON_STATE_UNMOUNTED)
	    {
		gint i = EDV_DEVICE_ICON_STATE_UNMOUNTED;
                pixmap = dev_ptr->large_pixmap[i];
                mask = dev_ptr->large_mask[i];
                if(pixmap == NULL)
                {
                    pixmap = dev_ptr->medium_pixmap[i];
                    mask = dev_ptr->medium_mask[i];
                }
                if(pixmap == NULL)
                {
                    pixmap = dev_ptr->small_pixmap[i];
                    mask = dev_ptr->small_mask[i];
                }
		i = EDV_DEVICE_ICON_STATE_STANDARD;
                if(pixmap == NULL)
                {
                    pixmap = dev_ptr->large_pixmap[i];
                    mask = dev_ptr->large_mask[i];
                }
                if(pixmap == NULL)
                {
                    pixmap = dev_ptr->medium_pixmap[i];
                    mask = dev_ptr->medium_mask[i];
                }
                if(pixmap == NULL)
                {
                    pixmap = dev_ptr->small_pixmap[i];
                    mask = dev_ptr->small_mask[i];
                }
	    }
	    if(pixmap != NULL)
	    {
		w = gtk_pixmap_new(pixmap, mask);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
	    }
	    /* Name label. */
	    if(dev_ptr->name != NULL)
		buf = g_strdup_printf(
		    "%s\n%s",
		    dev_ptr->name,
		    (dev_ptr->device_path != NULL) ?
			dev_ptr->device_path : "*path not set*"
		);
	    else
		buf = g_strdup_printf(
                    "%s",
                    (dev_ptr->device_path != NULL) ?
                        dev_ptr->device_path : "*path not set*"
                );
	    w = gtk_label_new(buf);
	    g_free(buf);
	    gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);


            /* FIle system type. */
            w = gtk_hbox_new(FALSE, border_minor);
            gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
            gtk_widget_show(w);
            parent3 = w;
	    buf = g_strdup_printf(
		"File System Type: %s",
		EDVDeviceGetFSStringFromNumber(dev_ptr->fs_type)
	    );
            w = gtk_label_new(buf);
            g_free(buf);
            gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
            gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
            gtk_widget_show(w);


	    /* Usage pie chart. */
           if(dev_ptr->is_mounted)
	   {
	    adj = (GtkAdjustment *)gtk_adjustment_new(
		0.0, 0.0,
		(gfloat)(dev_ptr->blocks_total / adj_division),
		0.0, 0.0, 0.0
	    );
	    c.red = 0.0 * (guint16)-1;
	    c.green = 0.0 * (guint16)-1;
	    c.blue = 1.0 * (guint16)-1;
	    buf = g_strdup_printf(
		"%s: %ld kb",
#ifdef PROG_LANGUAGE_ENGLISH
		"Total Capacity",
#endif
#ifdef PROG_LANGUAGE_SPANISH
		"La Capacidad Total",
#endif
#ifdef PROG_LANGUAGE_FRENCH
		"Capacit Totale",
#endif
		dev_ptr->blocks_total
	    );
	    buf2 = g_strdup_printf(
		"%ld kb",
		dev_ptr->blocks_free
	    );
	    zt->capacity_piechart = pc = PieChartNew(
		adj, &c, 110, 70,
		NULL, buf,
#ifdef PROG_LANGUAGE_ENGLISH
		"Free:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
		"Libre:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
		"Libre:",
#endif
		buf2
	    );
	    gtk_object_unref(GTK_OBJECT(adj));
	    g_free(buf);
	    g_free(buf2);
	    w = pc->toplevel;
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);

	    c.red = 0.0 * (guint16)-1;
	    c.green = 1.0 * (guint16)-1;
	    c.blue = 1.0 * (guint16)-1;
	    adj = (GtkAdjustment *)gtk_adjustment_new(
		0.0, 0.0,
		(gfloat)(
		    (dev_ptr->blocks_available) / adj_division),
                0.0, 0.0, 0.0
            );
	    buf = g_strdup_printf(
		"%ld kb", dev_ptr->blocks_available
	    );
	    PieChartValueAdd(
		pc, adj, &c,
#ifdef PROG_LANGUAGE_ENGLISH
		"Free & Available:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
		"Libre & Disponible:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
		"Libre & Disponible:",
#endif
		buf
	    );
	    gtk_object_unref(GTK_OBJECT(adj));
	    g_free(buf);

	    c.red = 1.0 * (guint16)-1;
	    c.green = 0.0 * (guint16)-1;
	    c.blue = 1.0 * (guint16)-1;
	    adj = (GtkAdjustment *)gtk_adjustment_new(
		0.0, 0.0,
		(gfloat)(
 (dev_ptr->blocks_total - dev_ptr->blocks_free) / adj_division),
		0.0, 0.0, 0.0
	    );
	    buf = g_strdup_printf(
		"%ld kb",
                dev_ptr->blocks_total - dev_ptr->blocks_free
	    );
	    PieChartValueAdd(
		pc, adj, &c,
#ifdef PROG_LANGUAGE_ENGLISH
                "Used:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
                "Usado:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
                "Utilis:",
#endif
                buf
	    );
	    gtk_object_unref(GTK_OBJECT(adj));
	    g_free(buf);
	   }
	   else	/* Is mounted. */
	   {
		w = gtk_label_new("(not mounted)");
		gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
	   }	/* Not mounted. */

            /* Right side column vbox. */
            w = gtk_vbox_new(FALSE, border_minor);
            gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
            gtk_widget_show(w);
            parent2 = w;

	    /* Hbox for locked/unlocked button. */
            w = gtk_hbox_new(FALSE, border_minor);
            gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
            gtk_widget_show(w);
            parent3 = w;
	    /* Locked icon and label. */
	    if(zt->device_locked)
	    {
		w = GUIButtonPixmapLabelH(
		    (u_int8_t **)icon_secure_20x20_xpm,
		    (zt->device_locked == 2) ?
			"Locked With Password" : "Write Protected",
		    NULL
		);
		gtk_widget_set_usize(w, -1, ZIP_TOOLS_BTN_HEIGHT);
		GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
		gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
	    }
	    else
	    {
                w = GUIButtonPixmapLabelH(
                    (u_int8_t **)icon_insecure_20x20_xpm,
		    "Not Protected",
		    NULL
                );
                gtk_widget_set_usize(w, -1, ZIP_TOOLS_BTN_HEIGHT);
                GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
		gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
                gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
                gtk_widget_show(w);
	    }
	    opid_ptr = ZipToolsOPIDMatchByID(
		zt->opid, zt->total_opids,
		ZIP_TOOLS_OPID_PASSWORD
	    );
	    if(opid_ptr != NULL)
	    {
                gtk_signal_connect(
                    GTK_OBJECT(w), "clicked",
                    GTK_SIGNAL_FUNC(opid_ptr->func_cb), opid_ptr->zt
                );
                gtk_signal_connect(
                    GTK_OBJECT(w), "enter_notify_event",
                    GTK_SIGNAL_FUNC(opid_ptr->func_enter_cb), opid_ptr
                );
                gtk_signal_connect(
                    GTK_OBJECT(w), "leave_notify_event",
                    GTK_SIGNAL_FUNC(opid_ptr->func_leave_cb), opid_ptr
                );
		if(opid_ptr->tooltip != NULL)
		    GUISetWidgetTip(w, opid_ptr->tooltip);
	    }


            /* Hbox for browse button. */
            w = gtk_hbox_new(FALSE, border_minor);
            gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
            gtk_widget_show(w);
            parent3 = w;
            /* Browse button. */
	    buf = g_strdup_printf(
		"Mount Path: %s",
		dev_ptr->mount_path
	    );
	    w = GUIButtonPixmapLabelH(
		(u_int8_t **)icon_browse_20x20_xpm, buf, NULL
	    );
	    g_free(buf);
            gtk_widget_set_usize(w, -1, ZIP_TOOLS_BTN_HEIGHT);
            GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
            gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
            gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
            gtk_widget_show(w);
            opid_ptr = ZipToolsOPIDMatchByID(
                zt->opid, zt->total_opids,
                ZIP_TOOLS_OPID_BROWSE
            );
            if(opid_ptr != NULL)
            {
                gtk_signal_connect(
                    GTK_OBJECT(w), "clicked",
                    GTK_SIGNAL_FUNC(opid_ptr->func_cb), opid_ptr->zt
                );
                gtk_signal_connect(
                    GTK_OBJECT(w), "enter_notify_event",
                    GTK_SIGNAL_FUNC(opid_ptr->func_enter_cb), opid_ptr
                );
                gtk_signal_connect(
                    GTK_OBJECT(w), "leave_notify_event",
                    GTK_SIGNAL_FUNC(opid_ptr->func_leave_cb), opid_ptr
                );
                if(opid_ptr->tooltip != NULL)
                    GUISetWidgetTip(w, opid_ptr->tooltip);
            }
	}

	gtk_widget_queue_resize(zt->toplevel);
}



/*
 *	Creates a new zip tools window.
 */
zip_tools_struct *ZipToolsNew(edv_context_struct *ctx)
{
        gint /* border_major = 5, */ border_minor = 2;
	gint i;
        GdkWindow *window;
        GtkWidget	*w, *parent, *parent2, *parent3,
			*main_vbox;
        GtkAccelGroup *accelgrp;
	zip_tools_opid_struct *opid_ptr;
	toolbar_item_struct **tb_item;
	gint total_tb_items;
        zip_tools_struct *zt = (zip_tools_struct *)g_malloc0(
            sizeof(zip_tools_struct)
        );
        if(zt == NULL)
            return(NULL);


        /* Reset values. */
        zt->initialized = TRUE;
        zt->map_state = FALSE;
	zt->ctx = ctx;
	zt->busy_count = 0;
	zt->opid = ZipToolsOPIDListNew(
	    zt, &zt->total_opids
	);
	zt->device = NULL;
	zt->device_locked = 0;

	/* Load cursors. */
	zt->busy_cur = gdk_cursor_new(GDK_WATCH);
	zt->text_cur = gdk_cursor_new(GDK_XTERM);

        /* Keyboard accelerator group. */
        zt->accelgrp = accelgrp = gtk_accel_group_new();

        /* Toplevel. */
        zt->toplevel = w = gtk_window_new(GTK_WINDOW_DIALOG);
        gtk_widget_set_usize(w, ZIP_TOOLS_WIDTH, ZIP_TOOLS_HEIGHT);
        gtk_widget_realize(w);
        window = w->window;
        if(window != NULL)
        {
            gdk_window_set_decorations(
                window,
                GDK_DECOR_TITLE | GDK_DECOR_MENU | GDK_DECOR_MINIMIZE
            );
            gdk_window_set_functions(
                window,
                GDK_FUNC_MOVE | GDK_FUNC_MINIMIZE | GDK_FUNC_CLOSE
            );
            GUISetWMIcon(window, (u_int8_t **)icon_drive_zip_48x48_xpm);
        }
        gtk_window_set_title(GTK_WINDOW(w), PROG_NAME);
        gtk_signal_connect(
            GTK_OBJECT(w), "delete_event",
            GTK_SIGNAL_FUNC(ZipToolsCloseCB), zt
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "destroy",
            GTK_SIGNAL_FUNC(ZipToolsDestroyCB), zt
        );
        gtk_container_border_width(GTK_CONTAINER(w), 0);
        gtk_accel_group_attach(accelgrp, GTK_OBJECT(w));
        parent = w;

        /* Main vbox. */
        zt->main_vbox = main_vbox = w = gtk_vbox_new(FALSE, 0);
        gtk_container_add(GTK_CONTAINER(parent), w);
        gtk_widget_show(w);
        parent = w;


	/* Generate list of tool bar items. */
	total_tb_items = 0;
	tb_item = NULL;
#define DO_APPEND_BUTTON	\
{ \
 if(opid_ptr != NULL) \
 { \
  total_tb_items++; \
  tb_item = (toolbar_item_struct **)g_realloc( \
   tb_item, total_tb_items * sizeof(toolbar_item_struct *) \
  ); \
\
  tb_item[total_tb_items - 1] = ToolBarItemNew( \
   TOOLBAR_ITEM_BUTTON, opid_ptr->button_name, \
   opid_ptr->icon_data, opid_ptr->tooltip, \
   opid_ptr->opid, \
   ZipToolsOPIDCB, opid_ptr, \
   ZipToolsOPIDEnterCB, opid_ptr, \
   ZipToolsOPIDLeaveCB, opid_ptr \
  ); \
 } \
}
#define DO_APPEND_SEPARATOR	\
{ \
 total_tb_items++; \
 tb_item = (toolbar_item_struct **)g_realloc( \
  tb_item, total_tb_items * sizeof(toolbar_item_struct *) \
 ); \
\
 tb_item[total_tb_items - 1] = ToolBarItemNew( \
  TOOLBAR_ITEM_SEPARATOR, NULL, NULL, NULL, 0, \
  NULL, NULL, NULL, NULL, NULL, NULL \
 ); \
}

        opid_ptr = ZipToolsOPIDMatchByID(
            zt->opid, zt->total_opids,
            ZIP_TOOLS_OPID_MOUNT
        );
	DO_APPEND_BUTTON
        opid_ptr = ZipToolsOPIDMatchByID(
            zt->opid, zt->total_opids,
            ZIP_TOOLS_OPID_UNMOUNT
        );
        DO_APPEND_BUTTON
	DO_APPEND_SEPARATOR
        opid_ptr = ZipToolsOPIDMatchByID(
            zt->opid, zt->total_opids,
            ZIP_TOOLS_OPID_SPIN_DOWN
        );
        DO_APPEND_BUTTON
        opid_ptr = ZipToolsOPIDMatchByID(
            zt->opid, zt->total_opids,
            ZIP_TOOLS_OPID_EJECT
        );
        DO_APPEND_BUTTON
        opid_ptr = ZipToolsOPIDMatchByID(
            zt->opid, zt->total_opids,
            ZIP_TOOLS_OPID_PASSWORD
        );
        DO_APPEND_BUTTON
        DO_APPEND_SEPARATOR
        opid_ptr = ZipToolsOPIDMatchByID(
            zt->opid, zt->total_opids,
            ZIP_TOOLS_OPID_FSCK
        );
        DO_APPEND_BUTTON
        DO_APPEND_SEPARATOR
        opid_ptr = ZipToolsOPIDMatchByID(
            zt->opid, zt->total_opids,
            ZIP_TOOLS_OPID_REFRESH
        );
        DO_APPEND_BUTTON
	DO_APPEND_SEPARATOR
        opid_ptr = ZipToolsOPIDMatchByID(
            zt->opid, zt->total_opids,
            ZIP_TOOLS_OPID_CLOSE
        );
        DO_APPEND_BUTTON


	/* Create tool bar. */
	zt->tool_bar_handle = w = gtk_handle_box_new();
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	zt->tool_bar = ToolBarNew(
            tb_item, total_tb_items, parent2,
            TOOLBAR_DISPLAY_PICTURES_AND_TEXT,
	    TOOLBAR_RELIEF_NONE, FALSE
        );
        ToolBarMap(zt->tool_bar);

        /* Deallocate tool bar items list, it is no longer needed. */
        for(i = 0; i < total_tb_items; i++)
            ToolBarItemDelete(tb_item[i]);
        g_free(tb_item);
        tb_item = NULL;
        total_tb_items = 0;
#undef DO_APPEND_BUTTON
#undef DO_APPEND_SEPARATOR



	/* Event box to hold display parent. */
	zt->display_event_box = w = gtk_event_box_new();
        gtk_widget_add_events(
            w,
	    GDK_EXPOSURE_MASK |
            GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "expose_event",
            GTK_SIGNAL_FUNC(ZipToolsExposeEventCB), zt
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "button_press_event",
            GTK_SIGNAL_FUNC(ZipToolsButtonPressEventCB), zt
        );
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_realize(w);
        gtk_widget_show(w);
	parent2 = w;

	/* Set style for event box. */
	if(1)
	{
#if 0
	    const gchar *rcstr = "\n\
style \"ziptool-base-style\" { \n\
 bg[NORMAL] = \"#ffffc0\"\n\
 fg[NORMAL] = \"#000000\"\n\
}\n\
widget \"*ziptool-base\" style \"ziptool-base-style\"\n\
";
	    gtk_rc_parse_string(rcstr);

#endif
	}

        /* Display parent vbox. */
	zt->display_parent = w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent2), w);
        gtk_widget_show(w);
        parent2 = w;

	zt->display_child = NULL;
	zt->capacity_piechart = NULL;




        /* Status bar frame. */
        w = gtk_frame_new(NULL);
        gtk_widget_set_usize(w, -1, EDV_STATUS_BAR_HEIGHT);
        gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_OUT);
        gtk_container_border_width(GTK_CONTAINER(w), 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent2 = w;

        /* Table in main frame. */
        w = gtk_table_new(1, 1, FALSE);
        gtk_container_add(GTK_CONTAINER(parent2), w);
        gtk_widget_show(w);
        parent2 = w;

        /* Frame for label. */
        w = gtk_frame_new(NULL);
        gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
        gtk_table_attach(
            GTK_TABLE(parent2), w,
            0, 1, 0, 1,
            GTK_SHRINK | GTK_EXPAND | GTK_FILL,
            GTK_SHRINK,
            border_minor, border_minor
        );
        gtk_widget_show(w);
        parent3 = w;
	/* Hbox in frame for label. */
	w = gtk_hbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent3), w);
	gtk_widget_show(w);
        parent3 = w;
	/* Label. */
	zt->status_label = w = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, border_minor);
	gtk_widget_show(w);



        /* Display area menu. */
        if(1)
        {
#define DO_ADD_MENU_ITEM_LABEL	\
if(opid_ptr != NULL) \
{ \
 w = GUIMenuItemCreate( \
  menu, GUI_MENU_ITEM_TYPE_LABEL, accel_group, \
  opid_ptr->icon_data, opid_ptr->name, \
  accel_key, accel_mods, (gpointer *)&fw, \
  zt, opid_ptr->func_cb \
 ); \
 GUISetMenuItemCrossingCB( \
  w, \
  (gpointer)opid_ptr->func_enter_cb, opid_ptr, \
  (gpointer)opid_ptr->func_leave_cb, opid_ptr \
 ); \
}
#define DO_ADD_MENU_SEP		\
{ \
 w = GUIMenuItemCreate( \
  menu, GUI_MENU_ITEM_TYPE_SEPARATOR, NULL, \
  NULL, NULL, 0, 0, NULL, \
  NULL, NULL \
 ); \
}

            GtkWidget *menu = (GtkWidget *)GUIMenuCreate();
            gint accel_key;
            guint accel_mods;
            gpointer accel_group = NULL;
            gpointer fw;

            zt->menu = menu;

            opid_ptr = ZipToolsOPIDMatchByID(
                zt->opid, zt->total_opids,
                ZIP_TOOLS_OPID_MOUNT
            );
	    accel_key = 0;
	    accel_mods = 0;
	    DO_ADD_MENU_ITEM_LABEL
	    zt->mount_mi = w;

            opid_ptr = ZipToolsOPIDMatchByID(
                zt->opid, zt->total_opids,
                ZIP_TOOLS_OPID_UNMOUNT
            );
            accel_key = 0;
            accel_mods = 0;
            DO_ADD_MENU_ITEM_LABEL
            zt->unmount_mi = w;

            DO_ADD_MENU_SEP

            opid_ptr = ZipToolsOPIDMatchByID(
                zt->opid, zt->total_opids,
                ZIP_TOOLS_OPID_SPIN_DOWN
            );
            accel_key = 0;
            accel_mods = 0;
            DO_ADD_MENU_ITEM_LABEL
            zt->spin_down_mi = w;

            opid_ptr = ZipToolsOPIDMatchByID(
                zt->opid, zt->total_opids,
                ZIP_TOOLS_OPID_EJECT
            );
            accel_key = 0;
            accel_mods = 0;
            DO_ADD_MENU_ITEM_LABEL
            zt->eject_mi = w;

            opid_ptr = ZipToolsOPIDMatchByID(
                zt->opid, zt->total_opids,
                ZIP_TOOLS_OPID_PASSWORD
            );
            accel_key = 0;
            accel_mods = 0;
            DO_ADD_MENU_ITEM_LABEL
            zt->password_mi = w;

	    DO_ADD_MENU_SEP

            opid_ptr = ZipToolsOPIDMatchByID(
                zt->opid, zt->total_opids,
                ZIP_TOOLS_OPID_FSCK
            );
            accel_key = 0;
            accel_mods = 0;
            DO_ADD_MENU_ITEM_LABEL
	    zt->fsck_mi = w;

            DO_ADD_MENU_SEP

            opid_ptr = ZipToolsOPIDMatchByID(
                zt->opid, zt->total_opids,
                ZIP_TOOLS_OPID_REFRESH
            );
            accel_key = 0;
            accel_mods = 0;
            DO_ADD_MENU_ITEM_LABEL
            zt->refresh_mi = w;

#undef DO_ADD_MENU_ITEM_LABEL
#undef DO_ADD_MENU_SEP
        }





	ZipToolsUpdateMenus(zt);

	return(zt);
}

/*
 *	Updates widgets to reflect local data values.
 */
void ZipToolsUpdateMenus(zip_tools_struct *zt)
{
        gboolean sensitive;
	GtkWidget *w;
	toolbar_struct *tb;
	edv_device_struct *dev_ptr;


	if(zt == NULL)
	    return;

        if(!zt->initialized)
            return;

#define DO_SET_SENSITIVE        \
{ \
 if(w != NULL) \
  gtk_widget_set_sensitive(w, sensitive); \
}

	tb = zt->tool_bar;
	dev_ptr = zt->device;

	/* Mount. */
	sensitive = (dev_ptr != NULL) ? !dev_ptr->is_mounted : TRUE;
	ToolBarItemSetSensitiveID(
	    tb, ZIP_TOOLS_OPID_MOUNT, sensitive
	);
        w = zt->mount_mi;
        DO_SET_SENSITIVE

        /* Unmount. */
        sensitive = (dev_ptr != NULL) ? dev_ptr->is_mounted : FALSE;
        ToolBarItemSetSensitiveID(
            tb, ZIP_TOOLS_OPID_UNMOUNT, sensitive
        );
        w = zt->unmount_mi;
        DO_SET_SENSITIVE

        /* Eject. */
        sensitive = (dev_ptr != NULL) ? !dev_ptr->is_mounted : TRUE;
        ToolBarItemSetSensitiveID(
            tb, ZIP_TOOLS_OPID_EJECT, sensitive
        );
	w = zt->eject_mi;
	DO_SET_SENSITIVE

	/* Lock/unlock. */
        ToolBarItemUpdateByID(
            tb,
            ZIP_TOOLS_OPID_PASSWORD,
            (zt->device_locked == 0) ? "Lock" : "Unlock",
            NULL,
            NULL
        );
        sensitive = (dev_ptr != NULL) ? !dev_ptr->is_mounted : TRUE;
        ToolBarItemSetSensitiveID(
            tb, ZIP_TOOLS_OPID_PASSWORD, sensitive
        );

	w = zt->password_mi,
	GUIMenuItemSetLabel(
	    w,
            (zt->device_locked == 0) ? "Lock" : "Unlock"
	);
	DO_SET_SENSITIVE

	/* FSCK. */
	sensitive = (dev_ptr != NULL) ? !dev_ptr->is_mounted : TRUE;
        ToolBarItemSetSensitiveID(
            tb, ZIP_TOOLS_OPID_FSCK, sensitive
        );
        w = zt->fsck_mi;
        DO_SET_SENSITIVE


#undef DO_SET_SENSITIVE
}

/*
 *      Marks the zip tools window as busy or ready.
 */
void ZipToolsSetBusy(zip_tools_struct *zt, gboolean is_busy)
{
        GdkCursor *cur;
        GtkWidget *w;


        if(zt == NULL)
            return;

        if(!zt->initialized)
            return;

        w = zt->toplevel;
        if(w != NULL)
        {
            if(is_busy)
            {
                /* Increase busy count. */
                zt->busy_count++;

                /* If already busy then don't change anything. */
                if(zt->busy_count > 1)
                    return;

                cur = zt->busy_cur;
            }
            else
            {
                /* Reduce busy count. */
                zt->busy_count--;
                if(zt->busy_count < 0)
                    zt->busy_count = 0;

                /* If still busy do not change anything. */
                if(zt->busy_count > 0)
                    return;

                cur = NULL;	/* Use default cursor. */
            }

            /* Update toplevel window's cursor. */
            if(w->window != NULL)
            {
                gdk_window_set_cursor(w->window, cur);
                gdk_flush();
            }
        }
}

/*
 *	Updates the status bar message.
 */
void ZipToolsStatusMessage(
	zip_tools_struct *zt, const gchar *mesg,
	gboolean allow_gtk_iteration

)
{
	GtkWidget *w;

        if(zt == NULL)
            return;

	w = zt->status_label;
	if(w != NULL)
	    gtk_label_set_text(
		GTK_LABEL(w), (mesg != NULL) ? mesg : ""
	    );

	while((gtk_events_pending() > 0) && allow_gtk_iteration)
	    gtk_main_iteration();
}

/*
 *	Maps the zip tools window.
 */
void ZipToolsMap(zip_tools_struct *zt)
{
        GtkWidget *w;

        if(zt == NULL)
            return;

        if(!zt->initialized)
            return;

	w = zt->toplevel;
	gtk_widget_show_raise(w);
	zt->map_state = TRUE;
}

/*
 *      Unmaps the zip tools window.
 */
void ZipToolsUnmap(zip_tools_struct *zt)
{
        GtkWidget *w;

        if(zt == NULL)
            return;

        if(!zt->initialized)
            return;

        w = zt->toplevel;
        if(w == NULL)
            return;

        if(zt->map_state)
        {
            zt->map_state = FALSE;
            gtk_widget_hide(w);
        }
}

/*
 *      Destroys zip tools widgets and deallocates resources including
 *      the structure itself.
 */
void ZipToolsDelete(zip_tools_struct *zt)
{
	GdkCursor **cur;
	GtkWidget **w;


        if(zt == NULL)
            return;

        if(zt->initialized)
        {
#define DO_DESTROY_CURSOR	\
{ \
 if((*cur) != NULL) \
 { \
  GdkCursor *tmp_cur = *cur; \
  (*cur) = NULL; \
  gdk_cursor_destroy(tmp_cur); \
 } \
}
#define DO_DESTROY_WIDGET       \
{ \
 if((*w) != NULL) \
 { \
  GtkWidget *tmp_w = *w; \
  (*w) = NULL; \
  gtk_widget_destroy(tmp_w); \
 } \
}
            w = &zt->menu;
	    zt->mount_mi = NULL;
            zt->unmount_mi = NULL;
	    zt->spin_down_mi = NULL;
            zt->eject_mi = NULL;
            zt->password_mi = NULL;
	    zt->fsck_mi = NULL;
            zt->refresh_mi = NULL;
            DO_DESTROY_WIDGET

	    ToolBarDelete(zt->tool_bar);
	    zt->tool_bar = NULL;
	    w = &zt->tool_bar_handle;
	    DO_DESTROY_WIDGET

	    w = &zt->status_label;
	    DO_DESTROY_WIDGET

	    PieChartDelete(zt->capacity_piechart);
	    zt->capacity_piechart = NULL;
            w = &zt->display_child;
            DO_DESTROY_WIDGET
	    w = &zt->display_parent;
            DO_DESTROY_WIDGET
	    w = &zt->display_event_box;
	    DO_DESTROY_WIDGET

            w = &zt->main_vbox;
            DO_DESTROY_WIDGET
	    w = &zt->toplevel;
            DO_DESTROY_WIDGET

            if(zt->accelgrp != NULL)
            {
                gtk_accel_group_unref(zt->accelgrp);
                zt->accelgrp = NULL;
            }

            ZipToolsOPIDListDelete(zt->opid, zt->total_opids);
            zt->opid = NULL;
            zt->total_opids = 0;

	    cur = &zt->busy_cur;
	    DO_DESTROY_CURSOR
            cur = &zt->text_cur;
            DO_DESTROY_CURSOR

	    EDVDeviceDelete(zt->device);

#undef DO_DESTROY_WIDGET
#undef DO_DESTROY_CURSOR
        }

        /* Deallocate the structure itself. */
        g_free(zt);
}

