#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/time.h>
#include <signal.h>
#include <unistd.h>
#include <gtk/gtk.h>

#include "../../include/string.h"
#include "../../include/fio.h"
#include "../../include/disk.h"
#include "../../include/prochandle.h"

#include "../guiutils.h"
#include "../cdialog.h"
#include "../pdialog.h"
#include "../fb.h"
#include "../progressdialog.h"

#include "../edvtypes.h"
#include "../edvcfglist.h"
#include "../edvutils.h"
#include "../lib/endeavour2.h"

#include "downloadcfgio.h"
#include "config.h"

#include "../images/icon_planet_32x32.xpm"

#include "../images/icon_download_file_32x32.xpm"
#include "../images/icon_download_file_48x48.xpm"

#include "../images/pdi_file01_20x20.xpm"
#include "../images/pdi_file02_20x20.xpm"
#include "../images/pdi_file03_20x20.xpm"
#include "../images/pdi_file04_20x20.xpm"
#include "../images/pdi_file05_20x20.xpm"
#include "../images/pdi_file06_20x20.xpm"
#include "../images/pdi_folder_32x32.xpm"
#include "../images/pdi_folderfile_32x32.xpm"

#define DOWNLOAD_PROG	"/usr/bin/wget"


static glong CurrentMillitime(void);

static void DownloadSignalCB(int s);
static gchar *DownloadBrowseTargetPathCB(
	gpointer d, gpointer data, gint prompt_num
);

static gulong DownloadTryGetTotalLength(FILE *fp);
static gint DownloadMonitor(
	edv_context_struct *ctx, gint p,
	const gchar *src, const gchar *tar,
	const gchar *stdout_file
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *      Returns the current time in milliseconds.
 */
static glong CurrentMillitime(void)
{
#ifdef __MSW__
	SYSTEMTIME t;   /* Current time of day structure */

	GetSystemTime(&t);
	return(
	    (glong)(
		(((((t.wHour * 60.0) + t.wMinute) * 60) + t.wSecond) * 1000) +
		t.wMilliseconds
	    )
	);
#else
	struct timeval tv[1];

	if(gettimeofday(tv, NULL) < 0)
	    return(-1);

	return(((tv->tv_sec % 86400) * 1000) + (tv->tv_usec / 1000));
#endif
}

/*
 *      UNIX signal callback.
 */
static void DownloadSignalCB(int s)
{
	switch(s)
	{
	  case SIGINT:
	  case SIGTERM:
	  case SIGSEGV:
	    exit(1);
	    break;
	}
}

/*
 *	Prompt dialog browse target path callback.
 */
static gchar *DownloadBrowseTargetPathCB(
	gpointer d, gpointer data, gint prompt_num
)
{
	gboolean status;
	fb_type_struct **ftype = NULL, *ftype_rtn = NULL;
	gint total_ftypes = 0;
	gchar **path_rtn = NULL;
	gint total_path_rtns = 0;
	GtkWidget *toplevel = NULL;

	if(FileBrowserIsQuery())
	    return(NULL);

	/* Create file types list */
	FileBrowserTypeListNew(
	    &ftype, &total_ftypes,
	    "*.*", "All files"
	);

	/* Query user for path */
	FileBrowserSetTransientFor(toplevel);
	status = FileBrowserGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
"El Sendero Selecto",
"Selecto",
"Cancele",
#elif defined(PROG_LANGUAGE_FRENCH)
"Choisir Le Sentier",
"Choisir",
"Annuler",
#elif defined(PROG_LANGUAGE_GERMAN)
"Erlesener Pfad",
"Erlesen.",
"Heben",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scegliere Il Sentiero",
"Scegliere",
"Annullare",
#elif defined(PROG_LANGUAGE_DUTCH)
"Uitgezocht Pad",
"Uitgezocht",
"Annuleer",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Selecione Caminho",
"Selecione",
"Cancelamento",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Utvalgt Sti",
"Utvalgt",
"Kanseller",
#else
"Select Path",
"Select",
"Cancel",
#endif
	    PDialogGetPromptValue(prompt_num),
	    ftype, total_ftypes,
	    &path_rtn, &total_path_rtns,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	/* Delete file types list */
	FileBrowserDeleteTypeList(ftype, total_ftypes);

	/* Reset due to possible file related change */
	FileBrowserReset();

	return((total_path_rtns > 0) ? path_rtn[0] : NULL);
}

/*
 *	Reads the first 1024 bytes (or less) of the file specified by
 *	fp, searching for the total length header (prefixed by the
 *	string "\nLength:".
 *
 *	This function may return (gulong)-1, indicating that a permanent
 *	error has occured and not to call it anymore.  It normally
 *	returns the total length (or 0 if it was not found).
 */
static gulong DownloadTryGetTotalLength(FILE *fp)
{
	gulong total_length = 0;
	gchar *buf;
	gint buf_len, bytes_read;
	struct stat stat_buf;


	if(fp == NULL)
	    return((gulong)-1);

	if(fstat(fileno(fp), &stat_buf))
	    return((gulong)-1);

	buf_len = stat_buf.st_size;
	if(buf_len <= 0)
	    return((gulong)-1);

	/* Allocate read buffer */
	buf = (gchar *)g_malloc((buf_len + 1) * sizeof(gchar));
	if(buf == NULL)
	    return((gulong)-1);

	/* Read the beginning of the file, searching for the content
	 * length "\nLength:" string.
	 */
	rewind(fp);
	bytes_read = fread(buf, sizeof(gchar), buf_len, fp);
	if(bytes_read > 0)
	{
	    gchar *strptr, num_str[80];


	    /* Null terminate buffer */
	    if(bytes_read >= buf_len)
		buf[buf_len - 1] = '\0';
	    else
		buf[bytes_read] = '\0';

	    /* Look for the content length string and that we got the
	     * entire line.
	     */
	    strptr = strstr(buf, "\nLength:");
	    if((strptr != NULL) ?
		(strchr(strptr + 1, '\n') != NULL) : FALSE
	    )
	    {
		/* Found string, now seek past it to the next space
		 * space separated string.
		 */
		while(!ISBLANK(*strptr) && (*strptr != '\0'))
		    strptr++;
		while(ISBLANK(*strptr))
		    strptr++;

		/* Now strptr is positioned at the string we want,
		 * it is assumed to be a number so copy it to num_str.
		 */
		strncpy(num_str, strptr, 80);
		num_str[80 - 1] = '\0';
		/* Null terminate number string */
		strptr = num_str;
		while(!ISBLANK(*strptr) && (*strptr != '\0'))
		    strptr++;
		*strptr = '\0';

		/* Now we need to get rid of the ',' characters (if any)
		 * in the number string.
		 */
		substr(num_str, ",", "");
		total_length = atol(num_str);
		/* Number string was 0 or contained no numbers?
		 * Then set total_length to 1, indicating that although
		 * we got the total length, it was undeciperable
		 * (which can happen if the total length is not given).
		 */
		if(total_length == 0)
		    total_length = (gulong)-1;

	    }	/* Look for the content length string */
	}

	/* Deallocate read buffer */
	g_free(buf);

	return(total_length);
#undef buf_len
}


/*
 *	Monitors the download process p.
 *
 *	Returns:
 *
 *	0	Success
 *	-2	Process exited but object not downloaded completely
 *	-3	Object does not exist or unable to download from remote
 *		server (downloaded object will not exist)
 *	-4	User aborted
 */
static gint DownloadMonitor(
	edv_context_struct *ctx, gint p,
	const gchar *src, const gchar *tar,
	const gchar *stdout_file
)
{
	FILE *fp;
	gboolean	notified_object_added = FALSE,
			user_abort = FALSE;
	const gchar *cstrptr;
	gint check_count = 0, notify_count = 0;
	gulong last_length = 0, current_length = 0, total_length = 0;
	glong last_millitime = CurrentMillitime();
	gchar *title, *message, *p1, *p2, *tar_name;
	guint8        **start_icon_data[3],
			**icon_data[6],
			**end_icon_data[3];
	struct stat stat_buf;


	/* Get just the name of the target */
	cstrptr = strrchr(tar, '/');
	if(cstrptr != NULL)
	    tar_name = STRDUP(cstrptr + 1);
	else
	    tar_name = STRDUP(tar);

	/* Set up icon data */
	start_icon_data[0] = (guint8 **)icon_planet_32x32_xpm;
	start_icon_data[1] = (guint8 **)icon_planet_32x32_xpm;
	start_icon_data[2] = (guint8 **)icon_planet_32x32_xpm;

	icon_data[0] = (guint8 **)pdi_file01_20x20_xpm;
	icon_data[1] = (guint8 **)pdi_file02_20x20_xpm;
	icon_data[2] = (guint8 **)pdi_file03_20x20_xpm;
	icon_data[3] = (guint8 **)pdi_file04_20x20_xpm;
	icon_data[4] = (guint8 **)pdi_file05_20x20_xpm;
	icon_data[5] = (guint8 **)pdi_file06_20x20_xpm;

	end_icon_data[0] = (guint8 **)pdi_folder_32x32_xpm;
	end_icon_data[1] = (guint8 **)pdi_folder_32x32_xpm;
	end_icon_data[2] = (guint8 **)pdi_folderfile_32x32_xpm;

	/* Format title */
	title = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Descargar"
#elif defined(PROG_LANGUAGE_FRENCH)
"Tlcharger"
#elif defined(PROG_LANGUAGE_GERMAN)
"Laden"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scaricare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Downloaden"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Download"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Downloading"
#else
"Downloading"
#endif
	    " %s", tar_name
	);

	/* Format source and target paths for message */
	p1 = EDVCopyShortenPath(
	    src, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX - 10
	);
	p2 = EDVCopyShortenPath(
	    tar, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX - 10
	);
	/* Format initial message */
	message = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Descargar:\n\
\n\
    %s (conectar...)\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Tlcharger:\n\
\n\
    %s (connecter...)\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Laden:\n\
\n\
    %s (verbinden...)\n\
\n\
Zu:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scaricare:\n\
\n\
    %s (collegare...)\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Downloaden:\n\
\n\
    %s (verbinden...)\n\
\n\
Te:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Download:\n\
\n\
    %s (ligar...)\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Downloading:\n\
\n\
    %s (forbinding...)\n\
\n\
Til:\n\
\n\
    %s\n"
#else
"Downloading:\n\
\n\
    %s (connecting...)\n\
\n\
To:\n\
\n\
    %s\n"
#endif
	    , p1, p2
	);

	/* Map progress dialog */
	ProgressDialogSetWMIconData(
	    (const guint8 **)icon_download_file_48x48_xpm
	);
	ProgressDialogMapAnimation(
	    title,
	    message,
#if defined(PROG_LANGUAGE_SPANISH)
"Parada"
#elif defined(PROG_LANGUAGE_FRENCH)
"Arrt"
#elif defined(PROG_LANGUAGE_GERMAN)
"Halt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Fermata"
#elif defined(PROG_LANGUAGE_DUTCH)
"Einde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Parada"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stans"
#else
"Stop"
#endif
	    , start_icon_data, 3,
	    icon_data, 6,
	    end_icon_data, 3,
	    500l,
	    10000
	);

	g_free(title);
	title = NULL;
	g_free(message);
	message = NULL;


	/* Open stdout file */
	fp = FOpen(stdout_file, "rb");

	/* While the download process is running... */
	while(TRUE)
	{
	    /* User aborted? */
	    if(ProgressDialogStopCount() > 0)
	    {
		kill(p, SIGINT);
		user_abort = TRUE;
		break;
	    }

	    /* Time to do some checks (at least every 1 second)? */
	    if(check_count >= 50)
	    {
		/* Check if download process is done */
		if(!ExecProcessExists(p))
		    break;

		/* Need to get total length? Only call this to try and
		 * get the total length if we have not obtained it yet
		 *
		 * If total_length is non-zero or (gulong)-1 then it
		 * means it has already been obtained or is not
		 * available. In either case we should not try to get
		 * it anymore
		 */
		if(total_length == 0)
		    total_length = DownloadTryGetTotalLength(fp);

		/* Make sure the target object exists and get its 
		 * current length
		 */
		if(!stat(tar, &stat_buf))
		{
		    glong current_millitime, delta_millitime;
		    glong time_left_seconds;
		    gulong delta_length, remaining_length;
		    gchar time_left_str[80];

		    /* Update current length and time lapse between
		     * length checks
		     */
		    last_length = current_length;
		    current_length = stat_buf.st_size;
		    delta_length = current_length - last_length;
		    remaining_length = total_length - current_length;

		    current_millitime = CurrentMillitime();
		    if((current_millitime > last_millitime) &&
		       (last_millitime > 0)
		    )
			delta_millitime = current_millitime - last_millitime;
		    else
			delta_millitime = 0;
		    last_millitime = current_millitime;

		    /* Calculate time left in seconds, if the result
		     * is 0 then it means the transfer has stalled
		     */
		    if((total_length > 0) && (total_length != (gulong)-1) &&
		       (delta_length > 0) && (remaining_length > 0)
		    )
			time_left_seconds = (glong)(
			    (gfloat)delta_millitime / (gfloat)delta_length *
			    (gfloat)remaining_length / 1000.0f
			);
		    else
			time_left_seconds = 0;

		    /* Format time left string */
		    if(time_left_seconds > 3600)
			g_snprintf(
			    time_left_str, sizeof(time_left_str),
			    "%ld:%.2ld:%.2ld "
#if defined(PROG_LANGUAGE_SPANISH)
"quedndose"
#elif defined(PROG_LANGUAGE_FRENCH)
"rester"
#elif defined(PROG_LANGUAGE_GERMAN)
"bleiben"
#elif defined(PROG_LANGUAGE_ITALIAN)
"rimanere"
#elif defined(PROG_LANGUAGE_DUTCH)
"blijven"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"permanecer"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"forbliing"
#else
"left"
#endif
			    , (time_left_seconds / 3600),
			    (time_left_seconds / 60) % 60,
			    (time_left_seconds / 1) % 60
			);
		    else if(time_left_seconds > 0)
			g_snprintf(
			    time_left_str, sizeof(time_left_str),
			    "%ld:%.2ld "
#if defined(PROG_LANGUAGE_SPANISH)
"quedndose"
#elif defined(PROG_LANGUAGE_FRENCH)
"rester"
#elif defined(PROG_LANGUAGE_GERMAN)
"bleiben"
#elif defined(PROG_LANGUAGE_ITALIAN)
"rimanere"
#elif defined(PROG_LANGUAGE_DUTCH)
"blijven"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"permanecer"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"forbliing"
#else
"left"
#endif
			    , (time_left_seconds / 60),
			    (time_left_seconds / 1) % 60
			);
		    else
			g_snprintf(
			    time_left_str, sizeof(time_left_str),
#if defined(PROG_LANGUAGE_SPANISH)
"atascado"
#elif defined(PROG_LANGUAGE_FRENCH)
"cal"
#elif defined(PROG_LANGUAGE_GERMAN)
"hingehalten"
#elif defined(PROG_LANGUAGE_ITALIAN)
"fermato"
#elif defined(PROG_LANGUAGE_DUTCH)
"geblokkeerde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"imvel"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"stoppet"
#else
"stalled"
#endif
			);

#if 0
g_print("Time left %ld seconds (%ld left) %ld %ld\n",
 time_left_seconds, remaining_length, current_length, total_length);
#endif

		    /* Need to notify about the object being
		     * added?
		     *
		     * Note that since we stat() it we know it exists
		     */
		    if(!notified_object_added)
		    {
			EDVNotifyQueueObjectAdded(ctx, tar);
			EDVContextFlush(ctx);
			notified_object_added = TRUE;
		    }

		    /* Update title string */
		    g_free(title);
		    if((total_length > 0) && (total_length != (gulong)-1))
			title = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Descargar"
#elif defined(PROG_LANGUAGE_FRENCH)
"Tlcharger"
#elif defined(PROG_LANGUAGE_GERMAN)
"Laden"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scaricare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Downloaden"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Download"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Downloading"
#else
"Downloading"
#endif
			    " %s %.0f%%",
			    tar_name,
			    ((gfloat)current_length /
				(gfloat)total_length * 100.0f)
			);
		    else
			title = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Descargar"
#elif defined(PROG_LANGUAGE_FRENCH)
"Tlcharger"
#elif defined(PROG_LANGUAGE_GERMAN)
"Laden"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scaricare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Downloaden"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Download"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Downloading"
#else
"Downloading"
#endif
			     " %s", tar_name
			);

		    /* Update message string */
		    g_free(message);
		    if((total_length > 0) && (total_length != (gulong)-1))
		    {
			gchar	*total_size_str = STRDUP(EDVSizeStrDelim(total_length)),
				*cur_size_str = STRDUP(EDVSizeStrDelim(current_length));

			message = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Descargar:\n\
\n\
    %s (%s bytes)\n\
\n\
A:\n\
\n\
    %s (%s bytes) %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Tlcharger:\n\
\n\
    %s (%s bytes)\n\
\n\
A:\n\
\n\
    %s (%s bytes) %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Laden:\n\
\n\
    %s (%s bytes)\n\
\n\
Zu:\n\
\n\
    %s (%s bytes) %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scaricare:\n\
\n\
    %s (%s bytes)\n\
\n\
A:\n\
\n\
    %s (%s bytes) %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Downloaden:\n\
\n\
    %s (%s bytes)\n\
\n\
Te:\n\
\n\
    %s (%s bytes) %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Download:\n\
\n\
    %s (%s bytes)\n\
\n\
A:\n\
\n\
    %s (%s bytes) %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Downloading:\n\
\n\
    %s (%s bytes)\n\
\n\
Til:\n\
\n\
    %s (%s bytes) %s\n"
#else
"Downloading:\n\
\n\
    %s (%s bytes)\n\
\n\
To:\n\
\n\
    %s (%s bytes) %s\n"
#endif
			    , p1, total_size_str,
			    p2, cur_size_str, time_left_str
			);
			g_free(total_size_str);
			g_free(cur_size_str);
		    }
		    else
		    {
			gchar *cur_size_str = STRDUP(EDVSizeStrDelim(current_length));

			message = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Descargar:\n\
\n\
    %s (??? bytes)\n\
\n\
A:\n\
\n\
    %s (%s bytes)\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Tlcharger:\n\
\n\
    %s (??? bytes)\n\
\n\
A:\n\
\n\
    %s (%s bytes)\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Laden:\n\
\n\
    %s (??? bytes)\n\
\n\
Zu:\n\
\n\
    %s (%s bytes)\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scaricare:\n\
\n\
    %s (??? bytes)\n\
\n\
A:\n\
\n\
    %s (%s bytes)\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Downloaden:\n\
\n\
    %s (??? bytes)\n\
\n\
Te:\n\
\n\
    %s (%s bytes)\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Download:\n\
\n\
    %s (??? bytes)\n\
\n\
A:\n\
\n\
    %s (%s bytes)\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Downloading:\n\
\n\
    %s (??? bytes)\n\
\n\
Til:\n\
\n\
    %s (%s bytes)\n"
#else
"Downloading:\n\
\n\
    %s (??? bytes)\n\
\n\
To:\n\
\n\
    %s (%s bytes)\n"
#endif
			    , p1, p2, cur_size_str
			);
			g_free(cur_size_str);
		    }
		}	/* If total length is known, then get current length */

		check_count = 0;	/* Reset timmer */
	    }
	    else
	    {
		/* Not time to check, so just increment check timmer */
		check_count++;
	    }

	    /* Time to do notifies (every 10 seconds or longer)? */
	    if(notify_count >= 500)
	    {
		if(notified_object_added)
		{
		    /* Send object modified notify to Endeavour */
		    EDVNotifyQueueObjectModified(ctx, tar, NULL);
		    EDVContextFlush(ctx);
		}

		notify_count = 0;	/* Reset timmer */
	    }
	    else
	    {
		/* Not time to notify, so just increment notify timmer */
		notify_count++;
	    }


	    /* Manage GTK+ events */
	    if(gtk_events_pending() > 0)
		gtk_main_iteration();

	    /* Update progress and set message (if it is not NULL).
	     * If the total length is know then we use a known progress
	     * method, otherwise we use the unknown method.
	     */
	    if((total_length > 0) && (total_length != (gulong)-1))
		ProgressDialogUpdate(
		    title, message, NULL, NULL,
		    (gfloat)current_length / (gfloat)total_length,
		    EDV_DEF_PROGRESS_BAR_TICKS,
		    TRUE
		);
	    else
		ProgressDialogUpdateUnknown(
		    title, message, NULL, NULL,
		    TRUE
		);

	    /* Reset title and message strings */
	    if(title != NULL)
	    {
		g_free(title);
		title = NULL;
	    }
	    if(message != NULL)
	    {
		g_free(message);
		message = NULL;
	    }

	    usleep(20000);
	}

	/* Need to unmap progress dialog just in case */
	ProgressDialogBreakQuery(TRUE);

	/* Close standard output file */
	FClose(fp);

	/* Deallocate progress message */
	g_free(title);
	g_free(message);
	g_free(p1);
	g_free(p2);
	g_free(tar_name);


	/* Check if the object was downloaded by checking if the
	 * target object exists.  Also make sure that the user did not
	 * abort.
	 */
	if(stat(tar, &stat_buf) && !user_abort)
	{
	    /* Unable to stat destination file, this suggests that the
	     * object does not exist on the remote server or that the
	     * remote server is not responding.
	     */
	    gchar *buf = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Incapaz de descargar:\n\
\n\
    %s\n\
\n\
El objeto no existe ni el camarero remoto no responde.\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Incapable de tlcharger:\n\
\n\
    %s\n\
\n\
L'objet n'existe pas ou le serveur loign ne rpond pas.\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Unfhig zu laden:\n\
\n\
    %s\n\
\n\
Das objekt existiert nicht oder der entfernte diener antwortet nicht.\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Incapace per scaricare:\n\
\n\
    %s\n\
\n\
L'oggetto non esiste o il server remoto non risponde.\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Onbekwaam te downloaden:\n\
\n\
    %s\n\
\n\
Het voorwerp bestaat niet of de vere kelner antwoordt niet.\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Incapaz de baixar:\n\
\n\
    %s\n\
\n\
O objeto no existe nem o servidor remoto no responde.\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Maktesls til download:\n\
\n\
    %s\n\
\n\
Objektet finnes ikke eller den fjerne tjeneren responderer ikke.\n"
#else
"Unable to download:\n\
\n\
    %s\n\
\n\
The object does not exist or the remote server is not responding.\n"
#endif
		, src
	    );
	    CDialogSetTransientFor(NULL);
	    CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
"Cargue Fallado",
		buf,
"El objeto no puede existir en el camarero remoto ni el camarero\n\
remoto no responde. Verifique por favor que el URL ese reffers al\n\
objeto es correcto y que el objeto existe verdaderamente en el\n\
remoto camarero. Si el camarero remoto no responde entonces usted\n\
puede querer tratar otra vez luego.\n",
#elif defined(PROG_LANGUAGE_FRENCH)
"Charger A Echou",
		buf,
"L'objet ne peut pas exister sur le serveur loign ou le serveur\n\
loign ne rpond pas. S'il vous plat vrifier que le URL ce\n\
reffers  l'objet est exact et que l'objet existe en fait sur\n\
l'loign le serveur. Si le serveur loign ne rpond pas alors\n\
vous pouvez vouloir essayer encore plus tard.\n",
#elif defined(PROG_LANGUAGE_GERMAN)
"Laden Sie Versagt",
		buf,
"Das Objekt kann auf dem entfernten Diener nicht existieren\n\
oder der entfernte Diener antwortet nicht. Beglaubigen sie bitte,\n\
da die URL jener reffers zum objekt richtig ist, und, da das\n\
objekt tatschlich auf dem entfernten diener existiert. Wenn der\n\
entfernte diener dann sie nicht antwortet, wollen knnen, wieder\n\
spter zu versuchen.\n",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scaricare Fallito",
		buf,
"L'oggetto non pu esistere sul server remoto o il server remoto\n\
non risponde. Per favore di verificare che l'URL quel reffers\n\
all'oggetto  corretto e che l'oggetto esiste effettivamente sul\n\
server remoto. Se il server remoto non risponde poi lei pu volere\n\
tentare ancora dopo.\n",
#elif defined(PROG_LANGUAGE_DUTCH)
"Download Geverzuimenene",
		buf,
"Het voorwerp zou op de vere kelner niet kunnen bestaan of de vere\n\
kelner antwoordt niet. Bevestiig alstublieft dat de URL dat reffers\n\
aan het voorwerp correct is en dat het voorwerp eigenlijk op de vere\n\
kelner bestaat. Indien de vere kelner dan u niet antwoordt kunnen\n\
willen zou om opnieuw later te proberen.\n",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Baixe Fracassado",
		buf,
"O objeto no pode existir no servidor remoto nem o servidor remoto\n\
no responde. Por favor verifique-se que o URL aquele reffers ao\n\
objeto  correto e que o objeto realmente existe no servidor remoto.\n\
Se o servidor remoto no responde ento pode querer tentar novamente\n\
mais tarde.\n",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Download Sviktet",
		buf,
"Objektet finnes ikke p den fjerne tjeneren eller den fjerne\n\
tjeneren responderer ikke. Vr s snill og bekreft at URL den\n\
reffers til objektet er riktig og at objektet faktisk finnes p\n\
den fjerne tjeneren. Om den fjerne tjeneren ikke responderer da de\n\
vil ha prve igjen senere.\n",
#else
"Download Failed",
		buf,
"The object may not exist on the remote server or\n\
the remote server is not responding.  Please verify\n\
that the URL that reffers to the object is correct\n\
and that the object actually exists on the remote\n\
server.  If the remote server is not responding then\n\
you may want to try again later.",
#endif
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(buf);
	    return(-3);		/* Target object does not exist */
	}
	else
	{
	    current_length = stat_buf.st_size;

	    /* Target object exists, suggesting that we downloaded
	     * it successfully (but also that the user may have
	     * aborted). So note that the target object size might be
	     * smaller than the one reported in the header.
	     *
	     * Notify Endeavour one last time about object modified,
	     * but notify it as an object added since 
	     */
	    if(notified_object_added)
		EDVNotifyQueueObjectModified(ctx, tar, NULL);
	    else
		EDVNotifyQueueObjectAdded(ctx, tar);
	    EDVContextFlush(ctx);

	    /* Total length known and does not match the current
	     * length?
	     *
	     * Also make sure the user did not abort
	     */
	    if((total_length > 0) && (total_length != (gulong)-1) &&
	       (total_length != current_length) && !user_abort
	    )
	    {
		gchar	*cur_size_str = STRDUP(EDVSizeStrDelim(current_length)),
			*total_size_str = STRDUP(EDVSizeStrDelim(total_length));

		gchar *buf = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"El objeto cargado tiene un tamao de byte de %s, pero el\n\
el tamao esperado del objeto debe ser los byte de %s.\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"L'objet charg a une taille d'octets de %s, mais le\n\
la taille prvue de l'objet doit tre les octets de %s.\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Das geladene objekt hat eine Gre von %s bytes, aber die\b\
erwartete gre des objekts soll %s bytes sein.\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'oggetto scaricato ha una misura di byte di %s, ma la\n\
misura aspettata dell'oggetto dovrebbe essere i byte di %s.\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Het gedownload voorwerp heeft een maat van %s bytes, maar de\n\
verwachte maat van het voorwerp zou %s bytes moeten zijn.\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O objeto baixado tem um tamanho de bytes de %s, mas o tamanho\n\
esperado do objeto deve ser bytes de %s.\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Downloaded objekt har en strrelse av %s byter, men den\n\
ventede strrelsen av objektet er %s byter.\n"
#else
"The downloaded object has a size of %s bytes, but the\n\
expected size of the object should be %s bytes.\n"
#endif
		    , cur_size_str, total_size_str
		);
		g_free(cur_size_str);
		g_free(total_size_str);

		CDialogSetTransientFor(NULL);
		CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
"Descargado Se Opone El Tamao Inconsistancy",
		    buf,
"La descarga puede haber terminado prematuramente, se recomienda\n\
que usted reasuma la descarga descargando el objeto otra vez.\n",
#elif defined(PROG_LANGUAGE_FRENCH)
"Inconsistancy Tlcharg De Taille D'Objet",
		    buf,
"Le chargement a pu terminer prmaturment, il est recommand\n\
que vous repreniez le chargement en tlchargeant l'objet encore.\n",
#elif defined(PROG_LANGUAGE_GERMAN)
"Geladene Objekt Gre Inconsistancy",
		    buf,
"Das wird verfrht, es empfohlen ldt kann beendet haben, da\n\
sie das wiederaufnehmen, durch Laden des Objekts wieder zu laden.\n",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Inconsistancy Di Misura Di Oggetto Scaricato",
		    buf,
"Lo scarica pu avere finito il prematurly,  raccomandato che lei\n\
riprende lo scarica da scaricare l'oggetto ancora.\n",
#elif defined(PROG_LANGUAGE_DUTCH)
"Gedownloade Voorwerp Maat Inconsistancy",
		    buf,
"Het downloadt voortijdig, het door downloaden heeft misschien\n\
beindigd wordt aangeraden dat u het hervat van het voorwerp\n\
downloadt opnieuw.\n",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Inconsistancy Baixado De Tamanho De Objeto",
		    buf,
"O download pode ter acabado prematuramente,  recomendado que\n\
resume o download por baixar o objeto novamente.\n",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Downloaded Objekt Strrelse Inconsistancy",
		    buf,
"Download sluttet forhastet, det anbefaler at De gjenopptar den\n\
download ved downloading objektet igjen.\n",
#else
"Downloaded Object Size Inconsistancy",
		    buf,
"The download may have ended prematurly, it is recommended\n\
that you resume the download by downloading the object again.",
#endif
		    CDIALOG_ICON_WARNING,
		    CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
		g_free(buf);
		return(-2);	/* Target object smaller in size */
	    }

	    /* We know the object appears to have been downloaded,
	     * but we need to report if it was successful or the user
	     * aborted
	     */
	    return(user_abort ? -4 : 0);
	}
}

/*
 *	Returns:
 *
 *	0	Success
 *	1	General error (object probably not downloaded)
 *	2	Object not downloaded (object not exist, server problem, etc)
 *	3	Download program that will be used to download the
 *		object does not exist
 *	4	User abort
 */
int main(int argc, char *argv[])
{
	gboolean initialized_gtk = FALSE;
	gboolean	need_confirmation = FALSE,
			beep_when_complete = FALSE,
			open_object = FALSE,
			download_last_object = FALSE;
	gint i, status = 0;
	const gchar *arg_ptr;
	const gchar *download_prog = DOWNLOAD_PROG;
	gchar *source_url = NULL;
	gchar *destination_path = NULL;
	gchar *destination_file = NULL;
	gchar *stdout_file = NULL;
	edv_context_struct *ctx = NULL;

#define DO_SHUTDOWN	{		\
 if(ctx != NULL) {			\
  EDVContextSync(ctx);			\
  EDVContextDelete(ctx);		\
  ctx = NULL;				\
 }					\
					\
 g_free(source_url);			\
 source_url = NULL;			\
 g_free(destination_path);		\
 destination_path = NULL;		\
 g_free(destination_file);		\
 destination_file = NULL;		\
					\
 if(stdout_file != NULL) {		\
  unlink(stdout_file);			\
  g_free(stdout_file);			\
  stdout_file = NULL;			\
 }					\
					\
 if(initialized_gtk) {			\
  /* Shutdown dialogs */		\
  CDialogShutdown();			\
  PDialogShutdown();			\
  FileBrowserShutdown();		\
  ProgressDialogShutdown();		\
  initialized_gtk = FALSE;		\
 }					\
					\
 /* Reset the DND Icon */		\
 GUIDNDSetDragIcon(NULL, NULL, 0, 0);	\
}

	/* Set up time zone */
	tzset();

	/* Set up signal callbacks */
#ifdef SIGINT
	signal(SIGINT, DownloadSignalCB);
#endif
#ifdef SIGTERM
	signal(SIGTERM, DownloadSignalCB);
#endif
#ifdef SIGKILL
	signal(SIGKILL, DownloadSignalCB);
#endif
#ifdef SIGSEGV
	signal(SIGSEGV, DownloadSignalCB);
#endif
#ifdef SIGSTOP
	signal(SIGSTOP, DownloadSignalCB);
#endif
#ifdef SIGCONT
	signal(SIGCONT, DownloadSignalCB);
#endif
#ifdef SIGPIPE
	signal(SIGPIPE, DownloadSignalCB);
#endif


	/* Parse arguments */
	for(i = 1; i < argc; i++)
	{
	    arg_ptr = argv[i];
	    if(arg_ptr == NULL)
		continue;

	    /* Help */
	    if(!g_strcasecmp(arg_ptr, "--help") ||
	       !g_strcasecmp(arg_ptr, "-help") ||
	       !g_strcasecmp(arg_ptr, "--h") ||
	       !g_strcasecmp(arg_ptr, "-h") ||
	       !g_strcasecmp(arg_ptr, "-?")
	    )
	    {
		g_print("%s", PROG_HELP_MESG);
		status = 0;
		DO_SHUTDOWN
		return(status);
	    }
	    /* Version */
	    else if(!g_strcasecmp(arg_ptr, "--version") ||
		    !g_strcasecmp(arg_ptr, "-version") ||
		    !g_strcasecmp(arg_ptr, "--v") ||
		    !g_strcasecmp(arg_ptr, "-v")
	    )
	    {
		g_print("%s %s\n%s", PROG_NAME, PROG_VERSION, PROG_COPYRIGHT);
		status = 0;
		DO_SHUTDOWN
		return(status);
	    }
	    /* Beep when download is complete? */
	    else if(!g_strcasecmp(arg_ptr, "--beep") ||
		    !g_strcasecmp(arg_ptr, "-beep") ||
		    !g_strcasecmp(arg_ptr, "--b") ||
		    !g_strcasecmp(arg_ptr, "-b")
	    )
	    {
		beep_when_complete = TRUE;
	    }
	    /* Need confirmation? */
	    else if(!g_strcasecmp(arg_ptr, "--confirm") ||
		    !g_strcasecmp(arg_ptr, "-confirm") ||
		    !g_strcasecmp(arg_ptr, "--c") ||
		    !g_strcasecmp(arg_ptr, "-c")
	    )
	    {
		need_confirmation = TRUE;
	    }
	    /* Open object after (successful and complete) download? */
	    else if(!g_strcasecmp(arg_ptr, "--open") ||
		    !g_strcasecmp(arg_ptr, "-open") ||
		    !g_strcasecmp(arg_ptr, "--o") ||
		    !g_strcasecmp(arg_ptr, "-o")
	    )
	    {
		open_object = TRUE;
	    }
	    /* (Re)download last object? */
	    else if(!g_strcasecmp(arg_ptr, "--last") ||
		    !g_strcasecmp(arg_ptr, "-last") ||
		    !g_strcasecmp(arg_ptr, "--l") ||
		    !g_strcasecmp(arg_ptr, "-l")
	    )
	    {
		download_last_object = TRUE;
	    }

	    /* All else assume source url or destination path */
	    else
	    {
		if(source_url == NULL)
		    source_url = STRDUP(arg_ptr);
		else if(destination_path == NULL)
		    destination_path = STRDUP(arg_ptr);
	    }
	}

	/* Set GTK+ locale */
	gtk_set_locale();

	/* Initialize GTK+ as needed */
	if(!initialized_gtk)
	{
	    if(!gtk_init_check(&argc, &argv))
	    {
		g_printerr("Unable to initialize GTK.\n");
		status = 1;
		DO_SHUTDOWN
		return(status);
	    }
	    initialized_gtk = TRUE;
	}

	/* Initialize GDK RGB buffers system */
	gdk_rgb_init();

	/* Initialize dialogs */
	CDialogInit();
	PDialogInit();
	FileBrowserInit();
	ProgressDialogInit();

	/* Initialize Endeavour context */
	ctx = EDVContextNew();
	EDVContextLoadConfigurationFile(ctx, NULL);


	/* (Re)download last object? */
	if(download_last_object)
	{
	    /* Read last download configuration file and see if there
	     * was an object recorded, make sure that we at least have
	     * the source URL
	     */
	    gint download_status = 0;
	    DownloadCFGGetLast(
		ctx,
		&source_url,
		&destination_path,
		&download_status
	    );
	    if(source_url == NULL)
	    {
		CDialogSetTransientFor(NULL);
		CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
"Ningn Objeto Para Descargar",
"No hay el registro de un previamente descargado se opone\n\
para descargar.",
#elif defined(PROG_LANGUAGE_FRENCH)
"Aucun Objet Pour Tlcharger",
"Il n'y a pas de dossier de qu'un objet prcdemment\n\
tlcharg pour tlcharger.",
#elif defined(PROG_LANGUAGE_GERMAN)
"Kein Objekt Zu Laden",
"Es gibt keine aufzeichnung von einem vorher geladenen objekt\n\
zu laden.",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Nessuno Oggetto Di Scaricar",
"Non ci  disco di un oggetto precedentemente scaricato di\n\
scaricare.",
#elif defined(PROG_LANGUAGE_DUTCH)
"Geen Voorwerp Te Downloaden",
"Er is geen verslag van een vroeger gedownload voorwerp te\n\
downloaden.",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Nenhum Objeto Baixar",
"No h nenhum registro de um objeto previamente baixado\n\
baixar.",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Ingen Objekt Til Download",
"Der er ikke noen protokoll av et tidligere downloaded objekt\n\
til download.",
#else
"No Object To Download",
"There is no record of a previously downloaded object\n\
to (re)download.",
#endif
		    NULL,
		    CDIALOG_ICON_INFO,
		    CDIALOG_BTNFLAG_OK,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);

		status = 2;
		DO_SHUTDOWN
		return(status);
	    }
	}

	/* Use current working directory as destination path if the
	 * destination path is not obtained from the command line.
	 */
	if((destination_path != NULL) ? (*destination_path == '\0') : TRUE)
	{
	    gchar cwd[PATH_MAX];
	    if(getcwd(cwd, PATH_MAX) != NULL)
	    {
		cwd[PATH_MAX - 1] = '\0';
		g_free(destination_path);
		destination_path = STRDUP(cwd);
	    }
	}


	/* No source url given? then query user for url */
	if((source_url != NULL) ? (*source_url == '\0') : TRUE)
	{
	    gchar **strv;
	    gint strc;

	    PDialogDeleteAllPrompts();
	    PDialogAddPrompt(
		NULL, "URL:", NULL
	    );
	    PDialogSetPromptTip(
		-1,
#if defined(PROG_LANGUAGE_SPANISH)
"Entre el URL (comenzar con http:// o el ftp://) del objeto eso\
 usted quiere cargar"
#elif defined(PROG_LANGUAGE_FRENCH)
"Entrer le URL (le commencer avec http:// ou ftp://) de l'objet\
 cela vous voulez charger"
#elif defined(PROG_LANGUAGE_GERMAN)
"Tragen sie die URL (der mit http:// oder ftp:// anfngt) vom objekt,\
 das sie laden wollen ein"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Entrare l'URL (cominciando con http:// o con ftp://) dell'oggetto che\
 lei vuole scaricare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Ga de (, die met http:// of ftp:// begint) URL van het voorwerp binnen\
 dat u downloaden wil"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Entre o URL (comeando com http:// ou ftp://) do objeto que voc quer\
 baixar"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"G inn i URL (start med http:// eller ftp://) av objektet som De vil\
 ha download"
#else
"Enter the URL (starting with http:// or ftp://) of the\
 object that you want to download"
#endif
	    );
	    PDialogAddPromptWithBrowse(
		NULL,
#if defined(PROG_LANGUAGE_SPANISH)
"A"
#elif defined(PROG_LANGUAGE_FRENCH)
"A"
#elif defined(PROG_LANGUAGE_GERMAN)
"Zu"
#elif defined(PROG_LANGUAGE_ITALIAN)
"A"
#elif defined(PROG_LANGUAGE_DUTCH)
"Te"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"A"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Til"
#else
"To"
#endif
		":", destination_path,
		NULL, DownloadBrowseTargetPathCB
	    );
	    PDialogSetPromptTip(
		-1,
#if defined(PROG_LANGUAGE_SPANISH)
"Entre el sendero repleto de la gua que usted quiere para\
 descargar el objeto a o le sale blanco para descargar a la\
 gua actual"
#elif defined(PROG_LANGUAGE_FRENCH)
"Entrer le sentier plein de l'annuaire que vous voulez\
 tlcharger l'objet  ou le part le vide pour tlcharger \
 l'annuaire actuel"
#elif defined(PROG_LANGUAGE_GERMAN)
"Tragen sie den vollen pfad des verzeichnisses, das sie das\
 objekt zu oder laden wollen, verlassen es leerstelle ein, zum\
 jetzigen verzeichnis zu laden"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Entrare il sentiero pieno dell'elenco che lei vuole scaricare\
 l'oggetto a o lasciare lo spazio vuoto di esso per scaricare\
 all'elenco attuale"
#elif defined(PROG_LANGUAGE_DUTCH)
"Ga het vol pad van de gids dat u het voorwerp te downloaden wil\
 binnen of verlaat het leegte om te de huidig gids te downloaden"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Entre o pleno caminho do guia que voc quer baixar o objeto a\
 ou sai de ele lacuna baixar ao guia atual"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"G inn i den fulle stien av katalogen som De vil ha download\
 objektet til eller forlate det tomrom til download til den\
 nvrendee katalogen"
#else
"Enter the full path of the directory that you want to download the\
 object to or leave it blank to download to the current directory"
#endif
	    );
	    PDialogSetSize(450, -1);
	    strv = PDialogGetResponseIconData(
		PROG_NAME,
#if defined(PROG_LANGUAGE_SPANISH)
"Entre el URL (empezando con http:// o ftp://) del objeto que usted\n\
quiere descargar. Si usted sale el blanco de campo de A: entonces\n\
el objeto ser descargado a la gua actual."
#elif defined(PROG_LANGUAGE_FRENCH)
"Entrer le URL (commenant avec http:// ou ftp://) de l'objet que\n\
vous voulez tlcharger. Si vous partez le vide de champ de A: alors\n\
l'objet sera tlcharg  l'annuaire actuel."
#elif defined(PROG_LANGUAGE_GERMAN)
"Tragen sie die URL (der mit http:// oder ftp:// anfngt) vom objekt,\n\
das sie laden wollen ein. Wenn sie die Zu: feld leerstelle dann das\n\
objekt verlassen, zum jetzigen verzeichnis wird geladen werden."
#elif defined(PROG_LANGUAGE_ITALIAN)
"Entrare l'URL (cominciando con http:// o con ftp://) dell'oggetto\n\
che lei vuole scaricare. Se lei lascia lo spazio vuoto di campo di\n\
A: di il poi l'oggetto sar scaricato all'elenco attuale."
#elif defined(PROG_LANGUAGE_DUTCH)
"Ga de (, die met http:// of ftp:// begint) URL van het voorwerp\n\
binnen dat u downloaden wil. Indien u de Te: veld leegte dan het\n\
voorwerp verlaat te de huidig gids zal gedownload worden."
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Entre o URL (comeando com http:// ou ftp://) do objeto que voc\n\
quer baixar. Se sai da lacuna de campo de A: ento o objeto ser\n\
baixado ao guia atual."
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"G inn i URL (start med http:// eller ftp://) av objektet som\n\
de vil ha download. Om De forlater Til: felttomrom da objektet\n\
er downloaded til den nvrendee katalogen."
#else
"Enter the URL (starting with http:// or ftp://) of the object that\n\
you want to download. If you leave the To: field blank then the\n\
object will be downloaded to the current directory."
#endif
		, NULL,
		(guint8 **)icon_download_file_32x32_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Descarga",
"Cancele",
#elif defined(PROG_LANGUAGE_FRENCH)
"Chargement",
"Annuler",
#elif defined(PROG_LANGUAGE_GERMAN)
"Laden",
"Heben",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scaricare",
"Annullare",
#elif defined(PROG_LANGUAGE_DUTCH)
"Download",
"Annuleer",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Download",
"Cancelamento",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Download",
"Kanseller",
#else
"Download",
"Cancel",
#endif
		PDIALOG_BTNFLAG_SUBMIT | PDIALOG_BTNFLAG_CANCEL,
		PDIALOG_BTNFLAG_SUBMIT,
		&strc
	    );
	    if((strv != NULL) && (strc > 0))
	    {
		/* Get source URL from first string */
		const gchar *s = strv[0];
		if(!STRISEMPTY(s))
		{
		    g_free(source_url);
		    source_url = STRDUP(s);
		}

		/* Get destination path from second string (if any) */
		if(strc > 1)
		{
		    s = strv[1];
		    if(!STRISEMPTY(s))
		    {
			g_free(destination_path);
			destination_path = STRDUP(s);
		    }
		}
	    }

	    /* If no source url was obtained than that implies the user
	     * has canceled
	     */
	    if(STRISEMPTY(source_url))
	    {
		status = 4;
		DO_SHUTDOWN
		return(status);
	    }
	}


	/* Use current working directory as destination path if the
	 * destination path is not given
	 *
	 * Note that we have to do this again since the user may have
	 * been queried above and no destination path was given
	 */
	if(STRISEMPTY(destination_path))
	{
	    gchar cwd[PATH_MAX];
	    if(getcwd(cwd, PATH_MAX) != NULL)
	    {
		cwd[PATH_MAX - 1] = '\0';
		g_free(destination_path);
		destination_path = STRDUP(cwd);
	    }
	}


	/* Change working directory to the destination path only if the
	 * destination path is given, otherwise use current directory
	 */
	if(ISPATHDIR(destination_path))
	{
	    const gchar *s = strrchr(source_url, '/');
	    if(s != NULL)
		destination_file = g_strdup_printf(
		    "%s/%s", destination_path, s + 1
		);
	    else
		destination_file = STRDUP(destination_path);

	    chdir(destination_path);
	}
	else if(destination_path != NULL)
	{
	    /* Destination path is given, but it does not appear to be
	     * a directory, so instead get its parent path and change
	     * to its directory
	     */
	    const gchar	*s = strrchr(source_url, '/'),
			*parent = GetParentDir(destination_path);
	    if(parent != NULL)
		chdir(parent);

	    if(s != NULL)
		destination_file = g_strdup_printf(
		    "%s/%s", parent, s + 1
		);
	    else
		destination_file = STRDUP(destination_path);
	}
	else
	{
	    gchar cwd[PATH_MAX];
	    if(getcwd(cwd, PATH_MAX) != NULL)
	    {
		cwd[PATH_MAX - 1] = '\0';
		if(destination_path == NULL)
		    destination_path = STRDUP(cwd);
	    }
	}


	/* Need to make initial confirmation? */
	if(need_confirmation)
	{
	    gint response;
	    gchar *buf = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Descarga:\n\
\n\
    %s\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Chargement:\n\
\n\
    %s\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Laden:\n\
\n\
    %s\n\
Zu:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scaricare:\n\
\n\
    %s\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Download:\n\
\n\
    %s\n\
Te:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Download:\n\
\n\
    %s\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Download:\n\
\n\
    %s\n\
Til:\n\
\n\
    %s\n"
#else
"Download:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n"
#endif
		, source_url, destination_file
	    );
	    CDialogSetTransientFor(NULL);
	    response = CDialogGetResponseIconData(
#if defined(PROG_LANGUAGE_SPANISH)
"Confirme La Descarga"
#elif defined(PROG_LANGUAGE_FRENCH)
"Confirmer Le Chargement"
#elif defined(PROG_LANGUAGE_GERMAN)
"Besttigen Sie Ldt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Confermare Scaricare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bevestiig Downloadt"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Confirme Download"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Bekreft Download"
#else
"Confirm Download"
#endif
		, buf, NULL,
		(guint8 **)icon_download_file_32x32_xpm,
		CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
		CDIALOG_BTNFLAG_YES
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(buf);

	    switch(response)
	    {
	      case CDIALOG_RESPONSE_NO:
	      case CDIALOG_RESPONSE_NOT_AVAILABLE:
		status = 4;
		DO_SHUTDOWN
		return(status);
		break;
	    }
	}

	/* Destination file already exists? */
	if(!access(destination_file, F_OK))
	{
	    gint response;
	    gchar *buf = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Reasuma cargue de existente se opone:\n\
\n\
    %s\n\
\n\
El chasquido en el \"Si\" de para reasumir carga.\n\
El chasquido en el \"No\" de para escribir para reemplazar objeto\n\
local con objeto remoto.\n\
El chasquido en el \"Cancela\" de para cancelar el carga.\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Reprendre le chargement d'objet existant:\n\
\n\
    %s\n\
\n\
Le dclic sur \"Oui\" reprendre le chargement.\n\
Le dclic sur \"Non\" superposer l'objet local avec l'objet loign.\n\
Le dclic sur \"Annule\" pour annuler le chargement.\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Lebenslauf ldt von existieren objekt:\n\
\n\
    %s\n\
\n\
Klicken auf \"Ja\" wiederaufzunehmen zu laden.\n\
Klicken auf \"Nein\", rtliches objekt mit entferntem objekt zu\n\
berschreiben.\n\
Klicken auf "Hebt", das aufzuheben, ldt.\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Riprendere scaricare di oggetto esistente:\n\
\n\
    %s\n\
\n\
Scattare su \"S\" di riprendere scaricare.\n\
Scattare su \"No\" di sovrascrivere l'oggetto locale con\n\
l'oggetto remoto.\n\
Lo scatto su \"Annulla\" per annullare lo scarica.\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Cv download van bestaan voorwerp:\n\
\n\
    %s\n\
\n\
Klik op \"Ja\" te hervatten downloadt.\n\
Klik op \"Geen\" plaatselijk voorwerp met ver voorwerp te\n\
beschrijven.\n\
Klik \"Annuleert\" op om het te annuleren downloadt.\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Resuma download de objeto existente:\n\
\n\
    %s\n\
\n\
Clique em \"Sim\" resumir download.\n\
Clique em \"No\" a overwrite objeto local com objeto remoto.\n\
Clique em \"Cancelamento\" cancelar o download.\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Gjenoppta download av  finnes objekt:\n\
\n\
    %s\n\
\n\
Klikk p \"Ja\" gjenoppta download.\n\
Klikk p \"Ingen\" overskrive lokalt objekt med fjernt objekt.\n\
Klikk p \"Kansellerer\" kansellere download.\n"
#else
"Resume download of existing object:\n\
\n\
    %s\n\
\n\
Click on \"Yes\" to resume download.\n\
Click on \"No\" to overwrite local object with remote object.\n\
Click on \"Cancel\" to cancel the download.\n"
#endif
		, destination_file
	    );
	    CDialogSetTransientFor(NULL);
	    response = CDialogGetResponseIconData(
#if defined(PROG_LANGUAGE_SPANISH)
"Confirme Reasuma",
		buf,
"El objeto que usted tratan de cargar aparece a\n\
existe ya localmente. Si el objeto local existente\n\
era slo parcialmente cargado (e.g. debido a un interrumpido\n\
la transferencia) usted debe chasquear en \"S\" reasumir\n\
cargarlo. Si usted desea escribir para  reemplazar el\n\
existente objeto local con el objeto que usted cargan\n\
entonces el chasquido en \"No\". Si usted est no seguro lo\n\
que hacer ni usted quiere no cargar el objeto entonces\n\
chasquido en \"Cancela\".\n",
#elif defined(PROG_LANGUAGE_FRENCH)
"Confirmer Reprendre",
		buf,
"L'objet que vous essayez charger apparat \n\ existe dj\n\
localement. Si l'existant local objet seulement a t charg\n\
(e.g. grce  un interrompu transfert) vous doit cliqueter\n\
sur \"Oui\" reprendre chargement il. Si vous souhaitez\n\
superposer l'existant l'objet local avec l'objet que vous\n\
chargez alors le dclic sur \"No\". Si vous n'tes pas sr ce\n\
que de faire ou vous ne veut pas charger l'objet l'alors\n\
dclic sur \"Annule\".\n",
#elif defined(PROG_LANGUAGE_GERMAN)
"Besttigen Sie Lebenslauf",
		buf,
"Das objekt, das sie versuchen, zu laden, erscheint schon,\n\
rtlich zu existieren. Wenn das existierende rtliche objekt nur\n\
teilweise (e.g. aufgrund einer unterbrochenen bertragung) sie\n\
geladen wurde, auf sollen \"Ja\" klicken, wieder anzufangen, es zu\n\
laden. Wenn sie wnschen, das existierende rtliche objekt mit dem\n\
objekt sie zu berschreiben, dann laden klicken auf \"Nein\".\n\
Wenn sie nicht sicher sind was, zu machen oder sie \"Aufhebt\" das\n\
objekt dann klicken auf nicht wollen laden.\n",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Confermare Riprendere",
		buf,
"L'oggetto che lei tentano scaricare appare gi esistere localmente.\n\
Se l'esistere l'oggetto locale parzialmente  stato soltanto\n\
scaricato (e.g. dovuto a un trasferimento interrotto) lei dovrebbe\n\
scattare su \"S\" di riprendere per scaricare esso. Se lei desidera\n\
sovrascrivere l'esistere l'oggetto locale con l'oggetto che lei\n\
scaricano poi lo scatto su \"No\". Se lei non sono sicuro ci che di\n\
fare o lei non vuole scaricare l'oggetto il poi scatto su \"Annulla\".\n",
#elif defined(PROG_LANGUAGE_DUTCH)
"Bevestiig Cv",
		buf,
"Het voorwerp dat u probeert te downloaden verschijnt reeds om\n\
plaatselijk te bestaan. Indien het bestaand plaatselijk voorwerp\n\
enige gedeeltelijk werd gedownload (e.g. tengevolge van een\n\
onderbrekenene overdracht) u op \"Ja\" zou moeten klikken downloaden\n\
het te hervatten. Indien u wenst het bestaand plaatselijk voorwerp\n\
met het voorwerp u te beschrijven dan downloadt klik op \"Geen\".\n\
Indien u niet zeker bent wat te doen of u \"Annuleert\" het voorwerp\n\
dan klik op niet wil downloaden.\n",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Confirme Resume",
		buf,
"O objeto que voc tentam baixar aparece j existir localmente.\n\
Se o existir objeto local s parcialmente foi baixado (e.g. devido\n\
a uma transferncia interrompida) deve clicar em \"Sim\" resumir\n\
download ele. Se deseja a overwrite o existir objeto local com o\n\
objeto que voc baixam ento estalido em \"No\". Se voc no esto\n\
seguro o que fazer nem voc no quer baixar o objeto ento estalido\n\
em \"Cancelamento\".\n",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Bekreft Fortsetter",
		buf,
"Objektet som De prver til download kommer fram allerede finnes\n\
lokalt. Om finnesingen av lokalt objekt var bare delvis downloaded\n\
(e.g. p grunn av en avbrytet overfring) De klikker p \"Ja\"\n\
gjenoppta downloading det. Om De nsker overskrive finnesingen av\n\
lokalt objekt med objektet De er downloading da klikk p \"Ingen\".\n\
Om De er ikke sikker hva gjre eller De gjr ikke vil ha download\n\
objektet da klikk p \"Kansellerer\".\n",
#else
"Confirm Resume",
		buf,
"The object that you are trying to download appears to\n\
already exist locally.  If the existing local object\n\
was only partially downloaded (e.g. due to an interrupted\n\
transfer) you should click on \"Yes\" to resume\n\
downloading it.  If you wish to overwrite the existing\n\
local object with the object you are downloading then\n\
click on \"No\".  If you are not sure what to do or you\n\
do not want to download the object then click on \"Cancel\".",
#endif
		(guint8 **)icon_download_file_32x32_xpm,
		CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO |
		CDIALOG_BTNFLAG_CANCEL | CDIALOG_BTNFLAG_HELP,
		CDIALOG_BTNFLAG_YES
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(buf);

	    switch(response)
	    {
	      case CDIALOG_RESPONSE_CANCEL:
	      case CDIALOG_RESPONSE_NOT_AVAILABLE:
		status = 4;
		DO_SHUTDOWN
		return(status);
		break;

	      case CDIALOG_RESPONSE_NO:
		/* Overwrite, so remove the destination file */
		unlink(destination_file);
		/* Do not notify Endeavour about the removed file */
		break;
	    }
	}

	/* Check if the download program does not exist */
	if(access(download_prog, F_OK))
	{
	    gchar *buf = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Incapaz de encontrar el programa de la descarga:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Incapable de trouver le programme de chargement:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Unfhig, zu finden, programm ldt:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Incapace per trovare scarica il programma:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Onbekwaam te vinden programma downloadt:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Incapaz de achar programa de download:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Maktesls finne download program:\n\
\n\
    %s\n"
#else
"Unable to find download program:\n\
\n\
    %s\n"
#endif
		, download_prog
	    );
	    CDialogSetTransientFor(NULL);
	    CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
"El Programa De La Descarga No Encontr",
		buf,
"El programa de la descarga se utiliza para realizar la descarga\n\
verdadera, se debe instalar en su sistema en la ubicacin\n\
especificada encima de para permite este programa es capaz de\n\
descargar objetos.\n",
#elif defined(PROG_LANGUAGE_FRENCH)
"Le Programme de chargement n'A pas Trouv",
		buf,
"Le programme de chargement est utilis pour excuter le\n\
chargement vritable, il doit tre install sur votre systme\n\
 l'emplacement spcifi afin de rendre capable au-dessus ce\n\
programme peut tlcharger des objets.\n",
#elif defined(PROG_LANGUAGE_GERMAN)
"Laden Sie Programm, Das Gefunden Wird Nicht",
		buf,
"Das ist programm benutzt ldt, das eigentliche durchzufhren,\n\
es auf ihrem system am ort ldt oben mu installiert werden hat\n\
angegeben, um zu ermglichen, dieses programm kann objekte\n\
laden.\n",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Non scaricare il Programma Trovato",
		buf,
"Lo scarica il programma  usato per eseguire il reale scarica,\n\
deve essere installato sul suo sistema alla posizione\n\
specificata sopra per permettere questo programma  in grado di\n\
scaricare gli oggetti.\n",
#elif defined(PROG_LANGUAGE_DUTCH)
"Download Niet Programma Vond",
		buf,
"Het is programma gebruikt downloadt om het eigenlijk te\n\
verrichten, downloadt het op je systeem aan de plaats, die\n\
boven om gespecificeerd is om aan de gelegenheid deze programma\n\
te geven voorwerpen moet genstalleerd worden kan downloaden.\n",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Programa De Download No Achou",
		buf,
"O programa de download  usado para executar o download real,\n\
deve ser instalado em seu sistema na localidade especificado\n\
acima para capacitar este programa  capaz de baixar objetos.\n",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Download Program Funnet Ikke",
		buf,
"Download program bruker gjennomfre den aktuelle download,\n\
installert det p Deres system p plasseringen som spesifisert\n\
over muliggjre dette programet er kyndig til download objekt.\n",
#else
"Download Program Not Found",
		buf,
"The download program is used to perform the actual download,\n\
it must be installed on your system at the location specified\n\
above in order to enable this program is able to download\n\
objects.\n",
#endif
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(buf);

	    status = 3;
	    DO_SHUTDOWN
	    return(status);
	}

	/* Generate tempory stdout file */
	stdout_file = EDVTmpName(NULL);


	/* Execute download program and monitor download */
	if(destination_file != NULL)
	{
	    gchar *cmd = g_strdup_printf(
		"\"%s\" \"%s\" --tries=1 --progress=dot -c",
		download_prog, source_url
	    );
	    /* Execute the download command
	     *
	     * Note that wget writes stdout output to stderr
	     */
	    gint p = ExecOE(cmd, NULL, stdout_file);
	    if(p <= 0)
	    {
		/* Unable to execute download command */
		gchar *buf = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Incapaz de ejecutar la orden de la descarga:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Incapable d'excuter l'ordre de chargement:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Unfhig, durchzufhren, befehl ldt:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Incapace per eseguire scarica il comando:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Onbekwaam uit te voeren bevel downloadt:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Incapaz de executar comando de download:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Maktesls utfre download kommando:\n\
\n\
    %s\n"
#else
"Unable to execute download command:\n\
\n\
    %s\n"
#endif
		    , cmd
		);
		CDialogSetTransientFor(NULL);
		CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
"Descargue Fallado",
		    buf,
"La orden utiliz para descargar el objeto no se podra\n\
ejecutar, verifica por favor que el programa de la descarga\n\
existe en la ubicacin especificada encima de y es ejecutable.\n",
#elif defined(PROG_LANGUAGE_FRENCH)
"Tlcharger Echou",
		    buf,
"L'ordre a utilis pour tlcharger l'objet ne pourrait pas\n\
tre excut, s'il vous plat vrifier que le programme de\n\
chargement existe  l'emplacement spcifi au-dessus de et est\n\
ralisable.\n",
#elif defined(PROG_LANGUAGE_GERMAN)
"Laden Sie Versagt",
		    buf,
"Der befehl knnte das objekt nicht durchgefhrt werden hatte\n\
geladen, beglaubigt bitte, da das programm ldt, am Ort ober\n\
existiert hat angegeben und ist ausfhrbar.\n",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scaricare Fallito",
		    buf,
"Il comando ha usato per scaricare l'oggetto non potrebbe\n\
essere eseguito, verifica per favore che lo scarica il programma\n\
esiste alla posizione specificata e  sopra eseguibile.\n",
#elif defined(PROG_LANGUAGE_DUTCH)
"Download Geverzuimenene",
		    buf,
"Het bevel zou het voorwerp niet kunnen uitgevoerd worden heeft\n\
gedownload, alstublieft bevestigt dat het programma downloadt\n\
aan de plaats bovenstaande bestaat specificeerde en uitvoerbaar\n\
is.\n",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Baixe Fracassado",
		    buf,
"O comando baixava o objeto no podia ser executado, por favor\n\
verifica-se que o programa de download existe na localidade\n\
especificado acima e est executable.\n",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Download Sviktet",
		    buf,
"Kommandoen som brukt til download objektet ikke utfrer, vr\n\
s snill og bekreft at download program finnes p plasseringen\n\
spesifisert ovenfor og er gjennomfrbar.\n",
#else
"Download Failed",
		    buf,
"The command used to download the object could not be executed,\n\
please verify that the download program exists at the location\n\
specified above and is executable.\n",
#endif
		    CDIALOG_ICON_ERROR,
		    CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
		g_free(buf);
		status = 1;
	    }
	    else
	    {
		/* Download command executed, now monitor the download
	         * and handle the result
		 */
		gint download_status = DownloadMonitor(
		    ctx, p, source_url, destination_file, stdout_file
		);
		switch(download_status)
		{
		  case 0:	/* Success */
		    if(beep_when_complete)
			EDVPlaySoundCompleted(ctx);
		    if(open_object)
			EDVOpen(ctx, destination_file, NULL);
		    break;

		  case -2:	/* Process exited but object not completely downloaded */
		    status = 2;
		    if(beep_when_complete)
			EDVPlaySoundCompleted(ctx);
		    break;

		  case -3:	/* Object not downloaded (does not exist) */
		    status = 2;
		    break;

		  case -4:	/* User abort */
		    status = 4;
		    break;
		}

		/* Record the last downloaded URL regardless of the
		 * above result.
		 */
		DownloadCFGSetLast(
		    ctx, source_url, destination_path, download_status
		);

	    }
	}	/* Execute download program and monitor download */


	DO_SHUTDOWN
	return(status);
#undef DO_SHUTDOWN
}
