#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <signal.h>
#include <gtk/gtk.h>
#include <unistd.h>

#include "../../include/fio.h"
#include "../../include/disk.h"
#include "../../include/prochandle.h"

#include "../guiutils.h"
#include "../cdialog.h"
#include "../edvtypes.h"

#include "fsckmanager.h"
#include "fsckcb.h"
#include "config.h"


gint FSCKManagerButtonPressEventCB(
	GtkWidget *widget, GdkEventButton *button, gpointer data
);
gint FSCKManagerDiskIconExposeCB(
	GtkWidget *widget, GdkEventExpose *expose, gpointer data
);

gint FSCKManagerFSCKTOCB(gpointer data);

gint FSCKManagerDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);

static void FSCKManagerDoStart(
	fsck_manager_struct *fm, gint dev_num
);
void FSCKManagerStartCB(GtkWidget *widget, gpointer data);
void FSCKManagerStopCB(GtkWidget *widget, gpointer data);
void FSCKManagerClearCB(GtkWidget *widget, gpointer data);
void FSCKManagerCloseBtnCB(GtkWidget *widget, gpointer data);

void FSCKManagerSelectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
void FSCKManagerUnselectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)

#define KILL(p,s)       (((p) > 0) ? kill((p), (s)) : -1)
#define UNLINK(p)       (((p) != NULL) ? unlink(p) : -1)


/*
 *	FSCK Manager Devices GtkCList "button_press_event" signal
 *	callback.
 */
gint FSCKManagerButtonPressEventCB(
	GtkWidget *widget, GdkEventButton *button, gpointer data
)
{
	gint status = FALSE;
	gint row, column;
	GtkWidget *w;
	GtkCList *clist;
	fsck_manager_struct *fm = FSCK_MANAGER(data);
	if(fm == NULL)
	    return(status);

	clist = GTK_CLIST(fm->devices_clist);
	if(clist->clist_window != ((GdkEventAny *)button)->window)
	    return(status);

	if(!gtk_clist_get_selection_info(
	    clist, button->x, button->y, &row, &column
	))
	{
	    row = -1;
	    column = 0;
	}

	switch((gint)button->type)
	{
	  case GDK_2BUTTON_PRESS:
	    switch(button->button)
	    {
	      case 1:
		/* Start FSCK */
		FSCKManagerStartCB(widget, data);
		status = TRUE;
		break;
	    }
	    break;

	  case GDK_BUTTON_PRESS:
	    switch(button->button)
	    {
	      case 3:
		gtk_clist_freeze(clist);
		if(!(button->state & GDK_CONTROL_MASK) &&
		   !(button->state & GDK_SHIFT_MASK)
		)
		    gtk_clist_unselect_all(clist);
		clist->focus_row = row;
		gtk_clist_select_row(clist, row, column);
		gtk_clist_thaw(clist);

		/* Get right click menu widget and map it */
		w = fm->devices_menu;
		if(w != NULL)
		    gtk_menu_popup(
			GTK_MENU(w), NULL, NULL,
			NULL, NULL,
			button->button, button->time
		    );
		status = TRUE;
		break;
	    }
	    break;
	}

	return(status);
}

/*
 *	Redraws FSCK Manager's disk animation icon.
 */
gint FSCKManagerDiskIconExposeCB(
	GtkWidget *widget, GdkEventExpose *expose, gpointer data
)
{
	gint i, width, height;
	GdkWindow *window;
	GdkPixmap *pixmap, *bg_pixmap;
	GdkGC *gc;
	GtkStateType state;
	GtkStyle *style;
	GtkWidget *w;
	fsck_manager_struct *fm = FSCK_MANAGER(data);
	if(fm == NULL)
	    return(FALSE);

	w = fm->disk_anim_da;
	if(w == NULL)
	    return(FALSE);

	gc = fm->gc;
	window = w->window;
	pixmap = fm->disk_anim_pm;
	if((gc == NULL) || (window == NULL) || (pixmap == NULL))
	    return(FALSE);

	state = GTK_WIDGET_STATE(w);
	style = gtk_widget_get_style(w);
	gdk_window_get_size((GdkWindow *)pixmap, &width, &height);
	if((style == NULL) || (width <= 0) || (height <= 0))
	    return(FALSE);


	/* Begin drawing */
	bg_pixmap = style->bg_pixmap[state];
	if(bg_pixmap != NULL)
	{
	    gint tx, ty;

	    gdk_gc_set_tile(gc, bg_pixmap);
	    gdk_gc_set_fill(gc, GDK_TILED);
	    gdk_window_get_position(window, &tx, &ty);
	    gdk_gc_set_ts_origin(gc, -tx, -ty);
	}
	else
	{
	    gdk_gc_set_fill(gc, GDK_SOLID);
	}
	gdk_gc_set_clip_mask(gc, NULL);

	/* Draw background to cover entire pixmap buffer */
	gdk_gc_set_foreground(gc, &style->bg[state]);
	gdk_draw_rectangle(
	    (GdkDrawable *)pixmap, gc, TRUE,
	    0, 0, width, height
	);

	gdk_gc_set_fill(gc, GDK_SOLID);
	gdk_gc_set_ts_origin(gc, 0, 0);

	/* Find which disk animation icon we want to draw */
	i = fm->last_disk_icon_num;
	if((i >= 0) && (i < fm->total_disk_icons))
	{
	    gdk_gc_set_clip_mask(gc, fm->disk_icon_mask[i]);

	    if(fm->disk_icon_pixmap[i] != NULL)
		gdk_draw_pixmap(
		    pixmap, gc,
		    fm->disk_icon_pixmap[i],
		    0, 0,
		    0, 0,
		    width, height
		);
	}

	/* At this point the pixmap buffer has been drawn, now put the
	 * pixmap buffer to the drawing area's window.
	 */
	gdk_draw_pixmap(
	    window, gc,
	    pixmap,
	    0, 0,
	    0, 0,
	    width, height
	);

	return(TRUE);
}

/*
 *	FSCK Manager FSCK timeout callback.
 */
gint FSCKManagerFSCKTOCB(gpointer data)
{
	FILE *fp;
	pid_t p;
	fsck_manager_struct *fm = FSCK_MANAGER(data);
	if(fm == NULL)
	    return(FALSE);

/* Resets the stop count, restores progress bar to non-checking state,
 * sets fsck_toid=0, closes the fsck stdout and stderr streams, removes
 * fsck stdout and stderr tempory files, removes the current device from
 * the queued list of devices, and starts another check process for the
 * next queued device (if any)
 */
#define DO_PROCESS_END	{			\
						\
 fm->stop_count = 0;				\
						\
 FSCKManagerSetProgress(fm, 0.0f, FALSE);	\
						\
 fm->fsck_toid = 0;				\
						\
 KILL(fm->fsck_pid, SIGTERM);			\
 fm->fsck_pid = 0;				\
						\
 FClose(fm->stdout_fp);				\
 fm->stdout_fp = NULL;				\
 FClose(fm->stderr_fp);				\
 fm->stderr_fp = NULL;				\
						\
 UNLINK(fm->stdout_path);			\
 g_free(fm->stdout_path);			\
 fm->stdout_path = NULL;			\
						\
 UNLINK(fm->stderr_path);			\
 g_free(fm->stderr_path);			\
 fm->stderr_path = NULL;			\
						\
 /* Another queued device to check? */		\
 if(fm->queued_device_to_check != NULL) {	\
  /* Remove the last queued device from the	\
   * list */					\
  GList *glist = fm->queued_device_to_check;	\
  GList *glist2 = NULL;				\
  if(glist->next != NULL)			\
   glist2 = g_list_copy(glist->next);		\
  fm->queued_device_to_check = glist2;		\
  g_list_free(glist);				\
						\
  /* Check if there is a current device to	\
   * check */					\
  glist = fm->queued_device_to_check;		\
  if(glist != NULL) {				\
   gint dev_num = (gint)gtk_clist_get_row_data(	\
    (GtkCList *)fm->devices_clist,		\
    (gint)glist->data	/* Row index */		\
   );						\
   FSCKManagerDoStart(fm, dev_num);		\
  }						\
  else						\
  {						\
   /* Done checking all devices */		\
   gtk_widget_show_raise(fm->toplevel);		\
   EDVPlaySoundCompleted(fm->ctx);		\
   FSCKManagerAppendMessage(			\
    fm, fm->text_font, NULL, NULL,		\
    "Check done.\n", TRUE			\
   );						\
  }						\
 }						\
						\
 FSCKManagerUpdateMenus(fm);			\
						\
}

	/* Need to stop? */
	if(fm->stop_count > 0)
	{
	    /* Remove all queued devices */
	    g_list_free(fm->queued_device_to_check);
	    fm->queued_device_to_check = NULL;

	    /* Print interrupted message */
	    FSCKManagerAppendMessage(
		fm, fm->text_font, &fm->error_color, NULL,
		"\n*** Check Interrupted! ***\n",
		TRUE
	    );

	    DO_PROCESS_END
	    return(FALSE);
	}


	/* Read from stdout output file */
	fp = fm->stdout_fp;
	if(fp != NULL)
	{
	    gint bytes_read = 1;
	    gchar buf[1024];

	    while(bytes_read > 0)
	    {
		bytes_read = (gint)fread(
		    (char *)buf, sizeof(gchar), sizeof(buf), fp
		);
		if(bytes_read > 0)
		{
		    gint len = bytes_read;
		    if(len >= sizeof(buf))
			len = sizeof(buf) - 1;

		    buf[len] = '\0';

		    FSCKManagerAppendMessage(
			fm,
			fm->text_font,
			NULL, NULL,
			buf, TRUE
		    );
		}
	    }
	}

	/* Read from stderr output file */
	fp = fm->stderr_fp;
	if(fp != NULL)
	{
	    gint bytes_read = 1;
	    gchar buf[1024];

	    while(bytes_read > 0)
	    {
		bytes_read = (gint)fread(
		    (char *)buf, sizeof(gchar), sizeof(buf), fp
		);
		if(bytes_read > 0)
		{
		    gint len = bytes_read;
		    if(len >= sizeof(buf))
			len = sizeof(buf) - 1;

		    buf[len] = '\0';

		    FSCKManagerAppendMessage(
			fm,
			fm->text_font,
			&fm->error_color, NULL,
			buf, TRUE
		    );
		}
	    }
	}


	/* Process no longer exists? */
	p = fm->fsck_pid;
	if(!ExecProcessExists(p))
	{
	    DO_PROCESS_END
	    return(FALSE);
	}

	/* Update progress bar and disk animation icon */
	FSCKManagerSetProgress(fm, -1.0f, FALSE);

	/* Increment icon animation */
	fm->last_disk_icon_num++;
	if(fm->last_disk_icon_num >= fm->total_disk_icons)
	    fm->last_disk_icon_num = 0;
	FSCKManagerDiskIconExposeCB(fm->disk_anim_da, NULL, fm);

	return(TRUE);
#undef DO_PROCESS_END
}


/*
 *	FSCK Manager toplevel GtkWindow "delete_event" signal callback.
 */
gint FSCKManagerDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	FSCKManagerCloseBtnCB(widget, data);

	return(TRUE);
}


/*
 *	Sets up the fsck manager window to start checking the given
 *	device. The fsck program will be runned and timeout callback
 *	set.
 *
 *	Note that his function still needs work to be truely
 *	iterateable.
 */
static void FSCKManagerDoStart(
	fsck_manager_struct *fm, gint dev_num
)
{
	pid_t p;
	gboolean bad_blocks, verbose;
	const gchar *fs_type_name, *tmp_dir;
	gchar *cmd, *prog, *prog_name, *startup_message;
	GtkWidget *toplevel;
	edv_device_struct *dev;

	if(fm == NULL)
	    return;

	toplevel = fm->toplevel;

	/* Get the fsck parameters */
	bad_blocks = GTK_TOGGLE_BUTTON_GET_ACTIVE(
	    fm->bad_blocks_check
	);
	verbose = GTK_TOGGLE_BUTTON_GET_ACTIVE(
	    fm->verbose_check
	);

	/* Get the device */
	dev = ((dev_num >= 0) && (dev_num < fm->total_devices)) ?
	    fm->device[dev_num] : NULL;
	if(dev == NULL)
	    return;

	/* Get the filesystem's conical name */
	fs_type_name = EDVDeviceGetFSNameFromType(dev->fs_type);


	/* Check if this device is already mounted, in which case we
	 * cannot fsck it
	 */
	if(EDV_DEVICE_IS_MOUNTED(dev))
	{
	    gchar *buf = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"El artefacto \"%s\" se monta actualmente.\n\
\n\
Usted debe unmount el artefacto primero antes de verificar.",
#elif defined(PROG_LANGUAGE_FRENCH)
"\"%s\" d'appareil est actuellement mont.\n\
\n\
Vous devez unmount l'appareil premirement avant de vrifiant.",
#else
"Device \"%s\" is currently mounted.\n\
\n\
You must unmount the device before checking.",
#endif
		dev->device_path
	    );
	    EDVPlaySoundWarning(fm->ctx);
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
		"El Cheque Fall",
		buf,
"El artefacto se debe unmounted antes se puede verificar,\n\
esto deber prevenir los errores debido a lectura/escritura\n\
que pueden occure durante el proceso que verifica. Use la\n\
Endeavour otra aplicacin al unmount el artefacto primero.",
#elif defined(PROG_LANGUAGE_FRENCH)
		"Le Chque A Echou",
		buf,
"L'appareil doit tre unmounted avant d'il peut tre\n\
vrifi, ceci sera oblig  empcher des erreurs grce aux\n\
lecture/criture qui peut occure pendant le procd qui\n\
vrifiant. Utilise Endeavour ou un autre a la premierement.",
#else
		"Check Failed",
		buf,
"The device must be unmounted before it can be\n\
checked, this is to prevent errors due to read/writes\n\
that may occure during the checking process.\n\
Use Endeavour or another application to unmount\n\
the device first.",
#endif
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(buf);
	    return;
	}


	/* Check if there is already an fsck process running, in which
	 * case we will increment the stop count so the next call to
	 * the timeout callback stops it
	 */
	p = fm->fsck_pid;
	if(p != 0)
	{
	    if(ExecProcessExists(p))
	    {
		fm->stop_count++;
		return;
	    }
	}

	/* If there is already an existing fsck process timeout, then
	 * we will increment the stop count so the next call to the
	 * timeout callback stops it
	 */
	if(fm->fsck_toid > 0)
	{
	    fm->stop_count++;
	    return;
	}

	/* Get the FSCK program name based on the filesystem type */
	prog_name = NULL;
	switch(dev->fs_type)
	{
	  case EDV_FS_TYPE_MSDOS:
	  case EDV_FS_TYPE_VFAT:
	    prog_name = g_strconcat(FSCK_PROG, ".", "msdos", NULL);
	    break;
	  case EDV_FS_TYPE_EXT2_OLD:
	  case EDV_FS_TYPE_EXT2:
	    prog_name = g_strconcat(FSCK_PROG, ".", "ext2", NULL);
	    break;
	  case EDV_FS_TYPE_MINIX_ORIG:
	  case EDV_FS_TYPE_MINIX:
	    prog_name = g_strconcat(FSCK_PROG, ".", "minix", NULL);
	    break;
	}
	/* Filesystem type not supported? */
	if(prog_name == NULL)
	{
	    gchar *buf = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Este programa actualmente no sostiene el\n\
filesystem:\n\
\n\
    %s",
#elif defined(PROG_LANGUAGE_FRENCH)
"Ce programme ne soutient pas actuellement\n\
le filesystem:\n\
\n\
    %s",
#else
"This program currently does not support\n\
the filesystem:\n\
\n\
    %s",
#endif
		fs_type_name
	    );
	    EDVPlaySoundWarning(fm->ctx);
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
		"Filesystem No Sostuvo",
#elif defined(PROG_LANGUAGE_FRENCH)
		"Filesystem n'A Pas Soutenu",
#else
		"Filesystem Not Supported",
#endif
		buf,
		NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(buf);
	    return;
	}
	/* Check if the FSCK program exists */
	prog = EDVWhich(prog_name);
	if(prog == NULL)
	{
	    gchar *buf = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Incapaz de encontrar el programa de FSCK:\n\
\n\
    %s\n\
\n\
Verifique por favor ese programa es instalado en una\n\
ubicacion especificada por el ambiente de %s y que es\n\
ejecutable por este programa.",
#elif defined(PROG_LANGUAGE_FRENCH)
"Incapable de trouver de programme de FSCK:\n\
\n\
    %s\n\
\n\
S'il vous plait verifier ce programme est installe dans\n\
un emplacement specifie par l'environnement de %s et\n\
que c'est realisable par ce programme.",
#else
"Unable to find the FSCK program:\n\
\n\
    %s\n\
\n\
Please verify that the program is installed at a location\n\
specified by the %s environment and that it is\n\
executable by this program.",
#endif
		prog_name,
		ENV_VAR_NAME_PATH
	    );
	    EDVPlaySoundWarning(fm->ctx);
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
		"Incapaz De Encontrar El Programa De FSCK",
#elif defined(PROG_LANGUAGE_FRENCH)
		"Incapable De Trouver De Programme De FSCK",
#else
		"Unable To Find FSCK Program",
#endif
		buf,
		NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(buf);
	    g_free(prog_name);
	    return;
	}

	/* Format the command based on the filesystem type */
	cmd = NULL;
	switch(dev->fs_type)
	{
	  case EDV_FS_TYPE_MSDOS:
	  case EDV_FS_TYPE_VFAT:
	    cmd = g_strdup_printf(
		"%s -a %s %s %s",
		prog,
		verbose ? "-v" : "",
		bad_blocks ? "-t" : "",
		dev->device_path
	    );
	    break;
	  case EDV_FS_TYPE_EXT2_OLD:
	  case EDV_FS_TYPE_EXT2:
	    cmd = g_strdup_printf(
		"%s -p %s %s -f %s",
		prog,
		verbose ? "-v" : "",
		bad_blocks ? "-c" : "",
		dev->device_path
	    );
	    break;
	  case EDV_FS_TYPE_MINIX_ORIG:
	  case EDV_FS_TYPE_MINIX:
	    cmd = g_strdup_printf(
		"%s -a -v -f %s",
		prog, dev->device_path
	    );
	    break;
	}
	/* Unable to generate command? */
	if(cmd == NULL)
	{
	    gchar *buf = g_strdup_printf(
"Unable to generate command to check filesystem type:\n\
\n\
    %s",
		fs_type_name
	    );
	    EDVPlaySoundError(fm->ctx);
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Internal Error",
		buf,
		NULL,
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(buf);
	    g_free(prog);
	    g_free(prog_name);
	    return;
	}

startup_message = g_strdup_printf(
"-------------------------------------------------\n\
Checking device %s...\n",
 dev->device_path
);
FSCKManagerAppendMessage(
 fm, fm->text_font, NULL, NULL,
 startup_message,
 TRUE
);
g_free(startup_message);
startup_message = NULL;


	/* Generate tempory names for stdout and stderr paths */
	tmp_dir = g_getenv("TMPDIR");
	if(STRISEMPTY(tmp_dir))
#ifdef P_tmpdir
	    tmp_dir = P_tmpdir;
#else
	    tmp_dir = "/tmp";   
#endif

	g_free(fm->stdout_path);
	fm->stdout_path = STRDUP(PrefixPaths(tmp_dir, "fsckXXXXXX"));
	mkstemp(fm->stdout_path);

	g_free(fm->stderr_path);
	fm->stderr_path = STRDUP(PrefixPaths(tmp_dir, "fsckXXXXXX"));
	mkstemp(fm->stderr_path);

	/* Execute */
	fm->fsck_pid = p = ExecOE(
	    cmd,
	    fm->stdout_path,
	    fm->stderr_path
	);
	/* Unable to execute the command? */
	if(p == 0)
	{
	    gchar *buf = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Incapaz de ejecutar la orden:\n\
\n\
    %s",
#elif defined(PROG_LANGUAGE_FRENCH)
"Incapable d'executer l'ordre:\n\
\n\
    %s",
#else
"Unable to execute the command:\n\
\n\
    %s",
#endif
		cmd
	    );
	    EDVPlaySoundError(fm->ctx);
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
		"Incapaz De Ejecutar La Prden",
#elif defined(PROG_LANGUAGE_FRENCH)
		"Incapable D'Executer L'Ordre",
#else
		"Unable To Execute Command",
#endif
		buf,
		NULL,
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(buf);

	    UNLINK(fm->stdout_path);
	    g_free(fm->stdout_path);
	    fm->stdout_path = NULL;

	    UNLINK(fm->stderr_path);
	    g_free(fm->stderr_path);
	    fm->stderr_path = NULL;

	    g_free(cmd);
	    g_free(prog);
	    g_free(prog_name);

	    return;
	}

	/* Open stdout and stderr paths */
	FClose(fm->stdout_fp);
	fm->stdout_fp = FOpen(fm->stdout_path, "rb");

	FClose(fm->stderr_fp);
	fm->stderr_fp = FOpen(fm->stderr_path, "rb");


	/* Schedual timeout callback to monitor this process */
	fm->fsck_toid = gtk_timeout_add(
	    50l,		/* ms */
	    FSCKManagerFSCKTOCB, fm
	);



	FSCKManagerUpdateMenus(fm);

	g_free(cmd);
	g_free(prog);
	g_free(prog_name);
}


/*
 *      Start callback.
 */
void FSCKManagerStartCB(GtkWidget *widget, gpointer data)
{
	gint dev_num;
	GList *glist;
	GtkCList *clist;
	fsck_manager_struct *fm = FSCK_MANAGER(data);
	if(fm == NULL)
	    return;

	clist = (GtkCList *)fm->devices_clist;
	if(clist == NULL)
	    return;

	/* Process already running? */
	if(fm->fsck_pid != 0)
	    return;

	/* Update the mount states of each device */
	EDVDevicesListUpdateMountStates(
	    fm->device, fm->total_devices
	);

	/* Clear output */
	FSCKManagerClearCB(fm->clear_btn, fm);

	/* Clear old queued devices list to check */
	if(fm->queued_device_to_check != NULL)
	{
	    g_list_free(fm->queued_device_to_check);
	    fm->queued_device_to_check = NULL;
	}

	/* Copy list of selected rows on the devices clist as a list
	 * of queued devices to check.
	 */
	if(clist->selection != NULL)
	    fm->queued_device_to_check = g_list_copy(clist->selection);

	/* Start checking the first queued device */
	glist = fm->queued_device_to_check;
	if(glist != NULL)
	{
	    dev_num = (gint)gtk_clist_get_row_data(
		clist,
		(gint)glist->data	/* Row index */
	    );
	    FSCKManagerDoStart(fm, dev_num);
	}
}

/*
 *	Stop callback.
 */
void FSCKManagerStopCB(GtkWidget *widget, gpointer data)
{
	fsck_manager_struct *fm = FSCK_MANAGER(data);
	if(fm == NULL)
	    return;

	fm->stop_count++;
}

/*
 *	Clear callback.
 */
void FSCKManagerClearCB(GtkWidget *widget, gpointer data)
{
	GtkWidget *w;
	GtkEditable *editable;
	GtkText *text;
	fsck_manager_struct *fm = FSCK_MANAGER(data);
	if(fm == NULL)
	    return;

	w = fm->output_text;
	if(w == NULL)
	    return;

	editable = (GtkEditable *)w;
	text = (GtkText *)w;

	gtk_editable_delete_text(editable, 0, -1);
}

/*
 *	Close button callback.
 */
void FSCKManagerCloseBtnCB(GtkWidget *widget, gpointer data)
{
	fsck_manager_struct *fm = FSCK_MANAGER(data);
	if(fm == NULL)
	    return;

	FSCKManagerUnmap(fm);
}


/*
 *	Devices clist "select_row" signal callback.
 */
void FSCKManagerSelectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	fsck_manager_struct *fm = FSCK_MANAGER(data);
	if(fm == NULL)
	    return;

	/* Check which clist this event is for */
	if(GTK_WIDGET(clist) == fm->devices_clist)
	{
	    /* Check if selected row is fully visible, if not then
	     * adjust scroll position to try and make it visible.
	     */
	    if(gtk_clist_row_is_visible(clist, row) !=
		GTK_VISIBILITY_FULL
	    )
		gtk_clist_moveto(
		    clist,
		    row, column,        /* Row, column */
		    0.5, 0.0            /* Row, column */
		);

	    FSCKManagerUpdateMenus(fm);
	}
}

/*
 *      Devices clist "unselect_row" signal callback.
 */
void FSCKManagerUnselectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	fsck_manager_struct *fm = FSCK_MANAGER(data);
	if(fm == NULL)
	    return;

	/* Check which clist this event is for */
	if(GTK_WIDGET(clist) == fm->devices_clist)
	{

	    FSCKManagerUpdateMenus(fm);
	}
}
