#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <time.h>
#include <signal.h>
#include <sys/stat.h>
#ifdef HAVE_LIBXAR
# include <xar/xar.h>
/* The X Archive library does not allow public error callback
 * setting... yet, when it does you can uncomment the below
 */
/* # define USE_LIBXAR_ERROR_CB */
#endif
#include <gtk/gtk.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "cdialog.h"
#include "progressdialog.h"

#include "edv_types.h"
#include "edv_archive_obj.h"
#include "edv_archive_add.h"
#include "edv_archive_add_xar.h"
#include "endeavour2.h"
#include "edv_utils.h"
#include "edv_utils_gtk.h"
#include "config.h"


#ifdef HAVE_LIBXAR
# ifdef USE_LIBXAR_ERROR_CB
typedef struct _EDVArchXArErrorData	EDVArchXArErrorData;
#define EDV_ARCH_XAR_ERROR_DATA(p)	((EDVArchXArErrorData *)(p))
# endif
#endif


#ifdef HAVE_LIBXAR
# ifdef USE_LIBXAR_ERROR_CB
static int32_t EDVArchAddXArErrorCB(
	int32_t sev, int32_t err, xar_errctx_t ctx, void *data
);
# endif
static void EDVArchAddXArIterate(
	edv_core_struct *core, gint *status,
	const gchar *arch_path, xar_t xar,
	const gchar *password,
	const gchar *parent,		/* Parent directory of the archive */
	const gchar *path,		/* The object to add to the archive */
	GList **new_paths_list_rtn,
	gulong *cur_size, const gulong total_size,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive, const gboolean dereference_links
);
#endif	/* HAVE_LIBXAR */
gint EDVArchAddXArchive(
	edv_core_struct *core, const gchar *arch_path,
	GList *tar_paths_list,
	GList **new_paths_list_rtn,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive, const gint compression,
	const gboolean dereference_links
);


#ifdef HAVE_LIBXAR
# ifdef USE_LIBXAR_ERROR_CB
/*
 *	Error Data:
 */
struct _EDVArchXArErrorData {
	edv_core_struct	*core;
	const gchar	*arch_path;
};
# endif	/* USE_LIBXAR_ERROR_CB */
#endif	/* HAVE_LIBXAR */


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


#ifdef HAVE_LIBXAR
# ifdef USE_LIBXAR_ERROR_CB
/*
 *	X Archive library error callback.
 */
static int32_t EDVArchAddXArErrorCB(
	int32_t sev, int32_t err, xar_errctx_t ctx, void *data
)
{
	xar_file_t xar_fp;
	gint error_code;
	gchar *path;
	const gchar *error_msg;
	edv_core_struct *core;
	EDVArchXArErrorData *d = EDV_ARCH_XAR_ERROR_DATA(data);
	if(d == NULL)
	    return(0);

	core = d->core;

	xar_fp = xar_err_get_file(ctx);
	error_msg = (const gchar *)xar_err_get_string(ctx);
	error_code = (gint)xar_err_get_errno(ctx);

	switch(sev)
	{
	  case XAR_SEVERITY_DEBUG:
	  case XAR_SEVERITY_INFO:
	  case XAR_SEVERITY_WARNING:
	  case XAR_SEVERITY_NORMAL:
	    /* Ignore these */
	    break;

	  case XAR_SEVERITY_NONFATAL:
	  case XAR_SEVERITY_FATAL:
	    path = (xar_fp != NULL) ? (gchar *)xar_get_path(xar_fp) : NULL;
	    if(path != NULL)
	    {
		gchar *msg = g_strdup_printf(
"Unable to add:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
%s, %s.",
		    path, d->arch_path,
		    error_msg, g_strerror(error_code)
		);
		EDVArchAddCopyErrorMessage(core, msg);
		g_free(msg);
		g_free(path);
	    }
	    else
	    {
		gchar *msg = g_strdup_printf(
"Unable to add an object to the archive:\n\
\n\
    %s\n\
\n\
%s, %s.",
		    d->arch_path,
		    error_msg, g_strerror(error_code)
		);
		EDVArchAddCopyErrorMessage(core, msg);
		g_free(msg);
	    }
	    break;
	}

	return(0);
}
# endif	/* USE_LIBXAR_ERROR_CB */

/*
 *	Adds one object to the X Archive.
 *
 *	The path specifies the full path to the object to add to the
 *	archive.
 *
 *	If recursive is TRUE and path refers to a directory then the
 *	directory and all the objects within the directory will be
 *	added to the archive.
 */
static void EDVArchAddXArIterate(
	edv_core_struct *core, gint *status,
	const gchar *arch_path, xar_t xar,
	const gchar *password,
	const gchar *parent,		/* Parent directory of the archive */
	const gchar *path,		/* The object to add to the archive */
	GList **new_paths_list_rtn,
	gulong *cur_size, const gulong total_size,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive, const gboolean dereference_links
)
{
	mode_t m;
	struct stat stat_buf;
	gulong this_size;
	gchar *in_arch_path = NULL;

#define CLEANUP_RETURN	{		\
 g_free(in_arch_path);			\
					\
 return;				\
}

	if(STRISEMPTY(path))
	{
	    CLEANUP_RETURN;
	}

	/* Skip the archive itself */
	if(!strcmp((const char *)arch_path, path))
	{
	    CLEANUP_RETURN;
	}

	/* Get this object's local stats */
	if(lstat((const char *)path, &stat_buf))
	{
	    const gint error_code = (gint)errno;
	    gchar *msg = g_strdup_printf(
"Unable to add:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
%s.",
		path, arch_path, g_strerror(error_code)
	    );
	    EDVArchAddCopyErrorMessage(core, msg);
	    g_free(msg);
	    *status = -1;
	    CLEANUP_RETURN;
	}

	m = stat_buf.st_mode;
	this_size = (gulong)stat_buf.st_size;

	/* Generate the path of the object within the archive */
	if(EDVIsParentPath(parent, path))
	{
	    const char *s = path + STRLEN(parent);
	    while(*s == G_DIR_SEPARATOR)
		s++;
	    in_arch_path = STRDUP(s);
	}
	else
	{
	    in_arch_path = STRDUP(path);
	}
	if(in_arch_path == NULL)
	{
	    core->archive_last_error = "Memory allocation error.";
	    *status = -3;
	    CLEANUP_RETURN;
	}

	/* Update the progress dialog */
	if(show_progress)
	{
	    gchar	*p1 = EDVShortenPath(
		path,
		EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
	    ),		*p2 = EDVShortenPath(
		arch_path,
		EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
	    ),
			*msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Agregar:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Addition:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Hinzufgen:\n\
\n\
    %s\n\
\n\
Zu:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Aggiunta:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Toevoegen:\n\
\n\
    %s\n\
\n\
Te:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Adicionar:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Tilfying:\n\
\n\
    %s\n\
\n\
Til:\n\
\n\
    %s\n"
#else
"Adding:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n"
#endif
		, p1, p2
	    );
	    g_free(p1);
	    g_free(p2);
	    EDVArchAddMapProgressDialog(
		msg,
		(total_size > 0l) ?
		    ((gfloat)(*cur_size) / (gfloat)total_size) : 0.0f,
		toplevel, FALSE
	    );
	    g_free(msg);
	    if(ProgressDialogIsQuery())
	    {
		if(ProgressDialogStopCount() > 0)
		{
		    *status = -4;
		    CLEANUP_RETURN;
		}
	    }
	}

/* Reports an add error to the user and queries the user to
 * continue adding objects
 */
#define REPORT_ERROR_QUERY_CONTINUE_ADDING	{	\
 if(interactive) {					\
  if(!(*yes_to_all)) {					\
   /* Query the user to continue adding */		\
   gint response;					\
   gchar *msg = g_strdup_printf(			\
"An error occured while adding the object:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
Continue adding subsequent objects?",			\
    path, arch_path					\
   );							\
   EDVPlaySoundError(core);				\
   CDialogSetTransientFor(toplevel);			\
   response = CDialogGetResponse(			\
    "Add Failed",					\
    msg,						\
    NULL,						\
    CDIALOG_ICON_ERROR,					\
    CDIALOG_BTNFLAG_YES |				\
     CDIALOG_BTNFLAG_YES_TO_ALL |			\
     CDIALOG_BTNFLAG_NO,				\
    CDIALOG_BTNFLAG_YES					\
   );							\
   g_free(msg);						\
   CDialogSetTransientFor(NULL);			\
							\
   /* Stop adding? */					\
   if((response == CDIALOG_RESPONSE_NO) ||		\
      (response == CDIALOG_RESPONSE_CANCEL) ||		\
      (response == CDIALOG_RESPONSE_NOT_AVAILABLE)	\
   )							\
   {							\
    *status = -5;					\
    CLEANUP_RETURN;					\
   }							\
							\
   if(response == CDIALOG_RESPONSE_YES_TO_ALL)		\
    *yes_to_all = TRUE;					\
  }							\
 }							\
}

/* Report progress and return if user aborted */
#define REPORT_PROGRESS	{				\
 if(show_progress && (status == 0) && ProgressDialogIsQuery()) { \
  ProgressDialogUpdate(					\
   NULL, NULL, NULL, NULL,				\
   (total_size > 0l) ?					\
    ((gfloat)(*cur_size) / (gfloat)total_size) : 0.0f,	\
   EDV_DEF_PROGRESS_BAR_TICKS, TRUE			\
  );							\
  if(ProgressDialogStopCount() > 0) {			\
   *status = -4;					\
   CLEANUP_RETURN;					\
  }							\
 }							\
}

/* Increment *cur_size with _size_ and add _path_ to
 * new_paths_list_rtn
 */
#define COUNT_OBJECT_PATH(_path_,_size_)	{	\
 if((_size_) > 0l)					\
  *cur_size = (*cur_size) + (_size_);			\
							\
 /* Add the path to the list of objects added */	\
 if(((_path_) != NULL) && (new_paths_list_rtn != NULL))	\
  *new_paths_list_rtn = g_list_append(			\
   *new_paths_list_rtn, g_strdup(_path_)		\
  );							\
}

	/* Add this object to the archive based on its type */

	/* Directory? */
#ifdef S_ISDIR
	if(S_ISDIR(m))
#else
	if(FALSE)
#endif
	{
	    /* Add this directory to the X Archive */
	    if(xar_add(xar, (const char *)in_arch_path) != NULL)
	    {
		COUNT_OBJECT_PATH(path, 0l);
	    }
	    else
	    {
		if(core->archive_last_error == NULL)
		{
		    gchar *msg = g_strdup_printf(
"Unable to add:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s",
			path, arch_path
		    );
		    EDVArchAddCopyErrorMessage(core, msg);
		    g_free(msg);
		}
		*status = -1;
		REPORT_ERROR_QUERY_CONTINUE_ADDING
	    }

	    REPORT_PROGRESS;

	    /* Recurse into this directory? */
	    if(recursive && (*status != -4) && (*status != -5))
	    {
		/* Add the objects within this directory into the
		 * X Archive
		 */
		gint i, strc;
		const gchar *name;
		gchar **strv = (gchar **)GetDirEntNames2((const char *)path, (int *)&strc);
		StringQSort(strv, strc);
		for(i = 0; i < strc; i++)
		{
		    name = strv[i];
		    if(name == NULL)
			continue;

		    /* Ignore current and parent directory notations */
		    if(!strcmp((const char *)name, ".") ||
		       !strcmp((const char *)name, "..")
		    )
		    {
			g_free(strv[i]);
			continue;
		    }

		    if((*status != -4) && (*status != -5))
		    {
			gchar *full_path = STRDUP(PrefixPaths(
			    (const char *)path, (const char *)name
			));
			EDVArchAddXArIterate(
			    core, status,
			    arch_path, xar,
			    password,
			    parent,
			    full_path,
			    new_paths_list_rtn,
			    cur_size, total_size,
			    toplevel,
			    show_progress, interactive, yes_to_all,
			    recursive, dereference_links
			);
			g_free(full_path);
		    }
		    g_free(strv[i]);
		}
		g_free(strv);
	    }
	}
	/* Link? */
#ifdef S_ISLNK
	else if(S_ISLNK(m))
#else
	else if(FALSE)
#endif
	{
	    /* Dereference this link? */
	    if(dereference_links)
	    {
		/* Get the destination object's stats */
		if(stat((const char *)path, &stat_buf))
		{
		    const gint error_code = (gint)errno;
		    gchar *msg = g_strdup_printf(
"Unable to add:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
%s.",
			path, arch_path, g_strerror(error_code)
	            );
	            EDVArchAddCopyErrorMessage(core, msg);
	            g_free(msg);
		    *status = -1;
		    CLEANUP_RETURN;
		}
		else
		{
		    const mode_t m = stat_buf.st_mode;

		    /* Destination is a directory? */
#ifdef S_ISDIR
		    if(S_ISDIR(m))
#else
		    if(FALSE)
#endif
		    {
			/* Add this directory to the X Archive */
			if(xar_add(xar, (const char *)in_arch_path) != NULL)
			{
			    COUNT_OBJECT_PATH(path, 0l);
			}
			else
			{
			    if(core->archive_last_error == NULL)
			    {
				gchar *msg = g_strdup_printf(
"Unable to add:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s",
				    path, arch_path
				);
				EDVArchAddCopyErrorMessage(core, msg);
				g_free(msg);
			    }
			    *status = -1;
			    REPORT_ERROR_QUERY_CONTINUE_ADDING
			}

			REPORT_PROGRESS

			/* Recurse into this directory? */
			if(recursive && (*status != -4) && (*status != -5))
			{
			    /* Add the objects within this directory
			     * into the X Archive
			     */
			    gint i, strc;
			    const gchar *name;
			    gchar **strv = (gchar **)GetDirEntNames2((const char *)path, (int *)&strc);
			    StringQSort(strv, strc);
			    for(i = 0; i < strc; i++)
			    {
				name = strv[i];
				if(name == NULL)
				    continue;

				/* Ignore current and parent directory notations */
				if(!strcmp((const char *)name, ".") ||
				   !strcmp((const char *)name, "..")
				)
				{
				    g_free(strv[i]);
				    continue;
				}

				/* User not canceled or responded with
				 * "no"?
				 */
				if((*status != -4) && (*status != -5))
				{
				    gchar *full_path = STRDUP(PrefixPaths(
					(const char *)path, (const char *)name
				    ));
				    EDVArchAddXArIterate(
					core, status,
					arch_path, xar,
					password,
					parent,
					full_path,
					new_paths_list_rtn,
					cur_size, total_size,
					toplevel,
					show_progress, interactive, yes_to_all,
					recursive, dereference_links
				    );
				    g_free(full_path);
				}
				g_free(strv[i]);
			    }
			    g_free(strv);
			}
		    }
		    /* Destination is a file or other object */
		    else
		    {
			/* Add this object to the archive */
			if(xar_add(xar, (const char *)in_arch_path) != NULL)
			{
#ifdef S_ISREG
			    if(S_ISREG(m))
#else
			    if(FALSE)
#endif
			    {
				COUNT_OBJECT_PATH(path, this_size);
			    }
			    else
			    {
				COUNT_OBJECT_PATH(path, 0l);
			    }
			}
			else
			{
			    if(core->archive_last_error == NULL)
			    {
				gchar *msg = g_strdup_printf(
"Unable to add:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s",
				    path, arch_path
				);
				EDVArchAddCopyErrorMessage(core, msg);
				g_free(msg);
			    }
			    *status = -1;
			    REPORT_ERROR_QUERY_CONTINUE_ADDING
			}

			REPORT_PROGRESS
		    }
		}
	    }
	    else
	    {
		/* Add this link to the archive */
		if(xar_add(xar, (const char *)in_arch_path) != NULL)
		{
		    COUNT_OBJECT_PATH(path, this_size);
		}
		else
		{
		    if(core->archive_last_error == NULL)
		    {
			gchar *msg = g_strdup_printf(
"Unable to add:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s",
			    path, arch_path
			);
			EDVArchAddCopyErrorMessage(core, msg);
			g_free(msg);
		    }
		    *status = -1;
		    REPORT_ERROR_QUERY_CONTINUE_ADDING
		}

		REPORT_PROGRESS
	    }
	}
	/* File or other object */
	else
	{
	    /* Add this object to the archive */
	    if(xar_add(xar, (const char *)in_arch_path) != NULL)
	    {
#ifdef S_ISREG
		if(S_ISREG(m))
#else
		if(FALSE)
#endif
		{
		    COUNT_OBJECT_PATH(path, this_size);
		}
		else
		{
		    COUNT_OBJECT_PATH(path, 0l);
		}
	    }
	    else
	    {
		if(core->archive_last_error == NULL)
		{
		    gchar *msg = g_strdup_printf(
"Unable to add:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s",
			path, arch_path
		    );
		    EDVArchAddCopyErrorMessage(core, msg);
		    g_free(msg);
		}
		*status = -1;
		REPORT_ERROR_QUERY_CONTINUE_ADDING
	    }

	    REPORT_PROGRESS
	}

#undef REPORT_ERROR_QUERY_CONTINUE_ADDING
#undef REPORT_PROGRESS
#undef COUNT_OBJECT_PATH

	CLEANUP_RETURN;
#undef CLEANUP_RETURN
}
#endif	/* HAVE_LIBXAR */

/*
 *	Add object to an X Archive.
 */
gint EDVArchAddXArchive(
	edv_core_struct *core, const gchar *arch_path,
	GList *tar_paths_list,
	GList **new_paths_list_rtn,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive, const gint compression,
	const gboolean dereference_links
)
{
#ifdef HAVE_LIBXAR
	xar_t xar;
	gint status, nobjs;
	gulong cur_size, total_size;
	gchar *parent, *pwd;
	GList *glist;
#if defined(USE_LIBXAR_ERROR_CB)
	EDVArchXArErrorData *error_data;
#endif

	/* Get the parent directory of the archive, this will be the
	 * base directory in which all objects added into the X
	 * Archive will be relative to (have their parent paths striped
	 * of if their parent paths match this parent path)
	 */
	parent = g_dirname(arch_path);
	if(parent == NULL)
	{
	    core->archive_last_error = "Unable to obtain the parent directory of the archive.";
	    return(-1);
	}

	/* Calculate the total uncompressed size of all the objects to
	 * be added to the archive
	 */
	nobjs = 0;
	total_size = 0l;
	for(glist = tar_paths_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	    total_size += EDVArchAddCalculateTotalSize(
		arch_path, (const gchar *)glist->data,
		&nobjs,
		recursive, dereference_links
	    );

	/* Record the current working directory and then change to the
	 * parent directory of the archive
	 */
	pwd = EDVGetCWD();
	EDVSetCWD(parent);

	/* Open the X Archive for writing */
	xar = xar_open(arch_path, WRITE);
	if(xar == NULL)
	{
	    gchar *msg = g_strdup_printf(
"Unable to open the X Archive for writing:\n
\n\
    %s",
		arch_path
	    );
	    EDVArchAddCopyErrorMessage(core, msg);
	    g_free(msg);
	    EDVSetCWD(pwd);
	    g_free(parent);
	    g_free(pwd);
	    return(-1);
	}

	/* Set the compression
	 *
	 *	60% - 100%	BZip2
	 *	1% - 59%	GZip
	 *	0%		None
	 */
	if(compression >= 60)
#if defined(XAR_OPT_VAL_BZIP2)
	    xar_opt_set(xar, XAR_OPT_COMPRESSION, XAR_OPT_VAL_BZIP2);
#elif defined(XAR_OPT_VAL_BZIP)
	    xar_opt_set(xar, XAR_OPT_COMPRESSION, XAR_OPT_VAL_BZIP);
#elif defined(XAR_OPT_VAL_GZIP)
	    xar_opt_set(xar, XAR_OPT_COMPRESSION, XAR_OPT_VAL_GZIP);
#else
	    xar_opt_set(xar, XAR_OPT_COMPRESSION, XAR_OPT_VAL_NONE);
#endif
	else if(compression > 0)
#if defined(XAR_OPT_VAL_GZIP)
	    xar_opt_set(xar, XAR_OPT_COMPRESSION, XAR_OPT_VAL_GZIP);
#else
	    xar_opt_set(xar, XAR_OPT_COMPRESSION, XAR_OPT_VAL_NONE);
#endif
	else
	    xar_opt_set(xar, XAR_OPT_COMPRESSION, XAR_OPT_VAL_NONE);

	/* Set the checksum */
	xar_opt_set(xar, XAR_OPT_TOCCKSUM, XAR_OPT_VAL_SHA1);

#if defined(USE_LIBXAR_ERROR_CB)
	/* Set the error callback */
	error_data = EDV_ARCH_XAR_ERROR_DATA(g_malloc0(
	    sizeof(EDVArchXArErrorData)
	));
	if(error_data != NULL)
	{
	    error_data->core = core;
	    error_data->arch_path = arch_path;
	    xar_register_errhandler(
		xar,
		EDVArchAddXArErrorCB, error_data
	    );
	}
#endif

	/* Add each object to the X Archive */
	status = 0;
	cur_size = 0l;
	for(glist = tar_paths_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    EDVArchAddXArIterate(
		core, &status,
		arch_path, xar,
		password,
		parent,
		(const gchar *)glist->data,	/* Full path to the object to be added */
		new_paths_list_rtn,
		&cur_size, total_size,
		toplevel,
		show_progress, interactive, yes_to_all,
		recursive, dereference_links
	    );
	    /* User canceled or responded with "no"? */
	    if((status == -4) || (status == -5))
		break;
	}

	/* Close the X Archive */
	if(xar_close(xar) != 0)
	{
	    /* If not interrupted during the write and close then
	     * set the status to indicate error, otherwise the error
	     * was that the user aborted
	     */
	    if((status == 0) || (status != -4))
	    {
		if(core->archive_last_error == NULL)
		{
		    gchar *msg = g_strdup_printf(
"Unable to close the X Archive:\n\
\n\
    %s",
			arch_path
		    );
		    EDVArchAddCopyErrorMessage(core, msg);
		    g_free(msg);
		}
		status = -1;
	    }
	}

	/* Restore the previous working directory */
	EDVSetCWD(pwd);
	g_free(parent);
	g_free(pwd);

	/* Report the final progress */
	if(show_progress && (status == 0) && ProgressDialogIsQuery())
	{
	    ProgressDialogUpdate(
		NULL, NULL, NULL, NULL,
		1.0f, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
	    );
	    if(ProgressDialogStopCount() > 0)
		status = -4;
	}

#if defined(USE_LIBXAR_ERROR_CB)
	/* Delete the error callback data */
	g_free(error_data);
#endif

	return(status);
#else
	core->archive_last_error = "Unsupported archive format.";
	return(-2);
#endif
}
