#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <time.h>
#include <utime.h>
#include <signal.h>
#include <sys/stat.h>
#include <gtk/gtk.h>
#include <unistd.h>

#include "../include/string.h"
#include "../include/fio.h"
#include "../include/disk.h"
#include "../include/prochandle.h"

#include "cdialog.h"
#include "progressdialog.h"

#include "cfg.h"
#include "edv_types.h"
#include "edv_date.h"
#include "edv_archive_obj.h"
#include "edv_archive_extract.h"
#include "edv_archive_extract_tar.h"
#include "edv_archive_extract_xar.h"
#include "edv_archive_extract_zip.h"
#include "endeavour2.h"
#include "edv_op.h"
#include "edv_utils.h"
#include "edv_utils_gtk.h"
#include "edv_cfg_list.h"
#include "config.h"

#include "animations/file01_20x20.xpm"
#include "animations/file02_20x20.xpm"
#include "animations/file03_20x20.xpm"
#include "animations/file04_20x20.xpm"
#include "animations/file05_20x20.xpm"
#include "animations/file06_20x20.xpm"
#include "animations/folder_32x32.xpm"
#include "animations/folderfile_32x32.xpm"
#include "animations/packagefile_32x32.xpm"

#include "images/icon_replace_file_32x32.xpm"


/*
 *	Return values legend:
 *
 *	0	Success.
 *	-1	General error.
 *	-2	Invalid value.
 *	-3	Systems error, out of memory, or out of disk space.
 *	-4	User responded with "Cancel".
 *	-5	User responded with "No" or response was not available.
 *	-6	An operation is already in progress.
 */


/* Error Message */
const gchar *EDVArchExtractGetError(edv_core_struct *core);
void EDVArchExtractCopyErrorMessage(
	edv_core_struct *core, const gchar *msg
);

static gint EDVArchExtractConfirmOverwrite(
	edv_core_struct *core, GtkWidget *toplevel,
	const gchar *src_path, const gchar *tar_path,
	edv_archive_object_struct *src_obj_stat,
	const struct stat *tar_lstat_buf
);

/* Progress Dialog */
void EDVArchExtractMapProgressDialog(
	const gchar *label, const gfloat progress,
	GtkWidget *toplevel, const gboolean force_remap
);
#if 0
static void EDVArchExtractMapProgressDialogUnknown(
	const gchar *label, GtkWidget *toplevel,
	const gboolean force_remap
);
#endif

/* Mkdir Recursive */
gint EDVArchExtractRMkDir(const gchar *path, GList **new_paths_list_rtn);

/* Remove Recursive */
gint EDVArchExtractRemove(const gchar *path);

/* Extract From Archive */
static gint EDVArchExtractARJ(
	edv_core_struct *core,
	const gchar *arch_path,
	GList *objs_list, const gint nobjs,
	const gboolean extract_all,
	const gchar *dest_path,
	GList **new_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean preserve_directories,
	const gboolean preserve_timestamps
);
static gint EDVArchExtractLHA(
	edv_core_struct *core,
	const gchar *arch_path,
	GList *objs_list, const gint nobjs,
	const gboolean extract_all,
	const gchar *dest_path,
	GList **new_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean preserve_directories,
	const gboolean preserve_timestamps
);
static gint EDVArchExtractRAR(
	edv_core_struct *core,
	const gchar *arch_path,
	GList *objs_list, const gint nobjs,
	const gboolean extract_all,
	const gchar *dest_path,
	GList **new_paths_list_rtn,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean preserve_directories,
	const gboolean preserve_timestamps
);
gint EDVArchExtract(
	edv_core_struct *core,
	const gchar *arch_path,
	GList *objs_list,
	const gboolean extract_all,
	const gchar *dest_path,
	GList **new_paths_list_rtn,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean preserve_directories,
	const gboolean preserve_timestamps
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)

#define UNLINK(p)	(((p) != NULL) ? (gint)unlink((const char *)(p)) : -1)
#define INTERRUPT(i)	(((i) > 0) ? (gint)kill((int)(i), SIGINT) : -1)

#define ISCR(c)		(((c) == '\n') || ((c) == '\r'))


static gchar *G_STRCAT(gchar *s, const gchar *s2)
{
	if(s != NULL) {
	    if(s2 != NULL) {
		gchar *sr = g_strconcat(s, s2, NULL);
		g_free(s);
		s = sr;
	    }
	} else {
	    if(s2 != NULL)
		s = STRDUP(s2);
	    else
		s = STRDUP("");
	}
	return(s);
}


/*
 *	Returns the last error message as a statically allocated
 *	string or NULL if there was no previous error.
 */
const gchar *EDVArchExtractGetError(edv_core_struct *core)
{
	return((core != NULL) ? core->archive_last_error : NULL);
}

/*
 *	Coppies the error message to the core's archive_last_error_buf
 *	and sets the core's archive_last_error to point to it.
 */
void EDVArchExtractCopyErrorMessage(
	edv_core_struct *core, const gchar *msg
)
{
	if(core == NULL)
	    return;

	core->archive_last_error = NULL;

	g_free(core->archive_last_error_buf);
	core->archive_last_error_buf = STRDUP(msg);

	core->archive_last_error = core->archive_last_error_buf;
}


/*
 *	Maps the confirmation dialog and queries user for replacing the
 *	given src_path with the tar_path.
 *
 *	Returns one of CDIALOG_RESPONSE_*.
 */
static gint EDVArchExtractConfirmOverwrite(
	edv_core_struct *core, GtkWidget *toplevel,
	const gchar *src_path, const gchar *tar_path,
	edv_archive_object_struct *src_obj_stat,
	const struct stat *tar_lstat_buf
)
{
	gchar *msg, *src_date, *tar_date;
	gint response;
	edv_date_relativity relativity;
	const gchar *format;
	gulong src_size, tar_size;
	const cfg_item_struct *cfg_list;

	if(core == NULL)
	    return(CDIALOG_RESPONSE_NOT_AVAILABLE);

	cfg_list = core->cfg_list;

	/* Get date relativity and format */
	relativity = (edv_date_relativity)EDV_GET_I(
	    EDV_CFG_PARM_DATE_RELATIVITY
	);
	format = EDV_GET_S(
	    EDV_CFG_PARM_DATE_FORMAT
	);

	/* Get date strings for source and target objects */
	src_date = STRDUP(EDVDateFormatString(
	    (src_obj_stat != NULL) ?
		(gulong)src_obj_stat->modify_time : 0l,
	    format, relativity
	));
	tar_date = STRDUP(EDVDateFormatString(
	    (tar_lstat_buf != NULL) ?
		(gulong)tar_lstat_buf->st_mtime : 0l,
	    format, relativity
	));

	/* Get sizes of source and target objects in units of bytes */
	src_size = (gulong)((src_obj_stat != NULL) ?
	    src_obj_stat->size : 0
	);
	tar_size = (gulong)((tar_lstat_buf != NULL) ?
	    tar_lstat_buf->st_size : 0
	);

	/* Format confirm overwrite message */
	msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Reemplace:\n\
\n\
    %s (%ld byte%s) %s\n\
\n\
Con:\n\
\n\
    %s (%ld byte%s) %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Remplacer:\n\
\n\
    %s (%ld byte%s) %s\n\
\n\
Avec:\n\
\n\
    %s (%ld byte%s) %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Ersetzen Sie:\n\
\n\
    %s (%ld byte%s) %s\n\
\n\
Mit:\n\
\n\
    %s (%ld byte%s) %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Sostituire:\n\
\n\
    %s (%ld byte%s) %s\n\
\n\
Con:\n\
\n\
    %s (%ld byte%s) %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Vervang:\n\
\n\
    %s (%ld byte%s) %s\n\
\n\
Met:\n\
\n\
    %s (%ld byte%s) %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Reponha:\n\
\n\
    %s (%ld byte%s) %s\n\
\n\
Com:\n\
\n\
    %s (%ld byte%s) %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Erstatt:\n\
\n\
    %s (%ld byte%s) %s\n\
\n\
Med:\n\
\n\
    %s (%ld byte%s) %s\n"
#else
"Replace:\n\
\n\
    %s (%ld byte%s) %s\n\
\n\
With:\n\
\n\
    %s (%ld byte%s) %s\n"
#endif
	    ,
	    tar_path, tar_size, (tar_size == 1) ? "" : "s", tar_date,
	    src_path, src_size, (src_size == 1) ? "" : "s", src_date
	);

	EDVPlaySoundWarning(core);
	CDialogSetTransientFor(toplevel);
	response = CDialogGetResponseIconData(
#if defined(PROG_LANGUAGE_SPANISH)
"Confirme Escriba Para Reemplazar"
#elif defined(PROG_LANGUAGE_FRENCH)
"Confirmer Superposer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Besttigen Sie berschreibt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Confermare Sovrascrivere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bevestiig Beschrijft"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Confirme Overwrite"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Bekreft Overwrite"
#else
"Confirm Overwrite"
#endif
	    , msg, NULL,
	    (guint8 **)icon_replace_file_32x32_xpm,
/* Note that we are unable to have a "No" option, it must be yes or
 * cancel the entire operation
 */
	    CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_YES_TO_ALL |
	    CDIALOG_BTNFLAG_CANCEL,
	    CDIALOG_BTNFLAG_CANCEL
	);
	CDialogSetTransientFor(NULL);

	g_free(msg);
	g_free(tar_date);
	g_free(src_date);

	return(response);
}


/*
 *	Maps the progress dialog as needed in animation mode for
 *	extracting.
 */
void EDVArchExtractMapProgressDialog(
	const gchar *label, const gfloat progress,
	GtkWidget *toplevel, const gboolean force_remap
)
{
	guint8	**start_icon_data[3],
		**icon_data[6],
		**end_icon_data[3];

	/* Already mapped? */
	if(ProgressDialogIsQuery())
	{
	    /* Check if the progress dialog needs to be unmapped and
	     * remapped again
	     */
	    if(force_remap)
	    {
		ProgressDialogBreakQuery(FALSE);
	    }
	    else
	    {
		/* Already mapped and does not need unmapping, so just
		 * update the progress message
		 */
		ProgressDialogUpdate(
		    NULL, label, NULL, NULL,
		    progress, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
		);
		return;
	    }
	}

	ProgressDialogSetTransientFor(toplevel);

	start_icon_data[0] = (guint8 **)packagefile_32x32_xpm;
	start_icon_data[1] = (guint8 **)packagefile_32x32_xpm;
	start_icon_data[2] = (guint8 **)packagefile_32x32_xpm;
	icon_data[0] = (guint8 **)file01_20x20_xpm;
	icon_data[1] = (guint8 **)file02_20x20_xpm;
	icon_data[2] = (guint8 **)file03_20x20_xpm;
	icon_data[3] = (guint8 **)file04_20x20_xpm;
	icon_data[4] = (guint8 **)file05_20x20_xpm;
	icon_data[5] = (guint8 **)file06_20x20_xpm;
	end_icon_data[0] = (guint8 **)folder_32x32_xpm;
	end_icon_data[1] = (guint8 **)folder_32x32_xpm;
	end_icon_data[2] = (guint8 **)folderfile_32x32_xpm;

	ProgressDialogMapAnimation(
#if defined(PROG_LANGUAGE_SPANISH)
	    "Extraer",
	    label,
	    "Parada",
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Extraire",
	    label,
	    "Arrt",
#elif defined(PROG_LANGUAGE_GERMAN)
	    "Extrahieren",
	    label,
	    "Halt",
#elif defined(PROG_LANGUAGE_ITALIAN)
	    "Estrarre",
	    label,
	    "Fermata",
#elif defined(PROG_LANGUAGE_DUTCH)
	    "Onttrekken",
	    label,
	    "Einde",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	    "Extrair",
	    label,
	    "Parada",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	    "Trekking Ut",
	    label,
	    "Stans",
#else
	    "Extracting",
	    label,
	    "Stop",
#endif
	    start_icon_data, 3,
	    icon_data, 6,
	    end_icon_data, 3,
	    EDV_DEF_PROGRESS_DLG_ANIM_INT,
	    EDV_DEF_PROGRESS_DLG_ANIM_INC
	);
	ProgressDialogUpdate(
	    NULL, NULL, NULL, NULL,
	    progress, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
	);

	/* Flush output so dialog gets mapped and we catch the beginning
	 * of the operation (some WM need this)
	 */
	gdk_flush();
}

#if 0
/*
 *	Maps the progress dialog as needed in animation mode for
 *	extracting.
 */
static void EDVArchExtractMapProgressDialogUnknown(
	const gchar *label, GtkWidget *toplevel,
	const gboolean force_remap
)
{
	guint8	**start_icon_data[3],
		**icon_data[6],
		**end_icon_data[3];

	/* Already mapped? */
	if(ProgressDialogIsQuery())
	{
	    /* Check if the progress dialog needs to be unmapped and
	     * remapped again
	     */
	    if(force_remap)
	    {
		ProgressDialogBreakQuery(FALSE);
	    }
	    else
	    {
		/* Already mapped and does not need unmapping, so just
		 * update the progress message
		 */
		ProgressDialogUpdateUnknown(
		    NULL, label, NULL, NULL, TRUE
		);
		return;
	    }
	}

	ProgressDialogSetTransientFor(toplevel);

	start_icon_data[0] = (guint8 **)packagefile_32x32_xpm;
	start_icon_data[1] = (guint8 **)packagefile_32x32_xpm;
	start_icon_data[2] = (guint8 **)packagefile_32x32_xpm;
	icon_data[0] = (guint8 **)file01_20x20_xpm;
	icon_data[1] = (guint8 **)file02_20x20_xpm;
	icon_data[2] = (guint8 **)file03_20x20_xpm;
	icon_data[3] = (guint8 **)file04_20x20_xpm;
	icon_data[4] = (guint8 **)file05_20x20_xpm;
	icon_data[5] = (guint8 **)file06_20x20_xpm;
	end_icon_data[0] = (guint8 **)folder_32x32_xpm;
	end_icon_data[1] = (guint8 **)folder_32x32_xpm;
	end_icon_data[2] = (guint8 **)folderfile_32x32_xpm;

	ProgressDialogMapAnimation(
#if defined(PROG_LANGUAGE_SPANISH)
	    "Extraer",
	    label,
	    "Parada",
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Extraire",
	    label,
	    "Arrt",
#elif defined(PROG_LANGUAGE_GERMAN)
	    "Extrahieren",
	    label,
	    "Halt",
#elif defined(PROG_LANGUAGE_ITALIAN)
	    "Estrarre",
	    label,
	    "Fermata",
#elif defined(PROG_LANGUAGE_DUTCH)
	    "Onttrekken",
	    label,
	    "Einde",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	    "Extrair",
	    label,
	    "Parada",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	    "Trekking Ut",
	    label,
	    "Stans",
#else
	    "Extracting",
	    label,
	    "Stop",
#endif
	    start_icon_data, 3,
	    icon_data, 6,
	    end_icon_data, 3,
	    EDV_DEF_PROGRESS_DLG_ANIM_INT,
	    EDV_DEF_PROGRESS_DLG_ANIM_INC
	);
	ProgressDialogUpdateUnknown(
	    NULL, NULL, NULL, NULL, TRUE
	);

	/* Flush output so dialog gets mapped and we catch the beginning
	 * of the operation (some WM need this)
	 */
	gdk_flush();
}
#endif


/*
 *	Creates the parent directories as needed of the directory
 *	specified by path and adds the created directories to
 *	new_paths_list_rtn.
 *
 *	The specified path must be a directory.
 */
gint EDVArchExtractRMkDir(const gchar *path, GList **new_paths_list_rtn)
{
	struct stat stat_buf;
	gchar *dpath;

	if(path == NULL)
	{
	    errno = EINVAL;
	    return(-2);
	}

	dpath = g_strdup(path);
	if(dpath == NULL)
	{
	    errno = ENOMEM;
	    return(-3);
	}

	/* Path does not exist? */
	if(stat((const char *)dpath, &stat_buf))
	{
	    const guint m = EDVGetUMask();

	    if(new_paths_list_rtn != NULL)
	    {
		/* Seek s to the first deliminator past the toplevel
		 * prefix
		 */
		gchar *s = dpath;
		while((*s != G_DIR_SEPARATOR) &&
		      (*s != '\0')
		)
		    s++;

		/* Seek s past the toplevel prefix */
		if(*s == G_DIR_SEPARATOR)
		    s++;

		/* Seek s to the first deliminator */
		s = (gchar *)strchr((char *)s, G_DIR_SEPARATOR);

		/* Iterate through each directory compoent in the
		 * path, by tempory setting the deliminator to '\0'
		 * and checking for its existance
		 */
		while(s != NULL)
		{
		    /* Tempory set deliminator to '\0' */
		    *s = '\0';

		    /* Does this compoent not exist? */
		    if(stat((const char *)dpath, &stat_buf))
			*new_paths_list_rtn = g_list_append(
			    *new_paths_list_rtn, STRDUP(dpath)
			);

		    /* Restore the deliminator in dpath */
		    *s = G_DIR_SEPARATOR;

		    /* Seek to next deliminator (if any) */
		    s = strchr(s + 1, G_DIR_SEPARATOR);
		}

		/* Last directory compoent does not exist (from the
		 * very first check), so add it to the list of new
		 * objects
		 */
		*new_paths_list_rtn = g_list_append(
		    *new_paths_list_rtn, STRDUP(dpath)
		);
	    }

	    /* Create the destination directory */
	    if(rmkdir(
		dpath,
		(~m) &
		    (S_IRUSR | S_IWUSR | S_IXUSR |
		     S_IRGRP | S_IWGRP | S_IXGRP |
		     S_IROTH | S_IWOTH | S_IXOTH)
	    ))
	    {
		gint error_code = (gint)errno;
		g_free(dpath);
		errno = (int)error_code;
		return(-1);
	    }
	}

	g_free(dpath);

	return(0);
}

/*
 *	Removes the object.
 *
 *	If the object is a directory then all objects within that
 *	directory will be removed.
 *
 *	Returns 0 on success or if the object does not exist. Returns
 *	non-zero on error.
 */
gint EDVArchExtractRemove(const gchar *path)
{
	if(path == NULL)
	{
	    errno = EINVAL;
	    return(-2);
	}

	/* Remove this object */
	if(unlink(path))
	{
	    /* Unable to remove this object */
	    const gint error_code = (gint)errno;
#ifdef ENOENT
	    /* Object does not exist? */
	    if(error_code == ENOENT)
		return(0);
#endif
#ifdef ENOTDIR
	    /* A directory in the path compoent is not a directory? */
	    if(error_code == ENOTDIR)
		return(0);
#endif
#ifdef EISDIR
	    /* This object is a directory? */
	    if(error_code == EISDIR)
	    {
		/* Remove all the objects in this directory and
		 * remove this directory itself
		 */
		gchar **names_list = GetDirEntNames(path);
		if(names_list != NULL)
		{
		    gint i, status = 0, error_code = 0;
		    gchar *name, *child_path;

		    /* Remove all the objects in this directory */
		    for(i = 0; names_list[i] != NULL; i++)
		    {
			name = names_list[i];

			if(!strcmp((const char *)name, "..") ||
			   !strcmp((const char *)name, ".")
			)
			{
			    g_free(name);
			    continue;
			}

			child_path = g_strconcat(
			    path, G_DIR_SEPARATOR_S, name, NULL
			);
			if(EDVArchExtractRemove(child_path))
			{
			    status = -1;
			    error_code = (gint)errno;
			}
			g_free(child_path);

			g_free(name);
		    }

		    g_free(names_list);

		    /* Unable to remove all the objects in this directory? */
		    if(status != 0)
		    {
			errno = (int)error_code;
			return(status);
		    }
		}

		/* Remove this directory */
		if(rmdir(path))
		{
		    /* Unable to remove this directory */
		    const gint error_code = (gint)errno;
#ifdef ENOENT
		    /* Directory does not exist? */
		    if(error_code == ENOENT)
			return(0);
#endif
#ifdef ENOTDIR
		    /* A directory in the path compoent is not a directory? */
		    if(error_code == ENOTDIR)
			return(0);
#endif
		    /* Unable to remove this directory, return error */
		    return(-1);
		}

		return(0);
	    }
#endif  /* EISDIR */

	    /* Unable to remove this object, return error */
	    return(-1);
	}

	return(0);
}


/*
 *	Extract the objects from the ARJ archive.
 *
 *	Inputs assumed valid.
 */
static gint EDVArchExtractARJ(
	edv_core_struct *core,
	const gchar *arch_path,
	GList *objs_list, const gint nobjs,
	const gboolean extract_all,
	const gchar *dest_path,
	GList **new_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive, gboolean *yes_to_all,
	const gboolean preserve_directories,
	const gboolean preserve_timestamps
)
{
	const gchar *prog_arj = CFGItemListGetValueS(
	    core->cfg_list, EDV_CFG_PARM_PROG_ARJ
	);
	FILE *fp;
	gint p, status, nobjs_extracted;
	gchar	*cmd = NULL,
		*pwd = NULL,
		*stdout_path = NULL,
		*stderr_path = NULL;
	const gchar *src_path;
	GList *glist;
	edv_archive_object_struct *obj;

#define CLEANUP_RETURN(_v_)	{		\
 g_free(cmd);					\
 g_free(stdout_path);				\
 g_free(stderr_path);				\
						\
 /* Restore the previous working dir */		\
 if(pwd != NULL) {				\
  EDVSetCWD(pwd);				\
  g_free(pwd);					\
 }						\
						\
 return(_v_);					\
}

	/* Record previous working dir and set new working dir */
	pwd = EDVGetCWD();
	if(EDVSetCWD(dest_path))
	{
	    core->archive_last_error =
"Unable to change working directory to the destination directory.";
	    CLEANUP_RETURN(-1);
	}

	/* Format extract object from archive command */
	cmd = g_strdup_printf(
	    "\"%s\" %c -i -r -y \"%s\"",
	    prog_arj,
	    preserve_directories ? 'x' : 'e',
	    arch_path
	);
	if(cmd == NULL)
	{
	    core->archive_last_error = "Unable to generate the extract command.";
	    CLEANUP_RETURN(-1);
	}
	/* Append the objects to extract to the command string
	 * only if not extracting all the objects
	 */
	if(!extract_all)
	{
	    for(glist = objs_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		obj = EDV_ARCHIVE_OBJECT(glist->data);
		if(obj == NULL)
		    continue;

		src_path = obj->full_path;
		if(STRISEMPTY(src_path))
		    continue;

		if(obj->type == EDV_OBJECT_TYPE_DIRECTORY)
		{
		    const gint len = STRLEN(src_path);

		    cmd = G_STRCAT(cmd, " \"");
		    cmd = G_STRCAT(cmd, src_path);
		    /* If directory does not have a tailing deliminator
		     * then we must append one or else it will not get
		     * matched
		     */
		    if(len > 1)
		    {
			if(src_path[len - 1] != G_DIR_SEPARATOR)
			{
			    gchar delim_str[2];
			    delim_str[0] = G_DIR_SEPARATOR;
			    delim_str[1] = '\0';
			    cmd = G_STRCAT(cmd, delim_str);
			}
		    }
		    cmd = G_STRCAT(cmd, "\"");
		}
		else
		{
		    cmd = G_STRCAT(cmd, " \"");
		    cmd = G_STRCAT(cmd, src_path);
		    cmd = G_STRCAT(cmd, "\"");
		}
	    }
	}
	if(cmd == NULL)
	{
	    core->archive_last_error = "Unable to generate the extract command.";
	    CLEANUP_RETURN(-1);
	}


	/* Generate output file paths */
	stdout_path = EDVTmpName(NULL);
	stderr_path = EDVTmpName(NULL);

	/* Begin extracting */

	status = 0;
	nobjs_extracted = 0;

	/* Execute the extract objects from archive command */
	p = (gint)ExecOE((const char *)cmd, (const char *)stdout_path, (const char *)stderr_path);
	if(p <= 0)
	{
	    core->archive_last_error = "Unable to execute the extract command.";
	    CLEANUP_RETURN(-1);
	}
	else
	{
	    /* Delete extract command, it is no longer needed */
	    g_free(cmd);
	    cmd = NULL;
	}


	/* Open the output file for reading */
	fp = fopen((const char *)stdout_path, "rb");
	if(fp != NULL)
	{
	    gint buf_pos = 0, line_count = 0;
	    gfloat progress = 0.0f;
	    gchar buf[10000];
	    gboolean need_break = FALSE;

	    /* Begin reading the output file */
	    while(TRUE)
	    {
		/* Update progress? */
		if(show_progress && ProgressDialogIsQuery())
		{
		    ProgressDialogUpdate(
			NULL, NULL, NULL, NULL,
			progress, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
		    );
		    if(ProgressDialogStopCount() > 0)
		    {
			INTERRUPT(p);
			p = 0;
			status = -4;
			break;
		    }
		}

		/* Check if there is new data to be read from the output
		 * file
		 */
		if(FPending(fp))
		{
		    gint c;
		    gboolean got_complete_line = FALSE;

		    /* Copy all available data from the current output
		     * file position to its end to the line buffer buf
		     */
		    while(TRUE)
		    {
			c = (gint)fgetc(fp);
			if((int)c == EOF)
			{
			    clearerr(fp);
			    break;
			}

			if(ISCR(c))
			{
			    got_complete_line = TRUE;
			    line_count++;

			    if(buf_pos < sizeof(buf))
				buf[buf_pos] = '\0';
			    else
				buf[sizeof(buf) - 1] = '\0';
			    buf_pos = 0;

			    break;
			}

			if(buf_pos < sizeof(buf))
			{
			    buf[buf_pos] = c;
			    buf_pos++;
			}
		    }
		    /* Extracting small objects from an ARJ archive
		     * sometimes causes lines to be missed and objects
		     * being extracted but not being reported as
		     * extracted
		     */
		    if(got_complete_line &&
		       strcasepfx(buf, "Extracting")
		    )
		    {
			gchar *s = buf, *s2, *extracted_path;

			/* Seek s past the prefix to the path value */
			while(!ISBLANK(*s) && (*s != '\0'))
			    s++;
			while(ISBLANK(*s))
			    s++;

			/* Cap the first blank character after the path */
			for(s2 = s; *s2 != '\0'; s2++)
			{
			    if(ISBLANK(*s2))
			    {
				*s2 = '\0';
				break;
			    }
			}

			extracted_path = STRDUP(PrefixPaths(dest_path, s));
			StripPath(extracted_path);

			/* Append this path to the list of paths
			 * extracted from the archive
			 */
			if(new_paths_list_rtn != NULL)
			    *new_paths_list_rtn = g_list_append(
				*new_paths_list_rtn, STRDUP(extracted_path)
			    );

			/* Update the progress dialog's label? */
			if(show_progress && !STRISEMPTY(extracted_path))
			{
			    gchar	*p1 = EDVShortenPath(
				s, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
			    ),
					*p2 = EDVShortenPath(
				extracted_path, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
			    ),
					*msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Extraer:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Extraire:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Extrahieren:\n\
\n\
    %s\n\
\n\
Zu:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Estrarre:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Onttrekken:\n\
\n\
    %s\n\
\n\
Te:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Extrair:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Trekking Ut:\n\
\n\
    %s\n\
\n\
Til:\n\
\n\
    %s\n"
#else
"Extracting:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n"
#endif
				, p1, p2
			    );
			    EDVArchExtractMapProgressDialog(
				msg, progress, toplevel, FALSE
			    );
			    g_free(msg);
			    g_free(p1);
			    g_free(p2);
			}

			nobjs_extracted++;
			progress = (nobjs > 0) ?
			    ((gfloat)nobjs_extracted / (gfloat)nobjs) : 0.0f;

			g_free(extracted_path);

			continue;
		    }
		}

		if(need_break)
		    break;

		/* Check if the extract process has exited, if it has
		 * then we set need_break to TRUE. Which will be
		 * tested on the next loop if there is still no more
		 * data to be read
		 */
		if(!ExecProcessExists(p))
		    need_break = TRUE;

		usleep(8000);
	    }

	    fclose(fp);
	}

	/* Remove output files */
	UNLINK(stdout_path);
	UNLINK(stderr_path);

	/* Report the final progress? */
	if(show_progress && (status == 0) &&
	   ProgressDialogIsQuery()
	)
	{
	    ProgressDialogUpdate(
		NULL, NULL, NULL, NULL,
		1.0f, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
	    );
	    if(ProgressDialogStopCount() > 0)
		status = -4;
	}

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}

/*
 *	Extract object from a LHA archive.
 *
 *	Inputs assumed valid.
 */
static gint EDVArchExtractLHA(
	edv_core_struct *core,
	const gchar *arch_path,
	GList *objs_list, const gint nobjs,
	const gboolean extract_all,
	const gchar *dest_path,
	GList **new_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive, gboolean *yes_to_all,
	const gboolean preserve_directories,
	const gboolean preserve_timestamps
)
{
	const gchar *prog_lha = CFGItemListGetValueS(
	    core->cfg_list, EDV_CFG_PARM_PROG_LHA
	);
	FILE *fp;
	gint status, p, nobjs_extracted;
	gchar   *cmd = NULL,
		*pwd = NULL,
		*stdout_path = NULL,
		*stderr_path = NULL;
	const gchar *src_path;
	GList *glist;
	edv_archive_object_struct *obj;

#define CLEANUP_RETURN(_v_)	{		\
 g_free(cmd);					\
 g_free(stdout_path);				\
 g_free(stderr_path);				\
						\
 /* Restore the previous working dir */		\
 if(pwd != NULL) {				\
  EDVSetCWD(pwd);				\
  g_free(pwd);					\
 }						\
						\
 return(_v_);					\
}

	/* Record previous working dir and set new working dir */
	pwd = EDVGetCWD();
	if(EDVSetCWD(dest_path))
	{
	    core->archive_last_error =
"Unable to change working directory to the destination directory.";
	    CLEANUP_RETURN(-1);
	}

	/* Format extract object from archive command */
	cmd = g_strdup_printf(
	    "\"%s\" -%cvf \"%s\"",
	    prog_lha,
	    preserve_directories ? 'x' : 'e',
	    arch_path
	);
	if(cmd == NULL)
	{
	    core->archive_last_error = "Unable to generate the extract command.";
	    CLEANUP_RETURN(-1);
	}
	/* Append the objects to extract to the command string
	 * only if not extracting all the objects
	 */
	if(!extract_all)
	{
	    for(glist = objs_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		obj = EDV_ARCHIVE_OBJECT(glist->data);
		if(obj == NULL)
		    continue;

		src_path = obj->full_path;
		if(STRISEMPTY(src_path))
		    continue;

		if(obj->type == EDV_OBJECT_TYPE_DIRECTORY)
		{
		    const gint len = STRLEN(src_path);

		    cmd = G_STRCAT(cmd, " \"");
		    cmd = G_STRCAT(cmd, src_path);
		    /* If directory does not have a tailing deliminator
		     * then we must append one or else it will not get
		     * matched
		     */
		    if(len > 1)
		    {
			if(src_path[len - 1] != G_DIR_SEPARATOR)
			{
			    gchar delim_str[2];
			    delim_str[0] = G_DIR_SEPARATOR;
			    delim_str[1] = '\0';
			    cmd = G_STRCAT(cmd, delim_str);
			}
		    }
		    cmd = G_STRCAT(cmd, "\"");
		}
		else
		{
		    cmd = G_STRCAT(cmd, " \"");
		    cmd = G_STRCAT(cmd, src_path);
		    cmd = G_STRCAT(cmd, "\"");
		}
	    }
	}
	if(cmd == NULL)
	{
	    core->archive_last_error = "Unable to generate the extract command.";
	    CLEANUP_RETURN(-1);
	}


	/* Generate output file paths */
	stdout_path = EDVTmpName(NULL);
	stderr_path = EDVTmpName(NULL);

	/* Begin extracting */

	status = 0;
	nobjs_extracted = 0;

	/* Execute the extract objects from archive command */
	p = (gint)ExecOE(
	    (const char *)cmd,
	    (const char *)stdout_path,
	    (const char *)stderr_path
	);
	if(p <= 0)
	{
	    core->archive_last_error = "Unable to execute the extract command.";
	    CLEANUP_RETURN(-1);
	}

	g_free(cmd);
	cmd = NULL;

	/* Open the output file for reading */
	fp = fopen((const char *)stdout_path, "rb");
	if(fp != NULL)
	{
	    gint buf_pos = 0, line_count = 0;
	    gfloat progress = 0.0f;
	    gchar buf[10000];
	    gboolean need_break = FALSE;

	    /* Begin reading the output file */
	    while(TRUE)
	    {
		/* Update progress? */
		if(show_progress && ProgressDialogIsQuery())
		{
		    ProgressDialogUpdate(
			NULL, NULL, NULL, NULL,
			progress, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
		    );
		    if(ProgressDialogStopCount() > 0)
		    {
			INTERRUPT(p);
			p = 0;
			status = -4;
			break;
		    }
		}

		/* Check if there is new data to be read from the output
		 * file
		 */
		if(FPending(fp))
		{
		    gint c;
		    gboolean got_complete_line = FALSE;

		    /* Copy all available data from the current output
		     * file position to its end to the line buffer buf
		     */
		    while(TRUE)
		    {
			c = (gint)fgetc(fp);
			if((int)c == EOF)
			{
			    clearerr(fp);
			    break;
			}

			if(ISCR(c))
			{
			    got_complete_line = TRUE;
			    line_count++;

			    if(buf_pos < sizeof(buf))
				buf[buf_pos] = '\0';
			    else
				buf[sizeof(buf) - 1] = '\0';
			    buf_pos = 0;

			    break;
			}

			if(buf_pos < sizeof(buf))
			{
			    buf[buf_pos] = c;
			    buf_pos++;
			}
		    }
		    if(got_complete_line)
		    {
			gchar *s = buf, *s2;

			while(ISBLANK(*s))
			    s++;

			s2 = (gchar *)strstr((char *)s, " :");

			if(strcasepfx((char *)s, "Making directory \""))
			{
			    gchar *extracted_path;

			    s += STRLEN("Making directory \"");
			    s2 = (gchar *)strrchr((char *)s, '"');
			    if(s2 != NULL)
				*s2 = '\0';

			    extracted_path = STRDUP(PrefixPaths(dest_path, s));
			    StripPath(extracted_path);

			    /* Append this path to the list of paths
			     * extracted from the archive
			     */
			    if(new_paths_list_rtn != NULL)
				*new_paths_list_rtn = g_list_append(
				    *new_paths_list_rtn, STRDUP(extracted_path)
				);
			}
			else if(s2 != NULL)
			{
			    gchar *extracted_path;

			    s2 = (gchar *)strpbrk((char *)s, " \t");
			    if(s2 != NULL)
				*s2 = '\0';

			    extracted_path = STRDUP(PrefixPaths(dest_path, s));
			    StripPath(extracted_path);

			    /* Append this path to the list of paths
			     * extracted from the archive
			     */
			    if(new_paths_list_rtn != NULL)
				*new_paths_list_rtn = g_list_append(
				    *new_paths_list_rtn, STRDUP(extracted_path)
				);

			    /* Update the progress dialog's label? */
			    if(show_progress && !STRISEMPTY(extracted_path))
			    {
			        gchar	*p1 = EDVShortenPath(
				    s, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
				),
					*p2 = EDVShortenPath(
				extracted_path, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
				),
					*msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Extraer:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Extraire:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Extrahieren:\n\
\n\
    %s\n\
\n\
Zu:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Estrarre:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Onttrekken:\n\
\n\
    %s\n\
\n\
Te:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Extrair:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Trekking Ut:\n\
\n\
    %s\n\
\n\
Til:\n\
\n\
    %s\n"
#else
"Extracting:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n"
#endif
				    , p1, p2
				);
				EDVArchExtractMapProgressDialog(
				    msg, progress, toplevel, FALSE
				);
				g_free(msg);
				g_free(p1);
				g_free(p2);
			    }

			    nobjs_extracted++;
			    progress = (nobjs > 0) ?
				((gfloat)nobjs_extracted / (gfloat)nobjs) : 0.0f;

			    g_free(extracted_path);
			}

			continue;
		    }
		}

		if(need_break)
		    break;

		/* Check if the extract process has exited, if it has
		 * then we set need_break to TRUE. Which will be
		 * tested on the next loop if there is still no more
		 * data to be read
		 */
		if(!ExecProcessExists(p))
		    need_break = TRUE;

		usleep(8000);
	    }

	    fclose(fp);
	}

	/* Remove output files */
	UNLINK(stdout_path);
	UNLINK(stderr_path);

	/* Report the final progress? */
	if(show_progress && (status == 0) &&
	   ProgressDialogIsQuery()
	)
	{
	    ProgressDialogUpdate(
		NULL, NULL, NULL, NULL,
		1.0f, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
	    );
	    if(ProgressDialogStopCount() > 0)
		status = -4;
	}

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}

/*
 *	Extract object from a RAR archive.
 *
 *	Inputs assumed valid.
 */
static gint EDVArchExtractRAR(
	edv_core_struct *core,
	const gchar *arch_path,
	GList *objs_list, const gint nobjs,
	const gboolean extract_all,
	const gchar *dest_path,
	GList **new_paths_list_rtn,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive, gboolean *yes_to_all,
	const gboolean preserve_directories,
	const gboolean preserve_timestamps
)
{
	const gchar *prog_rar = CFGItemListGetValueS(
	    core->cfg_list, EDV_CFG_PARM_PROG_RAR
	);
	FILE *fp;
	gint status, p, nobjs_extracted;
	gchar	*cmd = NULL,
		*pwd = NULL,
		*stdout_path = NULL,
		*stderr_path = NULL;
	const gchar *src_path;
	GList *glist;
	edv_archive_object_struct *obj;

#define CLEANUP_RETURN(_v_)	{		\
 g_free(cmd);					\
 g_free(stdout_path);				\
 g_free(stderr_path);				\
						\
 /* Restore the previous working dir */		\
 if(pwd != NULL) {				\
  EDVSetCWD(pwd);				\
  g_free(pwd);					\
 }						\
						\
 return(_v_);					\
}

	/* Record previous working dir and set new working dir */
	pwd = EDVGetCWD();
	if(EDVSetCWD(dest_path))
	{
	    core->archive_last_error =
"Unable to change working directory to the destination directory.";
	    CLEANUP_RETURN(-1);
	}

	/* Format the extract object(s) from the archive command */
	cmd = g_strdup_printf(
	    "\"%s\" %c%s \"-p%s\" -kb -o+ -y -c- \"%s\"",
	    prog_rar,
	    preserve_directories ? 'x' : 'e',
	    preserve_timestamps ? " -tsm -tsc -tsa" : "",
	    (STRISEMPTY(password)) ? "-" : password,
	    arch_path
	);
	if(cmd == NULL)
	{
	    core->archive_last_error =
"Unable to generate the extract command.";
	    CLEANUP_RETURN(-1);
	}
	/* Append the objects to extract to the command string
	 * only if not extracting all the objects
	 */
	if(!extract_all)
	{
	    for(glist = objs_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		obj = EDV_ARCHIVE_OBJECT(glist->data);
		if(obj == NULL)
		    continue;

		src_path = obj->full_path;
		if(STRISEMPTY(src_path))
		    continue;

		/* Do not put tailing deliminators on directories
		 * for directory objects in RAR archives
		 */

		cmd = G_STRCAT(cmd, " \"");
		cmd = G_STRCAT(cmd, src_path);
		cmd = G_STRCAT(cmd, "\"");
	    }
	}
	if(cmd == NULL)
	{
	    core->archive_last_error =
"Unable to generate the extract command.";
	    CLEANUP_RETURN(-1);
	}


	/* Generate output file paths */
	stdout_path = EDVTmpName(NULL);
	stderr_path = EDVTmpName(NULL);

	/* Begin extracting */

	status = 0;
	nobjs_extracted = 0;

	/* Execute the extract objects from archive command */
	p = (gint)ExecOE(
	    (const char *)cmd,
	    (const char *)stdout_path,
	    (const char *)stderr_path
	);
	if(p <= 0)
	{
	    core->archive_last_error = "Unable to execute the extract command.";
	    CLEANUP_RETURN(-1);
	}

	g_free(cmd);
	cmd = NULL;

	/* Open the output file for reading */
	fp = fopen((const char *)stdout_path, "rb");
	if(fp != NULL)
	{
	    gint buf_pos = 0, line_count = 0;
	    gfloat progress = 0.0f;
	    gchar buf[10000];
	    gboolean need_break = FALSE;

	    /* Begin reading the output file */
	    while(TRUE)
	    {
		/* Update progress? */
		if(show_progress && ProgressDialogIsQuery())
		{
		    ProgressDialogUpdate(
			NULL, NULL, NULL, NULL,
			progress, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
		    );
		    if(ProgressDialogStopCount() > 0)
		    {
			INTERRUPT(p);
			p = 0;

			status = -4;
			break;
		    }
		}

		/* Check if there is new data to be read from the output
		 * file
		 */
		if(FPending(fp))
		{
		    gint c;
		    gboolean got_complete_line = FALSE;

		    /* Copy all available data from the current output
		     * file position to its end to the line buffer buf
		     */
		    while(TRUE)
		    {
			c = (gint)fgetc(fp);
			if((int)c == EOF)
			{
			    clearerr(fp);
			    break;
			}

			if(ISCR(c))
			{
			    got_complete_line = TRUE;
			    line_count++;

			    if(buf_pos < sizeof(buf))
				buf[buf_pos] = '\0';
			    else
				buf[sizeof(buf) - 1] = '\0';
			    buf_pos = 0;

			    break;
			}

			if(buf_pos < sizeof(buf))
			{
			    buf[buf_pos] = c;
			    buf_pos++;
			}
		    }
		    if(got_complete_line &&
		       (strcasepfx(buf, "Extracting  ") ||
		        strcasepfx(buf, "Creating  "))
		    )
		    {
			gchar *s = buf, *s2, *extracted_path;

			/* Seek s past the prefix to the path value */
			while(ISBLANK(*s))
			    s++;
			while(!ISBLANK(*s) && (*s != '\0'))
			    s++;
			while(ISBLANK(*s))
			    s++;

			/* Cap s at the end deliminator "  "
			 *
			 * The RAR archiver has no formal deliminator
			 * to denote the end of the object's name and
			 * the "OK" string for each line
			 */
			s2 = strstr(s, "  ");
			if(s2 == NULL)
			    s2 = strchr(s, ' ');
			if(s2 == NULL)
			    s2 = strchr(s, '\t');
			if(s2 != NULL)
			    *s2 = '\0';

			extracted_path = STRDUP(PrefixPaths(dest_path, s));
			StripPath(extracted_path);

			/* Append this path to the list of paths
			 * extracted from the archive
			 */
			if(new_paths_list_rtn != NULL)
			    *new_paths_list_rtn = g_list_append(
				*new_paths_list_rtn, STRDUP(extracted_path)
			    );

			/* Update the progress dialog's label? */
			if(show_progress && !STRISEMPTY(extracted_path))
			{
			    gchar	*p1 = EDVShortenPath(
				s, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
			    ),
					*p2 = EDVShortenPath(
				extracted_path, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
			    ),
					*msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Extraer:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Extraire:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Extrahieren:\n\
\n\
    %s\n\
\n\
Zu:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Estrarre:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Onttrekken:\n\
\n\
    %s\n\
\n\
Te:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Extrair:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Trekking Ut:\n\
\n\
    %s\n\
\n\
Til:\n\
\n\
    %s\n"
#else
"Extracting:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n"
#endif
				, p1, p2
			    );
			    EDVArchExtractMapProgressDialog(
				msg, progress, toplevel, FALSE
			    );
			    g_free(msg);
			    g_free(p1);
			    g_free(p2);
			}

			nobjs_extracted++;
			progress = (nobjs > 0) ?
			    ((gfloat)nobjs_extracted / (gfloat)nobjs) : 0.0f;

			g_free(extracted_path);

			continue;
		    }
		}

		if(need_break)
		    break;

		/* Check if the extract process has exited, if it has
		 * then we set need_break to TRUE. Which will be
		 * tested on the next loop if there is still no more
		 * data to be read
		 */
		if(!ExecProcessExists(p))
		    need_break = TRUE;

		usleep(8000);
	    }

	    fclose(fp);
	}

	/* Remove output files */
	UNLINK(stdout_path);
	UNLINK(stderr_path);

	/* Report the final progress? */
	if(show_progress && (status == 0) &&
	   ProgressDialogIsQuery()
	)
	{
	    ProgressDialogUpdate(
		NULL, NULL, NULL, NULL,
		1.0f, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
	    );
	    if(ProgressDialogStopCount() > 0)
		status = -4;
	}

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}

/*
 *	Extracts objects from the archive.
 *
 *	The arch_path specifies the archive.
 *
 *	The objs_list specifies the list of edv_archive_object_struct *
 *	objects to extract from the archive.
 *
 *	The dest_path specifies the absolute path of the location to
 *	extract the objects in the archive to.
 *
 *	If new_paths_list_rtn is not NULL then a list of gchar * paths
 *	describing the objects that have been extracted from the archive
 *	will be returned. The calling function must delete the returned
 *	list and each string.
 */
gint EDVArchExtract(
	edv_core_struct *core,
	const gchar *arch_path,
	GList *objs_list,
	const gboolean extract_all,
	const gchar *dest_path,
	GList **new_paths_list_rtn,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive, gboolean *yes_to_all,
	const gboolean preserve_directories,
	const gboolean preserve_timestamps
)
{
	const gulong time_start = (gulong)time(NULL);
	gint status, nobjs;
	const gchar *arch_name;
	gchar *ldest_path = NULL;
	GList *glist;

#define CLEANUP_RETURN(_v_)	{		\
 g_free(ldest_path);				\
						\
 return(_v_);					\
}
	if(ProgressDialogIsQuery())
	{
	    EDVArchExtractCopyErrorMessage(
		core,
"An operation is already in progress, please try again later."
	    );
	    return(-6);
	}

	/* Clear the last error message */
	EDVArchExtractCopyErrorMessage(core, NULL);

	if(new_paths_list_rtn != NULL)
	    *new_paths_list_rtn = NULL;

	if((core == NULL) || STRISEMPTY(arch_path) ||
	   STRISEMPTY(dest_path) || (yes_to_all == NULL)
	)
	{
	    EDVArchExtractCopyErrorMessage(core, "Invalid value.");
	    return(-2);
	}

	/* Nothing to extract? */
	if(objs_list == NULL)
	    return(0);

	arch_name = g_basename(arch_path);

	/* Make a copy of the destination path and simplify it */
	ldest_path = STRDUP(dest_path);
	EDVSimplifyPath(ldest_path);

	if(!g_path_is_absolute(ldest_path))
	{
	    gchar *msg = g_strdup_printf(
"Extract to location is not an absolute path:\n\
\n\
    %s",
		ldest_path
	    );
	    EDVArchExtractCopyErrorMessage(core, msg);
	    g_free(msg);
	    CLEANUP_RETURN(-2);
	}

	/* Get the total number of objects to extract */
	nobjs = g_list_length(objs_list);

	/* Do overwrite check? */
	if(interactive && !(*yes_to_all))
	{
	    /* Do overwrite check by iterating through the given list of
	     * objects in the archive and check an object at the would
	     * be extracted location exists
	     */
	    gboolean got_cancel = FALSE;
	    gchar *tar_path;
	    const gchar *src_path;
	    edv_archive_object_struct *obj;
	    struct stat lstat_buf;

	    /* Iterate through the list of objects to extract and check
	     * if their extracted locations already exist, prompt the
	     * user to overwrite or cancel in each case
	     */
	    for(glist = objs_list; glist != NULL; glist = g_list_next(glist))
	    {
		obj = EDV_ARCHIVE_OBJECT(glist->data);
		if(obj == NULL)
		    continue;

		/* Get the path of the object within the archive */
		src_path = obj->full_path;
		if(STRISEMPTY(src_path))
		    continue;

		/* Generate the path of the would be extracted location
		 * for this object in the archive
		 */
		if(preserve_directories)
		    tar_path = STRDUP(PrefixPaths(
			ldest_path, src_path
		    ));
		else
		    tar_path = STRDUP(PrefixPaths(
			ldest_path, g_basename(src_path)
		    ));
		if(tar_path == NULL)
		    continue;

		SimplifyPath(tar_path);

		/* Check if an object already exists at the would be
		 * extracted location, if an object exists then prompt
		 * for extract overwrite
		 */
		if(lstat((const char *)tar_path, &lstat_buf))
		    status = CDIALOG_RESPONSE_YES;
		else
		    status = EDVArchExtractConfirmOverwrite(
			core, toplevel,
			src_path,	/* Source object in archive path*/
			tar_path,	/* Target extracted object path */
			obj,		/* Source object in archive stats */
			&lstat_buf	/* Target extracted object stats */
		    );

		g_free(tar_path);

		/* Check user response */
		switch(status)
		{
		  case CDIALOG_RESPONSE_YES_TO_ALL:
		    *yes_to_all = TRUE;
		  case CDIALOG_RESPONSE_YES:
		    break;
		  default:	/* All else assume cancel */
		    got_cancel = TRUE;
		    break;
		}
		if(*yes_to_all)
		    break;
		if(got_cancel)
		    break;
	    }

	    /* User aborted? */
	    if(got_cancel)
	    {
		CLEANUP_RETURN(-4);
	    }
	}

	/* Check if the destination path does not exist, if it does not
	 * then we first need to update the list of new path returns
	 * with all non-existent compoent directories of the destination
	 * path and create all compoent directories of the destination
	 * path
	 */
	if(EDVArchExtractRMkDir(ldest_path, new_paths_list_rtn))
	{
	    const gint error_code = (gint)errno;
	    gchar *msg = g_strdup_printf(
"Unable to create the extract to location:\n\
\n\
    %s\n\
\n\
%s.",
		ldest_path, g_strerror(error_code)
	    );
	    EDVArchExtractCopyErrorMessage(core, msg);
	    g_free(msg);
	    CLEANUP_RETURN(-1);
	}


	/* Begin extracting the source object to the destination specified
	 * by dest_obj from the archive object arch_path
	 *
	 * The extracting method will be determined by taking the
	 * extension of the archive object's name
	 */

	/* ARJ Archive */
	if(EDVIsExtension(arch_name, ".arj"))
	{
	    status = EDVArchExtractARJ(
		core,
		arch_path,
		objs_list, nobjs, extract_all,
		ldest_path, new_paths_list_rtn,
		toplevel, show_progress, interactive, yes_to_all,
		preserve_directories,
		preserve_timestamps
	    );
	}
	/* LHA Archive */
	else if(EDVIsExtension(arch_name, ".lha"))
	{
	    status = EDVArchExtractLHA(
		core,
		arch_path,
		objs_list, nobjs, extract_all,
		ldest_path, new_paths_list_rtn,
		toplevel, show_progress, interactive, yes_to_all,
		preserve_directories,
		preserve_timestamps
	    );
	}
	/* RAR Archive */
	else if(EDVIsExtension(arch_name, ".rar"))
	{
	    status = EDVArchExtractRAR(
		core,
		arch_path,
		objs_list, nobjs, extract_all,
		ldest_path, new_paths_list_rtn,
		password,
		toplevel, show_progress, interactive, yes_to_all,
		preserve_directories,
		preserve_timestamps
	    );
	}
	/* Tape Archive (Compressed) */
	else if(EDVIsExtension(arch_name, ".tar.Z"))
	{
	    status = EDVArchExtractTapeArchive(
		core,
		arch_path,
		objs_list, nobjs, extract_all,
		ldest_path, new_paths_list_rtn,
		toplevel, show_progress, interactive, yes_to_all,
		preserve_directories,
		preserve_timestamps,
		TRUE,		/* Is compress compressed */
		FALSE,		/* Not gzip compressed */
		FALSE           /* Not bzip2 compressed */
	    );
	}
	/* Tape Archive (GZip) */
	else if(EDVIsExtension(arch_name, ".tgz .tar.gz"))
	{
	    status = EDVArchExtractTapeArchive(
		core,
		arch_path,
		objs_list, nobjs, extract_all,
		ldest_path, new_paths_list_rtn,
		toplevel, show_progress, interactive, yes_to_all,
		preserve_directories,
		preserve_timestamps,
		FALSE,		/* Not compress compressed */
		TRUE,		/* Is gzip compressed */
		FALSE		/* Not bzip2 compressed */
	    );
	}
	/* Tape Archive (BZip2) */
	else if(EDVIsExtension(arch_name, ".tar.bz2"))
	{
	    status = EDVArchExtractTapeArchive(
		core,
		arch_path,
		objs_list, nobjs, extract_all,
		ldest_path, new_paths_list_rtn,
		toplevel, show_progress, interactive, yes_to_all,
		preserve_directories,
		preserve_timestamps,
		FALSE,		/* Not compress compressed */
		FALSE,		/* Not gzip compressed */
		TRUE		/* Is bzip2 compressed */
	    );
	}
	/* Tape Archive */
	else if(EDVIsExtension(arch_name, ".tar"))
	{
	    status = EDVArchExtractTapeArchive(
		core,
		arch_path,
		objs_list, nobjs, extract_all,
		ldest_path, new_paths_list_rtn,
		toplevel, show_progress, interactive, yes_to_all,
		preserve_directories,
		preserve_timestamps,
		FALSE,		/* Not compress compressed */
		FALSE,		/* Not gzip compressed */
		FALSE		/* Not bzip2 compressed */
	    );
	}
	/* X Archive */
	else if(EDVIsExtension(arch_name, ".xar"))
	{
	    status = EDVArchExtractXArchive(
		core,
		arch_path,
		objs_list, nobjs, extract_all,
		ldest_path, new_paths_list_rtn,
		password,
		toplevel, show_progress, interactive, yes_to_all,
		preserve_directories,
		preserve_timestamps
	    );
	}
	/* PKZip Archive */
	else if(EDVIsExtension(arch_name, ".xpi .zip"))
	{
	    status = EDVArchExtractPKZip(
		core,
		arch_path,
		objs_list, nobjs, extract_all,
		ldest_path, new_paths_list_rtn,
		password,
		toplevel, show_progress, interactive, yes_to_all,
		preserve_directories,
		preserve_timestamps
	    );
	}
	else
	{
	    core->archive_last_error = "Unsupported archive format.";
	    status = -2;
	}

	/* Record history */
	EDVAppendHistory(
	    core,
	    EDV_HISTORY_ARCHIVE_OBJECT_EXTRACT,
	    time_start, (gulong)time(NULL),
	    status,
	    arch_path,		/* Source */
	    ldest_path,		/* Target */
	    core->archive_last_error		/* Comment */
	);

	/* Need to flush disk changes since the archive may have been
	 * modified on another process and the changes have not reached
	 * our process yet
	 */
	sync();

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}
