#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <fnmatch.h>
#include <sys/stat.h>
#include <glib.h>

#include "../include/string.h"
#include "../include/fio.h"
#include "../include/disk.h"

#include "edv_types.h"
#include "edv_recycled_obj.h"
#include "edv_recbin_index.h"
#include "edv_archive_obj.h"
#include "edv_archive_stat.h"
#include "edv_find.h"


static gchar *EDVFindGrepExcerptFile(
	const gchar *path, const gchar *expression,
	gboolean case_sensitive, gint *line_index
);

/* Object In Archive Finding */
static void EDVFindArchiveObjectByNameIterate(
	edv_core_struct *core,
	const gchar *arch_path,
	const gchar *expression,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, const struct stat *, gpointer),
	gpointer match_data,
	gint *nmatches, gboolean *stop_find
);
gint EDVFindArchiveObjectByName(
	edv_core_struct *core,
	const gchar *arch_path,
	const gchar *expression,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, const struct stat *, gpointer),
	gpointer match_data
);

/* Recycled Object Finding */
static void EDVFindRecycledObjectByContentIterate(
	const gchar *recycled_index_file,
	const gchar *expression,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *, const struct stat *,
		const gchar *, gint,
		gpointer
	),
	gpointer match_data,
	gint *nmatches, gboolean *stop_find
);
gint EDVFindRecycledObjectByContent(
	const gchar *recycled_index_file,
	const gchar *expression,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *, const struct stat *,
		const gchar *, gint,
		gpointer
	),
	gpointer match_data
);

static void EDVFindRecycledObjectByNameIterate(
	const gchar *recycled_index_file,
	const gchar *expression,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, const struct stat *, gpointer),
	gpointer match_data,
	gint *nmatches, gboolean *stop_find
);
gint EDVFindRecycledObjectByName(
	const gchar *recycled_index_file,
	const gchar *expression,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, const struct stat *, gpointer),
	gpointer match_data
);


/* Object Finding */
static void EDVFindObjectByContentIterate(
	const gchar *path,
	const gchar *expression,
	const gboolean recursive,
	const gboolean follow_links,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *, const struct stat *,
		const gchar *, gint,
		gpointer
	),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
);
gint EDVFindObjectByContent(
	const gchar *start_dir,
	const gchar *expression,
	const gboolean recursive,
	const gboolean follow_links,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *, const struct stat *,
		const gchar *, gint,
		gpointer
	),
	gpointer match_data
);

static void EDVFindObjectByNameIterate(
	const gchar *path,
	const gchar *expression,
	const gboolean recursive,
	const gboolean follow_links,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, const struct stat *, gpointer),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
);
gint EDVFindObjectByName(
	const gchar *start_dir,
	const gchar *expression,
	const gboolean recursive,
	const gboolean follow_links,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, const struct stat *, gpointer),
	gpointer match_data
);


#define EDV_FIND_EXCERPT_PAD_MAX	40


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)

#define ISCR(c)		(((c) == '\n') || ((c) == '\r'))
#define ISBLANK(c)	(((c) == ' ') || ((c) == '\t'))


/*
 *	Opens the file specified by path and searches for a string
 *	within it that matches the given expression.
 *
 *	If a match is made then a dynamically allocated string will be
 *	returned containing the excerpt of the matched line.
 */
static gchar *EDVFindGrepExcerptFile(
	const gchar *path, const gchar *expression,
	gboolean case_sensitive, gint *line_index
)
{
	FILE *fp;
	gint c;
	gulong line_start_pos;
	const gchar *expre_ptr;
	gchar *excerpt;

	if(line_index != NULL)
	    *line_index = 0;

	if(STRISEMPTY(path) || STRISEMPTY(expression))
	    return(NULL);

	/* Open the file for reading */
	fp = fopen((const char *)path, "rb");
	if(fp == NULL)
	    return(NULL);

	/* Set the expression to the beginning */
	expre_ptr = expression;

	line_start_pos = 0l;
	excerpt = NULL;

	/* Search for the expression in the file */
	while(TRUE)
	{
	    /* Get the next character */
	    c = (gint)fgetc(fp);
	    if((int)c == EOF)
		break;

	    /* If not performing case sensitive search then make the
	     * character that was just read uppercase
	     */
	    if(!case_sensitive)
		c = (gint)toupper((int)c);

	    /* If this character marks the end of the line then record
	     * the last line starting position as one character ahead
	     * of the current position
	     */
	    if(ISCR(c))
	    {
		if(line_index != NULL)
		    *line_index = (*line_index) + 1;

		/* Record the line start position */
		line_start_pos = (gulong)ftell(fp);
	    }

	    /* Does this character matches the current character of the
	     * expression?
	     */
	    if(((gchar)c == *expre_ptr) && ((gchar)c != '\0'))
	    {
		/* Increment to the next character in the expression */
		expre_ptr++;
	    }
	    /* Reset expression back to the beginning as needed */
	    else if(expre_ptr != expression)
	    {
		expre_ptr = expression;
	    }

	    /* Matched the entire expression?
	     *
	     * Test if the expression pointer has been incremented all
	     * the way to the end which suggests that the entire
	     * expression has been matched
	     */
	    if(*expre_ptr == '\0')
	    {
		gchar *s;

		/* Seek to the start of the line */
		fseek(fp, (long)line_start_pos, SEEK_SET);

		/* Get current line as a string */
		s = (gchar *)FGetStringLiteral(fp);
		if(s != NULL)
		{
		    /* Seek to the expression in the excerpt */
		    gchar *s_exp_start = (case_sensitive) ?
			(gchar *)strstr((char *)s, (const char *)expression) :
			(gchar *)strcasestr((char *)s, (const char *)expression);
		    if(s_exp_start != NULL)
		    {
			const gint	left_pad_max = EDV_FIND_EXCERPT_PAD_MAX,
					right_pad_max = left_pad_max;
			gchar	*s2,
				*s_exp_end = s_exp_start + STRLEN(expression);

			/* Sanitize and limit length of the excerpt */
			for(s2 = s; *s2 != '\0'; s2++)
			{
			    /* Non-ACSII character? */
			    if(!isascii((int)(*s2)))
				*s2 = ' ';	/* Replace with space */

			    /* Exceeded the maximum excerpt length? */
			    if(s2 > s_exp_end)
			    {
				if((gint)(s2 - s_exp_end) >= right_pad_max)
				{
				    /* Cap the excerpt and break */
				    *s2 = '\0';
				    break;
				}
			    }
			}

			/* Copy the sanitized excerpt to the return */
			g_free(excerpt);
			if((s_exp_start - s) > left_pad_max)
			    excerpt = STRDUP(s_exp_start - left_pad_max);
			else
			    excerpt = STRDUP(s);
		    }

		    g_free(s);
		}

		break;	/* Break after the first match */

	    }	/* Matched the entire expression? */
	}

	/* Close the file */
	fclose(fp);

	return(excerpt);
}


/*
 *      Called by EDVFindArchiveObjectByName() to perform one find iteration
 *      on the given path.
 */
static void EDVFindArchiveObjectByNameIterate(
	edv_core_struct *core,
	const gchar *arch_path,
	const gchar *expression,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, const struct stat *, gpointer),
	gpointer match_data,
	gint *nmatches, gboolean *stop_find
)
{
	gint i, nobjs;
	GList *glist, *obj_list;
	edv_archive_object_struct *obj;

	if(*stop_find)
	    return;

	/* Get the listing of all the objects in the archive */
	obj_list = EDVArchStatList(
	    core,
	    arch_path,
	    NULL,			/* Get all the objects */
	    NULL,			/* No filter */
	    NULL,			/* No password */
	    NULL, NULL			/* No progress callback */
	);

	/* Iterate through all the objects in the archive */
	nobjs = g_list_length(obj_list);
	for(glist = obj_list, i = 0;
	    glist != NULL;
	    glist = g_list_next(glist), i++
	)
	{
	    obj = EDV_ARCHIVE_OBJECT(glist->data);
	    if(obj == NULL)
		continue;

	    /* User requested that the search be stopped? */
	    if(*stop_find)
		break;

	    /* Report progress */
	    if(progress_cb != NULL)
	    {
		if(progress_cb(
		    obj->name,
		    (gfloat)i / (gfloat)nobjs,
		    progress_data
		))
		{
		    *stop_find = TRUE;
		    continue;
		}
	    }

	    /* Does the object in the archive have a name? */
	    if(!STRISEMPTY(obj->name))
	    {
		gchar *s = STRDUP(obj->name);
		if(!case_sensitive)
		    g_strup(s);

		/* Names match? */
		if(!fnmatch((const char *)expression, (const char *)s, 0))
		{
		    /* Convert the object in the archive stats */
		    struct stat *lstat_buf = (struct stat *)g_malloc0(
			sizeof(struct stat)
		    );
		    if(lstat_buf != NULL)
		    {
			lstat_buf->st_mode = 
 EDVObjectGetTypeFromEDVType(obj->type) |
 EDVObjectGetPermissionsFromEDVPermissions(obj->permissions)
			;
			lstat_buf->st_size = (size_t)obj->size; 
			lstat_buf->st_atime = (time_t)obj->access_time;
			lstat_buf->st_mtime = (time_t)obj->modify_time;
			lstat_buf->st_ctime = (time_t)obj->change_time;
		    }

		    /* Increment the match count */
		    *nmatches = (*nmatches) + 1;

		    /* Repor the match */
		    if(match_cb != NULL)
			match_cb(
			    obj->full_path,	/* Path */
			    lstat_buf,		/* Stats */
			    match_data		/* Data */
			);

		    g_free(lstat_buf);
		}

		g_free(s);	/* Delete copy of object's name */
	    }
	}

	/* Delete the archive object stats */
	if(obj_list != NULL)
	{
	    g_list_foreach(obj_list, (GFunc)EDVArchObjectDelete, NULL);
	    g_list_free(obj_list);
	}
}

/*
 *      Searches for a object who's name matches the given expression
 *      in the archive.
 *
 *      If progress_cb is not NULL then it will be called frequently to
 *      update progress. If progress_cb returns non-zero then the
 *      operation will end prematurly.
 *
 *      If match_cb is not NULL then each matched object will be reported
 *      to it.
 *
 *      Returns the number of occurances found.
 */
gint EDVFindArchiveObjectByName(
	edv_core_struct *core,
	const gchar *arch_path,
	const gchar *expression,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, const struct stat *, gpointer),
	gpointer match_data
)
{
	gboolean stop_find;
	gint nmatches;
	gchar *lexpression;

	if((core == NULL) || STRISEMPTY(arch_path) ||
	   STRISEMPTY(expression)
	)
	    return(0);

	if(!strcmp((const char *)expression, "*"))
	    return(0);

	/* Convert expression to upper case if not performing case
	 * sensitive matching
	 */
	lexpression = STRDUP(expression);
	if(!case_sensitive)
	    g_strup(lexpression);

	/* Begin search */
	nmatches = 0;
	stop_find = FALSE;
	EDVFindArchiveObjectByNameIterate(
	    core,
	    arch_path, lexpression,
	    case_sensitive,
	    progress_cb, progress_data,
	    match_cb, match_data,
	    &nmatches, &stop_find
	);

	g_free(lexpression);

	return(nmatches);
}


/*
 *	Called by EDVFindRecycledObjectByContent() to perform one find
 *	iteration on the given path.
 */
static void EDVFindRecycledObjectByContentIterate(
	const gchar *recycled_index_file,
	const gchar *expression,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *, const struct stat *,
		const gchar *, gint,
		gpointer
	),
	gpointer match_data,
	gint *nmatches, gboolean *stop_find
)
{
	gint i, nobjs;
	gchar *recycled_dir;
	edv_recbin_index_struct *rbi_ptr;
	edv_recycled_object_struct *obj;

	if(*stop_find)
	    return;

	/* Get recycled objects directory from the recycled objects
	 * index file path
	 */
	recycled_dir = g_dirname(recycled_index_file);

	nobjs = EDVRecBinIndexGetTotal(recycled_index_file);

	/* Get recycled objects directory index pointer used to iterate
	 * through all recycled objects
	 */
	i = 0;
	rbi_ptr = EDVRecBinIndexOpen(recycled_index_file);
	while(!EDVRecBinIndexNext(rbi_ptr))
	{
	    obj = rbi_ptr->obj;
	    if(obj == NULL)
	    {
		i++;
		continue;
	    }

	    if(*stop_find)
		break;

	    /* Report progress */
	    if(progress_cb != NULL)
	    {
		if(progress_cb(
		    obj->name,
		    (gfloat)i / (gfloat)nobjs,
		    progress_data
		))
		{
		    *stop_find = TRUE;
		    i++;
		    continue;
		}
	    }

	    /* Do match */
	    if(TRUE)
	    {
		/* Contents match? */
		gint line_index;
		gchar	*full_path = g_strdup_printf(
		    "%s%c%i",
		    recycled_dir, G_DIR_SEPARATOR, obj->index
		),
			*excerpt = EDVFindGrepExcerptFile(
		    full_path, expression, case_sensitive,
		    &line_index
		);
		if(excerpt != NULL)
		{
		    /* Got match, increment the match count and report
		     * the match
		     */
		    gchar *index_str = g_strdup_printf("%u", obj->index);

		    /* Convert the recycled object's stats */
		    struct stat *lstat_buf = (struct stat *)g_malloc0(
			sizeof(struct stat)
		    );
		    if(lstat_buf != NULL)
		    {
			lstat_buf->st_mode =
 EDVObjectGetTypeFromEDVType(obj->type) |
 EDVObjectGetPermissionsFromEDVPermissions(obj->permissions)
			;
			lstat_buf->st_size = (size_t)obj->size;
			lstat_buf->st_uid = (uid_t)obj->owner_id;
			lstat_buf->st_gid = (gid_t)obj->group_id;
			lstat_buf->st_atime = (time_t)obj->access_time;
			lstat_buf->st_mtime = (time_t)obj->modify_time;
			lstat_buf->st_ctime = (time_t)obj->change_time;
		    }

		    /* Increment the match count */
		    *nmatches = (*nmatches) + 1;

		    /* Report the match */
		    if(match_cb != NULL)
			match_cb(
			    index_str,		/* Path as index string */
			    lstat_buf,		/* Stats */
			    excerpt,		/* Excerpt */
			    line_index,		/* Line Index */
			    match_data		/* Data */
			); 

		    g_free(lstat_buf);
		    g_free(index_str);
		    g_free(excerpt);
		}

		g_free(full_path);	/* Delete full path */
	    }
	}

	EDVRecBinIndexClose(rbi_ptr);

	g_free(recycled_dir);
}

/*
 *      Searches for recycled object who contains a string that matches
 *	the given expression in the recycled objects directory obtained
 *	from the given recycle objects index file.
 *
 *      If progress_cb is not NULL then it will be called frequently to
 *      update progress. If progress_cb returns non-zero then the
 *      operation will end prematurly.
 *
 *      If match_cb is not NULL then each matched object will be reported
 *      to it.
 *
 *      Returns the number of occurances found.
 */
gint EDVFindRecycledObjectByContent(
	const gchar *recycled_index_file,
	const gchar *expression,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *, const struct stat *,
		const gchar *, gint,
		gpointer
	),
	gpointer match_data
)
{
	gboolean stop_find;
	gint nmatches;
	gchar *lexpression;

	if(STRISEMPTY(recycled_index_file) || STRISEMPTY(expression))
	    return(0);

	/* Convert expression to upper case if not performing case
	 * sensitive matching
	 */
	lexpression = STRDUP(expression);
	if(!case_sensitive)
	    g_strup(lexpression);

	/* Begin search */
	nmatches = 0;
	stop_find = FALSE;
	EDVFindRecycledObjectByContentIterate(
	    recycled_index_file, lexpression,
	    case_sensitive,
	    progress_cb, progress_data,
	    match_cb, match_data,
	    &nmatches, &stop_find
	);

	g_free(lexpression);

	return(nmatches);
}


/*
 *	Called by EDVFindRecycledObjectByName() to perform one find iteration
 *	on the given path.
 */
static void EDVFindRecycledObjectByNameIterate(
	const gchar *recycled_index_file,
	const gchar *expression,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, const struct stat *, gpointer),
	gpointer match_data,
	gint *nmatches, gboolean *stop_find
)
{
	gint i, nobjs;
	gchar *recycled_dir;
	edv_recbin_index_struct *rbi_ptr;
	const edv_recycled_object_struct *obj;

	if(*stop_find)
	    return;

	/* Get recycled objects directory from the recycled objects
	 * index file path
	 */
	recycled_dir = g_dirname(recycled_index_file);

	nobjs = EDVRecBinIndexGetTotal(recycled_index_file);

	/* Get recycled objects directory index pointer used to iterate
	 * through all recycled objects
	 */
	i = 0;
	rbi_ptr = EDVRecBinIndexOpen(recycled_index_file);
	while(!EDVRecBinIndexNext(rbi_ptr))
	{
	    obj = rbi_ptr->obj;
	    if(obj == NULL)
	    {
		i++;
		continue;
	    }

	    if(*stop_find)
		break;

	    /* Report progress */
	    if(progress_cb != NULL)
	    {
		if(progress_cb(
		    obj->name,
		    (gfloat)i / (gfloat)nobjs,
		    progress_data
		))
		{
		    *stop_find = TRUE;
		    i++;
		    continue;
		}
	    }

	    /* Does the object have a name? */
	    if(!STRISEMPTY(obj->name))
	    {
		gchar *s = STRDUP(obj->name);

		if(!case_sensitive)
		    g_strup(s);

		/* Names match? */
		if(!fnmatch((const char *)expression, (const char *)s, 0))
		{
		    /* Got match, increment the match count and report
		     * the match
		     */
		    gchar *index_str = g_strdup_printf("%u", obj->index);

		    /* Convert the recycled object's stats */
		    struct stat *lstat_buf = (struct stat *)g_malloc0(
			sizeof(struct stat)
		    );
		    if(lstat_buf != NULL)
		    {
			lstat_buf->st_mode = 
 EDVObjectGetTypeFromEDVType(obj->type) |
 EDVObjectGetPermissionsFromEDVPermissions(obj->permissions)
			;
			lstat_buf->st_size = (size_t)obj->size;
			lstat_buf->st_uid = (uid_t)obj->owner_id;
			lstat_buf->st_gid = (gid_t)obj->group_id;
			lstat_buf->st_atime = (time_t)obj->access_time;
			lstat_buf->st_mtime = (time_t)obj->modify_time;
			lstat_buf->st_ctime = (time_t)obj->change_time;
		    }

		    /* Increment the match count */
		    *nmatches = (*nmatches) + 1;

		    /* Report the match */
		    if(match_cb != NULL)
			match_cb(
			    index_str,		/* Index as path */
			    lstat_buf,		/* Stats */
			    match_data		/* Data */
			);

		    g_free(lstat_buf);
		    g_free(index_str);
		}

		g_free(s);
	    }
	}

	EDVRecBinIndexClose(rbi_ptr);

	g_free(recycled_dir);
}

/*
 *	Searches for a recycled object who's name matches the given
 *	expression in the recycled objects directory obtained from the
 *	given recycled objects index file.
 *
 *      If progress_cb is not NULL then it will be called frequently to
 *      update progress. If progress_cb returns non-zero then the
 *      operation will end prematurly.
 *
 *      If match_cb is not NULL then each matched object will be reported
 *      to it.
 *
 *      Returns the number of occurances found.
 */
gint EDVFindRecycledObjectByName(
	const gchar *recycled_index_file,
	const gchar *expression,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, const struct stat *, gpointer),
	gpointer match_data
)
{
	gboolean stop_find;
	gint nmatches;
	gchar *lexpression;

	if(STRISEMPTY(recycled_index_file) || STRISEMPTY(expression))
	    return(0);
	if(!strcmp((const char *)expression, "*"))
	    return(0);

	/* Convert expression to upper case if not matching case
	 * sensitive
	 */
	lexpression = STRDUP(expression);
	if(!case_sensitive)
	    g_strup(lexpression);

	/* Begin search */
	nmatches = 0;
	stop_find = FALSE;
	EDVFindRecycledObjectByNameIterate(
	    recycled_index_file, lexpression,
	    case_sensitive,
	    progress_cb, progress_data,
	    match_cb, match_data,
	    &nmatches, &stop_find
	);

	g_free(lexpression);

	return(nmatches);
}


/*
 *	Called by EDVFindObjectByContent() to perform one find iteration
 *	on the given path and its child objects if path is a directory and
 *	recurisive is TRUE.
 */
static void EDVFindObjectByContentIterate(
	const gchar *path,
	const gchar *expression,
	const gboolean recursive,
	const gboolean follow_links,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *, const struct stat *,
		const gchar *, gint,
		gpointer
	),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
)
{
	gint strc;
	gchar **strv;

	if(*stop_find)
	    return;

	/* Get listing of child objects if the given path is a directory */
	strv = (gchar **)GetDirEntNames2((const char *)path, (int *)&strc);
	if(strv != NULL)
	{
	    gint i, status;
	    const gchar *name;
	    gchar *full_path = NULL;
	    struct stat stat_buf;

	    strv = (gchar **)StringQSort((char **)strv, (int)strc);
	    if(strv == NULL)
		return;

	    for(i = 0; i < strc; i++)
	    {
#define DO_FREE_CONTINUE	{	\
 g_free(full_path);			\
 full_path = NULL;			\
 g_free(strv[i]);			\
 strv[i] = NULL;			\
 continue;				\
}
		name = strv[i];
		if(STRISEMPTY(name))
		    DO_FREE_CONTINUE

		/* User requested stop? */
		if(*stop_find)
		    DO_FREE_CONTINUE

		/* Skip special directory notations */
		if(!strcmp((const char *)name, "..") ||
		   !strcmp((const char *)name, ".")
		)
		    DO_FREE_CONTINUE

		/* Get full path */
		full_path = STRDUP(PrefixPaths(
		    (const char *)path, (const char *)name
		));
		if(full_path == NULL)
		    DO_FREE_CONTINUE

		/* Report progress */
		if(progress_cb != NULL)
		{
		    if(progress_cb(
			full_path,
			recursive ? -1.0f : ((gfloat)i / (gfloat)strc),
			progress_data
		    ))
		    {
			*stop_find = TRUE;
			DO_FREE_CONTINUE
		    }
		}

		/* Match object content */
		if(TRUE)
		{
		    gint line_index;
		    gchar *excerpt = EDVFindGrepExcerptFile(
			full_path, expression,
			case_sensitive, &line_index
		    );
		    if(excerpt != NULL)
		    {
			/* Got match */
			struct stat lstat_buf;

			/* Increment number of matches */
			*nmatches = (*nmatches) + 1;

			/* Get local stats and call match callback */
			status = (gint)lstat((const char *)full_path, &lstat_buf);
			if((match_cb != NULL) && (status == 0))
			    match_cb(
				full_path,	/* Path */
				&lstat_buf,	/* Local Stats */
				excerpt,	/* Excerpt */
				line_index,	/* Line Index */
				match_data	/* Data */
			    );

			g_free(excerpt);
		    }
		}

		/* Get destination stats */
		if(stat((char *)full_path, &stat_buf))
		    DO_FREE_CONTINUE

		/* Is it a directory and should we recurse into it? */
#ifdef S_ISDIR
		if(recursive && S_ISDIR(stat_buf.st_mode))
#else
		if(FALSE)
#endif
		{
		    struct stat lstat_buf;

		    status = (gint)lstat(
			(const char *)full_path, &lstat_buf
		    );
#ifdef S_ISLNK
		    /* Skip links that go to directories in order to
		     * avoid possible infinate recusion
		     */
		    if(status == 0)
		    {
			if(S_ISLNK(lstat_buf.st_mode) && !follow_links)
			    DO_FREE_CONTINUE
		    }
#endif

		    EDVFindObjectByContentIterate(
			full_path, expression,
			recursive, follow_links, case_sensitive,
			progress_cb, progress_data,
			match_cb, match_data,
			nmatches, stop_find
		    );
		}

		DO_FREE_CONTINUE
#undef DO_FREE_CONTINUE
	    }

	    g_free(strv);
	}
}

/*
 *	Searches for an object who contains a string that matches the
 *	given expression starting from the directory specified by
 *	start_dir.
 *
 *	If progress_cb is not NULL then it will be called frequently to
 *	update progress. If progress_cb returns non-zero then the
 *	operation will end prematurly.
 *
 *	If match_cb is not NULL then each matched object will be reported
 *	to it.
 *
 *	Returns the number of occurances found.
 */
gint EDVFindObjectByContent(
	const gchar *start_dir,
	const gchar *expression,
	const gboolean recursive,
	const gboolean follow_links,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *, const struct stat *,
		const gchar *, gint,
		gpointer
	),
	gpointer match_data
)
{
	gboolean stop_find;
	gint nmatches;
	gchar *lexpression;

	if(STRISEMPTY(start_dir) || STRISEMPTY(expression))
	    return(0);

	/* Convert expression to upper case if not matching case
	 * sensitive
	 */
	lexpression = STRDUP(expression);
	if(!case_sensitive)
	    g_strup(lexpression);

	/* Begin search */
	nmatches = 0;
	stop_find = FALSE;
	EDVFindObjectByContentIterate(
	    start_dir, lexpression,
	    recursive, follow_links, case_sensitive,
	    progress_cb, progress_data,
	    match_cb, match_data,
	    &nmatches, &stop_find
	);

	g_free(lexpression);

	return(nmatches);
}


/*
 *	Called by EDVFindObjectByName() to perform one find iteration on
 *	the given path and its child objects if path is a directory and
 *	recurisive is TRUE.
 */
static void EDVFindObjectByNameIterate(
	const gchar *path,
	const gchar *expression,
	const gboolean recursive,
	const gboolean follow_links,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, const struct stat *, gpointer),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
)
{
	gint strc;
	gchar **strv;

	if(*stop_find)
	    return;

	/* Get listing of child objects if the given path is a directory */
	strv = (gchar **)GetDirEntNames2((const char *)path, (int *)&strc);
	if(strv != NULL)
	{
	    gint i, status;
	    const gchar *name;
	    gchar *full_path = NULL;
	    struct stat stat_buf;

	    strv = (gchar **)StringQSort((char **)strv, (int)strc);
	    if(strv == NULL)
		return;

	    for(i = 0; i < strc; i++)
	    {
#define DO_FREE_CONTINUE	{	\
 g_free(full_path);			\
 full_path = NULL;			\
 g_free(strv[i]);			\
 strv[i] = NULL;			\
 continue;				\
}
		name = strv[i];
		if(STRISEMPTY(name))
		    DO_FREE_CONTINUE

		/* User requested stop? */
		if(*stop_find)
		    DO_FREE_CONTINUE

		/* Skip special directory notations */
		if(!strcmp((const char *)name, "..") ||
		   !strcmp((const char *)name, ".")
		)
		    DO_FREE_CONTINUE

		/* Get full path */
		full_path = STRDUP(PrefixPaths(
		    (const char *)path, (const char *)name
		));
		if(full_path == NULL)
		    DO_FREE_CONTINUE

		/* Report progress */
		if(progress_cb != NULL)
		{
		    if(progress_cb(
			full_path,
			recursive ? -1.0f : ((gfloat)i / (gfloat)strc),
			progress_data
		    ))
		    {
			*stop_find = TRUE;
			DO_FREE_CONTINUE
		    }
		}

		/* Match object name */
		if(TRUE)
		{
		    gchar *s = STRDUP(name);

		    if(!case_sensitive)
			g_strup(s);

		    if(!fnmatch((const char *)expression, (const char *)s, 0))
		    {
			/* Got match */
			struct stat lstat_buf;

			/* Increment number of matches */
			*nmatches = (*nmatches) + 1;

			/* Get local stats and call match callback */
			status = (gint)lstat((const char *)full_path, &lstat_buf);
			if((match_cb != NULL) && (status == 0))
			    match_cb(
				full_path,	/* Path */
				&lstat_buf,	/* Stats */
				match_data	/* Data */
			    );
		    }

		    g_free(s);
		}

		/* Get destination stats */
		if(stat((char *)full_path, &stat_buf))
		    DO_FREE_CONTINUE

		/* Is it a directory and should we recurse into it? */
#ifdef S_ISDIR
		if(recursive && S_ISDIR(stat_buf.st_mode))
#else
		if(FALSE)
#endif
		{
		    struct stat lstat_buf;

		    status = (gint)lstat(
			(char *)full_path, &lstat_buf
		    );
#ifdef S_ISLNK
		    /* Skip links that go to directories in order to
		     * avoid possible infinate recusion
		     */
		    if(status == 0)
		    {
			if(S_ISLNK(lstat_buf.st_mode) && !follow_links)
			    DO_FREE_CONTINUE
		    }
#endif

		    EDVFindObjectByNameIterate(
			full_path, expression,
			recursive, follow_links, case_sensitive,
			progress_cb, progress_data,
			match_cb, match_data,
			nmatches, stop_find
		    );
		}

		DO_FREE_CONTINUE
#undef DO_FREE_CONTINUE
	    }

	    g_free(strv);
	}
}

/*
 *	Searches for an object who's name matches the given expression
 *	starting from the directory specified by start_dir.
 *
 *	If progress_cb is not NULL then it will be called frequently to
 *	update progress. If progress_cb returns non-zero then the
 *	operation will end prematurly.
 *
 *	If match_cb is not NULL then each matched object will be reported
 *	to it.
 *
 *	Returns the number of occurances found.
 */
gint EDVFindObjectByName(
	const gchar *start_dir,
	const gchar *expression,
	const gboolean recursive,
	const gboolean follow_links,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, const struct stat *, gpointer),
	gpointer match_data
)
{
	gboolean stop_find;
	gint nmatches;
	gchar *lexpression;

	if(STRISEMPTY(start_dir) || STRISEMPTY(expression))
	    return(0);
	if(!strcmp((const char *)expression, "*"))
	    return(0);

	/* Convert expression to upper case if not matching case
	 * sensitive
	 */
	lexpression = STRDUP(expression);
	if(!case_sensitive)
	    g_strup(lexpression);

	/* Begin search */
	nmatches = 0;
	stop_find = FALSE;
	EDVFindObjectByNameIterate(
	    start_dir, lexpression,
	    recursive, follow_links, case_sensitive,
	    progress_cb, progress_data,
	    match_cb, match_data,
	    &nmatches, &stop_find
	);

	g_free(lexpression);

	return(nmatches);
}
