/*
 *	Lists the UIDs and GIDs
 */
#include <stdlib.h>
#include <time.h>
#include <fnmatch.h>
#include <glib.h>
#include <endeavour2.h>
#include "../config.h"


typedef enum {
	LSIDS_PRINT_UIDS			= (1 << 0),
	LSIDS_PRINT_GIDS			= (1 << 1),
	LSIDS_PRINT_LONG_LIST			= (1 << 2),
	LSIDS_PRINT_GROUP_MEMBERS		= (1 << 3)
} lsids_print_flags;


static void print_usage(const gchar *prog_name);

static gboolean match_name(const gchar *name, GList *names_list);

static void print_uid(
	edv_context_struct *ctx,
	edv_uid_struct *uid,
	const lsids_print_flags print_flags
);
static void print_gid(
	edv_context_struct *ctx,
	edv_gid_struct *gid,
	const lsids_print_flags print_flags
);
static void list_uids(
	edv_context_struct *ctx,
	GList *uids_list, GList *names_list,
	const lsids_print_flags print_flags
);
static void list_gids(
	edv_context_struct *ctx,
	GList *gids_list, GList *names_list,
	const lsids_print_flags print_flags
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


static void print_usage(const gchar *prog_name)
{
	g_print(
"Usage: %s [names(s)...] [options]\n",
	    prog_name
	);

	g_print("\n\
    The [names(s)...] specifies the UID or GID names to list.\n\
\n\
    Note that if you specify wildcards (* or ?) then you need to put\n\
    them between double quotes (\") such as \"*\" or else the shell\n\
    will expand them as regular object names instead of UID or GID\n\
    names.\n\
\n\
    The [options] can be any of the following:\n\
\n\
        -u                      List UIDs (default).\n\
        -g                      List GIDs.\n\
\n\
        -l                      Print long list format.\n\
        -m                      Print group members.\n\
\n\
	--help                  Prints this help screen and exits.\n\
	--version               Prints version information and exits.\n\
\n\
    Return values:\n\
\n\
	0       Success.\n\
	1       General error.\n\
	2       Invalid value.\n\
	3       Systems error or memory allocation error.\n\
	4       User aborted.\n\
\n"
	);
}


/*
 *	Checks if name matches one of the names in names_list.
 *
 *	If the names_list is NULL then this function will always
 *	return TRUE.
 *
 *	Returns TRUE if a match is made, otherwise FALSE.
 */
static gboolean match_name(const gchar *name, GList *names_list)
{
	const gchar *s;
	GList *glist;

	if(name == NULL)
	    return(FALSE);

	if(names_list == NULL)
	    return(TRUE);

	for(glist = names_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    s = (const gchar *)glist->data;
	    if(s == NULL)
		continue;

	    if(!fnmatch(s, name, 0))
		return(TRUE);
	}

	return(FALSE);
}


/*
 *	Prints the UID.
 */
static void print_uid(
        edv_context_struct *ctx,
        edv_uid_struct *uid,
	const lsids_print_flags print_flags
)
{
	if(uid == NULL)
	    return;

	if(print_flags & LSIDS_PRINT_LONG_LIST)
	    g_print(
		"%i\t%i\t%s %s %s\n",
		uid->user_id,
		uid->group_id,
		uid->name,
		uid->home_directory,
		uid->shell_program
	    );
	else if(uid->name != NULL)
	    g_print("%s\n", uid->name);
	else
	    g_print("%i\n", uid->user_id);
}

/*
 *	Prints the GID.
 */
static void print_gid(
        edv_context_struct *ctx,
        edv_gid_struct *gid,
	const lsids_print_flags print_flags
)
{
	const gchar *s;
	GList *glist;

	if(gid == NULL)
	    return;

        if(print_flags & LSIDS_PRINT_LONG_LIST)
	    g_print("%i\t%s\n", gid->group_id, gid->name);
	else if(gid->name != NULL)
	    g_print("%s\n", gid->name);
	else
	    g_print("%i\n", gid->group_id);

	if((print_flags & LSIDS_PRINT_GROUP_MEMBERS) &&
	   (gid->group_members_list != NULL)
	)
	{
	    for(glist = gid->group_members_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		s = (const gchar *)glist->data;
		if(s == NULL)
		    continue;

		g_print("\t\t%s\n", s);
	    }
	}
}


/*
 *	Prints the list of UIDs.
 */
static void list_uids(
	edv_context_struct *ctx,
	GList *uids_list, GList *names_list,
	const lsids_print_flags print_flags
)
{
	GList *glist;
	edv_uid_struct *uid;

	for(glist = uids_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    uid = EDV_UID(glist->data);
	    if(uid == NULL)
		continue;

	    if(!match_name(uid->name, names_list))
		continue;

	    print_uid(ctx, uid, print_flags);
	}
}

/*
 *	Prints the list of GIDs.
 */
static void list_gids(
	edv_context_struct *ctx,
	GList *gids_list, GList *names_list,
	const lsids_print_flags print_flags
)
{
	GList *glist;
	edv_gid_struct *gid;

	for(glist = gids_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    gid = EDV_GID(glist->data);
	    if(gid == NULL)
		continue;

	    if(!match_name(gid->name, names_list))
		continue;

	    print_gid(ctx, gid, print_flags);
	}
}


int main(int argc, char *argv[])
{
	lsids_print_flags print_flags = 0;
	gint i;
	const gchar *arg;
	GList *names_list = NULL;

        /* Initialize the Endeavour2 Context */
	edv_context_struct *ctx = EDVContextNew();
	EDVContextLoadConfigurationFile(ctx, NULL);

#define CLEANUP_RETURN(_v_)	{	\
 if(names_list != NULL) {		\
  g_list_foreach(			\
   names_list, (GFunc)g_free, NULL	\
  );					\
  g_list_free(names_list);		\
 }					\
					\
 /* Shutdown the Endeavour2 context */	\
 EDVContextDelete(ctx);			\
					\
 return(_v_);				\
}

	/* Parse arguments */
	for(i = 1; i < argc; i++)
	{
	    arg = argv[i];
	    if(arg == NULL)
		continue;

	    /* Help */
	    if(!g_strcasecmp(arg, "--help") ||
	       !g_strcasecmp(arg, "-help") ||
	       !g_strcasecmp(arg, "--h") ||
	       !g_strcasecmp(arg, "-h")
	    )
	    {
		print_usage(argv[0]);
		CLEANUP_RETURN(0);
	    }
	    /* Version */
	    else if(!g_strcasecmp(arg, "--version") ||
		    !g_strcasecmp(arg, "-version")
	    )
	    {
		g_print(
		    "%s",
"Endeavour Mark II MIME Types List Version " PROG_VERSION "\n"
PROG_COPYRIGHT
		);
		CLEANUP_RETURN(0);
	    }
	    /* Switch */
	    else if(*arg == '-')
	    {
		while(*arg == '-')
		    arg++;

		while(*arg != '\0')
		{
		    switch(*arg)
		    {
		      case 'u':		/* UIDs */
			print_flags |= LSIDS_PRINT_UIDS;
			break;
		      case 'g':		/* GIDs */
			print_flags |= LSIDS_PRINT_GIDS;
			break;
		      case 'l':		/* Long List */
			print_flags |= LSIDS_PRINT_LONG_LIST;
			break;
		      case 'm':		/* Group Members */
			print_flags |= LSIDS_PRINT_GROUP_MEMBERS;
			break;
		      default:
                        g_printerr(
"-%c: Unsupported argument.\n",
                            *arg
                        );
                        CLEANUP_RETURN(2);
		    }
		    arg++;
		}
	    }
	    /* All else assume UID or GID */
	    else if((*arg != '-') && (*arg != '+'))
	    {
		names_list = g_list_append(names_list, STRDUP(arg));
	    }
	}

	/* Default to print UIDs if none were specified */
	if(!(print_flags & LSIDS_PRINT_UIDS) &&
	   !(print_flags & LSIDS_PRINT_GIDS)
	)
	    print_flags |= LSIDS_PRINT_UIDS;

	/* Print UIDs? */
	if(print_flags & LSIDS_PRINT_UIDS)
	{
	    list_uids(
		ctx,
		EDVUIDsList(ctx),
		names_list,
		print_flags
	    );
	}

	/* Print GIDs? */
	if(print_flags & LSIDS_PRINT_GIDS)
	{
	    list_gids(
		ctx,
		EDVGIDsList(ctx),
		names_list,
		print_flags
	    );
	}

        /* Flush any pending Endeavour2 operations */
	EDVContextSync(ctx);

	CLEANUP_RETURN(0);

#undef CLEANUP_RETURN
}
