#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <time.h>
#include <signal.h>
#include <sys/stat.h>
#ifdef HAVE_LIBTAR
/* Support for adding using libtar is not supported since libtar 
 * currently does not support appending objects to a Tape Archive
 */
#endif
#include <gtk/gtk.h>
#include <unistd.h>

#include "../include/string.h"
#include "../include/fio.h"
#include "../include/disk.h"
#include "../include/prochandle.h"

#include "cdialog.h"
#include "progressdialog.h"

#include "cfg.h"
#include "edv_types.h"
#include "edv_archive_obj.h"
#include "edv_archive_add.h"
#include "edv_archive_add_tar.h"
#include "endeavour2.h"
#include "edv_utils.h"
#include "edv_utils_gtk.h"
#include "edv_cfg_list.h"
#include "config.h"


gint EDVArchAddTar(
	edv_core_struct *core, const gchar *arch_path,
	GList *tar_paths_list,
	GList **new_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive, const gint compression,
	const gboolean dereference_links,
	const gboolean is_compress_compressed,
	const gboolean is_gzip_compressed,
	const gboolean is_bzip2_compressed
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)

#define UNLINK(p)	(((p) != NULL) ? (gint)unlink((const char *)(p)) : -1)
#define INTERRUPT(i)	(((i) > 0) ? (gint)kill((int)(i), SIGINT) : -1)

#define ISCR(c)		(((c) == '\n') || ((c) == '\r'))


/*
 *	Add object to a Tape Archive.
 */
gint EDVArchAddTar(
	edv_core_struct *core, const gchar *arch_path,
	GList *tar_paths_list,
	GList **new_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive, const gint compression,
	const gboolean dereference_links,
	const gboolean is_compress_compressed,
	const gboolean is_gzip_compressed,
	const gboolean is_bzip2_compressed
)
{
	const gchar *prog_tar = CFGItemListGetValueS(
	    core->cfg_list, EDV_CFG_PARM_PROG_TAR
	);
	const gchar *prog_compress = CFGItemListGetValueS(
	    core->cfg_list, EDV_CFG_PARM_PROG_COMPRESS
	);
	const gchar *prog_uncompress = CFGItemListGetValueS(
	    core->cfg_list, EDV_CFG_PARM_PROG_UNCOMPRESS
	);
	const gchar *prog_gzip = CFGItemListGetValueS(
	    core->cfg_list, EDV_CFG_PARM_PROG_GZIP
	);
	const gchar *prog_gunzip = CFGItemListGetValueS(
	    core->cfg_list, EDV_CFG_PARM_PROG_GUNZIP
	);
	const gchar *prog_bzip2 = CFGItemListGetValueS(
	    core->cfg_list, EDV_CFG_PARM_PROG_BZIP2
	);
	const gchar *prog_bunzip2 = CFGItemListGetValueS(
	    core->cfg_list, EDV_CFG_PARM_PROG_BUNZIP2
	);
	gint p, status, nobjs;
	gulong total_size;
	gchar *s, *s2;
	gchar	*parent_dir = NULL,
		*pwd = NULL,
		*cmd = NULL,
		*stdout_path = NULL,
		*stderr_path = NULL,
		*arch_uncompressed_path = NULL;
	FILE *fp;

#define CLEANUP_RETURN(_v_)	{		\
 g_free(parent_dir);				\
 g_free(cmd);					\
 g_free(stdout_path);				\
 g_free(stderr_path);				\
 g_free(arch_uncompressed_path);		\
						\
 /* Restore the previous working dir */		\
 if(pwd != NULL) {				\
  EDVSetCWD(pwd);				\
  g_free(pwd);					\
 }						\
						\
 return(_v_);					\
}

	/* Record the previous working directory and set the new
	 * working directory
	 */
	pwd = EDVGetCWD();
	parent_dir = g_dirname(arch_path);
	EDVSetCWD(parent_dir);

	/* Generate the .tar path from the given arch_path which is
	 * gauranteed to have a .tar, .tgz, .tar.gz, or .tar.bz2
	 * postfix
	 */
	arch_uncompressed_path = STRDUP(arch_path);
	s = g_basename(arch_uncompressed_path);
	s2 = (gchar *)strstr((char *)s, ".tgz");
	if(s2 == NULL)
	    s2 = (gchar *)strstr((char *)s, ".tar.gz");
	if(s2 == NULL)
	    s2 = (gchar *)strstr((char *)s, ".tar.bz2");
	if(s2 == NULL)
	    s2 = (gchar *)strstr((char *)s, ".tar.Z");

	/* Set the new extension of the uncompressed path */
	if(s2 != NULL)
	    strcpy((char *)s2, ".tar");


	/* Format the decompress archive command */
	if(is_compress_compressed)
	    cmd = g_strdup_printf(
		"\"%s\" \"%s\"",
		prog_uncompress, arch_path
	    );
	else if(is_gzip_compressed)
	    cmd = g_strdup_printf(
		"\"%s\" \"%s\"",
		prog_gunzip, arch_path
	    );
	else if(is_bzip2_compressed)
	    cmd = g_strdup_printf(
		"\"%s\" -q \"%s\"",
		prog_bunzip2, arch_path
	    );
	else
	    cmd = NULL;

	/* Need to decompress the archive? */
	status = 0;
	p = 0;
	if(cmd != NULL)
	{
	    gulong last_sec = (gulong)time(NULL);

	    /* Execute the decompress archive command */
	    p = (gint)ExecOE((const char *)cmd, "/dev/null", "/dev/null");
	    if(p <= 0)
	    {
		core->archive_last_error =
"Unable to execute the decompress command.";
		CLEANUP_RETURN(-1);
	    }

	    /* Delete the command */
	    g_free(cmd);
	    cmd = NULL;

	    /* Map the progress dialog */
	    if(show_progress)
	    {
		gchar	*p1 = EDVShortenPath(
		    arch_path,
		    EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX - 10
		),	*msg = g_strdup_printf(
"Decompressing Archive:\n\
\n\
    %s\n",
		    p1
		);
		EDVArchAddMapProgressDialogUnknown(msg, toplevel, TRUE);
		g_free(msg);
		g_free(p1);
	    }

	    /* Wait for the decompress archive process to finish */
	    while(ExecProcessExists(p))
	    {
		if(show_progress && ProgressDialogIsQuery())
		{
		    const gulong	cur_sec = (gulong)time(NULL),
					d_sec = cur_sec - last_sec;

		    /* Update the label every second */
		    if(d_sec >= 1l)
		    {
			struct stat stat_buf;
			gchar *p1, *s1, *msg;
			gulong cur_size;

			p1 = EDVShortenPath(
			    arch_path,
			    EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX - 10
			);
			if(stat((const char *)arch_uncompressed_path, &stat_buf))
			    memset(&stat_buf, 0x00, sizeof(struct stat));
			cur_size = (gulong)stat_buf.st_size;
			if(cur_size > 0l)
			{
			    s1 = STRDUP(EDVSizeStrDelim(cur_size));
			    msg = g_strdup_printf(
"Decompressing Archive:\n\
\n\
    %s (%s %s)\n",
			        p1, s1, (cur_size == 1l) ? "byte" : "bytes"
			    );
			}
			else
			{
			    s1 = NULL;
			    msg = NULL;
			}
			g_free(p1);
			g_free(s1);
			ProgressDialogUpdateUnknown(
			    NULL, msg, NULL, NULL, TRUE
			);
			g_free(msg);
			last_sec = cur_sec;
		    }
		    else
		    {
			ProgressDialogUpdateUnknown(
			    NULL, NULL, NULL, NULL, TRUE
			);
		    }
		    if(ProgressDialogStopCount() > 0)
		    {
			INTERRUPT(p);
			p = 0;
			status = -4;
			break;
		    }
		}
		usleep(8000);
	    }
	}
	/* Error or user aborted? */
	if(status != 0)
	{
	    CLEANUP_RETURN(status);
	}


	/* Format the add to archive command and calculate the number
	 * of objects and the total size
	 */
	nobjs = 0;
	total_size = 0l;
	if(cmd == NULL)
	{
	    gchar *s;
	    const gchar *tpath;
	    GList *glist;

	    /* Format the add object(s) to archive command */
	    cmd = g_strdup_printf(
		"\"%s\" -r%s%s -v -f \"%s\"",
		prog_tar,
		recursive ? "" : " --no-recursion",
		dereference_links ? " -h" : "",
		arch_uncompressed_path
	    );
	    for(glist = tar_paths_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		tpath = (const gchar *)glist->data;
		if(STRISEMPTY(tpath))
		    continue;

		/* Count the size and number of objects */
		total_size += EDVArchAddCalculateTotalSize(
		    arch_path, tpath,
		    &nobjs,
		    recursive, dereference_links
		);

		/* Seek tpath to the relative path as needed */
		if(EDVIsParentPath(parent_dir, tpath))
		{
		    tpath += STRLEN(parent_dir);
		    while(*tpath == G_DIR_SEPARATOR)
			tpath++;
		}

		s = g_strconcat(
		    cmd, " \"", tpath, "\"", NULL
		);
		g_free(cmd);
		cmd = s;
	    }
	    if(cmd == NULL)
	    {
		core->archive_last_error = "Unable to generate the add command.";
		CLEANUP_RETURN(-1);
	    }

	    /* Generate the output file paths */
	    stdout_path = EDVTmpName(NULL);
	    stderr_path = EDVTmpName(NULL);

	    /* Execute the add object(s) to archive command */
	    p = (gint)ExecOE(
		(const char *)cmd,
		(const char *)stdout_path,
		(const char *)stderr_path
	    );
	    if(p <= 0)
	    {
		core->archive_last_error = "Unable to execute the add command.";
	        CLEANUP_RETURN(-1);
	    }

	    /* Delete the command */
	    g_free(cmd);
	    cmd = NULL;
	}

	/* Open the output file for reading */
	fp = fopen((const char *)stdout_path, "rb");
	if(fp != NULL)
	{
	    gint buf_pos = 0, nobjs_added = 0;
	    gfloat progress = 0.0f;
	    gchar buf[10000];
	    gboolean need_break = FALSE;

	    /* Begin reading the output file */
	    while(TRUE)
	    {
		/* Update progress? */
		if(show_progress && ProgressDialogIsQuery())
		{
		    ProgressDialogUpdate(
			NULL, NULL, NULL, NULL,
			progress, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
		    );
		    if(ProgressDialogStopCount() > 0)
		    {
			INTERRUPT(p);
			p = 0;
			status = -4;
			break;
		    }
		}

		/* Check if there is new data to be read from the output
		 * file
		 */
		if(FPending(fp))
		{
		    gint c;
		    gboolean got_complete_line = FALSE;

		    /* Copy all available data from the current output
		     * file position to its end to the line buffer buf
		     */
		    while(TRUE)
		    {
			c = (gint)fgetc(fp);
			if((int)c == EOF)
			{
			    clearerr(fp);
			    break;
			}

			if(ISCR(c))
			{
			    got_complete_line = TRUE;
			    if(buf_pos < sizeof(buf))
				buf[buf_pos] = '\0';
			    else
				buf[sizeof(buf) - 1] = '\0';
			    buf_pos = 0;
			    break;
			}

			if(buf_pos < sizeof(buf))
			{
			    buf[buf_pos] = c;
			    buf_pos++;
			}
		    }
		    if(got_complete_line)
		    {
			const gchar *added_path = buf;

			/* Append path to the list of new paths added
			 * to the archive
			 */
			if(new_paths_list_rtn != NULL)
			    *new_paths_list_rtn = g_list_append(
				*new_paths_list_rtn, STRDUP(added_path)
			    );

			/* Update the progress dialog's label? */
			if(show_progress && !STRISEMPTY(added_path))
			{
			    gchar	*p1 = EDVShortenPath(
				added_path, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
			    ),
					*p2 = EDVShortenPath(
				arch_path, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
			    ),
					*msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Agregar:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Addition:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Hinzufgen:\n\
\n\
    %s\n\
\n\
Zu:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Aggiunta:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Toevoegen:\n\
\n\
    %s\n\
\n\
Te:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Adicionar:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Tilfying:\n\
\n\
    %s\n\
\n\
Til:\n\
\n\
    %s\n"
#else
"Adding:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n"
#endif
				, p1, p2
			    );
			    EDVArchAddMapProgressDialog(
				msg, progress, toplevel, FALSE
			    );
			    g_free(msg);
			    g_free(p1);
			    g_free(p2);
			}

			nobjs_added++;
			progress = (nobjs > 0) ?
			    ((gfloat)nobjs_added / (gfloat)nobjs) : 0.0f;

			continue;
		    }
		}

		if(need_break)
		    break;

		/* Check if the add process has exited, if it has then
		 * we set need_break to TRUE. Which will be tested on
		 * the next loop if there is still no more data to be
		 * read
		 */
		if(!ExecProcessExists(p))
		    need_break = TRUE;

		usleep(8000);
	    }

	    fclose(fp);
	}

	/* Remove the output files */
	UNLINK(stdout_path);
	UNLINK(stderr_path);

	/* Report the final progress? */
	if(show_progress && (status == 0) && ProgressDialogIsQuery())
	{
	    ProgressDialogUpdate(
		NULL, NULL, NULL, NULL,
		1.0f, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
	    );
	    if(ProgressDialogStopCount() > 0)
		status = -4;
	}

	/* Error or user aborted? */
	if(status != 0)
	{
	    CLEANUP_RETURN(status);
	}

	/* Format the compress archive command as needed */
	if(is_compress_compressed)
	    cmd = g_strdup_printf(
		"\"%s\" -c \"%s\"",
		prog_compress, arch_uncompressed_path
	    );
	else if(is_gzip_compressed)
	    cmd = g_strdup_printf(
		"\"%s\" -%i -c \"%s\"",
		prog_gzip,
		CLIP(compression * 9 / 100, 1, 9),
		arch_uncompressed_path
	    );
	else if(is_bzip2_compressed)
	    cmd = g_strdup_printf(
		"\"%s\" -%i -c \"%s\"",
		prog_bzip2,
		CLIP(compression * 9 / 100, 1, 9),
		arch_uncompressed_path
	    );
	else
	    cmd = NULL;

	/* Need to compress the archive? */
	if(cmd != NULL)
	{
	    gulong last_sec = (gulong)time(NULL);

	    /* Execute the compress archive command */
	    p = (gint)ExecOE(
		(const char *)cmd,
		(const char *)arch_path,		/* stdout is the archive object */
		"/dev/null"
	    );
	    if(p <= 0)
	    {
		core->archive_last_error = "Unable to execute the compress command.";
		CLEANUP_RETURN(-1);
	    }

	    /* Delete the command */
	    g_free(cmd);
	    cmd = NULL;

	    /* Map the progress dialog? */
	    if(show_progress)
	    {
		gchar	*p1 = EDVShortenPath(
		    arch_path,
		    EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX - 10
		),
			*msg = g_strdup_printf(
"Compressing Archive:\n\
\n\
    %s\n",
		    p1
		);
		EDVArchAddMapProgressDialogUnknown(msg, toplevel, TRUE);
		g_free(msg);
		g_free(p1);
	    }

	    /* Wait for the compress process to finish */
	    while(ExecProcessExists(p))
	    {
		if(show_progress && ProgressDialogIsQuery())
		{
		    const gulong	cur_sec = (gulong)time(NULL),
					d_sec = cur_sec - last_sec;

		    /* Update the label every second */
		    if(d_sec >= 1l)
		    {
			struct stat stat_buf;
			gchar *p1, *s1, *msg;
			gulong cur_size;

			p1 = EDVShortenPath(
			    arch_path,
			    EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX - 10
			);
			if(stat((const char *)arch_path, &stat_buf))
			    memset(&stat_buf, 0x00, sizeof(struct stat));
			cur_size = (gulong)stat_buf.st_size;
			if(cur_size > 0l)
			{
			    s1 = STRDUP(EDVSizeStrDelim(cur_size));
			    msg = g_strdup_printf(
"Compressing Archive:\n\
\n\
    %s (%s %s)\n",
				p1, s1, (cur_size == 1l) ? "byte" : "bytes"
			    );
			}
			else
			{
			    s1 = NULL;
			    msg = NULL;
			}
			g_free(p1);
			g_free(s1);
			ProgressDialogUpdateUnknown(
			    NULL, msg, NULL, NULL, TRUE
			);
			g_free(msg);
			last_sec = cur_sec;
		    }
		    else
		    {
			ProgressDialogUpdateUnknown(
			    NULL, NULL, NULL, NULL, TRUE
			);
		    }
		    if(ProgressDialogStopCount() > 0)
		    {
			INTERRUPT(p);
			p = 0;
			status = -4;
			break;
		    }
		}
		usleep(8000);
	    }

	    /* Remove the uncompressed version of the archive */
	    UNLINK(arch_uncompressed_path);
	}

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}
