#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <time.h>
#include <gtk/gtk.h>
#include <unistd.h>

#include "guiutils.h"
#include "cdialog.h"
#include "progressdialog.h"

#include "cfg.h"
#include "edv_types.h"
#include "edv_date.h"
#include "edv_id.h"
#include "edv_recycled_obj.h"
#include "edv_recbin_index.h"
#include "edv_recbin_stat.h"
#include "endeavour2.h"
#include "edv_recycled_obj_op.h"
#include "edv_utils.h"
#include "edv_utils_gtk.h"
#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_chmod_32x32.xpm"
#include "images/icon_owned_32x32.xpm"
#include "images/icon_time_stamp_32x32.xpm"


/*
 *	Return values legend:
 *
 *	0	Success.
 *	-1	General error.
 *	-2	Invalid value.
 *	-3	Systems error, out of memory, or out of disk space.
 *	-4	User responded with "Cancel".
 *	-5	User responded with "No" or response was not available.
 *	-6	An operation is already in progress.
 */


/* Error Message */
const gchar *EDVRecycledObjectOPGetError(edv_core_struct *core);
static void EDVRecycledObjectOPCopyErrorMessage(
	edv_core_struct *core, const gchar *msg
);

/* Relink */
gint EDVRecycledObjectOPRelink(
	edv_core_struct *core,
	const guint index,
	const gchar *new_target,
	GList **modified_indicies_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);

/* Rename */
gint EDVRecycledObjectOPRename(
	edv_core_struct *core,
	const guint index,
	const gchar *new_name,
	GList **modified_indicies_list,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all
);

/* Change Permissions */
static gint EDVRecycledObjectOPChModIterate(
	edv_core_struct *core,
	const gchar *index_file,
	edv_recycled_object_struct *obj,
	const edv_permission_flags permissions,
	const gchar *permissions_s,
	GList **modified_indicies_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	gint *nobjs_processed,
	const gint nobjs,
	const gulong time_start
);
gint EDVRecycledObjectOPChMod(
	edv_core_struct *core,
	GList *indices_list,
	const edv_permission_flags permissions,
	GList **modified_indicies_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);

/* Change Ownership */
static gint EDVRecycledObjectOPChOwnIterate(
	edv_core_struct *core,
	const gchar *index_file,
	edv_recycled_object_struct *obj,
	const gint owner_id, const gint group_id,
	const gchar *owner_group_s,
	GList **modified_indicies_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	gint *nobjs_processed,
	const gint nobjs,
	const gulong time_start
);
gint EDVRecycledObjectOPChOwn(
	edv_core_struct *core,
	GList *indices_list,
	const gint owner_id, const gint group_id,
	GList **modified_indicies_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);

/* Change Timestamps */
static gint EDVRecycledObjectOPChTimeIterate(
	edv_core_struct *core,
	const gchar *index_file,
	edv_recycled_object_struct *obj,
	const gulong atime, const gulong mtime, const gulong dtime,
	const gchar *time_s,
	GList **modified_indicies_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	gint *nobjs_processed,
	const gint nobjs,
	const gulong time_start
);
gint EDVRecycledObjectOPChTime(
	edv_core_struct *core,
	GList *indices_list,
	const gulong atime, const gulong mtime, const gulong dtime,
	GList **modified_indicies_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Returns the last error message or NULL if there was no error.
 *
 *	The returned pointer must not be deleted.
 */
const gchar *EDVRecycledObjectOPGetError(edv_core_struct *core)
{
	return((core != NULL) ? core->recbin_last_error : NULL);
}

/*
 *	Coppies the error message specified by msg to the last error
 *	message buffer and sets last_error to point to that buffer.
 */
static void EDVRecycledObjectOPCopyErrorMessage(
	edv_core_struct *core, const gchar *msg
)
{
	if(core == NULL)
	    return;

	core->recbin_last_error = NULL;

	g_free(core->recbin_last_error_buf);
	core->recbin_last_error_buf = STRDUP(msg);

	core->recbin_last_error = core->recbin_last_error_buf;
}


/*
 *	Relink.
 *
 *	The index specifies the recycled object to relink. The recycled
 *	object must be of type EDV_OBJECT_TYPE_LINK.
 *
 *	The new_target specifies the new target value of the recycled
 *	object (without the path).
 *
 *	If modified_indicies_list is not NULL then a dynamically
 *	allocated GList of guint indicies describing the modified
 *	recycled objects will be returned.
 *
 *	The toplevel specifies the reference toplevel GtkWidget.
 *
 *	If show_progress is TRUE then the progress dialog will be
 *	mapped during this operation.
 *
 *	If interactive is TRUE then the user will be queried about
 *	any problems that occure during this operation.
 *
 *	The yes_to_all specifies the pointer to the yes to all
 *	variable to be used during this operation. This variable must
 *	be initialized prior to this call.
 *
 *	Returns 0 on success and non-zero on error.
 */
gint EDVRecycledObjectOPRelink(
	edv_core_struct *core,
	const guint index,
	const gchar *new_target,
	GList **modified_indicies_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
	const gulong time_start = (gulong)time(NULL);
	gint status;
	const gchar *index_file;
	cfg_item_struct *cfg_list;
	edv_recycled_object_struct *obj = NULL;

	if(modified_indicies_list != NULL)
	    *modified_indicies_list = NULL;

	/* Clear the last error message */
	EDVRecycledObjectOPCopyErrorMessage(core, NULL);

	if((core == NULL) || (index == 0) || STRISEMPTY(new_target) ||
 	   (yes_to_all == NULL)
	)
	{
	    EDVRecycledObjectOPCopyErrorMessage(core, "Invalid value.");
	    return(-2);
	}

	/* Is an operation already in progress? */
	if(core->op_level > 0)
	{
	    core->recbin_last_error =
"An operation is already in progress, please try again later.";
	    return(-6);
	}

	core->op_level++;

	cfg_list = core->cfg_list;

#define CLEANUP_RETURN(_v_)	{	\
 EDVRecycledObjectDelete(obj);		\
					\
 return(_v_);				\
}

	/* Get the recycled objects index file */
	index_file = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);

	/* Get the current statistics for the recycled object */
	obj = EDVRecBinObjectStat(index_file, index);
	if(obj == NULL)
	{
	    gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    #%u",
		EDVRecBinIndexGetError(),
		index
	    );
	    EDVRecycledObjectOPCopyErrorMessage(core, msg);
	    g_free(msg);
	    core->op_level--;
	    CLEANUP_RETURN(-1);
	}

	/* Object must be of type link */
	if(obj->type != EDV_OBJECT_TYPE_LINK)
	{
	    gchar *msg = g_strdup_printf(
"Not a link:\n\
\n\
    %s",
		obj->name
	    );
	    EDVRecycledObjectOPCopyErrorMessage(core, msg);
	    g_free(msg);
	    core->op_level--;
	    CLEANUP_RETURN(-1);
	}

	/* Set the new target */
	g_free(obj->link_target);
	obj->link_target = STRDUP(new_target);

	/* Update the recycled object's changed time */
	obj->change_time = time_start;

	/* Commit the changes to the recycled objects index file */
	status = EDVRecBinIndexSet(
	    index_file,
	    index,
	    obj
	);
	if(status > 0)
	{
	    /* Add this index to the modified indicies list */
	    if(modified_indicies_list != NULL)
		*modified_indicies_list = g_list_append(
		    *modified_indicies_list,
		    (gpointer)index
		);

	    status = 0;
	}
	else
	{
	    /* Unable to commit the changes */
	    gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %u",
		EDVRecBinIndexGetError(),
		index
	    );
	    EDVRecycledObjectOPCopyErrorMessage(core, msg);
	    g_free(msg);
	}

	/* Record history */
	EDVAppendHistory(
	    core,
	    EDV_HISTORY_RECYCLED_OBJECT_LINK,
	    time_start,
	    (gulong)time(NULL),
	    status,
	    obj->name,				/* Source */
	    new_target,				/* Target */
	    core->recbin_last_error		/* Comment */
	);

	core->op_level--;

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}


/*
 *	Rename.
 *
 *	The index specifies the recycled object to rename.
 *
 *	The new_name specifies the new name of the recycled object
 *	(without the path).
 *
 *	If modified_indicies_list is not NULL then a dynamically
 *	allocated GList of guint indicies describing the modified
 *	recycled objects will be returned.
 *
 *	The toplevel specifies the reference toplevel GtkWidget.
 *
 *	If show_progress is TRUE then the progress dialog will be
 *	mapped during this operation.
 *
 *	If interactive is TRUE then the user will be queried about
 *	any problems that occure during this operation.
 *
 *	The yes_to_all specifies the pointer to the yes to all
 *	variable to be used during this operation. This variable must
 *	be initialized prior to this call.
 *
 *	Returns 0 on success and non-zero on error.
 */
gint EDVRecycledObjectOPRename(
	edv_core_struct *core,
	const guint index,
	const gchar *new_name,
	GList **modified_indicies_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
	const gulong time_start = (gulong)time(NULL);
	gint status;
	gchar *old_name = NULL;
	const gchar *index_file;
	cfg_item_struct *cfg_list;
	edv_recycled_object_struct *obj = NULL;

	if(modified_indicies_list != NULL)
	    *modified_indicies_list = NULL;

	/* Clear the last error message */
	EDVRecycledObjectOPCopyErrorMessage(core, NULL);

	if((core == NULL) || (index == 0) || STRISEMPTY(new_name) ||
 	   (yes_to_all == NULL)
	)
	{
	    EDVRecycledObjectOPCopyErrorMessage(core, "Invalid value.");
	    return(-2);
	}

	/* Is an operation already in progress? */
	if(core->op_level > 0)
	{
	    core->recbin_last_error =
"An operation is already in progress, please try again later.";
	    return(-6);
	}

	core->op_level++;

	cfg_list = core->cfg_list;

#define CLEANUP_RETURN(_v_)	{	\
 EDVRecycledObjectDelete(obj);		\
 g_free(old_name);			\
					\
 return(_v_);				\
}

	/* Check if the new name contains any invalid characters */
	if(!EDVIsObjectNameValid(new_name))
	{
	    gchar *msg = g_strdup_printf(
"Invalid name:\n\
\n\
    %s\n\
\n\
This name contains one or more invalid characters.",
		new_name
	    );
	    EDVRecycledObjectOPCopyErrorMessage(core, msg);
	    g_free(msg);
	    core->op_level--;
	    CLEANUP_RETURN(-1);
	}

	/* Get the recycled objects index file */
	index_file = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);

	/* Get the current statistics for the recycled object */
	obj = EDVRecBinObjectStat(index_file, index);
	if(obj == NULL)
	{
	    gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    #%u",
		EDVRecBinIndexGetError(),
		index
	    );
	    EDVRecycledObjectOPCopyErrorMessage(core, msg);
	    g_free(msg);
	    core->op_level--;
	    CLEANUP_RETURN(-1);
	}

	/* Record the old name */
	old_name = STRDUP(obj->name);

	/* Set the new name */
	g_free(obj->name);
	obj->name = STRDUP(new_name);

	/* Update the recycled object's changed time */
	obj->change_time = time_start;

	/* Commit the changes to the recycled objects index file */
	status = EDVRecBinIndexSet(
	    index_file,
	    index,
	    obj
	);
	if(status > 0)
	{
	    /* Add this index to the modified indicies list */
	    if(modified_indicies_list != NULL)
		*modified_indicies_list = g_list_append(
		    *modified_indicies_list,
		    (gpointer)index
		);

	    status = 0;
	}
	else
	{
	    /* Unable to commit the changes */
	    gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %u",
		EDVRecBinIndexGetError(),
		index
	    );
	    EDVRecycledObjectOPCopyErrorMessage(core, msg);
	    g_free(msg);
	}

	/* Record history */
	EDVAppendHistory(
	    core,
	    EDV_HISTORY_RECYCLED_OBJECT_RENAME,
	    time_start,
	    (gulong)time(NULL),
	    status,
	    old_name,				/* Source */
	    new_name,				/* Target */
	    core->recbin_last_error		/* Comment */
	);

	core->op_level--;

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}


/*
 *	Change permissions iteration.
 */
static gint EDVRecycledObjectOPChModIterate(
	edv_core_struct *core,
	const gchar *index_file,
	edv_recycled_object_struct *obj,
	const edv_permission_flags permissions,
	const gchar *permissions_s,
	GList **modified_indicies_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	gint *nobjs_processed,
	const gint nobjs,
	const gulong time_start
)
{
	gint status;

#define CLEANUP_RETURN(_v_)	{	\
 return(_v_);				\
}

	/* Update the progress dialog message? */
	if(show_progress)
	{
	    const gfloat progress = (nobjs > 0) ?
		((gfloat)(*nobjs_processed) / (gfloat)nobjs) : 0.0f;
	    gchar       *p1 = EDVShortenPath(
		obj->name,
		EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
	    ),          *msg = g_strdup_printf(
"Changing permissions of:\n\
\n\
    %s\n",
		p1
	    );
	    g_free(p1);
	    if(ProgressDialogIsQuery())
	    {
		ProgressDialogUpdate(
		    NULL, msg, NULL, NULL,
		    progress, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
		);
	    }
	    else
	    {
		ProgressDialogSetTransientFor(toplevel);
		ProgressDialogMap(
		    "Changing Permissions",
		    msg,
		    (const guint8 **)icon_chmod_32x32_xpm,
		    "Stop"
		);
		ProgressDialogUpdate(
		    NULL, NULL, NULL, NULL,
		    progress, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
		);
		gdk_flush();
	    }
	    g_free(msg);

	    if(ProgressDialogStopCount() > 0)
		CLEANUP_RETURN(-4);
	}

	/* Set the new permissions */
	obj->permissions = permissions;

	/* Update the recycled object's changed time */
	obj->change_time = time_start;

	/* Commit the changes to the recycled objects index file */
	status = EDVRecBinIndexSet(
	    index_file,
	    obj->index,
	    obj
	);
	if(status > 0)
	{
	    /* Add this index to the modified indicies list */
	    if(modified_indicies_list != NULL)
		*modified_indicies_list = g_list_append(
		    *modified_indicies_list,
		    (gpointer)obj->index
		);

	    status = 0;
	}
	else
	{
	    /* Unable to commit the changes */
	    gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %u",
		EDVRecBinIndexGetError(),
		obj->index
	    );
	    EDVRecycledObjectOPCopyErrorMessage(core, msg);
	    g_free(msg);
	}

	/* Count this object as processed */
	*nobjs_processed = (*nobjs_processed) + 1;

	if(show_progress && ProgressDialogIsQuery())
	{
	    const gfloat progress = (nobjs > 0) ?
		((gfloat)(*nobjs_processed) / (gfloat)nobjs) : 0.0f;
	    ProgressDialogUpdate(
		NULL, NULL, NULL, NULL,
		progress, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
	    );
	    if(ProgressDialogStopCount() > 0)
		CLEANUP_RETURN(-4);
	}

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}

/*
 *	Change permissions.
 *
 *	The indicies_list specifies a GList of guint indicies
 *	describing the recycled objects to change the permissions of.
 *
 *	The permissions specifies the new permissions which can be
 *	any of EDV_PERMISSION_*.
 *
 *	If modified_indicies_list is not NULL then a dynamically
 *	allocated GList of guint indicies describing the modified
 *	recycled objects will be returned.
 *
 *	The toplevel specifies the reference toplevel GtkWidget.
 *
 *	If show_progress is TRUE then the progress dialog will be
 *	mapped during this operation.
 *
 *	If interactive is TRUE then the user will be queried about
 *	any problems that occure during this operation.
 *
 *	The yes_to_all specifies the pointer to the yes to all
 *	variable to be used during this operation. This variable must
 *	be initialized prior to this call.
 *
 *	Returns 0 on success and non-zero on error.
 */
gint EDVRecycledObjectOPChMod(
	edv_core_struct *core,
	GList *indices_list,
	const edv_permission_flags permissions,
	GList **modified_indicies_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
	gint status, nobjs_processed, nobjs;
	guint index;
	gulong time_start;
	gchar *permissions_s = NULL;
	gchar *index_s;
	const gchar *index_file;
	GList *glist;
	cfg_item_struct *cfg_list;
	edv_recycled_object_struct *obj;

	if(modified_indicies_list != NULL)
	    *modified_indicies_list = NULL;

	/* Clear the last error message */
	EDVRecycledObjectOPCopyErrorMessage(core, NULL);

	if((core == NULL) || (indices_list == NULL) || (yes_to_all == NULL))
	{
	    EDVRecycledObjectOPCopyErrorMessage(core, "Invalid value.");
	    return(-2);
	}

	/* Is an operation already in progress? */
	if(core->op_level > 0)
	{
	    core->recbin_last_error =
"An operation is already in progress, please try again later.";
	    return(-6);
	}

	core->op_level++;

	cfg_list = core->cfg_list;

#define CLEANUP_RETURN(_v_)	{	\
 g_free(permissions_s);		\
					\
 return(_v_);				\
}

	/* Get the recycled objects index file */
	index_file = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);

	/* Format the permissions string */
	permissions_s = EDVGetPermissionsString(permissions);

	/* Iterate through the indicies list */
	status = 0;
	nobjs_processed = 0;
	nobjs = g_list_length(indices_list);
	for(glist = indices_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    time_start = (gulong)time(NULL);

	    index = (guint)glist->data;
	    if(index == 0)
		continue;

	    /* Format the index string to describe this recycled object */
	    index_s = g_strdup_printf("#%u", index);

	    /* Get the current statistics for this recycled object */
	    obj = EDVRecBinObjectStat(index_file, index);
	    if(obj != NULL)
	    {
		/* Change this recycled object's permissions */
		status = EDVRecycledObjectOPChModIterate(
		    core,
		    index_file,
		    obj,
		    permissions,
		    permissions_s,
		    modified_indicies_list,
		    toplevel,
		    show_progress,
		    interactive,
		    yes_to_all,
		    &nobjs_processed,
		    nobjs,
		    time_start
		);
	    }
	    else
	    {
		/* Unable to obtain the recycled object's statistics */
		gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    #%u",
		    EDVRecBinIndexGetError(),
		    index
		);
		EDVRecycledObjectOPCopyErrorMessage(core, msg);
		g_free(msg);
		status = -1;
	    }

	    /* Record history */
	    EDVAppendHistory(
		core,
		EDV_HISTORY_RECYCLED_OBJECT_CHMOD,
		time_start,
		(gulong)time(NULL),
		status,
		(obj != NULL) ? obj->name : index_s,	/* Source */
		permissions_s,			/* Target */
		core->recbin_last_error		/* Comment */
	    );

	    /* Delete the recycled object's statistics */
	    EDVRecycledObjectDelete(obj);

	    g_free(index_s);

	    if(status != 0)
		break;
	}

	core->op_level--;

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}


/*
 *	Change ownership iteration.
 */
static gint EDVRecycledObjectOPChOwnIterate(
	edv_core_struct *core,
	const gchar *index_file,
	edv_recycled_object_struct *obj,
	const gint owner_id, const gint group_id,
	const gchar *owner_group_s,
	GList **modified_indicies_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	gint *nobjs_processed,
	const gint nobjs,
	const gulong time_start
)
{
	gint status;

#define CLEANUP_RETURN(_v_)	{	\
 return(_v_);				\
}

	/* Update the progress dialog message? */
	if(show_progress)
	{
	    const gfloat progress = (nobjs > 0) ?
		((gfloat)(*nobjs_processed) / (gfloat)nobjs) : 0.0f;
	    gchar       *p1 = EDVShortenPath(
		obj->name,
		EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
	    ),          *msg = g_strdup_printf(
"Changing ownership of:\n\
\n\
    %s\n",
		p1
	    );
	    g_free(p1);
	    if(ProgressDialogIsQuery())
	    {
		ProgressDialogUpdate(
		    NULL, msg, NULL, NULL,
		    progress, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
		);
	    }
	    else
	    {
		ProgressDialogSetTransientFor(toplevel);
		ProgressDialogMap(
		    "Changing Ownership",
		    msg,
		    (const guint8 **)icon_owned_32x32_xpm,
		    "Stop"
		);
		ProgressDialogUpdate(
		    NULL, NULL, NULL, NULL,
		    progress, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
		);
		gdk_flush();
	    }
	    g_free(msg);

	    if(ProgressDialogStopCount() > 0)
		CLEANUP_RETURN(-4);
	}

	/* Set the new ownership */
	obj->owner_id = owner_id;
	obj->group_id = group_id;

	/* Update the recycled object's changed time */
	obj->change_time = time_start;

	/* Commit the changes to the recycled objects index file */
	status = EDVRecBinIndexSet(
	    index_file,
	    obj->index,
	    obj
	);
	if(status > 0)
	{
	    /* Add this index to the modified indicies list */
	    if(modified_indicies_list != NULL)
		*modified_indicies_list = g_list_append(
		    *modified_indicies_list,
		    (gpointer)obj->index
		);

	    status = 0;
	}
	else
	{
	    /* Unable to commit the changes */
	    gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %u",
		EDVRecBinIndexGetError(),
		obj->index
	    );
	    EDVRecycledObjectOPCopyErrorMessage(core, msg);
	    g_free(msg);
	}

	/* Count this object as processed */
	*nobjs_processed = (*nobjs_processed) + 1;

	if(show_progress && ProgressDialogIsQuery())
	{
	    const gfloat progress = (nobjs > 0) ?
		((gfloat)(*nobjs_processed) / (gfloat)nobjs) : 0.0f;
	    ProgressDialogUpdate(
		NULL, NULL, NULL, NULL,
		progress, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
	    );
	    if(ProgressDialogStopCount() > 0)
		CLEANUP_RETURN(-4);
	}

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}

/*
 *	Change ownership.
 *
 *	The indicies_list specifies a GList of guint indicies
 *	describing the recycled objects to change the ownership of.
 *
 *	The owner_id specifies the ID of the new owner.
 *
 *	The group_id specifies the ID of the new group.
 *
 *	If modified_indicies_list is not NULL then a dynamically
 *	allocated GList of guint indicies describing the modified
 *	recycled objects will be returned.
 *
 *	The toplevel specifies the reference toplevel GtkWidget.
 *
 *	If show_progress is TRUE then the progress dialog will be
 *	mapped during this operation.
 *
 *	If interactive is TRUE then the user will be queried about
 *	any problems that occure during this operation.
 *
 *	The yes_to_all specifies the pointer to the yes to all
 *	variable to be used during this operation. This variable must
 *	be initialized prior to this call.
 *
 *	Returns 0 on success and non-zero on error.
 */
gint EDVRecycledObjectOPChOwn(
	edv_core_struct *core,
	GList *indices_list,
	const gint owner_id, const gint group_id,
	GList **modified_indicies_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
	gint status, nobjs_processed, nobjs;
	guint index;
	gulong time_start;
	gchar *owner_s, *group_s, *owner_group_s = NULL;
	gchar *index_s;
	const gchar *index_file;
	GList *glist;
	cfg_item_struct *cfg_list;
	edv_recycled_object_struct *obj;

	if(modified_indicies_list != NULL)
	    *modified_indicies_list = NULL;

	/* Clear the last error message */
	EDVRecycledObjectOPCopyErrorMessage(core, NULL);

	if((core == NULL) || (indices_list == NULL) || (yes_to_all == NULL))
	{
	    EDVRecycledObjectOPCopyErrorMessage(core, "Invalid value.");
	    return(-2);
	}

	/* Is an operation already in progress? */
	if(core->op_level > 0)
	{
	    core->recbin_last_error =
"An operation is already in progress, please try again later.";
	    return(-6);
	}

	core->op_level++;

	cfg_list = core->cfg_list;

#define CLEANUP_RETURN(_v_)	{	\
 g_free(owner_group_s);			\
					\
 return(_v_);				\
}

	/* Get the recycled objects index file */
	index_file = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);

	/* Format the owner/group string */
	owner_s = EDVUIDGetNameFromUID(
	    core->uids_list,
	    owner_id, NULL
	);
	group_s = EDVGIDGetNameFromGID(
	    core->gids_list,
	    group_id, NULL
	);
	owner_group_s = g_strconcat(
	    owner_s, ".", group_s, NULL
	);
	g_free(owner_s);
	g_free(group_s);

	/* Iterate through the indicies list */
	status = 0;
	nobjs_processed = 0;
	nobjs = g_list_length(indices_list);
	for(glist = indices_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    time_start = (gulong)time(NULL);

	    index = (guint)glist->data;
	    if(index == 0)
		continue;

	    /* Format the index string to describe this recycled object */
	    index_s = g_strdup_printf("#%u", index);

	    /* Get the current statistics for this recycled object */
	    obj = EDVRecBinObjectStat(index_file, index);
	    if(obj != NULL)
	    {
		/* Change this recycled object's ownership */
		status = EDVRecycledObjectOPChOwnIterate(
		    core,
		    index_file,
		    obj,
		    owner_id, group_id,
		    owner_group_s,
		    modified_indicies_list,
		    toplevel,
		    show_progress,
		    interactive,
		    yes_to_all,
		    &nobjs_processed,
		    nobjs,
		    time_start
		);
	    }
	    else
	    {
		/* Unable to obtain the recycled object's statistics */
		gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    #%u",
		    EDVRecBinIndexGetError(),
		    index
		);
		EDVRecycledObjectOPCopyErrorMessage(core, msg);
		g_free(msg);
		status = -1;
	    }

	    /* Record history */
	    EDVAppendHistory(
		core,
		EDV_HISTORY_RECYCLED_OBJECT_CHOWN,
		time_start,
		(gulong)time(NULL),
		status,
		(obj != NULL) ? obj->name : index_s,	/* Source */
		owner_group_s,			/* Target */
		core->recbin_last_error		/* Comment */
	    );

	    /* Delete the recycled object's statistics */
	    EDVRecycledObjectDelete(obj);

	    g_free(index_s);

	    if(status != 0)
		break;
	}

	core->op_level--;

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}


/*
 *	Change time stamps iteration.
 */
static gint EDVRecycledObjectOPChTimeIterate(
	edv_core_struct *core,
	const gchar *index_file,
	edv_recycled_object_struct *obj,
	const gulong atime, const gulong mtime, const gulong dtime,
	const gchar *time_s,
	GList **modified_indicies_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	gint *nobjs_processed,
	const gint nobjs,
	const gulong time_start
)
{
	gint status;

#define CLEANUP_RETURN(_v_)	{	\
 return(_v_);				\
}

	/* Update the progress dialog message? */
	if(show_progress)
	{
	    const gfloat progress = (nobjs > 0) ?
		((gfloat)(*nobjs_processed) / (gfloat)nobjs) : 0.0f;
	    gchar       *p1 = EDVShortenPath(
		obj->name,
		EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
	    ),          *msg = g_strdup_printf(
"Changing time stamps of:\n\
\n\
    %s\n",
		p1
	    );
	    g_free(p1);
	    if(ProgressDialogIsQuery())
	    {
		ProgressDialogUpdate(
		    NULL, msg, NULL, NULL,
		    progress, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
		);
	    }
	    else
	    {
		ProgressDialogSetTransientFor(toplevel);
		ProgressDialogMap(
		    "Changing Time Stamps",
		    msg,
		    (const guint8 **)icon_time_stamp_32x32_xpm,
		    "Stop"
		);
		ProgressDialogUpdate(
		    NULL, NULL, NULL, NULL,
		    progress, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
		);
		gdk_flush();
	    }
	    g_free(msg);

	    if(ProgressDialogStopCount() > 0)
		CLEANUP_RETURN(-4);
	}

	/* Set the new time stamps */
	obj->access_time = atime;
	obj->modify_time = mtime;
	obj->deleted_time = dtime;

	/* Update the recycled object's changed time */
	obj->change_time = time_start;

	/* Commit the changes to the recycled objects index file */
	status = EDVRecBinIndexSet(
	    index_file,
	    obj->index,
	    obj
	);
	if(status > 0)
	{
	    /* Add this index to the modified indicies list */
	    if(modified_indicies_list != NULL)
		*modified_indicies_list = g_list_append(
		    *modified_indicies_list,
		    (gpointer)obj->index
		);

	    status = 0;
	}
	else
	{
	    /* Unable to commit the changes */
	    gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %u",
		EDVRecBinIndexGetError(),
		obj->index
	    );
	    EDVRecycledObjectOPCopyErrorMessage(core, msg);
	    g_free(msg);
	}

	/* Count this object as processed */
	*nobjs_processed = (*nobjs_processed) + 1;

	if(show_progress && ProgressDialogIsQuery())
	{
	    const gfloat progress = (nobjs > 0) ?
		((gfloat)(*nobjs_processed) / (gfloat)nobjs) : 0.0f;
	    ProgressDialogUpdate(
		NULL, NULL, NULL, NULL,
		progress, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
	    );
	    if(ProgressDialogStopCount() > 0)
		CLEANUP_RETURN(-4);
	}

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}

/*
 *	Change time stamps.
 *
 *	The indicies_list specifies a GList of guint indicies
 *	describing the recycled objects to change the time stamps
 *	of.
 *
 *	The atime specifies the new access time in seconds since
 *	EPOCH.
 *
 *	The mtime specifies the new modify time in seconds since
 *	EPOCH.
 *
 *	The dtime specifies the new delete time in seconds since
 *	EPOCH.
 *
 *	If modified_indicies_list is not NULL then a dynamically
 *	allocated GList of guint indicies describing the modified
 *	recycled objects will be returned.
 *
 *	The toplevel specifies the reference toplevel GtkWidget.
 *
 *	If show_progress is TRUE then the progress dialog will be
 *	mapped during this operation.
 *
 *	If interactive is TRUE then the user will be queried about
 *	any problems that occure during this operation.
 *
 *	The yes_to_all specifies the pointer to the yes to all
 *	variable to be used during this operation. This variable must
 *	be initialized prior to this call.
 *
 *	Returns 0 on success and non-zero on error.
 */
gint EDVRecycledObjectOPChTime(
	edv_core_struct *core,
	GList *indices_list,
	const gulong atime, const gulong mtime, const gulong dtime,
	GList **modified_indicies_list,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
	gint status, nobjs_processed, nobjs;
	guint index;
	gulong time_start;
	gchar *time_s = NULL;
	gchar *index_s;
	const gchar *format, *index_file;
	GList *glist;
	cfg_item_struct *cfg_list;
	edv_date_relativity relativity;
	edv_recycled_object_struct *obj;

	if(modified_indicies_list != NULL)
	    *modified_indicies_list = NULL;

	/* Clear the last error message */
	EDVRecycledObjectOPCopyErrorMessage(core, NULL);

	if((core == NULL) || (indices_list == NULL) || (yes_to_all == NULL))
	{
	    EDVRecycledObjectOPCopyErrorMessage(core, "Invalid value.");
	    return(-2);
	}

	/* Is an operation already in progress? */
	if(core->op_level > 0)
	{
	    core->recbin_last_error =
"An operation is already in progress, please try again later.";
	    return(-6);
	}

	core->op_level++;

	cfg_list = core->cfg_list;
	relativity = (edv_date_relativity)EDV_GET_I(
	    EDV_CFG_PARM_DATE_RELATIVITY
	);
	format = EDV_GET_S(EDV_CFG_PARM_DATE_FORMAT);

#define CLEANUP_RETURN(_v_)	{	\
 g_free(time_s);			\
					\
 return(_v_);				\
}

	/* Get the recycled objects index file */
	index_file = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);

	/* Format the time string */
	time_s = STRDUP(EDVDateFormatString(mtime, format, relativity));

	/* Iterate through the indicies list */
	status = 0;
	nobjs_processed = 0;
	nobjs = g_list_length(indices_list);
	for(glist = indices_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    time_start = (gulong)time(NULL);

	    index = (guint)glist->data;
	    if(index == 0)
		continue;

	    /* Format the index string to describe this recycled object */
	    index_s = g_strdup_printf("#%u", index);

	    /* Get the current statistics for this recycled object */
	    obj = EDVRecBinObjectStat(index_file, index);
	    if(obj != NULL)
	    {
		/* Change this recycled object's time stamps */
		status = EDVRecycledObjectOPChTimeIterate(
		    core,
		    index_file,
		    obj,
		    atime, mtime, dtime,
		    time_s,
		    modified_indicies_list,
		    toplevel,
		    show_progress,
		    interactive,
		    yes_to_all,
		    &nobjs_processed,
		    nobjs,
		    time_start
		);
	    }
	    else
	    {
		/* Unable to obtain the recycled object's statistics */
		gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    #%u",
		    EDVRecBinIndexGetError(),
		    index
		);
		EDVRecycledObjectOPCopyErrorMessage(core, msg);
		g_free(msg);
		status = -1;
	    }

	    /* Record history */
	    EDVAppendHistory(
		core,
		EDV_HISTORY_RECYCLED_OBJECT_CHTIME,
		time_start,
		(gulong)time(NULL),
		status,
		(obj != NULL) ? obj->name : index_s,	/* Source */
		time_s,				/* Target */
		core->recbin_last_error		/* Comment */
	    );

	    /* Delete the recycled object's statistics */
	    EDVRecycledObjectDelete(obj);

	    g_free(index_s);

	    if(status != 0)
		break;
	}

	core->op_level--;

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}
