#include <gtk/gtk.h>

#include "guiutils.h"

#include "edv_status_bar.h"
#include "cfg.h"
#include "edv_types.h"
#include "endeavour2.h"
#include "edv_cb.h"
#include "edv_utils_gtk.h"

#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_secure_16x16.xpm"
#include "images/icon_insecure_16x16.xpm"
#include "images/icon_trash_16x16.xpm"
#include "images/icon_purge_16x16.xpm"


static gint EDVStatusBarButtonEventCB(
	GtkWidget *widget, GdkEventButton *button, gpointer data
);
static gint EDVStatusBarCrossingEventCB(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);

edv_status_bar_struct *EDVStatusBarNew(
	edv_core_struct *core, GtkWidget *parent
);
void EDVStatusBarUpdate(edv_status_bar_struct *sb);
void EDVStatusBarSetBusy(
	edv_status_bar_struct *sb, const gboolean busy  
);
void EDVStatusBarMap(edv_status_bar_struct *sb);
void EDVStatusBarUnmap(edv_status_bar_struct *sb);
void EDVStatusBarDelete(edv_status_bar_struct *sb);

void EDVStatusBarMessage(
	edv_status_bar_struct *sb, const gchar *msg,
	const gboolean allow_gtk_iteration
);
void EDVStatusBarProgress(
	edv_status_bar_struct *sb, const gfloat v,
	const gboolean allow_gtk_iteration
);


#define STATUS_BAR_HEIGHT		26
#define STATUS_BAR_PROGRESS_BAR_WIDTH	100
#define STATUS_BAR_PROGRESS_BAR_HEIGHT	(STATUS_BAR_HEIGHT - 6)


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Status Bar "button_press_event" or "button_release_event"
 *	signal callback.
 */
static gint EDVStatusBarButtonEventCB(
	GtkWidget *widget, GdkEventButton *button, gpointer data
)
{
	gint status = FALSE;
	cfg_item_struct *cfg_list;
	edv_core_struct *core;
	edv_status_bar_struct *sb = EDV_STATUS_BAR(data);
	if((widget == NULL) || (sb == NULL))
	    return(status);

	core = sb->core;
	cfg_list = core->cfg_list;

	EDVStatusBarSetBusy(sb, TRUE);

	/* Check which widget this event is for
	 *
	 * Write protect enabled
	 */
	if(widget == sb->write_protect_enabled_eb)
	{
	    /* Disable master write protect */
	    EDV_SET_B(
		EDV_CFG_PARM_WRITE_PROTECT,
		FALSE
	    );
	    EDVWriteProtectChangedEmit(core, FALSE);
	    status = TRUE;
	}
	/* Write protect disabled */
	else if(widget == sb->write_protect_disabled_eb)
	{
	    /* Enable master write protect */
	    EDV_SET_B(
		EDV_CFG_PARM_WRITE_PROTECT,
		TRUE
	    );
	    EDVWriteProtectChangedEmit(core, TRUE);
	    status = TRUE;
	}
	/* Delete method recycle */
	else if(widget == sb->delete_method_recycle_eb)
	{
	    /* Set the delete method to purge on delete */
	    EDV_SET_I(
		EDV_CFG_PARM_DELETE_METHOD,
		EDV_DELETE_METHOD_PURGE
	    );
	    EDVReconfiguredEmit(core);
	    status = TRUE;
	}
	/* Delete method purge */
	else if(widget == sb->delete_method_purge_eb)
	{
	    /* Set the delete method to recycle */
	    EDV_SET_I(
		EDV_CFG_PARM_DELETE_METHOD,
		EDV_DELETE_METHOD_RECYCLE
	    );
	    EDVReconfiguredEmit(core);
	    status = TRUE;
	}

	EDVStatusBarSetBusy(sb, FALSE);

	return(status);
}

/*
 *	Status Bar "enter_notify_event" or "leave_notify_event"
 *	signal callback.
 */
static gint EDVStatusBarCrossingEventCB(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	gint status = FALSE;
	gint etype;
	edv_status_bar_struct *sb = EDV_STATUS_BAR(data);
	if((widget == NULL) || (crossing == NULL) || (sb == NULL))
	    return(status);

	etype = (gint)crossing->type;

	/* Check which widget this event is for
	 *
	 * Write protect enabled
	 */
	if(widget == sb->write_protect_enabled_eb)
	{
	    const gchar *msg = NULL;

	    switch(etype)
	    {
	      case GDK_ENTER_NOTIFY:
		msg =
#if defined(PROG_LANGUAGE_SPANISH)
"Proteccion de escritura habilitada, haga clic para anularla"
#elif defined(PROG_LANGUAGE_FRENCH)
"Protection criture En Service, cliquer pour annuler"
#elif defined(PROG_LANGUAGE_GERMAN)
"Schreibgeschtzt ist, klicken ermglicht, unwirksam es zu machen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scrivere proteggere  permesso, lo scatto esserlo incapace"
#elif defined(PROG_LANGUAGE_DUTCH)
"Schrijf beschermt, is gegeven de gelegenheid, klik onbruikbaar het te maken"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Escreva protege  capacitado, estalido incapacita-lo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Skrivebeskyttet muliggjrer, klikk udyktiggjre det"
#else
"Write protect is enabled, click to disable it"
#endif
		;
		break;
	      case GDK_LEAVE_NOTIFY:
		msg = NULL;
		break;
	    }
	    EDVStatusBarMessage(sb, msg, FALSE);
	    status = TRUE;
	}
	/* Write protect disabled */
	else if(widget == sb->write_protect_disabled_eb)
	{
	    const gchar *msg = NULL;

	    switch(etype)
	    {
	      case GDK_ENTER_NOTIFY:
		msg =
#if defined(PROG_LANGUAGE_SPANISH)
"Proteccion de escritura deshabilitada, haga clic para habilitarla"
#elif defined(PROG_LANGUAGE_FRENCH)
"Protection criture est Hors Service, cliquer pour la mettre En Service"
#elif defined(PROG_LANGUAGE_GERMAN)
"Schreibgeschtzt ist unwirksam, klicken gemacht, es zu ermglichen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scrivere proteggere  stato incapace, lo scatto permetterlo"
#elif defined(PROG_LANGUAGE_DUTCH)
"Schrijf beschermt, is gemaakt onbruikbaar, klik aan de gelegenheid het te geven"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Escreva protege  incapacitado, estalido capacita-lo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Skrivebeskyttet udyktiggjr, klikk muliggjre det"
#else
"Write protect is disabled, click to enable it"
#endif
		;
		break;

	      case GDK_LEAVE_NOTIFY:
		msg = "";
		break;
	    }
	    EDVStatusBarMessage(sb, msg, FALSE);
	    status = TRUE;
	}
	/* Delete method recycle */
	else if(widget == sb->delete_method_recycle_eb)
	{
	    const gchar *msg = NULL;

	    switch(etype)
	    {
	      case GDK_ENTER_NOTIFY:
		msg =
#if defined(PROG_LANGUAGE_SPANISH)
"Mtodo de borrado establecido a reciclar, hacer clic para ponerlo a purgar"
#elif defined(PROG_LANGUAGE_FRENCH)
"La mthode pour Supprimer est positionne sur mettre  la Corbeille, cliquer pour changer en Dtruire"
#elif defined(PROG_LANGUAGE_GERMAN)
"Lschen Sie Methode ist gesetzt wiederzuverwerten, zu klicken, es\
 zu setzen, zu reinigen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cancellare il metodo  regolato per riciclare, lo scatto regolarlo\
 per prosciogliere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Schrap methode is gezet te recyclen, klik het te zetten te\
 reinigen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Anule mtodo  posto reciclar, clicar por purgar"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stryk metode setter resirkulere, klikk sette det rense"
#else
"Delete method is set to recycle, click to set it to purge"
#endif
		;
		break;

	      case GDK_LEAVE_NOTIFY: 
		msg = "";
		break;
	    }
	    EDVStatusBarMessage(sb, msg, FALSE);
	    status = TRUE;
	}
	/* Delete method purge */
	else if(widget == sb->delete_method_purge_eb)
	{
	    const gchar *msg = NULL;

	    switch(etype)
	    {
	      case GDK_ENTER_NOTIFY:
		msg =
#if defined(PROG_LANGUAGE_SPANISH)
"Mtodo de borrado establecido a purgar, hacer clic para ponerlo a reciclar"
#elif defined(PROG_LANGUAGE_FRENCH)
"La mthode pour Supprimer est positionne sur dtruire, cliquer pour rgler  mettre dans la Corbeille"
#elif defined(PROG_LANGUAGE_GERMAN)
"Lschen Sie Methode ist gesetzt zu reinigen, zu klicken, es zu\
 setzen, wiederzuverwerten"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cancellare il metodo  regolato per prosciogliere, lo scatto\
 regolarlo per riciclare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Schrap methode is gezet te reinigen, klik het te zetten te recyclen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Anule mtodo  posto purgar, clicar por reciclar"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stryk metode setter rense, klikk sette det resirkulere"
#else
"Delete method is set to purge, click to set it to recycle"
#endif
		;
		break;

	      case GDK_LEAVE_NOTIFY:
		msg = "";
		break;
	    }
	    EDVStatusBarMessage(sb, msg, FALSE);
	    status = TRUE;
	}

	return(status);
}


/*
 *	Creates a new Status Bar.
 */
edv_status_bar_struct *EDVStatusBarNew(
	edv_core_struct *core, GtkWidget *parent
)
{
	const gint border_minor = 2;
	GtkStyle *style;
	GtkWidget *w, *parent2, *parent3, *toplevel;
	GtkAdjustment *adj;
	edv_status_bar_struct *sb;

	if(core == NULL)
	    return(NULL);

	sb = EDV_STATUS_BAR(g_malloc0(sizeof(edv_status_bar_struct)));
	if(sb == NULL)
	    return(NULL);

	sb->toplevel = toplevel = gtk_frame_new(NULL);
	sb->freeze_count = 0;
	sb->busy_count = 0;
	sb->core = core;

	sb->freeze_count++;

	/* Toplevel GtkFrame */
	w = toplevel;
	gtk_widget_set_usize(w, -1, STATUS_BAR_HEIGHT);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_OUT);
	gtk_container_border_width(GTK_CONTAINER(w), 0);
	if(parent != NULL)
	{
	    if(GTK_IS_BOX(parent))
		gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    else if(GTK_IS_CONTAINER(parent))
		gtk_container_add(GTK_CONTAINER(parent), w);
	}
	style = gtk_widget_get_style(w);
	parent2 = w;

	/* Main GtkTable */
	w = gtk_table_new(1, 4, FALSE);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);
	parent2 = w;

	/* Write protect enabled GtkEventBox */
	sb->write_protect_enabled_eb = w = gtk_event_box_new();
	gtk_widget_add_events(
	    w,
	    GDK_BUTTON_PRESS_MASK | GDK_ENTER_NOTIFY_MASK |
	    GDK_LEAVE_NOTIFY_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(EDVStatusBarButtonEventCB), sb
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "enter_notify_event",
	    GTK_SIGNAL_FUNC(EDVStatusBarCrossingEventCB), sb
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "leave_notify_event",
	    GTK_SIGNAL_FUNC(EDVStatusBarCrossingEventCB), sb
	);
	gtk_table_attach(
	    GTK_TABLE(parent2), w,
	    0, 1, 0, 1,
	    0,
	    0,
	    border_minor, 0
	);
	gtk_table_set_col_spacing(GTK_TABLE(parent2), 0, 0);
	GUISetWidgetTip(
	    w,
#if defined(PROG_LANGUAGE_SPANISH)
"Proteccion de escritura habilitada"
#elif defined(PROG_LANGUAGE_FRENCH)
"Protection criture est En-Service"
#elif defined(PROG_LANGUAGE_GERMAN)
"Schreibgeschtzt ist ermglicht"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scrivere proteggere  permesso"
#elif defined(PROG_LANGUAGE_DUTCH)
"Schrijf beschermt, is gegeven de gelegenheid"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Escreva protege  capacitado"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Skrivebeskyttet muliggjrer"
#else
"Write protect is enabled"
#endif
	);
	parent3 = w;
	/* Icon */
	w = EDVNewPixmapWidget(
	    core,
	    (guint8 **)icon_secure_16x16_xpm,
	    "icon_secure_16x16_xpm"
	);
	gtk_widget_set_usize(w, 16, 16);
	gtk_container_add(GTK_CONTAINER(parent3), w);
	gtk_widget_show(w);

	/* Write protect disabled GtkEventBox */
	sb->write_protect_disabled_eb = w = gtk_event_box_new();
	gtk_widget_add_events(
	    w,
	    GDK_BUTTON_PRESS_MASK | GDK_ENTER_NOTIFY_MASK |
	    GDK_LEAVE_NOTIFY_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(EDVStatusBarButtonEventCB), sb
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "enter_notify_event",
	    GTK_SIGNAL_FUNC(EDVStatusBarCrossingEventCB), sb
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "leave_notify_event",
	    GTK_SIGNAL_FUNC(EDVStatusBarCrossingEventCB), sb
	);
	gtk_table_attach(
	    GTK_TABLE(parent2), w,
	    0, 1, 0, 1,
	    0,
	    0,
	    border_minor, 0
	);
	GUISetWidgetTip(
	    w,
#if defined(PROG_LANGUAGE_SPANISH)
"Proteccion de escritura deshabilitada"
#elif defined(PROG_LANGUAGE_FRENCH)
"Protection criture est Hors Service"
#elif defined(PROG_LANGUAGE_GERMAN)
"Schreibgeschtzt ist unwirksam gemacht"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scrivere proteggere  stato incapace"
#elif defined(PROG_LANGUAGE_DUTCH)
"Schrijf beschermt, is gemaakt onbruikbaar"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Escreva protege  incapacitado"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Skrivebeskyttet udyktiggjr"
#else
"Write protect is disabled"
#endif
	);
	gtk_widget_show(w);
	parent3 = w;
	/* Icon */
	w = EDVNewPixmapWidget(
	    core,
	    (guint8 **)icon_insecure_16x16_xpm,
	    "icon_insecure_16x16_xpm"
	);
	gtk_widget_set_usize(w, 16, 16);
	gtk_container_add(GTK_CONTAINER(parent3), w);
	gtk_widget_show(w);


	/* Delete method recycle GtkEventBox */
	sb->delete_method_recycle_eb = w = gtk_event_box_new();
	gtk_widget_add_events(
	    w,
	    GDK_BUTTON_PRESS_MASK | GDK_ENTER_NOTIFY_MASK |
	    GDK_LEAVE_NOTIFY_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(EDVStatusBarButtonEventCB), sb
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "enter_notify_event",
	    GTK_SIGNAL_FUNC(EDVStatusBarCrossingEventCB), sb
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "leave_notify_event",
	    GTK_SIGNAL_FUNC(EDVStatusBarCrossingEventCB), sb
	);
	gtk_table_attach(
	    GTK_TABLE(parent2), w,
	    1, 2, 0, 1,
	    0,
	    0,
	    border_minor, 0
	);
	gtk_table_set_col_spacing(GTK_TABLE(parent2), 1, 1);
	GUISetWidgetTip(
	    w,
#if defined(PROG_LANGUAGE_SPANISH)
"Mtodo de borrado seleccionado a reciclar"
#elif defined(PROG_LANGUAGE_FRENCH)
"La mthode pour Supprimer est positionne sur mettre  la Corbeille"
#elif defined(PROG_LANGUAGE_GERMAN)
"Lschen sie methode ist gesetzt wiederzuverwerten"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cancellare il metodo  regolato per riciclare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Schrap methode is gezet te recyclen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Anule mtodo  posto reciclar"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stryk metode setter resirkulere"
#else
"Delete method is set to recycle"
#endif
	);
	gtk_widget_show(w);
	parent3 = w;
	/* Icon */
	w = EDVNewPixmapWidget(
	    core,
	    (guint8 **)icon_trash_16x16_xpm,
	    "icon_trash_16x16_xpm"
	);
	gtk_widget_set_usize(w, 16, 16);
	gtk_container_add(GTK_CONTAINER(parent3), w);
	gtk_widget_show(w);

	/* Delete method purge GtkEventBox */
	sb->delete_method_purge_eb = w = gtk_event_box_new();
	gtk_widget_add_events(
	    w,
	    GDK_BUTTON_PRESS_MASK | GDK_ENTER_NOTIFY_MASK |
	    GDK_LEAVE_NOTIFY_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(EDVStatusBarButtonEventCB), sb
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "enter_notify_event",
	    GTK_SIGNAL_FUNC(EDVStatusBarCrossingEventCB), sb
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "leave_notify_event",
	    GTK_SIGNAL_FUNC(EDVStatusBarCrossingEventCB), sb
	);
	gtk_table_attach(
	    GTK_TABLE(parent2), w,
	    1, 2, 0, 1,
	    0,
	    0,
	    border_minor, 0 
	);
	GUISetWidgetTip(
	    w,
#if defined(PROG_LANGUAGE_SPANISH)
"Mtodo de borrado seleccionado a purgar"
#elif defined(PROG_LANGUAGE_FRENCH)
"La mthode pour Supprimer est positionne sur dtruire"
#elif defined(PROG_LANGUAGE_GERMAN)
"Lschen sie methode ist gesetzt zu reinigen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cancellare il metodo  regolato per prosciogliere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Schrap methode is gezet te reinigen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Anule mtodo  posto purgar"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stryk metode setter rense"
#else
"Delete method is set to purge"
#endif
	);
	parent3 = w;
	/* Icon */
	w = EDVNewPixmapWidget(
	    core,
	    (guint8 **)icon_purge_16x16_xpm,
	    "icon_purge_16x16_xpm"
	);
	gtk_widget_set_usize(w, 16, 16);
	gtk_container_add(GTK_CONTAINER(parent3), w);
	gtk_widget_show(w);


	/* GtkProgressBar
	 *
	 * The GtkAdjustment range is from 1.0 to 100.0 (in units of
	 * percent), 0.0 is used as "no value" (so that no progress
	 * is shown)
	 */
	adj = (GtkAdjustment *)gtk_adjustment_new(
	    0.0f, 1.0f, 100.0f,
	    0.0f, 0.0f, 0.0f
	);
	sb->progress_bar = w = gtk_progress_bar_new_with_adjustment(adj);
	gtk_widget_set_usize(
	    w,
	    STATUS_BAR_PROGRESS_BAR_WIDTH,
	    STATUS_BAR_PROGRESS_BAR_HEIGHT
	);
	gtk_progress_bar_set_orientation(
	    GTK_PROGRESS_BAR(w), GTK_PROGRESS_LEFT_TO_RIGHT
	);
	gtk_progress_bar_set_bar_style(
	    GTK_PROGRESS_BAR(w), GTK_PROGRESS_CONTINUOUS
	);
	gtk_progress_set_activity_mode(
	    GTK_PROGRESS(w), FALSE
	);
	gtk_table_attach(
	    GTK_TABLE(parent2), w,
	    2, 3, 0, 1,
	    0,
	    0,
	    0, 1
	);
	gtk_table_set_col_spacing(GTK_TABLE(parent2), 2, 0);
	gtk_widget_show(w);

	/* Label GtkFrame */
	sb->label_frame = w = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
	gtk_table_attach(
	    GTK_TABLE(parent2), w,
	    3, 4, 0, 1,
	    GTK_SHRINK | GTK_EXPAND | GTK_FILL,
	    GTK_SHRINK | GTK_EXPAND | GTK_FILL,
	    1,		/* Use X padding as 1 so that both sides get
			 * padded since this is the last column and
			 * the right edge needs padding that can only
			 * be set this way */
	    1
	);
	gtk_widget_show(w);
	parent3 = w;

	w = gtk_hbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent3), w);
	gtk_widget_show(w);
	parent3 = w;

	/* GtkLabel */
	sb->label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, border_minor);
	gtk_widget_show(w);

	sb->freeze_count--;

	return(sb);
}

/*
 *	Updates the Status Bar widgets to reflect current values.
 */
void EDVStatusBarUpdate(edv_status_bar_struct *sb)
{
	gboolean write_protect;
	edv_delete_method delete_method;
	GtkWidget *w;
	const cfg_item_struct *cfg_list;
	edv_core_struct *core;

	if(sb == NULL)
	    return;

	sb->freeze_count++;

	core = sb->core;
	cfg_list = core->cfg_list;

	/* Get the configuration values */
	write_protect = CFGItemListGetValueI(
	    cfg_list, EDV_CFG_PARM_WRITE_PROTECT
	);
	delete_method = CFGItemListGetValueI(
	    cfg_list, EDV_CFG_PARM_DELETE_METHOD
	);

	/* Write protect */
	w = sb->write_protect_enabled_eb;
	if(w != NULL)
	{
	    if(write_protect)
		gtk_widget_show(w);
	    else
		gtk_widget_hide(w);
	}
	w = sb->write_protect_disabled_eb;
	if(w != NULL)
	{
	    if(write_protect)
		gtk_widget_hide(w);
	    else
		gtk_widget_show(w);
	}

	/* Delete method */
	w = sb->delete_method_recycle_eb;
	if(w != NULL)
	{
	    if(delete_method == EDV_DELETE_METHOD_RECYCLE)
		gtk_widget_show(w);
	    else   
		gtk_widget_hide(w);
	}
	w = sb->delete_method_purge_eb;
	if(w != NULL)
	{
	    if(delete_method == EDV_DELETE_METHOD_PURGE)
		gtk_widget_show(w);
	    else   
		gtk_widget_hide(w);
	}

	sb->freeze_count--;
}

/*
 *	Sets the Status Bar as busy or ready.
 */
void EDVStatusBarSetBusy(
	edv_status_bar_struct *sb, const gboolean busy  
)
{
	GdkCursor *cur;
	GtkWidget *w;
	edv_core_struct *core;

	if(sb == NULL)
	    return;

	w = gtk_widget_get_toplevel(sb->toplevel);
	if(w == NULL)
	    return;

	core = sb->core;

	if(busy)
	{
	    /* Increase busy count */
	    sb->busy_count++;

	    /* If already busy then don't change anything */
	    if(sb->busy_count > 1)
		return;

	    cur = EDVGetCursor(core, EDV_CURSOR_CODE_BUSY);
	}
	else
	{
	    /* Reduce busy count */
	    sb->busy_count--;
	    if(sb->busy_count < 0)
		sb->busy_count = 0;

	    /* If still busy do not change anything */
	    if(sb->busy_count > 0)
		return;

	    cur = NULL;		/* Use the default cursor */
	}

	/* Update the toplevel GdkWindow's GdkCursor */
	if(w->window != NULL)
	{
	    gdk_window_set_cursor(w->window, cur);
	    gdk_flush();
	}
}

/*
 *	Maps the Status Bar.
 */
void EDVStatusBarMap(edv_status_bar_struct *sb)
{
	if(sb == NULL)
	    return;

	gtk_widget_show(sb->toplevel);
}

/*
 *	Unmaps the Status Bar.
 */
void EDVStatusBarUnmap(edv_status_bar_struct *sb)
{
	if(sb == NULL)
	    return;

	gtk_widget_hide(sb->toplevel);

	/* Reset the last progress position */
	sb->progress_pos_last = -1.0f;
}

/*
 *	Deletes the Status Bar.
 */
void EDVStatusBarDelete(edv_status_bar_struct *sb)
{
	if(sb == NULL)
	    return;

	sb->freeze_count++;

	gtk_widget_destroy(sb->toplevel);

	sb->freeze_count--;

	g_free(sb);
}


/*
 *	Sets the Status Bar's message.
 */
void EDVStatusBarMessage(
	edv_status_bar_struct *sb, const gchar *msg,
	const gboolean allow_gtk_iteration
)
{
	if(sb == NULL)
	    return;

	if(!GTK_WIDGET_MAPPED(sb->toplevel))
	    return;

	gtk_label_set_text(
	    GTK_LABEL(sb->label),
	    (msg != NULL) ? msg : ""
	);

	/* Flush events? */
	while((gtk_events_pending() > 0) && allow_gtk_iteration)
	    gtk_main_iteration();
}

/*
 *	Sets the Status Bar's progress.
 *
 *	The v specifies the progress value from 0.0 to 1.0. If v is
 *	-1.0 then unknown progress is displayed.
 */
void EDVStatusBarProgress(
	edv_status_bar_struct *sb, const gfloat v,
	const gboolean allow_gtk_iteration
)
{
	gfloat value = v;
	GtkProgress *progress;
	GtkProgressBar *progress_bar;

	if(sb == NULL)
	    return;

	if(!GTK_WIDGET_MAPPED(sb->toplevel))
	    return;

	progress_bar = GTK_PROGRESS_BAR(sb->progress_bar);
	progress = GTK_PROGRESS(progress_bar);

	/* Do activity? */
	if(value < 0.0f)
	{
	    GtkAdjustment *adj = progress->adjustment;

	    /* Get new value based on unknown position */
	    value = gtk_progress_get_value(progress) + 1.0f;
	    if(value > adj->upper)
		value = adj->lower;

	    /* Update progress bar for `activity mode' (unknown limit)
	     * and set new value
	     */
	    gtk_progress_set_activity_mode(progress, TRUE);
	    gtk_progress_set_show_text(progress, FALSE);
	    gtk_progress_set_value(progress, value);

	    /* Reset the last progress position to -1.0, indicating
	     * that there is no known last position since we are
	     * doing activity mode
	     */
	    sb->progress_pos_last = -1.0f;
	}
	else
	{

	    /* Clip value to 0.0, 1.0 */
	    if(value > 1.0f)
		value = 1.0f;

	    /* Reset the last progress position if it is greater than
	     * the specified value, implying that the progress has
	     * warped back to the beginning
	     */
	    if(sb->progress_pos_last > value)
		sb->progress_pos_last = -1.0f;

	    /* Has the value not sufficiently changed? (less than
	     * 0.01 change in value)
	     */
	    if((sb->progress_pos_last > 0.0f) &&
	       ((value - sb->progress_pos_last) < 0.01f)
	    )
		return;

	    /* Update progress */
	    gtk_progress_set_activity_mode(progress, FALSE);
	    gtk_progress_set_format_string(
		progress, "%p%%"
	    );
	    /* Display text only if value is positive */
	    gtk_progress_set_show_text(
		progress, (value > 0.0f) ? TRUE : FALSE
	    );
	    gtk_progress_bar_update(progress_bar, value);

	    /* Record last position */
	    sb->progress_pos_last = value;
	}

	/* Flush events? */
	while((gtk_events_pending() > 0) && allow_gtk_iteration)
	    gtk_main_iteration();
}
