/*
			    Endeavour Utilities

	These functions do not depend on Endeavour or GTK+, see
	edv_utils_gtk.h for Endeavour and GTK+ dependent functions.
 */

#ifndef EDV_UTILS_H
#define EDV_UTILS_H

#include <sys/types.h>
#include <sys/stat.h>
#include <glib.h>
#include "edv_types.h"


/*
 *	Substitutes a format string containing tokens for values.
 *
 *	The fmt specifies the format string.
 *
 *	The token specifies the token string.
 *
 *	The val specifies the value string to replace each token
 *	string encountered in the format string.
 *
 *	Returns a dynamically allocated string describing the
 *	substituted format string or NULL on error.
 */
extern gchar *EDVStrSub(
	const gchar *fmt,
	const gchar *token,
	const gchar *val
);

/*
 *	Evaluates all occurances of "%HH" by replacing them with
 *	a single character described by HH in hexidecimal value.
 *
 *	The s specifies the string to replace.
 */
extern void EDVStrEvaluateTokenHexToChar(gchar *s);


/*
 *	Converts the stat() mode_t type to edv_object_type type.
 *
 *	The m specifies the stat() mode_t type value. Only the
 *	bit pertaining to the type in m is checked, all other bits
 *	are ignored.
 *
 *	Returns the edv_object_type type value or
 *	EDV_OBJECT_TYPE_UNKNOWN on failed match.
 */
extern edv_object_type EDVStatModeToObjectType(const mode_t m);

/*
 *	Converts the edv_object_type type to stat() mode_t type.
 *
 *	The type specifies the edv_object_type which must be one of
 *	EDV_OBJECT_TYPE_*.
 *
 *	Returns the mode_t type value or 0 on failed match.
 */
extern mode_t EDVObjectTypeToStatType(const edv_object_type type);

/*
 *	Converts the edv_object_type type to a string name.
 *
 *	The type specifies the edv_object_type which must be one of
 *	EDV_OBJECT_TYPE_*.
 *
 *	Returns a statically allocated string describing the type
 *	name.
 */
extern const gchar *EDVGetObjectTypeName(const edv_object_type type);

/*
 *	Converts the edv_object_type type to a string name in
 *	lowercase.
 *
 *	The type specifies the edv_object_type which must be one of
 *	EDV_OBJECT_TYPE_*.
 *
 *	Returns a statically allocated string describing the type
 *	name in lowercase.
 */
extern const gchar *EDVGetObjectTypeNameLower(const edv_object_type type);


/*
 *	Creates a string describing the value with ',' deliminators.
 *
 *	The i specifies the value.
 *
 *	Returns a statically allocated string describing the
 *	value with ',' deliminators or NULL on error.
 */
extern const gchar *EDVSizeStrDelim(const gulong i);
extern const gchar *EDVSizeStrDelimChar(
	const gulong i,
	const gchar delim_char
);

/*
 *      Formats a string describing the size with the specified
 *      format.
 *
 *      The size specifies the size in bytes.
 *
 *      The size_format specifies the format.
 *
 *      The block_size specifies the size of each block in bytes.
 *
 *      The delim_char specifies the character to use as the
 *      deliminator.
 *
 *      If allow_unit_conversion is TRUE then the returned string
 *      will describe the size in units other than bytes if size_format
 *      specifies a size format other than bytes. Otherwise FALSE
 *      forces the returned string to describe the size in bytes.
 *
 *      Returns a statically allocated string.
 */
extern const gchar *EDVSizeStrFormat(
	const gulong size,
	const edv_size_format size_format,
	const gulong block_size,
	const gchar delim_char,
	const gboolean allow_unit_conversion
);


/*
 *	Converts the stat() mode_t permissions to edv_permission_flags
 *	permissions.
 *
 *	The m specifies the stat() mode_t permissions.
 *
 *	Returns the edv_permission_flags permissions.
 */
extern edv_permission_flags EDVStatModeToEDVPermissions(const mode_t m);

/*
 *	Converts the edv_permission_flags permissions to stat() mode_t
 *	permissions.
 *
 *	The permissions specifies the edv_permission_flags, which can
 *	be any of EDV_PERMISSION_*.
 *
 *	Returns the stat() mode_t permissions.
 */
extern mode_t EDVEDVPermissionsToStatMode(const edv_permission_flags permissions);

/*
 *	Converts the edv_permission_flags permissions to a string
 *	description.
 *
 *	The permissions specifies the edv_permission_flags, which can
 *	be any of EDV_PERMISSION_*.
 *
 *	Returns a dynamically allocated string describing the
 *	permissions.
 */
extern gchar *EDVGetPermissionsString(const edv_permission_flags permissions);


/*
 *	Simplifies the path.
 *
 *	Removes any tailing path deliminators and reduces any
 *	occurances of "/.." or "/.".
 *
 *	The path specifies the string describing the path to simplify.
 */
extern void EDVSimplifyPath(gchar *path);

/*
 *	Checks if the parent path is a parent or grand parent of the
 *	child path.
 *
 *	The parent specifies the string describing the parent path.
 *	The parent path must be an absolute path that does not contain
 *	any occurances of "/.." or "/.".
 *
 *	The child specifies the string describing the child path.
 *	The child path must be an absolute path that does not contain
 *	any occurances of "/.." or "/.".
 *
 *	Returns TRUE if the parent path is a parent or grand parent
 *	of the child path.
 */
extern gboolean EDVIsParentPath(
	const gchar *parent,
	const gchar *child
);

/*
 *	Checks if at least one of the specified list of space-separated
 *	extensions matches the extension of the specified name.
 *
 *	Wild cards such as '*' and '?' are accepted in the
 *	space-separated extensions list.
 *
 *	The following example will return TRUE.
 *
 *	name	= "myfile.doc"
 *	ext	= ".txt .doc *rc Makefile*"
 */
extern gboolean EDVIsExtension(
	const gchar *name,
	const gchar *ext
);

/*
 *	Shortens a path string if it exceeds a maximum number of
 *	characters.
 *
 *	The path specifies the string describing the path to shorten
 *	as needed.
 *
 *	The max specifies the maximum number of characters allowed.
 *
 *	Returns a dynamically allocated copy of the path that is no
 *	longer than max characters. A "..." will be prefixed to the
 *	returned string contents if it was longer than max
 *	characters.
 */
extern gchar *EDVShortenPath(
	const gchar *path,
	const gint max
);

/*
 *	Evaluates/coppies a path.
 *
 *	The parent specifies the string describing the parent
 *	path or current location. It is used only if path is not
 *	an absolute path. If parent is NULL or not an absolute path
 *	then the toplevel directory will be used as the parent path
 *	instead.
 *
 *	The path specifies the string describing the path to be
 *	evaluated.
 *
 *	The path will be evaulated as follows:
 *
 *	Checks if the path has a "file://" prefix and, if it does,
 *	removes it from the path.
 *
 *	Checks if the path is "." and, if it does, replaces the
 *	path with the parent path.
 *
 *	Checks if the path is ".." and, if it does, replaces the
 *	path with the parent of the parent path.      
 *
 *	Checks if the path has a "~" prefix and, if it does,
 *	substitutes it with the home directory.
 *
 *	Postfixed to the parent path if path is not an absolute path.
 *
 *	Simplified, all occurances of ".." evaluated.
 *
 *	Tailing directory deliminators removed.
 *
 *	Returns a dynamically allocated string describing the
 *	evaluated path or NULL on error.
 */
extern gchar *EDVEvaluatePath(
	const gchar *parent,
	const gchar *path
);


typedef enum {
	EDV_COMPLETE_PATH_SUCCESS,	/* Completed path fully to a valid
					 * object */
	EDV_COMPLETE_PATH_NONE,		/* General error or not completed */
	EDV_COMPLETE_PATH_AMBIGUOUS,	/* Completed path fully to a valid
					 * object but there are other
					 * objects a with similar name */
	EDV_COMPLETE_PATH_PARTIAL	/* Path partially completed */
} edv_complete_path_status;

/*
 *	Completes the path.
 *
 *	The path specifies the string describing the path to complete,
 *	this string may be modified by this function. The path must
 *	be an absolute path. In the case of an empty path, the path
 *	will be returned describing toplevel.
 *
 *	If status is not NULL then *status will be set to one of
 *	EDV_COMPLETE_PATH_*.
 *
 *	Returns the reallocated path on success or NULL on error.
 */
extern gchar *EDVCompletePath(
	gchar *path,
	edv_complete_path_status *status
);

/*
 *      Generates a relative path from a starting path to a
 *	destination path.
 *
 *	The start_path specifies the starting path. The start_path
 *	must be an absolute path.
 *
 *	The dest_path specifies the destination path. The dest_path
 *	must be an absolute path.
 *
 *	Returns a dynamically allocated string describing the
 *	relative path or NULL on error.
 */
extern gchar *EDVPlotRelativePath(
	const gchar *start_path,
	const gchar *dest_path
);


/*
 *	Returns a dynamically allocated string describing the current
 *	working directory.
 */
extern gchar *EDVGetCWD(void);

/*
 *	Sets the current working directory.
 *
 *	Returns 0 on success or non-zero on error.
 */
extern gint EDVSetCWD(const gchar *path);

/*
 *	Returns the current umask.
 */
extern guint EDVGetUMask(void);

/*
 *	Sets the umask.
 */
extern void EDVSetUMask(const guint m);

/*
 *	Returns a dynamically allocated string describing a unique file
 *	name in the specified dir that is gauranteed to not exist.
 */
extern gchar *EDVTmpName(const gchar *dir);

/*
 *	Returns a dynamically allocated string path describing the
 *	completed absolute path to the program found from the PATH
 *	environment variable for the program specified name or NULL if
 *	there is no match.
 */
extern gchar *EDVWhich(const gchar *name);

/*
 *	Gets the link's target value.
 *
 *	The path specifies the full path to the link.
 *
 *	Returns a dynamically allocated string containing the link's
 *	target value or NULL on error. The global errno will be
 *	set if an error has occured.
 */
extern gchar *EDVGetLinkTarget(const gchar *path);

/*
 *	Same as EDVGetLinkTarget() except that it appends the parent
 *	of the path to the link's target value if the link's 
 *	target value is a relative path.
 */
extern gchar *EDVGetLinkTargetFull(const gchar *path);

/*
 *	Checks if the link specified by path has a target that may
 *	possibly be infinately recursive.
 */
extern gboolean EDVIsLinkInfinatelyRecursive(const gchar *path);

/*
 *	Gets the device major and minor numbers from the device value.
 *
 *	The rdev specifies the device value which should come from
 *	*stat()'s struct stat (dev_t)st_rdev value.
 *
 *	The major_rtn specifies the device major number return value.
 *
 *	The minor_rtn specifies the device minor number return value.
 */
extern void EDVGetDeviceNumbers(
	const gint rdev,
	gint *major_rtn, gint *minor_rtn
);

/*
 *	Combines the device major and minor numbers into a single
 *	device value.
 *
 *	The major specifies the device major number.
 *
 *	The minor specifies the device minor number.
 *
 *	Returns the device value which confirms to *stat()'s struct
 *	stat (dev_t)st_rdev value.
 */
extern gint EDVFormatDeviceNumbers(const gint major, const gint minor);


/*
 *	Checks if the process speecified by pid is still running.
 */
extern gboolean EDVProcessIsRunning(const gint pid);

/*
 *	Executes the command in the background.
 *
 *	The cmd specifies the command. The command must contain
 *	an absolute path to an executable and (optionally) any
 *	subsequent arguments. Shell tokens are accepted, the shell
 *	defaults to /bin/sh.
 *
 *	Returns the process ID or negative on error.
 */
extern gint EDVSystem(const gchar *cmd);

/*
 *	Executes the command in the background using a particular
 *	shell.
 *
 *	The cmd specifies the command. The command must contain
 *	an absolute path to an executable and (optionally) any
 *	subsequent arguments. Shell tokens are accepted.
 *
 *	The shell specifies the full path to the shell. If shell
 *	is NULL then "/bin/sh" will be used.
 *
 *	The shell specifies the full path to the shell. If shell
 *	is NULL then the shell specified by the environment
 *	variable SHELL will be used. If no environment variable is
 *	specified then "/bin/sh" will be used.
 *
 *	Returns the process ID or negative on error.
 */
extern gint EDVSystemShell(
	const gchar *cmd,
	const gchar *shell,
	const gchar *shell_args
);

/*
 *	Executes the command and blocks until it has exited.
 *
 *	The cmd specifies the command. The command must contain
 *	an absolute path to an executable and (optionally) any
 *	subsequent arguments. Shell tokens are accepted, the shell
 *	defaults to /bin/sh.
 *
 *	If status is not NULL then the status of the executed
 *	process will be returned.
 *
 *	Returns the process ID or negative on error.
 */
extern gint EDVSystemBlock(
	const gchar *cmd,
	gint *status
);

/*
 *	Executes the command and blocks until it has exited.
 *
 *	The cmd specifies the command. The command must contain
 *	an absolute path to an executable and (optionally) any
 *	subsequent arguments. Shell tokens are accepted.
 *
 *	The shell specifies the full path to the shell. If shell
 *	is NULL then the shell specified by the environment
 *	variable SHELL will be used. If no environment variable is
 *	specified then "/bin/sh" will be used.
 *
 *	The shell_args specifies the shell's arguments. If shell_args
 *	is NULL then "-c" will be used.
 *
 *	If status is not NULL then the return value of the executed
 *	process will be set to *status.
 *
 *	Returns the process ID or negative on error.
 */
extern gint EDVSystemShellBlock(
	const gchar *cmd,
	const gchar *shell,
	const gchar *shell_args,
	gint *status
);


/*
 *	Opens a GList of gchar * strings from a file.
 *
 *	The path specifies the full path to the file to open from.
 *
 *	If max_lines is positive then no more than max_lines will be
 *	read from the file.
 *
 *	Returns a dynamically allocated GList of gchar * strings.
 */
extern GList *EDVOpenFileGList(
	const gchar *path,
	const gint max_lines
);

/*
 *	Saves the GList of gchar * strings to a file.
 *
 *	The path specifies the full path to the file to save to.
 *
 *	The lines_list specifies a GList of gchar * strings to
 *	write to the file.
 */
extern void EDVSaveFileGList(
	const gchar *path,
	GList *lines_list
);


#endif	/* EDV_UTILS_H */
