/*
 * Copyright (C) 2002,2003 Daniel Heck
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: actors.cc,v 1.41.2.8 2003/10/09 16:32:09 dheck Exp $
 */
#include "enigma.hh"
#include "actors.hh"
#include "player.hh"
#include "sound.hh"
#include "objects.hh"
#include "object_mixins.hh"
#include "world.hh"

#include <cassert>
#include <iostream>

using px::V2;
using namespace world;

//----------------------------------------
// Actor implementation
//----------------------------------------

Actor::Actor(const char *kind, const px::V2 &p)
: Object(kind), actorinfo(p, V2()),
  m_sprite(),
  startingpos(p), respawnpos(), use_respawnpos(false),
  spikes(false)
{
    set_attrib("mouseforce", 0.0);
}

void
Actor::think(double dtime)
{
    GridPos  field(actorinfo.pos);
    Floor *fl = GetFloor(field);
    Item *it = GetItem(field);
    bool item_covers_floor = (it && it->covers_floor());
    if (!item_covers_floor && fl)
        fl->actor_contact(this);
}

void Actor::set_respawnpos(const V2& p) 
{
    respawnpos     = p;
    use_respawnpos = true;
}

void Actor::remove_respawnpos() {
    use_respawnpos = false;
}


void
Actor::respawn()
{
    V2 p =(use_respawnpos) ? respawnpos : startingpos;
    warp (p);
    on_respawn(p);
}

void
Actor::add_force (const px::V2 &f)
{
    actorinfo.forceacc += f;
}

void
Actor::init()
{
    m_sprite = display::AddSprite(actorinfo.pos);
}

void
Actor::set_attrib(const string &key, const Value &val)
{
    Object::set_attrib(key, val);
}

void
Actor::on_creation(const px::V2 &p)
{
    startingpos = p;
    set_model(get_kind());
    m_sprite.move (p);
    move();
}

void
Actor::on_respawn (const px::V2 &/*pos*/)
{
}

void
Actor::warp(const px::V2 &newpos)
{
    actorinfo.pos = newpos;
    actorinfo.vel = V2();
    m_sprite.move (newpos);
    move();
}

void Actor::move()
{
    using namespace world;
    GridPos  field(actorinfo.pos);
    GridPos ofield(actorinfo.oldpos);

    if (field != ofield) {
        // Actor entered a new field -> notify floor and item objects
        if (Floor *fl = GetFloor(field)) 
	    fl->actor_enter(this);
        if (Item *it = GetItem(field)) 
	    it->actor_enter(this);

        if (Floor *ofl = GetFloor(ofield))
            ofl->actor_leave(this);
        if (Item *oit = GetItem(ofield))
            oit->actor_leave(this);
    }

    Item *it = GetItem(field);
    if (it && it->actor_hit(this))
        player::PickupItem(this, field);

    if (Stone *st = GetStone(field))
        st->actor_inside(this);

    m_sprite.move (actorinfo.pos);
    on_motion(actorinfo.pos);
    actorinfo.oldpos = actorinfo.pos;
}

void
Actor::set_model(const string &name)
{
    m_sprite.replace_model (display::MakeModel(name));
}


//----------------------------------------
// Rotor implementation
//----------------------------------------
namespace
{
    class Rotor : public Actor {
        CLONEACTOR(Rotor);
    public:
        Rotor(const char *name, double radius);
    private:
        // Actor interface.
        void think (double dtime);
        bool is_dead() { return false; }
        bool is_flying() { return true; }

	void on_hit(Actor *a) {
	    SendMessage(a, "shatter");
	}
    };
}

Rotor::Rotor(const char *name, double radius)
: Actor(name, V2())
{
    world::ActorInfo *ai = get_actorinfo();
    ai->radius = radius;
    ai->mass = 0.8;

    set_attrib ("range", 5.0);
    set_attrib ("force", 10.0);
}

void Rotor::think (double dtime)
{
    double range = 0, force = 0;
    double_attrib("range", &range);
    double_attrib("force", &force);

    vector<Actor *> actors;
    GetActorsInRange (get_pos(), range, actors);

    for (size_t i=0; i<actors.size(); ++i) {
        Actor *a = actors[i];
        if (a->get_attrib ("whiteball") || a->get_attrib("blackball")) {
            this->add_force (normalize(a->get_pos() - get_pos()) * force);
        }
    }
    Actor::think(dtime);
}


//----------------------------------------
// Bug
//----------------------------------------
namespace
{
    class Bug : public Actor {
        CLONEACTOR(Bug);
    public:
        Bug() : Actor("ac-bug", V2()) {
            world::ActorInfo *ai = get_actorinfo();
            ai->radius = 12/64.0;
            ai->mass = 0.7;
        }
        bool is_flying() { return true; }
        bool is_dead() { return false; }
    };
}


//----------------------------------------
// Horse
//----------------------------------------
namespace
{
    class Horse : public Actor {
        CLONEACTOR(Horse);
    public:
        Horse() : Actor("ac-horse", V2()) {
            world::ActorInfo *ai = get_actorinfo();
            ai->radius = 24/64.0;
            ai->mass = 1.2;
        }
        bool is_flying() { return true; }
        bool is_dead() { return false; }
    };
}


//----------------------------------------
// Killerball
//----------------------------------------
namespace
{
    class Killerball : public Actor {
        CLONEACTOR(Killerball);
    public:

        Killerball() : Actor ("ac-killerball", V2()) {
            world::ActorInfo *ai = get_actorinfo();
            ai->radius = 13/64.0;
            ai->mass = 0.7;
        }
        bool is_dead() { return false; }

	void on_hit(Actor *a) {
	    SendMessage(a, "shatter");
	}
    };
}


//----------------------------------------
// CannonBall
//----------------------------------------
namespace
{
    class CannonBall : public Actor {
        CLONEACTOR(CannonBall);
    public:
        CannonBall() : Actor ("ac-cannonball", V2()) {
        }
        bool is_flying() { return true; }
        bool is_dead() { return false; }
    };
}


//----------------------------------------
// BasicBall
//----------------------------------------
namespace
{
    class BasicBall : public Actor, public display::ModelCallback
    {
    protected:
        BasicBall(const char *kind, double radius, double mass);

        enum State {
            NO_STATE,
            NORMAL,
            SHATTERING,
            SINKING,
            DROWNING,
            BUBBLING,
            FALLING,            // falling into abyss
            JUMPING,
            DEAD,               // marble is dead
            RESURRECTED,        // has been resurrected; about to respawn
            APPEARING,          // appearing when level starts/after respawn
            DISAPPEARING,       // disappearing when level finished
            FALLING_VORTEX,     // falling into vortex
            RISING_VORTEX,      // appear in vortex
            JUMP_VORTEX,        // jump out of vortex (here player controls actor)
        };

        enum HaloState { 
            NOHALO, HALOBLINK, HALONORMAL 
        };


	void disable_shield();
        void update_halo();
        void change_state_noshield (State newstate);


        void set_model_cb(const string &m) {
            set_model(m.c_str());
            get_sprite().set_callback (this);
        }

        void think (double dtime);

        void change_state(State newstate);

        void set_sink_model(const string &m);
        void set_shine_model (bool shinep);


        // ModelCallback interface.
        void animcb();

        /*
        ** Actor interface.
        */
        void on_creation(const px::V2 &p) {
            Actor::on_creation(p);
            change_state(APPEARING);
        }
        void on_respawn (const px::V2 &/*pos*/)
        {
            change_state(APPEARING);
        }

        bool is_dead();
	bool is_movable() { return (state!=DEAD && state!=RESURRECTED); }
        bool is_flying()        { return state == JUMPING; }
        bool is_on_floor();
        bool can_drop_items();
        bool can_pickup_items();
        bool has_shield() const;

        // Object interface.
        void message(const string &m, const Value &);


        /*
         * Variables.
         */
        State state;            // The marble's current state

        static const int minSinkDepth = 0; // normal level
        static const int maxSinkDepth = 7; // at that level the actor dies

        union {
            struct {
                double sinkDepth; // how deep actor has sunk
                int    sinkModel; // current model
            } sink;
            struct {
                double normal_time; // while jumping out of vortex: time at normal level
            } vortex;
        } shared;

        display::SpriteHandle m_halosprite;
        double                m_shield_rest_time;
        static const double   SHIELD_TIME = 10.0;
        HaloState             m_halostate;
    };
}

BasicBall::BasicBall(const char *kind, double radius, double mass)
: Actor(kind, V2()), state(NO_STATE),
  m_shield_rest_time(0),
  m_halostate(NOHALO)

    //, sinkDepth (0), sinkModel(-1)
{
    world::ActorInfo *ai = get_actorinfo();
    ai->radius = radius;
    ai->mass = mass;
}

bool BasicBall::is_dead()
{ 
    return state == DEAD; 
}


bool BasicBall::is_on_floor()
{ 
    return state == NORMAL || state == SINKING || 
        state == JUMP_VORTEX || state==APPEARING; 
}

bool BasicBall::can_drop_items()   
{ 
    return state == NORMAL || state == SINKING || 
        state == JUMP_VORTEX || state==JUMPING; 
}

bool BasicBall::can_pickup_items() 
{ 
    return state == NORMAL || state == SINKING || state == JUMP_VORTEX; 
}

void BasicBall::change_state_noshield (State newstate)
{
    if (!has_shield())
        change_state(newstate);
}

void BasicBall::message(const string &m, const Value &)
{
    switch (state) {
    case NORMAL:
        if (m == "shatter")         change_state_noshield(SHATTERING);
        else if (m == "laserhit")   change_state_noshield(SHATTERING);
        else if (m == "sink")       change_state_noshield(SINKING);
        else if (m == "drown")      change_state_noshield(DROWNING);
        else if (m == "fall")       change_state_noshield(FALLING);
        else if (m == "fallvortex") change_state(FALLING_VORTEX);
        else if (m == "jump")       change_state(JUMPING);
        else if (m == "appear")     change_state(APPEARING);
        else if (m == "disappear")  change_state(DISAPPEARING);
        break;
    case JUMPING:
        if (m == "shatter")         change_state_noshield(SHATTERING);
        else if (m == "disappear")  change_state(DISAPPEARING);
        break;
    case DEAD:
        if (m == "resurrect")       change_state(RESURRECTED);
        break;
    case SINKING:
        if (m == "shatter")         change_state_noshield(SHATTERING);
        else if (m == "getout")     change_state(NORMAL);
        break;
    case FALLING_VORTEX:
        if (m == "rise")            change_state(RISING_VORTEX); // vortex->vortex teleportation
        else if (m == "appear")     change_state(APPEARING); // vortex->non-vortex teleportation
        break;
    case JUMP_VORTEX:
        if (m == "laserhit")        change_state(SHATTERING);
        break;
    case APPEARING:
	if ((m == "shatter" || m == "drown" || m == "fall") 
	    && !has_shield())
	{
	    // Give an "emergency" shield to the actor if it would die
	    // after appearing.
	    m_shield_rest_time = 1.5;
	    update_halo();
	}
	break;
    default:
        break;
    }


    // Shield can be activated in all states except DEAD

    if (state != DEAD && m == "shield") {
        m_shield_rest_time += SHIELD_TIME;
        update_halo();
    }
}

void BasicBall::think(double dtime)
{
    ActorInfo *ai = get_actorinfo();

    if (state == NORMAL) {
        int xpos = static_cast<int>(ai->pos[0] * 32.0);
        int ypos = static_cast<int>(ai->pos[1] * 32.0);

        bool shinep = (xpos + ypos) % 2;
        set_shine_model (shinep);
    }
    else if (state == SINKING) {
        const double defaultSinkSpeed = 4.0; // 10.0 means : sink in 1 second (if absVelocity == 0)
        const double nonSinkVelocity  = 6.0; // at this velocity don't sink; above: raise

        double sinkSpeed = defaultSinkSpeed * (1 - length(ai->vel) / nonSinkVelocity);
        shared.sink.sinkDepth += sinkSpeed*dtime;

        if (shared.sink.sinkDepth >= maxSinkDepth) {
            set_model(string(get_kind())+"-sunk");
            ai->vel = V2();     // stop!
            sound::PlaySound("swamped");
            change_state(BUBBLING);
        }
        else {
            if (shared.sink.sinkDepth < minSinkDepth) shared.sink.sinkDepth = minSinkDepth;
            set_sink_model(get_kind());
        }
    }
    else if (state == JUMP_VORTEX) {
        shared.vortex.normal_time += dtime;
        if (shared.vortex.normal_time > 0.025) // same time as appear animation
            if (shared.vortex.normal_time > dtime) // ensure min. one tick in state JUMP_VORTEX!
                change_state(JUMPING); // end of short control over actor
    }

    // Update protection shield
    if (m_shield_rest_time > 0) {
        m_shield_rest_time -= dtime;
        update_halo();
    }
    Actor::think(dtime);
}

void BasicBall::set_sink_model(const string &m)
{
    int modelnum = static_cast<int>(shared.sink.sinkDepth);

    if (modelnum != shared.sink.sinkModel) {
        assert(modelnum >= minSinkDepth && modelnum < maxSinkDepth);

        string img = m+"-sink";
        img.append(1, static_cast<char>('0'+modelnum));
        set_model(img);

        shared.sink.sinkModel = modelnum;
    }
}

void BasicBall::set_shine_model (bool shinep)
{
    string modelname = get_kind();
    if (shinep)
        set_model (modelname + "-shine");
    else
        set_model (modelname);
}

void BasicBall::animcb()
{
    string kind=get_kind();

    switch (state) {
    case SHATTERING:
        set_model(kind+"-shattered");
        change_state(DEAD);
        break;
    case DROWNING:
    case BUBBLING:
        set_model("invisible");
        change_state(DEAD);
        break;
    case FALLING:
        set_model(kind+"-fallen"); // invisible
        sound::PlaySound("shatter");
        change_state(DEAD);
        break;
    case JUMPING:
        set_model(kind);
        change_state(NORMAL);
        break;
    case APPEARING:
        set_model(kind);
        change_state(NORMAL);
        break;
    case DISAPPEARING:
        set_model("ring-anim");
        break;
    case FALLING_VORTEX: {
        set_model(kind+"-fallen"); // invisible
        Item *it = GetItem(GridPos(get_pos()));
        if (it && it->is_kind("it-vortex")) {
            SendMessage(it, "warp");
        }
        break;
    }
    case RISING_VORTEX: {
        set_model(kind);
        Item *it = GetItem(GridPos(get_pos()));
        if (it && it->is_kind("it-vortex")) {
            SendMessage(it, "arrival"); // closes some vortex
        }
        change_state(JUMP_VORTEX);
        break;
    }
    default:
        break;
    }
}

void
BasicBall::change_state(State newstate)
{
    if (newstate == state)
        return;

    string kind=get_kind();

    State oldstate= state;
    state = newstate;
    switch (newstate) {
    case NORMAL:
        if (oldstate == APPEARING) {
            world::ActorInfo *ai = get_actorinfo();
            ai->forceacc = V2();
        }
        else if (oldstate == SINKING) {
            set_model(kind);
        }
        world::ReleaseActor(this);
        break;
    case SHATTERING:
        sound::PlaySound("shatter");
        world::GrabActor(this);
        set_model_cb(kind+"-shatter");
        break;
    case SINKING:
        if (oldstate != SINKING) {
            world::ReleaseActor(this);
            shared.sink.sinkDepth = minSinkDepth;
            shared.sink.sinkModel = -1;
            set_sink_model(kind);
        }
        break;
    case DROWNING:
        // @@@ FIXME: use same animation as SINKING ?
        world::GrabActor(this);
//         sound::PlaySound("drown");
        sound::PlaySound("dropinwater");
//         set_model_cb("ring-anim");
        set_model_cb("ac-drowned");
        break;
    case BUBBLING:
        world::GrabActor(this);
//         sound::PlaySound("drown");
        set_model_cb("ac-drowned");
        break;
    case FALLING:
    case FALLING_VORTEX:
        world::GrabActor(this);
        set_model_cb(kind+"-fall");
        break;
    case DEAD: break;
    case JUMPING:
        sound::PlaySound("boink");
        set_model_cb(kind+"-jump");
        break;
    case APPEARING:
    case RISING_VORTEX:
        set_model_cb(kind+"-appear");
        world::GrabActor(this);
        break;
    case JUMP_VORTEX:
        assert(oldstate == RISING_VORTEX);
        shared.vortex.normal_time = 0;
        set_model(kind);
        world::ReleaseActor(this);
        break;
    case DISAPPEARING:
        world::GrabActor(this);
	disable_shield();
        set_model_cb(kind+"-disappear");
        break;
    case RESURRECTED:
	disable_shield();
	break;
    default:
        break;
    }
}

void 
BasicBall::disable_shield()
{
    if (has_shield()) {
	m_shield_rest_time = 0;
	update_halo();
    }
}

bool
BasicBall::has_shield() const
{
    return m_shield_rest_time > 0;
}

void
BasicBall::update_halo()
{
    HaloState newstate = m_halostate;

    if (m_shield_rest_time <= 0)
        newstate = NOHALO;
    else if (m_shield_rest_time <= 3.0)
        newstate = HALOBLINK;
    else
        newstate = HALONORMAL;

    if (newstate != m_halostate) {
	if (m_halostate == NOHALO)
	    m_halosprite = display::AddSprite (get_pos(), "halo");
        switch (newstate) {
        case NOHALO:
            // remove halo
            m_halosprite.kill();
            m_halosprite = display::SpriteHandle();
            break;
        case HALOBLINK:
            // blink for the last 3 seconds
            m_halosprite.replace_model (display::MakeModel ("halo-blink"));
            break;
        case HALONORMAL:
	    m_halosprite.replace_model (display::MakeModel ("halo"));
            break;
        }
        m_halostate = newstate;
    }
    else if (m_halostate != NOHALO) {
        m_halosprite.move (get_pos());
    }
}



//----------------------------------------
// Balls of different sorts
//----------------------------------------

namespace
{
    class BlackBall : public BasicBall {
        CLONEACTOR(BlackBall);
    public:
        BlackBall() : BasicBall("ac-blackball", 19.0/64, 1.0)
        {
            set_attrib("mouseforce", Value(1.0));
            set_attrib("color", Value(0.0));
            set_attrib("blackball", Value(true));
            set_attrib("player", Value(0.0));
            set_attrib("controllers", Value(1.0));
        }
    };

    class WhiteBall : public BasicBall {
        CLONEACTOR(WhiteBall);
    public:
        WhiteBall() : BasicBall("ac-whiteball", 19.0/64, 1.0)
        {
            set_attrib("mouseforce", Value(1.0));
            set_attrib("color", Value(1.0));
            set_attrib("whiteball", Value(true));
            set_attrib("player", Value(1.0));
            set_attrib("controllers", Value(2.0));
        }
    };

    class WhiteBall_Small : public BasicBall {
        CLONEACTOR(WhiteBall_Small);
    public:
        WhiteBall_Small() : BasicBall("ac-whiteball-small", 13/64.0, 0.7)
        {
            set_attrib("mouseforce", Value(1.0));
            set_attrib("color", Value(1.0));
            set_attrib("whiteball", Value(true));
            set_attrib("controllers", Value(3.0));
        }

        void on_hit(Actor *a) {
            if (dynamic_cast<BlackBall*>(a) &&
                int_attrib("mouseforce") != 0) 
                // passive small whiteball do not shatter (see PerOxyd Linkgame #60)
            {
                // collision between WhiteBall_Small and blackball shatters blackball
//                 SendMessage(a, "shatter");
            }
        }
    };
}

void
actors::Init()
{
    Register(new Bug);
    Register(new Horse);
    Register(new Rotor("ac-rotor", 22./64));
    Register(new Rotor("ac-top",  16./64));
    Register(new BlackBall);
    Register(new WhiteBall);
    Register(new WhiteBall_Small);
    Register(new Killerball);
}
