/*
 * Copyright (C) 2002,2003 Daniel Heck
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: objects.hh,v 1.49.2.5 2003/10/09 16:32:09 dheck Exp $
 */
#ifndef OBJECTS_HH
#define OBJECTS_HH

#include "world.hh"
#include "display.hh"

#include "px/pxfwd.hh"
#include "px/math.hh"
#include "px/alist.hh"

#include <string>

/*
 * Object is the base class for all ``objects'' in the world.  The
 * most important facilities this class provides are:
 *
 * (1) A way to clone() and dispose() objects.  This is mainly used in
 *     function MakeObject() to create new objects of a given type.
 *
 * (2) A way to pass messages between unrelated objects via message().
 *     This allows us to send messages to objects from Lua and to
 *     decouple objects types as much as possible.
 *
 * (3) A way to get and set attributes.  These attributes are quite
 *     similar to instance variables, but they can be easily modified
 *     from Lua.  This makes it possible to modify certain object
 *     parameters (such as the text on a piece of paper or the color
 *     of an oxyd stone) in level descriptions.
 */

namespace world
{
    using std::string;
    using namespace enigma;


    enum ObjectType {
        OBJTYPE_None,
        OBJTYPE_Floor,
        OBJTYPE_Item,
        OBJTYPE_Stone,
        OBJTYPE_Actor
    };

    enum Signals {
        SIGNAL_None      = 0,
        SIGNAL_Trigger   = 0x01,
        SIGNAL_Open      = 0x02,
        SIGNAL_Close     = 0x04,
        SIGNAL_OpenClose = 0x08,
        SIGNAL_On        = 0x10,
        SIGNAL_Off       = 0x20,
        SIGNAL_OnOff     = 0x40,
    };

    /* (Not much used so far, might become useful for the level
       editor.) */
    struct ObjectTraits {
        ObjectTraits ()
        : type (OBJTYPE_None), signals_out (SIGNAL_None),
          signals_in(SIGNAL_None), min_version (0.0)
        {}

        ObjectTraits (const std::string &name_,
//                       const std::string &short_text_,
//                       const std::string &long_text_,
                      ObjectType type_)
        : name(name_),
//           short_text(short_text_),
//           long_text(long_text_),
          type(type_),
          signals_out (SIGNAL_None),
          signals_in (SIGNAL_None),
          min_version (0.0)
        {}

        string      name;
        string      short_text;
        string      long_text;
        ObjectType  type;
        Signals     signals_out;
        Signals     signals_in;
        float       min_version; // Required Enigma version
    };

    class Object {
    public:
        Object() {}
        Object(const char *kind);
        virtual ~Object() {}

        bool string_attrib (const string &name, string *val) const;
        int  int_attrib (const string &name) const;
        bool int_attrib (const string &name, int *val) const;
        bool double_attrib (const string &name, double *val) const;

        const char *get_kind() const;
        bool is_kind(const char *kind) const;
        bool is_kind(const string& kind) const;

        void send_impulse(const GridPos& dest, Direction dir);
        void send_impulse(const GridPos& dest, Direction dir, double delay);

        // Object interface
        virtual void message(const string& msg, const Value &val);
        virtual void set_attrib(const string& key, const Value &val);
        virtual const Value* get_attrib(const string& key) const;

        virtual Object *clone()=0;
        virtual void dispose()=0;

        virtual const ObjectTraits *get_traits() const { return 0; }
        virtual void warning(const char *format, ...) const;
    private:
        typedef px::AssocList<std::string, Value> AttribMap;
        AttribMap                                 attribs;
    };
}

/*
 * GridObject is the base class for everything that can only be placed
 * on "The Grid", i.e., for floor tiles, items, and stones.
 */
namespace world
{
    class GridObject : public Object, public display::ModelCallback {
    public:
        GridObject(const char * kind) : Object(kind) {}

        void creation(GridPos p) {
            pos = p;
            on_creation();
        }
        void removal() { on_removal(); }

        virtual bool on_laserhit(Direction /*d*/) { return false; }

        // GridObject interface
        virtual void actor_enter(Actor */*a*/) {}
        virtual void actor_leave(Actor */*a*/) {}

        GridPos get_pos() const { return pos; }
        void play_sound(const char *name);

        void warning(const char *format, ...) const;

    private:
        // ModelCallback interface.
        void animcb() {}

        // GridObject interface
        virtual void on_creation() = 0;
        virtual void on_removal() = 0;

        // Variables
        GridPos pos;
    };

    template <GridLayer LAYER>
    class TGridObject : public GridObject
    {
    public:
        TGridObject(const char *kind) : GridObject(kind) {}

    protected:

        // GridObject interface
        void on_creation() {
            init_model();
        }
        void on_removal() {
            display::KillModel(GridLoc(LAYER, get_pos()));
        }
        virtual void init_model() { set_model(get_kind()); }

        // Helper functions
        display::Model *set_model(const std::string &mname) {
            return set_model(mname.c_str());
        }
        display::Model *set_model(const char *mname) {
            return display::SetModel(GridLoc(LAYER, get_pos()), mname);
        }

        display::Model *set_anim(const std::string &mname) {
            return set_anim(mname.c_str());
        }
        display::Model *set_anim(const char *mname) {
            display::Model *m = set_model(mname);
            m->set_callback(this);
            return m;
        }
        display::Model *get_model() {
            return display::GetModel(GridLoc(LAYER, get_pos()));
        }
    };
}

//
// Floor
//
namespace world
{
    enum FloorFlags {
        FLOOR_CanCrack = 0x01,
        FLOOR_ItemsAllowed = 0x02,

        FLOOR_Normal = FLOOR_CanCrack | FLOOR_ItemsAllowed
    };

    struct FloorTraits {
        FloorTraits (const char *n, double f, double m, FloorFlags fl)
        : name(n), friction(f), mousefactor(m), flags(fl)
        {}

	string  name;
        double  friction;
        double  mousefactor;
        FloorFlags  flags;
     };

    class Floor : public TGridObject<GRID_FLOOR> {
    public:
        Floor (const FloorTraits &tr);
        Floor (const char *kind, double friction, double mfactor);

        // Object interface
        Floor *clone();
        void dispose();
        void message(const string& msg, const Value &val);

        // Floor interface
        virtual px::V2 process_mouseforce (Actor *a, px::V2 force);
        virtual px::V2 get_force(Actor */*a*/) { return px::V2(); }

        virtual void on_drop(Item* /*it*/) {}
        virtual void on_pickup(Item* /*it*/) {}

        virtual void stone_change(Stone */*st*/) {}

        virtual void actor_contact (Actor */*a*/) {}

        double friction() const { return traits.friction; }
        double mousefactor() const { return traits.mousefactor; }

	bool is_destroyable() const {return true;} 
    private:
        virtual void on_actorhit(Actor */*a*/) {}
        FloorTraits traits;
    };
}

//
// Item
//
namespace world
{
    // What may happen to an item _after_ it was activated?
    enum ItemAction {
        ITEM_DROP,              // drop it to the floor
        ITEM_DROP_AS_STONE,     // drop to floor as stone (e.g. Brake->BrakeStone)
        ITEM_KILL,              // remove it from the inventory and dispose it
        ITEM_KEEP,              // keep it in the inventory; do nothing further
    };

    struct ItemTraits {

    };

    class Item : public TGridObject<GRID_ITEMS> {
    public:
        Item(const char *kind);

        //
        // Item interface
        //
        virtual int get_id() const { return -1; }

        // Return true if item completely covers the floor. In this
        // case the floors `actor_contact' method will not be called
        // automatically; this must be done from `Item::actor_hit' (if
        // at all).
        virtual bool covers_floor() const { return false; }

        virtual px::V2 get_force(Actor *a);

        // Called when item is dropped or picked up by actor A
        virtual void on_drop(Actor *a);
        virtual void on_pickup(Actor *a);

        // Called when stone above item changes
        virtual void stone_change(Stone *st);

        // Called when item is ``hit'' by a moving stone.
        virtual void on_stonehit(Stone *st);

        // Return true if the item should be picked up.
        virtual bool actor_hit(Actor *a);

        // The model used for displaying this item in an inventory
        virtual string get_inventory_model();

        // Item is activated by the player
        virtual ItemAction activate(Actor* a, GridPos p);
    };
}

//
// Stone
//
namespace world
{
    struct StoneTraits {
        const char *collision_sound;
        double restitution;
        bool transparent;
    };

    class Stone : public TGridObject<GRID_STONES> {
    public:
        Stone(const char *kind);

        // Stone interface

        virtual StoneResponse collision_response(const StoneContact &sc);

        virtual px::V2 actor_impulse (const StoneContact &sc);
        virtual void   actor_hit (const StoneContact &sc);
        virtual void   actor_inside (Actor */*a*/) {}
        virtual void   actor_contact (Actor */*a*/) {}

        virtual bool   is_movable() { return false;}
        virtual bool   is_removable() { return true;}
        virtual bool   is_floating() const { return false; }

        virtual void   on_move() {}
        virtual void   floor_change() {}
        virtual void   on_impulse(const Impulse& impulse) { if (is_movable()) move_stone(impulse.dir); }

        virtual const char *collision_sound();

    protected:
        bool move_stone(Direction dir);
        static Direction get_push_direction (const StoneContact &sc);
        static bool maybe_push_stone (const StoneContact &sc);
    };
}

/*
** Actor
*/
namespace world
{
    class Actor : public Object {
    public:
        // Actor interface.
        virtual void think (double /*dtime*/);

        virtual void on_hit(Actor* /*a*/) {}
        virtual void on_creation(const px::V2 &pos);
        virtual void on_respawn (const px::V2 &pos);

        virtual bool is_dead() = 0;
	virtual bool is_movable() { return true; }
        virtual bool is_flying() { return false; }
        virtual bool is_on_floor() { return true; }

        virtual bool can_drop_items() { return false; }
        virtual bool has_shield() const { return false; }

        virtual void init();

        // Object interface
        void set_attrib(const string &key, const Value &val);

        // Methods

        void move();
        void warp(const px::V2 &newpos);

        void respawn();
        void set_respawnpos(const V2& p);
        void remove_respawnpos();

        void add_force (const px::V2 &f);

        // Accessors.
        world::ActorInfo *get_actorinfo() { return &actorinfo; }
        const px::V2 &get_pos() const { return actorinfo.pos; }
        const px::V2 &get_vel() const { return actorinfo.vel; }

        double get_radius() const { return actorinfo.radius; }
        double get_mass() const { return actorinfo.mass; }
        double get_charge() const { return actorinfo.charge; }

        bool has_spikes() const { return spikes; }
        void set_spikes(bool has) { spikes = has; }

    protected:
        Actor(const char *name, const px::V2 &pos);
        void set_model(const string &mname);
        const display::SpriteHandle &get_sprite() const { return m_sprite; }

    private:
        virtual void on_motion(px::V2 /*newpos*/) {}

        // Variables
        world::ActorInfo      actorinfo;
        display::SpriteHandle m_sprite;
        V2                    startingpos;
        V2                    respawnpos;
        bool                  use_respawnpos;
        bool                  spikes; // set by "it-pin"
    };
}

namespace world
{
    void SendMessage(Object *o, const string &msg);
    void SendMessage(Object *o, const string &msg, const Value& value);

    enum ExplosionType { DYNAMITE, BLACKBOMB, WHITEBOMB, BOMBSTONE };
    void SendExplosionEffect(GridPos p, ExplosionType type);

    /* This function is used by all triggers, switches etc. that
       perform some particular action when activated (like opening
       doors or switching lasers on and off). It interprets the
       "action" and "target" attributes of `o'. */
    void PerformAction(Object *o, bool onoff);

    Object *MakeObject (const char *kind);
    Floor  *MakeFloor (const char *kind);
    Item   *MakeItem (const char *kind);
    Stone  *MakeStone (const char *kind);
    Actor  *MakeActor (const char *kind);

    void DisposeObject(Object *o);

    void DefineSimpleStone(const string &kind, const string &sound, int hollow, int glass);
    void DefineSimpleStoneMovable(const string &kind, const string &sound, int glass);
    void DefineSimpleFloor(const string &kind, double friction, double mousefactor);

    /* Register a new object. */
    void Register (Object *obj);
    void Register (const string &kind, Object *obj);
    void Register (const string &kind, Floor *obj);
    void Register (const string &kind, Item *obj);
    void Register (const string &kind, Stone *obj);
    void Register (const string &kind, Actor *obj);

    /* Shutdown object repository */
    void Repos_Shutdown();

    Object *GetObjectTemplate(const string &kind);
}
#endif
