#-------------------------------------------------------------------------------
#
#  Define the event objects and traits used by Enable components.
#
#  Written by: David C. Morrill
#
#  Date: 09/22/2003
#
#  (c) Copyright 2003 by Enthought, Inc.
#
#  Classes defined: MouseEvent
#                   DragEvent
#                   MouseHandler
#
#  Traits defined:  mouse_event_trait
#                   drag_event_trait
#
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
#  Imports:
#-------------------------------------------------------------------------------
    
from enthought.traits.api import HasTraits, Event, ReadOnly, false

#-------------------------------------------------------------------------------
#  Miscellaneous events:  
#-------------------------------------------------------------------------------

simple_event_trait = Event( True )

#-------------------------------------------------------------------------------
#  'MouseEvent' class:
#-------------------------------------------------------------------------------

class MouseEvent ( HasTraits ):
    
    #---------------------------------------------------------------------------
    #  Trait definitions:
    #---------------------------------------------------------------------------

    x            = ReadOnly        
    y            = ReadOnly 
    alt_down     = ReadOnly
    control_down = ReadOnly
    shift_down   = ReadOnly
    left_down    = ReadOnly
    middle_down  = ReadOnly
    right_down   = ReadOnly
    mouse_wheel  = ReadOnly
    handled      = false
    
    #----------------------------------------------------------------------------
    #  Return the mouse coordinates relative to a specified component: 
    #----------------------------------------------------------------------------
    
    def xy ( self, component ):
        cx, cy = component.location()
        return ( self.x - cx, self.y - cy )

mouse_event_trait = Event( MouseEvent )

#-------------------------------------------------------------------------------
#  'DragEvent' class:
#-------------------------------------------------------------------------------

class DragEvent ( HasTraits ):
    
    #---------------------------------------------------------------------------
    #  Trait definitions:
    #---------------------------------------------------------------------------

    x           = ReadOnly        
    y           = ReadOnly
    x0          = ReadOnly
    y0          = ReadOnly
    copy        = ReadOnly
    components  = ReadOnly 
    start_event = ReadOnly
    handled     = false
    
    #----------------------------------------------------------------------------
    #  Return the mouse coordinates relative to a specified component: 
    #----------------------------------------------------------------------------
    
    def xy ( self, component ):
        cx, cy = component.location()
        return ( self.x - cx, self.y - cy )

drag_event_trait = Event( DragEvent )

#-------------------------------------------------------------------------------
#  'MouseHandler' class:
#-------------------------------------------------------------------------------

class MouseHandler ( HasTraits ):
    
    #---------------------------------------------------------------------------
    #  Trait definitions:
    #---------------------------------------------------------------------------

    pre_left_down     = mouse_event_trait
    pre_left_up       = mouse_event_trait
    pre_left_dclick   = mouse_event_trait
    pre_right_down    = mouse_event_trait
    pre_right_up      = mouse_event_trait
    pre_right_dclick  = mouse_event_trait
    pre_middle_down   = mouse_event_trait
    pre_middle_up     = mouse_event_trait
    pre_middle_dclick = mouse_event_trait
    pre_mouse_move    = mouse_event_trait
    pre_mouse_wheel   = mouse_event_trait
    pre_window_enter  = mouse_event_trait
    pre_window_leave  = mouse_event_trait
                                         
    left_down         = mouse_event_trait
    left_up           = mouse_event_trait
    left_dclick       = mouse_event_trait
    right_down        = mouse_event_trait
    right_up          = mouse_event_trait
    right_dclick      = mouse_event_trait
    middle_down       = mouse_event_trait
    middle_up         = mouse_event_trait
    middle_dclick     = mouse_event_trait
    mouse_move        = mouse_event_trait
    mouse_wheel       = mouse_event_trait
    window_enter      = mouse_event_trait
    window_leave      = mouse_event_trait

#-------------------------------------------------------------------------------
#  'KeyEvent' class:
#-------------------------------------------------------------------------------

class KeyEvent ( HasTraits ):
    
    #---------------------------------------------------------------------------
    #  Trait definitions:
    #---------------------------------------------------------------------------

    # 'character' is a single ASCII character or is a string describing the
    # high-bit and control characters.  (See subpackage wx.window.key_map)
    character    = ReadOnly 
    alt_down     = ReadOnly
    control_down = ReadOnly
    shift_down   = ReadOnly
    x            = ReadOnly        
    y            = ReadOnly
    event        = ReadOnly
    handled      = false

key_event_trait = Event( KeyEvent )

