#-------------------------------------------------------------------------------
#  
#  Defines the OMLabel class of the Enable 'om' (Object Model) package. 
#
#  An OMLabel is a subclass of Label suitable for use as an inner component
#  of an OMComponent object. That is, it allows an optional icon/text label
#  to be added to an OMComponent object, typically to label or identify the
#  component.
#
#  Written by: David C. Morrill
#  
#  Date: 02/11/2005
#  
#  (c) Copyright 2005 by Enthought, Inc.
#  
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
#  Imports:  
#-------------------------------------------------------------------------------

from om_base             import om_handler
from om_traits           import OffsetXY, SizeXY, ergba_color_editor
                        
from enthought.enable    import Label
                        
from enthought.traits.api    import Property
from enthought.traits.ui.api import View, Group, Item

#-------------------------------------------------------------------------------
#  'OMLabel' class:  
#-------------------------------------------------------------------------------

class OMLabel ( Label ):
    
    #---------------------------------------------------------------------------
    #  Trait definitions:  
    #---------------------------------------------------------------------------
    
    # Defines the original and size of the component:
    bounds = Property
    
    # The origin of the component:
    origin = OffsetXY
    
    # Size of the component:
    size   = SizeXY
    
    #---------------------------------------------------------------------------
    #  Trait view definitions:
    #---------------------------------------------------------------------------
    
    traits_view = View(
        [ 'text$',        ' ', 
          'font',         ' ', 
          Item( 'color',
                editor = ergba_color_editor ), ' ', 
          Item( 'shadow_color', 
                editor = ergba_color_editor ), ' ', 
          'style',
          '|{Text}@' ],
        [ Item( 'bg_color{Background Color}', 
                editor = ergba_color_editor ), '_',
          Item( 'border_color', 
                editor = ergba_color_editor ), '_', 
          'border_size',
          '|{Border}@' ],
        [ 'text_position',      '_',
          'image_position',     '_', 
          'image_orientation',  ' ', 
          'image',
          '|{Position}@' ],
        [ [ 'spacing_height{Height}', 'spacing_width{Width}', 
            '|[Spacing]' ],  
          [ 'padding_left{Left}',  'padding_right{Right}',  
            'padding_top{Top}',    'padding_bottom{Bottom}',
            '|[Padding]' ],
          [ 'margin_left{Left}', 'margin_right{Right}',   
            'margin_top{Top}',   'margin_bottom{Bottom}',
            '|[Margin]' ],
          '|{Margin}' ],
        handler = om_handler
    )
    
#-- Property Implementations ---------------------------------------------------

    #---------------------------------------------------------------------------
    #  Implementation of the 'bounds' property:  
    #---------------------------------------------------------------------------
    
    def _get_bounds ( self ):
        x, y, dx, dy = self.container.bounds
        ox, oy       = self.origin
        dx, dy       = self.size
        return ( x + ox, y + oy, dx, dy )
        
    def _set_bounds ( self, bounds ):
        old              = self.bounds
        x, y, dx, dy     = bounds
        cx, cy, cdx, cdy = self.container.bounds
        self.origin      = ( int( x - cx ), int( y - cy ) )
        self.size        = ( int( dx ), int( dy ) )
        self.trait_property_changed( 'bounds', old, bounds ) 
        
#-- Event Handlers -------------------------------------------------------------
  
    def _right_up_changed ( self, event ):
        pass
  
    def _left_dclick_changed ( self, event ):
        pass
        
#-- 'normal' State -------------------------------------------------------

    def normal_mouse_move ( self, event ):
        self.pointer = 'arrow'
        
    def normal_left_down ( self, event ):
        event.handled           = True
        self.window.mouse_owner = self
        self.event_state        = 'drag_pending'
        self._x_y               = ( event.x, event.y )
        
#-- 'drag_pending' State -------------------------------------------------------

    def drag_pending_left_up ( self, event ):
        event.handled           = True
        self.window.mouse_owner = None
        self.event_state        = 'normal'
        self.pointer            = 'arrow'
        component               =  self.container
        component.controller.label_clicked( component, event )
        
    def drag_pending_mouse_move ( self, event ):
        event.handled = True
        x, y          = self._x_y
        if (( abs( x - event.x ) + abs( y - event.y ))) > 2:
            component  = self.container
            controller = component.controller

            if event.alt_down:
                rc = controller.begin_alt_drag( component, event )
            elif event.control_down:
                rc = controller.begin_control_drag( component, event )
            elif event.shift_down:
                rc = controller.begin_shift_drag( component, event )
            elif controller.can_drag_component( component ): 
                self.pointer            = 'sizing'
                self.window.mouse_owner = None
                component.drag( event )
                rc = None
            else:
                rc = None

            if rc is None:
                self.event_state = 'normal'
        
        return
#-- Public Methods -------------------------------------------------------------

    #---------------------------------------------------------------------------
    #  Force the component to be updated:    
    #---------------------------------------------------------------------------

    def update ( self ):
        self.container.update( True )
        
