#-------------------------------------------------------------------------------
#
#  Define the abstract top-level WindowAbstract class (this class must be
#  subclassed by concrete classes which implement the Window functionality
#  for a specific GUI toolkit).
#
#  Written by: David C. Morrill
#
#  Date: 09/22/2003
#
#  (c) Copyright 2003 by Enthought, Inc.
#
#  Classes defined: WindowAbstract
#
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
#  Debug/Testing flags:
#-------------------------------------------------------------------------------

# Gotcha! enabled:
try:
    import enthought.gotcha as gotcha
    have_gotcha = True
except:
    have_gotcha = False

#-------------------------------------------------------------------------------
#  Imports:
#-------------------------------------------------------------------------------

from types            import ListType
from enthought.traits.api import HasTraits, Trait, ReadOnly, Any, Str, false
from base             import bounds_to_coordinates, coordinates_to_bounds, \
                      intersect_coordinates, union_coordinates, \
                      does_disjoint_intersect_coordinates, \
                      disjoint_union_coordinates, transparent_color, \
                      coordinates_to_size, empty_rectangle, bounding_box, \
                      bounding_coordinates, add_rectangles, send_event_to, \
                      xy_in_bounds, BOTTOM_LEFT, TOP, BOTTOM, LEFT, RIGHT
from base_container   import DefaultContainer
from component        import Component
from container        import Container
from events           import MouseHandler
from drag             import DragHandler
from drag_resize      import DragResizeHandler
from enable_traits    import white_color_trait, CURSOR_X, CURSOR_Y

#-------------------------------------------------------------------------------
#  Constants:
#-------------------------------------------------------------------------------

# Expand a rectangle by a small amount:
expand_1 = ( -1, -1, 1, 1 )

# Paint drag modes:
DRAG_NONE   = 0   # Normal
DRAG_MOVE   = 1   # Drag move is in progress
DRAG_RESIZE = 2   # Drag resize is in progress

#-------------------------------------------------------------------------------
#  'AbstractWindow' class:
#-------------------------------------------------------------------------------

class AbstractWindow ( DefaultContainer ):

    #---------------------------------------------------------------------------
    #  Trait definitions:
    #---------------------------------------------------------------------------

    component     = Trait( None, Component )
    focus_owner   = Trait( None, Component )
    mouse_owner   = Trait( None, MouseHandler )
    bg_color      = white_color_trait
    window        = ReadOnly
    alt_pressed   = false
    ctrl_pressed  = false
    shift_pressed = false
    tooltip       = Str
    __            = Any

    #---------------------------------------------------------------------------
    #  Initialize the object:
    #---------------------------------------------------------------------------
    
    def __init__ ( self, **traits ):
        self.window               = self
        self._size                = ( -1, -1 )
        self._scroll_origin       = ( 0.0, 0.0 )
        self._update_region       = None
        self._gc                  = None
        self._drag_bg_gc          = None
        self._drag_fg_gc          = None
        self._drag_bg_bounds_rect = None
        self._drag_fg_bounds_rect = None
        self._drag_update         = DRAG_NONE
        self._drag_update_region  = empty_rectangle
        self._gc_temp             = self._create_gc( ( 1, 1 ) )
        HasTraits.__init__( self, **traits )
       
        # Create a default component (if necessary):
        if self.component is None:
            self.component = Container()
       
    #---------------------------------------------------------------------------
    #  Handle the associated component being changed:
    #---------------------------------------------------------------------------

    def _component_changed ( self, old, new ):
        if old is not None:
            old.on_trait_change( self.on_bounds_change, 'bounds',
                                 remove = True )
            del old.container

        if new is None:
            self.component = Container()
            return
            
        new.container = self
        new.on_trait_change( self.on_bounds_change, 'bounds' )

        # If possible, size the new component according to the size of the 
        # toolkit control; otherwise, size it according to the new component's
        # desired size.
        size = self._get_control_size()
        if size is not None:
            self._set_component_size( size )
            
        else:
            self._set_component_size( new.bounds[2:] )
            
        self.redraw()
    
    #----------------------------------------------------------------------------
    #  Handle the associated component changing its size: 
    #----------------------------------------------------------------------------
    
    def on_bounds_change ( self, bounds ):
        self._set_component_size( bounds[2:] )
        
    #---------------------------------------------------------------------------
    #  Handle the 'mouse_owner' being changed:
    #---------------------------------------------------------------------------

    def _mouse_owner_changed ( self, mouse_owner ):
        if mouse_owner is None:
            self._release_mouse()
        else:
            self._capture_mouse()
          
    #---------------------------------------------------------------------------
    #  Add new components to the window:
    #---------------------------------------------------------------------------

    def add ( self, *components ):
        for component in components:
            self.component.add( component )

    #---------------------------------------------------------------------------
    #  Remove some components from the window:
    #---------------------------------------------------------------------------

    def remove ( self, *components ):
        # If no items were specified, remove our current contents:
        if not components:
            self.component = Container()
            return

        # Otherwise, remove the components specified from our component object:
        for component in components:
            self.component.remove( component )
          
    #---------------------------------------------------------------------------
    #  Handle the various mouse events that can occur:
    #---------------------------------------------------------------------------
 
    def _on_left_down ( self, event ):
        self._handle_mouse_event( 'left_down', event, set_focus = True )
 
    def _on_left_up ( self, event ):
        self._handle_mouse_event( 'left_up', event )
 
    def _on_left_dclick ( self, event ):
        self._handle_mouse_event( 'left_dclick', event )
 
    def _on_right_down ( self, event ):
        self._handle_mouse_event( 'right_down', event, set_focus = True )
 
    def _on_right_up ( self, event ):
        self._handle_mouse_event( 'right_up', event )
 
    def _on_right_dclick ( self, event ):
        self._handle_mouse_event( 'right_dclick', event )
 
    def _on_middle_down ( self, event ):
        self._handle_mouse_event( 'middle_down', event )
 
    def _on_middle_up ( self, event ):
        self._handle_mouse_event( 'middle_up', event )
 
    def _on_middle_dclick ( self, event ):
        self._handle_mouse_event( 'middle_dclick', event )
 
    def _on_mouse_move ( self, event ):
        self._handle_mouse_event( 'mouse_move', event, 1 )
        
    def _on_mouse_wheel ( self, event ):
        self._handle_mouse_event( 'mouse_wheel', event )
 
    def _on_window_enter ( self, event ):
        self._handle_mouse_event( 'window_enter', event )
 
    def _on_window_leave ( self, event ):
        self._handle_mouse_event( 'window_leave', event, -1 )
           
    #---------------------------------------------------------------------------
    #  Generic mouse event handler:
    #---------------------------------------------------------------------------
     
    # This routine is for performance testing purposes:
    def _handle_mouse_event ( self, event_name, event, 
                                    cursor_check = 0, set_focus = False ):
        if have_gotcha:
            gotcha.profile( self._timed_handle_mouse_event, event_name, event, 
                            cursor_check, set_focus )
        else:
            self._timed_handle_mouse_event( event_name, event, cursor_check, 
                                            set_focus )
   
    def _timed_handle_mouse_event ( self, event_name, event, cursor_check, 
                                          set_focus ):
        mouse_event = self._create_mouse_event( event )
        components  = self.component.components_at( mouse_event.x, 
                                                    mouse_event.y )
        mouse_owner = self.mouse_owner
        if mouse_owner is not None:
            setattr( mouse_owner, event_name, mouse_event )
            self._pointer_owner = mouse_owner
        else:    
            send_event_to( components, event_name, mouse_event )
            
            # Make sure the pointer is up to date:
            if len( components ) > 0:
                pointer_owner = components[-1]
                if pointer_owner is not self._pointer_owner:
                    self._pointer_owner = pointer_owner
                    self._set_pointer( pointer_owner.pointer )
            
        if cursor_check == 1:
            n = len( components )
            if n > 0:
                i = 0
                for i in xrange( n - 1, 0, -1 ):
                    if components[i].tooltip != '':
                        break
                self._tooltip_check( components[i] )
                    
            for i in xrange( n - 1, -1, -1 ):
                if self._cursor_test( components[i], mouse_event ):
                    break
            else:
                self._cursor_check()
            #self.set_tooltip( components )
        elif cursor_check == -1:
            self._cursor_check()

        # If this event requires setting the keyboard focus, set the first
        # component under the mouse pointer that accepts focus as the new focus
        # owner (otherwise, nobody owns the focus):
        if set_focus: 
            for i in xrange( len( components ) - 1, -1, -1 ):
                if components[i].accepts_focus:
                    new_focus_owner = components[i]
                    break
            else:
                new_focus_owner = None
                
            old_focus_owner  = self.focus_owner
            self.focus_owner = new_focus_owner
            if ((old_focus_owner is not None) and 
                (old_focus_owner is not new_focus_owner)):
                old_focus_owner.has_focus = False
            if new_focus_owner is not None:
                new_focus_owner.has_focus = True
                self._set_focus()
            
    #---------------------------------------------------------------------------
    #  Sets the window's tooltip (if necessary):  
    #---------------------------------------------------------------------------
                        
    def set_tooltip ( self, components ):
        for i in xrange( len( components ) - 1, -1, -1 ):
            tooltip = components[i].tooltip
            if tooltip != '':
                break
        else:
            tooltip = ''
        if tooltip != self.tooltip:
            self.tooltip = tooltip
            self._set_tooltip( tooltip )
            
    #---------------------------------------------------------------------------
    #  Set up the cursor style used by a specified component:
    #---------------------------------------------------------------------------
                    
    def _cursor_test ( self, component, event ):
        style = component.cursor_style_
        if style < 0:
            return False
        cursor_x = cursor_y = None
        bounds   = component._cursor_bounds()
        if xy_in_bounds( event.x, event.y, bounds ): 
            x, y, dx, dy = bounds
            if (style & CURSOR_X) != 0:
                cursor_x = ( event.x, y, dy )
            if (style & CURSOR_Y) != 0:
                cursor_y = ( event.y, x, dx )
        self._cursor_check( cursor_x, cursor_y, component.cursor_color_ )
        return True

    #---------------------------------------------------------------------------
    #  Request that a specified region of the window be redrawn:  
    #--------------------------------------------------------------------------- 
    
    def redraw ( self, bounds = None, coordinates = False ):
        if bounds is empty_rectangle:
            return
        if bounds is None:
            self._update_region = None
        elif self._update_region is not None:
            if not coordinates:
                bounds = bounds_to_coordinates( bounds )
            self._update_region = disjoint_union_coordinates( 
                                     self._update_region, bounds )
        self._redraw( bounds )
                                            
    #---------------------------------------------------------------------------
    #  Determine if a specified region intersects the update region:
    #---------------------------------------------------------------------------
    
    def _needs_redraw ( self, bounds ):
        return does_disjoint_intersect_coordinates( self._update_region,
                                               bounds_to_coordinates( bounds ) )
                                            
    #---------------------------------------------------------------------------
    #  Begin a 'drag resize' operation:  
    #---------------------------------------------------------------------------                                            

    def drag_resize ( self, component, bounds, event, 
                            anchor       = BOTTOM_LEFT, 
                            unconstrain  = TOP | BOTTOM | LEFT | RIGHT, 
                            drag_handler = None, 
                            inset        = expand_1, 
                            alpha        = 0.67, 
                            overlay      = False ):
                   
        # Assume the current window component is being used as the 'drag_root':
        drag_root = self.component
            
        # If the drag bounds are specified as a component, use its bounds
        # and make the component the 'drag_root':
        if isinstance( bounds, Component ):
            drag_root = bounds
            bounds    = bounds.bounds
            
        # Make sure the drag bounds is in coordinate form:
        bounds_rect = bounds
        if bounds_rect is not None:
            bounds_rect = bounds_to_coordinates( bounds_rect )

        # Get the bounding box of the component being resized:
        drag_rect = bounds_to_coordinates( component.bounds ) 

        # If there is a specified bounding area but the initial drag bounds
        # are not wholly contained within it, then abort the operation:
        if ((bounds_rect is not None) and 
            (intersect_coordinates( bounds_rect, drag_rect ) != drag_rect)):
            return
            
        # If the caller did not supply a drag handler, use a standard one:
        if not isinstance( drag_handler, DragResizeHandler ):
            handler = DragResizeHandler()
            
            # If the caller just supplied a 'drag validate' handler, connect
            # it to the standard handler:
            if drag_handler is not None:
                handler.drag_validate = drag_handler
            drag_handler = handler

        # Set up the appropriate 'drag_done' event handler based on which
        # mouse button is currently pressed:
        for name in [ 'left_down', 'right_down', 'middle_down' ]:
            if getattr( event, name ):
                self._drag_event = name.replace( 'down', 'up' )
                drag_handler.on_trait_change( self._drag_done, self._drag_event)                              
                break
        else:
            # If no mouse button is pressed, abort the operation:
            return
            
        # Initialize the drag handler:
        inset_drag_rect = add_rectangles( inset, drag_rect )
        drag_handler.init( component, bounds_rect, anchor, unconstrain, 
                           self._drag_event, event )
        
        # Make sure the drag handler gets all future mouse events:
        self.mouse_owner = drag_handler
               
        # Compute the area of the background drag layer:
        if overlay or (bounds_rect is None):
            bg_size        = self._size
            bg_bounds_rect = ( 0.0, 0.0 ) + bg_size
        else:
            bg_size        = coordinates_to_size( bounds_rect )
            bg_bounds_rect = bounds_rect
        self._drag_bg_bounds_rect = bg_bounds_rect
        
        # Save the current drag bounds for doing later screen updates:
        self._drag_fg_bounds_rect = inset_drag_rect 
               
        # Create the background image buffer context:
        self._drag_bg_gc = bg_gc = self._create_gc( bg_size )
        
        # Save the component being resized (for the 'paint' handler):
        self._drag_component = component
        
        if overlay:
            # For an overlay mode operation, use the current background:
            bg_gc.draw_image( self._gc, bg_bounds_rect )
        else:
            # Force an update of the entire background area:
            saved_update_region = self._update_region
            self._update_region = [ bg_bounds_rect ]
            
            # Make sure we don't draw the resized component in the static 
            # background image:
            component._drawable = False
            
            # Translate the origin so that all drawn components think they are
            # drawing on a full canvas:
            bg_gc.translate_ctm( -bg_bounds_rect[0], -bg_bounds_rect[1] )
            
            # Draw the root background component and all of its subcomponents:
            drag_root.draw( bg_gc )
            
            # Restore the previous update region:
            self._update_region = saved_update_region
            
            # Finally, redraw the portion of the screen bounded by the drag
            # source. This should not be necessary, since the contents of the 
            # screen really haven't changed yet, but anti-aliasing effects
            # can leave 'edge' artifacts around the drag source when it is 
            # moved if we don't do this (which is why we 'expand' the drag 
            # area slightly): 
            self.redraw( inset_drag_rect, coordinates = True )
            
            # Allow the resized component to be drawn again:
            component._drawable = True
            
            # Force the alpha channel to 1.0:
            new_gc = self._create_gc( bg_size )
            new_gc.clear( (1.0, 1.0, 1.0, 1.0) )
            new_gc.draw_image(bg_gc)
            self._drag_gc_gc = bg_gc = new_gc
            #bg_gc.alpha( 1.0 )
            
        # Make sure any 'paint' operations know a drag resize is in progress:
        self.drag_resize_update()

    #---------------------------------------------------------------------------
    #  Indicate a drag resize update is in progress:
    #---------------------------------------------------------------------------
            
    def drag_resize_update ( self ):
        self._drag_update = DRAG_RESIZE
                                            
    #---------------------------------------------------------------------------
    #  Begin a 'drag' operation:  
    #---------------------------------------------------------------------------

    def drag ( self, components, bounds, event, 
                     drag_copy    = False, 
                     drag_handler = None, 
                     alpha        = 0.67,
                     inset        = expand_1, 
                     overlay      = False ):
        # Make sure the event has been marked as handled:
        event.handled = True
        
        # Assume the current window component is being used as the 'drag_root':
        drag_root = self.component
        
        # Make sure that 'components' is always a list, even if only a single 
        # component is specified:
        if type( components ) != ListType:
            components = [ components ]
            
        # If the drag bounds are specified as a component, use its bounds
        # and make the component the 'drag_root':
        if isinstance( bounds, Component ):
            drag_root = bounds
            bounds    = bounds.bounds
            
        # Make sure the drag bounds is in coordinate form:
        bounds_rect = bounds
        if bounds_rect is not None:
            bounds_rect = bounds_to_coordinates( bounds_rect )

        # Compute the bounding box of all components being dragged:
        drag_rect = bounding_box( components ) 

        # If there is a specified bounding area but the initial drag bounds
        # are not wholly contained within it, then abort the operation:
        if ((bounds_rect is not None) and 
            (intersect_coordinates( bounds_rect, drag_rect ) != drag_rect)):
            return
            
        # If the caller did not supply a drag handler, use a standard one:
        if not isinstance( drag_handler, DragHandler ):
            handler = DragHandler()
            
            # If the caller just supplied a 'drag validate' handler, connect
            # it to the standard handler:
            if drag_handler is not None:
                handler.drag_validate = drag_handler
            drag_handler = handler

        # Set up the appropriate 'drag_done' event handler based on which
        # mouse button is currently pressed:
        for name in [ 'left_down', 'right_down', 'middle_down' ]:
            if getattr( event, name ):
                self._drag_event = name.replace( 'down', 'up' )
                drag_handler.on_trait_change( self._drag_done, self._drag_event)                              
                break
        else:
            # If no mouse button is pressed, abort the operation:
            return
            
        # Initialize the drag handler:
        inset_drag_rect = add_rectangles( inset, drag_rect )
        drag_handler.init( components, bounds_rect, drag_root, inset_drag_rect, 
                           drag_copy, self._drag_event, event )
        
        # Make sure the drag handler gets all future mouse events:
        self.mouse_owner = drag_handler
               
        # Compute the area of the background drag layer:
        if overlay or (bounds_rect is None):
            bg_size        = self._size
            bg_bounds_rect = ( 0.0, 0.0 ) + bg_size
        else:
            bg_size        = coordinates_to_size( bounds_rect )
            bg_bounds_rect = bounds_rect
        self._drag_bg_bounds_rect = bg_bounds_rect
        
        # Force an update of the entire background area:
        saved_update_region = self._update_region
        self._update_region = [ bg_bounds_rect ]
            
        # Save the current drag bounds for doing later screen updates:
        self._drag_fg_bounds_rect = inset_drag_rect 
        
        # Create the 'drag layer' containing all of the dragged components:
        self._drag_fg_gc = fg_gc = self._create_gc(
                                        coordinates_to_size( inset_drag_rect ),
                                        pix_format = "bgra32" )
        fg_gc.clear( transparent_color )
        
        # Translate the origin so that all drawn components think they are
        # drawing on a full canvas:
        fg_gc.translate_ctm( -inset_drag_rect[0], -inset_drag_rect[1] )
        
        # Draw each of the components into the drag layer:
        for component in components:
            component.draw( fg_gc )
        
        # To create a mask, we create a new GC and set its alpha channel to the
        # desired mask alpha using clear(). Then we draw the existing gc onto
        # it:
        if alpha >= 0.0:
            new_gc = self._create_gc( ( fg_gc.width(), fg_gc.height() ), 
                                      pix_format = "bgra32" )
            new_gc.clear( ( 1.0, 1.0, 1.0, alpha ) )
            new_gc.set_alpha( alpha )
            new_gc.draw_image( fg_gc )
            self._drag_fg_gc = fg_gc = new_gc
        
        # Create the background image buffer context:
        self._drag_bg_gc = bg_gc = self._create_gc( bg_size )
        
        # If we are not dragging a copy of the components, make sure that we 
        # don't draw the dragged components in the static background image:
        if not drag_copy:
            for component in components:
                component._drawable = False
        
        if overlay:
            # For an overlay mode operation, use the current background:
            bg_gc.draw_image( self._gc, bg_bounds_rect )
            
            # Restore the original update region:
            self._update_region = saved_update_region
        else:
            # Translate the origin so that all drawn components think they are
            # drawing on a full canvas:
            bg_gc.translate_ctm( -bg_bounds_rect[0], -bg_bounds_rect[1] )
            
            # Finally, redraw the portion of the screen bounded by the drag
            # source. This should not be necessary, since the contents of the 
            # screen really haven't changed yet, but anti-aliasing effects
            # can leave 'edge' artifacts around the drag source when it is 
            # moved if we don't do this (which is why we 'expand' the drag 
            # area slightly): 
            self.redraw( inset_drag_rect, coordinates = True )

    #---------------------------------------------------------------------------
    #  Update the current drag bounding box:
    #---------------------------------------------------------------------------
            
    def set_drag_bounds_rect ( self, drag_bounds_rect ):
           
        # Make sure that areas affected by the drag are updated:
        self._drag_update_region = union_coordinates( self._drag_update_region,
                                   union_coordinates( self._drag_fg_bounds_rect, 
                                                      drag_bounds_rect ) )
        self._redraw( self._drag_fg_bounds_rect )
        self._redraw( drag_bounds_rect )
        
        # Save the new drag bounds:
        self._drag_fg_bounds_rect = drag_bounds_rect

        # Tell the 'paint' routine that a drag update is in progress:
        self._drag_update = DRAG_MOVE
        
    #---------------------------------------------------------------------------
    #  Handle the end of a drag operation:
    #---------------------------------------------------------------------------
        
    def _drag_done ( self, drag_handler, trait_name, event ):               
        # Discard all temporary drag data:
        self._drag_bg_gc          = self._drag_fg_gc = None
        self._drag_bg_bounds_rect = self._drag_fg_bounds_rect = None
        self._drag_component      = None
        
        # Release the mouse capture:
        self.mouse_owner = None
        
        # Disconnect this event handler from the drag handler:
        drag_handler.on_trait_change( self._drag_done, self._drag_event, 
                                      remove = True )
        
    #---------------------------------------------------------------------------
    #  Paint the contents of the window:  
    #---------------------------------------------------------------------------
                                            
    # This routine is for performance testing purposes:
    def _paint ( self, event = None ):
        if have_gotcha:
            gotcha.profile( self._timed_paint, event )
        else:
            self._on_timed_paint( event )
        
    def _timed_paint ( self, event = None ):
        size = self._get_gc_size( self.component.bounds[2:] )
        if size != self._size:
            self._size = size
            self._gc   = self._create_gc( size )
            self._update_region = None
        
        update_region = self._update_region
        drag_update   = self._drag_update
        if ((update_region is None) or (len( update_region ) > 0) or 
            (drag_update != DRAG_NONE)):
            gc = self._gc
            gc.save_state()
            if update_region is not None:
                drag_update_region = self._drag_update_region
                if ((len( update_region ) > 0) or 
                    (drag_update_region is not empty_rectangle)):
                    region = update_region
                    if drag_update_region is not empty_rectangle:
                        region = update_region + [ drag_update_region ]
                    gc.clip_to_rects( 
                           [ coordinates_to_bounds(r) for r in region ] )
                    gc.set_fill_color( self.bg_color_ )
                    bounds = coordinates_to_bounds( 
                                 bounding_coordinates( region ) )
                    gc.begin_path()
                    gc.rect( *bounds )
                    gc.fill_path()
                    #gc.clear_clip_region( self.bg_color_, region )
            else:
                gc.clear( self.bg_color_ )
               
            if self._drag_bg_gc is not None:
                self._drag_update = DRAG_NONE
                
                # If screen updates are needed while in drag move mode, then 
                # the background image needs to be regenerated:
                bg_gc = self._drag_bg_gc
                if (drag_update != DRAG_RESIZE) and update_region:
                    
                    # Clear the (clipped) background graphics context:
                    tmprect = bounding_coordinates( update_region )
                    bg_gc.save_state()
                    bg_gc.clip_to_rect( *tmprect )
                    bg_gc.begin_path()
                    bg_gc.set_fill_color( self.bg_color_ )
                    bg_gc.rect( *tmprect )
                    bg_gc.fill_path()
                    
                    # Draw the component and all of its subcomponents:
                    self.component.draw( bg_gc )
                    
                    # Reset the clipping region:
                    bg_gc.restore_state()
        
                gc.draw_image( bg_gc,
                            coordinates_to_bounds( self._drag_bg_bounds_rect ) )
                if drag_update == DRAG_RESIZE:
                    self._drag_component.draw( gc )
                else:
                    gc.draw_image( self._drag_fg_gc, 
                            coordinates_to_bounds( self._drag_fg_bounds_rect ) )
                    self._drag_update_region = empty_rectangle
            else:
                self.component.draw( gc )
            gc.restore_state()
                
        self._window_paint( event )
        
        self._update_region = []       
        
    #---------------------------------------------------------------------------
    #  Abstract methods that must be implemented by concrete subclasses:
    #---------------------------------------------------------------------------
    
    #---------------------------------------------------------------------------
    #  Capture all future mouse events:
    #---------------------------------------------------------------------------
    
    def _capture_mouse ( self ):
        raise NotImplementedError        
    
    #---------------------------------------------------------------------------
    #  Release the mouse capture:
    #---------------------------------------------------------------------------
    
    def _release_mouse ( self ):
        raise NotImplementedError        
        
    #---------------------------------------------------------------------------
    #  Convert a GUI toolkit mouse event into a MouseEvent:
    #---------------------------------------------------------------------------
    
    def _create_mouse_event ( self, event ):
        raise NotImplementedError        
        
    #---------------------------------------------------------------------------
    #  Request a redraw of the window:
    #---------------------------------------------------------------------------
    
    def _redraw ( self, coordinates ):
        raise NotImplementedError  
        
    #---------------------------------------------------------------------------
    #  Get the appropriate gc size for a specified component size:
    #---------------------------------------------------------------------------
    
    def _get_gc_size ( self, size ):
        raise NotImplementedError  
        
    #---------------------------------------------------------------------------
    #  Set the size of the associated component:
    #---------------------------------------------------------------------------
    
    def _set_component_size ( self, size ):
        raise NotImplementedError  
        
    #---------------------------------------------------------------------------
    #  Get the size of the underlying toolkit control.
    #---------------------------------------------------------------------------
    
    def _get_control_size ( self ):
        raise NotImplementedError
    
    #---------------------------------------------------------------------------
    #  Create a Kiva graphics context of a specified size:
    #---------------------------------------------------------------------------
        
    def _create_gc ( self, size, pix_format = "bgr24" ):
        raise NotImplementedError  
        
    #---------------------------------------------------------------------------
    #  Do a GUI toolkit specific screen update:
    #---------------------------------------------------------------------------
        
    def _window_paint ( self, event ):
        raise NotImplementedError  
        
    #---------------------------------------------------------------------------
    #  Set the current pointer (i.e. cursor) shape:
    #---------------------------------------------------------------------------
        
    def _set_pointer ( self, pointer ):
        raise NotImplementedError  
        
    #---------------------------------------------------------------------------
    #  Set the current tooltip for the window:
    #---------------------------------------------------------------------------
        
    def _set_tooltip ( self, tooltip ):
        raise NotImplementedError  
        
    #---------------------------------------------------------------------------
    #  Set up or cancel a timer for a specified component:
    #---------------------------------------------------------------------------
        
    def _set_timer_interval ( self, component, interval ):
        raise NotImplementedError  
        
    #---------------------------------------------------------------------------
    #  Check to see if the 'cursor' needs to be updated on the screen:
    #---------------------------------------------------------------------------

    def _cursor_check ( self, cursor_x     = None, cursor_y = None, 
                              cursor_color = None ):
        raise NotImplementedError

    #---------------------------------------------------------------------------
    #  Check to see if the 'tooltip' infomation needs to be updated:
    #---------------------------------------------------------------------------

    def _tooltip_check ( self, component ):
        raise NotImplementedError
