#------------------------------------------------------------------------------
# Copyright (c) 2005, Enthought, Inc.
# All rights reserved.
# 
# This software is provided without warranty under the terms of the BSD
# license included in enthought/LICENSE.txt and may be redistributed only
# under the conditions described in the aforementioned license.  The license
# is also available online at http://www.enthought.com/licenses/BSD.txt
# Thanks for using Enthought open source!
# 
# Author: Enthought, Inc.
# Description: <Enthought pyface package component>
#------------------------------------------------------------------------------
""" The abstract base-class for all Pyface dialogs. """

# Standard library imports.
import logging

# Major package imports.
import wx

# Enthought library imports.
from enthought.traits.api import Bool, Enum, Int, Str

# Local imports.
from constant import OK, CANCEL, YES, NO
from window import Window


# Create a logger for this module.
logger = logging.getLogger(__name__)


class Dialog(Window):
    """ The abstract base-class for all Pyface dialogs.

    Usage: Sub-class this class and either override '_create_contents' or
    more simply, just override the two methods that do the real work:-

    1) '_create_dialog_area' creates the main content of the dialog.
    2) '_create_buttons'     creates the dialog buttons.

    """

    RESULT_MAP = {
        wx.ID_OK     : OK,
        wx.ID_CANCEL : CANCEL,
        wx.ID_YES    : YES,
        wx.ID_NO     : NO,
        wx.ID_CLOSE  : CANCEL,
        # fixme: There seems to be a bug in wx.SingleChoiceDialog that allows
        # it to return 0 when it is closed via the window (closing it via the
        # buttons works just fine).
        0            : CANCEL
    }
    
    # Default dialog style.
    STYLE = wx.DEFAULT_DIALOG_STYLE | wx.CLIP_CHILDREN

    #### 'Window' interface ###################################################

    # The dialog title.
    title = Str('Dialog')
    
    #### 'Dialog' interface ###################################################

    # The context sensitive help Id (the 'Help' button is only shown iff this
    # is set).
    help_id = Str

    # Is the dialog resizeable?
    resizeable = Bool(True)
    
    # The dialog style (is is modal or not!).
    style = Enum('modal', 'nonmodal')

    # The size of the border around the dialog area.
    dialog_area_border = Int(5)

    # Label for the 'ok' button
    ok_button_label = Str('OK')

    # Label for the 'cancel' button
    cancel_button_label = Str('Cancel')

    # Label for the 'help' button
    help_button_label = Str('Help')

    ###########################################################################
    # 'Window' interface.
    ###########################################################################

    def open(self):
        """ Opens the window. """

        if self.control is None:
            self._create()

            if self.style == 'modal':
                result = self.control.ShowModal()
                self.close()

                # Map the result from wx terms into Pyface terms.
                self.return_code = self.RESULT_MAP[result]
                
            else:
                self.control.Show(True)
                self.return_code = OK
        else:
            logger.debug('dialog is already open %s' % str(self))

        return self.return_code

    ###########################################################################
    # Protected 'Window' interface.
    ###########################################################################

    def _add_event_listeners(self, control):
        """ Adds any event listeners required by the window. """

        # For dialogs we don't need to catch any wx events.
        pass

    def _create_control(self, parent):
        """ Create the toolkit-specific control that represents the window. """

        style = self.STYLE
        
        if self.resizeable:
            style = style | wx.RESIZE_BORDER
            
        return wx.Dialog(parent, -1, self.title, style=style)

    def _create_contents(self, parent):
        """ Creates the window contents.

        Sub-classes can either override this method completely, or instead
        override the two methods that do the real work:-

        1) '_create_dialog_area' creates the main content of the dialog.
        2) '_create_buttons'     creates the dialog buttons.

        """

        dialog = parent

        sizer = wx.BoxSizer(wx.VERTICAL)
        dialog.SetSizer(sizer)
        dialog.SetAutoLayout(True)

        # The 'guts' of the dialog.
        dialog_area = self._create_dialog_area(dialog)
        sizer.Add(dialog_area, 1, wx.EXPAND | wx.ALL, self.dialog_area_border)

        # The buttons.
        buttons = self._create_buttons(dialog)
        sizer.Add(buttons, 0, wx.ALIGN_RIGHT | wx.ALL, 5)

        # Resize the dialog to match the sizer's minimal size.
        if self.size != (-1,-1):
            dialog.SetSize(self.size)

        else:
            sizer.Fit(dialog)

        dialog.CentreOnParent()
        return dialog

    ###########################################################################
    # Protected 'Dialog' interface.
    ###########################################################################

    def _create_dialog_area(self, parent):
        """ Creates the main content of the dialog. """

        panel = wx.Panel(parent, -1)
        panel.SetBackgroundColour("red")
        panel.SetSize((100, 200))
        
        return panel

    def _create_buttons(self, parent):
        """ Creates the buttons. """

        sizer = wx.BoxSizer(wx.HORIZONTAL)

        # 'OK' button.
        self._ok = ok = wx.Button(parent, wx.ID_OK, self.ok_button_label)
        ok.SetDefault()
        wx.EVT_BUTTON(parent, wx.ID_OK, self._on_ok)
        sizer.Add(ok)

        # 'Cancel' button.
        self._cancel = cancel = wx.Button(parent, wx.ID_CANCEL,
                                          self.cancel_button_label)
        wx.EVT_BUTTON(parent, wx.ID_CANCEL, self._on_cancel)
        sizer.Add(cancel, 0, wx.LEFT, 10)

        # 'Help' button.
        if len(self.help_id) > 0:
            help = wx.Button(parent, wx.ID_HELP, self.help_button_label)
            wx.EVT_BUTTON(parent, wx.ID_HELP, self._on_help)
            sizer.Add(help, 0, wx.LEFT, 10)

        return sizer

    ###########################################################################
    # 'Private' interface.
    ###########################################################################

    #### wx event handlers ####################################################

    def _on_ok(self, event):
        """ Called when the 'OK' button is pressed. """

        self.return_code = OK
        
        # Let the default handler close the dialog appropriately.
        event.Skip()
        
        return

    def _on_cancel(self, event):
        """ Called when the 'Cancel' button is pressed. """

        self.return_code = CANCEL

        # Let the default handler close the dialog appropriately.
        event.Skip()

        return

    def _on_help(self, event):
        """ Called when the 'Help' button is pressed. """

        print 'Heeeeelllllllllllllpppppppppppppppppppp'
        
        return

#### EOF ######################################################################
