""" How to show widgets without labels and labels without widgets.

    Code sample showing how to add annotation text to a View and how
    to suppress labels for one or more widgets.

    Demo class "employee" has eight attributes, which are displayed
    in three groups.  The 'general_info' group has no group label or
    annotation, but has labels for individual widgets.  The 'confidential'
    group has a group label, annotation and labels for each widget.
    The 'comments' group has a group label, but no annotation and no
    widget labels.

    These variations are accomplished as follows:

    1. Groups have no label by default, but one may be specified by
       means of the 'label' keyword.

    2. Widgets do have labels by default (the name of the attribute,
       unless otherwise specified), but they may be turned off for
       a given Group by setting 'show_labels' to False.

    3. Annotation is created by specifying an Item with an empty 'name'
       and a nonempty 'label'.  If such an Item is part of a larger Group,
       the annotation will line up with the widgets in the group rather
       than with their labels.  This can be avoided by placing the 
       annotation Item in a singleton Group as shown below.
"""

from enthought.traits.api \
    import HasTraits, Str, Int
    
from enthought.traits.ui.api \
    import View, Group, Item


class Employee ( HasTraits ):
    """ Demo class for demonstrating annotation and label suppression.
    """
    # General information:
    first_name = Str
    last_name  = Str
    department = Str

    # Confidential information:
    employee_number = Str
    salary          = Int

    # Additional blanks for comments:
    comment1 = Str
    comment2 = Str
    comment3 = Str

    # This group has a border and item labels, but no group label:
    general_info = Group( Item( 'first_name' ),
                          Item( 'last_name' ),
                          Item( 'department' ),
                          show_border = True )

    # This group has a border, a group label and item labels.  It also
    # has an annotation (the singleton Group containing "NOTE:..."
    confidential = Group( Group( Item( label = 'NOTE: These data are '
                                               'confidential!' ) ),
                          Group( Item( 'employee_number' ),
                                 Item( 'salary' ) ),
                          show_border = True,
                          label       = 'Internal' )

    # This group has a border and a group label, but no item labels:
    comments = Group( Item( 'comment1' ),
                      Item( 'comment2' ),
                      Item( 'comment3' ),
                      show_border = True,
                      show_labels = False,
                      label       = 'Comments' )

    # As always, the View is simple if the Groups are properly specified:
    view1 = View( Group( general_info, confidential, comments ) )


# Code to run the demo standalone:
if __name__ == '__main__':
    Employee().configure_traits()
    
