require 'runit/testcase'
require 'runit/cui/testrunner'
require 'open3'

class ERBCmdTest < RUNIT::TestCase
  def erb_cmd(arg = "", src=nil)
    cmd = ["../bin/erb", arg].join(" ")
    Open3.popen3(cmd) do | pin, pout, perr|
      pin.print(src) if src
      pin.close
      return [pout.read, pout.read]
    end
  end

  def test_01_trim
    out, err = erb_cmd("", "hello")
    assert_equal(out, "hello")

    src = <<EOS
%% hi
= hello
<% 3.times do |n| %>
% n=0
* <%= n %>
<% end %>
EOS

    ans = <<EOS
%% hi
= hello

% n=0
* 0

% n=0
* 1

% n=0
* 2

EOS
    out, err = erb_cmd("-P", src)
    assert_equal(ans, out)

    out, err = erb_cmd("-T 0 -P", src)
    assert_equal(ans, out)

    ans = <<EOS
%% hi
= hello
% n=0
* 0% n=0
* 1% n=0
* 2
EOS
    out, err = erb_cmd("-P -T 1", src)
    assert_equal(ans.chomp, out)

    ans  = <<EOS
%% hi
= hello
% n=0
* 0
% n=0
* 1
% n=0
* 2
EOS
    out, err = erb_cmd("-T 2 -P", src)
    assert_equal(ans, out)

    ans = <<EOS
% hi
= hello

* 0

* 0

* 0

EOS

    out, err = erb_cmd("", src)
    assert_equal(ans, out)

    out, err = erb_cmd("-T 0", src)
    assert_equal(ans, out)

    ans = <<EOS
% hi
= hello
* 0* 0* 0
EOS

    out, err = erb_cmd("-T 1 -S 1", src)
    assert_equal(ans.chomp, out)

    ans = <<EOS
% hi
= hello
* 0
* 0
* 0
EOS

    out, err = erb_cmd("-T 2", src)
    assert_equal(ans, out)

  end

  def test_02_safe
    out, err = erb_cmd("", '<%= $SAFE%>')
    assert_equal('0', out)

    out, err = erb_cmd("-S 0", '<%= $SAFE%>')
    assert_equal('0', out)

    out, err = erb_cmd("-S 1", '<%= $SAFE%>')
    assert_equal('1', out)

    out, err = erb_cmd("-S 2", '<%= $SAFE%>')
    assert_equal('2', out)

    out, err = erb_cmd("-S 3", '<%= $SAFE%>')
    assert_equal('3', out)

    out, err = erb_cmd("-S 4", '<%= $SAFE%>')
    assert_equal('4', out)
  end

  def test_03_kcode
    out, err = erb_cmd("", '<%= $KCODE%>')
    assert_equal('NONE', out)

    out, err = erb_cmd("-Kn", '<%= $KCODE%>')
    assert_equal('NONE', out)

    out, err = erb_cmd("-K n", '<%= $KCODE%>')
    assert_equal('NONE', out)

    out, err = erb_cmd("-K none", '<%= $KCODE%>')
    assert_equal('NONE', out)

    out, err = erb_cmd("-Ke", '<%= $KCODE%>')
    assert_equal('EUC', out)

    out, err = erb_cmd("-K e", '<%= $KCODE%>')
    assert_equal('EUC', out)

    out, err = erb_cmd("-K EUC", '<%= $KCODE%>')
    assert_equal('EUC', out)

    out, err = erb_cmd("-Ks", '<%= $KCODE%>')
    assert_equal('SJIS', out)

    out, err = erb_cmd("-K s", '<%= $KCODE%>')
    assert_equal('SJIS', out)

    out, err = erb_cmd("-K sjis", '<%= $KCODE%>')
    assert_equal('SJIS', out)

    out, err = erb_cmd("-Ku", '<%= $KCODE%>')
    assert_equal('UTF8', out)

    out, err = erb_cmd("-K u", '<%= $KCODE%>')
    assert_equal('UTF8', out)

    out, err = erb_cmd("-K utf8", '<%= $KCODE%>')
    assert_equal('UTF8', out)
  end

  def test_04_d_v
    out, err = erb_cmd("", '<%=$DEBUG%>, <%=$VERBOSE%>')
    assert_equal('false, false', out)

    out, err = erb_cmd("-d", '<%=$DEBUG%>, <%=$VERBOSE%>')
    assert_equal('true, false', out)

    out, err = erb_cmd("-v", '<%=$DEBUG%>, <%=$VERBOSE%>')
    assert_equal('false, true', out)

    out, err = erb_cmd("-dv", '<%=$DEBUG%>, <%=$VERBOSE%>')
    assert_equal('true, true', out)

    out, err = erb_cmd("-d -v", '<%=$DEBUG%>, <%=$VERBOSE%>')
    assert_equal('true, true', out)
  end

  def test_05_x_n
    src = <<EOS
% a=1
hello
world
EOS
    ans = <<EOS
_erbout = '';  a=1; _erbout.concat "hello\\n"
_erbout.concat "world\\n"
_erbout; 
EOS

    out, err = erb_cmd("-x", src)
    assert_equal(ans, out)

    ans = <<EOS
  1 _erbout = '';  a=1; _erbout.concat "hello\\n"
  2 _erbout.concat "world\\n"
  3 _erbout; 
EOS

    out, err = erb_cmd("-xn", src)
    assert_equal(ans, out)

    out, err = erb_cmd("-x -n", src)
    assert_equal(ans, out)

    out, err = erb_cmd("-n -x", src)
    assert_equal(ans, out)

    ans = <<EOS
hello
world
EOS
    out, err = erb_cmd("-n", src)
    assert_equal(ans, out)
  end
end

if __FILE__ == $0
  RUNIT::CUI::TestRunner.run(ERBCmdTest.suite)
end
