# -*- coding: utf-8 -*-

# Copyright (c) 2002 - 2005 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing the project management functionality.
"""

import os
import re
import time
import shutil
import glob

from qt import *

from KdeQt import KQFileDialog, KQMessageBox, KQInputDialog

from AddLanguageDialog import AddLanguageDialog
from AddFileDialog import AddFileDialog
from AddDirectoryDialog import AddDirectoryDialog
from PropertiesDialog import PropertiesDialog
from AddFoundFilesDialog import AddFoundFilesDialog

from XML.XMLUtilities import make_parser
from XML.XMLErrorHandler import XMLErrorHandler, XMLFatalParseError
from XML.ProjectHandler import ProjectHandler
from XML.ProjectWriter import ProjectWriter
from XML.SessionHandler import SessionHandler
from XML.SessionWriter import SessionWriter
from XML.XMLEntityResolver import XMLEntityResolver

import VCS
from VCS.Config import ConfigVcsSystemIndicators, ConfigVcsSystems, \
                       ConfigVcsSystemsDisplay
from VCS.CommandOptionsDialog import vcsCommandOptionsDialog
from VCS.LoginDialog import vcsLoginDialog
from VCS.LogoutDialog import vcsLogoutDialog

from DocumentationTools import DocumentationTools

from Checks.TabnannyDialog import TabnannyDialog
from Checks.SyntaxCheckerDialog import SyntaxCheckerDialog

from Graphics.ApplicationDiagram import ApplicationDiagram

from UI.CodeMetricsDialog import CodeMetricsDialog
from UI.PyCoverageDialog import PyCoverageDialog
from UI.PyProfileDialog import PyProfileDialog
import UI.PixmapCache

from UI.E3Action import E3Action, E3ActionGroup

import Preferences
import Utilities

class Project(QObject):
    """
    Module implementing the project management functionality.
    
    @signal dirty(int) emitted when the dirty state changes
    @signal projectSessionLoaded() emitted after a project session file was loaded
    @signal projectLanguageAdded(string) emitted after a new language was added
    @signal projectFormAdded(string) emitted after a new form was added
    @signal projectSourceAdded(string) emitted after a new source file was added
    @signal projectInterfaceAdded(string) emitted after a new IDL file was added
    @signal newProject() emitted after a new project was generated
    @signal pythonFile(string) emitted after a project file was read to open the main script
    @signal projectOpened() emitted after a project file was read
    @signal projectClosed() emitted after a project was closed
    @signal projectOthersAdded(string) emitted after a file or directory was added
            to the OTHERS project data area
    @signal projectFileRenamed(string, string) emitted after a file of the project
            has been renamed
    """
    keynames = [
        "DESCRIPTION", "VERSION",
        "AUTHOR", "EMAIL",
        "SOURCES", "FORMS",
        "TRANSLATIONS", "MAINSCRIPT",
        "VCS", "VCSOPTIONS", "VCSOTHERDATA",
        "ERIC3APIPARMS", "ERIC3DOCPARMS", "HAPPYDOCPARMS", 
        "OTHERS", "INTERFACES"
    ]
    
    def __init__(self, parent = None, filename = None):
        """
        Constructor
        
        @param parent parent widget (usually the ui object) (QWidget)
        @param filename optional filename of a project file to open (string)
        """
        QObject.__init__(self, parent)
        
        self.init()
        
        self.recent = QStringList()
        rp, ok = Preferences.Prefs.settings.readListEntry('/eric3/Recent/Projects')
        if ok:
            self.recent = rp
        
        if filename is not None:
            self.readProject(filename)
            
        self.vcs = self.initVCS()
        self.doctools = DocumentationTools(self, self.parent())
        
    def init(self):
        """
        Private method to initialize the project data part.
        """
        self.loaded = 0
        self.dirty = 0
        self.pfile = ""
        self.ppath = ""
        self.translationsRoot = ""
        self.name = ""
        self.opened = 0
        self.subdirs = [""] # record the project dir as a relative path (i.e. empty path)
        self.otherssubdirs = []
        self.vcs = None
        self.dbgCmdline = ''
        self.dbgWd = ''
        self.dbgEnv = ''
        self.dbgReportExceptions = 1
        self.dbgExcList = []
        self.dbgTracePython = 0
        self.dbgCovexcPattern = ''
        
        self.pdata = {}
        for key in self.__class__.keynames:
            self.pdata[key] = []
        self.pdata["AUTHOR"] = ['']
        self.pdata["EMAIL"] = ['']
    
    def setDirty(self, b):
        """
        Private method to set the dirty state.
        
        It emits the signal dirty(int).
        
        @param b dirty state (boolean)
        """
        self.dirty = b
        self.saveAct.setEnabled(b)
        self.emit(PYSIGNAL("dirty"), (b,))
        
    def isDirty(self):
        """
        Public method to return the dirty state.
        
        @return dirty state (boolean)
        """
        return self.dirty
        
    def isOpen(self):
        """
        Public method to return the opened state.
        
        @return open state (boolean)
        """
        return self.opened
        
    def checkFilesExist(self, index):
        """
        Private method to check, if the files in a list exist. 
        
        The files in the indicated list are checked for existance in the
        filesystem. Non existant files are removed from the list and the
        dirty state of the project is changed accordingly.
        
        @param index key of the list to be checked (string)
        """
        removed = 0
        removelist = []
        for file in self.pdata[index]:
            if not os.path.exists(os.path.join(self.ppath, file)):
                removelist.append(file)
                removed = 1
                
        if removed:
            for file in removelist:
                self.pdata[index].remove(file)
            self.setDirty(1)
        
    def readProject(self, fn):
        """
        Public method to read in a project (.e3p, .e3pz) file.
        
        @param fn filename of the project file to be read (string or QString)
        @return flag indicating success
        """
        fn = unicode(fn)
        try:
            if fn.lower().endswith("e3pz"):
                try:
                    import gzip
                except ImportError:
                    KQMessageBox.critical(None,
                        self.trUtf8("Read project file"),
                        self.trUtf8("""Compressed project files not supported. The compression library is missing."""),
                        self.trUtf8("&Abort"),
                        None,
                        None,
                        0, -1)
                    return 0
                f = gzip.open(fn, "rb")
            else:
                f = open(fn, "rb")
            line = f.readline()
            dtdLine = f.readline()
            f.close()
        except:
            KQMessageBox.critical(None,
                self.trUtf8("Read project file"),
                self.trUtf8("<p>The project file <b>%1</b> could not be read.</p>").arg(fn),
                self.trUtf8("&Abort"))
            return 0
            
        self.pfile = os.path.abspath(fn)
        self.ppath = os.path.abspath(os.path.dirname(fn))
        
        # insert filename into list of recently opened projects
        self.recent.remove(self.pfile)
        self.recent.prepend(self.pfile)
        if len(self.recent) > 9:
            self.recent = self.recent[:9]
        
        # now read the file
        if line.startswith('<?xml'):
            res = self.readXMLProject(fn, dtdLine.startswith("<!DOCTYPE"))
        else:
            KQMessageBox.critical(None,
                self.trUtf8("Read project file"),
                self.trUtf8("<p>The project file <b>%1</b> has an unsupported format.</p>").arg(fn),
                self.trUtf8("&Abort"))
            return 0
            
        if res:
            if len(self.pdata["MAINSCRIPT"]) == 1:
                (self.translationsRoot, dummy) = os.path.splitext(self.pdata["MAINSCRIPT"][0])
                
            (self.name, dummy) = os.path.splitext(os.path.basename(fn))
            
            # get the names of subdirectories the files are stored
            for fn in self.pdata["SOURCES"] + self.pdata["FORMS"] + self.pdata["INTERFACES"]:
                dn = os.path.dirname(fn)
                if dn not in self.subdirs:
                    self.subdirs.append(dn)
                    
            # get the names of other subdirectories
            for fn in self.pdata["OTHERS"]:
                dn = os.path.dirname(fn)
                if dn not in self.otherssubdirs:
                    self.otherssubdirs.append(dn)
                
            # check, if the files of the project still exist in the project directory
            self.checkFilesExist("SOURCES")
            self.checkFilesExist("FORMS")
            self.checkFilesExist("INTERFACES")
            
        return res

    def readXMLProject(self, fn, validating):
        """
        Public method to read the project data from an XML file.
        
        @param fn filename of the project file to be read (string or QString)
        @param validating flag indicating a validation of the XML file is
            requested (boolean)
        @return flag indicating success
        """
        fn = unicode(fn)
        
        parser = make_parser(validating)
        handler = ProjectHandler(self)
        er = XMLEntityResolver()
        eh = XMLErrorHandler()
        
        parser.setContentHandler(handler)
        parser.setEntityResolver(er)
        parser.setErrorHandler(eh)
        
        try:
            if fn.lower().endswith("e3pz"):
                try:
                    import gzip
                except ImportError:
                    KQMessageBox.critical(None,
                        self.trUtf8("Read project file"),
                        self.trUtf8("""Compressed project files not supported. The compression library is missing."""),
                        self.trUtf8("&Abort"),
                        None,
                        None,
                        0, -1)
                    return 0
                f = gzip.open(fn, "rb")
            else:
                f = open(fn, "rb")
            try:
                parser.parse(f)
            finally:
                f.close()
        except IOError:
            qApp.restoreOverrideCursor()
            KQMessageBox.critical(None,
                self.trUtf8("Read project file"),
                self.trUtf8("<p>The project file <b>%1</b> could not be read.</p>").arg(fn),
                self.trUtf8("&Abort"))
            return 0
            
        except XMLFatalParseError:
            qApp.restoreOverrideCursor()
            KQMessageBox.critical(None,
                self.trUtf8("Read project file"),
                self.trUtf8("<p>The project file <b>%1</b> has invalid contents.</p>").arg(fn),
                self.trUtf8("&Abort"))
            eh.showParseMessages()
            return 0
            
        qApp.restoreOverrideCursor()
        eh.showParseMessages()
        qApp.setOverrideCursor(Qt.waitCursor)
        qApp.processEvents()
        return 1
        
    def writeProject(self, fn = None):
        """
        Public method to save the project infos to a project file.
        
        @param fn optional filename of the project file to be written.
                If fn is None, the filename stored in the project object
                is used. This is the 'save' action. If fn is given, this filename
                is used instead of the one in the project object. This is the
                'save as' action.
        @return flag indicating success
        """
        if self.vcs is not None:
            self.pdata["VCSOPTIONS"] = [str(self.vcs.vcsGetOptions())]
            self.pdata["VCSOTHERDATA"] = [str(self.vcs.vcsGetOtherData())]
            
        if fn is None:
            fn = self.pfile
            
        res = self.writeXMLProject(fn)
            
        if res:
            self.pfile = os.path.abspath(fn)
            self.ppath = os.path.abspath(os.path.dirname(fn))
            (self.name, dummy) = os.path.splitext(os.path.basename(fn))
            self.setDirty(0)
            
            # insert filename into list of recently opened projects
            self.recent.remove(self.pfile)
            self.recent.prepend(self.pfile)
            if len(self.recent) > 9:
                self.recent = self.recent[:9]
                
        return res
        
    def writeXMLProject(self, fn = None):
        """
        Public method to write the project data to an XML file.
        
        @param fn the filename of the project file (string)
        """
        try:
            if fn.lower().endswith("e3pz"):
                try:
                    import gzip
                except ImportError:
                    KQMessageBox.critical(None,
                        self.trUtf8("Save project file"),
                        self.trUtf8("""Compressed project files not supported. The compression library is missing."""),
                        self.trUtf8("&Abort"),
                        None,
                        None,
                        0, -1)
                    return 0
                f = gzip.open(fn, "wb")
            else:
                f = open(fn, "wb")
            
            ProjectWriter(f, os.path.splitext(os.path.basename(fn))[0]).writeXML()
            
            f.close()
            
        except IOError:
            KQMessageBox.critical(None,
                self.trUtf8("Save project file"),
                self.trUtf8("<p>The project file <b>%1</b> could not be written.</p>").arg(fn),
                self.trUtf8("&Abort"))
            return 0
        
        return 1
        
    def readSession(self, quiet=0):
        """
        Private method to read in the project session file (.e3s)
        
        @param quiet flag indicating quiet operations.
                If this flag is true, no errors are reported.
        """
        if self.pfile is None:
            if not quiet:
                KQMessageBox.critical(None,
                    self.trUtf8("Read project session"),
                    self.trUtf8("Please save the project first."),
                    self.trUtf8("&Abort"))
            return
            
        fn, ext = os.path.splitext(self.pfile)
        
        try:
            if ext.lower() == ".e3pz":
                fn = '%s.e3sz' % fn
                try:
                    import gzip
                except ImportError:
                    if not quiet:
                        KQMessageBox.critical(None,
                            self.trUtf8("Read project session"),
                            self.trUtf8("""Compressed project session files not supported. The compression library is missing."""),
                            self.trUtf8("&Abort"),
                            None,
                            None,
                            0, -1)
                    return
                f = gzip.open(fn, "rb")
            else:
                fn = '%s.e3s' % fn
                f = open(fn, "rb")
            line = f.readline()
            dtdLine = f.readline()
            f.close()
        except:
            if not quiet:
                KQMessageBox.critical(None,
                    self.trUtf8("Read project session"),
                    self.trUtf8("<p>The project session <b>%1</b> could not be read.</p>").arg(fn),
                    self.trUtf8("&Abort"))
            return
            
        # now read the file
        if line.startswith('<?xml'):
            res = self.readXMLSession(fn, dtdLine.startswith("<!DOCTYPE"), quiet)
        else:
            if not quiet:
                KQMessageBox.critical(None,
                    self.trUtf8("Read project session"),
                    self.trUtf8("<p>The project session <b>%1</b> has an unsupported format.</p>").arg(fn),
                    self.trUtf8("&Abort"))
    
    def readXMLSession(self, fn, validating, quiet):
        """
        Public method to read the project data from an XML file.
        
        The data read is:
            <ul>
            <li>all open source filenames</li>
            <li>the active window</li>
            <li>all breakpoints</li>
            <li>the commandline</li>
            <li>the working directory</li>
            <li>the exception reporting flag</li>
            <li>the list of exception types to be highlighted</li>
            <li>all bookmarks</li>
            </ul>
            
        @param fn filename of the project session file to be read (string or QString)
        @param validating flag indicating a validation of the XML file is
            requested (boolean)
        @param quiet flag indicating quiet operations.
            If this flag is true, no errors are reported.
        """
        parser = make_parser(validating)
        handler = SessionHandler(self)
        er = XMLEntityResolver()
        eh = XMLErrorHandler()
        
        parser.setContentHandler(handler)
        parser.setEntityResolver(er)
        parser.setErrorHandler(eh)
        
        try:
            if fn.lower().endswith("e3sz"):
                try:
                    import gzip
                except ImportError:
                    if not quiet:
                        KQMessageBox.critical(None,
                            self.trUtf8("Read project session"),
                            self.trUtf8("<p>The project session <b>%1</b> could not be read.</p>").arg(fn),
                            self.trUtf8("&Abort"))
                    return
                f = gzip.open(fn, "rb")
            else:
                f = open(fn, "rb")
            try:
                parser.parse(f)
            finally:
                f.close()
        except IOError:
            if not quiet:
                KQMessageBox.critical(None,
                    self.trUtf8("Read project session"),
                    self.trUtf8("<p>The project session file <b>%1</b> could not be read.</p>")
                        .arg(fn),
                    self.trUtf8("&Abort"))
        except XMLFatalParseError:
            pass
            
        eh.showParseMessages()
        
    def writeSession(self, quiet=0):
        """
        Public method to write the session data to an XML file (.e3s).
        
        The data saved is:
            <ul>
            <li>all open source filenames belonging to the project</li>
            <li>the active window, if it belongs to the project</li>
            <li>all breakpoints</li>
            <li>the commandline</li>
            <li>the working directory</li>
            <li>the exception reporting flag</li>
            <li>the list of exception types to be highlighted</li>
            <li>all bookmarks of files belonging to the project</li>
            </ul>
        
        @param quiet flag indicating quiet operations.
                If this flag is true, no errors are reported.
        """
        if self.pfile is None:
            if not quiet:
                KQMessageBox.critical(None,
                    self.trUtf8("Save project session"),
                    self.trUtf8("Please save the project first."),
                    self.trUtf8("&Abort"))
            return
            
        fn, ext = os.path.splitext(self.pfile)
        
        try:
            if ext.lower() == ".e3pz":
                fn = '%s.e3sz' % fn
                try:
                    import gzip
                except ImportError:
                    if not quiet:
                        KQMessageBox.critical(None,
                            self.trUtf8("Save project session"),
                            self.trUtf8("""Compressed project session files not supported. The compression library is missing."""),
                            self.trUtf8("&Abort"),
                            None,
                            None,
                            0, -1)
                    return
                f = gzip.open(fn, "wb")
            else:
                fn = '%s.e3s' % fn
                f = open(fn, "wb")
            
            SessionWriter(f, os.path.splitext(os.path.basename(fn))[0]).writeXML()
            
            f.close()
            
        except IOError:
            if not quiet:
                KQMessageBox.critical(None,
                    self.trUtf8("Save project session"),
                    self.trUtf8("<p>The project session file <b>%1</b> could not be written.</p>")
                        .arg(fn),
                    self.trUtf8("&Abort"))
        
    def deleteSession(self):
        """
        Public method to delete the session file.
        """
        if self.pfile is None:
            if not quiet:
                KQMessageBox.critical(None,
                    self.trUtf8("Save project session"),
                    self.trUtf8("Please save the project first."),
                    self.trUtf8("&Abort"))
            return
            
        fname, ext = os.path.splitext(self.pfile)
        
        for fn in ["%s.e3sz" % fname, "%s.e3s" % fname]:
            if os.path.exists(fn):
                try:
                    os.remove(fn)
                except OSError:
                    KQMessageBox.critical(None,
                        self.trUtf8("Delete project session"),
                        self.trUtf8("<p>The project session file <b>%1</b> could not be deleted.</p>")
                            .arg(fn),
                        self.trUtf8("&Abort"))
        
    def setDbgInfo(self, argv, wd, env, excReporting, excList, tracePython=None,
                   covexcPattern=None):
        """
        Public method to set the debugging information.
        
        @param argv command line arguments to be used (string or QString)
        @param wd working directory (string or QString)
        @param env environment setting (string or QString)
        @param excReporting flag indicating the highlighting of exceptions
        @param excList list of exceptions to be highlighted (list of string)
        @keyparam tracePython flag to indicate if the Python library should be
            traced as well (boolean)
        @keyparam covexcPattern regexp pattern to exclude lines from code coverage
            (string or QString)
        """
        self.dbgCmdline = str(argv)
        self.dbgWd = unicode(wd)
        self.dbgEnv = env
        self.dbgReportExceptions = excReporting
        self.dbgExcList = excList[:]    # keep a copy of the list
        if tracePython is not None:
            self.dbgTracePython = tracePython
        if covexcPattern is not None:
            self.dbgCovexcPattern = covexcPattern
    
    def addLanguage(self):
        """
        Public slot used to add a language to the project.
        """
        if self.translationsRoot == '':
            KQMessageBox.critical(None,
                self.trUtf8("Add Language"),
                self.trUtf8("You have to specify the mainscript of the project first."),
                self.trUtf8("&OK"))
            return
                
        dlg = AddLanguageDialog(self.parent())
        if dlg.exec_loop() == QDialog.Accepted:
            lang = unicode(dlg.getSelectedLanguage())
            langFile = '%s_%s.ts' % (self.translationsRoot, lang)
            self.pdata["TRANSLATIONS"].append(langFile)
            self.emit(PYSIGNAL('projectLanguageAdded'), (langFile,))
            self.setDirty(1)
            
    def removeLanguage(self, lang):
        """
        Public slot to remove a translation from the project.
        
        The translation file is not deleted from the project directory.
        
        @param lang the language to be removed (string)
        @return filename of the translation file (string)
        """
        langFile = '%s_%s.ts' % (self.translationsRoot, lang)
        self.pdata["TRANSLATIONS"].remove(langFile)
        self.setDirty(1)
        return self.ppath+os.sep+langFile
        
    def deleteLanguage(self, lang):
        """
        Public slot to delete a translation from the project directory.
        
        @param lang the language to be removed (string)
        @return filename of the translation file. The return value
            is an empty string in case of an error (string)
        """
        langFile = '%s_%s.ts' % (self.translationsRoot, lang)
        try:
            fn = os.path.join(self.ppath, langFile)
            if os.path.exists(fn):
                os.remove(fn)
        except:
            KQMessageBox.critical(None,
                self.trUtf8("Delete translation"),
                self.trUtf8("<p>The selected translation file <b>%1</b> could not be deleted.</p>")
                    .arg(langFile),
                self.trUft8("&OK"))
            return ""
        
        self.removeLanguage(lang)
        
        # now get rid of the .qm file
        langFile2 = '%s_%s.qm' % (self.translationsRoot, lang)
        try:
            fn = os.path.join(self.ppath, langFile2)
            if os.path.exists(fn):
                os.remove(fn)
        except:
            pass    # never mind
            
        return self.ppath+os.sep+langFile
         
    def appendFile(self, fn, isPythonFile=0):
        """
        Public method to append a file to the project.
        
        @param fn filename to be added to the project (string or QString)
        @param isPythonFile flag indicating that this is a Python file
                even if it doesn't have the .py extension (boolean)
        """
        newfn = os.path.abspath(unicode(fn))
        newfn = newfn.replace(self.ppath+os.sep, '')
        newdir = os.path.dirname(newfn)
        if newfn.endswith('.ui.h'):
            ext = '.ui.h'
        else:
            (dummy, ext) = os.path.splitext(newfn)
        dirty = 0
        isPythonFile = isPythonFile or ext in ['.py', '.ptl']
        if ext in ['.ui', '.ui.h', '.idl'] or isPythonFile:
            if ext in ['.ui', '.ui.h']:
                if newfn not in self.pdata["FORMS"]:
                    self.pdata["FORMS"].append(newfn)
                    self.emit(PYSIGNAL('projectFormAdded'), (newfn,))
                    dirty = 1
            elif isPythonFile:
                if newfn not in self.pdata["SOURCES"]:
                    self.pdata["SOURCES"].append(newfn)
                    self.emit(PYSIGNAL('projectSourceAdded'), (newfn,))
                    dirty = 1
            elif ext == '.idl':
                if newfn not in self.pdata["INTERFACES"]:
                    self.pdata["INTERFACES"].append(newfn)
                    self.emit(PYSIGNAL('projectInterfaceAdded'), (newfn,))
                    dirty = 1
            if newdir not in self.subdirs:
                self.subdirs.append(newdir)
        else:
            if newfn not in self.pdata["OTHERS"]:
                self.pdata['OTHERS'].append(newfn)
                self.othersAdded(newfn)
                dirty = 1
            if newdir not in self.otherssubdirs:
                self.otherssubdirs.append(newdir)
            
        if dirty:
            self.setDirty(1)
        
    def addFile(self, filter = None):
        """
        Public slot used to add a file to the project.
        
        @param filter filter to be used by the add file dialog (string)
        """
        dlg = AddFileDialog(self, self.parent(), filter)
        if dlg.exec_loop() == QDialog.Accepted:
            (fn, target, isSource) = dlg.getData()
            (dummy, ext) = os.path.splitext(fn)
            if target != '':
                targetfile = os.path.join(target, os.path.basename(fn))
                if not Utilities.samepath(os.path.dirname(fn), target):
                    try:
                        if not os.path.isdir(target):
                            os.makedirs(target)
                                
                        if os.path.exists(targetfile):
                            res = KQMessageBox.warning(None,
                                self.trUtf8("Add file"),
                                self.trUtf8("<p>The file <b>%1</b> already exists.</p>")
                                    .arg(targetfile),
                                self.trUtf8("&Cancel"),
                                self.trUtf8("&Overwrite"))
                            if res == 0:
                                return  # operation was canceled
                                
                        shutil.copy(fn, target)
                        if ext == '.ui' and os.path.isfile(fn+'.h'):
                            shutil.copy(fn+'.h', target)
                    except IOError, why:
                        KQMessageBox.critical(None,
                            self.trUtf8("Add file"),
                            self.trUtf8("<p>The selected file <b>%1</b> could not be added to <b>%2</b>.</p>")
                                .arg(fn)
                                .arg(target),
                            self.trUtf8("&Abort"))
                        return
                        
                self.appendFile(targetfile, isSource or filter == 'py')
                if ext == '.ui' and os.path.isfile(targetfile + '.h'):
                    self.appendFile(targetfile + '.h')
            else:
                KQMessageBox.critical(None,
                    self.trUtf8("Add file"),
                    self.trUtf8("The target directory must not be empty."),
                    self.trUtf8("&Abort"))
        
    def addSingleDirectory(self, filter, source, target, quiet=0):
        """
        Private method used to add all files of a single directory to the project.
        
        @param filter filter to be applied (string)
        @param source source directory (string)
        @param target target directory (string)
        @param quiet flag indicating quiet operations (boolean)
        """
        sstring = "%s%s*%s%s" % (source, os.sep, os.extsep, filter)
        files = glob.glob(sstring)
        if len(files) == 0:
            if not quiet:
                KQMessageBox.information(None,
                    self.trUtf8("Add directory"),
                    self.trUtf8("<p>The source directory doesn't contain"
                        " any files having the extension <b>%1</b>.</p>")
                        .arg(filter),
                    self.trUtf8("&OK"))
            return
        
        if not Utilities.samepath(target, source) and not os.path.isdir(target):
            try:
                os.makedirs(target)
            except IOError, why:
                KQMessageBox.critical(None,
                    self.trUtf8("Add directory"),
                    self.trUtf8("<p>The target directory <b>%1</b> could not be created.</p>")
                        .arg(normtarget),
                    self.trUtf8("&Abort"))
                return
                
        for file in files:
            targetfile = os.path.join(target, os.path.basename(file))
            if not Utilities.samepath(target, source):
                try:
                    if os.path.exists(targetfile):
                        res = KQMessageBox.warning(None,
                            self.trUtf8("Add directory"),
                            self.trUtf8("<p>The file <b>%1</b> already exists.</p>")
                                .arg(targetfile),
                            self.trUtf8("&Cancel"),
                            self.trUtf8("&Overwrite"))
                        if res == 0:
                            continue  # operation was canceled, carry on with next file
                            
                    shutil.copy(file, target)
                    if filter == 'ui' and os.path.isfile(file+os.extsep+'h'):
                        shutil.copy(file+os.extsep+'h', target)
                except:
                    continue
            self.appendFile(targetfile)
            if filter == 'ui' and os.path.isfile(targetfile+os.extsep+'h'):
                self.appendFile(targetfile+os.extsep+'h')
        
    def addRecursiveDirectory(self, filter, source, target):
        """
        Private method used to add all files of a directory tree.
        
        The tree is rooted at source to another one rooted at target. This
        method decents down to the lowest subdirectory.
        
        @param filter filter to be applied (string)
        @param source source directory (string)
        @param target target directory (string)
        """
        # first perform the addition of source
        self.addSingleDirectory(filter, source, target, 1)
        
        # now recurse into subdirectories
        for name in os.listdir(source):
            ns = os.path.join(source, name)
            if os.path.isdir(ns):
                nt = os.path.join(target, name)
                self.addRecursiveDirectory(filter, ns, nt)
        
    def addDirectory(self, filter=None):
        """
        Private method used to add all files of a directory to the project.
        
        @param filter filter to be applied (string)
        """
        dlg = AddDirectoryDialog(self, filter, self.parent())
        if dlg.exec_loop() == QDialog.Accepted:
            filter, source, target, recursive = dlg.getData()
            if target == '':
                KQMessageBox.critical(None,
                    self.trUtf8("Add directory"),
                    self.trUtf8("The target directory must not be empty."),
                    self.trUtf8("&Abort"))
                return
                
            if filter == '':
                self.addToOthers(QString(source))
                return
                
            if source == '':
                KQMessageBox.critical(None,
                    self.trUtf8("Add directory"),
                    self.trUtf8("The source directory must not be empty."),
                    self.trUtf8("&Abort"))
                return
                
            if recursive:
                self.addRecursiveDirectory(filter, source, target)
            else:
                self.addSingleDirectory(filter, source, target)
        
    def addToOthers(self, fn):
        """
        Private method to add file/directory to the OTHERS project data.
        
        @param fn filename or directoryname to add
        """
        if not fn.isEmpty():
            fn = unicode(fn)
            
            # if it is below the project directory, make it relative to that
            fn = fn.replace(self.ppath+os.sep, '')
            
            # if it ends with the directory separator character, remove it
            if fn.endswith(os.sep):
                fn = fn[:-1]
            if fn not in self.pdata["OTHERS"]:
                self.pdata['OTHERS'].append(fn)
                self.othersAdded(fn)
                self.setDirty(1)
            if os.path.isdir(fn) and fn not in self.otherssubdirs:
                self.otherssubdirs.append(fn)
        
    def addPyFile(self):
        """
        Public slot to add a python file to the current project.
        """
        self.addFile('py')
        
    def addUiFile(self):
        """
        Public slot to add a form to the current project.
        """
        self.addFile('ui')
        
    def addIdlFile(self):
        """
        Public slot to add an IDL interface to the current project.
        """
        self.addFile('idl')
        
    def addOthersFile(self):
        """
        Private slot to add a file to the OTHERS project data.
        """
        self.addFile('others')
        
    def addPyDir(self):
        """
        Public slot to add all python files of a directory to the current project.
        """
        self.addDirectory('py')
        
    def addUiDir(self):
        """
        Public slot to add all forms of a directory to the current project.
        """
        self.addDirectory('ui')
        
    def addIdlDir(self):
        """
        Public slot to add all IDL interfaces of a directory to the current project.
        """
        self.addDirectory('idl')
        
    def addOthersDir(self):
        """
        Private slot to add a directory to the OTHERS project data.
        """
        self.addDirectory('others')
        
    def renameFile(self, oldfn, newfn=None):
        """
        Public slot to rename a file of the project.
        
        @param oldfn old filename of the file (string)
        @param newfn new filename of the file (string)
        @return flag indicating success
        """
        fn2 = unicode(oldfn)
        fn = fn2.replace(self.ppath+os.sep, '')
        isSourceFile = fn in self.pdata["SOURCES"]
        
        if newfn is None:
            newfn = KQFileDialog.getSaveFileName(\
                os.path.dirname(oldfn),
                None,
                None, None,
                self.trUtf8("Rename file"),
                None, 1)
            if newfn.isEmpty():
                return 0
            
        newfn = unicode(newfn)
        if os.path.exists(newfn):
            canceled = KQMessageBox.warning(None,
                self.trUtf8("Rename File"),
                self.trUtf8("""<p>The file <b>%1</b> already exists. Overwrite it?</p>""")
                    .arg(newfn),
                self.trUtf8("&Yes"),
                self.trUtf8("&No"),
                None,
                1, -1)
            if canceled:
                return 0
        
        try:
            os.rename(oldfn, newfn)
        except OSError, msg:
            KQMessageBox.critical(None,
                self.trUtf8("Rename File"),
                self.trUtf8("""<p>The file <b>%1</b> could not be renamed.<br />"""
                    """Reason: %s</p>""").arg(oldfn).arg(msg),
                self.trUtf8("&OK"),
                None,
                None,
                0, -1)
            return 0

        if fn in self.pdata["SOURCES"] or \
           fn in self.pdata["FORMS"] or \
           fn in self.pdata["TRANSLATIONS"] or \
           fn in self.pdata["INTERFACES"] or \
           fn in self.pdata["OTHERS"]:
            self.removeFile(oldfn)
            self.appendFile(newfn, isSourceFile)
            self.emit(PYSIGNAL('projectFileRenamed'), (oldfn, newfn))
        
        return 1
        
    def removeFile(self, fn):
        """
        Public slot to remove a file from the project.
        
        The file is not deleted from the project directory.
        
        @param fn filename to be removed from the project
        """
        fn2 = unicode(fn)
        fn = fn2.replace(self.ppath+os.sep, '')
        if fn in self.pdata["SOURCES"]:
            self.pdata["SOURCES"].remove(fn)
        elif fn in self.pdata["FORMS"]:
            self.pdata["FORMS"].remove(fn)
        elif fn in self.pdata["INTERFACES"]:
            self.pdata["INTERFACES"].remove(fn)
        elif fn in self.pdata["OTHERS"]:
            self.pdata["OTHERS"].remove(fn)
        self.setDirty(1)
            
    def deleteFile(self, fn):
        """
        Public slot to delete a file from the project directory.
        
        @param fn filename to be deleted from the project
        @return flag indicating success
        """
        fn = unicode(fn)
        try:
            os.remove(os.path.join(self.ppath, fn))
            dummy, ext = os.path.splitext(fn)
            if ext == '.ui':
                fn2 = os.path.join(self.ppath, '%s.h' % fn)
                if os.path.isfile(fn2):
                    os.remove(fn2)
        except:
            KQMessageBox.critical(None,
                self.trUtf8("Delete file"),
                self.trUtf8("<p>The selected file <b>%1</b> could not be deleted.</p>")
                    .arg(fn),
                self.trUtf8("&OK"))
            return 0
        
        self.removeFile(fn)
        if ext == '.ui':
            self.removeFile(fn + '.h')
        return 1
        
    def newProject(self):
        """
        Public slot to built a new project.
        
        This method displays the new project dialog and initializes
        the project object with the data entered.
        """
        if not self.checkDirty():
            return
            
        dlg = PropertiesDialog(self, 1)
        if dlg.exec_loop() == QDialog.Accepted:
            self.closeProject()
            dlg.storeData()
            self.pdata["VCS"] = ['None']
            self.opened = 1
            self.setDirty(1)
            self.closeAct.setEnabled(1)
            self.saveasAct.setEnabled(1)
            self.actGrp2.setEnabled(1)
            self.propsAct.setEnabled(1)
            self.vcsAddAct.setEnabled(1)
            self.sessActGrp.setEnabled(0)
            self.menu.setItemEnabled(self.menuCheckId, 1)
            self.menu.setItemEnabled(self.menuShowId, 1)
            self.menu.setItemEnabled(self.menuDiagramID, 1)
            try:
                self.menu.setItemEnabled(self.menuApidocId, 1)
            except:
                pass
        
            # create the project directory if it doesn't exist already
            if not os.path.isdir(self.ppath):
                try:
                    os.makedirs(self.ppath)
                except:
                    KQMessageBox.critical(None,
                        self.trUtf8("Create project directory"),
                        self.trUtf8("<p>The project directory <b>%1</b> could not be created.</p>")
                            .arg(self.ppath),
                        self.trUtf8("&OK"))
                    self.vcs = None
                    return
                self.saveProject()
            else:
                try:
                    ms = os.path.join(self.ppath, self.pdata["MAINSCRIPT"][0])
                    if os.path.exists(ms):
                        self.appendFile(ms)
                except IndexError:
                    pass
                
                # add existing files to the project
                res = KQMessageBox.question(None,
                    self.trUtf8("New Project"),
                    self.trUtf8("""Add existing files to the project?"""),
                    self.trUtf8("&Yes"),
                    self.trUtf8("&No"),
                    None,
                    0, -1)
                if res == 0:
                    # search the project directory for files with extensions
                    # of 'py', 'ptl', 'idl', 'ui' and 'ts'
                    for filespec in ['*.py', '*.ptl', '*.idl', '*.ui', '*.ui.h']:
                        files = Utilities.direntries(self.ppath, 1, filespec)
                        for file in files:
                            self.appendFile(file)
                            
                    # special handling for translation files (*.ts)
                    d = QDir(self.ppath)
                    tslist = d.entryList('*.ts')
                    if len(tslist) and tslist[0].contains('_'):
                        # the first entry determines the filename pattern
                        mainscriptname = tslist[0].section('_', 0, 0)
                        for ts in tslist:
                            if ts.startsWith(mainscriptname):
                                self.pdata["TRANSLATIONS"].append(str(ts))
                                self.emit(PYSIGNAL('projectLanguageAdded'), (str(ts),))
                        self.pdata["MAINSCRIPT"] = ['%s.py' % unicode(mainscriptname)]
                    self.setDirty(1)
                self.saveProject()
                
                # check, if the existing project directory is already under
                # VCS control
                for vcsSystem, indicator in ConfigVcsSystemIndicators:
                    if os.path.exists(os.path.join(self.ppath, indicator)):
                        self.pdata["VCS"] = [vcsSystem]
                        self.vcs = self.initVCS()
                        self.setDirty(1)
                        if self.vcs is not None:
                            # edit VCS command options
                            vcores = KQMessageBox.question(None,
                                self.trUtf8("New Project"),
                                self.trUtf8("""Would you like to edit the VCS command options?"""),
                                self.trUtf8("&Yes"),
                                self.trUtf8("&No"),
                                None,
                                1, -1)
                            if vcores == 0:
                                codlg = vcsCommandOptionsDialog(self.vcs)
                                if codlg.exec_loop() == QDialog.Accepted:
                                    self.vcs.vcsSetOptions(codlg.getOptions())
                            # add project file to repository
                            if res == 0:
                                apres = KQMessageBox.question(None,
                                    self.trUtf8("New project from repository"),
                                    self.trUtf8("Shall the project file be added to the repository?"),
                                    self.trUtf8("&Yes"), self.trUtf8("&No"))
                                if apres == 0:
                                    self.saveProject()
                                    self.vcs.vcsAdd(self.pfile)
                        else:
                            self.pdata["VCS"] = ['None']
                        self.saveProject()
                        break
                
            # put the project under VCS control
            if self.vcs is None:
                vcsSelected, ok = KQInputDialog.getItem(\
                    self.trUtf8("New Project"),
                    self.trUtf8("Select version control system for the project"),
                    ConfigVcsSystemsDisplay,
                    0, 0)
                vcsSystem = ConfigVcsSystems[ConfigVcsSystemsDisplay.findIndex(vcsSelected)]
                if ok:
                    self.pdata["VCS"] = [vcsSystem]
                else:
                    self.pdata["VCS"] = ['None']
                self.vcs = self.initVCS()
                if self.vcs is not None:
                    vcsdlg = self.vcs.vcsOptionsDialog(self, self.name)
                    if vcsdlg.exec_loop() == QDialog.Accepted:
                        vcsDir, archive, vendor, message, release, standardlayout =\
                            vcsdlg.getData()
                    else:
                        self.pdata["VCS"] = ['None']
                        self.vcs = None
                self.setDirty(1)
                if self.vcs is not None:
                    # edit VCS command options
                    vcores = KQMessageBox.question(None,
                        self.trUtf8("New Project"),
                        self.trUtf8("""Would you like to edit the VCS command options?"""),
                        self.trUtf8("&Yes"),
                        self.trUtf8("&No"),
                        None,
                        1, -1)
                    if vcores == 0:
                        codlg = vcsCommandOptionsDialog(self.vcs)
                        if codlg.exec_loop() == QDialog.Accepted:
                            self.vcs.vcsSetOptions(codlg.getOptions())
                        
                    # create the project in the VCS
                    self.vcs.vcsInit(vcsDir, 1)
                    self.vcs.vcsSetData('standardLayout', standardlayout)
                    self.saveProject()
                    success = self.vcs.vcsImport(vcsDir, message, archive, vendor, 
                        release, self.ppath)
                    if not success:
                        KQMessageBox.critical(None,
                            self.trUtf8("Create project in repository"),
                            self.trUtf8("""The project could not be created in the repository."""
                                        """ Maybe the given repository doesn't exist or the"""
                                        """ repository server is down."""),
                            self.trUtf8("&OK"),
                            None,
                            None,
                            0, -1)
                    else:
                        cwdIsPpath = 0
                        if os.getcwd() == self.ppath:
                            os.chdir(os.path.dirname(self.ppath))
                            cwdIsPpath = 1
                        tmpProjectDir = "%s_tmp" % self.ppath
                        os.rename(self.ppath, tmpProjectDir)
                        os.makedirs(self.ppath)
                        self.vcs.vcsCheckout(vcsDir, archive, self.ppath)
                        if cwdIsPpath:
                            os.chdir(self.ppath)
                        vcsName = self.vcs.vcsName()
                        if vcsName == 'CVS':
                            self.vcs.vcsCommit(os.path.join(self.ppath, archive), message, 1)
                            pfn = os.path.join(self.ppath, archive, os.path.basename(self.pfile))
                        elif vcsName == 'Subversion':
                            self.vcs.vcsCommit(self.ppath, message, 1)
                            pfn = self.pfile
                        if not os.path.isfile(pfn):
                            pfn += "z"
                        if not os.path.isfile(pfn):
                            KQMessageBox.critical(None,
                                self.trUtf8("New project"),
                                self.trUtf8("""The project could not be checked out of the repository.<br />"""
                                            """Restoring the original contents."""),
                                self.trUtf8("&OK"),
                                None,
                                None,
                                0, -1)
                            if os.getcwd() == self.ppath:
                                os.chdir(os.path.dirname(self.ppath))
                                cwdIsPpath = 1
                            else:
                                cwdIsPpath = 0
                            shutil.rmtree(self.ppath, 1)
                            os.rename(tmpProjectDir, self.ppath)
                            self.pdata["VCS"] = ['None']
                            self.vcs = None
                            self.setDirty(1)
                            self.saveProject()
                            return
                        shutil.rmtree(tmpProjectDir, 1)
                        self.openProject(pfn)
            
            else:
                self.emit(PYSIGNAL('newProject'), ())
    
    def showProperties(self):
        """
        Public slot to display the properties dialog.
        """
        dlg = PropertiesDialog(self, 0)
        if dlg.exec_loop() == QDialog.Accepted:
            dlg.storeData()
            self.setDirty(1)
            try:
                ms = os.path.join(self.ppath, self.pdata["MAINSCRIPT"][0])
                if os.path.exists(ms):
                    self.appendFile(ms)
            except IndexError:
                pass
        
    def openProject(self, fn = None):
        """
        Public slot to open a project.
        
        @param fn optional filename of the project file to be read
        """
        if not self.checkDirty():
            return
            
        if fn is None:
            fn = KQFileDialog.getOpenFileName(None,
                self.trUtf8("Project Files (*.e3p *.e3pz)"),
                self.parent(), None, self.trUtf8("Open project"))
            
            if fn.isNull():
                fn = None
            else:
                fn = unicode(fn)
            
        qApp.processEvents()
                
        if fn is not None:
            qApp.setOverrideCursor(Qt.waitCursor)
            qApp.processEvents()
            self.closeProject()
            if self.readProject(fn):
                self.opened = 1
                self.vcs = self.initVCS()
                if self.vcs is not None and not self.vcs.vcsRegisteredState(self.ppath):
                    self.vcs = None
                self.closeAct.setEnabled(1)
                self.saveasAct.setEnabled(1)
                self.actGrp2.setEnabled(1)
                self.propsAct.setEnabled(1)
                self.vcsAddAct.setEnabled(1)
                self.sessActGrp.setEnabled(1)
                self.menu.setItemEnabled(self.menuCheckId, 1)
                self.menu.setItemEnabled(self.menuShowId, 1)
                self.menu.setItemEnabled(self.menuDiagramID, 1)
                try:
                    self.menu.setItemEnabled(self.menuApidocId, 1)
                except:
                    pass
                if len(self.pdata["MAINSCRIPT"]) == 1:
                    self.emit(PYSIGNAL('pythonFile'), 
                        (os.path.join(self.ppath, self.pdata["MAINSCRIPT"][0]),))
                        
                self.emit(PYSIGNAL('projectOpened'), ())
                
                qApp.restoreOverrideCursor()
                
                if Preferences.getProject("SearchNewFiles"):
                    self.searchNewFiles()
                    
                # open a project session file being quiet about errors
                if Preferences.getProject("AutoLoadSession"):
                    self.readSession(1)
                    
            else:
                qApp.restoreOverrideCursor()
            
    def saveProject(self):
        """
        Public slot to save the current project.
        
        @return flag indicating success
        """
        if self.isDirty():
            if len(self.pfile) > 0:
                ok = self.writeProject()
            else:
                ok = self.saveProjectAs()
        else:
            ok = 1
        self.sessActGrp.setEnabled(ok)
        return ok
            
    def saveProjectAs(self):
        """
        Public slot to save the current project to a different file.
        
        @return flag indicating success
        """
        if Preferences.getProject("CompressedProjectFiles"):
            selectedFilter = self.trUtf8("Compressed Project Files (*.e3pz)")
        else:
            selectedFilter = self.trUtf8("Project Files (*.e3p)")
        fn = KQFileDialog.getSaveFileName(self.ppath,
            self.trUtf8("Project Files (*.e3p);;"
                "Compressed Project Files (*.e3pz)"),
            self.parent(), None, self.trUtf8("Save project as"), 
            selectedFilter, 0)
        
        if not fn.isNull():
            ext = QFileInfo(fn).extension()
            if ext.isEmpty():
                ex = selectedFilter.section('(*',1,1).section(')',0,0)
                if not ex.isEmpty():
                    fn.append(ex)
            if QFileInfo(fn).exists():
                abort = KQMessageBox.warning(self.parent(),
                    self.trUtf8("Save File"),
                    self.trUtf8("<p>The file <b>%1</b> already exists.</p>")
                        .arg(fn),
                    self.trUtf8("&Overwrite"),
                    self.trUtf8("&Abort"), None, 1)
                if abort:
                    return 0
                
            self.name = unicode(QFileInfo(fn).baseName(1))
            self.emit(PYSIGNAL('projectClosed'), ())
            self.emit(PYSIGNAL('projectOpened'), ())
            ok = self.writeProject(unicode(fn))
            self.sessActGrp.setEnabled(ok)
    
    def checkDirty(self):
        """
        Private method to check dirty status and open a message window.
        
        @return flag indicating whether this operation was successful
        """
        if self.isDirty():
            res = KQMessageBox.warning(self.parent(), 
                self.trUtf8("Close Project"),
                self.trUtf8("The current project has unsaved changes."),
                self.trUtf8("&Save"), self.trUtf8("&Discard changes"),
                self.trUtf8("&Abort"), 0, 2)
            if res == 0:
                return self.saveProject()
            elif res == 1:
                self.setDirty(0)
                return 1
            elif res == 2:
                return 0
            
        return 1
        
    def closeProject(self):
        """
        Public slot to close the current project.
        
        @return flag indicating success (boolean)
        """
        if not self.isOpen():
            return 1
            
        if not self.checkDirty():
            return 0
            
        # save the project session file being quiet about error
        if Preferences.getProject("AutoSaveSession"):
            self.writeSession(1)
        
        # save the list of recently opened projects
        ok = Preferences.Prefs.settings.writeEntry('/eric3/Recent/Projects', self.recent)
        
        # now save all open modified files of the project
        vm = self.parent().getViewManager()
        success = 1
        for key in ["SOURCES", "INTERFACES", "OTHERS"]:
            for fn in self.pdata[key]:
                fullname = os.path.join(self.ppath, fn)
                success &= vm.handleCloseWindow(fullname)
            
        if not success:
            return 0
                
        self.init()
        self.closeAct.setEnabled(0)
        self.saveasAct.setEnabled(0)
        self.saveAct.setEnabled(0)
        self.actGrp2.setEnabled(0)
        self.propsAct.setEnabled(0)
        self.vcsAddAct.setEnabled(0)
        self.sessActGrp.setEnabled(0)
        self.menu.setItemEnabled(self.menuCheckId, 0)
        self.menu.setItemEnabled(self.menuShowId, 0)
        self.menu.setItemEnabled(self.menuDiagramID, 0)
        try:
            self.menu.setItemEnabled(self.menuApidocId, 0)
        except:
            pass
        
        self.emit(PYSIGNAL('projectClosed'), ())
        
        return 1

    def saveAllScripts(self):
        """
        Public method to save all scripts belonging to the project.
        
        @return flag indicating success
        """
        vm = self.parent().getViewManager()
        success = 1
        for fn in self.pdata["SOURCES"]:
            fullname = os.path.join(self.ppath, fn)
            success &= vm.saveEditor(fullname)
            
        return success
        
    def getMainScript(self, normalized = 0):
        """
        Public method to return the main script filename.
        
        @param normalized flag indicating a normalized filename is wanted
        @return filename of the projects main script (string)
        """
        if len(self.pdata["MAINSCRIPT"]):
            if normalized:
                return os.path.join(self.ppath, self.pdata["MAINSCRIPT"][0])
            else:
                return self.pdata["MAINSCRIPT"]
        else:
            return None
            
    def getSources(self, normalized = 0):
        """
        Public method to return the source script files.
        
        @param normalized flag indicating a normalized filename is wanted
        @return list of the projects scripts (list of string)
        """
        if len(self.pdata["SOURCES"]):
            if normalized:
                flist = []
                for fn in self.pdata["SOURCES"]:
                    flist.append(os.path.join(self.ppath, fn))
                return flist
            else:
                return (self.pdata["SOURCES"])
                
    def isProjectSource(self, fn):
        """
        Public method used to check, if the passed in filename belongs to the project.
        
        @param fn filename to be checked (string or QString)
        @return flag indicating membership (boolean)
        """
        newfn = os.path.abspath(unicode(fn))
        newfn = newfn.replace(self.ppath + os.sep, '')
        return newfn in self.pdata["SOURCES"]
        
    def initActions(self):
        """
        Public slot to initialize the project related actions.
        """
        self.actions = []
        
        self.actGrp1 = E3ActionGroup(self)
        
        act = E3Action(self.trUtf8('New project'),
                QIconSet(UI.PixmapCache.getPixmap("projectNew.png")),
                self.trUtf8('&New...'),0,0,
                self.actGrp1,'project_new')
        act.setStatusTip(self.trUtf8('Generate a new project'))
        act.setWhatsThis(self.trUtf8(
            """<b>New...</b>"""
            """<p>This opens a dialog for entering the info for a"""
            """ new project.</p>"""
        ))
        act.connectIt(SIGNAL('activated()'),self.newProject)
        self.actions.append(act)

        act = E3Action(self.trUtf8('Open project'),
                QIconSet(UI.PixmapCache.getPixmap("projectOpen.png")),
                self.trUtf8('&Open...'),0,0,
                self.actGrp1,'project_open')
        act.setStatusTip(self.trUtf8('Open an existing project'))
        act.setWhatsThis(self.trUtf8(
            """<b>Open...</b>"""
            """<p>This opens an existing project.</p>"""
        ))
        act.connectIt(SIGNAL('activated()'),self.openProject)
        self.actions.append(act)

        self.closeAct = E3Action(self.trUtf8('Close project'),
                QIconSet(UI.PixmapCache.getPixmap("projectClose.png")),
                self.trUtf8('&Close'),0,0,self,'project_close')
        self.closeAct.setStatusTip(self.trUtf8('Close the current project'))
        self.closeAct.setWhatsThis(self.trUtf8(
            """<b>Close</b>"""
            """<p>This closes the current project.</p>"""
        ))
        self.closeAct.connectIt(SIGNAL('activated()'),self.closeProject)
        self.actions.append(self.closeAct)

        self.closeAct.setEnabled(0)
        
        self.saveAct = E3Action(self.trUtf8('Save project'),
                QIconSet(UI.PixmapCache.getPixmap("projectSave.png")),
                self.trUtf8('&Save'),0,0,self,'project_save')
        self.saveAct.setStatusTip(self.trUtf8('Save the current project'))
        self.saveAct.setWhatsThis(self.trUtf8(
            """<b>Save</b>"""
            """<p>This saves the current project.</p>"""
        ))
        self.saveAct.connectIt(SIGNAL('activated()'),self.saveProject)
        self.actions.append(self.saveAct)

        self.saveasAct = E3Action(self.trUtf8('Save project as'),
                QIconSet(UI.PixmapCache.getPixmap("projectSaveAs.png")),
                self.trUtf8('Save &as...'),0,0,self,'project_save_as')
        self.saveasAct.setStatusTip(self.trUtf8('Save the current project to a new file'))
        self.saveasAct.setWhatsThis(self.trUtf8(
            """<b>Save as</b>"""
            """<p>This saves the current project to a new file.</p>"""
        ))
        self.saveasAct.connectIt(SIGNAL('activated()'),self.saveProjectAs)
        self.actions.append(self.saveasAct)

        self.actGrp2 = E3ActionGroup(self)
        
        act = E3Action(self.trUtf8('Add file to project'),
                QIconSet(UI.PixmapCache.getPixmap("fileMisc.png")),
                self.trUtf8('Add &file...'),0,0,
                self.actGrp2,'project_add_file')
        act.setStatusTip(self.trUtf8('Add a file to the current project'))
        act.setWhatsThis(self.trUtf8(
            """<b>Add file...</b>"""
            """<p>This opens a dialog for adding a file"""
            """ to the current project. The place to add is"""
            """ determined by the file extension.</p>"""
        ))
        act.connectIt(SIGNAL('activated()'),self.addFile)
        self.actions.append(act)

        act = E3Action(self.trUtf8('Add directory to project'),
                QIconSet(UI.PixmapCache.getPixmap("dirOpen.png")),
                self.trUtf8('Add directory...'),0,0,
                self.actGrp2,'project_add_directory')
        act.setStatusTip(self.trUtf8('Add a directory to the current project'))
        act.setWhatsThis(self.trUtf8(
            """<b>Add directory...</b>"""
            """<p>This opens a dialog for adding a directory"""
            """ to the current project.</p>"""
        ))
        act.connectIt(SIGNAL('activated()'),self.addDirectory)
        self.actions.append(act)

        act = E3Action(self.trUtf8('Add translation to project'),
                QIconSet(UI.PixmapCache.getPixmap("linguist.png")),
                self.trUtf8('Add &translation...'),0,0,
                self.actGrp2,'project_add_translation')
        act.setStatusTip(self.trUtf8('Add a translation to the current project'))
        act.setWhatsThis(self.trUtf8(
            """<b>Add translation...</b>"""
            """<p>This opens a dialog for add a translation"""
            """ to the current project.</p>"""
        ))
        act.connectIt(SIGNAL('activated()'),self.addLanguage)
        self.actions.append(act)

        act = E3Action(self.trUtf8('Search new files'),
                self.trUtf8('Searc&h new files...'),0,0,
                self.actGrp2,'project_search_new_files')
        act.setStatusTip(self.trUtf8('Search new files in the project directory.'))
        act.setWhatsThis(self.trUtf8(
            """<b>Search new files...</b>"""
            """<p>This searches for new files (*.py, *.ptl, *.ui, *.idl) in the project"""
            """ directory and registered subdirectories.</p>"""
        ))
        act.connectIt(SIGNAL('activated()'),self.handleSearchNewFiles)
        self.actions.append(act)

        self.propsAct = E3Action(self.trUtf8('Project properties'),
                QIconSet(UI.PixmapCache.getPixmap("projectProps.png")),
                self.trUtf8('&Properties...'),0,0,self,'project_properties')
        self.propsAct.setStatusTip(self.trUtf8('Show the project properties'))
        self.propsAct.setWhatsThis(self.trUtf8(
            """<b>Properties...</b>"""
            """<p>This shows a dialog to edit the project properties.</p>"""
        ))
        self.propsAct.connectIt(SIGNAL('activated()'),self.showProperties)
        self.actions.append(self.propsAct)

        self.sessActGrp = E3ActionGroup(self)

        act = E3Action(self.trUtf8('Load session'),
                self.trUtf8('Load session'), 0, 0,
                self.sessActGrp, 'project_load_session')
        act.setStatusTip(self.trUtf8('Load the projects session file.'))
        act.setWhatsThis(self.trUtf8(
            """<b>Load session</b>"""
            """<p>This loads the projects session file. The session consists"""
            """ of the following data.<br>"""
            """- all open source files<br>"""
            """- all breakpoint<br>"""
            """- the commandline arguments<br>"""
            """- the working directory<br>"""
            """- the exception reporting flag</p>"""
        ))
        act.connectIt(SIGNAL('activated()'),self.readSession)
        self.actions.append(act)

        act = E3Action(self.trUtf8('Save session'),
                self.trUtf8('Save session'), 0, 0,
                self.sessActGrp, 'project_save_session')
        act.setStatusTip(self.trUtf8('Save the projects session file.'))
        act.setWhatsThis(self.trUtf8(
            """<b>Save session</b>"""
            """<p>This saves the projects session file. The session consists"""
            """ of the following data.<br>"""
            """- all open source files<br>"""
            """- all breakpoint<br>"""
            """- the commandline arguments<br>"""
            """- the working directory<br>"""
            """- the exception reporting flag</p>"""
        ))
        act.connectIt(SIGNAL('activated()'),self.writeSession)
        self.actions.append(act)
        
        act = E3Action(self.trUtf8('Delete session'),
                self.trUtf8('Delete session'), 0, 0,
                self.sessActGrp, 'project_delete_session')
        act.setStatusTip(self.trUtf8('Delete the projects session file.'))
        act.setWhatsThis(self.trUtf8(
            """<b>Delete session</b>"""
            """<p>This deletes the projects session file</p>"""
        ))
        act.connectIt(SIGNAL('activated()'),self.deleteSession)
        self.actions.append(act)
        
        self.chkGrp = E3ActionGroup(self)

        self.checkSyntaxAct = E3Action(self.trUtf8('Check Syntax'),
                self.trUtf8('&Syntax...'),0,0,
                self.chkGrp,'project_check_syntax')
        self.checkSyntaxAct.setStatusTip(self.trUtf8('Check project for syntax errors.'))
        self.checkSyntaxAct.setWhatsThis(self.trUtf8(
            """<b>Check Syntax...</b>"""
            """<p>This checks all Python files in the project"""
            """ for syntax errors.</p>"""
        ))
        self.checkSyntaxAct.connectIt(SIGNAL('activated()'),self.handleSyntaxCheck)
        self.actions.append(self.checkSyntaxAct)

        self.tabnannyAct = E3Action(self.trUtf8('Check Indentations'),
                self.trUtf8('&Indentations...'),0,0,
                self.chkGrp,'project_check_indentations')
        self.tabnannyAct.setStatusTip(self.trUtf8('Check project for bad indentations using tabnanny.'))
        self.tabnannyAct.setWhatsThis(self.trUtf8(
            """<b>Check Indentations...</b>"""
            """<p>This checks all Python files in the project"""
            """ for bad indentations using tabnanny.</p>"""
        ))
        self.tabnannyAct.connectIt(SIGNAL('activated()'),self.handleTabnanny)
        self.actions.append(self.tabnannyAct)

        self.codeMetricsAct = E3Action(self.trUtf8('Code Metrics'),
                self.trUtf8('&Code Metrics...'),0,0,
                self.chkGrp,'project_code_metrics')
        self.codeMetricsAct.setStatusTip(self.trUtf8('Show some code metrics for the project.'))
        self.codeMetricsAct.setWhatsThis(self.trUtf8(
            """<b>Code Metrics...</b>"""
            """<p>This shows some code metrics for all Python files in the project.</p>"""
        ))
        self.codeMetricsAct.connectIt(SIGNAL('activated()'),self.handleCodeMetrics)
        self.actions.append(self.codeMetricsAct)

        self.codeCoverageAct = E3Action(self.trUtf8('Python Code Coverage'),
                self.trUtf8('Code Co&verage...'),0,0,
                self.chkGrp,'project_code_coverage')
        self.codeCoverageAct.setStatusTip(self.trUtf8('Show code coverage information for the project.'))
        self.codeCoverageAct.setWhatsThis(self.trUtf8(
            """<b>Code Coverage...</b>"""
            """<p>This shows the code coverage information for all Python files in the project.</p>"""
        ))
        self.codeCoverageAct.connectIt(SIGNAL('activated()'),self.handleCodeCoverage)
        self.actions.append(self.codeCoverageAct)

        self.codeProfileAct = E3Action(self.trUtf8('Profile Data'),
                self.trUtf8('&Profile Data...'),0,0,
                self.chkGrp,'project_profile_data')
        self.codeProfileAct.setStatusTip(self.trUtf8('Show profiling data for the project.'))
        self.codeProfileAct.setWhatsThis(self.trUtf8(
            """<b>Profile Data...</b>"""
            """<p>This shows the profiling data for the project.</p>"""
        ))
        self.codeProfileAct.connectIt(SIGNAL('activated()'),self.handleProfileData)
        self.actions.append(self.codeProfileAct)

        self.codeCyclopsAct = E3Action(self.trUtf8('Cyclops Report'),
                self.trUtf8('Cyclops &Report...'),0,0,
                self.chkGrp,'project_cyclops_report')
        self.codeCyclopsAct.setStatusTip(self.trUtf8('Show cyclops report for the project.'))
        self.codeCyclopsAct.setWhatsThis(self.trUtf8(
            """<b>Cyclops Report...</b>"""
            """<p>This shows the cyclops report for the project.</p>"""
        ))
        self.codeCyclopsAct.connectIt(SIGNAL('activated()'),self.handleCyclopsReport)
        self.actions.append(self.codeCyclopsAct)

        self.removeCyclopsAct = E3Action(self.trUtf8('Remove Cyclops Report'),
                self.trUtf8('R&emove Cyclops Report'),0,0,
                self.chkGrp,'project_remove_cyclops_report')
        self.removeCyclopsAct.setStatusTip(self.trUtf8('Remove cyclops report for the project.'))
        self.removeCyclopsAct.setWhatsThis(self.trUtf8(
            """<b>Remove Cyclops Report</b>"""
            """<p>This removes the cyclops report for the project.</p>"""
        ))
        self.removeCyclopsAct.connectIt(SIGNAL('activated()'),self.handleRemoveCyclopsReport)
        self.actions.append(self.removeCyclopsAct)

        self.applicationDiagramAct = E3Action(self.trUtf8('Application Diagram'),
                self.trUtf8('&Application Diagram...'),0,0,
                self.chkGrp,'project_application_diagram')
        self.applicationDiagramAct.setStatusTip(self.trUtf8('Show a diagram of the project.'))
        self.applicationDiagramAct.setWhatsThis(self.trUtf8(
            """<b>Application Diagram...</b>"""
            """<p>This shows a diagram of the project.</p>"""
        ))
        self.applicationDiagramAct.connectIt(SIGNAL('activated()'),self.handleApplicationDiagram)
        self.actions.append(self.applicationDiagramAct)

        self.closeAct.setEnabled(0)
        self.saveAct.setEnabled(0)
        self.saveasAct.setEnabled(0)
        self.actGrp2.setEnabled(0)
        self.propsAct.setEnabled(0)
        self.sessActGrp.setEnabled(0)
        
        self.initVCSActions()
        self.doctools.initActions()
        
    def initMenu(self):
        """
        Public slot to initialize the project menu.
        
        @return the menu generated (QPopupMenu)
        """
        menu = QPopupMenu(self.parent())
        self.recentMenu = QPopupMenu(menu)
        self.vcsMenu = QPopupMenu(menu)
        self.checksMenu = QPopupMenu(menu)
        self.showMenu = QPopupMenu(menu)
        self.graphicsMenu = QPopupMenu(menu)
        self.apidocMenu = self.doctools.initMenu()
        
        # connect the aboutToShow signals
        self.connect(self.recentMenu, SIGNAL('aboutToShow()'), self.handleShowRecentMenu)
        self.connect(self.vcsMenu, SIGNAL('aboutToShow()'), self.handleShowVCSMenu)
        self.connect(self.showMenu, SIGNAL('aboutToShow()'), self.handleShowShowMenu)
        
        # build the checks menu
        self.checkSyntaxAct.addTo(self.checksMenu)
        self.tabnannyAct.addTo(self.checksMenu)
        
        # build the show menu
        self.codeMetricsAct.addTo(self.showMenu)
        self.codeCoverageAct.addTo(self.showMenu)
        self.codeProfileAct.addTo(self.showMenu)
        self.codeCyclopsAct.addTo(self.showMenu)
        self.removeCyclopsAct.addTo(self.showMenu)
        
        # build the diagrams menu
        self.applicationDiagramAct.addTo(self.graphicsMenu)
        
        # build the main menu
        menu.insertTearOffHandle()
        self.actGrp1.addTo(menu)
        menu.insertItem(self.trUtf8('Open &Recent Projects'), self.recentMenu)
        menu.insertSeparator()
        self.closeAct.addTo(menu)
        menu.insertSeparator()
        self.saveAct.addTo(menu)
        self.saveasAct.addTo(menu)
        menu.insertSeparator()
        self.sessActGrp.addTo(menu)
        menu.insertSeparator()
        self.actGrp2.addTo(menu)
        menu.insertSeparator()
        self.menuDiagramID = menu.insertItem(self.trUtf8('&Diagrams'), self.graphicsMenu)
        menu.insertSeparator()
        self.menuCheckId = menu.insertItem(self.trUtf8('Chec&k'), self.checksMenu)
        menu.insertSeparator()
        menu.insertItem(self.trUtf8('&Version Control'), self.vcsMenu)
        menu.insertSeparator()
        self.menuShowId = menu.insertItem(self.trUtf8('Sho&w'), self.showMenu)
        menu.insertSeparator()
        if self.apidocMenu is not None:
            self.menuApidocId = menu.insertItem(self.trUtf8('Source &Documentation'), self.apidocMenu)
            menu.insertSeparator()
        self.propsAct.addTo(menu)
        
        menu.setItemEnabled(self.menuCheckId, 0)
        menu.setItemEnabled(self.menuShowId, 0)
        menu.setItemEnabled(self.menuDiagramID, 0)
        try:
            menu.setItemEnabled(self.menuApidocId, 0)
        except:
            pass
        
        self.menu = menu
        return menu
        
    def initToolbar(self):
        """
        Public slot to initialize the project toolbar.
        
        @return the toolbar generated (QToolBar)
        """
        tb = QToolBar(self.parent())
        self.actGrp1.addTo(tb)
        self.closeAct.addTo(tb)
        tb.addSeparator()
        self.saveAct.addTo(tb)
        self.saveasAct.addTo(tb)
        tb.addSeparator()
        self.propsAct.addTo(tb)
        
        return tb
        
    def handleShowRecentMenu(self):
        """
        Private method to set up the recent projects menu.
        """
        idx = 0
        self.recentMenu.clear()
        
        for rp in self.recent:
            id = self.recentMenu.insertItem('&%d. %s' % (idx+1, unicode(rp)),
                                            self.handleOpenRecent)
            self.recentMenu.setItemParameter(id,idx)
            
            idx = idx + 1
            
        self.recentMenu.insertSeparator()
        self.recentMenu.insertItem(self.trUtf8('Clear'), self.handleClearRecent)
        
    def handleOpenRecent(self, idx):
        """
        Private method to open a project from the list of rencently opened projects.
        """
        self.openProject(unicode(self.recent[idx]))
        
    def handleClearRecent(self):
        """
        Private method to clear the recent projects menu.
        """
        self.recent = QStringList()
        
    def handleSearchNewFiles(self):
        """
        Private slot used to handle the search new files action.
        """
        self.searchNewFiles(0, 1)
        
    def searchNewFiles(self, AI=1, onUserDemand=0):
        """
        Private method to search for new files in the project directory.
        
        If new files were found it shows a dialog listing these files and
        gives the user the oportunity to select the ones he wants to
        include. If 'Automatic Inclusion' is enabled, the new files are
        automatically added to the project.
        
        @param AI flag indicating whether the automatic inclusion should
                be honoured
        @param onUserDemand flag indicating whether this method was 
                requested by the user via a menu action
        """
        autoInclude = Preferences.getProject("AutoIncludeNewFiles")
        newFiles = QStringList()
        for dir in self.subdirs:
            curpath = os.path.join(self.ppath, dir)
            try:
                newSources = os.listdir(curpath)
            except OSError:
                newSources = []
            for ns in newSources:
                # set fn to project relative name
                # then reset ns to fully qualified name for insertion, possibly.
                if dir == "":
                    fn = ns
                else:
                    fn = os.path.join(dir, ns)
                ns = os.path.abspath(os.path.join(curpath, ns))

                # do not bother with dirs here...
                if os.path.isdir(ns):
                    continue
                # we only deal with .py, .ptl, .idl, .ui and .ui.h, where .py and .ptl would be
                # catagorized as source, .ui and .ui.h are forms and .idl are interfaces, 
                # so deal with appropriately making sure to properly normalize the extension
                # case prior to compare.
                if fn.endswith('.ui.h'):
                    ext = '.ui.h'
                else:
                    dummy, ext = os.path.splitext(fn)
                ext = os.path.normcase(ext)
                if (ext in ['.py', '.ptl'] and not fn in self.pdata["SOURCES"]) or \
                   (ext in ['.ui', '.ui.h'] and not fn in self.pdata["FORMS"]) or \
                   (ext == '.idl' and not fn in self.pdata["INTERFACES"]):
                    if autoInclude and AI:
                        self.appendFile(ns)
                    else:
                        newFiles.append(ns)
                        
        # if autoInclude is set there is no more work left
        if (autoInclude and AI):
            return

        # if newfiles is empty, put up message box informing user nothing found
        if newFiles.isEmpty():
            if onUserDemand:
                KQMessageBox.information(None,
                    self.trUtf8("Search New Files"),
                    self.trUtf8("There were no new files found to be added."),
                    self.trUtf8("&OK"))
            return
            
        # autoInclude is not set, show a dialog
        dlg = AddFoundFilesDialog(newFiles, self.parent(), None, 1)
        res = dlg.exec_loop()
        
        # the 'Add All' button was pressed
        if res == 1:
            for file in newFiles:
                self.appendFile(unicode(file))
                
        # the 'Add Selected' button was pressed
        elif res == 2:
            files = dlg.getSelection()
            for file in files:
                self.appendFile(unicode(file))
                
    def othersAdded(self, fn):
        """
        Public slot to be called, if something was added to the OTHERS project data area.
        
        @param fn filename or directory name added (string or QString)
        """
        f = unicode(fn)
        self.emit(PYSIGNAL('projectOthersAdded'), (f,))
        
    def getActions(self):
        """
        Public method to get a list of all actions.
        
        @return list of all actions (list of E3Action)
        """
        return self.actions[:] + self.doctools.getActions()
        
    ##############################################################
    ## Below is the VCS interface
    ##############################################################
    
    def initVCS(self, vcsSystem=None):
        """
        Private method used to instantiate a vcs system.
        
        @param vcsSystem type of VCS to be used
        @return a reference to the vcs object
        """
        vcs = None
        forProject = 1
        
        if vcsSystem is None:
            if len(self.pdata["VCS"]):
                if self.pdata["VCS"][0] == 'None':
                    return vcs
                else:
                    vcsSystem = self.pdata["VCS"][0]
            else:
                return vcs
        else:
            vcsSystem = str(vcsSystem)
            forProject = 0
            
        try:
            vcs = VCS.factory(vcsSystem)
        except:
            return vcs
            
        if not vcs.vcsExists():
            qApp.restoreOverrideCursor()
            KQMessageBox.critical(None,
                self.trUtf8("Version Control System"),
                self.trUtf8("<p>The selected VCS <b>%1</b> could not be found.<br>"
                    "Disabling version control.</p>").arg(self.pdata["VCS"][0]),
                self.trUtf8("OK"))
            vcs = None
            if forProject:
                self.pdata["VCS"][0] = 'None'
                self.setDirty(1)
            return vcs
            
        if forProject:
            # set the vcs options
            try:
                vcsopt = eval(self.pdata["VCSOPTIONS"][0])
                vcs.vcsSetOptions(vcsopt)
            except:
                pass
            # set vcs specific data
            try:
                vcsother = eval(self.pdata["VCSOTHERDATA"][0])
                vcs.vcsSetOtherData(vcsother)
            except:
                pass
                
        return vcs
        
    def initVCSActions(self):
        """
        Private method to initialize the VCS actions.
        """
        self.vcsActGrp = E3ActionGroup(self)
        
        self.vcsNewAct = E3Action(self.trUtf8('New from repository'),
                self.trUtf8('&New from repository...'),0,0,self,'project_vcs_new')
        self.vcsNewAct.setStatusTip(self.trUtf8(
            'Create a new project from the VCS repository'
        ))
        self.vcsNewAct.setWhatsThis(self.trUtf8(
            """<b>New from repository</b>"""
            """<p>This creates a new local project from the VCS repository.</p>"""
        ))
        self.vcsNewAct.connectIt(SIGNAL('activated()'),self.vcsCheckout)
        self.actions.append(self.vcsNewAct)
        
        self.vcsUpdateAct = E3Action(self.trUtf8('Update from repository'),
                self.trUtf8('&Update from repository'),0,0,self.vcsActGrp,
                'project_vcs_update')
        self.vcsUpdateAct.setStatusTip(self.trUtf8(
            'Update the local project from the VCS repository'
        ))
        self.vcsUpdateAct.setWhatsThis(self.trUtf8(
            """<b>Update from repository</b>"""
            """<p>This updates the local project from the VCS repository.</p>"""
        ))
        self.vcsUpdateAct.connectIt(SIGNAL('activated()'),self.vcsUpdate)
        self.actions.append(self.vcsUpdateAct)
        
        self.vcsCommitAct = E3Action(self.trUtf8('Commit changes to repository'),
                self.trUtf8('&Commit changes to repository...'),0,0,self.vcsActGrp,
                'project_vcs_commit')
        self.vcsCommitAct.setStatusTip(self.trUtf8(
            'Commit changes to the local project to the VCS repository'
        ))
        self.vcsCommitAct.setWhatsThis(self.trUtf8(
            """<b>Commit changes to repository</b>"""
            """<p>This commits changes to the local project to the VCS repository.</p>"""
        ))
        self.vcsCommitAct.connectIt(SIGNAL('activated()'),self.vcsCommit)
        self.actions.append(self.vcsCommitAct)
        
        self.vcsAddAct = E3Action(self.trUtf8('Add to repository'),
                self.trUtf8('&Add to repository...'),0,0,self,'project_vcs_add')
        self.vcsAddAct.setStatusTip(self.trUtf8(
            'Add the local project to the VCS repository'
        ))
        self.vcsAddAct.setWhatsThis(self.trUtf8(
            """<b>Add to repository</b>"""
            """<p>This adds (imports) the local project to the VCS repository.</p>"""
        ))
        self.vcsAddAct.connectIt(SIGNAL('activated()'),self.vcsImport)
        self.actions.append(self.vcsAddAct)
        
        self.vcsRemoveAct = E3Action(self.trUtf8(
                'Remove from repository (and disk)'),
                self.trUtf8('&Remove from repository (and disk)'),
                0,0,self.vcsActGrp,'project_vcs_remove')
        self.vcsRemoveAct.setStatusTip(self.trUtf8(
            'Remove the local project from the VCS repository (and  disk)'
        ))
        self.vcsRemoveAct.setWhatsThis(self.trUtf8(
            """<b>Remove from repository</b>"""
            """<p>This removes the local project from the VCS repository (and disk).</p>"""
        ))
        self.vcsRemoveAct.connectIt(SIGNAL('activated()'),self.vcsRemove)
        self.actions.append(self.vcsRemoveAct)
        
        self.vcsLogAct = E3Action(self.trUtf8(
                'Show log'),
                self.trUtf8('Show &log'),
                0, 0, self.vcsActGrp, 'project_vcs_log')
        self.vcsLogAct.setStatusTip(self.trUtf8(
            'Show the log of the local project'
        ))
        self.vcsLogAct.setWhatsThis(self.trUtf8(
            """<b>Show log</b>"""
            """<p>This shows the log of the local project.</p>"""
        ))
        self.vcsLogAct.connectIt(SIGNAL('activated()'),self.vcsLog)
        self.actions.append(self.vcsLogAct)
        
        self.vcsDiffAct = E3Action(self.trUtf8('Show difference to repository'),
                self.trUtf8('Show &difference to repository'),
                0, 0, self.vcsActGrp, 'project_vcs_diff')
        self.vcsDiffAct.setStatusTip(self.trUtf8(
            'Show the difference of the local project to the repository'
        ))
        self.vcsDiffAct.setWhatsThis(self.trUtf8(
            """<b>Show difference to repository</b>"""
            """<p>This shows the difference of the local project to the repository.</p>"""
        ))
        self.vcsDiffAct.connectIt(SIGNAL('activated()'),self.vcsDiff)
        self.actions.append(self.vcsDiffAct)
        
        self.vcsStatusAct = E3Action(self.trUtf8('Show status'),
                self.trUtf8('Show &status'),
                0, 0, self.vcsActGrp, 'project_vcs_status')
        self.vcsStatusAct.setStatusTip(self.trUtf8(
            'Show the status of the local project'
        ))
        self.vcsStatusAct.setWhatsThis(self.trUtf8(
            """<b>Show status</b>"""
            """<p>This shows the status of the local project.</p>"""
        ))
        self.vcsStatusAct.connectIt(SIGNAL('activated()'),self.vcsStatus)
        self.actions.append(self.vcsStatusAct)
        
        self.vcsTagAct = E3Action(self.trUtf8('Tag in repository'), 
                self.trUtf8('&Tag in repository...'),
                0, 0, self.vcsActGrp, 'project_vcs_tag')
        self.vcsTagAct.setStatusTip(self.trUtf8(
            'Tag the local project in the repository'
        ))
        self.vcsTagAct.setWhatsThis(self.trUtf8(
            """<b>Tag in repository</b>"""
            """<p>This tags the local project in the repository.</p>"""
        ))
        self.vcsTagAct.connectIt(SIGNAL('activated()'),self.vcsTag)
        self.actions.append(self.vcsTagAct)
        
        self.vcsExportAct = E3Action(self.trUtf8('Export from repository'), 
                self.trUtf8('&Export from repository...'),
                0, 0, self, 'project_vcs_export')
        self.vcsExportAct.setStatusTip(self.trUtf8(
            'Export a project from the repository'
        ))
        self.vcsExportAct.setWhatsThis(self.trUtf8(
            """<b>Export from repository</b>"""
            """<p>This exports a project from the repository.</p>"""
        ))
        self.vcsExportAct.connectIt(SIGNAL('activated()'),self.vcsExport)
        self.actions.append(self.vcsExportAct)
        
        self.vcsPropsAct = E3Action(self.trUtf8('Command options'),
                self.trUtf8('Command &options...'),0,0,self.vcsActGrp,
                'project_vcs_options')
        self.vcsPropsAct.setStatusTip(self.trUtf8('Show the VCS command options'))
        self.vcsPropsAct.setWhatsThis(self.trUtf8(
            """<b>Command options...</b>"""
            """<p>This shows a dialog to edit the VCS command options.</p>"""
        ))
        self.vcsPropsAct.connectIt(SIGNAL('activated()'),self.vcsCommandOptions)
        self.actions.append(self.vcsPropsAct)
        
        self.vcsLoginAct = E3Action(self.trUtf8('Login to repository'),
                self.trUtf8('Login to repository...'),0,0,self,'project_vcs_login')
        self.vcsLoginAct.setStatusTip(self.trUtf8(
            'Login to repository'
        ))
        self.vcsLoginAct.setWhatsThis(self.trUtf8(
            """<b>Login to repository</b>"""
            """<p>This performs a login to the VCS repository.</p>"""
        ))
        self.vcsLoginAct.connectIt(SIGNAL('activated()'),self.vcsLogin)
        self.actions.append(self.vcsLoginAct)
        
        self.vcsLogoutAct = E3Action(self.trUtf8('Logout from repository'),
                self.trUtf8('Logout from repository...'),0,0,self,'project_vcs_logout')
        self.vcsLogoutAct.setStatusTip(self.trUtf8(
            'Logout from repository'
        ))
        self.vcsLogoutAct.setWhatsThis(self.trUtf8(
            """<b>Logout from repository</b>"""
            """<p>This performs a logout from the VCS repository.</p>"""
        ))
        self.vcsLogoutAct.connectIt(SIGNAL('activated()'),self.vcsLogout)
        self.actions.append(self.vcsLogoutAct)
        
        self.vcsRevertAct = E3Action(self.trUtf8('Revert changes'),
                self.trUtf8('Re&vert changes'),
                0, 0, self.vcsActGrp, 'project_vcs_revert')
        self.vcsRevertAct.setStatusTip(self.trUtf8(
            'Revert all changes made to the local project'
        ))
        self.vcsRevertAct.setWhatsThis(self.trUtf8(
            """<b>Revert changes</b>"""
            """<p>This reverts all changes made to the local project.</p>"""
        ))
        self.vcsRevertAct.connectIt(SIGNAL('activated()'),self.vcsRevert)
        self.actions.append(self.vcsRevertAct)
        
        self.vcsMergeAct = E3Action(self.trUtf8('Merge'),
                self.trUtf8('Mer&ge changes...'),
                0, 0, self.vcsActGrp, 'project_vcs_merge')
        self.vcsMergeAct.setStatusTip(self.trUtf8(
            'Merge changes of a tag/revision into the local project'
        ))
        self.vcsMergeAct.setWhatsThis(self.trUtf8(
            """<b>Merge</b>"""
            """<p>This merges changes of a tag/revision into the local project.</p>"""
        ))
        self.vcsMergeAct.connectIt(SIGNAL('activated()'),self.vcsMerge)
        self.actions.append(self.vcsMergeAct)
        
        self.vcsSwitchAct = E3Action(self.trUtf8('Switch'),
                self.trUtf8('S&witch...'),
                0, 0, self.vcsActGrp, 'project_vcs_switch')
        self.vcsSwitchAct.setStatusTip(self.trUtf8(
            'Switch the local copy to another tag/branch'
        ))
        self.vcsSwitchAct.setWhatsThis(self.trUtf8(
            """<b>Switch</b>"""
            """<p>This switches the local copy to another tag/branch.</p>"""
        ))
        self.vcsSwitchAct.connectIt(SIGNAL('activated()'),self.vcsSwitch)
        self.actions.append(self.vcsSwitchAct)
        
        self.vcsResolveAct = E3Action(self.trUtf8('Resolve conflicts'),
                self.trUtf8('Resolve con&flicts'),
                0, 0, self.vcsActGrp, 'project_vcs_resolve')
        self.vcsResolveAct.setStatusTip(self.trUtf8(
            'Resolve all conflicts of the local project'
        ))
        self.vcsResolveAct.setWhatsThis(self.trUtf8(
            """<b>Resolve conflicts</b>"""
            """<p>This resolves all conflicts of the local project.</p>"""
        ))
        self.vcsResolveAct.connectIt(SIGNAL('activated()'),self.svnResolve)
        self.actions.append(self.vcsResolveAct)
        
        self.vcsCleanupAct = E3Action(self.trUtf8('Cleanup'),
                self.trUtf8('Cleanu&p'),
                0, 0, self.vcsActGrp, 'project_vcs_cleanup')
        self.vcsCleanupAct.setStatusTip(self.trUtf8(
            'Cleanup the local project'
        ))
        self.vcsCleanupAct.setWhatsThis(self.trUtf8(
            """<b>Cleanup</b>"""
            """<p>This performs a cleanup of the local project.</p>"""
        ))
        self.vcsCleanupAct.connectIt(SIGNAL('activated()'),self.vcsCleanup)
        self.actions.append(self.vcsCleanupAct)
        
        self.vcsCommandAct = E3Action(self.trUtf8('Execute command'),
                self.trUtf8('E&xecute command...'),
                0, 0, self.vcsActGrp, 'project_vcs_command')
        self.vcsCommandAct.setStatusTip(self.trUtf8(
            'Execute an arbitrary VCS command'
        ))
        self.vcsCommandAct.setWhatsThis(self.trUtf8(
            """<b>Execute command</b>"""
            """<p>This opens a dialog to enter an arbitrary VCS command.</p>"""
        ))
        self.vcsCommandAct.connectIt(SIGNAL('activated()'),self.vcsCommand)
        self.actions.append(self.vcsCommandAct)
        
        #######################################################################
        ## Subversion specific actions below
        #######################################################################
        
        self.svnTagListAct = E3Action(self.trUtf8('List tags'), 
                self.trUtf8('List tags...'),
                0, 0, self.vcsActGrp, 'project_svn_list_tags')
        self.svnTagListAct.setStatusTip(self.trUtf8(
            'List tags of the project'
        ))
        self.svnTagListAct.setWhatsThis(self.trUtf8(
            """<b>List tags</b>"""
            """<p>This lists the tags of the project.</p>"""
        ))
        self.svnTagListAct.connectIt(SIGNAL('activated()'),self.svnTagList)
        self.actions.append(self.svnTagListAct)
        
        self.svnBranchListAct = E3Action(self.trUtf8('List branches'), 
                self.trUtf8('List branches...'),
                0, 0, self.vcsActGrp, 'project_svn_list_branches')
        self.svnBranchListAct.setStatusTip(self.trUtf8(
            'List branches of the project'
        ))
        self.svnBranchListAct.setWhatsThis(self.trUtf8(
            """<b>List branches</b>"""
            """<p>This lists the branches of the project.</p>"""
        ))
        self.svnBranchListAct.connectIt(SIGNAL('activated()'),self.svnBranchList)
        self.actions.append(self.svnBranchListAct)
        
        self.svnListAct = E3Action(self.trUtf8('List repository contents'), 
                self.trUtf8('List repository contents...'),
                0, 0, self.vcsActGrp, 'project_svn_contents')
        self.svnListAct.setStatusTip(self.trUtf8(
            'Lists the contents of the repository'
        ))
        self.svnListAct.setWhatsThis(self.trUtf8(
            """<b>List repository contents</b>"""
            """<p>This lists the contents of the repository.</p>"""
        ))
        self.svnListAct.connectIt(SIGNAL('activated()'),self.svnTagList)
        self.actions.append(self.svnListAct)
        
        self.svnPropSetAct = E3Action(self.trUtf8('Set Property'),
                self.trUtf8('Set Property...'),
                0, 0, self.vcsActGrp, 'project_svn_property_set')
        self.svnPropSetAct.setStatusTip(self.trUtf8(
            'Set a property for the project files'
        ))
        self.svnPropSetAct.setWhatsThis(self.trUtf8(
            """<b>Set Property</b>"""
            """<p>This sets a property for the project files.</p>"""
        ))
        self.svnPropSetAct.connectIt(SIGNAL('activated()'),self.svnPropSet)
        self.actions.append(self.svnPropSetAct)
        
        self.svnPropListAct = E3Action(self.trUtf8('List Properties'),
                self.trUtf8('List Properties...'),
                0, 0, self.vcsActGrp, 'project_svn_property_list')
        self.svnPropListAct.setStatusTip(self.trUtf8(
            'List properties of the project files'
        ))
        self.svnPropListAct.setWhatsThis(self.trUtf8(
            """<b>List Properties</b>"""
            """<p>This lists the properties of the project files.</p>"""
        ))
        self.svnPropListAct.connectIt(SIGNAL('activated()'),self.svnPropList)
        self.actions.append(self.svnPropListAct)
        
        self.svnPropDelAct = E3Action(self.trUtf8('Delete Property'),
                self.trUtf8('Delete Property...'),
                0, 0, self.vcsActGrp, 'project_svn_property_delete')
        self.svnPropDelAct.setStatusTip(self.trUtf8(
            'Delete a property for the project files'
        ))
        self.svnPropDelAct.setWhatsThis(self.trUtf8(
            """<b>Delete Property</b>"""
            """<p>This deletes a property for the project files.</p>"""
        ))
        self.svnPropDelAct.connectIt(SIGNAL('activated()'),self.svnPropDel)
        self.actions.append(self.svnPropDelAct)
        
        self.vcsAddAct.setEnabled(0)
        
    def handleShowVCSMenu(self):
        """
        Private slot called before the vcs menu is shown.
        """
        self.vcsMenu.clear()
        
        if self.vcs is None:
            self.vcsNewAct.addTo(self.vcsMenu)
            self.vcsExportAct.addTo(self.vcsMenu)
            self.vcsMenu.insertSeparator()
            self.vcsAddAct.addTo(self.vcsMenu)
            self.vcsMenu.insertSeparator()
            self.vcsLoginAct.addTo(self.vcsMenu)
            self.vcsLogoutAct.addTo(self.vcsMenu)
        else:
            vcsName = self.vcs.vcsName()
            self.vcsUpdateAct.addTo(self.vcsMenu)
            self.vcsCommitAct.addTo(self.vcsMenu)
            self.vcsMenu.insertSeparator()
            self.vcsNewAct.addTo(self.vcsMenu)
            self.vcsExportAct.addTo(self.vcsMenu)
            self.vcsMenu.insertSeparator()
            self.vcsAddAct.addTo(self.vcsMenu)
            self.vcsRemoveAct.addTo(self.vcsMenu)
            self.vcsMenu.insertSeparator()
            self.vcsTagAct.addTo(self.vcsMenu)
            if vcsName == 'Subversion':
                if self.vcs.otherData["standardLayout"]:
                    self.svnTagListAct.addTo(self.vcsMenu)
                    self.svnBranchListAct.addTo(self.vcsMenu)
                else:
                    self.svnListAct.addTo(self.vcsMenu)
            self.vcsMenu.insertSeparator()
            self.vcsLogAct.addTo(self.vcsMenu)
            if vcsName == 'Subversion':
                self.vcsStatusAct.addTo(self.vcsMenu)
            self.vcsDiffAct.addTo(self.vcsMenu)
            self.vcsMenu.insertSeparator()
            self.vcsRevertAct.addTo(self.vcsMenu)
            self.vcsMergeAct.addTo(self.vcsMenu)
            if vcsName == 'Subversion':
                self.vcsResolveAct.addTo(self.vcsMenu)
            self.vcsMenu.insertSeparator()
            self.vcsSwitchAct.addTo(self.vcsMenu)
            if vcsName == 'Subversion':
                self.vcsMenu.insertSeparator()
                self.svnPropSetAct.addTo(self.vcsMenu)
                self.svnPropListAct.addTo(self.vcsMenu)
                self.svnPropDelAct.addTo(self.vcsMenu)
            self.vcsMenu.insertSeparator()
            self.vcsCleanupAct.addTo(self.vcsMenu)
            if vcsName == 'CVS':
                self.vcsLoginAct.addTo(self.vcsMenu)
                self.vcsLogoutAct.addTo(self.vcsMenu)
            self.vcsMenu.insertSeparator()
            self.vcsCommandAct.addTo(self.vcsMenu)
            self.vcsMenu.insertSeparator()
            self.vcsPropsAct.addTo(self.vcsMenu)
        
    def vcsCheckout(self, export=0):
        """
        Private slot used to create a local project from the repository.
        
        @param export flag indicating whether an export or a checkout
                should be performed
        """
        if not self.checkDirty():
            return
            
        vcsSelected, ok = KQInputDialog.getItem(\
            self.trUtf8("New Project"),
            self.trUtf8("Select version control system for the project"),
            ConfigVcsSystemsDisplay[1:],
            0, 0)
        vcsSystem = ConfigVcsSystems[ConfigVcsSystemsDisplay.findIndex(vcsSelected)]
        if not ok:
            return
            
        self.pdata["VCS"] = [vcsSystem]
        self.vcs = self.initVCS(vcsSystem)
        if self.vcs is not None:
            vcsdlg = self.vcs.vcsNewProjectOptionsDialog()
            if vcsdlg.exec_loop() == QDialog.Accepted:
                self.closeProject()
                vcsDir, archive, tag, projectdir, standardlayout =\
                    vcsdlg.getData()
                self.pdata["VCS"] = [vcsSystem]
                self.vcs = self.initVCS(vcsSystem)
                # edit VCS command options
                vcores = KQMessageBox.question(None,
                    self.trUtf8("Import Project"),
                    self.trUtf8("""Would you like to edit the VCS command options?"""),
                    self.trUtf8("&Yes"),
                    self.trUtf8("&No"),
                    None,
                    1, -1)
                if vcores == 0:
                    codlg = vcsCommandOptionsDialog(self.vcs)
                    if codlg.exec_loop() == QDialog.Accepted:
                        self.vcs.vcsSetOptions(codlg.getOptions())
            
                # create the project directory if it doesn't exist already
                if not os.path.isdir(projectdir):
                    try:
                        os.makedirs(projectdir)
                    except:
                        KQMessageBox.critical(None,
                            self.trUtf8("Create project directory"),
                            self.trUtf8("<p>The project directory <b>%1</b> could not be created.</p>")
                                .arg(projectdir),
                            self.trUtf8("&OK"))
                        self.vcs = None
                        return
                
                # create the project from the VCS
                self.vcs.vcsSetData('standardLayout', standardlayout)
                if export:
                    self.vcs.vcsExport(vcsDir, archive, projectdir, tag)
                else:
                    self.vcs.vcsCheckout(vcsDir, archive, projectdir, 0, tag)
                vcsName = self.vcs.vcsName()
                if vcsName == 'CVS':
                    projectdir = "%s%s%s" % (projectdir, os.sep, archive)
                projectdir = Utilities.normcasepath(projectdir)
                d = QDir(projectdir)
                plist = d.entryInfoList("*.e3p *.e3pz")
                if len(plist):
                    if len(plist) == 1:
                        self.openProject(plist[0].absFilePath())
                        self.emit(PYSIGNAL('newProject'), ())
                    else:
                        pfilenamelist = d.entryList("*.e3p *.e3pz") # same as above
                        pfilename, ok = KQInputDialog.getItem(
                            self.trUtf8("New project from repository"),
                            self.trUtf8("Select a project file to open."),
                            pfilenamelist, 0, 0)
                        if ok:
                            self.openProject(QFileInfo(pfilename).absFilePath())
                            self.emit(PYSIGNAL('newProject'), ())
                else:
                    res = KQMessageBox.question(None,
                        self.trUtf8("New project from repository"),
                        self.trUtf8("The project retrieved from the repository"
                            " does not contain a project file (*.e3p *.e3pz)."
                            " Create it automatically?"),
                        self.trUtf8("&Yes"), self.trUtf8("&No"))
                    if res == 0:
                        self.ppath = projectdir
                        self.opened = 1
                        
                        # search the project directory for files with extensions
                        # of 'py', 'ptl', 'idl', 'ui' and 'ts'
                        for filespec in ['*.py', '*.ptl', '*.idl', '*.ui', '*.ui.h']:
                            files = Utilities.direntries(projectdir, 1, filespec)
                            for file in files:
                                self.appendFile(file)
                                
                        # special handling for translation files (*.ts)
                        tslist = d.entryList('*.ts')
                        if len(tslist) and tslist[0].contains('_'):
                            # the first entry determines the filename pattern
                            mainscriptname = tslist[0].section('_', 0, 0)
                            for ts in tslist:
                                if ts.startsWith(mainscriptname):
                                    self.pdata["TRANSLATIONS"].append(str(ts))
                                    self.emit(PYSIGNAL('projectLanguageAdded'), (str(ts),))
                            self.pdata["MAINSCRIPT"] = ['%s.py' % unicode(mainscriptname)]
                            
                        # now save the new project file and reload it
                        self.saveProjectAs()
                        self.openProject(self.pfile)
                        if not export:
                            res = KQMessageBox.question(None,
                                self.trUtf8("New project from repository"),
                                self.trUtf8("Shall the project file be added to the repository?"),
                                self.trUtf8("&Yes"), self.trUtf8("&No"))
                            if res == 0:
                                self.vcs.vcsAdd(self.pfile)
        
    def vcsExport(self):
        """
        Private slot used to export a project from the repository.
        """
        self.vcsCheckout(1)
        
    def vcsUpdate(self):
        """
        Private slot used to update the local project from the repository.
        """
        self.vcs.vcsUpdate(self.ppath)
        
    def vcsCommit(self):
        """
        Private slot used to commit changes to the local project to the repository.
        """
        self.vcs.vcsCommit(self.ppath, '')
        
    def vcsImport(self):
        """
        Private slot used to import the local project into the repository.
        
        <b>NOTE</b>: 
            This does not make the local project a vcs controlled project.
            You have to checkout the project from the repository in order
            to accomplish that.
        """
        pdata_vcs = self.pdata["VCS"]
        pdata_vcsoptions = self.pdata["VCSOPTIONS"]
        pdata_vcsother = self.pdata["VCSOTHERDATA"]
        vcs = self.vcs
        vcsSelected, ok = KQInputDialog.getItem(\
            self.trUtf8("New Project"),
            self.trUtf8("Select version control system for the project"),
            ConfigVcsSystemsDisplay[1:],
            0, 0)
        vcsSystem = ConfigVcsSystems[ConfigVcsSystemsDisplay.findIndex(vcsSelected)]
        if not ok:
            return
            
        self.pdata["VCS"] = [vcsSystem]
        self.vcs = self.initVCS(vcsSystem)
        if self.vcs is not None:
            vcsdlg = self.vcs.vcsOptionsDialog(self, self.name, 1)
            if vcsdlg.exec_loop() == QDialog.Accepted:
                vcsDir, archive, vendor, message, release, standardlayout =\
                    vcsdlg.getData()
                # edit VCS command options
                vcores = KQMessageBox.question(None,
                    self.trUtf8("Import Project"),
                    self.trUtf8("""Would you like to edit the VCS command options?"""),
                    self.trUtf8("&Yes"),
                    self.trUtf8("&No"),
                    None,
                    1, -1)
                if vcores == 0:
                    codlg = vcsCommandOptionsDialog(self.vcs)
                    if codlg.exec_loop() == QDialog.Accepted:
                        self.vcs.vcsSetOptions(codlg.getOptions())
                self.setDirty(1)
                self.vcs.vcsSetData('standardLayout', standardlayout)
                self.saveProject()
                self.vcs.vcsInit(vcsDir, 1)
                self.vcs.vcsImport(vcsDir, message, archive, vendor, release,
                    self.ppath)
                # revert the changes to the local project 
                # because the project dir is not a VCS directory
                self.pdata["VCS"] = pdata_vcs
                self.pdata["VCSOPTIONS"] = pdata_vcsoptions
                self.pdata["VCSOTHERDATA"] = pdata_vcsother
                self.vcs = vcs
                self.setDirty(1)
                self.saveProject()
        
    def vcsRemove(self):
        """
        Private slot used to remove the local project from the repository.
        
        Depending on the parameters set in the vcs object the project
        may be removed from the local disk as well.
        """
        res = KQMessageBox.warning(None,
            self.trUtf8("Remove project from repository"),
            self.trUtf8("Dou you really want to remove this project from the repository (and disk)?"),
            self.trUtf8("&Yes"), self.trUtf8("&No"), None, 1)
        if res == 0:
            self.vcs.vcsRemove(self.ppath, 1)
            self.vcsCommit()
            if not os.path.exists(self.pfile):
                ppath = self.ppath
                self.setDirty(0)
                self.closeProject()
                shutil.rmtree(ppath, 1)
            
    def vcsCommandOptions(self):
        """
        Private slot to edit the VCS command options.
        """
        codlg = vcsCommandOptionsDialog(self.vcs)
        if codlg.exec_loop() == QDialog.Accepted:
            self.vcs.vcsSetOptions(codlg.getOptions())
            self.setDirty(1)
        
    def vcsLog(self):
        """
        Private slot used to show the log of the local project.
        """
        self.vcs.vcsLog(self.ppath)
        
    def vcsDiff(self):
        """
        Private slot used to show the difference of the local project to the repository.
        """
        self.vcs.vcsDiff(self.ppath)
        
    def vcsStatus(self):
        """
        Private slot used to show the status of the local project.
        """
        self.vcs.vcsStatus(self.ppath)
        
    def vcsTag(self):
        """
        Private slot used to tag the local project in the repository.
        """
        self.vcs.vcsTag(self.ppath)
        
    def vcsLogin(self):
        """
        Private slot used to login to a remote repository.
        """
        dlg = vcsLoginDialog()
        if dlg.exec_loop() == QDialog.Accepted:
            vcsSystem, protocol, username, password, hostname, port, repository = dlg.getData()
            vcs = self.initVCS(vcsSystem)
            if vcs is not None:
                vcs.vcsLogin(protocol, username, password, hostname, port, repository)
        
    def vcsLogout(self):
        """
        Private slot used to logout from a remote repository.
        """
        dlg = vcsLogoutDialog()
        if dlg.exec_loop() == QDialog.Accepted:
            vcsSystem, protocol, username, hostname, port, repository = dlg.getData()
            vcs = self.initVCS(vcsSystem)
            if vcs is not None:
                vcs.vcsLogout(protocol, username, hostname, port, repository)
        
    def vcsRevert(self):
        """
        Private slot used to revert changes made to the local project.
        """
        self.vcs.vcsRevert(self.ppath)
        
    def vcsSwitch(self):
        """
        Private slot used to switch the local project to another tag/branch.
        """
        self.vcs.vcsSwitch(self.ppath)
        
    def vcsMerge(self):
        """
        Private slot used to merge changes of a tag/revision into the local project.
        """
        self.vcs.vcsMerge(self.ppath)
        
    def vcsCleanup(self):
        """
        Private slot used to cleanup the local project.
        """
        self.vcs.vcsCleanup(self.ppath)
        
    def vcsCommand(self):
        """
        Private slot used to execute an arbitrary vcs command.
        """
        self.vcs.vcsCommandLine(self.ppath)
        
    #########################################################################
    ## Subversion specific methods below
    #########################################################################
    
    def svnResolve(self):
        """
        Private slot used to resolve conflicts of the local project.
        """
        self.vcs.svnResolve(self.ppath)
        
    def svnPropList(self):
        """
        Private slot used to list the properties of the project files.
        """
        self.vcs.svnListProps(self.ppath, 1)
        
    def svnPropSet(self):
        """
        Private slot used to set a property for the project files.
        """
        self.vcs.svnSetProp(self.ppath, 1)
        
    def svnPropDel(self):
        """
        Private slot used to delete a property for the project files.
        """
        self.vcs.svnDelProp(self.ppath, 1)
        
    def svnTagList(self):
        """
        Private slot used to list the tags of the project.
        """
        self.vcs.svnListTagBranch(self.ppath, 1)
        
    def svnBranchList(self):
        """
        Private slot used to list the branches of the project.
        """
        self.vcs.svnListTagBranch(self.ppath, 0)
        
    #########################################################################
    ## Below is the interface to the checker tools
    #########################################################################
    
    def handleTabnanny(self):
        """
        Private slot used to check the project files for bad indentations.
        """
        self.saveAllScripts()
        files = [os.path.join(self.ppath, file) \
            for file in self.pdata["SOURCES"] \
                if file.endswith(".py") or file.endswith(".ptl")]
        self.tabnanny = TabnannyDialog(qApp.mainWidget().getViewManager())
        self.tabnanny.show()
        self.tabnanny.start(files)
    
    def handleSyntaxCheck(self):
        """
        Private slot used to check the project files for bad indentations.
        """
        self.saveAllScripts()
        files = [os.path.join(self.ppath, file) \
            for file in self.pdata["SOURCES"] \
                if file.endswith(".py") or file.endswith(".ptl")]
        self.syntaxcheck = SyntaxCheckerDialog(qApp.mainWidget().getViewManager())
        self.syntaxcheck.show()
        self.syntaxcheck.start(files)

    def handleCodeMetrics(self):
        """
        Private slot used to calculate some code metrics for the project files.
        """
        files = [os.path.join(self.ppath, file) \
            for file in self.pdata["SOURCES"] if file.endswith(".py")]
        self.codemetrics = CodeMetricsDialog()
        self.codemetrics.show()
        self.codemetrics.start(files)

    def handleCodeCoverage(self):
        """
        Private slot used to show the code coverage information for the project files.
        """
        fn = self.getMainScript(1)
        if fn is None:
            KQMessageBox.critical(self.ui,
                self.trUtf8("Coverage Data"),
                self.trUtf8("There is no main script defined for the"
                    " current project. Aborting"),
                self.trUtf8("&OK"))
            return
        
        tfn = Utilities.getTestFileName(fn)
        basename = os.path.splitext(fn)[0]
        tbasename = os.path.splitext(tfn)[0]
        
        # determine name of coverage file to be used
        files = []
        f = "%s.coverage" % basename
        tf = "%s.coverage" % tbasename
        if os.path.isfile(f):
            files.append(f)
        if os.path.isfile(tf):
            files.append(tf)
        
        if files:
            if len(files) > 1:
                filelist = QStringList()
                for f in files:
                    filelist.append(f)
                fn, ok = KQInputDialog.getItem(\
                    self.trUtf8("Code Coverage"),
                    self.trUtf8("Please select a coverage file"),
                    filelist,
                    0, 0)
                if not ok:
                    return
                fn = unicode(fn)
            else:
                fn = files[0]
        else:
            return
        
        files = [os.path.join(self.ppath, file) \
            for file in self.pdata["SOURCES"] if file.endswith(".py")]
        self.codecoverage = PyCoverageDialog()
        self.codecoverage.show()
        self.codecoverage.start(fn, files)

    def handleProfileData(self):
        """
        Private slot used to show the profiling information for the project.
        """
        fn = self.getMainScript(1)
        if fn is None:
            KQMessageBox.critical(self.ui,
                self.trUtf8("Profile Data"),
                self.trUtf8("There is no main script defined for the"
                    " current project. Aborting"),
                self.trUtf8("&OK"))
            return
        
        tfn = Utilities.getTestFileName(fn)
        basename = os.path.splitext(fn)[0]
        tbasename = os.path.splitext(tfn)[0]
        
        # determine name of profile file to be used
        files = []
        f = "%s.profile" % basename
        tf = "%s.profile" % tbasename
        if os.path.isfile(f):
            files.append(f)
        if os.path.isfile(tf):
            files.append(tf)
        
        if files:
            if len(files) > 1:
                filelist = QStringList()
                for f in files:
                    filelist.append(f)
                fn, ok = KQInputDialog.getItem(\
                    self.trUtf8("Profile Data"),
                    self.trUtf8("Please select a profile file"),
                    filelist,
                    0, 0)
                if not ok:
                    return
                fn = unicode(fn)
            else:
                fn = files[0]
        else:
            return
        
        self.profiledata = PyProfileDialog()
        self.profiledata.show()
        self.profiledata.start(fn)
        
    def handleCyclopsReport(self):
        """
        Private slot used to show the Cyclops report for the project.
        """
        fn = self.getMainScript(1)
        if fn is None:
            KQMessageBox.critical(self.ui,
                self.trUtf8("Cyclops Report"),
                self.trUtf8("There is no main script defined for the"
                    " current project. Aborting"),
                self.trUtf8("&OK"))
            return
        
        files = []
        tfn = Utilities.getTestFileName(fn)
        basename = os.path.splitext(fn)[0]
        tbasename = os.path.splitext(tfn)[0]
        
        f = "%s.cycles.html" % basename
        tf = "%s.cycles.html" % tbasename
        if os.path.isfile(f):
            files.append(f)
        if os.path.isfile(tf):
            files.append(tf)
        
        if files:
            if len(files) > 1:
                filelist = QStringList()
                for f in files:
                    filelist.append(f)
                fn, ok = KQInputDialog.getItem(\
                    self.trUtf8("Cyclops Report"),
                    self.trUtf8("Please select a Cyclops report file"),
                    filelist,
                    0, 0)
                if not ok:
                    return
                fn = unicode(fn)
            else:
                fn = files[0]
        else:
            return
            
        self.parent().launchHelpViewer(fn)
        
    def handleRemoveCyclopsReport(self):
        """
        Private slot used to remove the Cyclops report for the project.
        """
        fn = self.getMainScript(1)
        if fn is None:
            KQMessageBox.critical(self.ui,
                self.trUtf8("Remove Cyclops Report"),
                self.trUtf8("There is no main script defined for the"
                    " current project. Aborting"),
                self.trUtf8("&OK"))
            return
        
        files = []
        tfn = Utilities.getTestFileName(fn)
        basename = os.path.splitext(fn)[0]
        tbasename = os.path.splitext(tfn)[0]
        
        f = "%s.cycles.html" % basename
        tf = "%s.cycles.html" % tbasename
        if os.path.isfile(f):
            files.append(f)
        if os.path.isfile(tf):
            files.append(tf)
        
        if files:
            if len(files) > 1:
                filelist = QStringList()
                for f in files:
                    filelist.append(f)
                fn, ok = KQInputDialog.getItem(\
                    self.trUtf8("Remove Cyclops Report"),
                    self.trUtf8("Please select a Cyclops report file to be removed"),
                    filelist,
                    0, 0)
                if not ok:
                    return
                fn = unicode(fn)
            else:
                fn = files[0]
        else:
            return
            
        if os.path.exists(fn):
            os.remove(fn)
        
    def handleShowShowMenu(self):
        """
        Private slot called before the show menu is shown.
        """
        fn = self.getMainScript(1)
        if fn is not None:
            tfn = Utilities.getTestFileName(fn)
            basename = os.path.splitext(fn)[0]
            tbasename = os.path.splitext(tfn)[0]
            self.codeProfileAct.setEnabled(\
                os.path.isfile("%s.profile" % basename) or \
                os.path.isfile("%s.profile" % tbasename))
            self.codeCoverageAct.setEnabled(\
                os.path.isfile("%s.coverage" % basename) or \
                os.path.isfile("%s.coverage" % tbasename))
            self.codeCyclopsAct.setEnabled(\
                os.path.isfile("%s.cycles.html" % basename) or \
                os.path.isfile("%s.cycles.html" % tbasename))
            self.removeCyclopsAct.setEnabled(\
                os.path.isfile("%s.cycles.html" % basename) or \
                os.path.isfile("%s.cycles.html" % tbasename))
        else:
            self.codeProfileAct.setEnabled(0)
            self.codeCoverageAct.setEnabled(0)
            self.codeCyclopsAct.setEnabled(0)
            self.removeCyclopsAct.setEnabled(0)
        
    #########################################################################
    ## Below is the interface to the diagrams
    #########################################################################
    
    def handleApplicationDiagram(self):
        """
        Private method to handle the application diagram context menu action.
        """
        dlg = ApplicationDiagram(self, self.parent())
        dlg.show()
