#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (c) 2003 - 2005 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Eric3 Documentation Generator

This is the main Python script of the documentation generator. It is
this script that gets called via the source documentation interface.
This script can be used via the commandline as well.
"""

import glob
import os
import sys

import Utilities.ModuleParser
from DocumentationTools.ModuleDocumentor import ModuleDocument
from DocumentationTools.IndexGenerator import IndexGenerator
from UI.Info import Version
import Utilities

def usage():
    """
    Function to print some usage information.
    
    It prints a reference of all commandline parameters that may
    be used and ends the application.
    """
    print "eric3-doc (c) 2003-2004 by Detlev Offenbach <detlev@die-offenbachs.de>."
    print
    print "Usage:"
    print
    print "  eric3-doc [options] files..."
    print
    print "where files can be either python modules, package"
    print "directories or ordinary directories."
    print
    print "Options:"
    print
    print "  -p prefix or --prefix=prefix"
    print "        Prepend given prefix to file names."
    print "  -o directory or --outdir=directory"
    print "        Generate files in the named directory."
    print "  -R, -r or --recursive"
    print "        Perform a recursive search for Python files."
    print "  -x directory or --exclude=directory"
    print "        Specify a directory basename to be excluded."
    print "        This option may be repeated multiple times."
    print "  -i or --noindex"
    print "        Don't generate index files."
    print "  -e or --noempty"
    print "        Don't include empty modules."
    print "  -V or --version"
    print "        Show version information and exit."
    print "  -h or --help"
    print "        Show this help and exit."
    sys.exit(1)

def version():
    """
    Function to show the version information.
    """
    print \
"""eric3-doc  %s

Eric3 API documentation generator.

Copyright (c) 2003-2005 Detlev Offenbach <detlev@die-offenbachs.de>
This is free software; see the LICENSE.GPL for copying conditions.
There is NO warranty; not even for MERCHANTABILITY or FITNESS FOR A
PARTICULAR PURPOSE.""" % Version
    sys.exit(1)

def main():
    """
    Main entry point into the application.
    """

    import getopt

    try:
        opts, args = getopt.getopt(sys.argv[1:], "ehio:p:RrVx:",
            ["exclude=", "help", "noindex", "noempty", "outdir=",
             "prefix=", "recursive", "version"])
    except getopt.error:
        usage()

    prefix = "eric3doc"
    excludeDirs = ["CVS", ".svn", "dist", "build", "doc", "docs"]
    outputDir = "doc"
    recursive = 0
    doIndex = 1
    noempty = 0

    for k, v in opts:
        if k in ["-p", "--prefix"]:
            prefix = v
        elif k in ["-o", "--outdir"]:
            outputDir = v
        elif k in ["-R", "-r", "--recursive"]:
            recursive = 1
        elif k in ["-x", "--exclude"]:
            excludeDirs.append(v)
        elif k in ["-i", "--noindex"]:
            doIndex = 0
        elif k in ["-e", "--noempty"]:
            noempty = 1
        elif k in ["-h", "--help"]:
            usage()
        elif k in ["-V", "--version"]:
            version()

    if not args:
        usage()

    input = output = 0
    basename = ""

    if outputDir:
        if not os.path.isdir(outputDir):
            try:
                os.makedirs(outputDir)
            except:
                sys.stderr.write("Could not create output directory %s." % outputDir)
                sys.exit(2)
    else:
        outputDir = os.getcwd()
    outputDir = os.path.abspath(outputDir)
        
    indexGenerator = IndexGenerator(outputDir, prefix)
    
    for arg in args:
        if os.path.isdir(arg):
            if os.path.exists(os.path.join(arg, Utilities.joinext("__init__", ".py"))):
                basename = os.path.dirname(arg)
                if arg == '.':
                    sys.stderr.write("The directory '.' is a package.\n")
                    sys.stderr.write("Please repeat the call giving its real name.\n")
                    sys.stderr.write("Ignoring the directory.\n")
                    continue
            else:
                basename = arg
            if basename:
                basename = "%s%s" % (basename, os.sep)
                
            if recursive and not os.path.islink(arg):
                names = [arg] + Utilities.getDirs(arg, excludeDirs)
            else:
                names = [arg]
        else:
            basename = ""
            names = [arg]
    
        for filename in names:
    
            if os.path.isdir(filename):
                files = glob.glob(os.path.join(filename, Utilities.joinext("*", ".py"))) + \
                        glob.glob(os.path.join(filename, Utilities.joinext("*", ".ptl")))
                initFile = os.path.join(filename, Utilities.joinext("__init__", ".py"))
                if initFile in files:
                    files.remove(initFile)
                    files.insert(0, initFile)
            else:
                if sys.platform == "win32" and glob.has_magic(filename):
                    files = glob.glob(filename)
                else:
                    files = [filename]
    
            for file in files:
    
                try:
                    module = Utilities.ModuleParser.readModule(file, basename=basename)
                    moduleDocument = ModuleDocument(module)
                    doc = moduleDocument.genDocument()
                except IOError, v:
                    sys.stderr.write("%s error: %s\n" % (file, v[1]))
                    continue
                except ImportError, v:
                    sys.stderr.write("%s error: %s\n" % (file, v))
                    continue
    
                input = input + 1
    
                f = Utilities.joinext(os.path.join(outputDir, "%s-%s" % \
                    (prefix, moduleDocument.name())), ".html")
    
                # remember for index file generation
                indexGenerator.remember(file, moduleDocument, basename)
                if (noempty or file.endswith('__init__.py')) \
                   and moduleDocument.isEmpty():
                    continue
                    
                # generate output
                try:
                    out = open(f, "wb")
                    out.write(doc)
                    out.close()
                except IOError, v:
                    sys.stderr.write("%s error: %s\n" % (file, v[1]))
                else:
                    sys.stdout.write("%s ok\n" % f)
    
                output = output + 1

    # write index files
    if doIndex:
        indexGenerator.writeIndices(basename)

    sys.exit(0)

if __name__ == '__main__':
    main()
