# -*- coding: utf-8 -*-

# Copyright (c) 2004 - 2005 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Compatibility module to use the KDE Input Dialog instead of the Qt Input Dialog.
"""

try:
    import os
    if os.getenv("e3nokde"):
        raise ImportError
        
    from kdeui import KInputDialog, KPasswordDialog
    from qt import QLineEdit, QCString, QString
    
    def getText(caption, label, mode = QLineEdit.Normal, text = QString.null,
                parent = None, name = None):
        """
        Function to get some text from the user.
        
        @param caption caption of the dialog
        @param label text of the label for the line edit
        @param mode mode of the line edit (ignored)
        @param text initial text of the line edit
        @param parent parent widget of the dialog
        @param name name of the dialog
        @return tuple of (text, ok). text contains the text entered by the
            user, ok indicates a valid input.
        """
        if mode == QLineEdit.Normal:
            return KInputDialog.getText(caption, label, text, parent, name)
        else:
            password = QCString()
            res = KPasswordDialog.getPassword(password, label)[0]
            res = res == KPasswordDialog.Accepted
            return QString(password), res
    
    def getInteger(caption, label, value = 0, minValue = -2147483647, 
                   maxValue = 2147483647, step = 1, parent = None, name = None):
        """
        Function to get an integer value from the user.
        
        @param caption caption of the dialog
        @param label text of the label for the line edit
        @param value initial value of the spin box
        @param minValue minimal value of the spin box
        @param maxValue maximal value of the spin box
        @param step step size of the spin box
        @param parent parent widget of the dialog
        @param name name of the dialog
        @return tuple of (value, ok). value contains the integer entered by the
            user, ok indicates a valid input.
        """
        try:
            return KInputDialog.getInteger(caption, label, value, minValue, maxValue, step,
                                           parent, name)
        except TypeError:
            return KInputDialog.getInteger(caption, label, value, minValue, maxValue, step,
                                           10, parent, name)
    
    def getDouble(caption, label, value = 0, minValue = -2147483647, 
                  maxValue = 2147483647, decimals = 1, parent = None, name = None):
        """
        Function to get a double value from the user.
        
        @param caption caption of the dialog
        @param label text of the label for the line edit
        @param value initial value of the line edit
        @param minValue minimum value the user may choose
        @param maxValue maximum value the user may choose
        @param decimals maximum number of decimals the value may have
        @param parent parent widget of the dialog
        @param name name of the dialog
        @return tuple of (value, ok). value contains the double entered by the
            user, ok indicates a valid input.
        """
        try:
            return KInputDialog.getDouble(caption, label, value, minValue, maxValue, decimals,
                                           parent, name)
        except TypeError:
            return KInputDialog.getDouble(caption, label, value, minValue, maxValue, 0.1,
                                           decimals, parent, name)
    
    def getItem(caption, label, slist, current = 0, editable = 1, parent = None,
                name = None):
        """
        Function to get an item of a list from the user.
        
        @param caption caption of the dialog
        @param label text of the label for the line edit
        @param slist list of strings to select from (QStringList)
        @param current number of item, that should be selected as a default
        @param editable indicates whether the user can input their own text
        @param parent parent widget of the dialog
        @param name name of the dialog
        @return tuple of (value, ok). value contains the double entered by the
            user, ok indicates a valid input.
        """
        return KInputDialog.getItem(caption, label, slist, current, editable, parent,
                                    name)
    
except (ImportError, RuntimeError):
    import qt
    
    getText = qt.QInputDialog.getText
    getInteger = qt.QInputDialog.getInteger
    getDouble = qt.QInputDialog.getDouble
    getItem = qt.QInputDialog.getItem
