# -*- coding: utf-8 -*-

# Copyright (c) 2002 - 2006 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing a dialog to add files of a directory to the project.
"""

from qt import *

from KdeQt import KQFileDialog

from AddDirectoryForm import AddDirForm

class AddDirectoryDialog(AddDirForm):
    """
    Class implementing a dialog to add files of a directory to the project.
    """
    def __init__(self,pro,filter='py',parent = None,name = None,
                 modal = 0,fl = 0,startdir = None):
        """
        Constructor
        
        @param pro reference to the project object
        @param filter file type filter (string or QString)
        @param parent parent widget of this dialog (QWidget)
        @param name name of this dialog (string or QString)
        @param modal flag for a modal dialog (boolean)
        @param fl window flags
        @param startdir start directory for the selection dialog
        """
        AddDirForm.__init__(self,parent,name,1,fl)
        
        self.targetDirEdit.setText(pro.ppath)
        self.ppath = pro.ppath
        self.startdir = startdir
        self.handleFileTypeSelected(QString('(*.py)')) # enable all dialog elements
        filter = unicode(filter)
        if filter == 'py':  # it is a source file
            if pro.pdata["MIXEDLANGUAGE"][0]:
                sourceExtensions = pro.sourceExtensions["Mixed"]
                sourceFilters = pro.sourceFilters["Mixed"]
            else:
                sourceExtensions = pro.sourceExtensions[pro.pdata["PROGLANGUAGE"][0]]
                sourceFilters = pro.sourceFilters[pro.pdata["PROGLANGUAGE"][0]]
            filterList = QStringList.split(';;', QString(sourceFilters))
            filter = sourceExtensions[0][1:]
        elif filter == 'ui':
            filterList = QStringList.split(';;', self.trUtf8(\
                            "Qt User-Interface Files (*.ui);;"))
        elif filter == 'idl':
            filterList = QStringList.split(';;', self.trUtf8(\
                            "Interface Files (*.idl);;"))
        elif filter == 'others':
            filterList = QStringList.split(';;', self.trUtf8(\
                            "Other Files (*);;"))
            self.handleFileTypeSelected(QString('(*)'))
        else:
            if pro.pdata["MIXEDLANGUAGE"][0]:
                filters = pro.sourceFilters["Mixed"]
            else:
                filters = pro.sourceFilters[pro.pdata["PROGLANGUAGE"][0]]
            filters.append(self.trUtf8(\
                    "Qt User-Interface Files (*.ui);;Interface Files (*.idl);;"
                    "Other Files (*)"))
            filterList = QStringList.split(';;', filters)
        self.filterComboBox.insertStringList(filterList)
        fi = '(*.%s)' % str(filter)
        index = 0
        for filterItm in filterList:
            if filterItm.endsWith(fi):
                break
            index += 1
        self.filterComboBox.setCurrentItem(index)
        
    def handleFileTypeSelected(self, fileType):
        """
        Private slot to handle the selection of a file type.
        
        @param fileType the selected file type (QString)
        """
        if fileType.endsWith('(*)'):
            self.targetDirLabel.setEnabled(0)
            self.targetDirEdit.setEnabled(0)
            self.targetDirButton.setEnabled(0)
            self.recursiveCheckBox.setEnabled(0)
        else:
            self.targetDirLabel.setEnabled(1)
            self.targetDirEdit.setEnabled(1)
            self.targetDirButton.setEnabled(1)
            self.recursiveCheckBox.setEnabled(1)
        
    def handleDirDialog(self, textEdit):
        """
        Private slot to display a directory selection dialog.
        
        @param textEdit field for the display of the selected directory name
                (QLineEdit)
        """
        startdir = textEdit.text()
        if startdir.isEmpty() and self.startdir is not None:
            startdir = self.startdir
        
        directory = KQFileDialog.getExistingDirectory(startdir,
            self, None, self.trUtf8("Select directory"), 1)
            
        if not directory.isNull():
            textEdit.setText(QDir.convertSeparators(directory))
            
    def handleSDirDialog(self):
        """
        Private slot to handle the source dir button press.
        """
        self.handleDirDialog(self.sourceDirEdit)
        
    def handleTDirDialog(self):
        """
        Private slot to handle the target dir button press.
        """
        self.handleDirDialog(self.targetDirEdit)
        
    def handleSTextChanged(self, dir):
        """
        Private slot to handle the source dir text changed.
        
        If the entered source directory is a subdirectory of the current
        projects main directory, the target directory path is synchronized.
        It is assumed, that the user wants to add a bunch of files to
        the project in place.
        
        @param dir the text of the source directory line edit
        """
        if dir.startsWith(self.ppath):
            self.targetDirEdit.setText(dir)
            
    def getData(self):
        """
        Public slot to retrieve the dialogs data.
        
        @return tuple of four values (string, string, string, boolean) giving
            the selected file extension, the source and target directory and
            a flag indicating a recursive add
        """
        selectedFilter = self.filterComboBox.currentText()
        ext = selectedFilter.section('(*.',1,1).section(')',0,0)
        return (unicode(ext), unicode(self.sourceDirEdit.text()), 
            unicode(self.targetDirEdit.text()),
            self.recursiveCheckBox.isChecked())
