# -*- coding: utf-8 -*-

# Copyright (c) 2002 - 2006 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing the viewmanager base class.
"""

import os

from qt import *
from qtext import QextScintillaAPIs, QextScintilla

from KdeQt import KQFileDialog, KQMessageBox

import Preferences
from BookmarkedFilesDialog import BookmarkedFilesDialog
from QScintilla.QextScintillaCompat import QSCINTILLA_VERSION
from QScintilla.Editor import Editor
from QScintilla.GotoDialog import GotoDialog
from QScintilla.SearchReplaceDialog import SearchReplaceDialog
from QScintilla.ZoomDialog import ZoomDialog
import Utilities
import UI.PixmapCache

from UI.E3Action import E3Action, E3ActionGroup

class QuickSearchLineEdit(QLineEdit):
    """
    Class implementing a line edit that reacts to the press of the ESC key.
    
    @signal escPressed() emitted after the ESC key has been pressed
    """
    def keyPressEvent(self, evt):
        """
        Re-implemented to handle the press of the ESC key.
        
        @param ev key event (QKeyPressEvent)
        """
        if evt.key() == Qt.Key_Escape:
            self.emit(PYSIGNAL("escPressed()"), ())
        else:
            QLineEdit.keyPressEvent(self, evt)  # pass it on

class ViewManager:
    """
    Base class inherited by all specific viewmanager classes.
    
    It defines the interface to be implemented by specific
    viewmanager classes and all common methods.
    
    @signal lastEditorClosed emitted after the last editor window was closed
    @signal editorOpened(string) emitted after an editor window was opened
    @signal editorClosed(string) emitted just before an editor window gets closed
    @signal editorSaved(string) emitted after an editor window was saved
    @signal checkActions(editor) emitted when some actions should be checked
            for their status
    @signal cursorChanged(editor) emitted after the cursor position of the active
            window has changed
    @signal breakpointToggled(editor) emitted when a breakpoint is toggled.
    @signal breakpointEnabledToggled(editor) emitted when a breakpoint's enabled
            status is toggled
    @signal bookmarkToggled(editor) emitted when a bookmark is toggled.
    """
    def __init__(self, ui, dbs):
        """
        Constructor
        
        @param ui reference to the main user interface
        @param dbs reference to the debug server object
        """
        # initialize the instance variables
        self.ui = ui
        self.editors = []
        self.currentEditor = None
        self.dbs = dbs
        self.untitledCount = 0
        self.srHistory = {"search" : QStringList(), "replace" : QStringList()}
        self.editorsCheckFocusIn = 1
        
        self.recent = QStringList()
        rs, ok = Preferences.Prefs.settings.readListEntry('/eric3/Recent/Sources')
        if ok:
            self.recent = rs
            
        self.bookmarked = QStringList()
        bs, ok = Preferences.Prefs.settings.readListEntry('/eric3/Bookmarked/Sources')
        if ok:
            self.bookmarked = bs
            
        self.searchDlg = SearchReplaceDialog(0, self, ui)
        self.replaceDlg = SearchReplaceDialog(1, self, ui)
        
        # initialize the central store for api information (used by
        # autocompletion and calltips)
        self.apis = {}
        for apiLang in Preferences.Prefs.editorAPIDefaults.keys():
            self.apis[apiLang] = None
            
        self.initFileFilters()
        
        # initialize the autosave timer
        self.autosaveInterval = Preferences.getEditor("AutosaveInterval")
        self.autosaveTimer = QTimer(self, "AutosaveTimer")
        self.connect(self.autosaveTimer, SIGNAL('timeout()'), self.handleAutosave)
        
    def initFileFilters(self):
        """
        Private method to initialize the lists of supported filename filters.
        """
        self.fileFiltersString = self.trUtf8(\
            'Python Files (*.py);;'
            'Pyrex Files (*.pyx);;'
            'Quixote Template Files (*.ptl);;'
            'Ruby Files (*.rb);;'
            'IDL Files (*.idl);;'
            'C Files (*.h *.c);;'
            'C++ Files (*.h *.hpp *.hh *.cxx *.cpp *.cc);;'
            'C# Files (*.cs);;'
            'HTML Files (*.html *.htm *.asp *.shtml);;'
            'CSS Files (*.css);;'
            'PHP Files (*.php *.php3 *.php4 *.php5 *.phtml);;'
            'XML Files (*.xml *.xsl *.xslt *.dtd);;'
            'Java Files (*.java);;'
            'JavaScript Files (*.js);;'
            'SQL Files (*.sql);;'
            'Docbook Files (*.docbook);;'
            'Perl Files (*.pl *.pm *.ph);;'
            'Lua Files (*.lua);;'
            'Tex Files (*.tex *.sty *.aux *.toc *.idx);;'
            'Shell Files (*.sh);;'
            'Batch Files (*.bat *.cmd);;'
            'Diff Files (*.diff *.patch);;'
            'Makefiles (*.mak);;'
            'Properties Files (*.properties *.ini *.inf *.reg *.cfg *.cnf *.rc);;'
            'All Files (*)')

        fileFilters = QStringList.split(';;', self.fileFiltersString)
        
        self.ext2Filter = {}
        for fileFilter in fileFilters:
            extensions = QStringList.split('*', fileFilter)
            for extension in extensions[1:]:
                extension = unicode(extension).strip().replace(')', '')
                if extension:
                    self.ext2Filter[extension] = unicode(fileFilter)
                else:
                    self.ext2Filter["__ALL__"] = unicode(fileFilter)
        
    def setSbInfo(self, sbFile, sbLine, sbPos, sbWritable):
        """
        Public method to transfer statusbar info from the user interface to viewmanager.
        
        @param sbFile reference to the file part of the statusbar
        @param sbLine reference to the line number part of the statusbar
        @param sbPos reference to the character position part of the statusbar
        @param sbWritable reference to the writability indicator part of the statusbar
        """
        self.sbFile = sbFile
        self.sbLine = sbLine
        self.sbPos = sbPos
        self.sbWritable = sbWritable
        self.setSbFile()
        
    def canCascade(self):
        """
        Public method to signal if cascading of managed windows is available.
        
        @return flag indicating cascading of windows is available
        @exception RuntimeError Not implemented
        """
        raise RuntimeError, 'Not implemented'
        
    def canTile(self):
        """
        Public method to signal if tiling of managed windows is available.
        
        @return flag indicating tiling of windows is available
        @exception RuntimeError Not implemented
        """
        raise RuntimeError, 'Not implemented'
        
    def tile(self):
        """
        Public method to tile the managed windows.
        
        @exception RuntimeError Not implemented
        """
        raise RuntimeError, 'Not implemented'
    
    def cascade(self):
        """
        Public method to cascade the managed windows.
        
        @exception RuntimeError Not implemented
        """
        raise RuntimeError, 'Not implemented'

    def removeAllViews(self):
        """
        Private method to remove all views (i.e. windows)
        
        @exception RuntimeError Not implemented
        """
        raise RuntimeError, 'Not implemented'
    
    def removeView(self, win):
        """
        Private method to remove a view (i.e. window)
        
        @param win editor window to be removed
        @exception RuntimeError Not implemented
        """
        raise RuntimeError, 'Not implemented'
    
    def addView(self, win, fn=None):
        """
        Private method to add a view (i.e. window)
        
        @param win editor window to be added
        @param fn filename of this editor
        @exception RuntimeError Not implemented
        """
        raise RuntimeError, 'Not implemented'
    
    def showView(self, win, fn=None):
        """
        Private method to show a view (i.e. window)
        
        @param win editor window to be shown
        @param fn filename of this editor
        @exception RuntimeError Not implemented
        """
        raise RuntimeError, 'Not implemented'
    
    def activeWindow(self):
        """
        Private method to return the active (i.e. current) window.
        
        @return reference to the active editor
        @exception RuntimeError Not implemented
        """
        raise RuntimeError, 'Not implemented'
    
    def handleShowWindowMenu(self, windowMenu):
        """
        Private method to set up the viewmanager part of the Window menu.
        
        @param windowMenu reference to the window menu
        @exception RuntimeError Not implemented
        """
        raise RuntimeError, 'Not implemented'
    
    def initWindowActions(self):
        """
        Define the user interface actions for window handling.
        
        @exception RuntimeError Not implemented
        """
        raise RuntimeError, 'Not implemented'

    def setEditorName(self, editor, newName):
        """
        Change the displayed name of the editor.
        
        @param editor editor window to be changed
        @param newName new name to be shown (string or QString)
        @exception RuntimeError Not implemented
        """
        raise RuntimeError, 'Not implemented'
        
    def handleModificationStatusChanged(self, m, editor):
        """
        Private slot to handle the modificationStatusChanged signal.
        
        @param m flag indicating the modification status (boolean)
        @param editor editor window changed
        @exception RuntimeError Not implemented
        """
        raise RuntimeError, 'Not implemented'
        
    #####################################################################
    ## methods above need to be implemented by a subclass
    #####################################################################
    
    def canSplit(self):
        """
        public method to signal if splitting of the view is available.
        
        @return flag indicating splitting of the view is available.
        """
        return 0
        
    def addSplit(self):
        """
        Public method used to split the current view.
        """
        pass
        
    def removeSplit(self):
        """
        Public method used to remove the current split view.
        
        @return Flag indicating successful deletion
        """
        return 0
        
    def setSplitOrientation(self, orientation):
        """
        Public method used to set the orientation of the split view.
        
        @param orientation orientation of the split
                (QSplitter.Horizontal or QSplitter.Vertical)
        """
        pass
        
    def nextSplit(self):
        """
        Public slot used to move to the next split.
        """
        pass
        
    def prevSplit(self):
        """
        Public slot used to move to the previous split.
        """
        pass
        
    def eventFilter(self, object, event):
        """
        Private method called to filter an event.
        
        @param object object, that generated the event (QObject)
        @param event the event, that was generated by object (QEvent)
        @return flag indicating if event was filtered out
        """
        return 0
        
    def focusInEvent(self, event):
        """
        Public method called when the viewmanager receives focus.
        
        @param event the event object (QFocusEvent)
        """
        self.editorActGrp.setEnabled(1)
        
    def focusOutEvent(self, event):
        """
        Public method called when the viewmanager loses focus.
        
        @param event the event object (QFocusEvent)
        """
        self.editorActGrp.setEnabled(0)
        
    #####################################################################
    ## methods above need to be implemented by a subclass, that supports
    ## splitting of the viewmanager area.
    #####################################################################
    
    def initActions(self):
        """
        Public method defining the user interface actions.
        """
        # list containing all edit actions
        self.editActions = []
        
        # list containing all file actions
        self.fileActions = []
        
        # list containing all search actions
        self.searchActions = []
        
        # list containing all view actions
        self.viewActions = []
        
        # list containing all window actions
        self.windowActions = []
        
        # list containing all macro actions
        self.macroActions = []
        
        # list containing all bookmark actions
        self.bookmarkActions = []
        
        self.initWindowActions()
        self.initFileActions()
        self.initEditActions()
        self.initSearchActions()
        self.initViewActions()
        self.initMacroActions()
        self.initBookmarkActions()
        
    ##################################################################
    ## Initialize the file related actions, file menu and toolbar
    ##################################################################
    
    def initFileActions(self):
        """
        Private method defining the user interface actions for file handling.
        """
        self.newAct = E3Action(self.trUtf8('New'),
                QIconSet(UI.PixmapCache.getPixmap("new.png")),
                self.trUtf8('&New'),
                QKeySequence(self.trUtf8("CTRL+N","File|New")),0,
                self,'vm_file_new')
        self.newAct.setStatusTip(self.trUtf8('Open an empty editor window'))
        self.newAct.setWhatsThis(self.trUtf8(
            """<b>New</b>"""
            """<p>An empty editor window will be created.</p>"""
        ))
        self.newAct.connectIt(SIGNAL('activated()'),self.newEditor)
        self.fileActions.append(self.newAct)
        
        self.openAct = E3Action(self.trUtf8('Open'),
                QIconSet(UI.PixmapCache.getPixmap("open.png")),
                self.trUtf8('&Open...'),
                QKeySequence(self.trUtf8("CTRL+O","File|Open")),0,
                self,'vm_file_open')
        self.openAct.setStatusTip(self.trUtf8('Open a file'))
        self.openAct.setWhatsThis(self.trUtf8(
            """<b>Open a file</b>"""
            """<p>You will be asked for the name of a file to be opened"""
            """ in an editor window.</p>"""
        ))
        self.openAct.connectIt(SIGNAL('activated()'),self.handleOpen)
        self.fileActions.append(self.openAct)
        
        self.closeActGrp = E3ActionGroup(self)
        
        self.closeAct = E3Action(self.trUtf8('Close'),
                QIconSet(UI.PixmapCache.getPixmap("close.png")),
                self.trUtf8('&Close'),
                QKeySequence(self.trUtf8("CTRL+W","File|Close")),0,
                self.closeActGrp,'vm_file_close')
        self.closeAct.setStatusTip(self.trUtf8('Close the current window'))
        self.closeAct.setWhatsThis(self.trUtf8(
            """<b>Close Window</b>"""
            """<p>Close the current window.</p>"""
        ))
        self.closeAct.connectIt(SIGNAL('activated()'),self.handleClose)
        self.fileActions.append(self.closeAct)

        self.closeAllAct = E3Action(self.trUtf8('Close All'),
                self.trUtf8('Clos&e All'),
                0,0,self.closeActGrp,'vm_file_close_all')
        self.closeAllAct.setStatusTip(self.trUtf8('Close all editor windows'))
        self.closeAllAct.setWhatsThis(self.trUtf8(
            """<b>Close All Windows</b>"""
            """<p>Close all editor windows.</p>"""
        ))
        self.closeAllAct.connectIt(SIGNAL('activated()'),self.handleCloseAll)
        self.fileActions.append(self.closeAllAct)
        
        self.closeActGrp.setEnabled(0)
        
        self.saveActGrp = E3ActionGroup(self)

        self.saveAct = E3Action(self.trUtf8('Save'),
                QIconSet(UI.PixmapCache.getPixmap("fileSave.png")),
                self.trUtf8('&Save'),
                QKeySequence(self.trUtf8("CTRL+S","File|Save")),0,
                self.saveActGrp,'vm_file_save')
        self.saveAct.setStatusTip(self.trUtf8('Save the current file'))
        self.saveAct.setWhatsThis(self.trUtf8(
            """<b>Save File</b>"""
            """<p>Save the contents of current editor window.</p>"""
        ))
        self.saveAct.connectIt(SIGNAL('activated()'),self.saveCurrentEditor)
        self.fileActions.append(self.saveAct)

        self.saveAsAct = E3Action(self.trUtf8('Save as'),
                QIconSet(UI.PixmapCache.getPixmap("fileSaveAs.png")),
                self.trUtf8('Save &as...'),
                QKeySequence(self.trUtf8("SHIFT+CTRL+S","File|Save As")),0,
                self.saveActGrp,'vm_file_save_as')
        self.saveAsAct.setStatusTip(self.trUtf8('Save the current file to a new one'))
        self.saveAsAct.setWhatsThis(self.trUtf8(
            """<b>Save File as</b>"""
            """<p>Save the contents of current editor window to a new file."""
            """ The file can be entered in a file selection dialog.</p>"""
        ))
        self.saveAsAct.connectIt(SIGNAL('activated()'),self.saveAsCurrentEditor)
        self.fileActions.append(self.saveAsAct)

        self.saveAllAct = E3Action(self.trUtf8('Save all'),
                QIconSet(UI.PixmapCache.getPixmap("fileSaveAll.png")),
                self.trUtf8('Save a&ll...'),
                0,0,self.saveActGrp,'vm_file_save_all')
        self.saveAllAct.setStatusTip(self.trUtf8('Save all files'))
        self.saveAllAct.setWhatsThis(self.trUtf8(
            """<b>Save All Files</b>"""
            """<p>Save the contents of all editor windows.</p>"""
        ))
        self.saveAllAct.connectIt(SIGNAL('activated()'),self.saveAllEditors)
        self.fileActions.append(self.saveAllAct)
        
        self.saveActGrp.setEnabled(0)

        self.saveToProjectAct = E3Action(self.trUtf8('Save to Project'),
                QIconSet(UI.PixmapCache.getPixmap("fileSaveProject.png")),
                self.trUtf8('Save to Pro&ject'),
                0,0,self,'vm_file_save_to_project')
        self.saveToProjectAct.setStatusTip(self.trUtf8('Save the current file to the current project'))
        self.saveToProjectAct.setWhatsThis(self.trUtf8(
            """<b>Save to Project</b>"""
            """<p>Save the contents of the current editor window to the"""
            """ current project. After the file has been saved, it is"""
            """ automatically added to the current project.</p>"""
        ))
        self.saveToProjectAct.connectIt(SIGNAL('activated()'),self.saveCurrentEditorToProject)
        self.saveToProjectAct.setEnabled(0)
        self.fileActions.append(self.saveToProjectAct)
        
        self.printAct = E3Action(self.trUtf8('Print'),
                QIconSet(UI.PixmapCache.getPixmap("print.png")),
                self.trUtf8('&Print'),
                QKeySequence(self.trUtf8("CTRL+P","File|Print")),0,
                self,'vm_file_print')
        self.printAct.setStatusTip(self.trUtf8('Print the current file'))
        self.printAct.setWhatsThis(self.trUtf8(
            """<b>Print File</b>"""
            """<p>Print the contents of current editor window.</p>"""
        ))
        self.printAct.connectIt(SIGNAL('activated()'),self.printCurrentEditor)
        self.printAct.setEnabled(0)
        self.fileActions.append(self.printAct)

        self.printSelAct = E3Action(self.trUtf8('Print Selection'),
                QIconSet(UI.PixmapCache.getPixmap("print.png")),
                self.trUtf8('Prin&t Selection'),
                QKeySequence(self.trUtf8("SHIFT+CTRL+P","File|Print")),0,
                self,'vm_file_print_selection')
        self.printSelAct.setStatusTip(self.trUtf8('Print the selection of the current file'))
        self.printSelAct.setWhatsThis(self.trUtf8(
            """<b>Print Selection</b>"""
            """<p>Print the selection of the current editor window.</p>"""
        ))
        self.printSelAct.connectIt(SIGNAL('activated()'),self.printCurrentEditorSel)
        self.printSelAct.setEnabled(0)
        self.fileActions.append(self.printSelAct)

        self.findFileNameAct = E3Action(self.trUtf8('Search File'),
                self.trUtf8('Search &File...'),
                QKeySequence(self.trUtf8("ALT+CTRL+F","File|Search File")),0,
                self,'vm_file_search_file')
        self.findFileNameAct.setStatusTip(self.trUtf8('Search for a file'))
        self.findFileNameAct.setWhatsThis(self.trUtf8(
            """<b>Search File</b>"""
            """<p>Search for a file.</p>"""
        ))
        self.findFileNameAct.connectIt(SIGNAL('activated()'),self.handleFindFileName)
        self.fileActions.append(self.findFileNameAct)
        
    def initFileMenu(self):
        """
        Public method to create the File menu.
        
        @return the generated menu
        """
        menu = QPopupMenu(self.ui)
        self.recentMenu = QPopupMenu(menu)
        self.bookmarkedMenu = QPopupMenu(menu)
        menu.insertTearOffHandle()
        self.newAct.addTo(menu)
        self.openAct.addTo(menu)
        menu.insertItem(self.trUtf8('Open &Recent Files'), self.recentMenu)
        self.connect(self.recentMenu,SIGNAL('aboutToShow()'),self.handleShowRecentMenu)
        menu.insertItem(self.trUtf8('Open &Bookmarked Files'), self.bookmarkedMenu)
        self.connect(self.bookmarkedMenu,SIGNAL('aboutToShow()'),self.handleShowBookmarkedMenu)
        menu.insertSeparator()
        self.closeAct.addTo(menu)
        self.closeAllAct.addTo(menu)
        menu.insertSeparator()
        self.findFileNameAct.addTo(menu)
        menu.insertSeparator()
        self.saveAct.addTo(menu)
        self.saveAsAct.addTo(menu)
        self.saveAllAct.addTo(menu)
        self.saveToProjectAct.addTo(menu)
        menu.insertSeparator()
        self.printAct.addTo(menu)
        self.printSelAct.addTo(menu)

        return menu
        
    def initFileToolbar(self):
        """
        Public method to create the File toolbar.
        
        @return the generated toolbar
        """
        tb = QToolBar(self.ui)
        self.newAct.addTo(tb)
        self.openAct.addTo(tb)
        self.closeAct.addTo(tb)
        tb.addSeparator()
        self.saveAct.addTo(tb)
        self.saveAsAct.addTo(tb)
        self.saveAllAct.addTo(tb)
        self.saveToProjectAct.addTo(tb)
        tb.addSeparator()
        self.printAct.addTo(tb)
        
        return tb
        
    ##################################################################
    ## Initialize the edit related actions, edit menu and toolbar
    ##################################################################
    
    def initEditActions(self):
        """
        Private method defining the user interface actions for the edit commands.
        """
        self.editActGrp = E3ActionGroup(self)

        self.undoAct = E3Action(self.trUtf8('Undo'),
                QIconSet(UI.PixmapCache.getPixmap("editUndo.png")),
                self.trUtf8('&Undo'),
                QKeySequence(self.trUtf8("CTRL+Z","Edit|Undo")), 
                QKeySequence(self.trUtf8("Alt+Backspace","Edit|Undo")), 
                self.editActGrp, 'vm_edit_undo')
        self.undoAct.setStatusTip(self.trUtf8('Undo the last change'))
        self.undoAct.setWhatsThis(self.trUtf8(
            """<b>Undo</b>"""
            """<p>Undo the last change done in the current editor.</p>"""
        ))
        self.undoAct.connectIt(SIGNAL('activated()'),self.handleEditUndo)
        self.editActions.append(self.undoAct)
        
        self.redoAct = E3Action(self.trUtf8('Redo'),
                QIconSet(UI.PixmapCache.getPixmap("editRedo.png")),
                self.trUtf8('&Redo'),
                QKeySequence(self.trUtf8("CTRL+SHIFT+Z","Edit|Redo")), 0,
                self.editActGrp, 'vm_edit_redo')
        self.redoAct.setStatusTip(self.trUtf8('Redo the last change'))
        self.redoAct.setWhatsThis(self.trUtf8(
            """<b>Redo</b>"""
            """<p>Redo the last change done in the current editor.</p>"""
        ))
        self.redoAct.connectIt(SIGNAL('activated()'),self.handleEditRedo)
        self.editActions.append(self.redoAct)
        
        self.revertAct = E3Action(self.trUtf8('Revert to last saved state'),
                self.trUtf8('Re&vert to last saved state'),
                QKeySequence(self.trUtf8("CTRL+Y","Edit|Revert")), 0,
                self.editActGrp, 'vm_edit_revert')
        self.revertAct.setStatusTip(self.trUtf8('Revert to last saved state'))
        self.revertAct.setWhatsThis(self.trUtf8(
            """<b>Revert to last saved state</b>"""
            """<p>Undo all changes up to the last saved state of the current editor.</p>"""
        ))
        self.revertAct.connectIt(SIGNAL('activated()'),self.handleEditRevert)
        self.editActions.append(self.revertAct)
        
        self.cutAct = E3Action(self.trUtf8('Cut'),
                QIconSet(UI.PixmapCache.getPixmap("editCut.png")),
                self.trUtf8('Cu&t'),
                QKeySequence(self.trUtf8("CTRL+X","Edit|Cut")),
                QKeySequence(self.trUtf8("Shift+Del","Edit|Cut")),
                self.editActGrp, 'vm_edit_cut')
        self.cutAct.setStatusTip(self.trUtf8('Cut the selection'))
        self.cutAct.setWhatsThis(self.trUtf8(
            """<b>Cut</b>"""
            """<p>Cut the selected text of the current editor to the clipboard.</p>"""
        ))
        self.cutAct.connectIt(SIGNAL('activated()'),self.handleEditCut)
        self.editActions.append(self.cutAct)
        
        self.copyAct = E3Action(self.trUtf8('Copy'),
                QIconSet(UI.PixmapCache.getPixmap("editCopy.png")),
                self.trUtf8('&Copy'),
                QKeySequence(self.trUtf8("CTRL+C","Edit|Copy")), 
                QKeySequence(self.trUtf8("CTRL+Ins","Edit|Copy")), 
                self.editActGrp, 'vm_edit_copy')
        self.copyAct.setStatusTip(self.trUtf8('Copy the selection'))
        self.copyAct.setWhatsThis(self.trUtf8(
            """<b>Copy</b>"""
            """<p>Copy the selected text of the current editor to the clipboard.</p>"""
        ))
        self.copyAct.connectIt(SIGNAL('activated()'),self.handleEditCopy)
        self.editActions.append(self.copyAct)
        
        self.pasteAct = E3Action(self.trUtf8('Paste'),
                QIconSet(UI.PixmapCache.getPixmap("editPaste.png")),
                self.trUtf8('&Paste'),
                QKeySequence(self.trUtf8("CTRL+V","Edit|Paste")), 
                QKeySequence(self.trUtf8("Shift+Ins","Edit|Paste")), 
                self.editActGrp, 'vm_edit_paste')
        self.pasteAct.setStatusTip(self.trUtf8('Paste the last cut/copied text'))
        self.pasteAct.setWhatsThis(self.trUtf8(
            """<b>Paste</b>"""
            """<p>Paste the last cut/copied text from the clipboard to"""
            """ the current editor.</p>"""
        ))
        self.pasteAct.connectIt(SIGNAL('activated()'),self.handleEditPaste)
        self.editActions.append(self.pasteAct)
        
        self.deleteAct = E3Action(self.trUtf8('Clear'),
                QIconSet(UI.PixmapCache.getPixmap("editDelete.png")),
                self.trUtf8('Cl&ear'),
                0, 0, self.editActGrp, 'vm_edit_clear')
        self.deleteAct.setStatusTip(self.trUtf8('Clear all text'))
        self.deleteAct.setWhatsThis(self.trUtf8(
            """<b>Clear</b>"""
            """<p>Delete all text of the current editor.</p>"""
        ))
        self.deleteAct.connectIt(SIGNAL('activated()'),self.handleEditDelete)
        self.editActions.append(self.deleteAct)
        
        self.indentAct = E3Action(self.trUtf8('Indent'),
                QIconSet(UI.PixmapCache.getPixmap("editIndent.png")),
                self.trUtf8('&Indent'),
                QKeySequence(self.trUtf8("CTRL+I","Edit|Indent")), 0,
                self.editActGrp, 'vm_edit_indent')
        self.indentAct.setStatusTip(self.trUtf8('Indent line'))
        self.indentAct.setWhatsThis(self.trUtf8(
            """<b>Indent</b>"""
            """<p>Indents the current line or the lines of the"""
            """ selection by one level.</p>"""
        ))
        self.indentAct.connectIt(SIGNAL('activated()'),self.handleEditIndent)
        self.editActions.append(self.indentAct)
        
        self.unindentAct = E3Action(self.trUtf8('Unindent'),
                QIconSet(UI.PixmapCache.getPixmap("editUnindent.png")),
                self.trUtf8('U&nindent'),
                QKeySequence(self.trUtf8("CTRL+SHIFT+I","Edit|Unindent")), 0,
                self.editActGrp, 'vm_edit_unindent')
        self.unindentAct.setStatusTip(self.trUtf8('Unindent line'))
        self.unindentAct.setWhatsThis(self.trUtf8(
            """<b>Unindent</b>"""
            """<p>Unindents the current line or the lines of the"""
            """ selection by one level.</p>"""
        ))
        self.unindentAct.connectIt(SIGNAL('activated()'),self.handleEditUnindent)
        self.editActions.append(self.unindentAct)
        
        self.smartIndentAct = E3Action(self.trUtf8('Smart indent'),
                QIconSet(UI.PixmapCache.getPixmap("editSmartIndent.png")),
                self.trUtf8('Smart indent'),
                QKeySequence(self.trUtf8("CTRL+ALT+I","Edit|Smart indent")), 0,
                self.editActGrp, 'vm_edit_smart_indent')
        self.smartIndentAct.setStatusTip(self.trUtf8('Smart indent Line or Selection'))
        self.smartIndentAct.setWhatsThis(self.trUtf8(
            """<b>Smart indent</b>"""
            """<p>Indents the current line or the lines of the"""
            """ current selection smartly.</p>"""
        ))
        self.smartIndentAct.connectIt(SIGNAL('activated()'),self.handleEditSmartIndent)
        self.editActions.append(self.smartIndentAct)
        
        self.commentAct = E3Action(self.trUtf8('Comment'),
                QIconSet(UI.PixmapCache.getPixmap("editComment.png")),
                self.trUtf8('C&omment'),
                QKeySequence(self.trUtf8("CTRL+M","Edit|Comment")), 0,
                self.editActGrp, 'vm_edit_comment')
        self.commentAct.setStatusTip(self.trUtf8('Comment Line or Selection'))
        self.commentAct.setWhatsThis(self.trUtf8(
            """<b>Comment</b>"""
            """<p>Comments the current line or the lines of the"""
            """ current selection.</p>"""
        ))
        self.commentAct.connectIt(SIGNAL('activated()'),self.handleEditComment)
        self.editActions.append(self.commentAct)
        
        self.uncommentAct = E3Action(self.trUtf8('Uncomment'),
                QIconSet(UI.PixmapCache.getPixmap("editUncomment.png")),
                self.trUtf8('Unco&mment'),
                QKeySequence(self.trUtf8("ALT+CTRL+M","Edit|Uncomment")), 0,
                self.editActGrp, 'vm_edit_uncomment')
        self.uncommentAct.setStatusTip(self.trUtf8('Uncomment Line or Selection'))
        self.uncommentAct.setWhatsThis(self.trUtf8(
            """<b>Uncomment</b>"""
            """<p>Uncomments the current line or the lines of the"""
            """ current selection.</p>"""
        ))
        self.uncommentAct.connectIt(SIGNAL('activated()'),self.handleEditUncomment)
        self.editActions.append(self.uncommentAct)
        
        self.streamCommentAct = E3Action(self.trUtf8('Stream Comment'),
                self.trUtf8('Stream Comment'),
                0, 0, self.editActGrp, 'vm_edit_stream_comment')
        self.streamCommentAct.setStatusTip(self.trUtf8('Stream Comment Line or Selection'))
        self.streamCommentAct.setWhatsThis(self.trUtf8(
            """<b>Stream Comment</b>"""
            """<p>Stream comments the current line or the current selection.</p>"""
        ))
        self.streamCommentAct.connectIt(SIGNAL('activated()'),self.handleEditStreamComment)
        self.editActions.append(self.streamCommentAct)
        
        self.boxCommentAct = E3Action(self.trUtf8('Box Comment'),
                self.trUtf8('Box Comment'),
                0, 0, self.editActGrp, 'vm_edit_box_comment')
        self.boxCommentAct.setStatusTip(self.trUtf8('Box Comment Line or Selection'))
        self.boxCommentAct.setWhatsThis(self.trUtf8(
            """<b>Box Comment</b>"""
            """<p>Box comments the current line or the lines of the"""
            """ current selection.</p>"""
        ))
        self.boxCommentAct.connectIt(SIGNAL('activated()'),self.handleEditBoxComment)
        self.editActions.append(self.boxCommentAct)
        
        self.selectBraceAct = E3Action(self.trUtf8('Select to brace'),
                self.trUtf8('Select to &brace'),
                QKeySequence(self.trUtf8("CTRL+E","Edit|Select to brace")), 0,
                self.editActGrp, 'vm_edit_select_to_brace')
        self.selectBraceAct.setStatusTip(self.trUtf8('Select text to the matching brace'))
        self.selectBraceAct.setWhatsThis(self.trUtf8(
            """<b>Selct to brace</b>"""
            """<p>Select text of the current editor to the matching brace.</p>"""
        ))
        self.selectBraceAct.connectIt(SIGNAL('activated()'),self.handleEditSelectBrace)
        self.editActions.append(self.selectBraceAct)
        
        self.selectAllAct = E3Action(self.trUtf8('Select all'),
                self.trUtf8('&Select all'),
                QKeySequence(self.trUtf8("CTRL+A","Edit|Select all")), 0,
                self.editActGrp, 'vm_edit_select_all')
        self.selectAllAct.setStatusTip(self.trUtf8('Select all text'))
        self.selectAllAct.setWhatsThis(self.trUtf8(
            """<b>Selct All</b>"""
            """<p>Select all text of the current editor.</p>"""
        ))
        self.selectAllAct.connectIt(SIGNAL('activated()'),self.handleEditSelectAll)
        self.editActions.append(self.selectAllAct)
        
        self.deselectAllAct = E3Action(self.trUtf8('Deselect all'),
                self.trUtf8('&Deselect all'),
                QKeySequence(self.trUtf8("ALT+CTRL+A","Edit|Deselect all")), 0,
                self.editActGrp, 'vm_edit_deselect_all')
        self.deselectAllAct.setStatusTip(self.trUtf8('Deselect all text'))
        self.deselectAllAct.setWhatsThis(self.trUtf8(
            """<b>Deselct All</b>"""
            """<p>Deselect all text of the current editor.</p>"""
        ))
        self.deselectAllAct.connectIt(SIGNAL('activated()'),self.handleEditDeselectAll)
        self.editActions.append(self.deselectAllAct)
        
        self.convertEOLAct = E3Action(self.trUtf8('Convert Line End Characters'),
                self.trUtf8('Convert &Line End Characters'),
                0, 0, self.editActGrp, 'vm_edit_convert_eol')
        self.convertEOLAct.setStatusTip(self.trUtf8('Convert Line End Characters'))
        self.convertEOLAct.setWhatsThis(self.trUtf8(
            """<b>Convert Line End Characters</b>"""
            """<p>Convert the line end characters to the currently set type.</p>"""
        ))
        self.convertEOLAct.connectIt(SIGNAL('activated()'),self.handleConvertEOL)
        self.editActions.append(self.convertEOLAct)
        
        self.shortenEmptyAct = E3Action(self.trUtf8('Shorten empty lines'),
                self.trUtf8('Shorten empty lines'),
                0, 0, self.editActGrp, 'vm_edit_shorten_empty_lines')
        self.shortenEmptyAct.setStatusTip(self.trUtf8('Shorten empty lines'))
        self.shortenEmptyAct.setWhatsThis(self.trUtf8(
            """<b>Shorten empty lines</b>"""
            """<p>Shorten lines consisting solely of whitespace characters.</p>"""
        ))
        self.shortenEmptyAct.connectIt(SIGNAL('activated()'),self.handleShortenEmptyLines)
        self.editActions.append(self.shortenEmptyAct)
        
        self.autoCompleteAct = E3Action(self.trUtf8('Autocomplete'),
                self.trUtf8('&Autocomplete'),
                QKeySequence(self.trUtf8("CTRL+Space","Edit|Autocomplete")), 0,
                self.editActGrp, 'vm_edit_autocomplete')
        self.autoCompleteAct.setStatusTip(self.trUtf8('Autocomplete current word'))
        self.autoCompleteAct.setWhatsThis(self.trUtf8(
            """<b>Autocomplete</b>"""
            """<p>Performs an autocompletion of the word containing the cursor.</p>"""
        ))
        self.autoCompleteAct.connectIt(SIGNAL('activated()'),self.handleEditAutoComplete)
        self.editActions.append(self.autoCompleteAct)
        
        self.autoCompleteFromDocAct = E3Action(self.trUtf8('Autocomplete from Document'),
                self.trUtf8('Autocomplete from Document'),
                QKeySequence(self.trUtf8("CTRL+Shift+Space","Edit|Autocomplete from Document")), 0,
                self.editActGrp, 'vm_edit_autocomplete_from_document')
        self.autoCompleteFromDocAct.setStatusTip(self.trUtf8('Autocomplete current word from Document'))
        self.autoCompleteFromDocAct.setWhatsThis(self.trUtf8(
            """<b>Autocomplete from Document</b>"""
            """<p>Performs an autocompletion from document of the word containing the cursor.</p>"""
        ))
        self.autoCompleteFromDocAct.connectIt(SIGNAL('activated()'),self.handleEditAutoCompleteFromDoc)
        self.editActions.append(self.autoCompleteFromDocAct)
        
        self.autoCompleteFromAPIsAct = E3Action(self.trUtf8('Autocomplete from APIs'),
                self.trUtf8('Autocomplete from APIs'),
                QKeySequence(self.trUtf8("CTRL+Alt+Space","Edit|Autocomplete from APIs")), 0,
                self.editActGrp, 'vm_edit_autocomplete_from_api')
        self.autoCompleteFromAPIsAct.setStatusTip(self.trUtf8('Autocomplete current word from APIs'))
        self.autoCompleteFromAPIsAct.setWhatsThis(self.trUtf8(
            """<b>Autocomplete from APIs</b>"""
            """<p>Performs an autocompletion from APIs of the word containing the cursor.</p>"""
        ))
        self.autoCompleteFromAPIsAct.connectIt(SIGNAL('activated()'),self.handleEditAutoCompleteFromAPIs)
        self.editActions.append(self.autoCompleteFromAPIsAct)
        
        self.editActGrp.setEnabled(0)
        
        ####################################################################
        ## Below follow the actions for qscintilla standard commands.
        ####################################################################
        
        self.esm = QSignalMapper(self)
        self.connect(self.esm, SIGNAL('mapped(int)'), self.editorCommand)
        
        self.editorActGrp = E3ActionGroup(self.editActGrp)
        
        act = E3Action(self.trUtf8('Move left one character'), 
                      self.trUtf8('Move left one character'), 
                      QKeySequence(self.trUtf8('Left')), 0,
                      self.editorActGrp, 'vm_edit_move_left_char')
        act.mapIt(self.esm, QextScintilla.SCI_CHARLEFT)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Move right one character'), 
                      self.trUtf8('Move right one character'), 
                      QKeySequence(self.trUtf8('Right')), 0,
                      self.editorActGrp, 'vm_edit_move_right_char')
        act.mapIt(self.esm, QextScintilla.SCI_CHARRIGHT)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Move up one line'), 
                      self.trUtf8('Move up one line'), 
                      QKeySequence(self.trUtf8('Up')), 0,
                      self.editorActGrp, 'vm_edit_move_up_line')
        act.mapIt(self.esm, QextScintilla.SCI_LINEUP)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Move down one line'), 
                      self.trUtf8('Move down one line'), 
                      QKeySequence(self.trUtf8('Down')), 0,
                      self.editorActGrp, 'vm_edit_move_down_line')
        act.mapIt(self.esm, QextScintilla.SCI_LINEDOWN)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Move left one word part'), 
                      self.trUtf8('Move left one word part'), 
                      QKeySequence(self.trUtf8('Alt+Left')), 0,
                      self.editorActGrp, 'vm_edit_move_left_word_part')
        act.mapIt(self.esm, QextScintilla.SCI_WORDPARTLEFT)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Move right one word part'), 
                      self.trUtf8('Move right one word part'), 
                      QKeySequence(self.trUtf8('Alt+Right')), 0,
                      self.editorActGrp, 'vm_edit_move_right_word_part')
        act.mapIt(self.esm, QextScintilla.SCI_WORDPARTRIGHT)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Move left one word'), 
                      self.trUtf8('Move left one word'), 
                      QKeySequence(self.trUtf8('Ctrl+Left')), 0,
                      self.editorActGrp, 'vm_edit_move_left_word')
        act.mapIt(self.esm, QextScintilla.SCI_WORDLEFT)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Move right one word'), 
                      self.trUtf8('Move right one word'), 
                      QKeySequence(self.trUtf8('Ctrl+Right')), 0,
                      self.editorActGrp, 'vm_edit_move_right_word')
        act.mapIt(self.esm, QextScintilla.SCI_WORDRIGHT)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Move to first visible character in line'), 
                      self.trUtf8('Move to first visible character in line'), 
                      QKeySequence(self.trUtf8('Home')), 0,
                      self.editorActGrp, 'vm_edit_move_first_visible_char')
        act.mapIt(self.esm, QextScintilla.SCI_VCHOME)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Move to start of displayed line'), 
                      self.trUtf8('Move to start of displayed line'), 
                      QKeySequence(self.trUtf8('Alt+Home')), 0,
                      self.editorActGrp, 'vm_edit_move_start_line')
        act.mapIt(self.esm, QextScintilla.SCI_HOMEDISPLAY)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Move to end of line'), 
                      self.trUtf8('Move to end of line'), 
                      QKeySequence(self.trUtf8('End')), 0,
                      self.editorActGrp, 'vm_edit_move_end_line')
        act.mapIt(self.esm, QextScintilla.SCI_LINEEND)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Scroll view down one line'), 
                      self.trUtf8('Scroll view down one line'), 
                      QKeySequence(self.trUtf8('Ctrl+Down')), 0,
                      self.editorActGrp, 'vm_edit_scroll_down_line')
        act.mapIt(self.esm, QextScintilla.SCI_LINESCROLLDOWN)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Scroll view up one line'), 
                      self.trUtf8('Scroll view up one line'), 
                      QKeySequence(self.trUtf8('Ctrl+Up')), 0,
                      self.editorActGrp, 'vm_edit_scroll_up_line')
        act.mapIt(self.esm, QextScintilla.SCI_LINESCROLLUP)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Move up one paragraph'), 
                      self.trUtf8('Move up one paragraph'), 
                      QKeySequence(self.trUtf8('Alt+Up')), 0,
                      self.editorActGrp, 'vm_edit_move_up_para')
        act.mapIt(self.esm, QextScintilla.SCI_PARAUP)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Move down one paragraph'), 
                      self.trUtf8('Move down one paragraph'), 
                      QKeySequence(self.trUtf8('Alt+Down')), 0,
                      self.editorActGrp, 'vm_edit_move_down_para')
        act.mapIt(self.esm, QextScintilla.SCI_PARADOWN)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Move up one page'), 
                      self.trUtf8('Move up one page'), 
                      QKeySequence(self.trUtf8('PgUp')), 0,
                      self.editorActGrp, 'vm_edit_move_up_page')
        act.mapIt(self.esm, QextScintilla.SCI_PAGEUP)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Move down one page'), 
                      self.trUtf8('Move down one page'), 
                      QKeySequence(self.trUtf8('PgDown')), 0,
                      self.editorActGrp, 'vm_edit_move_down_page')
        act.mapIt(self.esm, QextScintilla.SCI_PAGEDOWN)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Move to start of text'), 
                      self.trUtf8('Move to start of text'), 
                      QKeySequence(self.trUtf8('Ctrl+Home')), 0,
                      self.editorActGrp, 'vm_edit_move_start_text')
        act.mapIt(self.esm, QextScintilla.SCI_DOCUMENTSTART)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Move to end of text'), 
                      self.trUtf8('Move to end of text'), 
                      QKeySequence(self.trUtf8('Ctrl+End')), 0,
                      self.editorActGrp, 'vm_edit_move_end_text')
        act.mapIt(self.esm, QextScintilla.SCI_DOCUMENTEND)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Indent one level'), 
                      self.trUtf8('Indent one level'), 
                      QKeySequence(self.trUtf8('Tab')), 0,
                      self.editorActGrp, 'vm_edit_indent_one_level')
        act.mapIt(self.esm, QextScintilla.SCI_TAB)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Unindent one level'), 
                      self.trUtf8('Unindent one level'), 
                      QKeySequence(self.trUtf8('Shift+Tab')), 0,
                      self.editorActGrp, 'vm_edit_unindent_one_level')
        act.mapIt(self.esm, QextScintilla.SCI_BACKTAB)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Extend selection left one character'), 
                      self.trUtf8('Extend selection left one character'), 
                      QKeySequence(self.trUtf8('Shift+Left')), 0,
                      self.editorActGrp, 'vm_edit_extend_selection_left_char')
        act.mapIt(self.esm, QextScintilla.SCI_CHARLEFTEXTEND)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Extend selection right one character'), 
                      self.trUtf8('Extend selection right one character'), 
                      QKeySequence(self.trUtf8('Shift+Right')), 0,
                      self.editorActGrp, 'vm_edit_extend_selection_right_char')
        act.mapIt(self.esm, QextScintilla.SCI_CHARRIGHTEXTEND)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Extend selection up one line'), 
                      self.trUtf8('Extend selection up one line'), 
                      QKeySequence(self.trUtf8('Shift+Up')), 0,
                      self.editorActGrp, 'vm_edit_extend_selection_up_line')
        act.mapIt(self.esm, QextScintilla.SCI_LINEUPEXTEND)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Extend selection down one line'), 
                      self.trUtf8('Extend selection down one line'), 
                      QKeySequence(self.trUtf8('Shift+Down')), 0,
                      self.editorActGrp, 'vm_edit_extend_selection_down_line')
        act.mapIt(self.esm, QextScintilla.SCI_LINEDOWNEXTEND)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Extend selection left one word part'), 
                      self.trUtf8('Extend selection left one word part'), 
                      QKeySequence(self.trUtf8('Alt+Shift+Left')), 0,
                      self.editorActGrp, 'vm_edit_extend_selection_left_word_part')
        act.mapIt(self.esm, QextScintilla.SCI_WORDPARTLEFTEXTEND)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Extend selection right one word part'), 
                      self.trUtf8('Extend selection right one word part'), 
                      QKeySequence(self.trUtf8('Alt+Shift+Right')), 0,
                      self.editorActGrp, 'vm_edit_extend_selection_right_word_part')
        act.mapIt(self.esm, QextScintilla.SCI_WORDPARTRIGHTEXTEND)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Extend selection left one word'), 
                      self.trUtf8('Extend selection left one word'), 
                      QKeySequence(self.trUtf8('Ctrl+Shift+Left')), 0,
                      self.editorActGrp, 'vm_edit_extend_selection_left_word')
        act.mapIt(self.esm, QextScintilla.SCI_WORDLEFTEXTEND)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Extend selection right one word'), 
                      self.trUtf8('Extend selection right one word'), 
                      QKeySequence(self.trUtf8('Ctrl+Shift+Right')), 0,
                      self.editorActGrp, 'vm_edit_extend_selection_right_word')
        act.mapIt(self.esm, QextScintilla.SCI_WORDRIGHTEXTEND)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Extend selection to first visible character in line'),
                      self.trUtf8('Extend selection to first visible character in line'), 
                      QKeySequence(self.trUtf8('Shift+Home')), 0,
                      self.editorActGrp, 'vm_edit_extend_selection_first_visible_char')
        act.mapIt(self.esm, QextScintilla.SCI_VCHOMEEXTEND)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Extend selection to start of line'), 
                      self.trUtf8('Extend selection to start of line'), 
                      QKeySequence(self.trUtf8('Alt+Shift+Home')), 0,
                      self.editorActGrp, 'vm_edit_extend_selection_start_line')
        act.mapIt(self.esm, QextScintilla.SCI_HOMEDISPLAYEXTEND)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Extend selection to end of line'), 
                      self.trUtf8('Extend selection to end of line'), 
                      QKeySequence(self.trUtf8('Shift+End')), 0,
                      self.editorActGrp, 'vm_edit_extend_selection_end_line')
        act.mapIt(self.esm, QextScintilla.SCI_LINEENDEXTEND)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Extend selection up one paragraph'), 
                      self.trUtf8('Extend selection up one paragraph'), 
                      QKeySequence(self.trUtf8('Alt+Shift+Up')), 0,
                      self.editorActGrp, 'vm_edit_extend_selection_up_para')
        act.mapIt(self.esm, QextScintilla.SCI_PARAUPEXTEND)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Extend selection down one paragraph'), 
                      self.trUtf8('Extend selection down one paragraph'), 
                      QKeySequence(self.trUtf8('Alt+Shift+Down')), 0,
                      self.editorActGrp, 'vm_edit_extend_selection_down_para')
        act.mapIt(self.esm, QextScintilla.SCI_PARADOWNEXTEND)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Extend selection up one page'), 
                      self.trUtf8('Extend selection up one page'), 
                      QKeySequence(self.trUtf8('Shift+PgUp')), 0,
                      self.editorActGrp, 'vm_edit_extend_selection_up_page')
        act.mapIt(self.esm, QextScintilla.SCI_PAGEUPEXTEND)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Extend selection down one page'), 
                      self.trUtf8('Extend selection down one page'), 
                      QKeySequence(self.trUtf8('Shift+PgDown')), 0,
                      self.editorActGrp, 'vm_edit_extend_selection_down_page')
        act.mapIt(self.esm, QextScintilla.SCI_PAGEDOWNEXTEND)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Extend selection to start of text'), 
                      self.trUtf8('Extend selection to start of text'), 
                      QKeySequence(self.trUtf8('Ctrl+Shift+Home')), 0,
                      self.editorActGrp, 'vm_edit_extend_selection_start_text')
        act.mapIt(self.esm, QextScintilla.SCI_DOCUMENTSTARTEXTEND)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Extend selection to end of text'), 
                      self.trUtf8('Extend selection to end of text'), 
                      QKeySequence(self.trUtf8('Ctrl+Shift+End')), 0,
                      self.editorActGrp, 'vm_edit_extend_selection_end_text')
        act.mapIt(self.esm, QextScintilla.SCI_DOCUMENTENDEXTEND)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Delete previous character'), 
                      self.trUtf8('Delete previous character'), 
                      QKeySequence(self.trUtf8('Backspace')), 
                      QKeySequence(self.trUtf8('Shift+Backspace')), 
                      self.editorActGrp, 'vm_edit_delete_previous_char')
        act.mapIt(self.esm, QextScintilla.SCI_DELETEBACK)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Delete previous character if not at line start'), 
                      self.trUtf8('Delete previous character if not at line start'), 
                      0, 0,
                      self.editorActGrp, 'vm_edit_delet_previous_char_not_line_start')
        act.mapIt(self.esm, QextScintilla.SCI_DELETEBACKNOTLINE)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Delete current character'), 
                      self.trUtf8('Delete current character'), 
                      QKeySequence(self.trUtf8('Del')), 0,
                      self.editorActGrp, 'vm_edit_delete_current_char')
        act.mapIt(self.esm, QextScintilla.SCI_CLEAR)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Delete word to left'), 
                      self.trUtf8('Delete word to left'), 
                      QKeySequence(self.trUtf8('Ctrl+Backspace')), 0,
                      self.editorActGrp, 'vm_edit_delete_word_left')
        act.mapIt(self.esm, QextScintilla.SCI_DELWORDLEFT)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Delete word to right'), 
                      self.trUtf8('Delete word to right'), 
                      QKeySequence(self.trUtf8('Ctrl+Del')), 0,
                      self.editorActGrp, 'vm_edit_delete_word_right')
        act.mapIt(self.esm, QextScintilla.SCI_DELWORDRIGHT)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Delete line to left'), 
                      self.trUtf8('Delete line to left'), 
                      QKeySequence(self.trUtf8('Ctrl+Shift+Backspace')), 0,
                      self.editorActGrp, 'vm_edit_delete_line_left')
        act.mapIt(self.esm, QextScintilla.SCI_DELLINELEFT)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Delete line to right'), 
                      self.trUtf8('Delete line to right'), 
                      QKeySequence(self.trUtf8('Ctrl+Shift+Del')), 0,
                      self.editorActGrp, 'vm_edit_delete_line_right')
        act.mapIt(self.esm, QextScintilla.SCI_DELLINERIGHT)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Insert new line'), 
                      self.trUtf8('Insert new line'), 
                      QKeySequence(self.trUtf8('Return')), 
                      QKeySequence(self.trUtf8('Enter')), 
                      self.editorActGrp, 'vm_edit_insert_line')
        act.mapIt(self.esm, QextScintilla.SCI_NEWLINE)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Delete current line'), 
                      self.trUtf8('Delete current line'), 
                      QKeySequence(self.trUtf8('Ctrl+U')), 
                      QKeySequence(self.trUtf8('Ctrl+Shift+L')),
                      self.editorActGrp, 'vm_edit_delete_current_line')
        act.mapIt(self.esm, QextScintilla.SCI_LINEDELETE)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Duplicate current line'), 
                      self.trUtf8('Duplicate current line'), 
                      QKeySequence(self.trUtf8('Ctrl+D')), 0,
                      self.editorActGrp, 'vm_edit_duplicate_current_line')
        act.mapIt(self.esm, QextScintilla.SCI_LINEDUPLICATE)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Swap current and previous lines'), 
                      self.trUtf8('Swap current and previous lines'), 
                      QKeySequence(self.trUtf8('Ctrl+T')), 0,
                      self.editorActGrp, 'vm_edit_swap_current_previous_line')
        act.mapIt(self.esm, QextScintilla.SCI_LINETRANSPOSE)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Cut current line'), 
                      self.trUtf8('Cut current line'), 
                      QKeySequence(self.trUtf8('Alt+Shift+L')), 0,
                      self.editorActGrp, 'vm_edit_cut_current_line')
        act.mapIt(self.esm, QextScintilla.SCI_LINECUT)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Copy current line'), 
                      self.trUtf8('Copy current line'), 
                      QKeySequence(self.trUtf8('Ctrl+Shift+T')), 0,
                      self.editorActGrp, 'vm_edit_copy_current_line')
        act.mapIt(self.esm, QextScintilla.SCI_LINECOPY)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Toggle insert/overtype'), 
                      self.trUtf8('Toggle insert/overtype'), 
                      QKeySequence(self.trUtf8('Ins')), 0,
                      self.editorActGrp, 'vm_edit_toggle_insert_overtype')
        act.mapIt(self.esm, QextScintilla.SCI_EDITTOGGLEOVERTYPE)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Convert selection to lower case'), 
                      self.trUtf8('Convert selection to lower case'), 
                      QKeySequence(self.trUtf8('Alt+Shift+U')), 0,
                      self.editorActGrp, 'vm_edit_convert_selection_lower')
        act.mapIt(self.esm, QextScintilla.SCI_LOWERCASE)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Convert selection to upper case'), 
                      self.trUtf8('Convert selection to upper case'), 
                      QKeySequence(self.trUtf8('Ctrl+Shift+U')), 0,
                      self.editorActGrp, 'vm_edit_convert_selection_upper')
        act.mapIt(self.esm, QextScintilla.SCI_UPPERCASE)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Move to end of displayed line'), 
                      self.trUtf8('Move to end of displayed line'), 
                      QKeySequence(self.trUtf8('Alt+End')), 0,
                      self.editorActGrp, 'vm_edit_move_end_displayed_line')
        act.mapIt(self.esm, QextScintilla.SCI_LINEENDDISPLAY)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Extend selection to end of displayed line'), 
                      self.trUtf8('Extend selection to end of displayed line'), 
                      0, 0,
                      self.editorActGrp, 'vm_edit_extend_selection_end_displayed_line')
        act.mapIt(self.esm, QextScintilla.SCI_LINEENDDISPLAYEXTEND)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Formfeed'), 
                      self.trUtf8('Formfeed'), 
                      0, 0,
                      self.editorActGrp, 'vm_edit_formfeed')
        act.mapIt(self.esm, QextScintilla.SCI_FORMFEED)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        act = E3Action(self.trUtf8('Escape'), 
                      self.trUtf8('Escape'), 
                      QKeySequence(self.trUtf8('Esc')), 0,
                      self.editorActGrp, 'vm_edit_escape')
        act.mapIt(self.esm, QextScintilla.SCI_CANCEL)
        act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
        self.editActions.append(act)
        
        # The following actions are only supported for QScintilla > 1.2
        # (i.e. build > 70)
        if QSCINTILLA_VERSION() > 0x010200:
            
            act = E3Action(self.trUtf8('Extend rectangular selection down one line'), 
                          self.trUtf8('Extend rectangular selection down one line'), 
                          QKeySequence(self.trUtf8('Alt+Ctrl+Down')), 0,
                          self.editorActGrp, 'vm_edit_extend_rect_selection_down_line')
            act.mapIt(self.esm, QextScintilla.SCI_LINEDOWNRECTEXTEND)
            act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
            self.editActions.append(act)
            
            act = E3Action(self.trUtf8('Extend rectangular selection up one line'), 
                          self.trUtf8('Extend rectangular selection up one line'), 
                          QKeySequence(self.trUtf8('Alt+Ctrl+Up')), 0,
                          self.editorActGrp, 'vm_edit_extend_rect_selection_up_line')
            act.mapIt(self.esm, QextScintilla.SCI_LINEUPRECTEXTEND)
            act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
            self.editActions.append(act)
            
            act = E3Action(self.trUtf8('Extend rectangular selection left one character'), 
                          self.trUtf8('Extend rectangular selection left one character'), 
                          QKeySequence(self.trUtf8('Alt+Ctrl+Left')), 0,
                          self.editorActGrp, 'vm_edit_extend_rect_selection_left_char')
            act.mapIt(self.esm, QextScintilla.SCI_CHARLEFTRECTEXTEND)
            act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
            self.editActions.append(act)
            
            act = E3Action(self.trUtf8('Extend rectangular selection right one character'), 
                          self.trUtf8('Extend rectangular selection right one character'), 
                          QKeySequence(self.trUtf8('Alt+Ctrl+Right')), 0,
                          self.editorActGrp, 'vm_edit_extend_rect_selection_right_char')
            act.mapIt(self.esm, QextScintilla.SCI_CHARRIGHTRECTEXTEND)
            act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
            self.editActions.append(act)
            
            act = E3Action(self.trUtf8('Extend rectangular selection to first visible character in line'), 
                          self.trUtf8('Extend rectangular selection to first visible character in line'), 
                          QKeySequence(self.trUtf8('Alt+Ctrl+Home')), 0,
                          self.editorActGrp, 'vm_edit_extend_rect_selection_first_visible_char')
            act.mapIt(self.esm, QextScintilla.SCI_VCHOMERECTEXTEND)
            act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
            self.editActions.append(act)
            
            act = E3Action(self.trUtf8('Extend rectangular selection to end of line'), 
                          self.trUtf8('Extend rectangular selection to end of line'), 
                          QKeySequence(self.trUtf8('Alt+Ctrl+End')), 0,
                          self.editorActGrp, 'vm_edit_extend_rect_selection_end_line')
            act.mapIt(self.esm, QextScintilla.SCI_LINEENDRECTEXTEND)
            act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
            self.editActions.append(act)
            
            act = E3Action(self.trUtf8('Extend rectangular selection up one page'), 
                          self.trUtf8('Extend rectangular selection up one page'), 
                          QKeySequence(self.trUtf8('Alt+Ctrl+PgUp')), 0,
                          self.editorActGrp, 'vm_edit_extend_rect_selection_up_page')
            act.mapIt(self.esm, QextScintilla.SCI_PAGEUPRECTEXTEND)
            act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
            self.editActions.append(act)
            
            act = E3Action(self.trUtf8('Extend rectangular selection down one page'), 
                          self.trUtf8('Extend rectangular selection down one page'), 
                          QKeySequence(self.trUtf8('Alt+Ctrl+PgDown')), 0,
                          self.editorActGrp, 'vm_edit_extend_rect_selection_down_page')
            act.mapIt(self.esm, QextScintilla.SCI_PAGEDOWNRECTEXTEND)
            act.connectIt(SIGNAL('activated()'),self.esm,SLOT('map()'))
            self.editActions.append(act)
            
            # weird things are going on here, had to enable and disable
            # it to make it work with the new QScintilla shell
            self.editorActGrp.setEnabled(1)
            self.editorActGrp.setEnabled(0)
    
    def initEditMenu(self):
        """
        Public method to create the Edit menu
        
        @return the generated menu
        """
        menu = QPopupMenu(self.ui)
        menu.insertTearOffHandle()
        self.undoAct.addTo(menu)
        self.redoAct.addTo(menu)
        self.revertAct.addTo(menu)
        menu.insertSeparator()
        self.cutAct.addTo(menu)
        self.copyAct.addTo(menu)
        self.pasteAct.addTo(menu)
        self.deleteAct.addTo(menu)
        menu.insertSeparator()
        self.indentAct.addTo(menu)
        self.unindentAct.addTo(menu)
        self.smartIndentAct.addTo(menu)
        menu.insertSeparator()
        self.commentAct.addTo(menu)
        self.uncommentAct.addTo(menu)
        self.streamCommentAct.addTo(menu)
        self.boxCommentAct.addTo(menu)
        menu.insertSeparator()
        self.autoCompleteAct.addTo(menu)
        self.autoCompleteFromDocAct.addTo(menu)
        self.autoCompleteFromAPIsAct.addTo(menu)
        menu.insertSeparator()
        self.quickSearchAct.addTo(menu)
        self.quickSearchBackAct.addTo(menu)
        self.searchAct.addTo(menu)
        self.searchAgainAct.addTo(menu)
        self.replaceAct.addTo(menu)
        menu.insertSeparator()
        self.searchFilesAct.addTo(menu)
        menu.insertSeparator()
        self.gotoAct.addTo(menu)
        self.gotoBraceAct.addTo(menu)
        menu.insertSeparator()
        self.selectBraceAct.addTo(menu)
        self.selectAllAct.addTo(menu)
        self.deselectAllAct.addTo(menu)
        menu.insertSeparator()
        self.shortenEmptyAct.addTo(menu)
        self.convertEOLAct.addTo(menu)
        
        return menu
        
    def initEditToolbar(self):
        """
        Public method to create the Edit toolbar
        
        @return the generated toolbar
        """
        tb = QToolBar(self.ui)
        self.undoAct.addTo(tb)
        self.redoAct.addTo(tb)
        tb.addSeparator()
        self.cutAct.addTo(tb)
        self.copyAct.addTo(tb)
        self.pasteAct.addTo(tb)
        self.deleteAct.addTo(tb)
        tb.addSeparator()
        self.indentAct.addTo(tb)
        self.unindentAct.addTo(tb)
        self.smartIndentAct.addTo(tb)
        tb.addSeparator()
        self.commentAct.addTo(tb)
        self.uncommentAct.addTo(tb)
        
        return tb
        
    ##################################################################
    ## Initialize the search related actions, search menu and toolbar
    ##################################################################
    
    def initSearchActions(self):
        """
        Private method defining the user interface actions for the search commands.
        """
        self.searchActGrp = E3ActionGroup(self)
        
        self.searchAct = E3Action(self.trUtf8('Search'),
                QIconSet(UI.PixmapCache.getPixmap("find.png")),
                self.trUtf8('&Search...'),
                QKeySequence(self.trUtf8("CTRL+F","Search|Search")), 0,
                self.searchActGrp, 'vm_search')
        self.searchAct.setStatusTip(self.trUtf8('Search for a text'))
        self.searchAct.setWhatsThis(self.trUtf8(
            """<b>Search</b>"""
            """<p>Search for some text in the current editor. A"""
            """ dialog is shown to enter the searchtext and options"""
            """ for the search.</p>"""
        ))
        self.searchAct.connectIt(SIGNAL('activated()'),self.handleSearch)
        self.searchActions.append(self.searchAct)
        
        self.searchAgainAct = E3Action(self.trUtf8('Search again'),
                QIconSet(UI.PixmapCache.getPixmap("findNext.png")),
                self.trUtf8('Search &again'),
                QKeySequence(self.trUtf8("F3","Search|Search")),0,
                self.searchActGrp,'vm_search_again')
        self.searchAgainAct.setStatusTip(self.trUtf8('Search again for text'))
        self.searchAgainAct.setWhatsThis(self.trUtf8(
            """<b>Search again</b>"""
            """<p>Search again for some text in the current editor."""
            """ The previously entered searchtext and options are reused.</p>"""
        ))
        self.searchAgainAct.connectIt(SIGNAL('activated()'),self.searchDlg.handleFindNext)
        self.searchActions.append(self.searchAgainAct)
        
        self.replaceAct = E3Action(self.trUtf8('Replace'),
                self.trUtf8('&Replace...'),
                QKeySequence(self.trUtf8("CTRL+R","Search|Replace")), 0,
                self.searchActGrp, 'vm_search_replace')
        self.replaceAct.setStatusTip(self.trUtf8('Replace some text'))
        self.replaceAct.setWhatsThis(self.trUtf8(
            """<b>Replace</b>"""
            """<p>Search for some text in the current editor and replace it. A"""
            """ dialog is shown to enter the searchtext, the replacement text"""
            """ and options for the search and replace.</p>"""
        ))
        self.replaceAct.connectIt(SIGNAL('activated()'),self.handleReplace)
        self.searchActions.append(self.replaceAct)
        
        self.quickSearchAct = E3Action(self.trUtf8('Quicksearch'),
                QIconSet(UI.PixmapCache.getPixmap("quickFindNext.png")),
                self.trUtf8('&Quicksearch'),
                QKeySequence(self.trUtf8("Ctrl+Shift+K","Search|Quicksearch")), 0,
                self.searchActGrp, 'vm_quicksearch')
        self.quickSearchAct.setStatusTip(self.trUtf8('Perform a quicksearch'))
        self.quickSearchAct.setWhatsThis(self.trUtf8(
            """<b>Quicksearch</b>"""
            """<p>This activates the quicksearch function of the IDE by"""
            """ giving focus to the quicksearch entry field. If this field"""
            """ is already active and contains text, it searches for the"""
            """ next occurrance of this text.</p>"""
        ))
        self.quickSearchAct.connectIt(SIGNAL('activated()'),self.handleQuickSearch)
        self.searchActions.append(self.quickSearchAct)
        
        self.quickSearchBackAct = E3Action(self.trUtf8('Quicksearch backwards'),
                QIconSet(UI.PixmapCache.getPixmap("quickFindPrev.png")),
                self.trUtf8('Quicksearch &backwards'),
                QKeySequence(self.trUtf8("Ctrl+Shift+J","Search|Quicksearch backwards")), 0,
                self.searchActGrp, 'vm_quicksearch_backwards')
        self.quickSearchBackAct.setStatusTip(self.trUtf8('Perform a quicksearch backwards'))
        self.quickSearchBackAct.setWhatsThis(self.trUtf8(
            """<b>Quicksearch backwards</b>"""
            """<p>This searches the previous occurrance of the quicksearch text.</p>"""
        ))
        self.quickSearchBackAct.connectIt(SIGNAL('activated()'),self.handleQuickSearchPrev)
        self.searchActions.append(self.quickSearchBackAct)
        
        self.quickSearchExtendAct = E3Action(self.trUtf8('Quicksearch extend'),
                QIconSet(UI.PixmapCache.getPixmap("quickFindExtend.png")),
                self.trUtf8('Quicksearch e&xtend'),
                QKeySequence(self.trUtf8("Ctrl+Shift+H","Search|Quicksearch extend")), 0,
                self.searchActGrp, 'vm_quicksearch_extend')
        self.quickSearchExtendAct.setStatusTip(self.trUtf8(\
            'Extend the quicksearch to the end of the current word'))
        self.quickSearchExtendAct.setWhatsThis(self.trUtf8(
            """<b>Quicksearch extend</b>"""
            """<p>This extends the quicksearch text to the end of the word"""
            """ currently found.</p>"""
        ))
        self.quickSearchExtendAct.connectIt(SIGNAL('activated()'),self.handleQuickSearchExtend)
        self.searchActions.append(self.quickSearchExtendAct)
        
        self.gotoAct = E3Action(self.trUtf8('Goto Line'),
                QIconSet(UI.PixmapCache.getPixmap("goto.png")),
                self.trUtf8('&Goto Line...'),
                QKeySequence(self.trUtf8("CTRL+G","Search|Goto Line")), 0,
                self.searchActGrp, 'vm_search_goto_line')
        self.gotoAct.setStatusTip(self.trUtf8('Goto Line'))
        self.gotoAct.setWhatsThis(self.trUtf8(
            """<b>Goto Line</b>"""
            """<p>Go to a specific line of text in the current editor."""
            """ A dialog is shown to enter the linenumber.</p>"""
        ))
        self.gotoAct.connectIt(SIGNAL('activated()'),self.handleGoto)
        self.searchActions.append(self.gotoAct)
        
        self.gotoBraceAct = E3Action(self.trUtf8('Goto Brace'),
                QIconSet(UI.PixmapCache.getPixmap("gotoBrace.png")),
                self.trUtf8('Goto &Brace'),
                QKeySequence(self.trUtf8("CTRL+L","Search|Goto Brace")), 0,
                self.searchActGrp, 'vm_search_goto_brace')
        self.gotoBraceAct.setStatusTip(self.trUtf8('Goto Brace'))
        self.gotoBraceAct.setWhatsThis(self.trUtf8(
            """<b>Goto Brace</b>"""
            """<p>Go to the matching brace in the current editor.</p>"""
        ))
        self.gotoBraceAct.connectIt(SIGNAL('activated()'),self.handleGotoBrace)
        self.searchActions.append(self.gotoBraceAct)
        
        self.searchActGrp.setEnabled(0)
        
        self.searchFilesAct = E3Action(self.trUtf8('Search in Files'),
                QIconSet(UI.PixmapCache.getPixmap("projectFind.png")),
                self.trUtf8('Search in &Files...'),
                QKeySequence(self.trUtf8("SHIFT+CTRL+F","Search|Search Files")), 0,
                self, 'vm_search_in_files')
        self.searchFilesAct.setStatusTip(self.trUtf8('Search for a text in files'))
        self.searchFilesAct.setWhatsThis(self.trUtf8(
            """<b>Search in Files</b>"""
            """<p>Search for some text in the files of a directory tree"""
            """ or the project. A dialog is shown to enter the searchtext"""
            """ and options for the search and to display the result.</p>"""
        ))
        self.searchFilesAct.connectIt(SIGNAL('activated()'),self.handleSearchFiles)
        self.searchActions.append(self.searchFilesAct)
        
    def initSearchToolbar(self):
        """
        Public method to create the Search toolbar
        
        @return the generated toolbar
        """
        tb = QToolBar(self.ui)
        
        self.quickFindLineEdit = QuickSearchLineEdit(self)
        self.quickFindtextCombo = QComboBox(1, tb)
        self.quickFindtextCombo.setLineEdit(self.quickFindLineEdit)
        self.quickFindtextCombo.setDuplicatesEnabled(0)
        self.quickFindtextCombo.setInsertionPolicy(QComboBox.AtTop)
        self.quickFindtextCombo.setEditable(1)
        self.quickFindtextCombo.lastActive = None
        self.quickFindtextCombo.lastCursorPos = None
        self.quickFindtextCombo.leForegroundColor = \
            self.quickFindtextCombo.lineEdit().paletteForegroundColor()
        self.quickFindtextCombo.leBackgroundColor = \
            self.quickFindtextCombo.lineEdit().paletteBackgroundColor()
        self.quickFindtextCombo.lastSearchText = QString.null
        self.quickFindtextCombo._editor = self.quickFindtextCombo.lineEdit()
        # this allows us not to jump across searched text
        # just because of autocompletion enabled
        self.quickFindtextCombo.setAutoCompletion(0)
        self.quickFindtextCombo.setMinimumWidth(250)
        self.quickFindtextCombo.insertItem("")
        tb.setStretchableWidget(self.quickFindtextCombo)
        QWhatsThis.add(self.quickFindtextCombo, self.trUtf8(
                """<p>Enter the searchtext directly into this field."""
                """ The search will be performed case insensitive."""
                """ The quicksearch function is activated upon activation"""
                """ of the quicksearch next action (default key CTRL+SHIFT+K),"""
                """ if this entry field does not have the input focus."""
                """ Otherwise it searches for the next occurrance of the"""
                """ text entered. The quicksearch backwards action"""
                """ (default key CTRL+SHIFT+J) searches backward."""
                """ Activating the 'quicksearch extend' action"""
                """ (default key CTRL+SHIFT+H) extends the current"""
                """ searchtext to the end of the currently found word."""
                """ The quicksearch can be ended by pressing the Return key"""
                """ while the quicksearch entry has the the input focus.</p>"""
        ))
        self.connect(self.quickFindtextCombo._editor, SIGNAL('returnPressed()'),
            self.handleQuickSearchEnter)
        self.connect(self.quickFindtextCombo._editor, 
            SIGNAL('textChanged(const QString&)'), self.handleQuickSearchText)
        self.connect(self.quickFindtextCombo._editor, PYSIGNAL('escPressed()'),
            self.handleQuickSearchEscape)
        self.quickSearchAct.addTo(tb)
        self.quickSearchBackAct.addTo(tb)
        self.quickSearchExtendAct.addTo(tb)
        self.quickFindtextCombo.setEnabled(0)
        
        tb.addSeparator()
        self.searchAct.addTo(tb)
        self.searchAgainAct.addTo(tb)
        tb.addSeparator()
        self.searchFilesAct.addTo(tb)
        tb.addSeparator()
        self.gotoAct.addTo(tb)
        self.gotoBraceAct.addTo(tb)
        
        return tb
        
    ##################################################################
    ## Initialize the view related actions, view menu and toolbar
    ##################################################################
    
    def initViewActions(self):
        """
        Protected method defining the user interface actions for the view commands.
        """
        self.viewActGrp = E3ActionGroup(self)
        self.viewFoldActGrp = E3ActionGroup(self)

        self.zoomInAct = E3Action(self.trUtf8('Zoom in'),
                            QIconSet(UI.PixmapCache.getPixmap("zoomIn.png")),
                            self.trUtf8('Zoom &in'),
                            QKeySequence(self.trUtf8("CTRL++", "View|Zoom in")), 0,
                            self.viewActGrp, 'vm_view_zoom_in')
        self.zoomInAct.setStatusTip(self.trUtf8('Zoom in on the text'))
        self.zoomInAct.setWhatsThis(self.trUtf8(
                """<b>Zoom in</b>"""
                """<p>Zoom in on the text. This makes the text bigger.</p>"""
                ))
        self.zoomInAct.connectIt(SIGNAL('activated()'),self.handleZoomIn)
        self.viewActions.append(self.zoomInAct)
        
        self.zoomOutAct = E3Action(self.trUtf8('Zoom out'),
                            QIconSet(UI.PixmapCache.getPixmap("zoomOut.png")),
                            self.trUtf8('Zoom &out'),
                            QKeySequence(self.trUtf8("CTRL+-", "View|Zoom out")), 0,
                            self.viewActGrp, 'vm_view_zoom_out')
        self.zoomOutAct.setStatusTip(self.trUtf8('Zoom out on the text'))
        self.zoomOutAct.setWhatsThis(self.trUtf8(
                """<b>Zoom out</b>"""
                """<p>Zoom out on the text. This makes the text smaller.</p>"""
                ))
        self.zoomOutAct.connectIt(SIGNAL('activated()'),self.handleZoomOut)
        self.viewActions.append(self.zoomOutAct)
        
        self.zoomToAct = E3Action(self.trUtf8('Zoom'),
                            QIconSet(UI.PixmapCache.getPixmap("zoomTo.png")),
                            self.trUtf8('&Zoom'),
                            0, 0, self.viewActGrp, 'vm_view_zoom')
        self.zoomToAct.setStatusTip(self.trUtf8('Zoom the text'))
        self.zoomToAct.setWhatsThis(self.trUtf8(
                """<b>Zoom</b>"""
                """<p>Zoom the text. This opens a dialog where the"""
                """ desired size can be entered.</p>"""
                ))
        self.zoomToAct.connectIt(SIGNAL('activated()'),self.handleZoom)
        self.viewActions.append(self.zoomToAct)
        
        self.toggleAllAct = E3Action(self.trUtf8('Toggle all folds'),
                            self.trUtf8('Toggle &all folds'),
                            0, 0, self.viewFoldActGrp, 'vm_view_toggle_all_folds')
        self.toggleAllAct.setStatusTip(self.trUtf8('Toggle all folds'))
        self.toggleAllAct.setWhatsThis(self.trUtf8(
                """<b>Toggle all folds</b>"""
                """<p>Toggle all folds of the current editor.</p>"""
                ))
        self.toggleAllAct.connectIt(SIGNAL('activated()'),self.handleToggleAll)
        self.viewActions.append(self.toggleAllAct)
        
        self.toggleCurrentAct = E3Action(self.trUtf8('Toggle current fold'),
                            self.trUtf8('Toggle &current fold'),
                            0, 0, self.viewFoldActGrp, 'vm_view_toggle_current_fold')
        self.toggleCurrentAct.setStatusTip(self.trUtf8('Toggle current fold'))
        self.toggleCurrentAct.setWhatsThis(self.trUtf8(
                """<b>Toggle current fold</b>"""
                """<p>Toggle the folds of the current line of the current editor.</p>"""
                ))
        self.toggleCurrentAct.connectIt(SIGNAL('activated()'),self.handleToggleCurrent)
        self.viewActions.append(self.toggleCurrentAct)
        
        self.unhighlightAct = E3Action(self.trUtf8('Remove all highlights'),
                            QIconSet(UI.PixmapCache.getPixmap("unhighlight.png")),
                            self.trUtf8('Remove all highlights'),
                            0, 0, self, 'vm_view_unhighlight')
        self.unhighlightAct.setStatusTip(self.trUtf8('Remove all highlights'))
        self.unhighlightAct.setWhatsThis(self.trUtf8(
                """<b>Remove all highlights</b>"""
                """<p>Remove the highlights of all editors.</p>"""
                ))
        self.unhighlightAct.connectIt(SIGNAL('activated()'),self.unhighlight)
        self.viewActions.append(self.unhighlightAct)
        
        self.splitViewAct = E3Action(self.trUtf8('Split view'),
                            QIconSet(UI.PixmapCache.getPixmap("splitVertical.png")),
                            self.trUtf8('&Split view'),
                            0, 0, self, 'vm_view_split_view')
        self.splitViewAct.setStatusTip(self.trUtf8('Add a split to the view'))
        self.splitViewAct.setWhatsThis(self.trUtf8(
                """<b>Split view</b>"""
                """<p>Add a split to the view.</p>"""
                ))
        self.splitViewAct.connectIt(SIGNAL('activated()'),self.handleSplitView)
        self.viewActions.append(self.splitViewAct)
        
        self.splitOrientationAct = E3Action(self.trUtf8('Arrange horizontally'),
                            self.trUtf8('Arrange &horizontally'),
                            0, 0, self, 'vm_view_arrange_horizontally', 1)
        self.splitOrientationAct.setStatusTip(self.trUtf8('Arrange the splitted views horizontally'))
        self.splitOrientationAct.setWhatsThis(self.trUtf8(
                """<b>Arrange horizontally</b>"""
                """<p>Arrange the splitted views horizontally.</p>"""
                ))
        self.splitOrientationAct.setOn(0)
        self.splitOrientationAct.connectIt(SIGNAL('activated()'),self.handleSplitOrientation)
        self.viewActions.append(self.splitOrientationAct)
        
        self.splitRemoveAct = E3Action(self.trUtf8('Remove split'),
                            QIconSet(UI.PixmapCache.getPixmap("remsplitVertical.png")),
                            self.trUtf8('&Remove split'),
                            0, 0, self, 'vm_view_remove_split')
        self.splitRemoveAct.setStatusTip(self.trUtf8('Remove the current split'))
        self.splitRemoveAct.setWhatsThis(self.trUtf8(
                """<b>Remove split</b>"""
                """<p>Remove the current split.</p>"""
                ))
        self.splitRemoveAct.connectIt(SIGNAL('activated()'),self.removeSplit)
        self.viewActions.append(self.splitRemoveAct)
        
        self.nextSplitAct = E3Action(self.trUtf8('Next split'),
                            self.trUtf8('&Next split'),
                            QKeySequence(self.trUtf8("CTRL+Alt+N", "View|Next split")), 0,
                            self, 'vm_next_split', 1)
        self.nextSplitAct.setStatusTip(self.trUtf8('Move to the next split'))
        self.nextSplitAct.setWhatsThis(self.trUtf8(
                """<b>Next split</b>"""
                """<p>Move to the next split.</p>"""
                ))
        self.nextSplitAct.connectIt(SIGNAL('activated()'),self.nextSplit)
        self.viewActions.append(self.nextSplitAct)
        
        self.prevSplitAct = E3Action(self.trUtf8('Previous split'),
                            self.trUtf8('&Previous split'),
                            QKeySequence(self.trUtf8("CTRL+Alt+P", "View|Previous split")), 0,
                            self, 'vm_previous_split', 1)
        self.prevSplitAct.setStatusTip(self.trUtf8('Move to the previous split'))
        self.prevSplitAct.setWhatsThis(self.trUtf8(
                """<b>Previous split</b>"""
                """<p>Move to the previous split.</p>"""
                ))
        self.prevSplitAct.connectIt(SIGNAL('activated()'),self.prevSplit)
        self.viewActions.append(self.prevSplitAct)
        
        self.viewActGrp.setEnabled(0)
        self.viewFoldActGrp.setEnabled(0)
        self.unhighlightAct.setEnabled(0)
        self.splitViewAct.setEnabled(0)
        self.splitOrientationAct.setEnabled(0)
        self.splitRemoveAct.setEnabled(0)
        self.nextSplitAct.setEnabled(0)
        self.prevSplitAct.setEnabled(0)
        
    def initViewMenu(self):
        """
        Public method to create the View menu
        
        @return the generated menu
        """
        menu = QPopupMenu(self.ui)
        menu.insertTearOffHandle()
        self.viewActGrp.addTo(menu)
        menu.insertSeparator()
        self.viewFoldActGrp.addTo(menu)
        menu.insertSeparator()
        self.unhighlightAct.addTo(menu)
        if self.canSplit():
            menu.insertSeparator()
            self.splitViewAct.addTo(menu)
            self.splitOrientationAct.addTo(menu)
            self.splitRemoveAct.addTo(menu)       
            self.nextSplitAct.addTo(menu)
            self.prevSplitAct.addTo(menu)
        return menu
        
    def initViewToolbar(self):
        """
        Public method to create the View toolbar
        
        @return the generated toolbar
        """
        tb = QToolBar(self.ui)
        self.viewActGrp.addTo(tb)
        tb.addSeparator()
        self.unhighlightAct.addTo(tb)
        if self.canSplit():
            tb.addSeparator()
            self.splitViewAct.addTo(tb)
            self.splitRemoveAct.addTo(tb)
        
        return tb
        
    ##################################################################
    ## Initialize the macro related actions and macro menu
    ##################################################################
    
    def initMacroActions(self):
        """
        Private method defining the user interface actions for the macro commands.
        """
        self.macroActGrp = E3ActionGroup(self)

        self.macroStartRecAct = E3Action(self.trUtf8('Start Macro Recording'),
                            self.trUtf8('S&tart Macro Recording'),
                            0, 0, self.macroActGrp, 'vm_macro_start_recording')
        self.macroStartRecAct.setStatusTip(self.trUtf8('Start Macro Recording'))
        self.macroStartRecAct.setWhatsThis(self.trUtf8(
                """<b>Start Macro Recording</b>"""
                """<p>Start recording editor commands into a new macro.</p>"""
                ))
        self.macroStartRecAct.connectIt(SIGNAL('activated()'),self.handleMacroStartRecording)
        self.macroActions.append(self.macroStartRecAct)
        
        self.macroStopRecAct = E3Action(self.trUtf8('Stop Macro Recording'),
                            self.trUtf8('Sto&p Macro Recording'),
                            0, 0, self.macroActGrp, 'vm_macro_stop_recording')
        self.macroStopRecAct.setStatusTip(self.trUtf8('Stop Macro Recording'))
        self.macroStopRecAct.setWhatsThis(self.trUtf8(
                """<b>Stop Macro Recording</b>"""
                """<p>Stop recording editor commands into a new macro.</p>"""
                ))
        self.macroStopRecAct.connectIt(SIGNAL('activated()'),self.handleMacroStopRecording)
        self.macroActions.append(self.macroStopRecAct)
        
        self.macroRunAct = E3Action(self.trUtf8('Run Macro'),
                            self.trUtf8('&Run Macro'),
                            0, 0, self.macroActGrp, 'vm_macro_run')
        self.macroRunAct.setStatusTip(self.trUtf8('Run Macro'))
        self.macroRunAct.setWhatsThis(self.trUtf8(
                """<b>Run Macro</b>"""
                """<p>Run a previously recorded editor macro.</p>"""
                ))
        self.macroRunAct.connectIt(SIGNAL('activated()'),self.handleMacroRun)
        self.macroActions.append(self.macroRunAct)
        
        self.macroDeleteAct = E3Action(self.trUtf8('Delete Macro'),
                            self.trUtf8('&Delete Macro'),
                            0, 0, self.macroActGrp, 'vm_macro_delete')
        self.macroDeleteAct.setStatusTip(self.trUtf8('Delete Macro'))
        self.macroDeleteAct.setWhatsThis(self.trUtf8(
                """<b>Delete Macro</b>"""
                """<p>Delete a previously recorded editor macro.</p>"""
                ))
        self.macroDeleteAct.connectIt(SIGNAL('activated()'),self.handleMacroDelete)
        self.macroActions.append(self.macroDeleteAct)
        
        self.macroLoadAct = E3Action(self.trUtf8('Load Macro'),
                            self.trUtf8('&Load Macro'),
                            0, 0, self.macroActGrp, 'vm_macro_load')
        self.macroLoadAct.setStatusTip(self.trUtf8('Load Macro'))
        self.macroLoadAct.setWhatsThis(self.trUtf8(
                """<b>Load Macro</b>"""
                """<p>Load an editor macro from a file.</p>"""
                ))
        self.macroLoadAct.connectIt(SIGNAL('activated()'),self.handleMacroLoad)
        self.macroActions.append(self.macroLoadAct)
        
        self.macroSaveAct = E3Action(self.trUtf8('Save Macro'),
                            self.trUtf8('&Save Macro'),
                            0, 0, self.macroActGrp, 'vm_macro_save')
        self.macroSaveAct.setStatusTip(self.trUtf8('Save Macro'))
        self.macroSaveAct.setWhatsThis(self.trUtf8(
                """<b>Save Macro</b>"""
                """<p>Save a previously recorded editor macro to a file.</p>"""
                ))
        self.macroSaveAct.connectIt(SIGNAL('activated()'),self.handleMacroSave)
        self.macroActions.append(self.macroSaveAct)
        
        self.macroActGrp.setEnabled(0)
        
    def initMacroMenu(self):
        """
        Public method to create the Macro menu
        
        @return the generated menu
        """
        menu = QPopupMenu(self.ui)
        menu.insertTearOffHandle()
        self.macroActGrp.addTo(menu)
        return menu
    
    #####################################################################
    ## Initialize the bookmark related actions, bookmark menu and toolbar
    #####################################################################
    
    def initBookmarkActions(self):
        """
        Private method defining the user interface actions for the bookmarks commands.
        """
        self.bookmarkActGrp = E3ActionGroup(self)

        self.bookmarkToggleAct = E3Action(self.trUtf8('Toggle Bookmark'),
                            QIconSet(UI.PixmapCache.getPixmap("bookmarkToggle.png")),
                            self.trUtf8('&Toggle Bookmark'),
                            QKeySequence(self.trUtf8("ALT+CTRL+T","Bookmark|Toggle")), 0,
                            self.bookmarkActGrp, 'vm_bookmark_toggle')
        self.bookmarkToggleAct.setStatusTip(self.trUtf8('Toggle Bookmark'))
        self.bookmarkToggleAct.setWhatsThis(self.trUtf8(
                """<b>Toggle Bookmark</b>"""
                """<p>Toggle a bookmark at the current line of the current editor.</p>"""
                ))
        self.bookmarkToggleAct.connectIt(SIGNAL('activated()'),self.handleToggleBookmark)
        self.bookmarkActions.append(self.bookmarkToggleAct)
        
        self.bookmarkNextAct = E3Action(self.trUtf8('Next Bookmark'),
                            QIconSet(UI.PixmapCache.getPixmap("bookmarkNext.png")),
                            self.trUtf8('&Next Bookmark'),
                            QKeySequence(self.trUtf8("CTRL+PgDown","Bookmark|Next")), 0,
                            self.bookmarkActGrp, 'vm_bookmark_next')
        self.bookmarkNextAct.setStatusTip(self.trUtf8('Next Bookmark'))
        self.bookmarkNextAct.setWhatsThis(self.trUtf8(
                """<b>Next Bookmark</b>"""
                """<p>Go to next bookmark of the current editor.</p>"""
                ))
        self.bookmarkNextAct.connectIt(SIGNAL('activated()'),self.handleNextBookmark)
        self.bookmarkActions.append(self.bookmarkNextAct)
        
        self.bookmarkPreviousAct = E3Action(self.trUtf8('Previous Bookmark'),
                            QIconSet(UI.PixmapCache.getPixmap("bookmarkPrevious.png")),
                            self.trUtf8('&Previous Bookmark'),
                            QKeySequence(self.trUtf8("CTRL+PgUp","Bookmark|Previous")), 0,
                            self.bookmarkActGrp, 'vm_bookmark_previous')
        self.bookmarkPreviousAct.setStatusTip(self.trUtf8('Previous Bookmark'))
        self.bookmarkPreviousAct.setWhatsThis(self.trUtf8(
                """<b>Previous Bookmark</b>"""
                """<p>Go to previous bookmark of the current editor.</p>"""
                ))
        self.bookmarkPreviousAct.connectIt(SIGNAL('activated()'),self.handlePreviousBookmark)
        self.bookmarkActions.append(self.bookmarkPreviousAct)
        
        self.bookmarkClearAct = E3Action(self.trUtf8('Clear Bookmarks'),
                            self.trUtf8('&Clear Bookmarks'),
                            QKeySequence(self.trUtf8("ALT+CTRL+C","Bookmark|Clear")), 0,
                            self.bookmarkActGrp, 'vm_bookmark_clear')
        self.bookmarkClearAct.setStatusTip(self.trUtf8('Clear Bookmarks'))
        self.bookmarkClearAct.setWhatsThis(self.trUtf8(
                """<b>Clear Bookmarks</b>"""
                """<p>Clear bookmarks of all editors.</p>"""
                ))
        self.bookmarkClearAct.connectIt(SIGNAL('activated()'),self.handleClearAllBookmarks)
        self.bookmarkActions.append(self.bookmarkClearAct)
        
        self.syntaxErrorGotoAct = E3Action(self.trUtf8('Goto Syntax Error'),
                            QIconSet(UI.PixmapCache.getPixmap("syntaxErrorGoto.png")),
                            self.trUtf8('&Goto Syntax Error'),
                            0, 0,
                            self.bookmarkActGrp, 'vm_syntaxerror_goto')
        self.syntaxErrorGotoAct.setStatusTip(self.trUtf8('Goto Syntax Error'))
        self.syntaxErrorGotoAct.setWhatsThis(self.trUtf8(
                """<b>Goto Syntax Error</b>"""
                """<p>Go to next syntax error of the current editor.</p>"""
                ))
        self.syntaxErrorGotoAct.connectIt(SIGNAL('activated()'),self.handleGotoSyntaxError)
        self.bookmarkActions.append(self.syntaxErrorGotoAct)
        
        self.syntaxErrorClearAct = E3Action(self.trUtf8('Clear Syntax Errors'),
                            self.trUtf8('Clear &Syntax Errors'),
                            0, 0,
                            self.bookmarkActGrp, 'vm_syntaxerror_clear')
        self.syntaxErrorClearAct.setStatusTip(self.trUtf8('Clear Syntax Errors'))
        self.syntaxErrorClearAct.setWhatsThis(self.trUtf8(
                """<b>Clear Syntax Errors</b>"""
                """<p>Clear syntax errors of all editors.</p>"""
                ))
        self.syntaxErrorClearAct.connectIt(SIGNAL('activated()'),self.handleClearAllSyntaxErrors)
        self.bookmarkActions.append(self.syntaxErrorClearAct)
        
        self.notcoveredNextAct = E3Action(self.trUtf8('Next uncovered line'),
                            QIconSet(UI.PixmapCache.getPixmap("notcoveredNext.png")),
                            self.trUtf8('&Next uncovered line'),
                            0, 0,
                            self.bookmarkActGrp, 'vm_uncovered_next')
        self.notcoveredNextAct.setStatusTip(self.trUtf8('Next uncovered line'))
        self.notcoveredNextAct.setWhatsThis(self.trUtf8(
                """<b>Next uncovered line</b>"""
                """<p>Go to next line of the current editor marked as not covered.</p>"""
                ))
        self.notcoveredNextAct.connectIt(SIGNAL('activated()'),self.handleNextUncovered)
        self.bookmarkActions.append(self.notcoveredNextAct)
        
        self.notcoveredPreviousAct = E3Action(self.trUtf8('Previous uncovered line'),
                            QIconSet(UI.PixmapCache.getPixmap("notcoveredPrev.png")),
                            self.trUtf8('&Previous uncovered line'),
                            0, 0,
                            self.bookmarkActGrp, 'vm_uncovered_previous')
        self.notcoveredPreviousAct.setStatusTip(self.trUtf8('Previous uncovered line'))
        self.notcoveredPreviousAct.setWhatsThis(self.trUtf8(
                """<b>Previous uncovered line</b>"""
                """<p>Go to previous line of the current editor marked as not covered.</p>"""
                ))
        self.notcoveredPreviousAct.connectIt(SIGNAL('activated()'),self.handlePreviousUncovered)
        self.bookmarkActions.append(self.notcoveredPreviousAct)
        
        self.bookmarkActGrp.setEnabled(0)
        
    def initBookmarkMenu(self):
        """
        Public method to create the Bookmark menu
        
        @return the generated menu
        """
        menu = QPopupMenu(self.ui)
        self.bookmarksMenu = QPopupMenu(menu)
        
        menu.insertTearOffHandle()
        self.bookmarkToggleAct.addTo(menu)
        self.bookmarkNextAct.addTo(menu)
        self.bookmarkPreviousAct.addTo(menu)
        self.bookmarkClearAct.addTo(menu)
        menu.insertSeparator()
        menu.insertItem(self.trUtf8('&Bookmarks'), self.bookmarksMenu)
        menu.insertSeparator()
        self.syntaxErrorGotoAct.addTo(menu)
        self.syntaxErrorClearAct.addTo(menu)
        menu.insertSeparator()
        self.notcoveredNextAct.addTo(menu)
        self.notcoveredPreviousAct.addTo(menu)
        
        self.connect(self.bookmarksMenu,SIGNAL('aboutToShow()'),self.handleShowBookmarksMenu)
        self.connect(self.bookmarksMenu,SIGNAL('activated(int)'),self.handleBookmarkSelected)
        
        return menu
        
    def initBookmarkToolbar(self):
        """
        Public method to create the Bookmark toolbar
        
        @return the generated toolbar
        """
        tb = QToolBar(self.ui)
        self.bookmarkToggleAct.addTo(tb)
        self.bookmarkNextAct.addTo(tb)
        self.bookmarkPreviousAct.addTo(tb)
        tb.addSeparator()
        self.syntaxErrorGotoAct.addTo(tb)
        tb.addSeparator()
        self.notcoveredNextAct.addTo(tb)
        self.notcoveredPreviousAct.addTo(tb)
        
        return tb
    
    ##################################################################
    ## Methods and slots that deal with file and window handling
    ##################################################################
    
    def handleOpen(self,prog=None):
        """
        Public slot to open a Python (or other) file.
        
        @param prog name of file to be opened (string or QString)
        """
        # Get the file name if one wasn't specified.
        if prog is None:
            # set the cwd of the dialog based on the following search criteria:
            #     1: Directory of currently active editor
            #     2: Directory of currently active Project
            #     3: CWD
            filter = self._getOpenFileFilter()
            progs = KQFileDialog.getOpenFileNames(self.fileFiltersString,
                            self._getOpenStartDir(), self.ui, None, QString.null, filter)
        else:
            progs = [prog]

        for prog in progs:
            prog = Utilities.normabspath(unicode(prog))
            # Open up the new files.
            self.handlePythonFile(prog)

    def checkDirty(self, editor):
        """
        Private method to check dirty status and open a message window.
        
        @param editor editor window to check
        @return flag indicating successful reset of the dirty flag (boolean)
        """
        if editor.isModified():
            fn = editor.getFileName()
            # ignore the dirty status, if there is more than one open editor
            # for the same file
            if fn and self.getOpenEditorCount(fn) > 1:
                return 1
            if fn is None:
                fn = self.trUtf8('Noname')
            res = KQMessageBox.warning(self.parent(), 
                self.trUtf8("File Modified"),
                self.trUtf8("<p>The file <b>%1</b> has unsaved changes.</p>")
                    .arg(fn),
                self.trUtf8("&Save"), self.trUtf8("&Discard changes"),
                self.trUtf8("&Abort"), 0, 2)
            if res == 0:
                (ok, newName) = editor.saveFile()
                if ok:
                    self.setEditorName(editor, newName)
                return ok
            elif res == 2:
                return 0
        
        return 1
        
    def checkAllDirty(self):
        """
        Public method to check the dirty status of all editors.
        
        @return flag indicating successful reset of all dirty flags (boolean)
        """
        for editor in self.editors:
            if not self.checkDirty(editor):
                return 0
                
        return 1
        
    def closeEditor(self, editor):
        """
        Private method to close an editor window.
        
        @param editor editor window to be closed
        @return flag indicating success (boolean)
        """
        # save file if necessary
        if not self.checkDirty(editor):
            return 0
        
        # get the filename of the editor for later use
        fn = editor.getFileName()
        
        # remove the window
        self.removeView(editor)
        self.editors.remove(editor)
        
        # send a signal, if it was the last editor for this filename
        if fn and self.getOpenEditor(fn) is None:
            self.emit(PYSIGNAL('editorClosed'), (fn,))
        
        # send a signal, if it was the very last editor
        if not len(self.editors):
            self.handleLastEditorClosed()
            self.emit(PYSIGNAL('lastEditorClosed'), ())
        
        return 1
    
    def handleClose(self):
        """
        Public method to close the current window.
        
        @return flag indicating success (boolean)
        """
        aw = self.activeWindow()
        if aw is None:
            return 0
            
        res = self.closeEditor(aw)
        if res and aw == self.currentEditor:
            self.currentEditor = None
            
        return res
            
    def handleCloseAll(self):
        """
        Private method to close all editor windows via file menu.
        """
        savedEditors = self.editors[:]
        for editor in savedEditors:
            self.closeEditor(editor)
            
    def handleCloseWindow(self, fn):
        """
        Public method to close an arbitrary source editor.
        
        @param fn filename of editor to be closed
        @return flag indicating success (boolean)
        """
        for editor in self.editors:
            if Utilities.samepath(fn, editor.getFileName()):
                break
        else:
            return 1
            
        res = self.closeEditor(editor)
        if res and editor == self.currentEditor:
            self.currentEditor = None
            
        return res
        
    def handleCloseEditor(self, editor):
        """
        Public method to close an arbitrary source editor.
        
        @param editor editor to be closed
        """
        if editor is None:
            return
            
        res = self.closeEditor(editor)
        if res and editor == self.currentEditor:
            self.currentEditor = None
    
    def handleExit(self):
        """
        Public method to handle the debugged program terminating.
        """
        if self.currentEditor is not None:
            self.currentEditor.highlight()
            self.currentEditor = None
            
        self.setSbFile()

    def handlePythonFile(self,pyfn,lineno=None,selection=None):
        """
        Public method to handle the user selecting a file for display.
        
        @param pyfn name of file to be opened
        @param lineno line number to place the cursor at
        @param selection tuple (start, end) of an area to be selected
        """
        try:
            self.displayPythonFile(pyfn,lineno,selection)
        except IOError:
            pass

    def displayPythonFile(self,fn,lineno=None,selection=None):
        """
        Public slot to display a file in an editor.
        
        @param fn name of file to be opened
        @param lineno line number to place the cursor at
        @param selection tuple (start, end) of an area to be selected
        """
        isPyFile = lineno and lineno < 0
        try:
            newWin, editor = self.getEditor(fn, isPythonFile=isPyFile)
        except IOError:
            return
        
        if newWin:
            self.handleModificationStatusChanged(editor.isModified(), editor)
        self.checkActions(editor)
            
        if lineno is not None and lineno >= 0:
            editor.ensureVisibleTop(lineno)
            editor.gotoLine(lineno)
            
        if selection is not None:
            editor.setSelection(lineno-1, selection[0], lineno-1, selection[1])
        
        # insert filename into list of recently opened files
        self.addToRecentList(fn)
        
    def newEditorView(self, fn, caller, isPythonFile=0):
        """
        Public method to create a new editor displaying the given document.
        
        @param fn filename of this view
        @param caller reference to the editor calling this method
        @param isPythonFile flag indicating that this is a Python file
                even if it doesn't have the .py extension (boolean)
        """
        editor = Editor(self.dbs,fn,self,isPythonFile=isPythonFile, editor=caller,
                        tv=self.ui.getTaskViewer())
        self.editors.append(editor)
        self.connect(editor, PYSIGNAL('modificationStatusChanged'),
            self.handleModificationStatusChanged)
        self.connect(editor, PYSIGNAL('cursorChanged'), self.handleCursorChanged)
        self.connect(editor, PYSIGNAL('editorSaved'), self.handleEditorSaved)
        self.connect(editor, PYSIGNAL('breakpointToggled'), self.handleBreakpointToggled)
        self.connect(editor, PYSIGNAL('breakpointEnabledToggled'), 
            self.handleBreakpointEnabledToggled)
        self.connect(editor, PYSIGNAL('bookmarkToggled'), self.handleBookmarkToggled)
        self.connect(editor, PYSIGNAL('syntaxerrorToggled'), self.handleSyntaxErrorToggled)
        self.connect(editor, PYSIGNAL('coverageMarkersShown'), self.handleCoverageMarkersShown)
        self.connect(editor, PYSIGNAL('autoCompletionAPIsAvailable'), 
            self.handleEditoracAPIsAvailable)
        self.connect(editor, PYSIGNAL('undoAvailable'), self.undoAct.setEnabled)
        self.connect(editor, PYSIGNAL('redoAvailable'), self.redoAct.setEnabled)
        self.handleEditorOpened()
        self.emit(PYSIGNAL('editorOpened'), (fn,))
        
        self.connect(caller, PYSIGNAL('editorRenamed'), editor.handleRenamed)
        self.connect(editor, PYSIGNAL('editorRenamed'), caller.handleRenamed)
        
        self.addView(editor, fn)
        self.handleModificationStatusChanged(editor.isModified(), editor)
        self.checkActions(editor)
        
    def addToRecentList(self, fn):
        """
        Public slot to add a filename to the list of recently opened files.
        
        @param fn name of the file to be added
        """
        self.recent.remove(fn)
        self.recent.prepend(fn)
        if len(self.recent) > 9:
            self.recent = self.recent[:9]

    def toggleWindow(self,w):
        """
        Private method to toggle a workspace window.
        
        @param w editor window to be toggled
        """
        if w.isHidden():
            w.show()
        else:
            w.hide()

    def setFileLine(self,fn,line,error=0,syntaxError=0):
        """
        Public method to update the user interface when the current program or line changes.
        
        @param fn filename of editor to update (string)
        @param line line number to highlight (int)
        @param error flag indicating an error highlight (boolean)
        @param syntaxError flag indicating a syntax error
        """
        self.setSbFile(fn,line)

        try:
            newWin, self.currentEditor = self.getEditor(fn)
        except IOError:
            return

        # Change the highlighted line.
        self.currentEditor.highlight(line,error,syntaxError)

        self.currentEditor.highlightVisible()
        self.checkActions(self.currentEditor, 0)
            
    def setSbFile(self,fn=None,line=None,pos=None):
        """
        Private method to set the file info in the status bar.
        
        @param fn filename to display (string)
        @param line line number to display (int)
        @param pos character position to display (int)
        """
        if fn is None:
            fn = ''
            writ = '   '
        else:
            if QFileInfo(fn).isWritable():
                writ = ' rw'
            else:
                writ = ' ro'
        
        self.sbWritable.setText(writ)
        self.sbFile.setText(self.trUtf8('File: %1').arg(fn,-50))

        if line is None:
            line = ''

        self.sbLine.setText(self.trUtf8('Line: %1').arg(line,5))

        if pos is None:
            pos = ''
            
        self.sbPos.setText(self.trUtf8('Pos: %1').arg(pos, 5))
        
    def unhighlight(self, current=0):
        """
        Public method to switch off all highlights.
        
        @param current flag indicating only the current editor should be unhighlighted
                (boolean)
        """
        if current: 
            if self.currentEditor is not None:
                self.currentEditor.highlight()
        else:
            for editor in self.editors:
                editor.highlight()

    def getOpenFilenames(self):
        """
        Public method returning a list of the filenames of all editors.
        
        @return list of all opened filenames (list of strings)
        """
        filenames = []
        for editor in self.editors:
            fn = editor.getFileName()
            if fn is not None and fn not in filenames:
                filenames.append(fn)
                
        return filenames
        
    def getEditor(self, fn, isPythonFile=0):
        """
        Private method to return the editor displaying the given file.
        
        If there is no editor with the given file, a new editor window is
        created.
        
        @param fn filename to look for
        @param isPythonFile flag indicating that this is a Python file
                even if it doesn't have the .py extension (boolean)
        @return tuple of two values giving a flag indicating a new window creation and
            a reference to the editor displaying this file
        """
        newWin = 0
        editor = self.activeWindow()
        if editor is None or not Utilities.samepath(fn, editor.getFileName()):
            for editor in self.editors:
                if Utilities.samepath(fn, editor.getFileName()):
                    break
            else:
                editor = Editor(self.dbs,fn,self,isPythonFile=isPythonFile,
                                tv=self.ui.getTaskViewer())
                self.editors.append(editor)
                self.connect(editor, PYSIGNAL('modificationStatusChanged'),
                    self.handleModificationStatusChanged)
                self.connect(editor, PYSIGNAL('cursorChanged'), self.handleCursorChanged)
                self.connect(editor, PYSIGNAL('editorSaved'), self.handleEditorSaved)
                self.connect(editor, PYSIGNAL('breakpointToggled'), self.handleBreakpointToggled)
                self.connect(editor, PYSIGNAL('breakpointEnabledToggled'), 
                    self.handleBreakpointEnabledToggled)
                self.connect(editor, PYSIGNAL('bookmarkToggled'), self.handleBookmarkToggled)
                self.connect(editor, PYSIGNAL('syntaxerrorToggled'), self.handleSyntaxErrorToggled)
                self.connect(editor, PYSIGNAL('coverageMarkersShown'), self.handleCoverageMarkersShown)
                self.connect(editor, PYSIGNAL('autoCompletionAPIsAvailable'), 
                    self.handleEditoracAPIsAvailable)
                self.connect(editor, PYSIGNAL('undoAvailable'), self.undoAct.setEnabled)
                self.connect(editor, PYSIGNAL('redoAvailable'), self.redoAct.setEnabled)
                self.handleEditorOpened()
                self.emit(PYSIGNAL('editorOpened'), (fn,))
                newWin = 1

        if newWin:
            self.addView(editor, fn)
        else:
            self.showView(editor, fn)
            
        return (newWin, editor)
        
    def getOpenEditor(self, fn):
        """
        Public method to return the editor displaying the given file.
        
        @param fn filename to look for
        @return a reference to the editor displaying this file or None, if
            no editor was found
        """
        for editor in self.editors:
            if Utilities.samepath(fn, editor.getFileName()):
                return editor
                
        return None

    def getOpenEditorCount(self, fn):
        """
        Public method to return the count of editors displaying the given file.
        
        @param fn filename to look for
        @return count of editors displaying this file (integer)
        """
        count = 0
        for editor in self.editors:
            if Utilities.samepath(fn, editor.getFileName()):
                count += 1
        return count
    
    def getActiveName(self):
        """
        Public method to retrieve the filename of the active window.
        
        @return filename of active window (string)
        """
        aw = self.activeWindow()
        if aw:
            return aw.getFileName()
        else:
            return None

    def saveEditor(self, fn):
        """
        Public method to save a named editor file.
        
        @param fn filename of editor to be saved (string)
        @return flag indicating success (boolean)
        """
        for editor in self.editors:
            if Utilities.samepath(fn, editor.getFileName()):
                break
        else:
            return 1
            
        if not editor.isModified():
            return 1
        else:
            ok, dummy = editor.saveFile()
            return ok
        
    def saveEditorEd(self, ed):
        """
        Public slot to save the contents of an editor.
        
        @param ed editor to be saved
        """
        if ed:
            ok, newName = ed.saveFile()
            if ok:
                self.setEditorName(ed, newName)
        else:
            return
        
    def saveCurrentEditor(self):
        """
        Public slot to save the contents of the current editor.
        """
        aw = self.activeWindow()
        self.saveEditorEd(aw)

    def saveAsEditorEd(self, ed):
        """
        Public slot to save the contents of an editor to a new file.
        
        @param ed editor to be saved
        """
        if ed:
            ok, newName = ed.saveFileAs()
            if ok:
                self.setEditorName(ed, newName)
        else:
            return
        
    def saveAsCurrentEditor(self):
        """
        Public slot to save the contents of the current editor to a new file.
        """
        aw = self.activeWindow()
        self.saveAsEditorEd(aw)

    def saveEditorsList(self, editors):
        """
        Public slot to save a list of editors.
        
        @param editors list of editors to be saved
        """
        for editor in editors:
            ok, newName = editor.saveFile()
            if ok:
                self.setEditorName(editor, newName)
        
    def saveAllEditors(self):
        """
        Public slot to save the contents of all editors.
        """
        for editor in self.editors:
            ok, newName = editor.saveFile()
            if ok:
                self.setEditorName(editor, newName)
        
        # restart autosave timer
        if self.autosaveInterval > 0:
            self.autosaveTimer.start(self.autosaveInterval * 60000, 1)

    def saveEditorToProjectEd(self, ed):
        """
        Public slot to save the contents of an editor the current project.
        
        @param ed editor to be saved
        """
        pro = self.ui.getProject()
        path = pro.ppath
        if ed:
            ok, newName = ed.saveFileAs(path)
            if ok:
                self.setEditorName(ed, newName)
                pro.appendFile(newName)
        else:
            return
        
    def saveCurrentEditorToProject(self):
        """
        Public slot to save the contents of the current editor to the current project.
        """
        aw = self.activeWindow()
        self.saveEditorToProjectEd(aw)
        
    def newEditor(self):
        """
        Public slot to generate a new empty editor.
        """
        editor = Editor(self.dbs, None, self, tv=self.ui.getTaskViewer())
        self.editors.append(editor)
        self.connect(editor, PYSIGNAL('modificationStatusChanged'),
            self.handleModificationStatusChanged)
        self.connect(editor, PYSIGNAL('cursorChanged'), self.handleCursorChanged)
        self.connect(editor, PYSIGNAL('editorSaved'), self.handleEditorSaved)
        self.connect(editor, PYSIGNAL('breakpointToggled'), self.handleBreakpointToggled)
        self.connect(editor, PYSIGNAL('breakpointEnabledToggled'), 
            self.handleBreakpointEnabledToggled)
        self.connect(editor, PYSIGNAL('bookmarkToggled'), self.handleBookmarkToggled)
        self.connect(editor, PYSIGNAL('syntaxerrorToggled'), self.handleSyntaxErrorToggled)
        self.connect(editor, PYSIGNAL('coverageMarkersShown'), self.handleCoverageMarkersShown)
        self.connect(editor, PYSIGNAL('autoCompletionAPIsAvailable'), 
            self.handleEditoracAPIsAvailable)
        self.connect(editor, PYSIGNAL('undoAvailable'), self.undoAct.setEnabled)
        self.connect(editor, PYSIGNAL('redoAvailable'), self.redoAct.setEnabled)
        self.addView(editor, None)
        self.handleEditorOpened()
        self.checkActions(editor)
        self.emit(PYSIGNAL('editorOpened'), (None,))
        
    def printEditor(self, editor):
        """
        Public slot to print an editor.
        
        @param editor editor to be printed
        """
        if editor:
            editor.printFile()
        else:
            return
        
    def printCurrentEditor(self):
        """
        Public slot to print the contents of the current editor.
        """
        aw = self.activeWindow()
        self.printEditor(aw)

    def printCurrentEditorSel(self):
        """
        Public slot to print the selection of the current editor.
        """
        aw = self.activeWindow()
        if aw:
            aw.printSelection()
        else:
            return

    def handleShowRecentMenu(self):
        """
        Private method to set up recent files menu.
        """
        idx = 0
        self.recentMenu.clear()
        
        for rp in self.recent:
            id = self.recentMenu.insertItem('&%d. %s' % (idx+1, unicode(rp)),
                                            self.handleOpenRecent)
            self.recentMenu.setItemParameter(id,idx)
            
            idx = idx + 1
            
        self.recentMenu.insertSeparator()
        self.recentMenu.insertItem(self.trUtf8('&Clear'), self.handleClearRecent)
        
    def handleOpenRecent(self, idx):
        """
        Private method to open a file from the list of rencently opened files.
        
        @param idx index of the selected entry (int)
        """
        self.handlePythonFile(unicode(self.recent[idx]))
        
    def handleClearRecent(self):
        """
        Private method to clear the recent files menu.
        """
        self.recent = QStringList()
        
    def handleShowBookmarkedMenu(self):
        """
        Private method to set up bookmarked files menu.
        """
        idx = 0
        self.bookmarkedMenu.clear()
        
        for rp in self.bookmarked:
            id = self.bookmarkedMenu.insertItem(rp, self.handleOpenBookmarked)
            self.bookmarkedMenu.setItemParameter(id,idx)
            
            idx = idx + 1
            
        self.bookmarkedMenu.insertSeparator()
        self.bookmarkedMenu.insertItem(self.trUtf8('&Add'), self.handleAddBookmarked)
        self.bookmarkedMenu.insertItem(self.trUtf8('&Edit...'), self.handleEditBookmarked)
        self.bookmarkedMenu.insertItem(self.trUtf8('&Clear'), self.handleClearBookmarked)
        
    def handleOpenBookmarked(self, idx):
        """
        Private method to open a file from the list of bookmarked files.
        
        @param idx index of the selected entry (int)
        """
        self.handlePythonFile(unicode(self.bookmarked[idx]))
        
    def handleAddBookmarked(self):
        """
        Private method to add the current file to the list of bookmarked files.
        """
        an = self.getActiveName()
        if an is not None and self.bookmarked.findIndex(QString(an)) == -1:
            self.bookmarked.append(an)
        
    def handleEditBookmarked(self):
        """
        Private method to edit the list of bookmarked files.
        """
        dlg = BookmarkedFilesDialog(self.bookmarked, self.ui)
        if dlg.exec_loop() == QDialog.Accepted:
            self.bookmarked = QStringList(dlg.getBookmarkedFiles())
        
    def handleClearBookmarked(self):
        """
        Private method to clear the bookmarked files menu.
        """
        self.bookmarked = QStringList()
        
    def handleNewProject(self):
        """
        Public slot to handle the NewProject signal.
        """
        self.saveToProjectAct.setEnabled(1)
        
    def handleProjectOpened(self):
        """
        Public slot to handle the projectOpened signal.
        """
        self.saveToProjectAct.setEnabled(1)
        
    def handleProjectClosed(self):
        """
        Public slot to handle the projectClosed signal.
        """
        self.saveToProjectAct.setEnabled(0)
        
    def handleProjectFileRenamed(self, oldfn, newfn):
        """
        Public slot to handle the projectFileRenamed signal.
        
        @param oldfn old filename of the file (string)
        @param newfn new filename of the file (string)
        """
        editor = self.getOpenEditor(oldfn)
        if editor:
            editor.fileRenamed(newfn)
        
    def enableEditorsCheckFocusIn(self, enabled):
        """
        Public method to set a flag enabling the editors to perform focus in checks.
        
        @param enabled flag indicating focus in checks should be performed (boolean)
        """
        self.editorsCheckFocusIn = enabled
        
    def editorsCheckFocusInEnabled(self):
        """
        Public method returning the flag indicating editors should perform focus in checks.
        
        @return flag indicating focus in checks should be performed (boolean)
        """
        return self.editorsCheckFocusIn

    def handleFindFileName(self):
        """
        Private method to handle the search for file action.
        """
        self.ui.findFileNameDialog.show()
        self.ui.findFileNameDialog.raiseW()
        self.ui.findFileNameDialog.setActiveWindow()
        
    ##################################################################
    ## Below are the action methods for the edit menu
    ##################################################################
    
    def handleEditUndo(self):
        """
        Private method to handle the undo action.
        """
        self.activeWindow().undo()
        
    def handleEditRedo(self):
        """
        Private method to handle the redo action.
        """
        self.activeWindow().redo()
        
    def handleEditRevert(self):
        """
        Private method to handle the revert action.
        """
        self.activeWindow().revertToUnmodified()
        
    def handleEditCut(self):
        """
        Private method to handle the cut action.
        """
        self.activeWindow().cut()
        
    def handleEditCopy(self):
        """
        Private method to handle the copy action.
        """
        if qApp.focusWidget() == self.ui.getShell():
            self.ui.getShell().copy()
        else:
            self.activeWindow().copy()
        
    def handleEditPaste(self):
        """
        Private method to handle the paste action.
        """
        if qApp.focusWidget() == self.ui.getShell():
            self.ui.getShell().paste()
        else:
            self.activeWindow().paste()
        
    def handleEditDelete(self):
        """
        Private method to handle the delete action.
        """
        self.activeWindow().clear()
        
    def handleEditIndent(self):
        """
        Private method to handle the indent action.
        """
        self.activeWindow().indentLineOrSelection()
        
    def handleEditUnindent(self):
        """
        Private method to handle the unindent action.
        """
        self.activeWindow().unindentLineOrSelection()
        
    def handleEditSmartIndent(self):
        """
        Private method to handle the smart indent action
        """
        self.activeWindow().smartIndentLineOrSelection()
        
    def handleEditComment(self):
        """
        Private method to handle the comment action.
        """
        self.activeWindow().commentLineOrSelection()
        
    def handleEditUncomment(self):
        """
        Private method to handle the uncomment action.
        """
        self.activeWindow().uncommentLineOrSelection()
        
    def handleEditStreamComment(self):
        """
        Private method to handle the stream comment action.
        """
        self.activeWindow().streamCommentLineOrSelection()
        
    def handleEditBoxComment(self):
        """
        Private method to handle the box comment action.
        """
        self.activeWindow().boxCommentLineOrSelection()
        
    def handleEditSelectBrace(self):
        """
        Private method to handle the select to brace action.
        """
        self.activeWindow().selectToMatchingBrace()
        
    def handleEditSelectAll(self):
        """
        Private method to handle the select all action.
        """
        self.activeWindow().selectAll(1)
        
    def handleEditDeselectAll(self):
        """
        Private method to handle the select all action.
        """
        self.activeWindow().selectAll(0)
        
    def handleConvertEOL(self):
        """
        Private method to handle the convert line end characters action.
        """
        aw = self.activeWindow()
        aw.convertEols(aw.eolMode())
        
    def handleShortenEmptyLines(self):
        """
        Private method to handle the shorten empty lines action.
        """
        self.activeWindow().handleShortenEmptyLines()
        
    def handleEditAutoComplete(self):
        """
        Private method to handle the autocomplete action.
        """
        aw = self.activeWindow()
        aw.autoComplete()
        
    def handleEditAutoCompleteFromDoc(self):
        """
        Private method to handle the autocomplete from document action.
        """
        aw = self.activeWindow()
        aw.autoCompleteFromDocument()
        
    def handleEditAutoCompleteFromAPIs(self):
        """
        Private method to handle the autocomplete from APIs action.
        """
        aw = self.activeWindow()
        aw.autoCompleteFromAPIs()
        
    def handleEditoracAPIsAvailable(self, available):
        """
        Private method to handle the availability of API autocompletion signal.
        """
        self.autoCompleteFromAPIsAct.setEnabled(available)
        
    ##################################################################
    ## Below are the action and utility methods for the search menu
    ##################################################################

    def getWord(self, text, index):
        """
        Private method to get the word at a position.
        
        @param text text to look at (string or QString)
        @param index position to look at (int)
        @return the word at that position
        """
        re = QRegExp('[^\w_]')
        start = text.findRev(re, index) + 1
        end = text.find(re, index)
        if end > start:
            word = text.mid(start, end-start)
        else:
            word = QString('')
        return word
        
    def textForFind(self, getCurrentWord=1):
        """
        Private method to determine the selection or the current word for the next find operation.
        
        @param getCurrentWord flag indicating to return the current word, if no selected
            text was found (boolean)
        @return selection or current word (QString)
        """
        aw = self.activeWindow()
        if aw is None:
            return QString('')
            
        if aw.hasSelectedText():
            text = aw.selectedText()
            if text.contains('\r') or text.contains('\n'):
                # the selection contains at least a newline, it is
                # unlikely to be the expression to search for
                return QString('')
                
            return text
            
        if getCurrentWord:
            # no selected text, determine the word at the current position
            line, index = aw.getCursorPosition()
            return self.getWord(aw.text(line), index)
        
        return QString('')
        
    def getSRHistory(self, key):
        """
        Private method to get the search or replace history list.
        
        @param key list to return (must be 'search' or 'replace')
        @return the requested history list (QStringList)
        """
        return self.srHistory[key]
        
    def handleSearchListChanged(self):
        """
        Private slot to handle the searchListChanged signal.
        """
        self.quickFindtextCombo.clear()
        self.quickFindtextCombo.insertStringList(self.srHistory["search"])
        
    def handleQuickSearch(self):
        """
        Private slot to handle the incremental quick search.
        """
        # first we have to check if quick search is active
        # and try to activate it if not
        if not self.quickFindtextCombo.lineEdit().hasFocus():
            self.quickFindtextCombo.lastActive = self.quickFindtextCombo.focusWidget()
            aw = self.activeWindow()
            if aw:
                self.quickFindtextCombo.lastCursorPos = aw.getCursorPosition()
            else:
                self.quickFindtextCombo.lastCursorPos = None
            tff = self.textForFind(0)
            if not tff.isEmpty():
                self.quickFindtextCombo.lineEdit().setText(tff)
            self.quickFindtextCombo.lineEdit().setFocus()
            self.quickFindtextCombo.lineEdit().selectAll()
        else:
            self.performQuickSearch(1, 0)

    def handleQuickSearchEnter(self):
        """
        Private slot to handle the incremental quick search return pressed (jump back to text)
        """
        if self.quickFindtextCombo.lastActive:
            self.quickFindtextCombo.lastActive.setFocus()
    
    def handleQuickSearchEscape(self):
        """
        Private slot to handle the incremental quick search escape pressed (jump back to text)
        """
        if self.quickFindtextCombo.lastActive:
            self.quickFindtextCombo.lastActive.setFocus()
            aw = self.activeWindow()
            if aw and self.quickFindtextCombo.lastCursorPos:
                aw.setCursorPosition(self.quickFindtextCombo.lastCursorPos[0],
                                     self.quickFindtextCombo.lastCursorPos[1])
    
    def handleQuickSearchText(self):
        """
        Private slot to handle the textChanged signal of the quicksearch edit.
        """
        self.performQuickSearch(0, 0)
        
    def handleQuickSearchPrev(self):
        """
        Private slot to handle the quickFindPrev toolbutton action.
        """
        self.performQuickSearch(1, 1)
        
    def performQuickSearch(self, again, back):
        """
        Private slot to perform a quick search.
        
        @param again flag indicating a repeat of the last search (boolean)
        @param back flag indicating a backwards search operation (boolean)
        @author Maciek Fijalkowski, 2005-07-23
        """
        aw = self.activeWindow()
        if not aw:
            return
        
        text = self.quickFindtextCombo.lineEdit().text()
        if text.isEmpty():
            text = self.quickFindtextCombo.lastSearchText
        if text.isEmpty():
            return
        else:
            self.quickFindtextCombo.lastSearchText = text
        
        lineFrom, indexFrom, lineTo, indexTo = aw.getSelection()
        cline, cindex = aw.getCursorPosition ()
        if again:
            if back:
                if indexFrom != 0:
                    index = indexFrom - 1
                    line = lineFrom
                elif lineFrom == 0:
                    return
                else:
                    line = lineFrom - 1
                    index = aw.lineLength(line)
                ok = aw.findFirst(text, 0, 0, 0, 1, 0, line, index)
            else:
                ok = aw.findFirst(text, 0, 0, 0, 1, not back, cline, cindex)
        else:
            ok = aw.findFirst(text, 0, 0, 0, 1, not back, lineFrom, indexFrom)
        if not ok:
            self.quickFindtextCombo.lineEdit().setPaletteBackgroundColor(QColor("red"))
            self.quickFindtextCombo.lineEdit().setPaletteForegroundColor(QColor("white"))
        else:
            self.quickFindtextCombo.lineEdit().setPaletteBackgroundColor(\
                self.quickFindtextCombo.leBackgroundColor)
            self.quickFindtextCombo.lineEdit().setPaletteForegroundColor(\
                self.quickFindtextCombo.leForegroundColor)
        
    def handleQuickSearchExtend(self):
        """
        Private method to handle the quicksearch extend action.
        """
        aw = self.activeWindow()
        if aw is None:
            return
        
        txt = self.quickFindtextCombo.lineEdit().text()
        if txt.isEmpty():
            return
        
        line, index = aw.getCursorPosition()
        text = aw.text(line)
        
        re = QRegExp('[^\w_]')
        end = text.find(re, index)
        if end > index:
            ext = text.mid(index, end-index)
            txt.append(ext)
            self.quickFindtextCombo.lineEdit().setText(txt)
        
    def handleSearch(self):
        """
        Private method to handle the search action.
        """
        self.searchDlg.showFind(self.textForFind())
        
    def handleReplace(self):
        """
        Private method to handle the replace action.
        """
        self.replaceDlg.showReplace(self.textForFind())
        
    def handleGoto(self):
        """
        Private method to handle the goto action.
        """
        aw = self.activeWindow()
        dlg = GotoDialog(self.ui, None, 1)
        dlg.selectAll()
        if dlg.exec_loop() == QDialog.Accepted:
            aw.gotoLine(min(dlg.getLinenumber(), aw.lines()))
        
    def handleGotoBrace(self):
        """
        Private method to handle the goto brace action.
        """
        self.activeWindow().moveToMatchingBrace()
        
    def handleSearchFiles(self):
        """
        Private method to handle the search in files action.
        """
        self.ui.findFilesDialog.show(self.textForFind())
        self.ui.findFilesDialog.raiseW()
        self.ui.findFilesDialog.setActiveWindow()
        
    ##################################################################
    ## Below are the action methods for the view menu
    ##################################################################
    
    def handleZoomIn(self):
        """
        Private method to handle the zoom in action.
        """
        self.activeWindow().zoomIn()
        
    def handleZoomOut(self):
        """
        Private method to handle the zoom out action.
        """
        self.activeWindow().zoomOut()
        
    def handleZoom(self):
        """
        Private method to handle the zoom action.
        """
        aw = self.activeWindow()
        dlg = ZoomDialog(aw.getZoom(), self.ui, None, 1)
        if dlg.exec_loop() == QDialog.Accepted:
            aw.zoomTo(dlg.getZoomSize())
            
    def handleToggleAll(self):
        """
        Private method to handle the toggle all folds action.
        """
        self.activeWindow().foldAll()
        
    def handleToggleCurrent(self):
        """
        Private method to handle the toggle current fold action.
        """
        aw = self.activeWindow()
        line, index = aw.getCursorPosition()
        aw.foldLine(line)
        
    def handleSplitView(self):
        """
        Private method to handle the split view action.
        """
        self.addSplit()
        
    def handleSplitOrientation(self):
        """
        Private method to handle the split orientation action.
        """
        if self.splitOrientationAct.isOn():
            self.setSplitOrientation(QSplitter.Horizontal)
            self.splitViewAct.setIconSet(\
                QIconSet(UI.PixmapCache.getPixmap("splitHorizontal.png")))
            self.splitRemoveAct.setIconSet(\
                QIconSet(UI.PixmapCache.getPixmap("remsplitHorizontal.png")))
        else:
            self.setSplitOrientation(QSplitter.Vertical)
            self.splitViewAct.setIconSet(\
                QIconSet(UI.PixmapCache.getPixmap("splitVertical.png")))
            self.splitRemoveAct.setIconSet(\
                QIconSet(UI.PixmapCache.getPixmap("remsplitVertical.png")))
    
    ##################################################################
    ## Below are the action methods for the macro menu
    ##################################################################
    
    def handleMacroStartRecording(self):
        """
        Private method to handle the start macro recording action.
        """
        self.activeWindow().handleStartMacroRecording()
        
    def handleMacroStopRecording(self):
        """
        Private method to handle the stop macro recording action.
        """
        self.activeWindow().handleStopMacroRecording()
        
    def handleMacroRun(self):
        """
        Private method to handle the run macro action.
        """
        self.activeWindow().handleRunMacro()
        
    def handleMacroDelete(self):
        """
        Private method to handle the delete macro action.
        """
        self.activeWindow().handleDeleteMacro()
        
    def handleMacroLoad(self):
        """
        Private method to handle the load macro action.
        """
        self.activeWindow().handleLoadMacro()
        
    def handleMacroSave(self):
        """
        Private method to handle the save macro action.
        """
        self.activeWindow().handleSaveMacro()
    
    ##################################################################
    ## Below are the action methods for the bookmarks menu
    ##################################################################
    
    def handleToggleBookmark(self):
        """
        Private method to handle the toggle bookmark action.
        """
        self.activeWindow().handleToggleBookmark()
        
    def handleNextBookmark(self):
        """
        Private method to handle the next bookmark action.
        """
        self.activeWindow().handleNextBookmark()
    
    def handlePreviousBookmark(self):
        """
        Private method to handle the previous bookmark action.
        """
        self.activeWindow().handlePreviousBookmark()
    
    def handleClearAllBookmarks(self):
        """
        Private method to handle the clear all bookmarks action.
        """
        for editor in self.editors:
            editor.handleClearBookmarks()
            
        self.bookmarkNextAct.setEnabled(0)
        self.bookmarkPreviousAct.setEnabled(0)
        self.bookmarkClearAct.setEnabled(0)
    
    def handleShowBookmarksMenu(self):
        """
        Private method to handle the show bookmarks menu signal.
        """
        self.bookmarks = {}
        self.bookmarksMenu.clear()
        
        filenames = self.getOpenFilenames()
        filenames.sort()
        for filename in filenames:
            editor = self.getOpenEditor(filename)
            for bookmark in editor.getBookmarks():
                if len(filename) > 50:
                    dots = "..."
                else:
                    dots = ""
                id = self.bookmarksMenu.insertItem(\
                        "%s%s : %d" % (dots, filename[-50:], bookmark))
                self.bookmarks[id] = (filename, bookmark)
    
    def handleBookmarkSelected(self, id):
        """
        Private method to handle the bookmark selected signal.
        
        @param id index of the selected menu entry
                This acts as an index into the list of bookmarks
                that was created, when the bookmarks menu was built.
        """
        self.displayPythonFile(self.bookmarks[id][0], self.bookmarks[id][1])
        
    def handleBookmarkToggled(self, editor):
        """
        Private slot to handle the bookmarkToggled signal.
        
        It checks some bookmark actions and reemits the signal.
        
        @param editor editor that sent the signal
        """
        if editor.hasBookmarks():
            self.bookmarkNextAct.setEnabled(1)
            self.bookmarkPreviousAct.setEnabled(1)
            self.bookmarkClearAct.setEnabled(1)
        else:
            self.bookmarkNextAct.setEnabled(0)
            self.bookmarkPreviousAct.setEnabled(0)
            self.bookmarkClearAct.setEnabled(0)
        self.emit(PYSIGNAL('bookmarkToggled'), (editor,))
    
    def handleGotoSyntaxError(self):
        """
        Private method to handle the goto syntax error action.
        """
        self.activeWindow().handleGotoSyntaxError()
    
    def handleClearAllSyntaxErrors(self):
        """
        Private method to handle the clear all syntax errors action.
        """
        for editor in self.editors:
            editor.handleClearSyntaxError()
    
    def handleSyntaxErrorToggled(self, editor):
        """
        Private slot to handle the syntaxerrorToggled signal.
        
        It checks some syntax error actions and reemits the signal.
        
        @param editor editor that sent the signal
        """
        if editor.hasSyntaxErrors():
            self.syntaxErrorGotoAct.setEnabled(1)
            self.syntaxErrorClearAct.setEnabled(1)
        else:
            self.syntaxErrorGotoAct.setEnabled(0)
            self.syntaxErrorClearAct.setEnabled(0)
        self.emit(PYSIGNAL('syntaxerrorToggled'), (editor,))
        
    def handleNextUncovered(self):
        """
        Private method to handle the next uncovered action.
        """
        self.activeWindow().handleNextUncovered()
        
    def handlePreviousUncovered(self):
        """
        Private method to handle the previous uncovered action.
        """
        self.activeWindow().handlePreviousUncovered()
        
    def handleCoverageMarkersShown(self, shown):
        """
        Private slot to handle the coverageMarkersShown signal.
        
        @param shown flag indicating whether the markers were shown or cleared
        """
        if shown:
            self.notcoveredNextAct.setEnabled(1)
            self.notcoveredPreviousAct.setEnabled(1)
        else:
            self.notcoveredNextAct.setEnabled(0)
            self.notcoveredPreviousAct.setEnabled(0)
    
    ##################################################################
    ## Below are general utility methods
    ##################################################################
    
    def handleResetUI(self):
        """
        Public slot to handle the resetUI signal.
        """
        editor = self.activeWindow()
        if editor is None:
            self.setSbFile()
        else:
            line, pos = editor.getCursorPosition()
            self.setSbFile(editor.getFileName(), line+1, pos)
        
    def closeViewManager(self):
        """
        Public method to shutdown the viewmanager. 
        
        If it cannot close all editor windows, it aborts the shutdown process.
        
        @return flag indicating success (boolean)
        """
        self.handleCloseAll()
            
        # save the list of recently opened projects
        ok = Preferences.Prefs.settings.writeEntry('/eric3/Recent/Sources', self.recent)
        
        # save the list of recently opened projects
        ok = Preferences.Prefs.settings.writeEntry('/eric3/Bookmarked/Sources', self.bookmarked)
        
        if len(self.editors):
            return 0
        else:
            return 1

    def handleLastEditorClosed(self):
        """
        Private slot to handle the lastEditorClosed signal.
        """
        self.closeActGrp.setEnabled(0)
        self.saveActGrp.setEnabled(0)
        self.printAct.setEnabled(0)
        self.printSelAct.setEnabled(0)
        self.editActGrp.setEnabled(0)
        self.searchActGrp.setEnabled(0)
        self.quickFindtextCombo.setEnabled(0)
        self.viewActGrp.setEnabled(0)
        self.viewFoldActGrp.setEnabled(0)
        self.unhighlightAct.setEnabled(0)
        self.splitViewAct.setEnabled(0)
        self.splitOrientationAct.setEnabled(0)
        self.macroActGrp.setEnabled(0)
        self.bookmarkActGrp.setEnabled(0)
        self.setSbFile()
        
        # reinitialize the central store for api information (used by
        # autocompletion and calltips)
        shellLanguage = self.ui.getShell().getClientType()
        for key in self.apis.keys():
            if key != shellLanguage:
                self.apis[key] = None
            
        # remove all split views, if this is supported
        if self.canSplit():
            while self.removeSplit(): pass
            
        # stop the autosave timer
        if self.autosaveTimer.isActive():
            self.autosaveTimer.stop()
        
    def handleEditorOpened(self):
        """
        Private slot to handle the editorOpened signal.
        """
        self.closeActGrp.setEnabled(1)
        self.saveActGrp.setEnabled(1)
        self.printAct.setEnabled(1)
        self.printSelAct.setEnabled(1)
        self.editActGrp.setEnabled(1)
        self.searchActGrp.setEnabled(1)
        self.quickFindtextCombo.setEnabled(1)
        self.viewActGrp.setEnabled(1)
        self.viewFoldActGrp.setEnabled(1)
        self.unhighlightAct.setEnabled(1)
        if self.canSplit():
            self.splitViewAct.setEnabled(1)
            self.splitOrientationAct.setEnabled(1)
        self.macroActGrp.setEnabled(1)
        self.bookmarkActGrp.setEnabled(1)
        
        # activate the autosave timer
        if not self.autosaveTimer.isActive() and \
           self.autosaveInterval > 0:
            self.autosaveTimer.start(self.autosaveInterval * 60000, 1)
        
    def handleAutosave(self):
        """
        Public slot to save the contents of all editors automatically.
        
        Only named editors will be saved by the autosave timer.
        """
        for editor in self.editors:
            if editor.shouldAutosave():
                ok, newName = editor.saveFile()
                if ok:
                    self.setEditorName(editor, newName)
        
        # restart autosave timer
        if self.autosaveInterval > 0:
            self.autosaveTimer.start(self.autosaveInterval * 60000, 1)
        
    def checkActions(self, editor, setSb=1):
        """
        Private slot to check some actions for their enable/disable status and set the statusbar info.
        
        @param editor editor window
        @param setSb flag indicating an update of the status bar is wanted (boolean)
        """
        if editor is not None:
            self.saveAct.setEnabled(editor.isModified())
            self.revertAct.setEnabled(editor.isModified())
            
            self.undoAct.setEnabled(editor.isUndoAvailable())
            self.redoAct.setEnabled(editor.isRedoAvailable())
            
            lex = editor.getLexer()
            if lex is not None:
                self.commentAct.setEnabled(lex.canBlockComment())
                self.uncommentAct.setEnabled(lex.canBlockComment())
                self.streamCommentAct.setEnabled(lex.canStreamComment())
                self.boxCommentAct.setEnabled(lex.canBoxComment())
            else:
                self.commentAct.setEnabled(0)
                self.uncommentAct.setEnabled(0)
                self.streamCommentAct.setEnabled(0)
                self.boxCommentAct.setEnabled(0)
            
            if editor.hasBookmarks():
                self.bookmarkNextAct.setEnabled(1)
                self.bookmarkPreviousAct.setEnabled(1)
                self.bookmarkClearAct.setEnabled(1)
            else:
                self.bookmarkNextAct.setEnabled(0)
                self.bookmarkPreviousAct.setEnabled(0)
                self.bookmarkClearAct.setEnabled(0)
            
            if editor.hasSyntaxErrors():
                self.syntaxErrorGotoAct.setEnabled(1)
                self.syntaxErrorClearAct.setEnabled(1)
            else:
                self.syntaxErrorGotoAct.setEnabled(0)
                self.syntaxErrorClearAct.setEnabled(0)
            
            if editor.hasCoverageMarkers():
                self.notcoveredNextAct.setEnabled(1)
                self.notcoveredPreviousAct.setEnabled(1)
            else:
                self.notcoveredNextAct.setEnabled(0)
                self.notcoveredPreviousAct.setEnabled(0)
            
            if editor.canAutoCompleteFromAPIs():
                self.autoCompleteFromAPIsAct.setEnabled(1)
            else:
                self.autoCompleteFromAPIsAct.setEnabled(0)
                
            if setSb:
                line, pos = editor.getCursorPosition()
                self.setSbFile(editor.getFileName(), line+1, pos)
                
            self.emit(PYSIGNAL('checkActions'), (editor,))
        
    def handleReloadAPIs(self):
        """
        Public slot to reload the api information.
        """
        shellLanguage = self.ui.getShell().getClientType()
        for language, api in self.apis.items():
            if api is not None:
                apifiles = Preferences.getEditorAPI(language)
                if len(apifiles):
                    api.clear()
                    for apifile in apifiles:
                        api.load(apifile)
                else:
                    if language != shellLanguage:
                        self.apis[language] = None
        
    def handlePreferencesChanged(self):
        """
        Public slot to handle the preferencesChanged signal.
        
        This method performs the following actions
            <ul>
            <li>reread the colours for the syntax highlighting</li>
            <li>reloads the already created API objetcs</li>
            <li>starts or stops the autosave timer</li>
            <li><b>Note</b>: changes in viewmanager type are activated
              on an application restart.</li>
            </ul>
        """
        # reload api information
        self.handleReloadAPIs()
        
        # reload editor settings
        for editor in self.editors:
            editor.readSettings()
            
        # reload the autosave timer setting
        self.autosaveInterval = Preferences.getEditor("AutosaveInterval")
        if len(self.editors):
            if self.autosaveTimer.isActive() and \
               self.autosaveInterval == 0:
                self.autosaveTimer.stop()
            elif not self.autosaveTimer.isActive() and \
               self.autosaveInterval > 0:
                self.autosaveTimer.start(self.autosaveInterval * 60000, 1)
        
    def handleEditorSaved(self, fn):
        """
        Public slot to handle the editorSaved signal.
        
        It simply reemits the signal.
        
        @param fn filename of the saved editor
        """
        self.emit(PYSIGNAL('editorSaved'), (fn,))
        
    def handleCursorChanged(self, fn, line, pos):
        """
        Private slot to handle the cursorChanged signal. 
        
        It emits the signal cursorChanged with parameter editor.
        
        @param fn filename (string)
        @param line line number of the cursor (int)
        @param pos position in line of the cursor (int)
        """
        self.setSbFile(fn, line, pos)
        self.emit(PYSIGNAL('cursorChanged'), (self.getOpenEditor(fn),))
        
    def handleBreakpointToggled(self, editor):
        """
        Private slot to handle the breakpointToggled signal.
        
        It simply reemits the signal.
        
        @param editor editor that sent the signal
        """
        self.emit(PYSIGNAL('breakpointToggled'), (editor,))
        
    def handleBreakpointEnabledToggled(self, editor):
        """
        Private slot to handle the breakpointEnabledToggled signal.
        
        It simply reemits the signal.
        
        @param editor editor that sent the signal
        """
        self.emit(PYSIGNAL('breakpointEnabledToggled'), (editor,))
        
    def getAPIs(self, language):
        """
        Public method to get an apis object for autocompletion/calltips
        
        This method creates and loads the QextScintillaAPIs object
        dynamically upon request. This saves memory for languages,
        that might not be needed at the moment.
        
        @param language the language of the requested api object (string or QString)
        @return the apis object (QextScintillaAPIs)
        """
        language = unicode(language)
        try:
            if self.apis[language] is None:
                # create the api object
                apifiles = Preferences.getEditorAPI(language)
                if len(apifiles):
                    self.apis[language] = QextScintillaAPIs()
                    for apifile in apifiles:
                        self.apis[language].load(apifile)
            return self.apis[language]
        except KeyError:
            return None
            
    def getProject(self):
        """
        Public method to get a reference to the Project object.
        
        @return Reference to the Project object (Project.Project)
        """
        return self.ui.getProject()
        
    def getActions(self, type):
        """
        Public method to get a list of all actions.
        
        @param type string denoting the action set to get.
                It must be one of "edit", "file", "search",
                "view" or "window"
        @return list of all actions (list of E3Action)
        """
        try:
            exec 'actionList = self.%sActions[:]' % type
        except:
            actionList = []
                
        return actionList
        
    def editorCommand(self, cmd):
        """
        Private method to send an editor command to the active window.
        
        @param cmd the scintilla command to be sent
        """
        if qApp.focusWidget() == self.ui.getShell():
            self.ui.getShell().editorCommand(cmd)
        else:
            aw = self.activeWindow()
            if aw:
                aw.editorCommand(cmd)
        
    ##################################################################
    ## Below are protected utility methods
    ##################################################################
    
    def _getOpenStartDir(self):
        """
        Protected method to return the starting directory for a file open dialog. 
        
        The appropriate starting directory is calculated
        using the following search order, until a match is found:<br />
            1: Directory of currently active editor<br />
            2: Directory of currently active Project<br />
            3: CWD

        @return String name of directory to start or None
        """
        # if we have an active source, return its path
        if self.activeWindow() is not None and \
           self.activeWindow().getFileName():
            return os.path.dirname(self.activeWindow().getFileName())
            
        # ok, try if there is an active project and return its path
        elif self.getProject().isOpen():
            return self.getProject().ppath
            
        else:
            # None will cause open dialog to start with cwd
            return QString.null

    def _getOpenFileFilter(self):
        """
        Protected method to return the active filename filter for a file open dialog.
        
        The appropriate filename filter is determined by file extension of
        the currently active editor.
        
        @return name of the filename filter (QString) or None
        """
        if self.activeWindow() is not None and \
           self.activeWindow().getFileName():
            ext = os.path.splitext(self.activeWindow().getFileName())[1]
            try:
                return QString(self.ext2Filter[ext])
            except KeyError:
                try:
                    return QString(self.ext2Filter["__ALL__"])
                except KeyError:
                    return None
                
        else:
            return None
