# -*- coding: utf-8 -*-

# Copyright (c) 2004 - 2007 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing the builtin API generator.

"""

import sys

class APIGenerator:
    """
    Class implementing the builtin documentation generator.
    """
    def __init__(self, module):
        """
        Constructor
        
        @param module The information of the parsed Python file.
        """
        self.module = module
        
    def genAPI(self, newStyle, basePackage):
        """
        Method to generate the source code documentation.
        
        @param newStyle flag indicating the api generation for QScintilla 1.7 and 
            newer (boolean)
        @param basePackage name of the base package (string)
        @return The API information. (string)
        """
        self.newStyle = newStyle
        if self.newStyle:
            modulePath = self.module.name.split('.')
            if modulePath[-1] == '__init__':
                del modulePath[-1]
            if basePackage:
                modulePath[0] = basePackage
            self.moduleName = "%s." % '.'.join(modulePath)
        else:
            self.moduleName = ""
        self.api = []
        self.addClassesAPI()
        self.addFunctionsAPI()
        return self.api
        
    def addClassesAPI(self):
        """
        Private method to generate the document section with details about classes.
        
        @return The classes details section. (string)
        """
        classNames = self.module.classes.keys()
        classNames.sort()
        for className in classNames:
            self.addMethodsAPI(className)
        
    def addMethodsAPI(self, className):
        """
        Private method to generate the method details section.
        
        @param classname Name of the class containing the method. (string)
        @return The method details section. (string)
        """
        _class = self.module.classes[className]
        methods = _class.methods.keys()
        methods.sort()
        
        # first do the constructor
        if '__init__' in methods:
            methods.remove('__init__')
            self.api.append('%s%s(%s)' % \
                (self.moduleName, _class.name, 
                 ', '.join(_class.methods['__init__'].parameters[1:])))
            self.api.append('%s%s.__init__(%s)' % \
                (self.moduleName, _class.name, 
                 ', '.join(_class.methods['__init__'].parameters)))
            
        if self.newStyle:
            classNameStr = "%s%s." % (self.moduleName, className)
        else:
            classNameStr = ""
        for method in methods:
            self.api.append('%s%s(%s)' % \
                (classNameStr, method,
                 ', '.join(_class.methods[method].parameters[1:])))
        
    def addFunctionsAPI(self):
        """
        Private method to generate the document section with details about functions.
        
        @return The functions details section. (string)
        """
        funcNames = self.module.functions.keys()
        funcNames.sort()
        for funcName in funcNames:
            self.api.append('%s%s(%s)' % \
                (self.moduleName, self.module.functions[funcName].name,
                 ', '.join(self.module.functions[funcName].parameters)))
