# -*- coding: utf-8 -*-

# Copyright (c) 2002 - 2007 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Package implementing the preferences interface.

The preferences interface consists of a class, which defines the default
values for all configuration items and stores the actual values. These
values are read and written to the eric3 preferences file by module
functions. On Windows the data is located in the registry, everywhere 
else it is stored in a file in a subdirectory of the users home directory.
The individual configuration data is accessed by accessor functions defined
on the module level. The module is simply imported wherever it is needed
with the statement 'import Preferences'. Do not use 'from Preferences import *'
to import it.
"""

import sys
import os
import fnmatch

from qt import *
from qtext import QextScintilla

from ViewManager.Config import ConfigViewManagers

class Prefs:
    """
    A class to hold all configuration items for the application.
    """
    # defaults for the variables window
    varDefaults = {
        "LocalsFilter" : [],
        "GlobalsFilter" : []
    }
        
    # defaults for the debugger
    debuggerDefaults = {
        "RemoteDbgEnabled" : 0,
        "RemoteHost" : "",
        "RemoteExecution" : "",
        "PassiveDbgEnabled" : 0,
        "PassiveDbgPort" : 42424,
        "AutomaticReset" : 0,
        "CustomPythonInterpreter" : 0,
        "PythonInterpreter" : "",
        "RubyInterpreter" : "/usr/bin/ruby",
        "DebugClientType" : 1,      # this coresponds with the radio button id
        "DebugClient" : "",
        "DebugEnvironmentReplace" : 0,
        "DebugEnvironment" : "",
        "PythonRedirect" : 1,
        "PythonNoEncoding" : 0,
        "RubyRedirect" : 1,
        "ConsoleDbgEnabled" : 0,
        "ConsoleDbgCommand" : "",
        "PathTranslation" : 0,
        "PathTranslationRemote" : "",
        "PathTranslationLocal" : "",
    }
    
    uiDefaults = {
        "Language" : "System",
        "ViewManager" : "tabview",
        "LayoutType" : 1,                   # 0 = splitters
                                            # 1 = dockable windows
                                            # 2 = floating windows
        "LayoutShellEmbedded" : 0,          # 0 = separate
                                            # 1 = embedded in debug browser
        "LayoutFileBrowserEmbedded" : 1,    # 0 = separate
                                            # 1 = embedded in debug browser
                                            # 2 = embedded in project browser
        "BrowsersListFoldersFirst" : 1,
        "BrowsersHideNonPublic" : 0,
        "SingleApplicationMode" : 0,
        "CaptionShowsFilename" : 1,
        # the order in ViewProfiles is Project-Browser, File-Browser,
        # Debug-Browser, Python-Shell, Log-Viewer, Task-Viewer,
        # Templates-Viewer
        "ViewProfiles" : {
            "edit"  : [[1, 0, 0, 1, 1, 1, 1], "", 
                [[0,0,300,300], [620,0,300,300], [310,0,300,300],
                 [310,350,400,100], [0,350,300,100], 
                 [0,500,400,125], [405,500,200,200]]],
            "debug" : [[0, 0, 1, 1, 1, 1, 0], "",
                [[0,0,300,300], [620,0,300,300], [310,0,300,300],
                 [310,350,400,100], [0,350,300,100], 
                 [0,500,400,125], [405,500,200,200]]],
        }
    }
    viewProfilesLength = len(uiDefaults["ViewProfiles"]["edit"][2])
    
    iconsDefaults = {
        "Path" : QStringList(),
    }
    
    # defaults for the editor settings
    editorDefaults = {
        "AutosaveInterval" : 0,
        "TabWidth" : 4,
        "IndentWidth" : 4,
        "LinenoWidth" : 4,
        "IndentationGuides" : 1,
        "LinenoMargin" : 1,
        "FoldingMargin" : 1,
        "FoldingStyle" : 1,
        "TabForIndentation" : 0,
        "TabIndents" : 1,
        "ConvertTabsOnLoad" : 0,
        "AutomaticEOLConversion" : 1,
        "ShowWhitespace" : 0,
        "ShowEOL" : 0,
        "UseMonospacedFont" : 0,
        "WrapLongLines" : 1,

        "EdgeMode" : QextScintilla.EDGE_NONE,
        "EdgeColumn" : 80,
        
        # All (most) lexers
        "AllFoldCompact" : 1,
        
        # Python specifics
        "PythonBadIndentation" : 1,
        "PythonFoldComment" : 1,
        "PythonFoldString" : 1,
        "PythonAutoIndent" : 0,
        
        # C++ specifics
        "CppFoldComment" : 1,
        "CppFoldPreprocessor" : 0,
        "CppFoldAtElse" : 0,
        "CppIndentOpeningBrace" : 0,
        "CppIndentClosingBrace" : 0,
        
        # SQL specifics
        "SqlFoldComment" : 1,
        "SqlBackslashEscapes" : 0,
        
        # HTML specifics
        "HtmlFoldPreprocessor" : 0,
        "HtmlCaseSensitiveTags" : 0,
        
        # Perl specifics
        "PerlFoldComment" : 1,
        
        # Bash specifics
        "BashFoldComment" : 1,
        
        # Ruby specifics
        "RubyBadIndentation" : 1,
        
        # CSS specifics
        "CssFoldComment" : 1,
        
        "AutoIndentation" : 1,
        "BraceHighlighting" : 1,
        "CreateBackupFile" : 0,
        "AutoCompletionEnabled" : 0,
        "AutoCompletionCaseSensitivity" : 1,
        "AutoCompletionReplaceWord" : 0,
        "AutoCompletionShowSingle" : 0,
        "AutoCompletionSource" : QextScintilla.AcsDocument,
        "AutoCompletionThreshold" : 2,
        "AutoCompletionFillups" : 0,
        "CallTipsEnabled" : 0,
        "CallTipsVisible" : -1,
        "CaretLineVisible" : 0,
        "CaretWidth" : 1,
        "ColourizeSelText" : 0,
        
        "AutoCheckSyntax" : 1,
        
        "DefaultEncoding" : "utf8"
    }
    if sys.platform == "win32":
        editorDefaults["EOLMode"] = QextScintilla.EolWindows
    else:
        editorDefaults["EOLMode"] = QextScintilla.EolUnix
    
    editorColourDefaults = {
        "CurrentMarker" : Qt.yellow,
        "ErrorMarker" : Qt.red,
        "MatchingBrace" : Qt.blue,
        "MatchingBraceBack" : Qt.white,
        "NonmatchingBrace" : Qt.red,
        "NonmatchingBraceBack" : Qt.white,
        "CallTipsBackground" : Qt.white,
        "CaretForeground" : Qt.black,
        "CaretLineBackground" : Qt.white,
        "Edge" : Qt.lightGray,
    }
    
    editorOtherFontsDefaults = {
        "MarginsFont" : "Helvetica,12,-1,5,50,0,0,0,0,0",
        "MonospacedFont" : "Courier,12,-1,5,50,0,0,0,0,0"
    }
    
    editorAPIDefaults = {
        "Python" : QStringList(),
        "C++" : QStringList(),
        "C#" : QStringList(),
        "IDL" : QStringList(),
        "Java" : QStringList(),
        "JavaScript" : QStringList()
    }
    
    editorLexerAssocDefaults = {
        '*.py' : "Python",
        '*.pyw' : "Python",
        '*.pyx' : "Python",
        '*.ptl' : "Python",
        '*.cpp' : "C++",
        '*.cxx' : "C++",
        '*.cc' : "C++",
        '*.c' : "C++",
        '*.hpp' : "C++",
        '*.hh' : "C++",
        '*.h' : "C++",
        '*.cs' : "C#",
        '*.idl' : "IDL",
        '*.java' : "Java",
        '*.js' : "JavaScript",
    }
    
    editorDefaultPatterns = \
        ['*.h', '*.cpp', '*.cs', '*.idl', '*.java', '*.js', 
         '*.pyx', '*.py', '*.ptl', '*.pyw']
    
    # the following languages are only supported for QScintilla > 1.0
    try:
        from qtext import QextScintillaLexerHTML
        editorAPIDefaults["HTML"] = QStringList()
        editorLexerAssocDefaults.update({
            '*.html' : "HTML",
            '*.htm' : "HTML",
            '*.asp' : "HTML",
            '*.shtml' : "HTML",
            '*.css' : "HTML",
            '*.php' : "HTML",
            '*.php3' : "HTML",
            '*.php4' : "HTML",
            '*.php5' : "HTML",
            '*.phtml' : "HTML",
            '*.xml' : "HTML",
            '*.xsl' : "HTML",
            '*.svg' : "HTML",
            '*.xsd' : "HTML",
            '*.xslt' : "HTML",
            '*.dtd' : "HTML",
            '*.rdf' : "HTML",
            '*.docbook' : "HTML"
        })
        editorDefaultPatterns.extend(['*.html', '*.xml', '*.php', '*.dtd'])
    except ImportError:
        pass
    try:
        from qtext import QextScintillaLexerSQL
        editorAPIDefaults["SQL"] = QStringList()
        editorLexerAssocDefaults["*.sql"] = "SQL"
        editorDefaultPatterns.append('*.sql')
    except ImportError:
        pass
    
    # the following languages are only supported for QScintilla > 1.2
    try:
        from qtext import QextScintillaLexerPerl
        editorAPIDefaults["Perl"] = QStringList()
        editorLexerAssocDefaults.update({
            '*.pl' : "Perl",
            '*.pm' : "Perl",
            '*.ph' : "Perl"
        })
        editorDefaultPatterns.append('*.pl')
    except ImportError:
        pass
    
    # the following languages are only supported for QScintilla > 1.3
    try:
        from qtext import QextScintillaLexerBash
        editorAPIDefaults["Bash"] = QStringList()
        editorLexerAssocDefaults["*.sh"] = "Bash"
        editorDefaultPatterns.append('*.sh')
    except ImportError:
        pass
    
    # the following languages are only supported for QScintilla > 1.4
    try:
        from qtext import QextScintillaLexerRuby
        editorAPIDefaults["Ruby"] = QStringList()
        editorLexerAssocDefaults.update({
            '*.rb' : "Ruby",
            '*.rbw' : "Ruby"
        })
        editorDefaultPatterns.extend(['*.rb', '*.rbw'])
    except ImportError:
        pass
    try:
        from qtext import QextScintillaLexerLua
        editorAPIDefaults["Lua"] = QStringList()
        editorLexerAssocDefaults["*.lua"] = "Lua"
        editorDefaultPatterns.append('*.lua')
    except ImportError:
        pass

    # the following languages are only supported for QScintilla > 1.5
    try:
        from qtext import QextScintillaLexerCSS
        editorAPIDefaults["CSS"] = QStringList()
        editorLexerAssocDefaults["*.css"] = "CSS"
        editorDefaultPatterns.append('*.css')
    except ImportError:
        pass
    try:
        from qtext import QextScintillaLexerTeX
        editorAPIDefaults["TeX"] = QStringList()
        editorLexerAssocDefaults.update({
            "*.tex" : "TeX",
            "*.sty" : "TeX",
            "*.aux" : "TeX",
            "*.toc" : "TeX",
            "*.idx" : "TeX"
        })
        editorDefaultPatterns.extend(['*.tex', '*.sty', '*.aux', '*.toc', '*.idx'])
    except ImportError:
        pass
    try:
        from qtext import QextScintillaLexerDiff
        editorAPIDefaults["Diff"] = QStringList()
        editorLexerAssocDefaults.update({
            "*.diff" : "Diff",
            "*.patch" : "Diff"
        })
        editorDefaultPatterns.extend(['*.diff', '*.patch'])
    except ImportError:
        pass
    try:
        from qtext import QextScintillaLexerMakefile
        editorAPIDefaults["Makefile"] = QStringList()
        editorLexerAssocDefaults.update({
            "*makefile" : "Makefile",
            "Makefile*" : "Makefile",
            "*.mak" : "Makefile"
        })
        editorDefaultPatterns.extend(['*makefile', 'Makefile*', '*.mak'])
    except ImportError:
        pass
    try:
        from qtext import QextScintillaLexerProperties
        editorAPIDefaults["Properties"] = QStringList()
        editorLexerAssocDefaults.update({
            "*.properties" : "Properties",
            "*.ini" : "Properties",
            "*.inf" : "Properties",
            "*.reg" : "Properties",
            "*.cfg" : "Properties",
            "*.cnf" : "Properties",
            "*.rc"  : "Properties",
        })
        editorDefaultPatterns.extend(['*.ini', '*.inf', '*.reg', '*.cfg', '*.cnf', '*.rc'])
    except ImportError:
        pass
    try:
        from qtext import QextScintillaLexerBatch
        editorAPIDefaults["Batch"] = QStringList()
        editorLexerAssocDefaults.update({
            "*.bat" : "Batch",
            "*.cmd" : "Batch"
        })
        editorDefaultPatterns.extend(['*.bat', '*.cmd'])
    except ImportError:
        pass


    # defaults for the printer settings
    printerDefaults = {
        "PrinterName" : "",
        "ColorMode" : 1,
        "FirstPageFirst" : 1,
        "Magnification" : 0,
        "Orientation" : 0,
        "PageSize": 0,
        "HeaderFont" : "Times,10,-1,5,50,0,0,0,0,0"
    }
    
    # defaults for the project settings
    projectDefaults = {
        "SearchNewFiles" : 0,
        "AutoIncludeNewFiles" : 0,
        "AutoLoadSession" : 0,
        "AutoSaveSession" : 0,
        "SessionAllBreakpoints" : 0,
        "CompressedProjectFiles" : 0,
        "XMLTimestamp" : 1,
        "AutoCompileForms" : 0,
        "AutoLoadDbgProperties" : 0,
        "AutoSaveDbgProperties" : 0,
    }
    
    # defaults for the help settings
    helpDefaults = {
        "HelpViewerType" : 1,      # this coresponds with the radio button id
        "Webbrowser" : "",
        "Pdfviewer" : "",
        "CustomViewer" : "",
        "PythonDocDir" : "",
        "QtDocDir" : "",
        "Qt4DocDir" : "",
    }
    
    # defaults for system settings
    sysDefaults = {
        "StringEncoding" : "utf8"
    }
    
    # defaults for the shell settings
    shellDefaults = {
        "LinenoWidth" : 4,
        "LinenoMargin" : 1,
        "AutoCompletionEnabled" : 1,
        "CallTipsEnabled" : 1,
        "WrapEnabled" : 1,
        "MaxHistoryEntries" : 100,
    }

    # defaults for Qt related stuff
    qtDefaults = {
        "QtDir" : "",
        "ExportQtDir" : 0,
        "Qt4Dir" : "",
        "ExportQt4Dir" : 0,
    }
    
    # defaults for corba related stuff
    corbaDefaults = {
        "omniidl" : "omniidl"
    }
    
    # defaults for the refactoring related stuff
    refactoringDefaults = {
        "Logging" : 1
    }
    
    # defaults for user related stuff
    userDefaults = {
        "Email" : "",
        "MailServer" : "",
        "Signature" : "",
        "MailServerAuthentication" : 0,
        "MailServerUser" : "",
        "MailServerPassword" : "",
    }
    
    # defaults for vcs related stuff
    vcsDefaults = {
        "AutoClose" : 0,
        "AutoSaveFiles" : 1,
        "AutoSaveProject" : 1,
    }
    
    # defaults for tasks related stuff
    tasksDefaults = {
        "TasksMarkers"       : "TO" + "DO:", 
        "TasksMarkersBugfix" : "FIX" + "ME:",
        # needed to keep it from being recognized as a task
        "TasksColour"          : Qt.black,
        "TasksBugfixColour"    : Qt.red,
        "TasksBgColour"        : Qt.white,
        "TasksProjectBgColour" : Qt.lightGray,
    }
    
    # defaults for templates related stuff
    templatesDefaults = {
        "AutoOpenGroups" : 1,
        "SingleDialog"   : 0,
        "ShowTooltip"    : 0,
        "SeparatorChar"  : "$",
    }
    
    # defaults for geometry
    geometryDefaults = {
        "MainGeometry"   : [0,0,0,0],
        "MainSplitter0"  : [0,0,0,0,0],
        "MainSplitter1"  : [0,0,0,0,0,0,0],
        "ProjectBrowser" : [0,0,300,300,1],
        "SBV"            : [310,0,300,300,1],
        "LogViewer"      : [0,350,300,100,1],
        "Shell"          : [310,350,400,100,1],
        "FileBrowser"    : [620,0,300,300,1],
        "TaskViewer"     : [0,500,400,125,1],
        "TemplateViewer" : [405,500,200,200,1],
    }

    # defaults for dock windows
    dockDefaults = {
        "MainDockLayout0" : "",
        "MainDockLayout1" : "",
        "MainDockLayout2" : "",
        "MainDockLayout3" : "",
    }

    # if true, revert layouts to factory defaults
    resetLayout = 0

def readPreferences(prefClass = Prefs):
    """
    Module function to read the preferences and transfer them into the central store.
    
    @param prefClass preferences class used as the storage area
    """
    # read the entries for the variables display
    prefClass.varPrefs = {}
    for key in prefClass.varDefaults.keys():
        v, ok = prefClass.settings.readEntry("/eric3/Variables/" + key)
        if ok:
            prefClass.varPrefs[key] = eval(unicode(v))
        else:
            prefClass.varPrefs[key] = prefClass.varDefaults[key]

    # read the entries for the debugger
    prefClass.debuggerPrefs = {}
    for key in "RemoteDbgEnabled", "PassiveDbgEnabled", \
                "PassiveDbgPort", "CustomPythonInterpreter", \
                "DebugClientType", "AutomaticReset", "DebugEnvironmentReplace", \
                "PythonRedirect", "PythonNoEncoding", "RubyRedirect", \
                "ConsoleDbgEnabled", "PathTranslation":
        prefClass.debuggerPrefs[key], ok = \
            prefClass.settings.readNumEntry("/eric3/Debugger/" + key,
                prefClass.debuggerDefaults[key])
                
    for key in "RemoteHost", "RemoteExecution", "PythonInterpreter", \
                "RubyInterpreter", "DebugClient", "DebugEnvironment", \
                "ConsoleDbgCommand", "PathTranslationRemote", \
                "PathTranslationLocal":
        prefClass.debuggerPrefs[key], ok = \
            prefClass.settings.readEntry("/eric3/Debugger/" + key,
                prefClass.debuggerDefaults[key])
    
    # read the entries for the general ui settings
    prefClass.uiPrefs = {}
    lang, ok = \
        prefClass.settings.readEntry("/eric3/UI/Language",
            prefClass.uiDefaults["Language"])
    if unicode(lang) == "None" or unicode(lang) == "" or lang is None:
        prefClass.uiPrefs["Language"] = None
    else:
        prefClass.uiPrefs["Language"] = unicode(lang)
        
    vm, ok = \
        prefClass.settings.readEntry("/eric3/UI/ViewManager",
            prefClass.uiDefaults["ViewManager"])
    if unicode(vm) in ConfigViewManagers:
        prefClass.uiPrefs["ViewManager"] = unicode(vm)
    else:
        prefClass.uiPrefs["ViewManager"] = prefClass.uiDefaults["ViewManager"]

    for key in ["LayoutType", "LayoutShellEmbedded", "LayoutFileBrowserEmbedded",
                "BrowsersListFoldersFirst", "BrowsersHideNonPublic", "SingleApplicationMode",
                "CaptionShowsFilename"]:
        prefClass.uiPrefs[key], ok = \
            prefClass.settings.readNumEntry("/eric3/UI/" + key,
                prefClass.uiDefaults[key])

    v, ok = prefClass.settings.readEntry("/eric3/UI/ViewProfiles")
    if ok:
        prefClass.uiPrefs["ViewProfiles"] = eval(unicode(v))
        for name in ["edit", "debug"]:
            vpLength = len(prefClass.uiPrefs["ViewProfiles"][name][0])
            if vpLength < prefClass.viewProfilesLength:
                prefClass.uiPrefs["ViewProfiles"][name][0].extend(\
                    prefClass.uiDefaults["ViewProfiles"][name][0][vpLength:])
            vpLength = len(prefClass.uiPrefs["ViewProfiles"][name][2])
            if vpLength < prefClass.viewProfilesLength:
                prefClass.uiPrefs["ViewProfiles"][name][2].extend(\
                    prefClass.uiDefaults["ViewProfiles"][name][2][vpLength:])
    else:
        prefClass.uiPrefs["ViewProfiles"] = prefClass.uiDefaults["ViewProfiles"]
    
    
    prefClass.iconsPrefs = {}
    for key in prefClass.iconsDefaults.keys():
        dirlist, ok = prefClass.settings.readListEntry("/eric3/UI/Icons/" + key)
        if ok:
            prefClass.iconsPrefs[key] = dirlist
        else:
            prefClass.iconsPrefs[key] = prefClass.iconsDefaults[key]
    
    # read the entries for the editor settings
    prefClass.editorPrefs = {}
    for key in prefClass.editorDefaults.keys():
        if key in ["DefaultEncoding"]:
            prefClass.editorPrefs[key], ok = \
                prefClass.settings.readEntry("/eric3/Editor/" + key,
                    prefClass.editorDefaults[key])
        else:
            prefClass.editorPrefs[key], ok = \
                prefClass.settings.readNumEntry("/eric3/Editor/" + key,
                    prefClass.editorDefaults[key])
    
    prefClass.editorColourPrefs = {}
    for key in prefClass.editorColourDefaults.keys():
        col, ok = prefClass.settings.readEntry("/eric3/Editor/Colour/" + key)
        if ok:
            prefClass.editorColourPrefs[key] = QColor(col)
        else:
            prefClass.editorColourPrefs[key] = prefClass.editorColourDefaults[key]
            
    prefClass.editorOtherFontsPrefs = {}
    for key in prefClass.editorOtherFontsDefaults.keys():
        prefClass.editorOtherFontsPrefs[key], ok = \
            prefClass.settings.readEntry("/eric3/Editor/Other Fonts/" + key,
                prefClass.editorOtherFontsDefaults[key])
                
    prefClass.editorAPI = {}
    for key in prefClass.editorAPIDefaults.keys():
        ap, ok = prefClass.settings.readListEntry("/eric3/Editor/APIs/" + key)
        if ok:
            prefClass.editorAPI[key] = ap
        else:
            prefClass.editorAPI[key] = prefClass.editorAPIDefaults[key]
        
    prefClass.editorLexerAssoc = {}
    keyList = prefClass.settings.entryList("/eric3/Editor/LexerAssociations")
    if len(keyList) == 0:
        keyList2 = prefClass.settings.entryList("/eric3/Editor/Lexers")
        if len(keyList2) > 0:
            # code to migrate extension based assocs to pattern based ones
            for key in keyList2:
                key = unicode(key)
                language, ok = \
                    prefClass.settings.readEntry("/eric3/Editor/Lexers/" + key)
                if ok:
                    prefClass.editorLexerAssoc["*.%s" % key] = language
                prefClass.settings.removeEntry("/eric3/Editor/Lexers/" + key)
            prefClass.settings.removeEntry("/eric3/Editor/Lexers")
        else:
            # build from scratch
            for key in prefClass.editorLexerAssocDefaults.keys():
                prefClass.editorLexerAssoc[key] = \
                    QString(prefClass.editorLexerAssocDefaults[key])
    else:
        for key in keyList:
            key = unicode(key)
            if prefClass.editorLexerAssocDefaults.has_key(key):
                defaultValue = prefClass.editorLexerAssocDefaults[key]
            else:
                defaultValue = QString.null
            prefClass.editorLexerAssoc[key], ok = \
                prefClass.settings.readEntry("/eric3/Editor/LexerAssociations/" + key,
                    defaultValue)
    for key in prefClass.editorDefaultPatterns:
        if not prefClass.editorLexerAssoc.has_key(key):
            prefClass.editorLexerAssoc[key] = \
                QString(prefClass.editorLexerAssocDefaults[key])
    
    # read the entries for the project settings
    prefClass.projectPrefs = {}
    for key in prefClass.projectDefaults.keys():
        prefClass.projectPrefs[key], ok = \
            prefClass.settings.readNumEntry("/eric3/Project/" + key,
                prefClass.projectDefaults[key])
    
    # read the entries for the help settings
    prefClass.helpPrefs = {}
    for key in ["HelpViewerType"]:
        prefClass.helpPrefs[key], ok = \
            prefClass.settings.readNumEntry("/eric3/Help/" + key,
                prefClass.helpDefaults[key])
    for key in ["Webbrowser", "Pdfviewer", "CustomViewer", \
                "PythonDocDir", "QtDocDir", "Qt4DocDir"]:
        prefClass.helpPrefs[key], ok = \
            prefClass.settings.readEntry("/eric3/Help/" + key,
                prefClass.helpDefaults[key])
    
    # read the entries for system settings
    prefClass.sysPrefs = {}
    prefClass.sysPrefs["StringEncoding"], ok = \
        prefClass.settings.readEntry("/eric3/System/StringEncoding",
            prefClass.sysDefaults["StringEncoding"])
            
    # read the entries for Qt settings
    prefClass.qtPrefs = {}
    for key in ["QtDir", "Qt4Dir"]:
        prefClass.qtPrefs[key], ok = \
            prefClass.settings.readEntry("/eric3/Qt/" + key,
                prefClass.qtDefaults[key])
    for key in ["ExportQtDir", "ExportQt4Dir"]:
        prefClass.qtPrefs[key], ok = \
            prefClass.settings.readNumEntry("/eric3/Qt/" + key,
                prefClass.qtDefaults[key])
            
    # read the entries for the display geometry
    prefClass.geometryPrefs = {}
    for key in prefClass.geometryDefaults.keys():
        v, ok = prefClass.settings.readEntry("/eric3/Geometry/" + key)
        if ok:
            prefClass.geometryPrefs[key] = eval(unicode(v))
        else:
            prefClass.geometryPrefs[key] = prefClass.geometryDefaults[key]

    # read the entries for the dock window layout
    prefClass.dockPrefs = {}
    for key in prefClass.dockDefaults.keys():
        v, ok = prefClass.settings.readEntry("/eric3/Docks/" + key)
        if ok:
            prefClass.dockPrefs[key] = unicode(v)
        else:
            prefClass.dockPrefs[key] = prefClass.dockDefaults[key]
        
    # read the entries for the printer settings
    prefClass.printerPrefs = {}
    for key in ["ColorMode", "FirstPageFirst", "Magnification", 
                "Orientation", "PageSize"]:
        prefClass.printerPrefs[key], ok = \
            prefClass.settings.readNumEntry("/eric3/Printer/" + key,
                prefClass.printerDefaults[key])
    for key in ["PrinterName", "HeaderFont"]:
        prefClass.printerPrefs[key], ok = \
            prefClass.settings.readEntry("/eric3/Printer/" + key,
                prefClass.printerDefaults[key])
                
    # read the entries for the shell settings
    prefClass.shellPrefs = {}
    for key in prefClass.shellDefaults.keys():
        prefClass.shellPrefs[key], ok = \
            prefClass.settings.readNumEntry("/eric3/Shell/" + key,
                prefClass.shellDefaults[key])
                
    # read the entries for the corba settings
    prefClass.corbaPrefs = {}
    for key in ["omniidl"]:
        prefClass.corbaPrefs[key], ok = \
            prefClass.settings.readEntry("/eric3/Corba/" + key,
                prefClass.corbaDefaults[key])
                
    # read the entries for the refactoring settings
    prefClass.refactoringPrefs = {}
    for key in prefClass.refactoringDefaults.keys():
        prefClass.refactoringPrefs[key], ok = \
            prefClass.settings.readNumEntry("/eric3/Refactoring/" + key,
                prefClass.refactoringDefaults[key])
                
    # read the entries for the user settings
    prefClass.userPrefs = {}
    for key in prefClass.userDefaults.keys():
        if key in ["MailServerAuthentication"]:
            prefClass.userPrefs[key], ok = \
                prefClass.settings.readNumEntry("/eric3/User/" + key,
                    prefClass.userDefaults[key])
        else:
            prefClass.userPrefs[key], ok = \
                prefClass.settings.readEntry("/eric3/User/" + key,
                    prefClass.userDefaults[key])
                    
    # read the entries for the VCS settings
    prefClass.vcsPrefs = {}
    for key in prefClass.vcsDefaults.keys():
        prefClass.vcsPrefs[key], ok = \
            prefClass.settings.readNumEntry("/eric3/VCS/" + key,
                prefClass.vcsDefaults[key])
    
    # read the entries for the tasks settings
    prefClass.tasksPrefs = {}
    for key in prefClass.tasksDefaults.keys():
        if key in ["TasksColour", "TasksBugfixColour", 
                   "TasksBgColour", "TasksProjectBgColour"]:
            col, ok = prefClass.settings.readEntry("/eric3/Tasks/" + key)
            if ok:
                prefClass.tasksPrefs[key] = QColor(col)
            else:
                prefClass.tasksPrefs[key] = prefClass.tasksDefaults[key]
        else:
            prefClass.tasksPrefs[key], ok = \
                prefClass.settings.readEntry("/eric3/Tasks/" + key,
                    prefClass.tasksDefaults[key])

    # read the entries for templates settings
    prefClass.templatesPrefs = {}
    for key in prefClass.templatesDefaults.keys():
        if key in ["SeparatorChar"]:
            prefClass.templatesPrefs[key], ok = \
                prefClass.settings.readEntry("/eric3/Templates/" + key,
                    prefClass.templatesDefaults[key])
        else:
            prefClass.templatesPrefs[key], ok = \
                prefClass.settings.readNumEntry("/eric3/Templates/" + key,
                    prefClass.templatesDefaults[key])    
    
def savePreferences(prefClass = Prefs):
    """
    Module function to write the preferences from the central store.
    
    @param prefClass preferences class used as the storage area
    """
    # write the entries for the variables display
    for key in prefClass.varPrefs.keys():
        prefClass.settings.writeEntry("/eric3/Variables/" + key,
            unicode(prefClass.varPrefs[key]))

    # write the entries for the debugger
    for key in prefClass.debuggerPrefs.keys():
        prefClass.settings.writeEntry("/eric3/Debugger/" + key,
            prefClass.debuggerPrefs[key])

    # write the entries for the general ui settings
    if prefClass.uiPrefs["Language"] is None:
        prefClass.settings.writeEntry("/eric3/UI/Language", "None")
    else:
        prefClass.settings.writeEntry("/eric3/UI/Language", prefClass.uiPrefs["Language"])
    prefClass.settings.writeEntry("/eric3/UI/ViewManager", prefClass.uiPrefs["ViewManager"])
    prefClass.settings.writeEntry("/eric3/UI/LayoutType",
        prefClass.uiPrefs["LayoutType"])
    prefClass.settings.writeEntry("/eric3/UI/LayoutShellEmbedded",
        prefClass.uiPrefs["LayoutShellEmbedded"])
    prefClass.settings.writeEntry("/eric3/UI/LayoutFileBrowserEmbedded",
        prefClass.uiPrefs["LayoutFileBrowserEmbedded"])
    prefClass.settings.writeEntry("/eric3/UI/BrowsersListFoldersFirst", 
        prefClass.uiPrefs["BrowsersListFoldersFirst"])
    prefClass.settings.writeEntry("/eric3/UI/BrowsersHideNonPublic",
        prefClass.uiPrefs["BrowsersHideNonPublic"])
    prefClass.settings.writeEntry("/eric3/UI/SingleApplicationMode", 
        prefClass.uiPrefs["SingleApplicationMode"])
    prefClass.settings.writeEntry("/eric3/UI/CaptionShowsFilename", 
        prefClass.uiPrefs["CaptionShowsFilename"])
    prefClass.settings.writeEntry("/eric3/UI/ViewProfiles",
        unicode(prefClass.uiPrefs["ViewProfiles"]))

    # write the entries for the icon settings
    for key in prefClass.iconsPrefs.keys():
        prefClass.settings.writeEntry("/eric3/UI/Icons/" + key,
            prefClass.iconsPrefs[key])
            
    # write the entries for the editor settings
    for key in prefClass.editorPrefs.keys():
        prefClass.settings.writeEntry("/eric3/Editor/" + key,
            prefClass.editorPrefs[key])
            
    for key in prefClass.editorColourDefaults.keys():
        prefClass.settings.writeEntry("/eric3/Editor/Colour/" + key,
            prefClass.editorColourPrefs[key].name())

    for key in prefClass.editorOtherFontsDefaults.keys():
        prefClass.settings.writeEntry("/eric3/Editor/Other Fonts/" + key,
            prefClass.editorOtherFontsPrefs[key])

    for key in prefClass.editorAPI.keys():
        prefClass.settings.writeEntry("/eric3/Editor/APIs/" + key,
            prefClass.editorAPI[key])
    
    # first remove lexer associations that no longer exist, than save the rest
    keyList = prefClass.settings.entryList("/eric3/Editor/LexerAssociations")
    for key in keyList:
        key = unicode(key)
        if not prefClass.editorLexerAssoc.has_key(key):
            prefClass.settings.removeEntry("/eric3/Editor/LexerAssociations/" + key)
    for key in prefClass.editorLexerAssoc.keys():
        prefClass.settings.writeEntry("/eric3/Editor/LexerAssociations/" + key,
            prefClass.editorLexerAssoc[key])
    
    # write the entries for the project settings
    for key in prefClass.projectPrefs.keys():
        prefClass.settings.writeEntry("/eric3/Project/" + key,
            prefClass.projectPrefs[key])
            
    # write the entries for the help settings
    for key in prefClass.helpPrefs.keys():
        prefClass.settings.writeEntry("/eric3/Help/" + key,
            prefClass.helpPrefs[key])
            
    # write the entries for the system settings
    prefClass.settings.writeEntry("/eric3/System/StringEncoding",
        prefClass.sysPrefs["StringEncoding"])
        
    # write the entries for the Qt settings
    for key in prefClass.qtPrefs.keys():
        prefClass.settings.writeEntry("/eric3/Qt/" + key,
            prefClass.qtPrefs[key])
        
    # write the entries for the display geometry
    for key in prefClass.geometryPrefs.keys():
        prefClass.settings.writeEntry("/eric3/Geometry/" + key,
            unicode(prefClass.geometryPrefs[key]))
        
    # write the entries for the dock layout
    for key in prefClass.dockPrefs.keys():
        prefClass.settings.writeEntry("/eric3/Docks/" + key,
            unicode(prefClass.dockPrefs[key]))
        
    # write the entries for the printer settings
    for key in prefClass.printerPrefs.keys():
        prefClass.settings.writeEntry("/eric3/Printer/" + key,
                prefClass.printerPrefs[key])
        
    # write the entries for the shell settings
    for key in prefClass.shellPrefs.keys():
        prefClass.settings.writeEntry("/eric3/Shell/" + key,
                prefClass.shellPrefs[key])
                
    # write the entries for the corba settings
    for key in prefClass.corbaPrefs.keys():
        prefClass.settings.writeEntry("/eric3/Corba/" + key,
                prefClass.corbaPrefs[key])
        
    # write the entries for the refactoring settings
    for key in prefClass.refactoringPrefs.keys():
        prefClass.settings.writeEntry("/eric3/Refactoring/" + key,
                prefClass.refactoringPrefs[key])
                
    # write the entries for the user settings
    for key in prefClass.userPrefs.keys():
        prefClass.settings.writeEntry("/eric3/User/" + key,
                prefClass.userPrefs[key])
                
    # write the entries for the VCS settings
    for key in prefClass.vcsPrefs.keys():
        prefClass.settings.writeEntry("/eric3/VCS/" + key,
                prefClass.vcsPrefs[key])
                
    # write the entries for the tasks settings
    for key in prefClass.tasksPrefs.keys():
        if key in ["TasksColour", "TasksBugfixColour",
                   "TasksBgColour", "TasksProjectBgColour"]:
            prefClass.settings.writeEntry("/eric3/Tasks/" + key,
                    prefClass.tasksPrefs[key].name())
        else:
            prefClass.settings.writeEntry("/eric3/Tasks/" + key,
                    prefClass.tasksPrefs[key])
    
    # write the entries for the templates settings
    for key in prefClass.templatesPrefs.keys():
        prefClass.settings.writeEntry("/eric3/Templates/" + key,
                prefClass.templatesPrefs[key])
    
def readToolbarSettings(mw, tbs, prefClass = Prefs):
    """
    Module function to read the toolbar settings from the central store.
    
    In addition to reading and storing the values, the toolbars are
    set accordingly.
    
    @param mw reference to the main window (QMainWindow)
    @param tbs list of tuples defining the different toolbars
    @param prefClass preferences class used as the storage area
    """
    tbList = []
    for ind, (dummy, tb) in tbs.items():
        hidden, ok = prefClass.settings.readNumEntry("/eric3/Toolbars/%d/hidden" % ind, 0)
        dock, ok = prefClass.settings.readNumEntry("/eric3/Toolbars/%d/dockarea" % ind, Qt.DockTop)
        index, ok = prefClass.settings.readNumEntry("/eric3/Toolbars/%d/index" % ind, ind)
        nl, ok = prefClass.settings.readNumEntry("/eric3/Toolbars/%d/nl" % ind, 0)
        eo, ok = prefClass.settings.readNumEntry("/eric3/Toolbars/%d/extraOffset" % ind, 0)
        tbList.append((index, dock, nl, eo, hidden, tb))

    tbList.sort()
    
    for index, dock, nl, eo, hidden, tb in tbList:
        try:
            dock = Qt.Dock(dock)
        except AttributeError:
            pass
        mw.moveDockWindow(tb, dock, nl, index, eo)
        if hidden:
            tb.hide()
        else:
            tb.show()
        
def saveToolbarSettings(mw, tbs, prefClass = Prefs):
    """
    Module function to write the toolbar settings to the central store.
    
    @param prefClass preferences class used as the storage area
    """
    for ind, (text, tb) in tbs.items():
        ok, dock, index, nl, eo = mw.getLocation(tb)
        prefClass.settings.writeEntry("/eric3/Toolbars/%d/hidden" % ind, tb.isHidden())
        prefClass.settings.writeEntry("/eric3/Toolbars/%d/text" % ind, text)
        if ok:
            prefClass.settings.writeEntry("/eric3/Toolbars/%d/dockarea" % ind, dock)
            prefClass.settings.writeEntry("/eric3/Toolbars/%d/index" % ind, index)
            prefClass.settings.writeEntry("/eric3/Toolbars/%d/nl" % ind, nl)
            prefClass.settings.writeEntry("/eric3/Toolbars/%d/extraOffset" % ind, eo)
        
def readToolsMenu(prefClass = Prefs):
    """
    Module function to read the tools menu configuration.
    
    @param prefClass preferences class used as the storage area
    @return list of tuples defing the tools menu entries
    """
    toollist = []
    items, ok = prefClass.settings.readNumEntry("/eric3/Toolsmenu/Items", 0)
    for ind in range(items):
        menutext, ok1 = prefClass.settings.readEntry("/eric3/Toolsmenu/%d/Menutext" % ind)
        executable, ok2 = prefClass.settings.readEntry("/eric3/Toolsmenu/%d/Executable" % ind)
        arguments, ok3 = prefClass.settings.readEntry("/eric3/Toolsmenu/%d/Arguments" % ind)
        redirect, ok4 = prefClass.settings.readNumEntry("/eric3/Toolsmenu/%d/Redirect" % ind, 0)
        
        if ok1 and ok2 and ok3:
            tool = (unicode(menutext), unicode(executable), unicode(arguments), redirect)
            toollist.append(tool)
            
    return toollist
    
def saveToolsMenu(toollist, prefClass = Prefs):
    """
    Module function to write the tools menu configuration.
    
    @param prefClass preferences class used as the storage area
    """
    # first step, remove all tool menu entries
    items, ok = prefClass.settings.readNumEntry("/eric3/Toolsmenu/Items", 0)
    for ind in range(items):
        prefClass.settings.removeEntry("/eric3/Toolsmenu/%d/Menutext" % ind)
        prefClass.settings.removeEntry("/eric3/Toolsmenu/%d/Executable" % ind)
        prefClass.settings.removeEntry("/eric3/Toolsmenu/%d/Arguments" % ind)
        prefClass.settings.removeEntry("/eric3/Toolsmenu/%d/Redirect" % ind)
        
    # second step, write the menu entries
    prefClass.settings.writeEntry("/eric3/Toolsmenu/Items", len(toollist))
    ind = 0
    for tool in toollist:
        prefClass.settings.writeEntry("/eric3/Toolsmenu/%d/Menutext" % ind, tool[0])
        prefClass.settings.writeEntry("/eric3/Toolsmenu/%d/Executable" % ind, tool[1])
        prefClass.settings.writeEntry("/eric3/Toolsmenu/%d/Arguments" % ind, tool[2])
        prefClass.settings.writeEntry("/eric3/Toolsmenu/%d/Redirect" % ind, tool[3])
        ind += 1
    
def initPreferences():
    """
    Module function to initialize the central configuration store. 
    
    This function is called once upon import of the module.
    """
    Prefs.settings = QSettings()
    if sys.platform == "win32":
        Prefs.settings.setPath("Eric", "", QSettings.User)
    else:
        hp = QDir.homeDirPath()
        dn = QDir(hp)
        dn.mkdir(".eric3")
        eric3rcPath = os.path.join(unicode(hp), ".eric3", "eric3rc")
        if os.path.exists(eric3rcPath):
            os.rename(eric3rcPath, os.path.join(unicode(hp), ".qt", "eric3rc"))
    readPreferences(Prefs)
    
def syncPreferences():
    """
    Module function to sync the preferences to disk.
    
    In addition to synching, the central configuration store is reinitialized as well.
    """
    savePreferences(Prefs)
    initPreferences()
    
def getVarFilters(prefClass = Prefs):
    """
    Module function to retrieve the variables filter settings.
    
    @param prefClass preferences class used as the storage area
    @return a tuple defing the variables filter
    """
    return (prefClass.varPrefs["LocalsFilter"], prefClass.varPrefs["GlobalsFilter"])
    
def setVarFilters(filters, prefClass = Prefs):
    """
    Module function to store the variables filter settings.
    
    @param prefClass preferences class used as the storage area
    """
    prefClass.varPrefs["LocalsFilter"], prefClass.varPrefs["GlobalsFilter"] = filters
    
def getDebugger(key, prefClass = Prefs):
    """
    Module function to retrieve the debugger settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested debugger setting
    """
    return prefClass.debuggerPrefs[key]
    
def setDebugger(key, value, prefClass = Prefs):
    """
    Module function to store the debugger settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.debuggerPrefs[key] = value

def getUILanguage(prefClass = Prefs):
    """
    Module function to retrieve the language for the user interface.
    
    @param prefClass preferences class used as the storage area
    @return the language for the UI
    """
    return prefClass.uiPrefs["Language"]
    
def setUILanguage(lang, prefClass = Prefs):
    """
    Module function to store the language for the user interface.
    
    @param lang the language
    @param prefClass preferences class used as the storage area
    """
    prefClass.uiPrefs["Language"] = lang

def getUILayout(prefClass = Prefs):
    """
    Module function to retrieve the layout for the user interface.
    
    @param prefClass preferences class used as the storage area
    @return the UI layout as a tuple of main layout, flag for
        an embedded shell and a value for an embedded file browser
    """
    layout = (prefClass.uiPrefs["LayoutType"],
              prefClass.uiPrefs["LayoutShellEmbedded"],
              prefClass.uiPrefs["LayoutFileBrowserEmbedded"])
    return layout
    
def setUILayout(layout, prefClass = Prefs):
    """
    Module function to store the layout for the user interface.
    
    @param layout the layout type
    @param prefClass preferences class used as the storage area
    """
    prefClass.uiPrefs["LayoutType"], \
    prefClass.uiPrefs["LayoutShellEmbedded"], \
    prefClass.uiPrefs["LayoutFileBrowserEmbedded"] = layout

def getViewManager(prefClass = Prefs):
    """
    Module function to retrieve the selected viewmanager type.
    
    @param prefClass preferences class used as the storage area
    @return the viewmanager type
    """
    return prefClass.uiPrefs["ViewManager"]
    
def setViewManager(vm, prefClass = Prefs):
    """
    Module function to store the selected viewmanager type.
    
    @param vm the viewmanager type
    @param prefClass preferences class used as the storage area
    """
    prefClass.uiPrefs["ViewManager"] = vm

def getUI(key, prefClass = Prefs):
    """
    Module function to retrieve the various UI settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested UI setting
    """
    return prefClass.uiPrefs[key]
    
def setUI(key, value, prefClass = Prefs):
    """
    Module function to store the various UI settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.uiPrefs[key] = value
    
def getIcons(key, prefClass = Prefs):
    """
    Module function to retrieve the various Icons settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested Icons setting
    """
    return prefClass.iconsPrefs[key]
    
def setIcons(key, value, prefClass = Prefs):
    """
    Module function to store the various Icons settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.iconsPrefs[key] = value
    
def getEditor(key, prefClass = Prefs):
    """
    Module function to retrieve the various editor settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested editor setting
    """
    return prefClass.editorPrefs[key]
    
def setEditor(key, value, prefClass = Prefs):
    """
    Module function to store the various editor settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.editorPrefs[key] = value
    
def getEditorColour(key, prefClass = Prefs):
    """
    Module function to retrieve the various editor marker colours.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested editor colour
    """
    return prefClass.editorColourPrefs[key]
    
def setEditorColour(key, value, prefClass = Prefs):
    """
    Module function to store the various editor marker colours.
    
    @param key the key of the colour to be set
    @param value the colour to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.editorColourPrefs[key] = value
    
def getEditorOtherFonts(key, prefClass = Prefs):
    """
    Module function to retrieve the various editor fonts except the lexer fonts.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested editor font (QFont)
    """
    f = QFont()
    f.fromString(prefClass.editorOtherFontsPrefs[key])
    return f
    
def setEditorOtherFonts(key, font, prefClass = Prefs):
    """
    Module function to store the various editor fonts except the lexer fonts.
    
    @param key the key of the font to be set
    @param font the font to be set (QFont)
    @param prefClass preferences class used as the storage area
    """
    prefClass.editorOtherFontsPrefs[key] = font.toString()
    
def getEditorAPI(key, prefClass = Prefs):
    """
    Module function to retrieve the various lists of api files.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested list of api files (QStringList)
    """
    return prefClass.editorAPI[key]
    
def setEditorAPI(key, apilist, prefClass = Prefs):
    """
    Module function to store the various lists of api files.
    
    @param key the key of the api to be set
    @param apilist the list of api files (QStringList)
    @param prefClass preferences class used as the storage area
    """
    prefClass.editorAPI[key] = apilist
    
def getEditorLexerAssocs(prefClass = Prefs):
    """
    Module function to retrieve all lexer associations.
    
    @param prefClass preferences class used as the storage area
    @return a reference to the list of lexer associations
        (dictionary of strings)
    """
    return prefClass.editorLexerAssoc
    
def setEditorLexerAssocs(assocs, prefClass = Prefs):
    """
    Module function to retrieve all lexer associations.
    
    @param assocs dictionary of lexer associations to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.editorLexerAssoc = assocs.copy()
    
def getEditorLexerAssoc(filename, prefClass = Prefs):
    """
    Module function to retrieve a lexer association.
    
    @param filename filename used to determine the associated lexer language (string)
    @param prefClass preferences class used as the storage area
    @return the requested lexer language (string)
    """
    for pattern, language in prefClass.editorLexerAssoc.items():
        if fnmatch.fnmatch(filename, pattern):
            return unicode(language)
        
    return ""
    
def getPrinter(key, prefClass = Prefs):
    """
    Module function to retrieve the various printer settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested printer setting
    """
    if key in ["HeaderFont"]:
        f = QFont()
        f.fromString(prefClass.printerPrefs[key])
        return f
    else:
        return prefClass.printerPrefs[key]

def setPrinter(key, value, prefClass = Prefs):
    """
    Module function to store the various printer settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    if key in ["HeaderFont"]:
        prefClass.printerPrefs[key] = value.toString()
    else:
        prefClass.printerPrefs[key] = value

def getShell(key, prefClass = Prefs):
    """
    Module function to retrieve the various shell settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested shell setting
    """
    return prefClass.shellPrefs[key]

def setShell(key, value, prefClass = Prefs):
    """
    Module function to store the various shell settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.shellPrefs[key] = value

def getProject(key, prefClass = Prefs):
    """
    Module function to retrieve the various project handling settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested project setting
    """
    return prefClass.projectPrefs[key]
    
def setProject(key, value, prefClass = Prefs):
    """
    Module function to store the various project handling settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.projectPrefs[key] = value
    
def getHelp(key, prefClass = Prefs):
    """
    Module function to retrieve the various help settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested help setting
    """
    return prefClass.helpPrefs[key]
    
def setHelp(key, value, prefClass = Prefs):
    """
    Module function to store the various help settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.helpPrefs[key] = value
    
def getSystem(key, prefClass = Prefs):
    """
    Module function to retrieve the various system settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested system setting
    """
    return prefClass.sysPrefs[key]
    
def setSystem(key, value, prefClass = Prefs):
    """
    Module function to store the various system settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.sysPrefs[key] = value
    
def getQtDir(prefClass = Prefs):
    """
    Module function to retrieve the QtDir setting.
    
    @param prefClass preferences class used as the storage area
    @return the requested QtDir setting (string)
    """
    if prefClass.qtPrefs["QtDir"].isEmpty():
        return os.getenv("QTDIR", "")
    else:
        return unicode(prefClass.qtPrefs["QtDir"])
    
def getQt4Dir(prefClass = Prefs):
    """
    Module function to retrieve the QtDir setting.
    
    @param prefClass preferences class used as the storage area
    @return the requested QtDir setting (string)
    """
    if prefClass.qtPrefs["Qt4Dir"].isEmpty():
        return os.getenv("QT4DIR", "")
    else:
        return unicode(prefClass.qtPrefs["Qt4Dir"])
    
def getQt(key, prefClass = Prefs):
    """
    Module function to retrieve the various Qt settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested Qt setting
    """
    if key == "QtDir":
        return getQtDir(prefClass)
    else:
        return prefClass.qtPrefs[key]
    
def setQt(key, value, prefClass = Prefs):
    """
    Module function to store the various Qt settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.qtPrefs[key] = value
    
def getCorba(key, prefClass = Prefs):
    """
    Module function to retrieve the various corba settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested corba setting
    """
    return prefClass.corbaPrefs[key]
    
def setCorba(key, value, prefClass = Prefs):
    """
    Module function to store the various corba settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.corbaPrefs[key] = value
    
def getRefactoring(key, prefClass = Prefs):
    """
    Module function to retrieve the various refactoring settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested refactoring setting
    """
    return prefClass.refactoringPrefs[key]
    
def setRefactoring(key, value, prefClass = Prefs):
    """
    Module function to store the various refactoring settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.refactoringPrefs[key] = value
    
def getUser(key, prefClass = Prefs):
    """
    Module function to retrieve the various user settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested user setting
    """
    return prefClass.userPrefs[key]
    
def setUser(key, value, prefClass = Prefs):
    """
    Module function to store the various user settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.userPrefs[key] = value
    
def getVCS(key, prefClass = Prefs):
    """
    Module function to retrieve the VCS related settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested user setting
    """
    return prefClass.vcsPrefs[key]
    
def setVCS(key, value, prefClass = Prefs):
    """
    Module function to store the VCS related settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.vcsPrefs[key] = value
    
def getTasks(key, prefClass = Prefs):
    """
    Module function to retrieve the Tasks related settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested user setting
    """
    return prefClass.tasksPrefs[key]
    
def setTasks(key, value, prefClass = Prefs):
    """
    Module function to store the Tasks related settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.tasksPrefs[key] = value
    
def getTemplates(key, prefClass = Prefs):
    """
    Module function to retrieve the Templates related settings.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested user setting
    """
    return prefClass.templatesPrefs[key]
    
def setTemplates(key, value, prefClass = Prefs):
    """
    Module function to store the Templates related settings.
    
    @param key the key of the setting to be set
    @param value the value to be set
    @param prefClass preferences class used as the storage area
    """
    prefClass.templatesPrefs[key] = value
    
def getGeometry(key, prefClass = Prefs):
    """
    Module function to retrieve the display geometry.
    
    @param key the key of the value to get
    @param prefClass preferences class used as the storage area
    @return the requested geometry setting
    """
    return prefClass.geometryPrefs[key]

def setGeometry(key, value, prefClass = Prefs):
    """
    Module function to store the display geometry.
    
    @param key the key of the setting to be set
    @param value the geometry to be set
    @param prefClass preferences class used as the storage area
    """
    if prefClass.resetLayout:
        prefClass.geometryPrefs[key] = prefClass.geometryDefaults[key]
    else:
        prefClass.geometryPrefs[key] = value

def getMainDockLayout(mw, which, prefClass = Prefs):
    """
    Module function to retrieve the layout of the dock windows.
    
    @param mw reference to the main window (QMainWindow)
    @param which basic type of the dock layout (0 to 3)
    @param prefClass preferences class used as the storage area
    @return flag indicating success
    """
    s = QString(prefClass.dockPrefs["MainDockLayout%d" % which])
    if s.isEmpty():
        return 0
    ts = QTextStream(s, IO_ReadOnly)
    ts >> mw
    return 1

def setMainDockLayout(mw, which, prefClass = Prefs):
    """
    Module function to store the layout of the dock windows.
    
    @param mw reference to the main window (QMainWindow)
    @param which basic type of the dock layout (0 or 1)
    @param prefClass preferences class used as the storage area
    """
    key = "MainDockLayout%d" % which
    if prefClass.resetLayout:
        s = QString(prefClass.dockDefaults[key])
    else:
        s = QString()
        ts = QTextStream(s, IO_WriteOnly)
        ts << mw
    prefClass.dockPrefs[key] = unicode(s)

def resetLayout(which, prefClass = Prefs):
    """
    Module function to set a flag not storing the current layout.
    
    @param which flag indicating a reset of the layout. (ignored)
    @param prefClass preferences class used as the storage area
    """
    prefClass.resetLayout = 1

from Shortcuts import readShortcuts, saveShortcuts, exportShortcuts, importShortcuts

initPreferences()
