# -*- coding: utf-8 -*-

# Copyright (c) 2003 - 2007 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing a dialog to show the output of the svn log command process.
"""

import os
import sys

from qt import *

from LogForm import LogForm
from DiffDialog import SvnDiffDialog

import Utilities

class SvnLogDialog(LogForm):
    """
    Module implementing a dialog to show the output of the svn log command process.
    
    The dialog is nonmodal. Clicking a link in the upper text pane shows 
    a diff of the versions.
    """
    def __init__(self, vcs, parent = None):
        """
        Constructor
        
        @param vcs reference to the vcs object
        @param parent parent widget (QWidget)
        """
        LogForm.__init__(self, parent)
        
        self.saveButton.hide()
        
        QWhatsThis.add(self.contents,self.tr("<b>Subversion Log</b>\n"
            "<p>This shows the output of the svn log command."
            " By clicking on the links you may show the difference"
            " between versions.</p>"
        ))
        QWhatsThis.add(self.errors,self.tr("<b>Subversion log errors</b>\n"
            "<p>This shows possible error messages of the svn log"
            " command.</p>"
        ))
        
        self.setWFlags(self.getWFlags() | Qt.WDestructiveClose)
        self.process = QProcess()
        self.vcs = vcs
        
        self.connect(self.process, SIGNAL('readyReadStdout()'),
            self.handleReadStdout)
        self.connect(self.process, SIGNAL('readyReadStderr()'),
            self.handleReadStderr)
        self.connect(self.process, SIGNAL('processExited()'),
            self.handleProcessExited)
        self.disconnect(self.contents, SIGNAL('linkClicked(const QString&)'),
            self.contents, SLOT('setSource(const QString&)'))
        self.connect(self.contents, SIGNAL('linkClicked(const QString&)'),
            self.handleLinkClicked)
        self.connect(self.contents, SIGNAL('anchorClicked(const QString&, const QString&)'),
            self.handleAnchorClicked)
            
        self.rx_sep = QRegExp('\\-+')
        self.rx_sep2 = QRegExp('=+')
        self.rx_rev = QRegExp('rev ([0-9]+):  ([^|]*) \| ([^|]*) \| ([0-9]+) lines?')
        # "rev" followed by one or more decimals followed by a colon followed
        # anything up to " | " (twice) followed by one or more decimals followed
        # by "line" and an optional "s"
        self.rx_rev2 = QRegExp('r([0-9]+) \| ([^|]*) \| ([^|]*) \| ([0-9]+) lines?')
        # "r" followed by one or more decimals followed by " | " followed
        # anything up to " | " (twice) followed by one or more decimals followed
        # by "line" and an optional "s"
        self.rx_flags = QRegExp('   ([ADM])( .*)')
        # three blanks followed by A or D or M
        self.rx_changed = QRegExp('Changed .*')
        
        self.flags = {
            'A' : self.trUtf8('Added'),
            'D' : self.trUtf8('Deleted'),
            'M' : self.trUtf8('Modified')
        }
        
        self.revisions = QStringList()
        self.revString = self.trUtf8('revision')
        
        self.buf = QStringList()    # buffer for stdout
            
    def closeEvent(self, e):
        """
        Private slot implementing a close event handler.
        
        @param e close event (QCloseEvent)
        """
        if self.process is not None:
            self.process.tryTerminate()
            QTimer.singleShot(2000, self.process, SLOT('kill()'))
            
        e.accept()
        
    def start(self, fn, noEntries=0):
        """
        Public slot to start the cvs log command.
        
        @param fn filename to show the log for (string)
        @param noEntries number of entries to show (integer)
        """
        self.intercept = 0
        self.filename = fn
        self.dname, self.fname = self.vcs.splitPath(fn)
        
        self.process.kill()
        
        self.process.clearArguments()
        self.process.addArgument('svn')
        self.process.addArgument('log')
        self.vcs.addArguments(self.process, self.vcs.options['global'])
        self.vcs.addArguments(self.process, self.vcs.options['log'])
        self.process.addArgument('--stop-on-copy')
        if noEntries:
            self.process.addArgument('--limit')
            self.process.addArgument(str(noEntries))
        try:
            self.setActiveWindow()
            self.raiseW()
        except:
            pass
        self.process.addArgument(self.fname)
        self.process.setWorkingDirectory(QDir(self.dname))
        
        self.process.start()
        self.setCaption(self.trUtf8('Subversion Log %1').arg(self.filename))
        
    def handleProcessExited(self):
        """
        Private slot to handle the processExited signal.
        
        After the process has exited, diff links are inserted into the contents pane.
        """
        self.closeButton.setFocus()
        self.inputGroup.setEnabled(0)
        
        self.contents.clear()
        lvers = 1
        for s in self.buf:
            rev_match = 0
            if self.rx_rev.exactMatch(s):
                ver = self.rx_rev.cap(1)
                author = self.rx_rev.cap(2)
                date = self.rx_rev.cap(3)
                # number of lines is ignored
                rev_match = 1
            elif self.rx_rev2.exactMatch(s):
                ver = self.rx_rev2.cap(1)
                author = self.rx_rev2.cap(2)
                date = self.rx_rev2.cap(3)
                # number of lines is ignored
                rev_match = 1
                
            if rev_match:
                dstr = QString('<b>%1 %2</b>').arg(self.revString).arg(ver)
                try:
                    lv = self.revisions[lvers]
                    lvers += 1
                except IndexError:
                    lv = None
                
                if lv is not None:
                    url = QUrl()
                    url.setProtocol("file")
                    url.setFileName(self.filename)
                    query = QString()
                    query.append(lv).append('_').append(ver)
                    url.setQuery(query)
                    dstr.append(' [<a href="')\
                        .append(url.toString())\
                        .append('" name="')\
                        .append(query)\
                        .append('">')\
                        .append(self.trUtf8('diff to %1').arg(lv))\
                        .append('</a>]')
                self.contents.append(dstr)
                
                dstr = self.trUtf8('<i>author: %1</i>').arg(author)
                self.contents.append(dstr)
                
                dstr = self.trUtf8('<i>date: %1</i>').arg(date)
                self.contents.append(dstr)
                
            elif self.rx_sep.exactMatch(s) or self.rx_sep2.exactMatch(s):
                self.contents.append('<hr>')
                
            elif self.rx_flags.exactMatch(s):
                dstr = QString(self.flags[str(self.rx_flags.cap(1))])
                dstr.append(self.rx_flags.cap(2))
                self.contents.append(dstr)
                
            elif self.rx_changed.exactMatch(s):
                dstr = QString('<br>%1').arg(s)
                self.contents.append(dstr)
                
            else:
                if s.isEmpty():
                    s = self.contents.append('<br>')
                else:
                    self.contents.append(Utilities.html_encode(unicode(s)))
        
        self.contents.setCursorPosition(0, 0)
        self.contents.ensureCursorVisible()
        
    def handleReadStdout(self):
        """
        Private slot to handle the readyReadStdout signal.
        
        It reads the output of the process and inserts it into a buffer.
        """
        while self.process.canReadLineStdout():
            s = self.process.readLineStdout()
            self.buf.append(s)
            if self.rx_rev.exactMatch(s):
                ver = self.rx_rev.cap(1)
                # save revision number for later use
                self.revisions.append(ver)
            elif self.rx_rev2.exactMatch(s):
                ver = self.rx_rev2.cap(1)
                # save revision number for later use
                self.revisions.append(ver)
        
    def handleReadStderr(self):
        """
        Private slot to handle the readyReadStderr signal. 
        
        It reads the error output of the process and inserts it into the
        error pane.
        """
        if self.process is not None:
            s = str(self.process.readStderr())
            self.errors.moveCursor(QTextEdit.MoveEnd, 0)
            self.errors.insert(s)
        
    def handleLinkClicked(self, link):
        """
        Private slot to handle the linkClicked signal of the contents pane.
        
        @param link the link that was clicked (QString)
        """
        self.contents.setSource('')
        link = QUrl(link)
        filename = unicode(link.path())
        if sys.platform == "win32":
            if filename.startswith("/"):
                filename = filename[1:]
        ver = link.query()
        v1 = ver.section('_', 0, 0)
        v2 = ver.section('_', 1, 1)
        if v1.isEmpty() or v2.isEmpty():
            return
            
        self.diff = SvnDiffDialog(self.vcs)
        self.diff.show()
        self.diff.start(filename, [v1, v2])
        
    def handleAnchorClicked(self, name, link):
        """
        Private slot to handle the anchorClicked signal of the contents pane.
        
        @param name name of the anchor that was clicked (QString)
        @param link the link that was clicked (QString)
        """
        self.contents.scrollToAnchor(name)
        
    def passwordMode(self, isOn):
        """
        Private slot to handle the password checkbox toggled.
        
        @param isOn flag indicating the status of the check box (boolean)
        """
        if isOn:
            self.input.setEchoMode(QLineEdit.Password)
        else:
            self.input.setEchoMode(QLineEdit.Normal)
        
    def sendInput(self):
        """
        Private slot to send the input to the subversion process.
        """
        input = self.input.text()
        input.append(os.linesep)
        
        self.errors.moveCursor(QTextEdit.MoveEnd, 0)
        if self.passwordCheckBox.isChecked():
            self.errors.insert(os.linesep)
        else:
            self.errors.insert(input)
        
        self.proc.writeToStdin(input)
        
        self.passwordCheckBox.setChecked(0)
        self.input.clear()
        
    def returnPressed(self):
        """
        Private slot to handle the press of the return key in the input field.
        """
        self.intercept = 1
        self.sendInput()
        
    def keyPressEvent(self, evt):
        """
        Protected slot to handle a key press event.
        
        @param evt the key press event (QKeyEvent)
        """
        if self.intercept:
            self.intercept = 0
            evt.accept()
            return
        LogForm.keyPressEvent(self, evt)
