# -*- coding: utf-8 -*-

# Copyright (c) 2003 - 2007 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing a dialog to show the output of the svn status command process.
"""

import types
import os

from qt import *

from KdeQt import KQMessageBox

from StatusForm import StatusForm

import Preferences

class SvnStatusDialog(StatusForm):
    """
    Module implementing a dialog to show the output of the svn status command process.
    """
    def __init__(self, vcs, parent = None):
        """
        Constructor
        
        @param vcs -- reference to the vcs object
        @param parent -- parent widget (QWidget)
        """
        StatusForm.__init__(self, parent)
        
        self.setWFlags(self.getWFlags() | Qt.WDestructiveClose)
        self.process = QProcess()
        self.vcs = vcs
        
        self.statusList.setColumnAlignment(0, Qt.AlignHCenter)
        self.statusList.setColumnAlignment(1, Qt.AlignHCenter)
        self.statusList.setColumnAlignment(2, Qt.AlignHCenter)
        self.statusList.setColumnAlignment(3, Qt.AlignHCenter)
        self.statusList.setColumnAlignment(4, Qt.AlignHCenter)
        self.statusList.setColumnAlignment(5, Qt.AlignHCenter)
        self.statusList.setColumnAlignment(6, Qt.AlignHCenter)
        self.statusList.setColumnAlignment(7, Qt.AlignRight)
        self.statusList.setColumnAlignment(8, Qt.AlignRight)
        self.statusList.setColumnAlignment(9, Qt.AlignLeft)
        self.statusList.setColumnAlignment(10, Qt.AlignLeft)
        
        self.statusList.setSorting(10)
        
        self.menuitems = []
        self.menu = QPopupMenu()
        self.menuitems.append(self.menu.insertItem(self.trUtf8("Commit changes to repository..."),
            self.handleCommit))
        if self.vcs.versionStr >= '1.2.0':
            self.menu.insertSeparator()
            self.menuitems.append(self.menu.insertItem(self.trUtf8("Lock"),
                self.handleLock))
            self.menuitems.append(self.menu.insertItem(self.trUtf8("Unlock"),
                self.handleUnlock))
            self.menuitems.append(self.menu.insertItem(self.trUtf8("Break lock"),
                self.handleBreakLock))
            self.menuitems.append(self.menu.insertItem(self.trUtf8("Steal lock"),
                self.handleStealLock))
        for itm in self.menuitems:
            self.menu.setItemEnabled(itm, 0)
        
        self.connect(self.statusList,SIGNAL('contextMenuRequested(QListViewItem *, const QPoint &, int)'),
                     self.handleContextMenu)
        self.connect(self.process, SIGNAL('readyReadStdout()'),
            self.handleReadStdout)
        self.connect(self.process, SIGNAL('readyReadStderr()'),
            self.handleReadStderr)
        self.connect(self.process, SIGNAL('processExited()'),
            self.handleProcessExited)
            
        self.modifiedIndicators = QStringList()
        self.modifiedIndicators.append(self.trUtf8('added'))
        self.modifiedIndicators.append(self.trUtf8('deleted'))
        self.modifiedIndicators.append(self.trUtf8('modified'))
        
        self.lockedIndicators = QStringList()
        self.lockedIndicators.append(self.trUtf8('locked'))
        
        self.stealBreakLockIndicators = QStringList()
        self.stealBreakLockIndicators.append(self.trUtf8('other lock'))
        self.stealBreakLockIndicators.append(self.trUtf8('stolen lock'))
        self.stealBreakLockIndicators.append(self.trUtf8('broken lock'))
        
        self.unlockedIndicators = QStringList()
        self.unlockedIndicators.append(self.trUtf8('not locked'))
        
        self.status = {
            ' ' : self.trUtf8('normal'),
            'A' : self.trUtf8('added'),
            'D' : self.trUtf8('deleted'),
            'M' : self.trUtf8('modified'),
            'R' : self.trUtf8('replaced'),
            'C' : self.trUtf8('conflict'),
            'X' : self.trUtf8('external'),
            'I' : self.trUtf8('ignored'),
            '?' : self.trUtf8('unversioned'),
            '!' : self.trUtf8('missing'),
            '~' : self.trUtf8('type error')
        }
        self.propStatus = {
            ' ' : self.trUtf8('normal'),
            'M' : self.trUtf8('modified'),
            'C' : self.trUtf8('conflict')
        }
        self.locked = {
            ' ' : self.trUtf8('no'),
            'L' : self.trUtf8('yes')
        }
        self.history = {
            ' ' : self.trUtf8('no'),
            '+' : self.trUtf8('yes')
        }
        self.switched = {
            ' ' : self.trUtf8('no'),
            'S' : self.trUtf8('yes')
        }
        self.lockinfo = {
            ' ' : self.trUtf8('not locked'),
            'K' : self.trUtf8('locked'),
            'O' : self.trUtf8('other lock'),
            'T' : self.trUtf8('stolen lock'),
            'B' : self.trUtf8('broken lock'),
        }
        self.uptodate = {
            ' ' : self.trUtf8('yes'),
            '*' : self.trUtf8('no')
        }
        
        self.rx_status = QRegExp('(.{8})\\s+([0-9-]+)\\s+([0-9?]+)\\s+([\\w?]+)\\s+(.+)')
        self.rx_status2 = QRegExp('(.{8})\\s+(.+)\\s*')
        
    def handleContextMenu(self,itm,coord,col):
        """
        Private slot to show the context menu of the listview.
        
        @param itm the selected listview item (QListViewItem)
        @param coord the position of the mouse pointer (QPoint)
        @param col the column of the mouse pointer (int)
        """
        self.menu.popup(coord)
        
    def closeEvent(self, e):
        """
        Private slot implementing a close event handler.
        
        @param e -- close event (QCloseEvent)
        """
        if self.process is not None:
            self.process.tryTerminate()
            QTimer.singleShot(2000, self.process, SLOT('kill()'))
            
        e.accept()
        
    def start(self, fn):
        """
        Public slot to start the svn status command.
        
        @param fn filename(s) to show the status of (string or list of strings)
        """
        self.intercept = 0
        self.args = fn
        
        if self.process:
            self.process.kill()
        else:
            self.process = QProcess()
            self.connect(self.process, SIGNAL('readyReadStdout()'),
                self.handleReadStdout)
            self.connect(self.process, SIGNAL('readyReadStderr()'),
                self.handleReadStderr)
            self.connect(self.process, SIGNAL('processExited()'),
                self.handleProcessExited)
        
        self.process.clearArguments()
        self.process.addArgument('svn')
        self.process.addArgument('status')
        self.vcs.addArguments(self.process, self.vcs.options['global'])
        self.vcs.addArguments(self.process, self.vcs.options['status'])
        self.process.addArgument('--verbose')
        if '--show-updates' in self.vcs.options['status'] or \
           '-u' in self.vcs.options['status']:
            try:
                self.setActiveWindow()
                self.raiseW()
            except:
                pass
        if type(fn) is types.ListType:
            self.dname, fnames = self.vcs.splitPathList(fn)
            self.vcs.addArguments(self.process, fnames)
        else:
            self.dname, fname = self.vcs.splitPath(fn)
            self.process.addArgument(fname)
        self.process.setWorkingDirectory(QDir(self.dname))
        
        self.process.start()
        self.setCaption(self.trUtf8('Subversion Status'))
        
    def finish(self):
        """
        Private slot called when the process finished or the user pressed the button.
        """
        if self.process is not None:
            self.process.tryTerminate()
            QTimer.singleShot(2000, self.process, SLOT('kill()'))
            
        self.cancelButton.setText(self.trUtf8('&OK'))
        self.cancelButton.setDefault(1)
        self.cancelButton.setFocus()
        
        self.inputGroup.setEnabled(0)
        self.refreshButton.setEnabled(1)
        
        for itm in self.menuitems:
            self.menu.setItemEnabled(itm, 1)
        
        self.process = None
        
    def buttonPressed(self):
        """
        Private slot connected to the button clicked signal.
        """
        if self.process is None:
            self.close()
        else:
            self.finish()
            
    def handleProcessExited(self):
        """
        Private slot to handle the processExited signal.
        
        Just call finish().
        """
        self.finish()
        
    def handleReadStdout(self):
        """
        Private slot to handle the readyReadStdout signal.
        
        It reads the output of the process, formats it and inserts it into
        the contents pane.<br />
        <b>Note</b>: If the order gets changed, the handle... and
        get...Items methods need to be changed accordingly.
        """
        while self.process.canReadLineStdout():
            s = self.process.readLineStdout()
            if self.rx_status.exactMatch(s):
                flags = str(self.rx_status.cap(1))
                rev = self.rx_status.cap(2)
                change = self.rx_status.cap(3)
                author = self.rx_status.cap(4)
                path = self.rx_status.cap(5).stripWhiteSpace()
                
                itm = QListViewItem(self.statusList)
                itm.setText(0, self.status[flags[0]])
                itm.setText(1, self.propStatus[flags[1]])
                itm.setText(2, self.locked[flags[2]])
                itm.setText(3, self.history[flags[3]])
                itm.setText(4, self.switched[flags[4]])
                itm.setText(5, self.lockinfo[flags[5]])
                itm.setText(6, self.uptodate[flags[7]])
                itm.setText(7, "%7s" % str(rev))
                itm.setText(8, "%7s" % str(change))
                itm.setText(9, author)
                itm.setText(10, path)
            elif self.rx_status2.exactMatch(s):
                flags = str(self.rx_status2.cap(1))
                path = self.rx_status2.cap(2).stripWhiteSpace()
                
                itm = QListViewItem(self.statusList)
                itm.setText(0, self.status[flags[0]])
                itm.setText(1, self.propStatus[flags[1]])
                itm.setText(2, self.locked[flags[2]])
                itm.setText(3, self.history[flags[3]])
                itm.setText(4, self.switched[flags[4]])
                itm.setText(5, self.lockinfo[flags[5]])
                itm.setText(6, self.uptodate[flags[7]])
                itm.setText(7, "%7s" % "")
                itm.setText(8, "%7s" % "")
                itm.setText(9, "")
                itm.setText(10, path)
        
    def handleReadStderr(self):
        """
        Private slot to handle the readyReadStderr signal.
        
        It reads the error output of the process and inserts it into the
        error pane.
        """
        if self.process is not None:
            s = str(self.process.readStderr())
            self.errors.moveCursor(QTextEdit.MoveEnd, 0)
            self.errors.insert(s)
        
    def passwordMode(self, isOn):
        """
        Private slot to handle the password checkbox toggled.
        
        @param isOn flag indicating the status of the check box (boolean)
        """
        if isOn:
            self.input.setEchoMode(QLineEdit.Password)
        else:
            self.input.setEchoMode(QLineEdit.Normal)
        
    def sendInput(self):
        """
        Private slot to send the input to the subversion process.
        """
        input = self.input.text()
        input.append(os.linesep)
        
        self.errors.moveCursor(QTextEdit.MoveEnd, 0)
        if self.passwordCheckBox.isChecked():
            self.errors.insert(os.linesep)
        else:
            self.errors.insert(input)
        
        self.proc.writeToStdin(input)
        
        self.passwordCheckBox.setChecked(0)
        self.input.clear()
        
    def returnPressed(self):
        """
        Private slot to handle the press of the return key in the input field.
        """
        self.intercept = 1
        self.sendInput()
        
    def keyPressEvent(self, evt):
        """
        Protected slot to handle a key press event.
        
        @param evt the key press event (QKeyEvent)
        """
        if self.intercept:
            self.intercept = 0
            evt.accept()
            return
        StatusForm.keyPressEvent(self, evt)
        
    def handleCommit(self):
        """
        Private slot to handle the Commit context menu entry.
        """
        names = [os.path.join(self.dname, unicode(itm.text(10))) \
                 for itm in self.getModifiedItems()]
        if not names:
            KQMessageBox.information(self,
                self.trUtf8("Commit"),
                self.trUtf8("""There are no uncommitted changes available/selected."""),
                self.trUtf8("&OK"),
                QString.null,
                QString.null,
                0, -1)
            return
        
        if Preferences.getVCS("AutoSaveFiles"):
            vm = qApp.mainWidget().getViewManager()
            for name in names:
                vm.saveEditor(name)
        self.vcs.vcsCommit(names, '', parent=self)
        self.refreshStatus()
        
    def getModifiedItems(self):
        """
        Private method to retrieve all emtries, that have a modified status.
        
        @return list of all items with a modified status
        """
        modifiedItems = []
        itm = self.statusList.firstChild()
        while itm:
            if itm.isSelected() and \
                    (self.modifiedIndicators.contains(itm.text(0)) or \
                     self.modifiedIndicators.contains(itm.text(1))):
                modifiedItems.append(itm)
            itm = itm.itemBelow()
        return modifiedItems

    def handleLock(self):
        """
        Private slot to handle the Break Lock context menu entry.
        """
        names = [os.path.join(self.dname, unicode(itm.text(10))) \
                 for itm in self.getLockActionItems(self.unlockedIndicators)]
        if not names:
            KQMessageBox.information(self,
                self.trUtf8("Lock"),
                self.trUtf8("""There are no unlocked files available/selected."""),
                self.trUtf8("&OK"),
                QString.null,
                QString.null,
                0, -1)
            return
        
        self.vcs.svnLock(names, parent=self)
        self.refreshStatus()

    def handleUnlock(self):
        """
        Private slot to handle the Unlock context menu entry.
        """
        names = [os.path.join(self.dname, unicode(itm.text(10))) \
                 for itm in self.getLockActionItems(self.lockedIndicators)]
        if not names:
            KQMessageBox.information(self,
                self.trUtf8("Unlock"),
                self.trUtf8("""There are no locked files available/selected."""),
                self.trUtf8("&OK"),
                QString.null,
                QString.null,
                0, -1)
            return
        
        self.vcs.svnUnlock(names, parent=self)
        self.refreshStatus()

    def handleBreakLock(self):
        """
        Private slot to handle the Break Lock context menu entry.
        """
        names = [os.path.join(self.dname, unicode(itm.text(10))) \
                 for itm in self.getLockActionItems(self.stealBreakLockIndicators)]
        if not names:
            KQMessageBox.information(self,
                self.trUtf8("Break Lock"),
                self.trUtf8("""There are no locked files available/selected."""),
                self.trUtf8("&OK"),
                QString.null,
                QString.null,
                0, -1)
            return
        
        self.vcs.svnUnlock(names, parent=self, breakIt=True)
        self.refreshStatus()

    def handleStealLock(self):
        """
        Private slot to handle the Break Lock context menu entry.
        """
        names = [os.path.join(self.dname, unicode(itm.text(10))) \
                 for itm in self.getLockActionItems(self.stealBreakLockIndicators)]
        if not names:
            KQMessageBox.information(self,
                self.trUtf8("Steal Lock"),
                self.trUtf8("""There are no locked files available/selected."""),
                self.trUtf8("&OK"),
                QString.null,
                QString.null,
                0, -1)
            return
        
        self.vcs.svnLock(names, parent=self, stealIt=True)
        self.refreshStatus()

    def getLockActionItems(self, indicators):
        """
        Private method to retrieve all emtries, that have a locked status.
        
        @return list of all items with a locked status
        """
        lockitems = []
        itm = self.statusList.firstChild()
        while itm:
            if itm.isSelected() and indicators.contains(itm.text(5)):
                lockitems.append(itm)
            itm = itm.itemBelow()
        return lockitems

    def refreshStatus(self):
        """
        Private slot to refresh the status display.
        """
        self.cancelButton.setText(self.trUtf8('&Cancel'))
        self.cancelButton.setDefault(1)
        self.cancelButton.setFocus()
        
        self.inputGroup.setEnabled(1)
        self.refreshButton.setEnabled(0)
        
        for itm in self.menuitems:
            self.menu.setItemEnabled(itm, 0)
        
        self.statusList.clear()
        
        self.start(self.args)
