# -*- coding: utf-8 -*-

# Copyright (c) 2006 - 2008 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing the Editor Styles configuration page.
"""

from PyQt4.QtCore import pyqtSignature
from PyQt4.QtGui import QPixmap, QIcon
from PyQt4.Qsci import QsciScintilla

from KdeQt import KQColorDialog, KQFontDialog

from ConfigurationPageBase import ConfigurationPageBase
from Ui_EditorStylesPage import Ui_EditorStylesPage

import Preferences

class EditorStylesPage(ConfigurationPageBase, Ui_EditorStylesPage):
    """
    Class implementing the Editor Styles configuration page.
    """
    def __init__(self):
        """
        Constructor
        """
        ConfigurationPageBase.__init__(self)
        self.setupUi(self)
        self.setObjectName("EditorStylesPage")
        
        self.foldStyles = [
            QsciScintilla.PlainFoldStyle,
            QsciScintilla.CircledFoldStyle,
            QsciScintilla.BoxedFoldStyle,
            QsciScintilla.CircledTreeFoldStyle,
            QsciScintilla.BoxedTreeFoldStyle
        ]
        self.edgeModes = [
            QsciScintilla.EdgeNone,
            QsciScintilla.EdgeLine,
            QsciScintilla.EdgeBackground
        ]
        
        # set initial values
        eolMode = Preferences.getEditor("EOLMode")
        if eolMode == QsciScintilla.EolWindows:
            self.crlfRadioButton.setChecked(True)
        elif eolMode == QsciScintilla.EolMac:
            self.crRadioButton.setChecked(True)
        elif eolMode == QsciScintilla.EolUnix:
            self.lfRadioButton.setChecked(True)
            
        self.foldingStyleComboBox.setCurrentIndex(
            self.foldStyles.index(Preferences.getEditor("FoldingStyle")))
        self.marginsFont = Preferences.getEditorOtherFonts("MarginsFont")
        self.marginsFontSample.setFont(self.marginsFont)
        self.defaultFont = Preferences.getEditorOtherFonts("DefaultFont")
        self.defaultFontSample.setFont(self.defaultFont)
        self.monospacedFont = Preferences.getEditorOtherFonts("MonospacedFont")
        self.monospacedFontSample.setFont(self.monospacedFont)
        self.monospacedCheckBox.setChecked(\
            Preferences.getEditor("UseMonospacedFont"))
        self.caretlineVisibleCheckBox.setChecked(\
            Preferences.getEditor("CaretLineVisible"))
        self.caretWidthSpinBox.setValue(\
            Preferences.getEditor("CaretWidth"))
        self.colourizeSelTextCheckBox.setChecked(\
            Preferences.getEditor("ColourizeSelText"))
        self.customSelColourCheckBox.setChecked(\
            Preferences.getEditor("CustomSelectionColours"))
        self.extentSelEolCheckBox.setChecked(\
            Preferences.getEditor("ExtendSelectionToEol"))
        
        self.edgeModeCombo.setCurrentIndex(
            self.edgeModes.index(Preferences.getEditor("EdgeMode")))
        self.edgeLineColumnSlider.setValue(\
            Preferences.getEditor("EdgeColumn"))
        self.edgeColour = Preferences.getEditorColour("Edge")
        size = self.edgeBackgroundColorButton.size()
        pm = QPixmap(size.width()/2, size.height()/2)
        pm.fill(self.edgeColour)
        self.edgeBackgroundColorButton.setIconSize(pm.size())
        self.edgeBackgroundColorButton.setIcon(QIcon(pm))
        
    def save(self):
        """
        Public slot to save the Editor Styles configuration.
        """
        if self.crlfRadioButton.isChecked():
            Preferences.setEditor("EOLMode", QsciScintilla.EolWindows)
        elif self.crRadioButton.isChecked():
            Preferences.setEditor("EOLMode", QsciScintilla.EolMac)
        elif self.lfRadioButton.isChecked():
            Preferences.setEditor("EOLMode", QsciScintilla.EolUnix)
        Preferences.setEditor("FoldingStyle",
            self.foldStyles[self.foldingStyleComboBox.currentIndex()])
        Preferences.setEditorOtherFonts("MarginsFont", self.marginsFont)
        Preferences.setEditorOtherFonts("DefaultFont", self.defaultFont)
        Preferences.setEditorOtherFonts("MonospacedFont", self.monospacedFont)
        Preferences.setEditor("UseMonospacedFont",
            int(self.monospacedCheckBox.isChecked()))
        Preferences.setEditor("CaretLineVisible",
            int(self.caretlineVisibleCheckBox.isChecked()))
        Preferences.setEditor("ColourizeSelText",
            int(self.colourizeSelTextCheckBox.isChecked()))
        Preferences.setEditor("CustomSelectionColours", 
            int(self.customSelColourCheckBox.isChecked()))
        Preferences.setEditor("ExtendSelectionToEol", 
            int(self.extentSelEolCheckBox.isChecked()))
        
        Preferences.setEditor("CaretWidth", 
            self.caretWidthSpinBox.value())
        Preferences.setEditor("EdgeMode",
            self.edgeModes[self.edgeModeCombo.currentIndex()])
        Preferences.setEditor("EdgeColumn",
            self.edgeLineColumnSlider.value())
        Preferences.setEditorColour("Edge", self.edgeColour)
        
    @pyqtSignature("")
    def on_edgeBackgroundColorButton_clicked(self):
        """
        Private slot to set the colour for the edge background or line.
        """
        colour = KQColorDialog.getColor(self.edgeColour)
        if colour.isValid():
            self.edgeColour = colour
            size = self.edgeBackgroundColorButton.iconSize()
            pm = QPixmap(size.width(), size.height())
            pm.fill(colour)
            self.edgeBackgroundColorButton.setIcon(QIcon(pm))
        
    @pyqtSignature("")
    def on_linenumbersFontButton_clicked(self):
        """
        Private method used to select the font for the editor margins.
        """
        font, ok = KQFontDialog.getFont(self.marginsFont)
        if ok:
            self.marginsFontSample.setFont(font)
            self.marginsFont = font
        
    @pyqtSignature("")
    def on_defaultFontButton_clicked(self):
        """
        Private method used to select the default font for the editor.
        """
        font, ok = KQFontDialog.getFont(self.defaultFont)
        if ok:
            self.defaultFontSample.setFont(font)
            self.defaultFont = font
        
    @pyqtSignature("")
    def on_monospacedFontButton_clicked(self):
        """
        Private method used to select the font to be used as the monospaced font.
        """
        font, ok = KQFontDialog.getFont(self.monospacedFont)
        if ok:
            self.monospacedFontSample.setFont(font)
            self.monospacedFont = font
        
    def polishPage(self):
        """
        Public slot to perform some polishing actions.
        """
        self.marginsFontSample.setFont(self.marginsFont)
        self.defaultFontSample.setFont(self.defaultFont)
        self.monospacedFontSample.setFont(self.monospacedFont)

def create(dlg):
    """
    Module function to create the configuration page.
    
    @param dlg reference to the configuration dialog
    """
    page = EditorStylesPage()
    return page
