# -*- coding: utf-8 -*-

# Copyright (c) 2004 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing a module widget.
"""

from qt import *
from qtcanvas import *

from UMLWidget import UMLWidget

class ModuleModel:
    """
    Class implementing the module model.
    """
    def __init__(self, name, classlist=[]):
        """
        Constructor
        
        @param name the module name (string)
        @param classlist list of class names (list of strings)
        """
        self.name = name
        self.classlist = classlist
        
    def addClass(self, classname):
        """
        Method to add a class to the module model.
        
        @param classname class name to be added (string)
        """
        self.classlist.append(classname)
        
    def getClasses(self):
        """
        Method to retrieve the classes of the module.
        
        @return list of class names (list of strings)
        """
        return self.classlist[:]
        
    def getName(self):
        """
        Method to retrieve the module name.
        
        @return module name (string)
        """
        return self.name
        
class ModuleWidget(UMLWidget):
    """
    Class implementing a module widget.
    """
    def __init__(self, canvas, model=None, x=0, y=0, rounded=0):
        """
        Constructor
        
        @param canvas canvas containing the class widget (QCanvas)
        @param model module model containing the module data (ModuleModel)
        @param x x-coordinate (integer)
        @param y y-coordinate (integer)
        @param rounded flag indicating a rounded corner (boolean)
        """
        UMLWidget.__init__(self, canvas, x, y, rounded)
        self.model = model
        
        if self.model:
            self.calculateSize()
        
        self.update()
        
    def calculateSize(self):
        """
        Method to calculate the size of the module widget.
        """
        if self.model is None:
            return
            
        classes = self.model.getClasses()
        
        font = QFont(self.font)
        font.setBold(1)
        font.setItalic(1)
        fm = QFontMetrics(font)
        
        # calculate height
        fontHeight = fm.lineSpacing()
        lines = 1
        if classes:
            lines += len(classes)
        else:
            lines += 1
        height = lines * fontHeight
        
        # calculate width
        width = fm.width(self.model.getName())
        
        for cls in classes:
            w = fm.width(cls)
            width = max(w, width)
        
        width += self.margin * 2
        width += 4  # a little bit more for strange fonts
        height += self.margin * 4
        
        self.setSize(width, height)
    
    def setModel(self, model):
        """
        Method to set the module model.
        
        @param model module model containing the module data (ModuleModel)
        """
        self.model = model
        self.calculateSize()
        self.update()
        
    def drawShape(self, painter):
        """
        Overriden method to draw the shape.
        
        @param painter painter the shape is drawn to (QPainter)
        """
        painter.setPen(Qt.black)
        offsetX = self.x()
        offsetY = self.y()
        w = self.width()
        h = self.height()
        fm = QFontMetrics(self.font)
        fontHeight = fm.lineSpacing() + 1 # one for the baseline itself
        
        painter.drawRect(offsetX, offsetY, w, h)
        
        y = self.margin
        self.font.setBold(1)
        painter.setFont(self.font)
        painter.drawText(offsetX + self.margin,
                         offsetY + y,
                         w - self.margin * 2,
                         fontHeight,
                         Qt.AlignHCenter,
                         self.model.getName())
        y += fontHeight + self.margin
        self.font.setBold(0)
        painter.setFont(self.font)
        painter.drawLine(offsetX, offsetY + y,
                         offsetX + w - 1, offsetY + y)
        y += self.margin
        
        classes = self.model.getClasses()
        if classes:
            for cls in classes:
                painter.drawText(offsetX + self.margin,
                                 offsetY + y,
                                 w - self.margin * 2,
                                 fontHeight,
                                 Qt.AlignHCenter,
                                 cls)
                y += fontHeight
        else:
            painter.drawText(offsetX + self.margin,
                             offsetY + y,
                             w - self.margin * 2,
                             fontHeight,
                             Qt.AlignHCenter,
                             "")
            y += fontHeight
        
        if self.isSelected():
            self.drawSelected(painter, offsetX, offsetY)
