# -*- coding: utf-8 -*-

# Copyright (c) 2003 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing a dialog for the configuration of a keyboard shortcut.
"""

from qt import *

from ShortcutForm import ShortcutForm


class ShortcutDialog(ShortcutForm):
    """
    Class implementing a dialog for the configuration of a keyboard shortcut.
    
    @signal shortcutChanged(QKeySequence) emitted after the OK button
            was pressed
    """
    def __init__(self,parent = None,name = None,modal = 0,fl = 0):
        """
        Constructor
        
        @param parent The parent widget of this dialog. (QWidget)
        @param name The name of this dialog. (QString)
        @param modal Flag indicating a modal dialog. (boolean)
        @param fl The window flags.
        """
        ShortcutForm.__init__(self,parent,name,modal,fl)
        
        self.keyIndex = 0
        self.keys = [0, 0, 0, 0]

    def setKey(self, key):
        """
        Public method to set the key to be configured.
        
        @param ey key sequence to be changed (QKeySequence)
        """
        self.keyIndex = 0
        self.keys = [0, 0, 0, 0]
        self.keyLabel.setText(QString(key))
        
    def handleOk(self):
        """
        Private slot to handle the OK button press.
        """
        self.hide()
        self.emit(PYSIGNAL('shortcutChanged'), (QKeySequence(self.keyLabel.text()),))

    def handleClear(self):
        """
        Private slot to handle the Clear button press.
        """
        self.keyIndex = 0
        self.keys = [0, 0, 0, 0]
        self.keyLabel.setText("")
        
    def keyPressEvent(self, evt):
        """
        Private method to handle a key press event.
        
        @param evt the key event (QKeyEvent)
        """
        if evt.key() == Qt.Key_Control:
            return;
        if evt.key() == Qt.Key_Meta:
            return;
        if evt.key() == Qt.Key_Shift:
            return;
        if evt.key() == Qt.Key_Alt:
            return;
        if evt.key() == Qt.Key_Menu:
            return;
    
        if self.keyIndex == 4:
            self.keyIndex = 0
            self.keys = [0, 0, 0, 0]
    
        self.keys[self.keyIndex] = evt.key();
        
        if evt.state() & Qt.ShiftButton:
            self.keys[self.keyIndex] += Qt.SHIFT
        if evt.state() & Qt.ControlButton:
            self.keys[self.keyIndex] += Qt.CTRL
        if evt.state() & Qt.AltButton:
            self.keys[self.keyIndex] += Qt.ALT
        # the following is a workaround for a bug in PyQt 3.7
        try:
            if evt.state() & Qt.MetaButton:
                self.keys[self.keyIndex] += Qt.META
        except AttributeError:
            if evt.state() & 0x0800:
                self.keys[self.keyIndex] += Qt.META
        
        self.keyIndex += 1
        
        if self.keyIndex == 1:
            self.keyLabel.setText(QString(QKeySequence(self.keys[0])))
        elif self.keyIndex == 2:
            self.keyLabel.setText(QString(QKeySequence(self.keys[0], self.keys[1])))
        elif self.keyIndex == 3:
            self.keyLabel.setText(QString(QKeySequence(self.keys[0], self.keys[1],
                self.keys[2])))
        elif self.keyIndex == 4:
            self.keyLabel.setText(QString(QKeySequence(self.keys[0], self.keys[1],
                self.keys[2], self.keys[3])))
