#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (c) 2002 - 2004 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Eric3 Helpviewer

This is the main Python script that performs the neccessary initialization
of the helpviewer and starts the Qt event loop. This is a standalone version
of the integrated helpviewer.
"""

import sys
import os

from qt import QApplication, QTranslator, QTextCodec, QMimeSourceFactory, \
               SIGNAL, SLOT

import Preferences
import Utilities

def loadTranslator(dirs, tn):
    """
    Global function to find and load a specific translation.

    @param dirs Searchpath for the translations. (list of strings)
    @param tn The translation to be loaded. (string)
    @return Tuple of a status flag and the loaded translator. (int, QTranslator)
    """
    trans = QTranslator(None)
    for dir in dirs:
        loaded = trans.load(tn, dir)
        if loaded:
            return (trans, 1)
            
    print "eric3-helpviewer: no translation file '" + tn + "'found."
    print "Using default."
    return (None, 0)

def initializeMimeSourceFactory(ericDir):
    """
    Global function to initialize the default mime source factory.
    
    @param ericDir directory name of the eric installation
    """
    defaultFactory = QMimeSourceFactory.defaultFactory()
    defaultIconPath = os.path.join(ericDir, "icons", "default")
    iconPaths = Preferences.getIcons("Path")
    for iconPath in iconPaths:
        defaultFactory.addFilePath(iconPath)
    if not defaultIconPath in iconPaths:
        defaultFactory.addFilePath(defaultIconPath)

def main():
    """
    Main entry point into the application.
    """
    ericDir = os.path.dirname(sys.argv[0])
    if ericDir == '':
        ericDir = '.'
    app = QApplication(sys.argv)

    # set the searchpath for icons
    initializeMimeSourceFactory(ericDir)

    # We can only import these after creating the QApplication because they
    # make Qt calls that need the QApplication to exist.
    from Helpviewer.HelpWindow import HelpWindow

    # Set the applications string encoding
    try:
        sys.setappdefaultencoding(str(Preferences.getSystem("StringEncoding")))
    except:
        pass
        
    # get the Qt installation directory
    qtdir = Preferences.getQtDir()
    if qtdir is not None:
        if Preferences.getQt("ExportQtDir"):
            os.environ["QTDIR"] = qtdir
    
    # Use the program on the command line if one was given.
    try:
        home = sys.argv[1]
    except:
        if qtdir is not None:
            home = Utilities.normjoinpath(qtdir, 'doc', 'html', 'index.html')
        else:
            home = None
    
    # Load translation files and install them
    loaded = 0
    loc = Preferences.getUILanguage()
    if loc is not None:
        if loc == "System":
            loc = str(QTextCodec.locale())
            
        if loc != "C":
            # setup the searchpath for qt
            dirs = [ericDir, "."]
            if qtdir is not None:
                dirs.insert(-1, os.path.join(qtdir, "translations"))
                
            # 1) load translations for qt
            qtTrans, ok = loadTranslator(dirs, "qt_" + loc)
            if ok:
                app.installTranslator(qtTrans)
                
            # 2) load translations for helpviewer
            dirs = [".", ericDir]
            ericTrans, ok = loadTranslator(dirs, "eric3_" + loc)
            if ok:
                app.installTranslator(ericTrans)
            else:
                print "Please report to <eric-bugs@die-offenbachs.de>."
        
    help = HelpWindow(home, '.', None, 'help viewer')
    help.setCaption('Helpviewer')

    app.setMainWidget(help)
    if QApplication.desktop().width() > 400 and \
       QApplication.desktop().height() > 500:
        help.show()
    else:
        help.showMaximized()

    app.connect(app, SIGNAL('lastWindowClosed()'),
                app, SLOT('quit()'))
    res = app.exec_loop()
    sys.exit(res)

if __name__ == '__main__':
    main()
