# -*- coding: utf-8 -*-

# Copyright (c) 2002 Detlev Offenbach <detlev@die-offenbachs.de>
#
# This is the install script for eric3. It is based on the
# build script found in the PyQt distribution. The original is 
# Copyright (c) 2002 Riverbank Computing Limited <info@riverbankcomputing.co.uk>

import sys
import os
import string
import re
import compileall
import glob
import shutil
import fnmatch
import distutils.sysconfig

# Define the globals.
progName = None
modDir = None
platBinDir = None
distDir = None

def usage(rcode = 2):
    """Display a usage message and exit.

    rcode is the return code passed back to the calling process.
    """
    global progName, platBinDir, modDir, distDir

    print "Usage:"
    print "    %s [-hx] [-b dir] [-d dir] [-i dir]" % (progName)
    print "where:"
    print "    -h             display this help message"
    print "    -b dir         where eric3, qtunittest and helpviewer will be installed [default %s]" % (platBinDir)
    print "    -d dir         where eric3 python files will be installed [default %s]" % (modDir)
    if sys.platform <> "win32":
        print "    -i dir         temporary install prefix [default %s]" % (distDir)
    print "    -x             don't perform dependency checks (use on your own risk)"

    sys.exit(rcode)


def initGlobals():
    """Sets the values of globals that need more than a simple assignment.
    """
    global platBinDir, modDir

    if sys.platform == "win32":
        platBinDir = sys.exec_prefix
    else:
        platBinDir = "/usr/local/bin"

    modDir = distutils.sysconfig.get_python_lib()


def copyToFile(name,text):
    """Copy a string to a file.

    name is the name of the file.
    text is the contents to copy to the file.
    """
    f = open(name,"w")
    f.write(text)
    f.close()


def wrapperName(dname,wfile):
    """Create the platform specific name for the wrapper script.
    """
    if sys.platform == "win32":
        wname = dname + "\\" + wfile + ".bat"
    else:
        wname = dname + "/" + wfile

    return wname


def createPyWrapper(pydir,wfile):
    """Create an executable wrapper for a Python script.

    pydir is the name of the directory where the Python script will eventually
    be installed.
    wfile is the basename of the wrapper.

    Returns the platform specific name of the wrapper.
    """
    if sys.platform == "win32":
        wname = wfile + ".bat"
        wrapper = \
"""@%s\\pythonw %s\\%s.py %%1 %%2 %%3 %%4 %%5 %%6 %%7 %%8 %%9
""" % (platBinDir,pydir,wfile)
    else:
        wname = wfile
        wrapper = \
"""#!/bin/sh

exec %s %s/%s.py $*
""" % (sys.executable,pydir,wfile)

    copyToFile(wname,wrapper)
    os.chmod(wname,0755)

    return wname


def copyTree(src, dst):
    """Copy Python, translation, documentation, wizards configuration
    and designer template files of a directory tree.
    """
    names = os.listdir(src)
    if not os.path.isdir(dst):
        os.mkdir(dst)
    for name in names:
        srcname = os.path.join(src, name)
        dstname = os.path.join(dst, name)
        if fnmatch.fnmatch(srcname, '*.py') or \
           fnmatch.fnmatch(srcname, '*.pyc') or \
           fnmatch.fnmatch(srcname, '*.pyo') or \
           fnmatch.fnmatch(srcname, '*.qm') or \
           fnmatch.fnmatch(srcname, '*.html') or \
           fnmatch.fnmatch(srcname, '*.e3w') or \
           fnmatch.fnmatch(srcname, '*.tmpl'):
            shutil.copy2(srcname, dstname)
        elif os.path.isdir(srcname):
            copyTree(srcname, dstname)


def copyIconTree(src, dst):
    """Copy icon files of a directory tree.
    """
    names = os.listdir(src)
    if not os.path.isdir(dst):
        os.mkdir(dst)
    for name in names:
        srcname = os.path.join(src, name)
        dstname = os.path.join(dst, name)
        if fnmatch.fnmatch(srcname, '*.png'):
            shutil.copy2(srcname, dstname)
        elif os.path.isdir(srcname):
            copyIconTree(srcname, dstname)


def installEric():
    """Create the python wrapper files for eric3, qtunittest and helpviewer
    and install all the necessary files comprising the package.
    """
    global modDir, platBinDir, distDir
    
    ericdir = os.path.join(modDir, "eric3")
    ericpixdir = os.path.join(ericdir, "pixmaps")
    ericicondir = os.path.join(ericdir, "icons")
    
    # Create the platform specific wrappers.
    ename = createPyWrapper(ericdir,"eric3")
    uname = createPyWrapper(ericdir,"eric3-unittest")
    hname = createPyWrapper(ericdir,"eric3-helpviewer")
    dname = createPyWrapper(ericdir,"eric3-doc")
    qname = createPyWrapper(ericdir,"eric3-qregexp")
    rname = createPyWrapper(ericdir,"eric3-re")

    # set install prefix, if not None
    #
    # ToDo: This has to be reworked, if there is a need
    #       for distDir on "win32".
    #       Don't use os.path.join because absolute paths
    #       seem to be assumed, but not assured.
    #       At least for linux, double slashes make no problems.
    #
    if distDir:
        ericdir = distDir + os.sep + ericdir
        ericpixdir = distDir + os.sep + ericpixdir
        ericicondir = distDir + os.sep + ericicondir
        bindir  = distDir + os.sep +  platBinDir
        mdir = distDir + os.sep + modDir
    else:
        bindir = platBinDir
        mdir = modDir
    
    # Cleanup old stuff
    # Remove the wrapper scripts
    rem_ename = wrapperName(platBinDir,"eric3")
    rem_uname = wrapperName(platBinDir,"eric3-unittest")
    rem_hname = wrapperName(platBinDir,"eric3-helpviewer")
    rem_dname = wrapperName(platBinDir,"eric3-doc")
    rem_qname = wrapperName(platBinDir,"eric3-qregexp")
    rem_rname = wrapperName(platBinDir,"eric3-re")
    rem_ouname = wrapperName(platBinDir,"qtunittest")
    rem_ohname = wrapperName(platBinDir,"helpviewer")
    if os.path.exists(rem_ename):
        os.remove(rem_ename)
    if os.path.exists(rem_uname):
        os.remove(rem_uname)
    if os.path.exists(rem_hname):
        os.remove(rem_hname)
    if os.path.exists(rem_dname):
        os.remove(rem_dname)
    if os.path.exists(rem_qname):
        os.remove(rem_qname)
    if os.path.exists(rem_rname):
        os.remove(rem_rname)
    if os.path.exists(rem_ouname):
        os.remove(rem_ouname)
    if os.path.exists(rem_ohname):
        os.remove(rem_ohname)
    
    # Cleanup the install directory
    try:
        shutil.rmtree(ericdir, 1)
    except:
        pass

    # Install the files
    # make the install directory tree
    os.makedirs(ericpixdir)
    if not os.path.isdir(bindir):
        os.makedirs(bindir)
    # copy the new sitecustomize.py
    shutil.copy('eric%ssitecustomize.py' % os.sep, mdir)
    # copy the python files and the translations
    copyTree('eric', ericdir)
    
    # copy the icons
    copyIconTree('eric%sicons' % os.sep, ericicondir)
    
    # copy the pixmaps
    for fn in glob.glob('eric%spixmaps%s*.png' % (os.sep, os.sep)):
        shutil.copy(fn, ericpixdir)
    for fn in glob.glob('eric%spixmaps%s*.xpm' % (os.sep, os.sep)):
        shutil.copy(fn, ericpixdir)
    # copy the wrappers
    shutil.copy(ename, bindir)
    shutil.copy(uname, bindir)
    shutil.copy(hname, bindir)
    shutil.copy(dname, bindir)
    shutil.copy(qname, bindir)
    shutil.copy(rname, bindir)
    # copy the license file
    shutil.copy('eric%sLICENSE.GPL' % os.sep, ericdir)


def createSitecustomize():
    """Generate the sitecustomize.py script merging an already
       installed script with our additions.
    """
    global modDir, distDir

    exists = 1

    if not distDir:
        try:
            f = open(os.path.join(modDir, 'sitecustomize.py'), 'r')
        except:
            exists = 0
    else:
        exists = 0

    if exists:
        lines = f.readlines()
        f.close()
        s = open(os.path.join("eric", "sitecustomize.py"), "w")
        if not "sys.setappdefaultencoding = sys.setdefaultencoding\n" in lines:
            s.write("#\n# added by the eric install script\n#\n\n")
            s.write("import sys\n\n")
            s.write("sys.setappdefaultencoding = sys.setdefaultencoding\n\n")
        for line in lines:
            s.write(line)
        s.close()
    else:
        s = open(os.path.join("eric", "sitecustomize.py"), "w")
        s.write("#\n# sitecustomize.py - saving a useful function.\n")
        s.write("# generated by the eric install script\n#\n\n")
        s.write("import sys\n\n")
        s.write("sys.setappdefaultencoding = sys.setdefaultencoding\n\n")
        s.close()

def main(argv):
    """The main function of the script.

    argv is the list of command line arguments.
    """
    import getopt

    # Parse the command line.
    global progName, modDir
    progName = os.path.basename(argv[0])

    initGlobals()

    try:
        if sys.platform == "win32":
            optlist, args = getopt.getopt(argv[1:],"hxb:d:")
        else:
            optlist, args = getopt.getopt(argv[1:],"hxb:d:i:")
    except getopt.GetoptError:
        usage()

    global platBinDir
    
    depChecks = 1

    for opt, arg in optlist:
        if opt == "-h":
            usage(0)
        elif opt == "-b":
            platBinDir = arg
        elif opt == "-d":
            global modDir
            modDir = arg
        elif opt == "-i":
            global distDir
            distDir = os.path.normpath(arg)
        elif opt == "-x":
            depChecks = 0
    
    if depChecks:
        # perform dependency checks
        if sys.version_info < (2,1,0):
            print 'Sorry, you must have Python 2.1.0 or higher.'
            sys.exit(5)
            
        try:
            from qt import qVersion, PYQT_VERSION
        except ImportError, msg:
            print 'Sorry, please install PyQt.'
            print 'Error: %s' % msg
            sys.exit(1)
            
        try:
            from qtnetwork import QServerSocket
        except ImportError, msg:
            print 'Sorry, PyQt is missing QtNetwork support.'
            print 'Error: %s' % msg
            sys.exit(1)
            
        try:
            from qtcanvas import QCanvas
        except ImportError, msg:
            print 'Sorry, PyQt is missing QtCanvas support.'
            print 'Error: %s' % msg
            sys.exit(1)
            
        try:
            from qttable import QTable
        except ImportError, msg:
            print 'Sorry, PyQt is missing QtTable support.'
            print 'Error: %s' % msg
            sys.exit(1)
            
        try:
            from qtext import QextScintilla
        except ImportError, msg:
            print 'Sorry, please install QScintilla and/or reinstall'
            print 'PyQt with QScintilla support.'
            print 'Error: %s' % msg
            sys.exit(1)
            
        # check version of Qt
        qtMajor = int(qVersion().split('.')[0])
        qtMinor = int(qVersion().split('.')[1])
        if qtMajor < 3 or (qtMajor == 3 and qtMinor < 1):
            print 'Sorry, you must have Qt version 3.1.0 or higher.'
            sys.exit(2)
            
        #check version of PyQt
        try:
            from qt import PYQT_VERSION_STR
            pyqtVersion = PYQT_VERSION_STR
        except:
            pyqtVersion = PYQT_VERSION
        if pyqtVersion[:9] == "snapshot-":
            if long(pyqtVersion[9:]) < 20030418L:
                print 'Sorry, you must have PyQt snapshot-20030418 or higher.'
                sys.exit(3)
        else:
            while pyqtVersion.count('.') < 2:
                pyqtVersion += '.0'
            (maj, min, pat) = pyqtVersion.split('.')
            maj = int(maj)
            min = int(min)
            pat = int(pat)
            if maj < 3 or (maj == 3 and min < 6):
                print 'Sorry, you must have PyQt 3.6.0 or snapshot-20030418 or higher.'
                sys.exit(3)
            if maj == 3 and min == 9:
                print 'PyQt 3.9 contains a bug, which prevents eric3 from functioning.'
                print 'Please get PyQt 3.8 or 3.10 or newer (or snapshot-20031226 or newer).'
                sys.exit(3)
            
        #check version of QScintilla
        try:
            from qtext import QSCINTILLA_VERSION_STR
            scintillaVersion = QSCINTILLA_VERSION_STR
            if scintillaVersion[:9] == "snapshot-":
                if long(scintillaVersion[9:]) < 20030418L:
                    print 'Sorry, you must have QScintilla snapshot-20030418 or higher.'
                    sys.exit(4)
            else:
                while scintillaVersion.count('.') < 2:
                    scintillaVersion += '.0'
                (maj, min, pat) = scintillaVersion.split('.')
                maj = int(maj)
                min = int(min)
                pat = int(pat)
                if maj < 1 or (maj == 1 and min < 0):
                    print 'Sorry, you must have QScintilla 1.0 or snapshot-20030418 or higher.'
                    sys.exit(4)
        except:
            print 'Sorry, you must have QScintilla 1.0 or snapshot-20030418 or higher.'
            sys.exit(4)
        
    createSitecustomize()
    print "Compiling debugger files."
    if distDir:
        compileall.compile_dir("eric", ddir=os.path.join(distDir, modDir))
    else:
        compileall.compile_dir("eric", ddir=modDir)
    print "\nInstalling eric3 ..."
    installEric()
    print "\nInstallation complete."
    print "\nIf you would like to use the autocompletion feature please call"
    print "the gen_pyqt_api.py and/or gen_python_api.py script available"
    print "on the eric3 contribution page or get ready made api-files"
    print "from the contribution page."
    print
    
    
if __name__ == "__main__":
    try:
        main(sys.argv)
    except SystemExit:
        raise
    except:
        print \
"""An internal error occured.  Please report all the output of the program,
including the following traceback, to eric-bugs@die-offenbachs.de.
"""
        raise

