"""

Interface to cdrdao

"""

from tools import cmdoutput, striplist, which, TRUE, FALSE
from string import split, join, digits, find, strip, letters, atoi, lower
from types import StringType
from log4py import Logger, LOGLEVEL_NORMAL
from tempfile import mktemp

import os

class cdrdao:

    speed = 1
    dummy_mode = FALSE
    eject = FALSE
    overburn = FALSE
    multisession = FALSE
    swap_byte_order = FALSE
    toc_file = ""

    def __init__(self, bus, target, lun, device, loglevel = LOGLEVEL_NORMAL):
        self.__cdrdao_logger = Logger().get_instance(self)
        self.__cdrdao_logger.set_loglevel(loglevel)
        self.__cdrdao_command = which("cdrdao")
        if (type(bus) == StringType): bus = atoi(bus)
        if (type(target) == StringType): target = atoi(target)
        if (type(lun) == StringType): lun = atoi(lun)
        self.__cdrdao_bus = bus
        self.__cdrdao_target = target
        self.__cdrdao_lun = lun
        self.__cdrdao_device = device
        self.__cdrdao_tracks = []
        self.__cdrdao_read_version()

    def version(self):
        """ Returns the version string of cdrdao. """
        if (self.__cdrdao_version_micro != None):
            return "%d.%d.%d" % (self.__cdrdao_version_major, self.__cdrdao_version_minor, self.__cdrdao_version_micro)
        else:
            return "%d.%d" % (self.__cdrdao_version_major, self.__cdrdao_version_minor)

    def device(self):
        return self.__cdrdao_device

    def add_track(self, value):
        self.__cdrdao_tracks.append("\"%s\"" % value)

    def command_line(self):
        cmdline = self.__cdrdao_command
        if (self.dummy_mode):
            cmdline ="%s simulate -n -v 2" % cmdline
        else:
            cmdline ="%s write -n -v 2" % cmdline

        if (self.eject): cmdline = "%s --eject" % cmdline
        if (self.overburn): cmdline = "%s --overburn" % cmdline
        if (self.multisession): cmdline = "%s --multi" % cmdline
        if (self.swap_byte_order): cmdline = "%s --swap" % cmdline

        cmdline = "%s --device %d,%d,%d" % (cmdline, self.__cdrdao_bus, self.__cdrdao_target, self.__cdrdao_lun)
        if (type(self.speed) == StringType): self.speed = atoi(self.speed)
        cmdline = "%s --speed %d" % (cmdline, self.speed)
        if (self.toc_file == ""):
            cmdline = "%s %s" % (cmdline, self.__cdrdao_create_toc_file())
        else:
            cmdline = "%s \"%s\"" % (cmdline, self.toc_file)
        return cmdline
    
    def installed(self):
        """ Returns, wether cdrdao is installed on the system or not. """
        return (self.__cdrdao_command != "")

    # Private methods of the cdrecord class
    def __cdrdao_read_version(self):
        """ Reads the version string by executing "cdrdao". """
        if (self.__cdrdao_command == ""):
            self.__cdrdao_logger.warn("cdrdao not found - couldn't get version info")
        else:
            output = cmdoutput("%s 2>&1" % self.__cdrdao_command, strip = TRUE)
            versionLine = output[1]
            splitted = split(versionLine, " ")
            version = split(splitted[2], ".")
            self.__cdrdao_version_major = atoi(version[0])
            self.__cdrdao_version_minor = atoi(version[1])
            self.__cdrdao_version_micro = atoi(version[2])

    def __cdrdao_create_toc_file(self):
        """ Creates a temporary toc file. """
        filename = mktemp("_eroaster.toc")
        file = open(filename, "w")
        if (self.multisession):
            file.write("CD_ROM_XA\n\n")
        else: 
            file.write("CD_DA\n\n")

        for i in range(len(self.__cdrdao_tracks)):
            file.write("TRACK AUDIO\n")
            trackname = "FILE %s 0\n\n" % self.__cdrdao_tracks[i]
            file.write(trackname)
        file.close()
        return filename
