/*===========================================================================
  Copyright (C) 2009-2011 European Southern Observatory (ESO)
 
  This program is free software; you can redistribute it and/or 
  modify it under the terms of the GNU General Public License as 
  published by the Free Software Foundation; either version 2 of 
  the License, or (at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public 
  License along with this program; if not, write to the Free 
  Software Foundation, Inc., 675 Massachusetts Ave, Cambridge, 
  MA 02139, USA.
 
  Correspondence concerning ESO-MIDAS should be addressed as follows:
        Internet e-mail: midas@eso.org
        Postal address: European Southern Observatory
                        Data Management Division 
                        Karl-Schwarzschild-Strasse 2
                        D 85748 Garching bei Muenchen 
                        GERMANY
===========================================================================*/

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

.IDENT        sort.c
.LANGUAGE     C
.AUTHOR       K. Banse   ESO/IPG
.KEYWORDS     sort algorithms
.PURPOSE      different sorting routines
.VERSION 
 090630		creation 
 110517         last modif
---------------------------------------------------------------------*/

 

void sortr(rka,ndim)
float *rka;
int   ndim;

/*
ndim  = dimension of array rka for sorting
but we pass the arrays with 1 element in front, so that the algorithm
sorts from [1] -> [ndim]
this is the Heapsort algorithm from "Numerical Recipes", page 231

In case you an array has to be sorted starting at index 0 then you have
to pass the reduced adress to rka:
   float *fp, farr;
   fp = farr-1;
   sortit(fp,ndim);
*/
  

{
register int  m, j, i;

float  zka;
  

m = (ndim>>1) + 1; 		     /* ndim/2 + 1 */
  
for (;;)
   {
   if (m > 1)			     /* still hiring */
      zka = rka[--m];
   else			             /* in retirement + promotion phase */
      {
   /* for (k=1;k<=ntrue;k++) printf("%f ",rka[k]); printf("\nir=%d\n",ndim); */
      zka = rka[ndim];	             /* clear a space at end of array rka */
      rka[ndim] = rka[1];            /* retire the top of the heap into it */
      if (--ndim == 1)	             /* done with last promotion */
         {
         rka[1] = zka;	             /* the least competent guy of all... */
         return;		     /* that's it folks  */
         }
      }
     
/* in hiring as well as in promotion phase */
/* here set up to shift down zka to its right level */

   i = m;	
   j = m << 1;                       /* in FORTRAN: j = m + m */
  

   while (j <= ndim)	
      {
      if (j < ndim)
         {
         if (rka[j] < rka[j+1]) j++;
         }
   
      if (zka < rka[j]) 	     /* demote zka */
         {
         rka[i] = rka[j];
         i = j;
         j = j << 1;		     /* in FORTRAN: j = j + j */
         }
      else
         j = ndim + 1;
 
      }
   rka[i] = zka;		     /* put zka into its slot */
   }
}


int sortmed(ra,zbins,num,ll,valu)
int   num, ll;
float *ra, *zbins, *valu;
 
/*

sort an array to get the median value
 
we only sort fully until index of median - then we insert
for sorting we use the Heapsort algorithm from "Numerical Recipes", page 231
 
use via stat = sortmed(ra,zbins,n,ll,valu)
 
input par:
zbins:	float array	low, high excess bins, if lo >= hi, no excess bins
num:	int		size of array RA
ll:	int		index of array element to be returned
 
in/output par:
ra:	float array	array to be sorted
 
output par:
valu:	float 		value = ra(ll) of sorted array ra
status:	int		return status, = 0 is o.k.
 
*/


{
int   ntrue, lltrue, l, i, status;

float zl, zh;

status = 0;

  /*  first check for excess bins */
  if (zbins[2] > zbins[1])
    {
      zl = zbins[1];
      zh = zbins[2];
      i = 0;
      for (l=0;l<num;l++)
	if ((ra[l] >= zl) && (ra[l] <= zh)) {
	  ra[i] = ra[l];
	  i++;
	}
      if (i > 3)
	{
	  ntrue = i ;
	  lltrue = (ntrue+1)/2;
	}
      else if (i < 1)
	{
	  status = -1;
	  return(status);
	}
      else if (i <= 2)
	{
	  *valu = ra[0];
	  return(status);
	}
      else                     /* i == 3 */
	{
	  *valu = ra[1];
	  return(status);
	}
    }
  else
    {
      ntrue = num;
      lltrue = ll;
    }
 
  sortr(ra,ntrue);
  *valu = ra[lltrue];

return 0;
}


void heapsort(n, ra)
int        n;                    /* no. of elements in array           */
float      *ra;                   /* pointer to array to be sorted      */

/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
.PURPOSE   sort array in place using heapsort
.RETURN    none
-----------------------------------------------------------------------*/

{
int      k, j, ir, i;
float    rra;

k = n >> 1;
ir = n - 1;

while (1) 
   {
   if (k>0)
      rra = ra[--k];
   else 
      {
      rra = ra[ir];
      ra[ir] = ra[0];
      if (--ir == 0) 
         {
         ra[0] = rra;
         return;
         }
      }
   i = k;
   j = (k << 1) + 1;
   while (j<=ir) 
      {
      if (j<ir && ra[j]<ra[j+1]) ++j;
      if (rra<ra[j]) 
         {
         ra[i] = ra[j];
         j += (i=j) + 1;
         }
      else j = ir + 1;
      }
   ra[i] = rra;
   }
}


void hsort2(nsiz,ra,ia)
int      nsiz;		/* no. of elements in array */
float    *ra;		/* pointer to array to be sorted */
int      *ia;		/* pointer to corresp. pixnos */

/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
.PURPOSE   sort array and pixnos in place using heapsort
.RETURN    none
-----------------------------------------------------------------------*/

{
int      k, j, ir, i;

int      iia;
float    rra;

k = nsiz >> 1;			/* k = nsiz/2  */
ir = nsiz - 1;

while (1) 			/* forever... */
   {
   if (k>0)
      {
      --k;
      rra = ra[k];
      iia = ia[k];
      }
   else
      {
      rra = ra[ir];
      iia = ia[ir];
      ra[ir] = ra[0];
      ia[ir] = ia[0];

      if (--ir == 0) 
         {
         ra[0] = rra;
         ia[0] = iia;
         return;
         }
      }

   i = k;
   j = (k << 1) + 1;
   while (j<=ir) 
      {
      if ((j<ir) && (ra[j]<ra[j+1])) ++j;

      if (rra<ra[j]) 
         {
         ra[i] = ra[j];
         ia[i] = ia[j];
         j += (i=j) + 1;
         }
      else 
         j = ir + 1;
      }
     ra[i] = rra;
     ia[i] = iia;
   }
}


