#include <wintypes.h>
#include <pcsclite.h>
#include <usb.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <syslog.h>

#include "ifdhandler.h"
#include "etoken.h"

#define T1MAXBUF (254+4)

/* auxiliary functions */
void hexdump(const u_int8_t * buffer, int length)
{
	char line[256];
	int n, i;

	for (i = 0; i < length; i++) {
		if ((i & 0xf) == 0) {
			snprintf(line, sizeof(line), "%04x:", i);
		}
		n = strlen(line);
		snprintf(line + n, sizeof(line) - n, " %02hhx", buffer[i]);
		if ((i & 0xf) == 0xf)
			syslog(LOG_DEBUG, "%s", line);
	};
	if ((i & 0xf) != 0xf)
		syslog(LOG_DEBUG, "%s", line);
}

int
do_usb(usb_dev_handle * dev, int requesttype, int request,
       int value, int index, char *bytes, int size, int timeout)
{
	int rc;

	rc = usb_control_msg(dev, requesttype, request, value, index,
			     bytes, size, timeout);
	if (rc == -1) {
		printf("usb_control_msg returned %u, error is %s\n",
		       rc, usb_strerror());
	}
	return rc;
}

int init_etoken(struct eToken *eToken)
{
	int rc, len;
	u_int8_t buffer[1024];
	u_int8_t cookie[] = { 0x00, 0x00, 0x01, 0x00, 0x88, 0x13 };

	/* request atr */
	rc = do_usb(eToken->usb, 0x40, 0x01, 0x0000, 0x0000, buffer, 0x000,
		    10000);

	/* receive atr */
	rc = do_usb(eToken->usb, 0xc0, 0x81, 0x0000, 0x0000, buffer,
		    0x0023, 10000);
	if ((rc == -1) || (rc == 0)) {
		/* failed, we should get an atr */
		return 0;
	}
	len = buffer[0];
	if (rc < len) {
		/* failed, we need to get a whole atr */
		return 0;
	}

	if (len > MAX_ATR_SIZE) {
		/* too big */
		return 0;
	}

	memcpy(eToken->atr, &buffer[1], len);
	eToken->atrlen = len;

	/* ask for something strange */
	rc = do_usb(eToken->usb, 0x40, 0x03, 0x0000, 0x0000, buffer,
		    0x0000, 10000);

	/* receive strange data */
	rc = do_usb(eToken->usb, 0xc0, 0x83, 0x0000, 0x0000, buffer,
		    0x000d, 10000);

	/* send something strange */
	memcpy(buffer, cookie, sizeof(cookie));
	rc = do_usb(eToken->usb, 0x40, 0x02, 0x0000, 0x0000, buffer,
		    sizeof(cookie), 10000);
	if (rc != sizeof(cookie)) {
		/* the whole cookie should have been send */
		return 0;
	}

	/* get strange answer */
	rc = do_usb(eToken->usb, 0xc0, 0x82, 0x0000, 0x0000, buffer,
		    0x0001, 10000);
	if (rc != 1) {
		/* we should have got one byte */
		return 0;
	}

	if (buffer[0] != 0) {
		/* the answer should have bin 0x00 */
		return 0;
	}

	/* fine. eToken is operational. */
	eToken->nad = 0;	/* start with sequence 0 */
	eToken->ns = 0;		/* sequence 0 for s blocks */
	eToken->nr = 0;		/* sequence 0 for r blocks */
	eToken->ifsc = 0x20;	/* initial ifsc */
	eToken->ifsd = 0x20;	/* initial ifsc */
	eToken->rc = T1_CHECKSUM_LRC;	/* start with sequence 0 */

	return 1;
}

int power_up_etoken(struct eToken *eToken)
{
	struct usb_bus *mybus;
	struct usb_device *mydev;

	usb_init();
	usb_find_busses();
	usb_find_devices();

	mybus = usb_busses;
	while (mybus) {
		mydev = mybus->devices;
		while (mydev) {
			if ((mydev->descriptor.idVendor == 0x529) &&
			    ((mydev->descriptor.idProduct == 0x50c) ||
			     (mydev->descriptor.idProduct == 0x514))) {
				goto found;
			}
			mydev = mydev->next;
		}
		mybus = mybus->next;
	}

	return 0;

      found:
	eToken->usb = usb_open(mydev);
	if (!eToken->usb) {
		printf("%s %d %s: usb_open failed: %s\n",
		       __FILE__, __LINE__, __PRETTY_FUNCTION__,
		       usb_strerror());
		return 0;
	}

	return init_etoken(eToken);
}

int power_down_etoken(struct eToken *eToken)
{
	if (eToken->usb) {
		usb_reset(eToken->usb);
		sleep(1);
		eToken->usb = NULL;
	}
	return 0;
}

int usb_transfer(struct eToken *eToken, u_int8_t * buffer_out,
		 u_int8_t * buffer_in, int len_out, int *len_in)
{
	int rc;

	// printf("%s %d %s: sending %d, hexdump follows\n",
	//              __FILE__, __LINE__, __PRETTY_FUNCTION__, len_out);
	// hexdump (buffer_out, len_out);

	if (eToken == NULL || eToken->usb == NULL) {
		/* we need some token */
		printf("token unavailable!\n");
		return 0;
	}

	/* send via usb */
	rc = do_usb(eToken->usb, 0x40, 0x06, 0x0000, 0x0000, buffer_out,
		    len_out, 10000);
	if (rc != len_out) {
		/* not the whole buffer was transmitted ! */
		return 0;
	}

	/* receive answer via usb */
	rc = do_usb(eToken->usb, 0xc0, 0x86, 0x0000, 0x0000, buffer_in,
		    eToken->ifsc + 5, 10000);
	if (rc == -1) {
		/* failed. */
		return 0;
	}
	// printf("%s %d %s: received %d, hexdump follows\n",
	//              __FILE__, __LINE__, __PRETTY_FUNCTION__, rc);
	// hexdump (buffer_in, rc);

	*len_in = rc;

	return 1;
}
