#include "etpan-abook-local.h"

#include "etpan-abook-driver.h"
#include "etpan-cfg-abook.h"
#include "etpan-cfg-local-abook.h"
#include "etpan-errors.h"
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stdio.h>

static int lookup(struct etpan_abook * abook,
    const char * key, carray ** result);

static int connect(struct etpan_abook * abook);

static void uninitialize(struct etpan_abook * abook);

static struct etpan_abook_driver local_abook_local = {
  .name = "local",
  .connect = connect,
  .lookup = lookup,
  .uninitialize = uninitialize,
};

static int connect(struct etpan_abook * abook)
{
  return NO_ERROR;
}

struct etpan_abook *
etpan_abook_local_new(struct etpan_global_config * global_config,
    char * filename)
{
  struct etpan_abook * abook;
  struct etpan_local_abook_config * abook_config;
  int r;
  
  abook = malloc(sizeof(* abook));
  if (abook == NULL)
    return NULL;
  
  r = etpan_local_abook_config_read(global_config,
      filename, &abook_config);
  if (r != NO_ERROR)
    return NULL;
  
  abook->data = abook_config;
  abook->driver = &local_abook_local;
  abook->connected = 0;
  
  return abook;
}

static void uninitialize(struct etpan_abook * abook)
{
  struct etpan_local_abook_config * abook_config;

  abook_config = abook->data;
  
  etpan_local_abook_config_free(abook_config);
}


static int match_abook(char * pattern, int pattern_len, char * value)
{
  char * dup_value;
  int r;
  char * p;

  r = 0;

  if (value == NULL)
    goto err;

  dup_value = strdup(value);
  if (dup_value == NULL)
    goto err;

  for(p = dup_value ; * p != 0 ; p ++)
    * p = (char) toupper((unsigned char) * p);

  if (strncasecmp(pattern, dup_value, pattern_len) == 0)
    r = 1;
  
  free(dup_value);
  
  return r;
  
 err:
  return r;
}

static struct etpan_abook_entry *
get_entry_from_local_entry(struct etpan_local_abook_entry * local_entry)
{
  struct etpan_abook_entry * entry;
  char * name;
  char * addr;
  char * nick;
  
  if (local_entry->name != NULL) {
    name = strdup(local_entry->name);
    if (name == NULL)
      goto err;
  }
  else {
    name = NULL;
  }
  
  if (local_entry->addr != NULL) {
    addr = strdup(local_entry->addr);
    if (addr == NULL)
      goto free_name;
  }
  else {
    addr = NULL;
  }
  
  if (local_entry->nick != NULL) {
    nick = strdup(local_entry->nick);
    if (nick == NULL)
      goto free_addr;
  }
  else {
    nick = NULL;
  }

  entry = etpan_abook_entry_new(name, addr, nick);
  if (entry == NULL)
    goto free_nick;
  
  return entry;
  
 free_nick:
  free(nick);
 free_addr:
  free(addr);
 free_name:
  free(name);
 err:
  return NULL;
}

static int lookup(struct etpan_abook * abook,
    const char * key, carray ** result)
{
  struct etpan_local_abook_config * abook_config;
  char * dup_key;
  size_t key_len;
  carray * entry_list;
  char * p;
  clistiter * cur;
  int r;
  unsigned int i;
  
  abook_config = abook->data;

  dup_key = strdup(key);
  if (dup_key == NULL)
    goto err;
  
  key_len = strlen(dup_key);
  
  for(p = dup_key ; * p != 0 ; p ++)
    * p = (char) toupper((unsigned char) * p);
  
  entry_list = carray_new(16);
  if (entry_list == NULL)
    goto free_key;
  
  for(cur = clist_begin(abook_config->entry_list) ;
      cur != NULL ; cur = cur->next) {
    struct etpan_local_abook_entry * local_entry;
    struct etpan_abook_entry * entry;
    
    local_entry = cur->data;

    if (match_abook(dup_key, key_len, local_entry->name) || 
        match_abook(dup_key, key_len, local_entry->nick) ||
        match_abook(dup_key, key_len, local_entry->addr)) {
      /* entry is the abook response from lookup */
      /* local_entry is the abook entry of local abook configuration */
      entry = get_entry_from_local_entry(local_entry);
      if (entry == NULL)
        goto free_entry_list;
      
      r = carray_add(entry_list, entry, NULL);
      if (r != NO_ERROR) {
        etpan_abook_entry_free(entry);
        goto free_entry_list;
      }
    }
  }

  * result = entry_list;

  return NO_ERROR;
  
 free_entry_list:
  for(i = 0 ; i < carray_count(entry_list) ; i ++) {
    struct etpan_abook_entry * entry;
    
    entry = carray_get(entry_list, i);
    etpan_abook_entry_free(entry);
  }
 free_key:
  free(dup_key);
 err:
  return ERROR_MEMORY;
}
