/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-account-edit.c,v 1.12 2004/11/12 12:13:27 nyoxi Exp $
 */

#include "etpan-account-edit.h"
#include "etpan-subapp.h"
#include "etpan-app.h"
#include "etpan-app-subapp.h"
#include "etpan-errors.h"
#include "etpan-tools.h"
#include "etpan-cfg-storage.h"
#include "etpan-cfg-common.h"
#include <stdlib.h>
#include "etpan-thread-manager.h"
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "etpan-cfg-edit-common.h"
#include "etpan-imf-helper.h"
#include "etpan-mime-tools.h"
#include "etpan-storage-sel.h"
#include "etpan-folder-sel-app.h"
#include "etpan-account-sel.h"
#include "etpan-help-viewer.h"
#include <libetpan/libetpan.h>
#include "etpan-search-input.h"

static void handle_key(struct etpan_subapp * app, int key);
static void display(struct etpan_subapp * app, WINDOW * w);
static void set_color(struct etpan_subapp * app);
static int init(struct etpan_subapp * app);
static void done(struct etpan_subapp * app);
static int display_init(struct etpan_subapp * app);
static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app);

static struct etpan_subapp_driver etpan_account_editor_app_driver = {
  .name = "account-edit",
  .always_handle_key = 0,
  .always_on_top = 1,
  .get_idle_delay = NULL,
  .idle = NULL,
  .set_fd = NULL,
  .handle_fd = NULL,
  .handle_key = handle_key,
  .handle_resize = NULL,
  .display = display,
  .set_color = set_color,
  .init = init,
  .done = done,
  .enter = NULL,
  .leave = leave,
  .display_init = display_init,
  .display_done = NULL,
};

#define INFO_TYPE_MAX 20

enum {
  INFO_TYPE_NAME,
  INFO_TYPE_DISPLAY_NAME,
  INFO_TYPE_ADDRESS,
  INFO_TYPE_SIGNATURE,
#if 0
  INFO_TYPE_SMIME_KEY,
#endif
  INFO_TYPE_PRIVACY,
};

static int info_type_tab[] = {
  INFO_TYPE_NAME,
  INFO_TYPE_DISPLAY_NAME,
  INFO_TYPE_ADDRESS,
  INFO_TYPE_SIGNATURE,
#if 0
  INFO_TYPE_SMIME_KEY,
#endif
  INFO_TYPE_PRIVACY,
};

#define TABLE_SIZE(a) (sizeof(a) / sizeof(a[0]))

#define PRIVACY_NAME_MAX 256

struct app_state {
  int storage_type;
  
  char id[256];
  char display_name[256];
  char address[256];
  char signature[PATH_MAX];
#if 0
  char smime_key[PATH_MAX];
#endif
  char initial_privacy_driver[PRIVACY_NAME_MAX];
  char initial_privacy_encryption[PRIVACY_NAME_MAX];
  char * privacy_driver;
  char * privacy_encryption;
  
  unsigned int index;
  unsigned int first;
  unsigned int count;
  
  int main_attr;
  int selection_attr;
  int status_attr;
  
  /* upcall */
  void (* upcall)(struct etpan_subapp *, int, void *);
  void * upcall_data;
};

static int show_help(struct etpan_subapp * app);

static int get_next_privacy(struct etpan_subapp * app, int direction);

static void ask_quit(struct etpan_subapp * app);

static void handle_key(struct etpan_subapp * app, int key)
{
  int list_lines;
  int current_type;
  struct app_state * state;

  list_lines = app->display_height - 1;
  
  state = app->data;
  
  current_type = info_type_tab[state->index];
  
  switch (key) {
  case KEY_F(1):
  case '?':
    show_help(app);
    break;
    
  case 'k':
  case KEY_UP:
    if (state->index > 0)
      state->index --;
    break;
    
  case 'j':
  case KEY_DOWN:
    if (state->index < state->count - 1)
      state->index ++;
    break;
    
  case KEY_NPAGE:
    if (state->index + list_lines - 1 < state->count)
      state->index += list_lines - 1;
    else
      state->index = state->count - 1;
    break;
    
  case KEY_PPAGE:
    if ((int) state->index >= list_lines - 1)
      state->index -= list_lines - 1;
    else
      state->index = 0;
    break;
    
  case KEY_HOME:
    state->index = 0;
    break;
    
  case KEY_END:
    state->index = state->count - 1;
    break;

  case KEY_CTRL('G'):
    if (state->upcall != NULL)
      state->upcall(app, ETPAN_ACCOUNT_EDIT_CANCEL, state->upcall_data);
    break;

  case 'y':
    if (state->upcall != NULL)
      state->upcall(app, ETPAN_ACCOUNT_EDIT_VALID, state->upcall_data);
    break;

  case 'q':
    ask_quit(app);
    break;
    
  case KEY_LEFT:
    switch (current_type) {
    case INFO_TYPE_PRIVACY:
      get_next_privacy(app, 0);
      break;
    }
    break;
      
  case KEY_RIGHT:
    switch (current_type) {
    case INFO_TYPE_PRIVACY:
      get_next_privacy(app, 1);
      break;
    }
    break;
    
  case '\n':
    switch (current_type) {
    case INFO_TYPE_NAME:
      etpan_cfg_edit_string(app, "Enter identifier: ",
          state->id, sizeof(state->id), 0);
      break;
    case INFO_TYPE_DISPLAY_NAME:
      etpan_cfg_edit_string(app, "Enter display name: ",
          state->display_name, sizeof(state->display_name), 0);
      break;
    case INFO_TYPE_ADDRESS:
      etpan_cfg_edit_string(app, "Enter address: ",
          state->address, sizeof(state->address), 0);
      break;
    case INFO_TYPE_SIGNATURE:
      etpan_cfg_edit_filename(app, "Signature file: ",
          state->signature, sizeof(state->signature));
      break;
#if 0
    case INFO_TYPE_SMIME_KEY:
      etpan_cfg_edit_filename(app, "S/MIME key file: ",
          state->smime_key, sizeof(state->smime_key));
      break;
#endif
    }
    break;
  }
}

static void display(struct etpan_subapp * app, WINDOW * w)
{
  struct app_state * state;
  unsigned int info_type_count;
  unsigned int i;
  int y;
  int list_lines;
  char * buffer;
  char * output;
  char * fill;
  int percent;

  buffer = app->app->buffer;
  output = app->app->output;
  fill = app->app->fill;
  
  list_lines = app->display_height - 1;
  
  state = app->data;
  
  /* update view */
  
  info_type_count = TABLE_SIZE(info_type_tab);
  
  state->count = info_type_count;
  
  if (state->index > state->count - 1)
    state->index = state->count - 1;
  
  if (state->index < state->first)
    state->first = state->index;
  if (state->index - state->first + 1 > (unsigned int) list_lines)
    state->first = state->index - list_lines + 1;
  
  /* display */
  
  wattron(w, state->main_attr);
  y = 0;
  i = state->first;
  while (y < list_lines) {
    char * privacy_name;
    
    if (i > state->count - 1)
      break;
    
    switch (info_type_tab[i]) {
    case INFO_TYPE_NAME:
      snprintf(buffer, app->display_width, "identifier: %s", state->id);
      break;
    case INFO_TYPE_DISPLAY_NAME:
      snprintf(buffer, app->display_width, "display name: %s",
          state->display_name);
      break;
    case INFO_TYPE_ADDRESS:
      snprintf(buffer, app->display_width, "address: %s", state->address);
      break;
    case INFO_TYPE_SIGNATURE:
      if (state->signature[0] == '\0')
        snprintf(buffer, app->display_width, "signature: (no signature)");
      else
        snprintf(buffer, app->display_width, "signature: %s",
            state->signature);
      break;
#if 0
    case INFO_TYPE_SMIME_KEY:
      if (state->smime_key[0] == '\0')
        snprintf(buffer, app->display_width, "S/MIME private key: (no private key)");
      else
        snprintf(buffer, app->display_width, "S/MIME private key: %s",
            state->smime_key);
      break;
#endif
    case INFO_TYPE_PRIVACY:
      if ((state->privacy_driver != NULL) &&
          (state->privacy_encryption != NULL)) {
        privacy_name = mailprivacy_get_encryption_name(app->app->privacy,
            state->privacy_driver, state->privacy_encryption);
      }
      else {
        privacy_name = NULL;
      }
      
      if (privacy_name == NULL)
        snprintf(buffer, app->display_width, "security: (no security option)");
      else
        snprintf(buffer, app->display_width, "security: %s",
            privacy_name);
      break;
    }
    if (i == state->index) {
      wattroff(w, state->main_attr);
      wattron(w, state->selection_attr);
    }
    snprintf(output, app->display_width, "%s%s", buffer, fill);
    mvwprintw(w, y, 0, "%s", output);
    if (i == state->index) {
      wattroff(w, state->selection_attr);
      wattron(w, state->main_attr);
    }
    
    i ++;
    y ++;
  }

  while (y < list_lines) {
    mvwprintw(w, y, 0, "%s", fill);
    y ++;
  }

  wattroff(w, state->main_attr);
  
  /* status bar */

  wattron(w, state->status_attr);
  
  if (state->count == 0)
    percent = 0;
  if (state->count == 1)
	percent = 100;
  else
    percent = state->index * 100 / (state->count-1);
  
  snprintf(output, app->display_width + 1,
      " %3i %% | y: ok  ^G: cancel%s", percent, fill);
  
  mvwprintw(w, app->display_height - 1, 0, "%s", output);
  
  wattroff(w, state->status_attr);
}

static void set_color(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_app_set_color(app->app, "main",
      &state->main_attr, A_NORMAL);
  etpan_app_set_color(app->app, "selection",
      &state->selection_attr, A_REVERSE);
  etpan_app_set_color(app->app, "status",
      &state->status_attr, A_REVERSE);
}

static int init(struct etpan_subapp * app)
{
  struct app_state * state;

  state = malloc(sizeof(* state));
  if (state == NULL)
    goto err;
  
  state->id[0] = '\0';
  
  state->display_name[0] = '\0';
  state->address[0] = '\0';
  snprintf(state->signature, sizeof(state->signature), "%s/%s",
      etpan_get_home_dir(), ".signature");
#if 0
  state->smime_key[0] = '\0';
#endif
  state->initial_privacy_driver[0] = '\0';
  state->initial_privacy_encryption[0] = '\0';
  state->privacy_driver = NULL;
  state->privacy_encryption = NULL;
  
  state->index = 0;
  state->first = 0;
  state->count = 0;
  
  /* colors */
  state->main_attr = A_NORMAL;
  state->selection_attr = A_REVERSE;
  state->status_attr = A_REVERSE;
  
  state->upcall = NULL;
  state->upcall_data = NULL;
  
  app->data = state;
  
  return NO_ERROR;
  
 err:
  return ERROR_MEMORY;
}

static void done(struct etpan_subapp * app)
{
  struct app_state * state;

  state = app->data;
  
  free(state);
}

static int display_init(struct etpan_subapp * app)
{
  etpan_subapp_set_title(app, "etPan! - edit account");
  return etpan_app_subapp_display_init(app);
}


struct etpan_subapp * etpan_account_editor_app_new(struct etpan_app * app)
{
  return etpan_subapp_new(app, &etpan_account_editor_app_driver);
}

void etpan_account_editor_app_flush(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  state->id[0] = '\0';
  
  state->display_name[0] = '\0';
  state->address[0] = '\0';
  snprintf(state->signature, sizeof(state->signature), "%s/%s",
      etpan_get_home_dir(), ".signature");
#if 0
  state->smime_key[0] = '\0';
#endif
  state->initial_privacy_driver[0] = '\0';
  state->initial_privacy_encryption[0] = '\0';
  state->privacy_driver = NULL;
  state->privacy_encryption = NULL;
  
  state->index = 0;
  state->first = 0;
  state->count = 0;
  
  state->upcall = NULL;
  state->upcall_data = NULL;
}

static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app)
{
  etpan_account_editor_app_flush(app);
}

void etpan_account_editor_app_set(struct etpan_subapp * app,
    struct etpan_account_info * account,
    void (* upcall)(struct etpan_subapp *, int, void *),
    void * upcall_data)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_account_editor_app_flush(app);
  
  state->upcall = upcall;
  state->upcall_data = upcall_data;
  
  if (account == NULL)
    return;
  
  if (account->id != NULL) {
    strncpy(state->id, account->id, sizeof(state->id));
    state->id[sizeof(state->id) - 1] = '\0';
  }

  if (account->name != NULL) {
    char * decoded;
    
    decoded = etpan_decode_mime_header(app->app, account->name);
    if (decoded != NULL) {
      strncpy(state->display_name, decoded,
          sizeof(state->display_name));
      free(decoded);
    }
    else {
      strncpy(state->display_name, account->name,
          sizeof(state->display_name));
    }
    state->display_name[sizeof(state->display_name) - 1] = '\0';
  }

  if (account->addr != NULL) {
    strncpy(state->address, account->addr,
        sizeof(state->address));
    state->address[sizeof(state->address) - 1] = '\0';
  }
  
  if (account->signature != NULL) {
    strncpy(state->signature, account->signature,
        sizeof(state->signature));
    state->signature[sizeof(state->signature) - 1] = '\0';
  }

#if 0
  if (account->smime_key != NULL) {
    strncpy(state->smime_key, account->smime_key,
        sizeof(state->smime_key));
    state->smime_key[sizeof(state->smime_key) - 1] = '\0';
  }
#endif

  if (account->privacy_driver != NULL) {
    strncpy(state->initial_privacy_driver, account->privacy_driver,
        sizeof(state->initial_privacy_driver));
    state->initial_privacy_driver[sizeof(state->initial_privacy_driver) - 1] = '\0';
    state->privacy_driver = state->initial_privacy_driver;
  }

  if (account->privacy_encryption != NULL) {
    strncpy(state->initial_privacy_encryption, account->privacy_encryption,
        sizeof(state->initial_privacy_encryption));
    state->initial_privacy_encryption[sizeof(state->initial_privacy_encryption) - 1] = '\0';
    state->privacy_encryption = state->initial_privacy_encryption;
  }
}



struct etpan_account_info *
etpan_account_editor_app_get_account(struct etpan_subapp * app)
{
  struct app_state * state;
  char * id;
  char * name;
  char * addr;
  char * signature;
#if 0
  char * smime_key;
#endif
  char * privacy_driver;
  char * privacy_encryption;
  struct etpan_account_info * account;
  
  state = app->data;
  
  id = NULL;
  name = NULL;
  addr = NULL;
  signature = NULL;
#if 0
  smime_key = NULL;
#endif
  privacy_driver = NULL;
  privacy_encryption = NULL;
  
  id = strdup(state->id);
  if (id == NULL) {
    goto err;
  }
  
  if (state->display_name[0] != '\0') {
    name = etpan_encode_mime_header(app->app, state->display_name);
    if (name == NULL) {
      goto free_id;
    }
  }
  
  if (state->address[0] != '\0') {
    addr = strdup(state->address);
    if (addr == NULL) {
      goto free_name;
    }
  }
  
  if (state->signature[0] != '\0') {
    signature = strdup(state->signature);
    if (signature == NULL) {
      goto free_addr;
    }
  }
  
#if 0
  if (state->smime_key[0] != '\0') {
    smime_key = strdup(state->smime_key);
    if (smime_key == NULL) {
      goto free_signature;
    }
  }
#endif
  
  if (state->privacy_driver != NULL) {
    privacy_driver = strdup(state->privacy_driver);
    if (privacy_driver == NULL) {
      goto free_signature;
    }
  }
  
  if (state->privacy_encryption != NULL) {
    privacy_encryption = strdup(state->privacy_encryption);
    if (privacy_encryption == NULL) {
      goto free_privacy_driver;
    }
  }
  
  account = etpan_account_info_new(id, name, addr,
      signature, privacy_driver, privacy_encryption);
  if (account == NULL) {
    goto free_privacy_encryption;
  }
  
  return account;
  
 free_privacy_encryption:
  free(privacy_encryption);
 free_privacy_driver:
  free(privacy_driver);
#if 0
 free_smime_key:
  free(smime_key);
#endif
 free_signature:
  free(signature);
 free_addr:
  free(addr);
 free_name:
  free(name);
 free_id: 
  free(id);
 err:
  return NULL;
}


static int get_next_privacy(struct etpan_subapp * app, int direction)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_privacy_encrypt_next(app->app->privacy,
      state->privacy_driver, state->privacy_encryption,
      &state->privacy_driver, &state->privacy_encryption, direction);
  
  return NO_ERROR;
}


static void ask_quit_upcall(struct etpan_subapp * input_app,
    int valid, void * data)
{
  struct app_state * state;
  char * name;
  char * dup_name;
  struct etpan_subapp * app;
  int r;
  char * result;
  int do_quit;

  app = data;
  state = app->data;
  
  if (valid == ETPAN_INPUT_COMMON_CANCEL) {
    etpan_app_quit_subapp(input_app);
    return;
  }
  
  result = etpan_search_input_get_value(input_app);
  if ((* result != 'y') && (* result != 'n')) {
    etpan_app_quit_subapp(input_app);
    return;
  }
  
  do_quit = (* result == 'y');
  etpan_app_quit_subapp(input_app);
  
  if (do_quit) {
    if (state->upcall != NULL)
      state->upcall(app, ETPAN_ACCOUNT_EDIT_VALID, state->upcall_data);
  }
  else {
    if (state->upcall != NULL)
      state->upcall(app, ETPAN_ACCOUNT_EDIT_CANCEL, state->upcall_data);
  }
}


static void ask_quit(struct etpan_subapp * app)
{
  struct app_state * state;
  struct etpan_subapp * input;
  int r;
  
  input = etpan_app_find_subapp(app->app, "search-input",
      0, NULL, NULL);
  if (input == NULL) {
    input = etpan_search_input_new(app->app);
    if (input == NULL)
      goto err;
  }
  
  r = etpan_search_input_set(input,
      "save the account (y/n) ? ", 1,
      NULL, 0,
      ask_quit_upcall, app);
  if (input == NULL)
    goto err;
  
  etpan_subapp_set_parent(input, app);
  etpan_app_switch_subapp(input, 0);
  
  return;
  
 err:
  ETPAN_APP_LOG((app->app, "configuration - not enough memory"));
}


#define HELP_TEXT \
"\
Help for account editor\n\
-----------------------\n\
\n\
This application will let you edit the selected account.\n\
\n\
description\n\
-----------\n\
\n\
- Identifier of the account is the string that will be displayed in\n\
the account selection dialog.\n\
\n\
- Display name is the name that will be displayed to your recipient\n\
in the From line.\n\
\n\
- Address is your e-mail address.\n\
\n\
- Signature is the file that will contain the signature.\n\
\n\
- S/MIME key is the file that contains your S/MIME private key.\n\
\n\
- Privacy driver and encryption tells which default encryption to use when\n\
sending the message,\n\
\n\
\n\
keys\n\
----\n\
\n\
- arrow keys       : move cursor\n\
\n\
- y                : finished\n\
- Ctrl-G           : cancel\n\
\n\
- [Enter]          : edit given information\n\
\n\
- ?                : help\n\
- Ctrl-L           : Console log\n\
\n\
(? or q to exit help)\n\
"

static int show_help(struct etpan_subapp * app)
{
  return etpan_show_help(app, HELP_TEXT, sizeof(HELP_TEXT) - 1);
}
