/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-cfg-local-abook.c,v 1.1 2004/03/18 01:32:02 hoa Exp $
 */

#include "etpan-cfg-local-abook.h"

#include <stdlib.h>
#include <string.h>

#include <libetpan/clist.h>
#include <libetpan/charconv.h>

#include "etpan-db-read.h"
#include "etpan-errors.h"

static struct etpan_local_abook_entry *
etpan_local_abook_entry_new(char * name, char * nick, char * addr)
{
  struct etpan_local_abook_entry * entry;
  
  entry = malloc(sizeof(struct etpan_local_abook_entry));
  if (entry == NULL)
    return NULL;
  
  entry->name = name;
  entry->nick = nick;
  entry->addr = addr;

  return entry;
}

static void
etpan_local_abook_entry_free(struct etpan_local_abook_entry * entry)
{
  if (entry->addr != NULL)
    free(entry->addr);
  if (entry->nick != NULL)
    free(entry->nick);
  if (entry->name != NULL)
    free(entry->name);
  free(entry);
}

struct etpan_local_abook_config *
etpan_local_abook_config_new(struct etpan_global_config * global_config,
    clist * entry_list, clist * list_list)
{
  struct etpan_local_abook_config * config;

  config = malloc(sizeof(struct etpan_local_abook_config));
  if (config == NULL)
    goto err;

  config->global_config = global_config;

  config->entry_list = entry_list;
  config->list_list = list_list;

  return config;

 err:
  return NULL;
}

void etpan_local_abook_config_free(struct etpan_local_abook_config * config)
{
  clist_free(config->list_list);
  clist_foreach(config->entry_list,
      (clist_func) etpan_local_abook_entry_free, NULL);
  clist_free(config->entry_list);
  free(config);
}

static int abook_entry_read(struct etpan_global_config * global_config,
    char * current_charset,
    chash * entry,
    struct etpan_local_abook_entry ** result)
{
  char * name;
  char * nick;
  char * addr;
  struct etpan_local_abook_entry * abook_entry;
  int res;
  int r;
  
  name = etpan_db_entry_get_value(entry, "name");
  nick = etpan_db_entry_get_value(entry, "nick");
  addr = etpan_db_entry_get_value(entry, "addr");
  
  if ((name == NULL) && (nick != NULL) && (addr != NULL)) {
    res = ERROR_INVAL;
    goto err;
  }
  
  if (name != NULL) {
    r = charconv(global_config->display_charset,
        current_charset, name, strlen(name), &name);
    if (r != MAIL_CHARCONV_NO_ERROR) {
      res = ERROR_PARSE;
      goto err;
    }
  }
  
  if (nick != NULL) {
    r = charconv(global_config->display_charset,
        current_charset, nick, strlen(nick), &nick);
    if (r != MAIL_CHARCONV_NO_ERROR) {
      res = ERROR_PARSE;
      goto free_name;
    }
  }
  
  if (addr != NULL) {
    addr = strdup(addr);
    if (addr == NULL) {
      res = ERROR_MEMORY;
      goto free_nick;
    }
  }
  
  abook_entry = etpan_local_abook_entry_new(name, nick, addr);
  if (entry == NULL) {
    res = ERROR_MEMORY;
    goto free_addr;
  }
  
  * result = abook_entry;

  return NO_ERROR;
  
 free_addr:
  free(addr);
 free_nick:
  free(nick);
 free_name:
  free(name);
 err:
  return res;
}


int etpan_local_abook_config_read(struct etpan_global_config * global_config,
    char * filename,
    struct etpan_local_abook_config ** result)
{
  struct etpan_db * db;
  char * current_charset;
  int r;
  int res;
  clist * list;
  clist * list_list;
  struct etpan_local_abook_config * abook_config;
  unsigned int i;
  
  r = etpan_read_config(filename, &db);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }
  
  list = clist_new();
  if (list == NULL) {
    res = ERROR_MEMORY;
    goto free_db;
  }

  list_list = clist_new();
  if (list_list == NULL) {
    res = ERROR_MEMORY;
    goto free_list;
  }
  
  current_charset = "utf-8";
  
  for(i = 0 ; i < carray_count(db->data) ; i ++) {
    chash * entry;
    char * id;
    
    entry = carray_get(db->data, i);
    if (entry == NULL)
      continue;
    
    id = etpan_db_entry_get_value(entry, "id");
    if (strcasecmp(id, "default") != 0)
      continue;
    
    current_charset = etpan_db_entry_get_value(entry, "charset");
    
    break;
  }
  
  for(i = 0 ; i < db->data->len ; i ++) {
    chash * entry;
    struct etpan_local_abook_entry * abook_entry;

    entry = carray_get(db->data, i);
    if (entry == NULL)
      continue;
    
    r = abook_entry_read(global_config, current_charset,
        entry, &abook_entry);
    if (r != NO_ERROR) 
      continue;
    
    r = clist_append(list, abook_entry);
    if (r < 0) {
      etpan_local_abook_entry_free(abook_entry);
      res = ERROR_MEMORY;
      goto free_ll;
    }
  }
  
  abook_config = etpan_local_abook_config_new(global_config, list, list_list);
  if (abook_config == NULL) {
    res = ERROR_MEMORY;
    goto free_ll;
  }

  * result = abook_config;
  
  etpan_db_free(db);
  
  return NO_ERROR;

 free_ll:
  clist_free(list_list);
 free_list:
  clist_foreach(list, (clist_func) etpan_local_abook_entry_free, NULL);
  clist_free(list);
 free_db:
  etpan_db_free(db);
 err:
  return res;
}
