/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-mime-edit-app.c,v 1.30 2004/11/12 13:57:02 hoa Exp $
 */

#include "etpan-mime-edit-app.h"

#include "etpan-app-subapp.h"
#include "etpan-subapp.h"
#include "etpan-subapp-thread.h"
#include "etpan-errors.h"
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ncurses.h>
#include <libetpan/libetpan.h>
#include "etpan-msg-renderer.h"
#include "etpan-part-viewer-app.h"
#include "etpan-mime-params.h"
#include "etpan-tools.h"
#include "etpan-imf-helper.h"
#include "etpan-header-editor-app.h"
#include "etpan-mime-common.h"
#include "etpan-filename-input.h"
#include "etpan-mime-tools.h"
#include <libetpan/libetpan.h>
#include "etpan-msg-new.h"
#include "etpan-folder-sel-app.h"
#include "etpan-help-viewer.h"
#include "etpan-search-input.h"

static void set_fd(struct etpan_subapp * app, fd_set * fds, int * maxfd);
static void handle_fd(struct etpan_subapp * app, fd_set * fds);
static void handle_key(struct etpan_subapp * app, int key);
static void display(struct etpan_subapp * app, WINDOW * w);
static void set_color(struct etpan_subapp * app);
static int init(struct etpan_subapp * subapp);
static void done(struct etpan_subapp * subapp);
static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app);
static int display_init(struct etpan_subapp * app);

static struct etpan_subapp_driver etpan_mime_edit_app_driver = {
  .name = "mime-edit",
  .always_handle_key = 0,
  .always_on_top = 0,
  .get_idle_delay = NULL,
  .idle = NULL,
  .set_fd = set_fd,
  .handle_fd = handle_fd,
  .handle_key = handle_key,
  .handle_resize = NULL,
  .display = display,
  .set_color = set_color,
  .init = init,
  .done = done,
  .enter = NULL,
  .leave = leave,
  .display_init = display_init,
  .display_done = NULL,
};

struct app_state {
  struct etpan_mime_common_app_state common_state;
  struct mailfolder * post_from_folder;
  void (* action)(void *);
  void * action_arg;
  void (* cancel_action)(void *);

  int auto_privacy;
};


void clear_auto_privacy(struct etpan_subapp * app);

void set_auto_privacy(struct etpan_subapp * app);


static void set_fd(struct etpan_subapp * app,
    fd_set * fds, int * maxfd)
{
  struct app_state * state;
  
  state = app->data;

  etpan_subapp_thread_set_fd(app, fds, maxfd);
  etpan_mime_common_set_fd(app, &state->common_state, fds, maxfd);
}


/*
  fetch_for_send_callback()
  
  WARNING : twice calls to RESULT_FREE but in two different cases
*/

static void fetch_for_send_callback(struct etpan_subapp * app,
        struct etpan_thread_op * op, int app_op_type, void * data)
{
  struct app_state * state;
  struct etpan_message_fetch_result * fetch_result;
  FILE * f;
  char filename[PATH_MAX];
  mailmessage * msg;
  size_t write_len;
  int r;

  state = app->data;
  
  if (op->err != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "message send - error while getting message"));
    return;
  }
  
  fetch_result = op->result;
  
  /* write result to a file */
  f = etpan_get_tmp_file(filename, sizeof(filename));
  if (f == NULL) {
    ETPAN_APP_LOG((app->app, "message send - error writing file"));
    goto exit;
  }
  
  write_len = fwrite(fetch_result->content, 1, fetch_result->len, f);
  if (write_len != fetch_result->len) {
    ETPAN_APP_LOG((app->app, "message send - error writing file"));
    goto close;
  }
  
  fclose(f);
  
  /*
    this must precede leave so that message result free operation is done
    before leaving (the message is freed when switching app to quit)
  */
  r = etpan_subapp_thread_op_add(app, THREAD_ID_MIMEEDIT_FETCH_FOR_SEND_FREE,
      ETPAN_THREAD_MESSAGE_FETCH_RESULT_FREE,
      op->data.mailaccess.storage, op->data.mailaccess.folder,
      op->data.mailaccess.msg, NULL,
      fetch_result->content,
      NULL, NULL, NULL);
  if (r != NO_ERROR) {
    /* show the warning */
    ETPAN_APP_LOG((app->app, "message send - not enough memory"));
  }
  free(fetch_result);
  
  /* check if the message has a news post */
  msg = etpan_mime_common_get_msg(app, &state->common_state);
  if (msg->msg_mime->mm_type != MAILMIME_MESSAGE) {
    ETPAN_APP_DEBUG((app->app, "BUG detected - MIME structure of message is not of type MAILMIME_MESSAGE"));
    return;
  }
  
  r = etpan_message_send(app, state->post_from_folder,
      msg->msg_mime->mm_data.mm_message.mm_fields, filename);
  if (r != NO_ERROR) {
    /*
      leave file "filename" so that the user can get his file
      when this operation failed
    */
    ETPAN_APP_LOG((app->app, "message send - failed"));
    return;
  }
  
  ETPAN_APP_LOG((app->app, "message queued"));
  
  unlink(filename);
  
  if (state->action != NULL)
    state->action(state->action_arg);
  
  etpan_app_quit_subapp(app);
  
  return;
  
 close:
  fclose(f);
  unlink(filename);
 exit:
  r = etpan_subapp_thread_op_add(app, THREAD_ID_MIMEEDIT_FETCH_FOR_SEND_FREE,
      ETPAN_THREAD_MESSAGE_FETCH_RESULT_FREE,
      op->data.mailaccess.storage, op->data.mailaccess.folder,
      op->data.mailaccess.msg, NULL,
      fetch_result->content,
      NULL, NULL, NULL);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "message send - not enough memory"));
  }
  free(fetch_result);
}

static void handle_fd(struct etpan_subapp * app, fd_set * fds)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_subapp_thread_handle_fd(app, fds);
  
  etpan_mime_common_handle_fd(app, &state->common_state, fds);
}


static int attach_file(struct etpan_subapp * app);

static int attach_text(struct etpan_subapp * app);

static int attach_multipart(struct etpan_subapp * app);

static int attach_message(struct etpan_subapp * app);

static int delete_part(struct etpan_subapp * app);

static int send_message(struct etpan_subapp * app);

static int postpone_message(struct etpan_subapp * app);

static int get_next_privacy(struct etpan_subapp * app);

static int etpan_edit_headers(struct etpan_subapp * app,
    struct mailmime * mime);

static void ask_quit(struct etpan_subapp * app);

static int show_help(struct etpan_subapp * app);

static void handle_key(struct etpan_subapp * app,
    int key)
{
  struct app_state * state;
  struct mailmime * mime;
  mailmessage * msg;
  int r;
  
  state = app->data;
  
  etpan_mime_common_handle_key(app, &state->common_state, key);
  
  switch (key) {
  case KEY_F(1):
  case '?':
    show_help(app);
    break;
    
  case 'y':
    send_message(app);
    break;

  case 'p':
    postpone_message(app);
    break;
    
  case 'e':
    mime = etpan_mime_common_get_selected_mime(app, &state->common_state);
    msg = etpan_mime_common_get_msg(app, &state->common_state);
    
    if (mime->mm_type == MAILMIME_MESSAGE)
      etpan_edit_headers(app, mime);
    else if (mime->mm_type == MAILMIME_SINGLE) {
      if (mime->mm_data.mm_single != NULL) {
        struct mailmime_data * data;
        
        data = mime->mm_data.mm_single;
        if ((data->dt_type == MAILMIME_DATA_FILE) &&
            etpan_mime_is_text(mime)) {
          r = etpan_edit_file(app->app, data->dt_data.dt_filename, 1, NULL);
          if (r == NO_ERROR) {
            etpan_set_text_content_type(app->app, mime);
          }
        }
        else {
          ETPAN_APP_LOG((app->app, "edit file - this is not a text part"));
        }
      }
    }
    break;
    
  case 'A':
    /* attach file */
    attach_file(app);
    break;

  case 'T':
    /* add text part */
    attach_text(app);
    break;
    
  case 'M':
    /* add multipart */
    attach_multipart(app);
    break;

  case KEY_CTRL('m'):
    /* add message */
    attach_message(app);
    break;

  case 'd':
    /* delete part */
    delete_part(app);
    break;

  case 'P':
    /* set privacy option of this part */
    get_next_privacy(app);
    break;
    
  case 'q':
    ask_quit(app);
    break;
    
  case KEY_CTRL('g'):
    if (state->cancel_action != NULL)
      state->cancel_action(state->action_arg);
    etpan_app_quit_subapp(app);
    break;
  }
}

static int display_init(struct etpan_subapp * app)
{
  etpan_subapp_set_title(app, "etPan! - MIME message editor");
  return etpan_app_subapp_display_init(app);
}

static void set_color(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_mime_common_set_color(app, &state->common_state);
}

static void display(struct etpan_subapp * app, WINDOW * w)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_mime_common_display(app, &state->common_state, w,
      "y: send  p: postpone  q: quit");
}

static int init(struct etpan_subapp * subapp)
{
  struct app_state * state;
  int r;
  
  state = malloc(sizeof(* state));
  if (state == NULL)
    goto err;
  
  r = etpan_mime_common_init(&state->common_state);
  if (r != NO_ERROR)
    goto free;
  
  state->post_from_folder = NULL;

  state->action = NULL;
  state->action_arg = NULL;
  state->cancel_action = NULL;

  state->auto_privacy = 0;
  
  subapp->data = state;
  
  return NO_ERROR;
  
 free:
  free(state);
 err:
  return ERROR_MEMORY;
}

static void done(struct etpan_subapp * subapp)
{
  struct app_state * state;
  
  state = subapp->data;
  
  etpan_mime_common_done(subapp, &state->common_state);
  
  free(state);
}

static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app)
{
  struct app_state * state;
  mailmessage * msg;
  struct mailfolder * folder;
  
  state = app->data;
  
  etpan_subapp_thread_cancel_all(app);
  
  msg = etpan_mime_common_get_msg(app, &state->common_state);
  folder = etpan_mime_common_get_folder(app, &state->common_state);
  
  /* message is unref before freeing it */
  etpan_mime_common_leave(app, &state->common_state, new_app);
  
  if (msg != NULL) {
    etpan_message_mime_clear(msg->msg_mime);
    
    etpan_queue_unref_msg(app, msg);
#if 0
    etpan_subapp_thread_folder_op_add(app, THREAD_ID_MIMEEDIT_FREE_MSG,
        ETPAN_THREAD_MESSAGE_FREE,
        folder,
        msg, NULL,
        NULL,
        NULL, NULL, NULL);
#endif
  }
  
  state->post_from_folder = NULL;
  
  state->action = NULL;
  state->action_arg = NULL;
  state->cancel_action = NULL;
  
  state->auto_privacy = 0;
}

struct etpan_subapp *
etpan_mime_edit_app_new(struct etpan_app * app)
{
  return etpan_subapp_new(app, &etpan_mime_edit_app_driver);
}

int etpan_mime_edit_app_set_msg(struct etpan_subapp * app,
    mailmessage * msg,
    struct mailfolder * post_from_folder,
    void (* action)(void *), void * action_arg,
    void (* cancel_action)(void *))
{
  struct app_state * state;
  int r;
  
  state = app->data;
  state->post_from_folder = post_from_folder;
  
  state->action = action;
  state->action_arg = action_arg;
  state->cancel_action = cancel_action;
  
  state->auto_privacy = 1;
  
  etpan_subapp_thread_cancel_all(app);
  
  r = etpan_mime_common_set_msg(app, &state->common_state, NULL, msg);
  if (r != NO_ERROR)
    return r;
  
  set_auto_privacy(app);
  
  return NO_ERROR;
}

mailmessage * etpan_mime_edit_app_get_msg(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  return etpan_mime_common_get_msg(app, &state->common_state);
}

struct mailfolder * etpan_mime_edit_app_get_folder(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  return etpan_mime_common_get_folder(app, &state->common_state);
}

static void attach_file_upcall(struct etpan_subapp * input_app, int valid,
    void * data);

static int attach_file(struct etpan_subapp * app)
{
  char current_dir[PATH_MAX];
  char * default_filename;
  struct etpan_subapp * input_app;
  int r;
  struct mailmime * parent_mime;
  struct app_state * state;
  
  state = app->data;
  
  parent_mime =
    etpan_mime_common_get_selected_mime(app, &state->common_state);
  if (parent_mime == NULL) {
    ETPAN_APP_LOG((app->app, "attach file - invalid"));
    return ERROR_INVAL;
  }
  
  if ((parent_mime->mm_type != MAILMIME_MESSAGE) &&
      (parent_mime->mm_type != MAILMIME_MULTIPLE)) {
    ETPAN_APP_LOG((app->app, "attach file - not a multipart or a message part"));
    return ERROR_INVAL;
  }
  
  input_app = etpan_app_find_subapp(app->app, "filename-input",
    0, NULL, NULL);
  if (input_app == NULL) {
    input_app = etpan_filename_input_new(app->app);
    if (input_app == NULL) {
      ETPAN_APP_LOG((app->app, "attach file - not enough memory"));
      return ERROR_MEMORY;
    }
  }
  
  default_filename = NULL;
  
  if (getcwd(current_dir, sizeof(current_dir)) != NULL) {
    size_t len;
    
    len = strlen(current_dir);
    if (current_dir[len - 1] != '/') {
      if (len + 1 < sizeof(current_dir) - 1) {
        current_dir[len] = '/';
        current_dir[len + 1] = '\0';
      }
    }
    
    default_filename = current_dir;
  }
  
  etpan_subapp_set_parent(input_app, app);
  
  r = etpan_filename_input_set(input_app,
      "filename: ", 256, default_filename,
      attach_file_upcall, app);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "attach file - not enough memory"));
    return ERROR_MEMORY;
  }
  
  ETPAN_APP_LOG((app->app, "attach file - switch to miniapp"));
  etpan_app_switch_subapp(input_app, 0);
  
  return NO_ERROR;
}

static void attach_file_upcall(struct etpan_subapp * input_app, int valid,
    void * data)
{
  struct etpan_subapp * app;
  struct app_state * state;
  struct mailmime * parent_mime;
  struct mailmime * mime;
  char * val;
  int r;
  mailmessage * msg;
  int fd;
  
  app = data;
  state = app->data;
  
  if (valid == ETPAN_INPUT_COMMON_CANCEL) {
    ETPAN_APP_LOG((app->app, "attach file - cancel"));
    goto err;
  }
  
  parent_mime =
    etpan_mime_common_get_selected_mime(app, &state->common_state);
  if (parent_mime == NULL) {
    ETPAN_APP_LOG((app->app, "attach file - invalid"));
    goto err;
  }
  
  val = etpan_filename_input_get_value(input_app);
  if (val == NULL) {
    ETPAN_APP_LOG((app->app, "attach file - BUG detected"));
    goto err;
  }
  
  fd = open(val, O_RDONLY);
  if (fd < 0) {
    ETPAN_APP_LOG((app->app, "attach file - file not found"));
    goto err;
  }
  close(fd);
  
  mime = etpan_mime_new_file_part(app->app, 1, val, NULL, -1);
  if (mime == NULL) {
    ETPAN_APP_LOG((app->app, "attach file - not enough memory"));
    goto err;
  }
  
  r = mailmime_smart_add_part(parent_mime, mime);
  if (r != MAILIMF_NO_ERROR) {
    ETPAN_APP_LOG((app->app, "attach file - not enough memory"));
    goto free_mime;
  }
  
  msg = etpan_mime_common_get_msg(app, &state->common_state);
  r = etpan_mime_common_set_msg(app, &state->common_state, NULL, msg);
  /* ignore error */
  
  etpan_app_quit_subapp(input_app);
  
  return;

 free_mime:
  etpan_message_mime_clear(mime);
  mailmime_free(mime);
 err:
  etpan_app_quit_subapp(input_app);
}

static int attach_text(struct etpan_subapp * app)
{
  char filename[PATH_MAX];
  FILE * f;
  int res;
  struct mailmime * mime;
  int r;
  struct app_state * state;
  struct mailmime * parent_mime;
  
  state = app->data;
  
  parent_mime =
    etpan_mime_common_get_selected_mime(app, &state->common_state);
  if (parent_mime == NULL) {
    ETPAN_APP_LOG((app->app, "attach file - invalid"));
    res = ERROR_INVAL;
    goto err;
  }
  
  if ((parent_mime->mm_type != MAILMIME_MESSAGE) &&
      (parent_mime->mm_type != MAILMIME_MULTIPLE)) {
    ETPAN_APP_LOG((app->app, "attach file - not a multipart or a message part"));
    return ERROR_INVAL;
  }
  
  f = etpan_get_mime_tmp_file(filename, sizeof(filename));
  if (f == NULL) {
    res = ERROR_FILE;
    goto err;
  }
  fclose(f);
  
  mime = etpan_mime_new_file_part(app->app, 0, filename,
      "text/plain", MAILMIME_MECHANISM_8BIT);
  if (mime == NULL) {
    ETPAN_APP_LOG((app->app, "attach file - not enough memory"));
    res = ERROR_MEMORY;
    goto unlink;
  }
  
  r = mailmime_smart_add_part(parent_mime, mime);
  if (r != MAILIMF_NO_ERROR) {
    ETPAN_APP_LOG((app->app, "attach file - not enough memory"));
    res = ERROR_MEMORY;
    goto free_mime;
  }
  
  unlink(filename);
  
  clear_auto_privacy(app);
  etpan_mime_common_display_update(app, &state->common_state);
  set_auto_privacy(app);
  
  return NO_ERROR;
  
 free_mime:
  etpan_message_mime_clear(mime);
  mailmime_free(mime);
 unlink:
  unlink(filename);
 err:
  return res;
}

static int attach_multipart(struct etpan_subapp * app)
{
  struct app_state * state;
  struct mailmime * parent_mime;
  int res;
  int r;
  struct mailmime * mime;
  
  state = app->data;
  
  parent_mime =
    etpan_mime_common_get_selected_mime(app, &state->common_state);
  if (parent_mime == NULL) {
    ETPAN_APP_LOG((app->app, "attach file - invalid"));
    res = ERROR_INVAL;
    goto err;
  }
  
  if ((parent_mime->mm_type != MAILMIME_MESSAGE) &&
      (parent_mime->mm_type != MAILMIME_MULTIPLE)) {
    ETPAN_APP_LOG((app->app, "attach file - not a multipart or a message part"));
    return ERROR_INVAL;
  }
  
  mime = etpan_mime_new_file_part(app->app, 0, NULL,
      "multipart/mixed", MAILMIME_MECHANISM_8BIT);
  if (mime == NULL) {
    ETPAN_APP_LOG((app->app, "attach file - not enough memory"));
    res = ERROR_MEMORY;
    goto err;
  }
  
  r = mailmime_smart_add_part(parent_mime, mime);
  if (r != MAILIMF_NO_ERROR) {
    ETPAN_APP_LOG((app->app, "attach file - not enough memory"));
    res = ERROR_MEMORY;
    goto free_mime;
  }
  
  clear_auto_privacy(app);
  etpan_mime_common_display_update(app, &state->common_state);
  set_auto_privacy(app);
  
  return NO_ERROR;
  
 free_mime:
  etpan_message_mime_clear(mime);
  mailmime_free(mime);
 err:
  return res;
}

static int attach_message(struct etpan_subapp * app)
{
  struct app_state * state;
  struct mailmime * parent_mime;
  int res;
  int r;
  struct mailmime * mime;
  struct mailimf_fields * fields;
  
  state = app->data;
  
  parent_mime =
    etpan_mime_common_get_selected_mime(app, &state->common_state);
  if (parent_mime == NULL) {
    ETPAN_APP_LOG((app->app, "attach file - invalid"));
    res = ERROR_INVAL;
    goto err;
  }
  
  if ((parent_mime->mm_type != MAILMIME_MESSAGE) &&
      (parent_mime->mm_type != MAILMIME_MULTIPLE)) {
    ETPAN_APP_LOG((app->app, "attach file - not a multipart or a message part"));
    return ERROR_INVAL;
  }
  
  mime = etpan_mime_new_file_part(app->app, 0, NULL,
      "message/rfc822", MAILMIME_MECHANISM_8BIT);
  if (mime == NULL) {
    ETPAN_APP_LOG((app->app, "attach file - not enough memory"));
    res = ERROR_MEMORY;
    goto err;
  }

  r = etpan_new_get_fields(app->app,
      state->post_from_folder, NULL, &fields);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "attach file - not enough memory"));
    res = ERROR_MEMORY;
    goto free_mime;
  }
  mailmime_set_imf_fields(mime, fields);
  
  r = mailmime_smart_add_part(parent_mime, mime);
  if (r != MAILIMF_NO_ERROR) {
    ETPAN_APP_LOG((app->app, "attach file - not enough memory"));
    res = ERROR_MEMORY;
    goto free_mime;
  }
  
  clear_auto_privacy(app);
  etpan_mime_common_display_update(app, &state->common_state);
  set_auto_privacy(app);
  
  return NO_ERROR;
  
 free_mime:
  etpan_message_mime_clear(mime);
  mailmime_free(mime);
 err:
  return res;
}

static int delete_part(struct etpan_subapp * app)
{
  struct mailmime * mime;
  mailmessage * msg;
  struct app_state * state;
  int res;

  state = app->data;
  
  mime =
    etpan_mime_common_get_selected_mime(app, &state->common_state);
  if (mime == NULL) {
    ETPAN_APP_LOG((app->app, "remove part - invalid"));
    res = ERROR_INVAL;
    goto err;
  }
  
  msg = etpan_mime_common_get_msg(app, &state->common_state);
  if (msg->msg_mime == mime) {
    ETPAN_APP_LOG((app->app, "remove part - could not remove main mime part"));
    res = ERROR_INVAL;
    goto err;
  }
  
  switch (mime->mm_type) {
  case MAILMIME_MULTIPLE:
    if (clist_begin(mime->mm_data.mm_multipart.mm_mp_list) != NULL) {
      ETPAN_APP_LOG((app->app, "remove part - mime part not empty"));
      res = ERROR_INVAL;
      goto err;
    }
    break;
    
  case MAILMIME_MESSAGE:
    if (mime->mm_data.mm_message.mm_msg_mime != NULL) {
      ETPAN_APP_LOG((app->app, "remove part - mime part not empty"));
      res = ERROR_INVAL;
      goto err;
    }
    break;
  }
  
  etpan_message_mime_remove(mime);
  
  clear_auto_privacy(app);
  etpan_mime_common_display_update(app, &state->common_state);
  set_auto_privacy(app);
  
  return NO_ERROR;
  
 err:
  return res;
}


static int get_next_privacy(struct etpan_subapp * app)
{
  struct app_state * state;
  struct mailmime * mime;
  char * privacy_driver;
  char * privacy_encryption;
  
  state = app->data;
  
  mime =
    etpan_mime_common_get_selected_mime(app, &state->common_state);
  if (mime == NULL)
    return ERROR_INVAL;

  if (mime->mm_parent == NULL) {
    if (mime->mm_type == MAILMIME_MESSAGE) {
      mime = mime->mm_data.mm_message.mm_msg_mime;
      if (mime == NULL)
        return ERROR_INVAL;
    }
  }
  
  if (mime->mm_parent == NULL) {
    /* can't encode root MIME part */
    return ERROR_INVAL;
  }
  
  etpan_mime_get_privacy(state->common_state.params,
      mime, &privacy_driver, &privacy_encryption);
  
#if 0
  etpan_security_encrypt_next(app->app,
      security_driver, security_encryption,
      &security_driver, &security_encryption, 1);
#endif
  etpan_privacy_encrypt_next(app->app->privacy,
      privacy_driver, privacy_encryption,
      &privacy_driver, &privacy_encryption, 1);
  
  etpan_mime_set_privacy(state->common_state.params,
      mime, privacy_driver, privacy_encryption);
  
  state->auto_privacy = 0;
  
  return NO_ERROR;
}


static int apply_encryption(struct etpan_subapp * app,
    struct mailfolder * folder, struct mailmime * mime)
{
  int r;
  struct mailmime * parent;
  struct app_state * state;
  char * privacy_driver;
  char * privacy_encryption;
  clistiter * cur;
  
  state = app->data;
  
  switch (mime->mm_type) {
  case MAILMIME_MULTIPLE:
    for(cur = clist_begin(mime->mm_data.mm_multipart.mm_mp_list) ;
        cur != NULL ; cur = clist_next(cur)) {
      struct mailmime * child;
      
      child = clist_content(cur);
      
      r = apply_encryption(app, folder, child);
      if (r != NO_ERROR)
        return r;
    }
    break;
    
  case MAILMIME_MESSAGE:
    if (mime->mm_data.mm_message.mm_msg_mime != NULL) {
      r = apply_encryption(app, folder, mime->mm_data.mm_message.mm_msg_mime);
      if (r != NO_ERROR)
        return r;
    }
    break;
  }
  
  etpan_mime_get_privacy(state->common_state.params,
      mime, &privacy_driver,  &privacy_encryption);
  
  if ((privacy_driver != NULL) && (privacy_encryption != NULL)) {
    struct mailmime * encrypted_mime;
    
    if (mime->mm_parent == NULL)
      return ERROR_INVAL;

#if 0    
    r = etpan_security_encrypt(app->app,
        security_driver, security_encryption,
        folder, mime, &encrypted_mime);
#endif
    r = mailprivacy_encrypt(app->app->privacy,
        privacy_driver, privacy_encryption,
        mime, &encrypted_mime);
    if (r != MAIL_NO_ERROR) {
      switch (r) {
      case MAIL_ERROR_INVAL:
        ETPAN_APP_LOG((app->app, "could not encrypt - missing public key or certificate"));
        return ERROR_INVAL;
      case MAIL_ERROR_MEMORY:
        ETPAN_APP_LOG((app->app, "could not encrypt - not enough memory"));
        return ERROR_MEMORY;
      }
      return ERROR_PRIVACY;
    }
    
    parent = mime->mm_parent;

    if (parent->mm_type == MAILMIME_MULTIPLE)
      mime->mm_multipart_pos->data = encrypted_mime;
    else
      parent->mm_data.mm_message.mm_msg_mime = encrypted_mime;
    encrypted_mime->mm_parent = parent;
    mime->mm_parent = NULL;
    
    etpan_message_mime_clear(mime);
    mailmime_free(mime);
  }
  
  return NO_ERROR;
}

static int send_message(struct etpan_subapp * app)
{
  struct mailfolder * folder;
  int send_type;
  int r;
  mailmessage * msg;
  struct app_state * state;

  state = app->data;
  
  folder = etpan_mime_common_get_folder(app, &state->common_state);
  
  msg = etpan_mime_common_get_msg(app, &state->common_state);
  
  send_type = ETPAN_SEND_FLAG_NONE;
  
  if (msg->msg_mime->mm_type == MAILMIME_MESSAGE) {
    if (msg->msg_mime->mm_data.mm_message.mm_fields != NULL) {
      send_type = etpan_get_send_type(msg->msg_mime->mm_data.mm_message.mm_fields,
          state->post_from_folder);
    }
  }
  
  if (send_type == ETPAN_SEND_FLAG_NONE) {
    ETPAN_APP_LOG((app->app, "send message - error, no recipient"));
    return ERROR_INVAL;
  }

  r = apply_encryption(app, state->post_from_folder, msg->msg_mime);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app,
                      "send message - error while applying encryption"));
    return ERROR_INVAL;
  }
  
  /* does not apply clear_security because security is already applied */
  etpan_mime_common_display_update(app, &state->common_state);
  
  r = etpan_subapp_thread_folder_op_add(app, THREAD_ID_MIMEEDIT_FETCH_FOR_SEND,
      ETPAN_THREAD_MESSAGE_FETCH,
      folder,
      msg, NULL,
      NULL,
      fetch_for_send_callback, NULL, NULL);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "send message - not enough memory"));
    return ERROR_MEMORY;
  }
  
  return NO_ERROR;
}



static int headereditor_match_fields(struct etpan_subapp * app, void * data)
{
  return (etpan_header_editor_app_get_fields(app) == data);
}

static int etpan_edit_headers(struct etpan_subapp * app,
    struct mailmime * mime)
{
  struct etpan_subapp * headereditor_app;
  
  if (mime->mm_type != MAILMIME_MESSAGE)
    return ERROR_INVAL;

  headereditor_app = etpan_app_find_subapp(app->app, "header-editor",
      1, headereditor_match_fields, mime->mm_data.mm_message.mm_fields);
  if (headereditor_app != NULL) {
    etpan_app_switch_subapp(headereditor_app, 0);
    return NO_ERROR;
  }
  
  /* reuse a previous interface */
  headereditor_app = etpan_app_find_subapp(app->app, "header-editor",
      0, NULL, NULL);
  if (headereditor_app == NULL) {
    /* create new */
    headereditor_app = etpan_header_editor_app_new(app->app);
    if (headereditor_app == NULL)
      return ERROR_MEMORY;
  }

  if (mime->mm_data.mm_message.mm_fields == NULL) {
    mime->mm_data.mm_message.mm_fields = mailimf_fields_new_empty();
    if (mime->mm_data.mm_message.mm_fields == NULL)
      return ERROR_MEMORY;
  }
  
  etpan_subapp_set_parent(headereditor_app, app);
  
  etpan_header_editor_app_set(headereditor_app,
      mime->mm_data.mm_message.mm_fields);
  
  etpan_app_switch_subapp(headereditor_app, 0);
  
  return NO_ERROR;
}


/*
  fetch_for_postpone_callback()
  
  WARNING : twice calls to RESULT_FREE but in two different cases
*/

static void fetch_for_postpone_callback(struct etpan_subapp * app,
        struct etpan_thread_op * op, int app_op_type, void * data)
{
  struct app_state * state;
  struct etpan_message_fetch_result * fetch_result;
  FILE * f;
  char filename[PATH_MAX];
  size_t write_len;
  int r;
  struct mailfolder * postpone_folder;
  char * dup_filename;
  
  postpone_folder = data;

  state = app->data;
  
  if (op->err != NO_ERROR) {
    ETPAN_APP_LOG((app->app,
                      "message postpone - error while getting message"));
    return;
  }
  
  fetch_result = op->result;
  
  /* write result to a file */
  f = etpan_get_tmp_file(filename, sizeof(filename));
  if (f == NULL) {
    ETPAN_APP_LOG((app->app, "message postpone - error writing file"));
    goto exit;
  }
  
  write_len = fwrite(fetch_result->content, 1, fetch_result->len, f);
  if (write_len != fetch_result->len) {
    ETPAN_APP_LOG((app->app, "message postpone - error writing file"));
    goto close;
  }
  
  fclose(f);

  dup_filename = strdup(filename);
  if (dup_filename == NULL) {
    ETPAN_APP_LOG((app->app, "message postpone - not enough memory"));
    goto close;
  }
  
  /*
    this must precede leave so that message result free operation is done
    before leaving (the message is freed when switching app to quit)
  */
  r = etpan_subapp_thread_op_add(app,
      THREAD_ID_MIMEEDIT_FETCH_FOR_POSTPONE_FREE,
      ETPAN_THREAD_MESSAGE_FETCH_RESULT_FREE,
      op->data.mailaccess.storage, op->data.mailaccess.folder,
      op->data.mailaccess.msg, NULL,
      fetch_result->content,
      NULL, NULL, NULL);
  if (r != NO_ERROR) {
    /* show the warning */
    ETPAN_APP_LOG((app->app, "message send - not enough memory"));
  }
  free(fetch_result);
  
  r = etpan_subapp_thread_folder_op_add(app, THREAD_ID_MIMEEDIT_POSTPONE,
      ETPAN_THREAD_FOLDER_APPEND_MESSAGE,
      postpone_folder,
      NULL, NULL,
      dup_filename,
      NULL, NULL, NULL);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "message postpone - not enough memory"));
    return;
  }
  
  ETPAN_APP_LOG((app->app, "message postponed"));
  
  if (state->cancel_action != NULL)
    state->cancel_action(state->action_arg);
  
  etpan_app_quit_subapp(app);
  
  return;
  
 close:
  fclose(f);
  unlink(filename);
 exit:
  r = etpan_subapp_thread_op_add(app,
      THREAD_ID_MIMEEDIT_FETCH_FOR_POSTPONE_FREE,
      ETPAN_THREAD_MESSAGE_FETCH_RESULT_FREE,
      op->data.mailaccess.storage, op->data.mailaccess.folder,
      op->data.mailaccess.msg, NULL,
      fetch_result->content,
      NULL, NULL, NULL);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "message postpone - not enough memory"));
  }
  free(fetch_result);
}

static int add_postpone_header(struct mailmessage * msg,
    struct mailfolder * post_from_folder)
{
  struct mailimf_field * field;
  char * name;
  char * value;
  int r;
  
  if (post_from_folder == NULL)
    return NO_ERROR;
  
  name = strdup("X-etPan-Draft-Folder");
  if (name == NULL)
    goto err;
  
  value = etpan_folder_get_virtual_path(post_from_folder);
  if (value == NULL)
    goto free_name;

  field = mailimf_field_new_custom(name, value);
  if (field == NULL)
    goto free_value;
  
  r = mailimf_fields_add(msg->msg_mime->mm_data.mm_message.mm_fields, field);
  if (r != MAILIMF_NO_ERROR) {
    mailimf_field_free(field);
    goto err;
  }
  
  return NO_ERROR;
  
 free_value:
  free(value);
 free_name:
  free(name); 
 err:
 return ERROR_MEMORY; 
}

static int postpone_message(struct etpan_subapp * app)
{
  struct mailfolder * folder;
  int r;
  mailmessage * msg;
  struct app_state * state;
  struct mailfolder * postpone_folder;
  int res;
  
  state = app->data;
  
  postpone_folder =
    etpan_vfolder_get_draft_folder(app->app->config.vfolder_config,
        state->post_from_folder);
  if (postpone_folder == NULL) {
    ETPAN_APP_LOG((app->app, "postpone message - no draft folder was defined"));
    res = ERROR_COULD_NOT_POSTPONE;
    goto err;
  }
  
  folder = etpan_mime_common_get_folder(app, &state->common_state);
  
  msg = etpan_mime_common_get_msg(app, &state->common_state);
  
  r = add_postpone_header(msg, state->post_from_folder);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "postpone message - not enough memory"));
    res = r;
    goto err;
  }
  
  r = etpan_subapp_thread_folder_op_add(app,
      THREAD_ID_MIMEEDIT_FETCH_FOR_POSTPONE,
      ETPAN_THREAD_MESSAGE_FETCH,
      folder,
      msg, NULL,
      NULL,
      fetch_for_postpone_callback, postpone_folder, NULL);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "postpone message - not enough memory"));
    res = r;
    goto err;
  }
  
  return NO_ERROR;
  
 err:
  return res;
}



static void clear_privacy_recursive(struct etpan_mime_params * params,
    struct mailmime * root)
{
  clistiter * cur;
  struct mailmime * child;

  etpan_mime_set_privacy(params, root, NULL, NULL);
  
  switch (root->mm_type) {
  case MAILMIME_SINGLE:
    break;

  case MAILMIME_MULTIPLE:
    for(cur = clist_begin(root->mm_data.mm_multipart.mm_mp_list) ;
        cur != NULL ; cur = clist_next(cur)) {
      child = clist_content(cur);
      
      clear_privacy_recursive(params, child);
    }
    break;
  case MAILMIME_MESSAGE:
    if (root->mm_data.mm_message.mm_msg_mime != NULL) {
      child = root->mm_data.mm_message.mm_msg_mime;

      clear_privacy_recursive(params, child);
    }
    break;
  }
}


void clear_auto_privacy(struct etpan_subapp * app)
{
  struct app_state * state;
  mailmessage * msg;
  
  state = app->data;
  
  if (!state->auto_privacy)
    return;
  
  msg = etpan_mime_common_get_msg(app, &state->common_state);
  
  clear_privacy_recursive(state->common_state.params, msg->msg_mime);
}

void set_auto_privacy(struct etpan_subapp * app)
{
  struct app_state * state;
  mailmessage * msg;
  struct etpan_account_info * account;
  struct mailmime * first_part;
  
  state = app->data;
  
  if (!state->auto_privacy)
    return;
  
  msg = etpan_mime_common_get_msg(app, &state->common_state);
  if (msg->msg_mime->mm_type != MAILMIME_MESSAGE)
    return;
  
  first_part = msg->msg_mime->mm_data.mm_message.mm_msg_mime;
  
  if (first_part == NULL)
    return;
  
  account = etpan_vfolder_get_account(app->app->config.vfolder_config,
      app->app->config.account_config, state->post_from_folder);
  if (account == NULL)
    return;

  etpan_mime_set_privacy(state->common_state.params, first_part,
      account->privacy_driver, account->privacy_encryption);
}

static void ask_quit_upcall(struct etpan_subapp * input_app,
    int valid, void * data)
{
  struct app_state * state;
  char * name;
  char * dup_name;
  struct etpan_subapp * app;
  int r;
  char * result;
  int do_quit;

  app = data;
  state = app->data;
  
  if (valid == ETPAN_INPUT_COMMON_CANCEL) {
    etpan_app_quit_subapp(input_app);
    return;
  }
  
  result = etpan_search_input_get_value(input_app);
  
  do_quit = (* result == 'y');
  etpan_app_quit_subapp(input_app);
  
  if (do_quit) {
    if (state->cancel_action != NULL)
      state->cancel_action(state->action_arg);
    etpan_app_quit_subapp(app);
  }
  
}


static void ask_quit(struct etpan_subapp * app)
{
  struct app_state * state;
  struct etpan_subapp * input;
  int r;
  
  input = etpan_app_find_subapp(app->app, "search-input",
      0, NULL, NULL);
  if (input == NULL) {
    input = etpan_search_input_new(app->app);
    if (input == NULL)
      goto err;
  }
  
  r = etpan_search_input_set(input,
      "close message (y/n) ? ", 1,
      NULL, 0,
      ask_quit_upcall, app);
  if (input == NULL)
    goto err;
  
  etpan_subapp_set_parent(input, app);
  etpan_app_switch_subapp(input, 0);
  
  return;
  
 err:
  ETPAN_APP_LOG((app->app, "composer - not enough memory"));
}




#define HELP_TEXT \
"\
Help for MIME message editor\n\
----------------------------\n\
\n\
This application will let you edit the MIME structure of the\n\
message.\n\
\n\
- Ctrl-W,\n\
  Ctrl-X     : switch between applications\n\
\n\
- arrow keys : move cursor\n\
\n\
- e          : if the part is message/rfc822, it runs the headers editor,\n\
                 if this is a text part, you can edit the content of\n\
                 the part.\n\
- Shift-A    : adds an attachment\n\
- Shift-T    : adds a text part\n\
- Shift-M    : adds a multipart\n\
- Ctrl-M     : adds a message part\n\
- d          : deletes the selected part\n\
- Shift-P    : sets privacy option to use on the selected part\n\
\n\
- +          : expand/close node\n\
- *          : expand all nodes\n\
\n\
- [Enter]    : view selected part\n\
- s          : save selected MIME part\n\
- h          : view header of selected MIME part\n\
- Ctrl-H     : view MIME headers of selected MIME part\n\
- t          : view source of selected MIME part\n\
\n\
- m          : compose new message\n\
- a          : reply to the author of the selected MIME part\n\
- r          : reply to the selected MIME part\n\
- g          : reply to all recipient of the selected MIME part\n\
- f          : forward selected MIME part\n\
- F          : forward selected MIME part as attachment\n\
- b          : bounce the selected MIME part\n\
- B          : redirect (Subject line is modified)\n\
\n\
- y          : send message\n\
- p          : postpone message\n\
- q          : exit\n\
\n\
- ?          : help\n\
- Ctrl-L     : Console log\n\
\n\
(? or q to exit help)\n\
"

static int show_help(struct etpan_subapp * app)
{
  return etpan_show_help(app, HELP_TEXT, sizeof(HELP_TEXT) - 1);
}
