/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-msg-new.c,v 1.20 2004/12/16 15:54:15 hoa Exp $
 */

#include "etpan-msg-new.h"

#include "etpan-app-subapp.h"
#include "etpan-subapp.h"
#include <libetpan/libetpan.h>
#include "etpan-imf-helper.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <unistd.h>
#include "etpan-app.h"
#include "etpan-cfg-vfolder.h"
#include "etpan-mime-edit-app.h"
#include "etpan-errors.h"
#include <string.h>
#include <stdlib.h>
#include "etpan-tools.h"
#include "etpan-header-editor-app.h"
#include "etpan-subapp-thread.h"
#include "etpan-mime-tools.h"
#include "etpan-compose-app.h"
#include "etpan-cfg-sender.h"

int etpan_add_signature(FILE * f, struct etpan_account_info * account)
{
  FILE * sig_f;
  char str[1024];
  int r;

  if (account == NULL)
    return NO_ERROR;

  if (account->signature == NULL)
    return NO_ERROR;
  
  sig_f = fopen(account->signature, "r");
  if (sig_f == NULL)
    return NO_ERROR;

  r = fputs("\r\n", f);
  if (r < 0)
    goto err;
  
  r = fputs("-- \r\n", f);
  if (r < 0)
    goto err;
  
  while (fgets(str, 1024, sig_f)) {
    r = fputs(str, f);
    if (r < 0)
      goto err;
  }
  fclose(sig_f);
  
  return NO_ERROR;
  
 err:
  return ERROR_FILE;
}





int etpan_new_get_fields(struct etpan_app * app,
    struct mailfolder * folder,
    struct mailimf_address_list * dest_addr,
    struct mailimf_fields ** result)
{
  struct mailimf_fields * new_fields;
  struct mailimf_mailbox_list * from;
  struct mailimf_address_list * to;
  char * subject;
  int nntp_protocol;
  struct etpan_account_info * account;
  int r;

  nntp_protocol = etpan_is_nntp_protocol(folder);

  account = etpan_vfolder_get_account(app->config.vfolder_config,
      app->config.account_config, folder);

  /* build headers */

  if (nntp_protocol)
    to = NULL;
  else {
    clist * list;
    
    if (dest_addr == NULL) {
      list = clist_new();
      to = mailimf_address_list_new(list);
    }
    else {
      to = etpan_dup_address_list(dest_addr);
    }
  }

  if (account == NULL)
    from = NULL;
  else {
    from = etpan_get_from_field(app, account);
    if (from == NULL)
      goto free_to;
  }

  subject = strdup("");
  if (subject == NULL) {
    goto free_from;
  }

  new_fields = mailimf_fields_new_with_data(from, /* from */
					    NULL, /* sender */
					    NULL, /* reply-to */
					    to,
					    NULL,
					    NULL,
					    NULL,
					    NULL,
					    subject);
  if (new_fields == NULL)
    goto free_subject;

  if (nntp_protocol) {
    r = etpan_set_current_newsgroups_header(new_fields, folder);
    if (r != NO_ERROR)
      goto free_fields;
  }

  r = etpan_set_xmailer(new_fields, nntp_protocol);
  if (r != NO_ERROR)
    goto free_fields;

  * result = new_fields;

  return NO_ERROR;

 free_fields:
  mailimf_fields_free(new_fields);
  goto err;
 free_subject:
  free(subject);
 free_from:
  if (from != NULL)
    mailimf_mailbox_list_free(from);
 free_to:
  mailimf_address_list_free(to);
 err:
  return ERROR_MEMORY;
}


int etpan_new_get_resent_fields(struct etpan_app * app,
    struct mailfolder * folder,
    struct mailimf_address_list * dest_addr,
    struct mailimf_fields ** result)
{
  struct mailimf_fields * new_fields;
  struct mailimf_mailbox_list * from;
  struct mailimf_address_list * to;
  int nntp_protocol;
  struct etpan_account_info * account;
  int r;
  
  nntp_protocol = etpan_is_nntp_protocol(folder);

  account = etpan_vfolder_get_account(app->config.vfolder_config,
      app->config.account_config, folder);

  /* build headers */

  if (nntp_protocol)
    to = NULL;
  else {
    clist * list;
    
    if (dest_addr == NULL) {
      list = clist_new();
      to = mailimf_address_list_new(list);
    }
    else {
      to = etpan_dup_address_list(dest_addr);
    }
  }

  if (account == NULL)
    from = NULL;
  else {
    from = etpan_get_from_field(app, account);
    if (from == NULL)
      goto free_to;
  }

  new_fields = mailimf_resent_fields_new_with_data(from, /* from */
      NULL /* sender */, to, NULL, NULL);
  if (new_fields == NULL)
    goto free_from;

  if (nntp_protocol) {
    r = etpan_set_current_newsgroups_header(new_fields, folder);
    if (r != NO_ERROR)
      goto free_fields;
  }

  r = etpan_set_xmailer(new_fields, nntp_protocol);
  if (r != NO_ERROR)
    goto free_fields;

  * result = new_fields;

  return NO_ERROR;

 free_fields:
  mailimf_fields_free(new_fields);
  goto err;
 free_from:
  if (from != NULL)
    mailimf_mailbox_list_free(from);
 free_to:
  mailimf_address_list_free(to);
 err:
  return ERROR_MEMORY;
}


int etpan_build_simple_mime(struct etpan_app * app,
    struct mailimf_fields * new_fields,
    char * body_filename,
    struct mailmime ** result,
    struct mailmime ** pedit_part)
{
  struct mailmime_fields * mime_fields;
  struct mailmime * mime_sub;
  struct mailmime * mime;
  struct mailmime_content * content;
  struct mailmime_parameter * param;
  int res;
  int r;

  /* text/plain part */

  mime_fields = mailmime_fields_new_encoding(MAILMIME_MECHANISM_8BIT);
  if (mime_fields == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }

  content = mailmime_content_new_with_str("text/plain");
  if (content == NULL) {
    res = ERROR_MEMORY;
    goto free_fields;
  }

  param = mailmime_param_new_with_data("charset",
      app->config.global_config->editor_charset);
  if (param == NULL) {
    res = ERROR_MEMORY;
    goto free_content;
  }

  r = clist_append(content->ct_parameters, param);
  if (r < 0) {
    mailmime_parameter_free(param);
    res = ERROR_MEMORY;
    goto free_content;
  }

  mime_sub = mailmime_new_empty(content, mime_fields);
  if (mime_sub == NULL) {
    res = ERROR_MEMORY;
    goto free_content;
  }

  body_filename = strdup(body_filename);
  if (body_filename == NULL) {
    res = ERROR_MEMORY;
    goto free_content;
  }
  
  r = mailmime_set_body_file(mime_sub, body_filename);
  if (r != MAILIMF_NO_ERROR) {
    free(body_filename);
    res = r;
    goto free_mime;
  }
  
  /* message */
  
  mime = mailmime_new_message_data(mime_sub);
  if (mime == NULL) {
    res = ERROR_MEMORY;
    goto free_mime;
  }

  mailmime_set_imf_fields(mime, new_fields);

  * result = mime;
  if (pedit_part != NULL)
    * pedit_part = mime_sub;

  return NO_ERROR;

 free_mime:
  mailmime_free(mime_sub);
  goto err;
 free_content:
  mailmime_content_free(content);
 free_fields:
  mailmime_fields_free(mime_fields);
 err:
  return res;
}


int etpan_message_new(struct etpan_subapp * app,
    struct mailfolder * folder, struct mailmime ** result,
    struct mailmime ** pedit_part)
{
  struct etpan_account_info * account;
  struct mailimf_address_list * addr_list;
  char body_filename[PATH_MAX];
  int res;
  struct mailmime * mime;
  mode_t old_umask;
  int r;
  struct mailimf_fields * new_fields;
  FILE * f;
  struct mailmime * edit_part;

  account = etpan_vfolder_get_account(app->app->config.vfolder_config,
      app->app->config.account_config, folder);
  
  addr_list = etpan_vfolder_get_to(app->app->config.vfolder_config, folder);

  r = etpan_new_get_fields(app->app, folder, addr_list, &new_fields);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }

  r = etpan_get_tmp_name(body_filename, sizeof(body_filename));
  if (r < 0) {
    res = ERROR_FILE;
    goto free_fields;
  }
  
  old_umask = umask(0077);
  f = fopen(body_filename, "w");
  umask(old_umask);
  if (f == NULL) {
    res = ERROR_FILE;
    goto unlink;
  }

  r = fputs("\r\n", f);
  if (r < 0) {
    fclose(f);
    res = ERROR_FILE;
    goto unlink;
  }
  
  r = etpan_add_signature(f, account);
  if (r != NO_ERROR) {
    fclose(f);
    res = r;
    goto unlink;
  }
  
  fclose(f);
  
  r = etpan_build_simple_mime(app->app, new_fields, body_filename,
      &mime, &edit_part);
  if (r != NO_ERROR) {
    res = r;
    goto unlink;
  }
  
  * result = mime;
  * pedit_part = edit_part;

  return NO_ERROR;
  
 free_fields:
  mailimf_fields_free(new_fields);
 unlink:
  unlink(body_filename);
 err:
  return res;
}

void etpan_message_mime_remove(struct mailmime * mime)
{
  etpan_message_mime_clear(mime);
  mailmime_smart_remove_part(mime);
}

void etpan_message_mime_clear(struct mailmime * mime)
{
  struct mailmime_data * data;
  clistiter * cur;
  
  switch (mime->mm_type) {
  case MAILMIME_SINGLE:
    data = mime->mm_data.mm_single;
    if (data != NULL) {
      if (data->dt_type == MAILMIME_DATA_FILE)
        unlink(data->dt_data.dt_filename);
    }
    break;
    
  case MAILMIME_MULTIPLE:
    data = mime->mm_data.mm_multipart.mm_preamble;
    if (data != NULL) {
      if (data->dt_type == MAILMIME_DATA_FILE)
        unlink(data->dt_data.dt_filename);
    }
    data = mime->mm_data.mm_multipart.mm_epilogue;
    if (data != NULL) {
      if (data->dt_type == MAILMIME_DATA_FILE)
        unlink(data->dt_data.dt_filename);
    }
    
    for(cur = clist_begin(mime->mm_data.mm_multipart.mm_mp_list) ;
        cur != NULL ; cur = clist_next(cur)) {
      struct mailmime * submime;
      
      submime = clist_content(cur);
      
      etpan_message_mime_clear(submime);
    }
    break;
    
  case MAILMIME_MESSAGE:
    if (mime->mm_data.mm_message.mm_msg_mime != NULL) {
      etpan_message_mime_clear(mime->mm_data.mm_message.mm_msg_mime);
    }
    break;
  }
}

static int etpan_edit_initial_headers(struct etpan_subapp * app,
    mailmessage * msg,
    struct mailmime * edit_part,
    int edit_part_line,
    struct mailfolder * post_from_folder,
    void (* action)(void *), void * action_arg,
    void (* cancel_action)(void *));

int etpan_compose_new(struct etpan_subapp * app, struct mailfolder * folder)
{
  int r;
  struct mailmime * mime;
  int res;
  struct mailmime * edit_part;

  r = etpan_message_new(app, folder, &mime, &edit_part);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }
  
  r = etpan_compose(app, folder, mime, edit_part, 1, NULL, NULL, NULL);
  if (r != NO_ERROR) {
    res = r;
    goto free_mime;
  }
  
  return NO_ERROR;
  
 free_mime:
  etpan_message_mime_clear(mime);
  mailmime_free(mime);
 err:
  return res;
}

int etpan_edit_file(struct etpan_app * app,
    char * filename, int line, size_t * size)
{
  int r;
  int res;
  struct stat buf_before;
  struct stat buf_after;
  char command[PATH_MAX];
  char quoted_filename[PATH_MAX];

  r = stat(filename, &buf_before);
  if (r < 0) {
    res = ERROR_FILE;
    goto err;
  }

  r = etpan_prepare_file_for_edition(filename);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }
  
  r = etpan_quote_filename(quoted_filename, sizeof(quoted_filename), filename);
  if (r < 0) {
    res = ERROR_MEMORY;
    goto err;
  }
  
  etpan_app_display_done(app);
  snprintf(command, PATH_MAX, "%s +%i %s", app->config.global_config->editor,
      line, quoted_filename);
  r = system(command);
  etpan_app_display_init(app);
  etpan_app_set_colors(app);
  if (WEXITSTATUS(r) != 0) {
    res = ERROR_COMMAND;
    goto err;
  }
  
  r = stat(filename, &buf_after);
  if (r < 0) {
    res = ERROR_FILE;
    goto err;
  }

  if ((buf_before.st_mtime == buf_after.st_mtime) &&
      (buf_before.st_size == buf_after.st_size)) {
    res = ERROR_CANCELLED;
    goto err;
  }

  if (size != NULL)
    * size = buf_after.st_size;

  return NO_ERROR;

 err:
  return res;
}

struct mailmime * etpan_message_get_first_text_part(struct mailmime * mime)
{
  struct mailmime_data * data;
  clistiter * cur;
  
  switch (mime->mm_type) {
  case MAILMIME_SINGLE:
    data = mime->mm_data.mm_single;
    if (data != NULL) {
      if ((data->dt_type == MAILMIME_DATA_FILE) && etpan_mime_is_text(mime))
        return mime;
    }
    break;
    
  case MAILMIME_MULTIPLE:
    for(cur = clist_begin(mime->mm_data.mm_multipart.mm_mp_list) ;
        cur != NULL ; cur = clist_next(cur)) {
      struct mailmime * submime;
      struct mailmime * first_mime;
      
      submime = clist_content(cur);
      
      first_mime = etpan_message_get_first_text_part(submime);
      if (first_mime != NULL)
        return first_mime;
    }
    break;
    
  case MAILMIME_MESSAGE:
    if (mime->mm_data.mm_message.mm_msg_mime != NULL) {
      struct mailmime * first_mime;
      
      first_mime =
        etpan_message_get_first_text_part(mime->mm_data.mm_message.mm_msg_mime);
      if (first_mime != NULL)
        return first_mime;
    }
    break;
  }
  
  return NULL;
}


/*
  etpan_compose()
*/

int etpan_compose(struct etpan_subapp * app,
    struct mailfolder * post_from_folder,
    struct mailmime * mime,
    struct mailmime * edit_part,
    int edit_part_line,
    void (* action)(void *), void * action_arg,
    void (* cancel_action)(void *))
{
  int r;
  mailmessage * msg;
  int res;
  char tmpdir[PATH_MAX];
  
  msg = mime_message_init(mime);
  if (msg == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }
  
  snprintf(tmpdir, sizeof(tmpdir),
      "%s/%s", getenv("HOME"), ETPAN_TMP_MIME_PATH);

#if 0  
  r = mime_message_set_tmpdir(msg, tmpdir);
  if (r != MAIL_NO_ERROR) {
    res = ERROR_MEMORY;
    goto free_msg;
  }
#endif
  
  r = libetpan_message_register(app->app->thread_manager->engine, NULL, msg);
  if (r != MAIL_NO_ERROR) {
    res = ERROR_MEMORY;
    goto free_msg;
  }
  
  libetpan_message_ref(app->app->thread_manager->engine, msg);
  
  r = etpan_edit_initial_headers(app, msg,
      edit_part, edit_part_line, post_from_folder,
      action, action_arg, cancel_action);
  if (r != NO_ERROR) {
    res = r;
    goto unref;
  }
  
  return NO_ERROR;
  
 unref:
  mime_message_detach_mime(msg);
  libetpan_message_unref(app->app->thread_manager->engine, msg);
  goto err;
 free_msg:
  mime_message_detach_mime(msg);
  mailmessage_free(msg);
 err:
  return res;
}

int etpan_set_text_content_type(struct etpan_app * app,
    struct mailmime * mime)
{
  struct mailmime_content * content;
  struct mailmime_parameter * param;
  int res;
  int r;
  
  content = mailmime_content_new_with_str("text/plain");
  if (content == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }
  
  param = mailmime_param_new_with_data("charset",
      app->config.global_config->editor_charset);
  if (param == NULL) {
    res = ERROR_MEMORY;
    goto free_content;
  }
  
  r = clist_append(content->ct_parameters, param);
  if (r < 0) {
    mailmime_parameter_free(param);
    res = ERROR_MEMORY;
    goto free_content;
  }
  
  mailmime_content_free(mime->mm_content_type);
  mime->mm_content_type = content;
  
  return NO_ERROR;
  
 free_content:
  mailmime_content_free(content);
 err:
  return res;
}


int etpan_get_send_type(struct mailimf_fields * new_fields,
    struct mailfolder * post_from_folder)
{
  clistiter * cur;
  int type;
  
  type = 0;

  for(cur = clist_begin(new_fields->fld_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    struct mailimf_field * field;

    field = clist_content(cur);

    switch (field->fld_type) {
    case MAILIMF_FIELD_TO:
    case MAILIMF_FIELD_CC:
    case MAILIMF_FIELD_BCC:
    case MAILIMF_FIELD_RESENT_TO:
    case MAILIMF_FIELD_RESENT_CC:
    case MAILIMF_FIELD_RESENT_BCC:
      type |= ETPAN_SEND_FLAG_SENDMAIL;
      break;

    case MAILIMF_FIELD_OPTIONAL_FIELD:
      if ((strcasecmp(field->fld_data.fld_optional_field->fld_name,
               "To") == 0) ||
          (strcasecmp(field->fld_data.fld_optional_field->fld_name,
              "Cc") == 0) ||
          (strcasecmp(field->fld_data.fld_optional_field->fld_name,
              "Bcc") == 0))
        type |= ETPAN_SEND_FLAG_SENDMAIL;
      if (strcasecmp(field->fld_data.fld_optional_field->fld_name,
              "Newsgroups") == 0)
        type |= ETPAN_SEND_FLAG_POSTNEWS;
      if ((strcasecmp(field->fld_data.fld_optional_field->fld_name,
               "Resent-To") == 0) ||
          (strcasecmp(field->fld_data.fld_optional_field->fld_name,
              "Resent-Cc") == 0) ||
          (strcasecmp(field->fld_data.fld_optional_field->fld_name,
              "Resent-Bcc") == 0)) {
        type |= ETPAN_SEND_FLAG_SENDMAIL;
        type &= ~ETPAN_SEND_FLAG_POSTNEWS;
      }
      break;
    }
  }
  
  if (!etpan_is_nntp_protocol(post_from_folder))
    type &= ~ETPAN_SEND_FLAG_POSTNEWS;
  
  return type;
}



/*
  etpan_message_send()  
  
  fields let the function how the message must be sent.
  post_from_folder is provided if message has to be posted on newsgroups.
  
  filename is the entire message text with headers.
*/

int etpan_message_send(struct etpan_subapp * app,
    struct mailfolder * post_from_folder,
    struct mailimf_fields * fields, char * filename)
{
  int send_type;
  int r;
  struct mailfolder * sent_folder;
  
  send_type = ETPAN_SEND_FLAG_NONE;
  if (fields != NULL) {
    send_type = etpan_get_send_type(fields, post_from_folder);
  }
  
  /* save message to sent folder */
  sent_folder = etpan_vfolder_get_sent_folder(app->app->config.vfolder_config,
      post_from_folder);
  
  if (sent_folder != NULL)
    if (sent_folder->fld_storage == NULL)
      sent_folder = NULL;
  
  if (sent_folder != NULL) {
    char * dup_filename;
    struct mailfolder * folder;
    
    /* duplicate the file because the thread will remove the file at the end */
    dup_filename = etpan_dup_file(filename);
    if (dup_filename == NULL) {
      ETPAN_APP_LOG((app->app, "message send - not enough memory"));
      return ERROR_MEMORY;
    }
    
    folder = sent_folder;
    
    r = etpan_subapp_thread_folder_op_add(app, THREAD_ID_MIMEEDIT_SEND,
        ETPAN_THREAD_FOLDER_APPEND_MESSAGE,
        folder,
        NULL, NULL,
        dup_filename,
        NULL, NULL, NULL);
    if (r != NO_ERROR) {
      ETPAN_APP_LOG((app->app, "message send - not enough memory"));
      unlink(dup_filename);
      free(dup_filename);
      return ERROR_MEMORY;
    }
  }
  
  /* post news message */
  if (send_type & ETPAN_SEND_FLAG_POSTNEWS) {
    char * dup_filename;
    struct mailfolder * folder;
    
    /* duplicate the file because the thread will remove the file at the end */
    dup_filename = etpan_dup_file(filename);
    if (dup_filename == NULL) {
      ETPAN_APP_LOG((app->app, "message send - not enough memory"));
      return ERROR_MEMORY;
    }
    
    folder = post_from_folder;
    
    r = etpan_subapp_thread_folder_op_add(app, THREAD_ID_MIMEEDIT_SEND,
        ETPAN_THREAD_FOLDER_APPEND_MESSAGE,
        folder,
        NULL, NULL,
        dup_filename,
        NULL, NULL, NULL);
    if (r != NO_ERROR) {
      ETPAN_APP_LOG((app->app, "message send - not enough memory"));
      unlink(dup_filename);
      free(dup_filename);
      return ERROR_MEMORY;
    }
  }
  
  /* send message through sendmail */
  if (send_type & ETPAN_SEND_FLAG_SENDMAIL) {
    char * dup_filename;
    struct etpan_sender_item * sender;
    
    /* duplicate the file because the thread will remove the file at the end */
    dup_filename = etpan_dup_file(filename);
    if (dup_filename == NULL) {
      ETPAN_APP_LOG((app->app, "message send - not enough memory"));
      return ERROR_MEMORY;
    }

#if 0
    r = etpan_subapp_thread_op_add(app, THREAD_ID_MIMEEDIT_SEND,
        ETPAN_THREAD_MESSAGE_SEND,
        NULL, NULL,
        NULL, NULL,
        dup_filename,
        NULL, NULL, NULL);
#endif
    
    sender = etpan_sender_config_get_item(app->app->config.sender_config);
    if (sender == NULL) {
      ETPAN_APP_LOG((app->app, "message send - missing config"));
      unlink(dup_filename);
      free(dup_filename);
      return ERROR_INVAL;
    }
    
    r = etpan_subapp_thread_sender_op_add(app, THREAD_ID_MIMEEDIT_SEND,
        ETPAN_THREAD_SENDER_SENDMAIL,
        sender, dup_filename,
        NULL, NULL, NULL);
    
    if (r != NO_ERROR) {
      ETPAN_APP_LOG((app->app, "message send - not enough memory"));
      unlink(dup_filename);
      free(dup_filename);
      return ERROR_MEMORY;
    }
  }
  
  return NO_ERROR;
}

static int etpan_edit_initial_headers(struct etpan_subapp * app,
    mailmessage * msg,
    struct mailmime * edit_part,
    int edit_part_line,
    struct mailfolder * post_from_folder,
    void (* action)(void *), void * action_arg,
    void (* cancel_action)(void *))
{
  struct etpan_subapp * compose_app;
  
  /* reuse a previous interface */
  compose_app = etpan_app_find_subapp(app->app, "compose-message",
      0, NULL, NULL);
  if (compose_app == NULL) {
    /* create new */
    compose_app = etpan_compose_app_new(app->app);
    if (compose_app == NULL)
      return ERROR_MEMORY;
  }

  if (msg->msg_mime->mm_type != MAILMIME_MESSAGE)
    return ERROR_INVAL;
  
  if (msg->msg_mime->mm_data.mm_message.mm_fields == NULL) {
    msg->msg_mime->mm_data.mm_message.mm_fields = mailimf_fields_new_empty();
    if (msg->msg_mime->mm_data.mm_message.mm_fields == NULL)
      return ERROR_MEMORY;
  }
  
  while (app->parent != NULL)
    app = app->parent;
  
  etpan_subapp_set_parent(compose_app, app);
  
  etpan_compose_app_set(compose_app, msg,
      edit_part, edit_part_line, post_from_folder,
      action, action_arg, cancel_action);
  
  etpan_app_switch_subapp(compose_app, 0);
  
  return NO_ERROR;
}
