/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-search.c,v 1.7 2004/11/12 12:13:27 nyoxi Exp $
 */

#include "etpan-search.h"
#include "etpan-app.h"
#include "etpan-subapp.h"
#include "etpan-subapp-thread.h"
#include "etpan-app-subapp.h"
#include "etpan-errors.h"
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "etpan-cfg-edit-common.h"
#include "etpan-help-viewer.h"
#include "etpan-msg-list-app.h"
#include "etpan-msg-params.h"
#include <unistd.h>
#include <sys/mman.h>
#include "etpan-msg-renderer.h"
#include <regex.h>
#include "etpan-imf-helper.h"
#include "etpan-mime-tools.h"
#include "etpan-cfg-vfolder.h"

static void set_fd(struct etpan_subapp * app, fd_set * fds, int * maxfd);
static void handle_fd(struct etpan_subapp * app, fd_set * fds);
static void handle_key(struct etpan_subapp * app, int key);
static void display(struct etpan_subapp * app, WINDOW * w);
static void set_color(struct etpan_subapp * app);
static int init(struct etpan_subapp * app);
static void done(struct etpan_subapp * app);
static int display_init(struct etpan_subapp * app);
static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app);

static struct etpan_subapp_driver etpan_search_app_driver = {
  .name = "search-app",
  .always_handle_key = 0,
  .always_on_top = 0,
  .get_idle_delay = NULL,
  .idle = NULL,
  .set_fd = set_fd,
  .handle_fd = handle_fd,
  .handle_key = handle_key,
  .handle_resize = NULL,
  .display = display,
  .set_color = set_color,
  .init = init,
  .done = done,
  .enter = NULL,
  .leave = leave,
  .display_init = display_init,
  .display_done = NULL,
};

enum {
  INFO_TYPE_OR,
  INFO_TYPE_AND,
  INFO_TYPE_RECURSIVE,
  INFO_TYPE_NEW,
  INFO_TYPE_SEEN,
  INFO_TYPE_FLAGGED,
  INFO_TYPE_DELETED,
  INFO_TYPE_ANSWERED,
  INFO_TYPE_FORWARDED,
  INFO_TYPE_FROM,
  INFO_TYPE_TO,
  INFO_TYPE_CC,
  INFO_TYPE_DATE_EQUAL,
  INFO_TYPE_DATE_SINCE,
  INFO_TYPE_DATE_BEFORE,
  INFO_TYPE_RECIPIENT,
  INFO_TYPE_SUBJECT,
  INFO_TYPE_HEADER,
  INFO_TYPE_BODY,
  INFO_TYPE_MESSAGE,
};

#define INPUT_MAX 1024

struct search_cond {
  int type;
  int condition_not;
  char value[INPUT_MAX];
  int case_sensitive;
  regex_t value_regex;
  int regex_init;
  time_t date;
};

struct search_cond * search_cond_new(int type)
{
  struct search_cond * elt;
  
  elt = malloc(sizeof(* elt));
  if (elt == NULL)
    return NULL;
  
  elt->type = type;
  elt->condition_not = 0;
  elt->value[0] = '\0';
  elt->case_sensitive = 0;
  elt->regex_init = 0;
  elt->date = time(NULL);
  
  return elt;
}

struct app_state {
  unsigned int index;
  unsigned int first;
  carray * info_type_tab;

  struct mailfolder * folder;

  int main_attr;
  int selection_attr;
  int status_attr;
};

static int show_help(struct etpan_subapp * app);

static int do_search(struct etpan_subapp * app);

static void set_fd(struct etpan_subapp * app, fd_set * fds, int * maxfd)
{
  etpan_subapp_thread_set_fd(app, fds, maxfd);
}

static void handle_fd(struct etpan_subapp * app, fd_set * fds)
{
  etpan_subapp_thread_handle_fd(app, fds);
}

static void add_search_cond(struct etpan_subapp * app, int type);

static void handle_key(struct etpan_subapp * app, int key)
{
  unsigned int list_lines;
  int current_type;
  struct app_state * state;
  struct search_cond * elt;
  unsigned int count;
  int r;

  state = app->data;
  
  count = carray_count(state->info_type_tab);
  
  list_lines = app->display_height - 1;
  
  if (carray_count(state->info_type_tab) == 0) {
    elt = NULL;
    current_type = -1;
  }
  else {
    elt = carray_get(state->info_type_tab, state->index);
    current_type = elt->type;
  }
  
  switch (key) {
  case KEY_F(1):
  case '?':
    show_help(app);
    break;
    
  case KEY_LEFT:
  case 'h':
    switch (current_type) {
    case INFO_TYPE_OR:
      elt->type = INFO_TYPE_AND;
      break;
    case INFO_TYPE_AND:
      break;
    default:
      if (elt->type != INFO_TYPE_RECURSIVE)
        elt->type --;
      elt->case_sensitive = 0;
      break;
    }
    break;
    
  case KEY_RIGHT:
  case 'l':
    switch (current_type) {
    case INFO_TYPE_AND:
      elt->type = INFO_TYPE_OR;
      break;
    case INFO_TYPE_OR:
      break;
    default:
      if (elt->type != INFO_TYPE_MESSAGE)
        elt->type ++;
      elt->case_sensitive = 0;
      break;
    }
    break;
    
  case 'k':
  case KEY_UP:
    if (state->index > 0)
      state->index --;
    break;
    
  case 'j':
  case KEY_DOWN:
    if (state->index < count - 1)
      state->index ++;
    break;
    
  case KEY_NPAGE:
    if (state->index + list_lines - 1 < count)
      state->index += list_lines - 1;
    else
      state->index = count - 1;
    break;
    
  case KEY_PPAGE:
    if (state->index >= list_lines - 1)
      state->index -= list_lines - 1;
    else
      state->index = 0;
    break;
    
  case KEY_HOME:
    state->index = 0;
    break;
    
  case KEY_END:
    state->index = count - 1;
    break;
    
  case KEY_CTRL('G'):
    etpan_app_quit_subapp(app);
    break;
    
  case 'y':
    do_search(app);
    break;
    
  case 'a':
    add_search_cond(app, INFO_TYPE_FROM);
    break;

  case 's':
    add_search_cond(app, INFO_TYPE_SUBJECT);
    break;
    
  case 'f':
    add_search_cond(app, INFO_TYPE_FROM);
    break;
    
  case 't':
    add_search_cond(app, INFO_TYPE_TO);
    break;

  case 'r':
    add_search_cond(app, INFO_TYPE_RECIPIENT);
    break;

  case 'R':
    add_search_cond(app, INFO_TYPE_RECURSIVE);
    break;
    
  case 'd':
    if ((current_type != INFO_TYPE_OR) && (current_type != INFO_TYPE_AND)) {
      free(elt);
      carray_delete(state->info_type_tab, state->index);
    }
    break;

  case 'i':
    switch (current_type) {
    case INFO_TYPE_FROM:
    case INFO_TYPE_TO:
    case INFO_TYPE_CC:
    case INFO_TYPE_RECIPIENT:
    case INFO_TYPE_SUBJECT:
    case INFO_TYPE_HEADER:
    case INFO_TYPE_BODY:
    case INFO_TYPE_MESSAGE:
      elt->case_sensitive = !elt->case_sensitive;
      break;
    }
    break;
    
  case '!':
    switch (current_type) {
    case INFO_TYPE_OR:
    case INFO_TYPE_AND:
    case INFO_TYPE_RECURSIVE:
    case INFO_TYPE_DATE_EQUAL:
    case INFO_TYPE_DATE_SINCE:
    case INFO_TYPE_DATE_BEFORE:
    case -1:
      break;
      
    default:
      elt->condition_not = !elt->condition_not;
      break;
    }
    break;
    
  case '\n':
    switch (current_type) {
    case INFO_TYPE_FROM:
      r = etpan_cfg_edit_string(app, "From: ",
          elt->value, sizeof(elt->value), 0);
      break;
    case INFO_TYPE_TO:
      r = etpan_cfg_edit_string(app, "To: ",
          elt->value, sizeof(elt->value), 0);
      break;
    case INFO_TYPE_CC:
      r = etpan_cfg_edit_string(app, "Cc: ",
          elt->value, sizeof(elt->value), 0);
      break;
    case INFO_TYPE_RECIPIENT:
      r = etpan_cfg_edit_string(app, "Recipient: ",
          elt->value, sizeof(elt->value), 0);
      break;
    case INFO_TYPE_SUBJECT:
      r = etpan_cfg_edit_string(app, "Subject: ",
          elt->value, sizeof(elt->value), 0);
      break;
    case INFO_TYPE_HEADER:
      r = etpan_cfg_edit_string(app, "Header line: ",
          elt->value, sizeof(elt->value), 0);
      break;
    case INFO_TYPE_BODY:
      r = etpan_cfg_edit_string(app, "Body content: ",
          elt->value, sizeof(elt->value), 0);
      break;
    case INFO_TYPE_MESSAGE:
      r = etpan_cfg_edit_string(app, "Message content: ",
          elt->value, sizeof(elt->value), 0);
      break;
    case INFO_TYPE_DATE_SINCE:
    case INFO_TYPE_DATE_BEFORE:
    case INFO_TYPE_DATE_EQUAL:
      r = etpan_cfg_edit_time(app, "Date: ", &elt->date);
      break;
    default:
      r = NO_ERROR;
      break;
    }
    if (r != NO_ERROR) {
      ETPAN_APP_LOG((app->app, "edit value - not enough memory"));
    }
    break;
  }
}

static void display(struct etpan_subapp * app, WINDOW * w)
{
  struct app_state * state;
  unsigned int i;
  unsigned int y;
  unsigned int list_lines;
  char * buffer;
  char * output;
  char * fill;
  int percent;
  unsigned int count;
  
  buffer = app->app->buffer;
  output = app->app->output;
  fill = app->app->fill;
  
  list_lines = app->display_height - 1;
  
  state = app->data;
  
  /* update view */
  
  count = carray_count(state->info_type_tab);
  
  if (state->index > count - 1)
    state->index = count - 1;
  
  if (state->index < state->first)
    state->first = state->index;
  if (state->index - state->first + 1 > (unsigned int) list_lines)
    state->first = state->index - list_lines + 1;
  
  /* display */
  
  wattron(w, state->main_attr);
  y = 0;
  i = state->first;
  while (y < list_lines) {
    struct search_cond * elt;
    char * value;
    char date_str[40];
    struct tm time_date;

    if (i >= count)
      break;
    
    elt = carray_get(state->info_type_tab, i);
    if (elt->value[0] != '\0')
      value = elt->value;
    else
      value = "(not set)";

    switch (elt->type) {
    case INFO_TYPE_DATE_SINCE:
    case INFO_TYPE_DATE_BEFORE:
    case INFO_TYPE_DATE_EQUAL:
      localtime_r(&elt->date, &time_date);
      snprintf(date_str, sizeof(date_str), "%02i/%02i/%i %02i:%02i:%02i",
          time_date.tm_mday, time_date.tm_mon + 1, time_date.tm_year + 1900,
          time_date.tm_hour, time_date.tm_min, time_date.tm_sec);
      break;
    }
    
    switch (elt->type) {
    case INFO_TYPE_OR:
      snprintf(buffer, app->width, "combinaison: or");
      break;
    case INFO_TYPE_AND:
      snprintf(buffer, app->width, "combinaison: and");
      break;
    case INFO_TYPE_RECURSIVE:
      snprintf(buffer, app->width, "recursive search");
      break;
    case INFO_TYPE_NEW:
      if (elt->condition_not)
        snprintf(buffer, app->width, "not new");
      else
        snprintf(buffer, app->width, "new");
      break;
    case INFO_TYPE_SEEN:
      if (elt->condition_not)
        snprintf(buffer, app->width, "unseen");
      else
        snprintf(buffer, app->width, "seen");
      break;
    case INFO_TYPE_FLAGGED:
      if (elt->condition_not)
        snprintf(buffer, app->width, "not flagged");
      else
        snprintf(buffer, app->width, "flagged");
      break;
    case INFO_TYPE_DELETED:
      if (elt->condition_not)
        snprintf(buffer, app->width, "not deleted");
      else
        snprintf(buffer, app->width, "deleted");
      break;
    case INFO_TYPE_ANSWERED:
      if (elt->condition_not)
        snprintf(buffer, app->width, "not answered");
      else
        snprintf(buffer, app->width, "answered");
      break;
    case INFO_TYPE_FORWARDED:
      if (elt->condition_not)
        snprintf(buffer, app->width, "not forwarded");
      else
        snprintf(buffer, app->width, "forwarded");
      break;
    case INFO_TYPE_FROM:
      if (elt->condition_not)
        snprintf(buffer, app->width, "From does not contain %s", value);
      else
        snprintf(buffer, app->width, "From contains %s", value);
      break;
    case INFO_TYPE_TO:
      if (elt->condition_not)
        snprintf(buffer, app->width, "To does not contain %s", value);
      else
        snprintf(buffer, app->width, "To contains %s", value);
      break;
    case INFO_TYPE_CC:
      if (elt->condition_not)
        snprintf(buffer, app->width, "Cc does not contain %s", value);
      else
        snprintf(buffer, app->width, "Cc contains %s", value);
      break;
    case INFO_TYPE_DATE_SINCE:
      snprintf(buffer, app->width, "Date since %s", date_str);
      break;
    case INFO_TYPE_DATE_BEFORE:
      snprintf(buffer, app->width, "Date before %s", date_str);
      break;
    case INFO_TYPE_DATE_EQUAL:
      snprintf(buffer, app->width, "Date is %s", date_str);
      break;
    case INFO_TYPE_RECIPIENT:
      if (elt->condition_not)
        snprintf(buffer, app->width,
            "recipient does not contain %s", value);
      else
        snprintf(buffer, app->width,
            "recipient contains %s", value);
      break;
    case INFO_TYPE_SUBJECT:
      if (elt->condition_not)
        snprintf(buffer, app->width, "Subject does not contain %s", value);
      else
        snprintf(buffer, app->width, "Subject contains %s", value);
      break;
    case INFO_TYPE_HEADER:
      if (elt->condition_not)
        snprintf(buffer, app->width, "Header does not contain %s", value);
      else
        snprintf(buffer, app->width, "Header contains %s", value);
      break;
    case INFO_TYPE_BODY:
      if (elt->condition_not)
        snprintf(buffer, app->width, "Body does not contain %s", value);
      else
        snprintf(buffer, app->width, "Body contains %s", value);
      break;
    case INFO_TYPE_MESSAGE:
      if (elt->condition_not)
        snprintf(buffer, app->width,
            "Message does not contain %s", value);
      else
        snprintf(buffer, app->width,
            "Message contains %s", value);
      break;
    }
    if (i == state->index) {
      wattroff(w, state->main_attr);
      wattron(w, state->selection_attr);
    }
    if (elt->case_sensitive)
      snprintf(output, app->display_width, "%s (case sensitive)%s",
          buffer, fill);
    else
      snprintf(output, app->display_width, "%s%s", buffer, fill);
    mvwprintw(w, y, 0, "%s", output);
    if (i == state->index) {
      wattroff(w, state->selection_attr);
      wattron(w, state->main_attr);
    }
    
    i ++;
    y ++;
  }

  while (y < list_lines) {
    mvwprintw(w, y, 0, "%s", fill);
    y ++;
  }

  wattroff(w, state->main_attr);
  
  /* status bar */

  wattron(w, state->status_attr);
  
  if (count == 0)
    percent = 0;
  else if (count == 1)
	percent = 100;
  else
    percent = state->index * 100 / (count-1);
  
  snprintf(output, app->display_width + 1,
      " %3i %% | y: ok  ^G: cancel%s", percent, fill);
  
  mvwprintw(w, app->display_height - 1, 0, "%s", output);
  
  wattroff(w, state->status_attr);
}

static void set_color(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_app_set_color(app->app, "main",
      &state->main_attr, A_NORMAL);
  etpan_app_set_color(app->app, "selection",
      &state->selection_attr, A_REVERSE);
  etpan_app_set_color(app->app, "status",
      &state->status_attr, A_REVERSE);
}

static int init(struct etpan_subapp * app)
{
  struct app_state * state;
  struct search_cond * elt;
  int r;
  
  state = malloc(sizeof(* state));
  if (state == NULL)
    goto err;
  
  state->index = 0;
  state->first = 0;
  
  elt = search_cond_new(INFO_TYPE_AND);
  if (elt == NULL)
    goto free;
  
  state->info_type_tab = carray_new(16);
  if (state->info_type_tab == NULL) {
    free(elt);
    goto free;
  }
  
  r = carray_add(state->info_type_tab, elt, NULL);
  if (r < 0) {
    free(elt);
    goto free_array;
  }

  state->folder = NULL;
  
  /* colors */
  state->main_attr = A_NORMAL;
  state->selection_attr = A_REVERSE;
  state->status_attr = A_REVERSE;
  
  app->data = state;
  
  return NO_ERROR;

 free_array:
  carray_free(state->info_type_tab);
 free:
  free(state);
 err:
  return ERROR_MEMORY;
}

static void done(struct etpan_subapp * app)
{
  struct app_state * state;

  state = app->data;
  carray_free(state->info_type_tab);
  
  free(state);
}

static int display_init(struct etpan_subapp * app)
{
  etpan_subapp_set_title(app, "etPan! - edit folder");
  return etpan_app_subapp_display_init(app);
}


struct etpan_subapp * etpan_search_app_new(struct etpan_app * app)
{
  return etpan_subapp_new(app, &etpan_search_app_driver);
}

void etpan_search_app_flush(struct etpan_subapp * app)
{
  struct app_state * state;
  unsigned int i;

  state = app->data;
  
  state->folder = NULL;
  
  state->index = 0;
  state->first = 0;
  for(i = 1 ; i < carray_count(state->info_type_tab) ; i ++) {
    struct search_cond * elt;
    
    elt = carray_get(state->info_type_tab, i);
    free(elt);
  }
  carray_set_size(state->info_type_tab, 1);
}

void etpan_search_app_set(struct etpan_subapp * app,
    struct mailfolder * folder)
{
  struct app_state * state;

  state = app->data;
  state->folder = folder;
}

static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app)
{
  /* do nothing */
}

static int reparent(struct mailmessage_tree * env_tree,
    struct mailmessage_tree * node)
{
  unsigned int i;
  int r;

  for(i = 0 ; i < carray_count(node->node_children) ; i ++) {
    struct mailmessage_tree * child;
    
    child = carray_get(node->node_children, i);
    
    r = reparent(env_tree, child);
    if (r != NO_ERROR)
      goto err;
  }
  
  if (node != env_tree) {
    unsigned int old_count;
    
    old_count = carray_count(env_tree->node_children);
    r = carray_set_size(env_tree->node_children, old_count +
        carray_count(node->node_children));
    if (r < 0)
      goto err;
    for(i = 0 ; i < carray_count(node->node_children) ; i ++) {
      struct mailmessage_tree * child;
      
      child = carray_get(node->node_children, i);
      carray_set(env_tree->node_children, old_count + i, child);
    }
    carray_set_size(node->node_children, 0);
  }
  
  return NO_ERROR;
  
 err:
  return ERROR_MEMORY;
}

static int do_finish_filter(struct etpan_subapp * app,
    struct mailmessage_tree * env_tree,
    struct etpan_node_msg_params * node_params)
{
  struct etpan_subapp * msglist_app;
  struct etpan_subapp * parent;
  
  /* reuse a previous interface */
  msglist_app = etpan_app_find_subapp(app->app, "msg-list",
      0, NULL, NULL);
  if (msglist_app == NULL) {
    /* create new */
    msglist_app = etpan_msg_list_app_new(app->app);
    if (msglist_app == NULL) {
      ETPAN_APP_LOG((app->app, "search - not enough memory"));
      etpan_free_msg_list(app, env_tree, node_params);
      return ERROR_MEMORY;
    }
  }
  
  parent = app;
  while (parent->parent != NULL)
    parent = parent->parent;
  
  /* set parent interface */
  etpan_subapp_set_parent(msglist_app, parent);
  
  /* set folder and run */
  etpan_msg_list_app_set_msglist(msglist_app, env_tree, node_params);
  
  etpan_app_quit_subapp(app);
  etpan_app_switch_subapp(msglist_app, 0);
  
  return NO_ERROR;
}

struct recursive_search_state {
  struct mailmessage_tree * env_tree;
  struct etpan_node_msg_params * node_params;
  unsigned int remaining;
};

struct filter_state {
  struct recursive_search_state * rec_search_state;
  
  struct mailfolder * folder;
  struct mailmessage_tree * env_tree;
  
  int resume;
  int cond;
  unsigned int current_node;
  int matched;
  unsigned int current_filter;
  int error_fetch;
  char * header;
  char * body;
  char * message;
  
  /* result */
  struct mailmessage_tree * matched_msg;
  struct mailmessage_tree * not_matched;
};


static int get_filter_fetch_result(struct etpan_subapp * app,
    struct filter_state * filter_state,
    struct etpan_thread_op * op)
{
  char * content;
  int res;
  int r;
  
  switch (op->cmd) {
  case ETPAN_THREAD_MESSAGE_FETCH:
  case ETPAN_THREAD_MESSAGE_FETCH_HEADER:
    {
      struct etpan_message_fetch_result * result;
      MMAPString * mmapstr;
      
      result = op->result;
      
      mmapstr = mmap_string_new_len(result->content, result->len);
      if (mmapstr == NULL) {
        res = ERROR_MEMORY;
        goto free_fetch;
      }
      
      r = mmap_string_ref(mmapstr);
      if (r < 0) {
        res = ERROR_MEMORY;
        goto free_mmap;
      }
      
      free(result);
      
      content = mmapstr->str;
      
      switch (op->cmd) {
      case ETPAN_THREAD_MESSAGE_FETCH:
        filter_state->message = content;
        break;
      case ETPAN_THREAD_MESSAGE_FETCH_HEADER:
        filter_state->header = content;
        break;
      }
      
      return NO_ERROR;
      
    free_mmap:
      mmap_string_free(mmapstr);
    free_fetch:
      etpan_subapp_thread_msg_op_add(app, THREAD_ID_SEARCH_FETCH_RESULT_FREE,
          ETPAN_THREAD_MESSAGE_FETCH_RESULT_FREE,
          op->data.mailaccess.msg, NULL, result->content,
          NULL, NULL, NULL);
      free(result);
      goto err;
    }
    break;
        
  case ETPAN_THREAD_MESSAGE_RENDER_MIME:
    {
      struct etpan_message_render_result * result;
      int fd;
      struct stat stat_info;
      char * mapping;
      MMAPString * mmapstr;
      
      result = op->result;
      
      etpan_text_prop_array_free(result->attr);
      
      fd = fileno(result->f);
      if (fd == -1) {
        res = ERROR_FILE;
        goto unlink;
      }
          
      r = fstat(fd, &stat_info);
      if (r == -1) {
        res = ERROR_FILE;
        goto unlink;
      }
          
      mapping = mmap(NULL, stat_info.st_size,
          PROT_READ, MAP_PRIVATE, fd, 0);
      if (mapping == MAP_FAILED) {
        res = ERROR_FILE;
        goto unlink;
      }
          
      mmapstr = mmap_string_new(mapping);
      if (mmapstr == NULL) {
        res = ERROR_MEMORY;
        goto unmap;
      }
          
      r = mmap_string_ref(mmapstr);
      if (r < 0) {
        res = ERROR_MEMORY;
        goto free_mmapstr;
        break;
      }
      
      munmap(mapping, stat_info.st_size);
      fclose(result->f);
      unlink(result->filename);
      free(result->filename);
      free(result);
      
      content = mmapstr->str;
      
      filter_state->body = content;
      
      return NO_ERROR;
      
    free_mmapstr:
      mmap_string_free(mmapstr);
    unmap:
      munmap(mapping, stat_info.st_size);
    unlink:
      fclose(result->f);
      unlink(result->filename);
      free(result->filename);
      free(result);
      goto err;
    }
    break;
  }
  
  res = ERROR_INVAL;
  
 err:
  return res;
}

static void handle_finish_search(struct etpan_subapp * app,
    struct recursive_search_state * rec_search_state)
{
  struct mailmessage_tree * env_tree;
  struct etpan_node_msg_params * node_params;
  
  /* merging results */
  
  rec_search_state->remaining --;
  
  if (rec_search_state->remaining > 0)
    return;
  
  /* if finished */
  
  env_tree = rec_search_state->env_tree;
  node_params = rec_search_state->node_params;
  free(rec_search_state);
  
  do_finish_filter(app, env_tree, node_params);
}

static int merge_nodes(struct etpan_subapp * app,
    struct recursive_search_state * rec_search_state,
    struct mailfolder * folder,
    struct mailmessage_tree * env_tree)
{
  struct etpan_folder_free_msg_list_arg * arg;
  int r;
  int res;

  /* add nodes */
  
  r = etpan_node_msg_params_add_recursive(rec_search_state->node_params,
      NULL, env_tree);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "search - not enough memory"));
    res = ERROR_MEMORY;
    goto free_list;
  }
  
  if (rec_search_state->env_tree == NULL) {
    rec_search_state->env_tree = env_tree;
  }
  else {
    unsigned int child_count;
    unsigned int  old_child_count;
    unsigned int i;
    
    /* merge nodes */
    
    old_child_count = carray_count(rec_search_state->env_tree->node_children);
    child_count = old_child_count + carray_count(env_tree->node_children);
    
    r = carray_set_size(rec_search_state->env_tree->node_children,
        child_count);
    if (r < 0) {
      ETPAN_APP_LOG((app->app, "search - not enough memory"));
      res = ERROR_MEMORY;
      goto free_list;
    }
    
    for(i = 0 ; i < carray_count(env_tree->node_children) ; i ++) {
      struct mailmessage_tree * node;
      
      node = carray_get(env_tree->node_children, i);
      /* reparent child */
      
      carray_set(rec_search_state->env_tree->node_children,
          old_child_count + i, node);
      node->node_parent = rec_search_state->env_tree;
    }
    carray_set_size(env_tree->node_children, 0);
    mailmessage_tree_free_recursive(env_tree);
  }
  
  return NO_ERROR;
  
 free_list:
  arg = malloc(sizeof(* arg));
  if (arg == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }
  
  arg->node_params = NULL;
  arg->env_tree = env_tree;
  
  etpan_subapp_thread_folder_op_add(app,
      THREAD_ID_SEARCH_FREE_MSG_LIST,
      ETPAN_THREAD_FOLDER_FREE_MSG_LIST,
      folder, NULL, NULL,
      arg,
      NULL, NULL, NULL);
 err:
  return res;
}

enum {
  MSG_MATCHED,
  MSG_RECEIVING,
  MSG_NOT_MATCHED,
};

static void filter_callback(struct etpan_subapp * app,
    struct etpan_thread_op * op, int app_op_type, void * data);

static void filter_handle_cancel(struct etpan_subapp * app,
    struct etpan_thread_op * op, int app_op_type, void * data);

static inline void value_regex_init(struct search_cond * elt)
{
  int flags;
  int r;
  
  if (elt->regex_init)
    return;
  
  flags = REG_EXTENDED | REG_NOSUB;
  
  if (!elt->case_sensitive)
    flags = REG_ICASE;
  
  r = regcomp(&elt->value_regex, elt->value, flags);
  if (r != 0)
    return;
  
  elt->regex_init = 1;
}

static inline int value_regex_match(struct search_cond * elt,
    const char * str)
{
  int r;
  
  value_regex_init(elt);
  if (!elt->regex_init)
    return 0;
  
  r = regexec(&elt->value_regex, str, 0, NULL, 0);
  if (r == 0)
    return 1;
  else
    return 0;
}

static inline int match_mb(struct search_cond * elt,
    struct mailimf_mailbox * mb)
{
  if (mb->mb_display_name != NULL)
    if (value_regex_match(elt, mb->mb_display_name))
      return 1;
  
  return value_regex_match(elt, mb->mb_addr_spec);
}

static inline int match_mb_list(struct search_cond * elt,
    struct mailimf_mailbox_list * mb_list)
{
  clistiter * cur;
  
  for(cur = clist_begin(mb_list->mb_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    if (match_mb(elt, clist_content(cur)))
      return 1;
  }
  
  return 0;
}

static inline int match_group(struct search_cond * elt,
    struct mailimf_group * group)
{
  if (group->grp_display_name != NULL)
    if (value_regex_match(elt, group->grp_display_name))
      return 1;
  
  if (group->grp_mb_list != NULL)
    return match_mb_list(elt, group->grp_mb_list);
  else
    return 0;
}

static inline int match_addr(struct search_cond * elt,
    struct mailimf_address * addr)
{
  if (addr->ad_type == MAILIMF_ADDRESS_MAILBOX)
    return match_mb(elt, addr->ad_data.ad_mailbox);
  else
    return match_group(elt, addr->ad_data.ad_group);
}

static inline int match_addr_list(struct search_cond * elt,
    struct mailimf_address_list * addr_list)
{
  clistiter * cur;
  
  for(cur = clist_begin(addr_list->ad_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    if (match_addr(elt, clist_content(cur)))
      return 1;
  }
  
  return 0;
}

static int match_message(struct etpan_subapp * app,
    struct filter_state * filter_state,
    struct mailmessage_tree * node)
{
  unsigned int j;
  int matched;
  unsigned int current_filter;
  int cond;
  struct app_state * state;
  int r;
  mailmessage * msg;
  
  if (node->node_msg == NULL)
    return MSG_NOT_MATCHED;
  
  msg = node->node_msg;

  state = app->data;
  
  cond = filter_state->cond;
  if (filter_state->resume) {
    filter_state->resume = 0;
    matched = filter_state->matched;
    current_filter = filter_state->current_filter;
  }
  else {
    if (cond == INFO_TYPE_OR)
      matched = 0;
    else
      matched = 1;
    current_filter = 0;
  }
  
  for(j = current_filter ;
      j < carray_count(state->info_type_tab) ; j ++) {
    struct search_cond * elt;
    int cond_matched;
    int got_cond;
    struct mailimf_address_list * addr_list;
    struct mailimf_mailbox_list * mb_list;
    struct mailimf_single_fields single_fields;
    
    mailimf_single_fields_init(&single_fields, msg->msg_fields);
    
    got_cond = 1;
    elt = carray_get(state->info_type_tab, j);
    
    switch (elt->type) {
    case INFO_TYPE_NEW:
      cond_matched = ((msg->msg_flags->fl_flags & MAIL_FLAG_NEW) != 0);
      break;
      
    case INFO_TYPE_SEEN:
      if (msg->msg_flags != NULL)
        cond_matched = ((msg->msg_flags->fl_flags & MAIL_FLAG_SEEN) != 0);
      else
        cond_matched = 0;
      break;
      
    case INFO_TYPE_FLAGGED:
      if (msg->msg_flags != NULL)
        cond_matched = ((msg->msg_flags->fl_flags & MAIL_FLAG_FLAGGED) != 0);
      else
        cond_matched = 0;
      break;
      
    case INFO_TYPE_DELETED:
      if (msg->msg_flags != NULL)
        cond_matched = ((msg->msg_flags->fl_flags & MAIL_FLAG_DELETED) != 0);
      else
        cond_matched = 0;
      break;
      
    case INFO_TYPE_ANSWERED:
      if (msg->msg_flags != NULL)
        cond_matched = ((msg->msg_flags->fl_flags & MAIL_FLAG_ANSWERED) != 0);
      else
        cond_matched = 0;
      break;
      
    case INFO_TYPE_FORWARDED:
      if (msg->msg_flags != NULL)
        cond_matched = ((msg->msg_flags->fl_flags & MAIL_FLAG_FORWARDED) != 0);
      else
        cond_matched = 0;
      break;
      
    case INFO_TYPE_FROM:
      mb_list = NULL;
      if (single_fields.fld_from != NULL)
        mb_list = etpan_dup_mailbox_list(single_fields.fld_from->frm_mb_list);
      
      if (mb_list != NULL) {
        etpan_mailbox_list_decode(app->app, mb_list);
        cond_matched = match_mb_list(elt, mb_list);
        mailimf_mailbox_list_free(mb_list);
      }
      else {
        cond_matched = 0;
      }
      break;
      
    case INFO_TYPE_TO:
      addr_list = NULL;
      if (single_fields.fld_to != NULL)
        addr_list = etpan_dup_address_list(single_fields.fld_to->to_addr_list);
      
      if (addr_list != NULL) {
        etpan_addr_list_decode(app->app, addr_list);
        cond_matched = match_addr_list(elt, addr_list);
        mailimf_address_list_free(addr_list);
      }
      else {
        cond_matched = 0;
      }
      break;
      
    case INFO_TYPE_CC:
      addr_list = NULL;
      if (single_fields.fld_cc != NULL)
        addr_list = etpan_dup_address_list(single_fields.fld_cc->cc_addr_list);
      
      if (addr_list != NULL) {
        etpan_addr_list_decode(app->app, addr_list);
        cond_matched = match_addr_list(elt, addr_list);
        mailimf_address_list_free(addr_list);
      }
      else {
        cond_matched = 0;
      }
      break;
      
    case INFO_TYPE_DATE_EQUAL:
      cond_matched = (node->node_date == elt->date);
      break;
      
    case INFO_TYPE_DATE_SINCE:
      cond_matched = (node->node_date >= elt->date);
      break;
      
    case INFO_TYPE_DATE_BEFORE:
      cond_matched = (node->node_date <= elt->date);
      break;
      
    case INFO_TYPE_RECIPIENT:
      addr_list = NULL;
      if (single_fields.fld_to != NULL)
        addr_list = etpan_dup_address_list(single_fields.fld_to->to_addr_list);
      
      if (addr_list != NULL) {
        etpan_addr_list_decode(app->app, addr_list);
        cond_matched = match_addr_list(elt, addr_list);
        mailimf_address_list_free(addr_list);
      }
      else {
        cond_matched = 0;
      }
      
      addr_list = NULL;
      if (single_fields.fld_cc != NULL)
        addr_list = etpan_dup_address_list(single_fields.fld_cc->cc_addr_list);
      
      if (addr_list != NULL) {
        etpan_addr_list_decode(app->app, addr_list);
        cond_matched = cond_matched || match_addr_list(elt, addr_list);
        mailimf_address_list_free(addr_list);
      }
      break;
      
    case INFO_TYPE_SUBJECT:
      if (single_fields.fld_subject != NULL)
        cond_matched = value_regex_match(elt,
            single_fields.fld_subject->sbj_value);
      else
        cond_matched = 0;
      break;
      
    case INFO_TYPE_HEADER:
      if (!filter_state->error_fetch) {
        if (filter_state->header != NULL) {
          cond_matched = value_regex_match(elt, filter_state->header);
        }
        else {
          r = etpan_subapp_thread_msg_op_add(app,
              THREAD_ID_SEARCH_FETCH,
              ETPAN_THREAD_MESSAGE_FETCH_HEADER,
              node->node_msg, NULL,
              NULL,
              filter_callback, filter_state, filter_handle_cancel);
          if (r == NO_ERROR)
            goto save_state;
          
          cond_matched = 0;
        }
      }
      else {
        cond_matched = 0;
      }
      break;
      
    case INFO_TYPE_BODY:
      if (!filter_state->error_fetch) {
        if (filter_state->body != NULL) {
          cond_matched = value_regex_match(elt, filter_state->body);
        }
        else {
          r = etpan_subapp_thread_msg_op_add(app,
              THREAD_ID_SEARCH_FETCH,
              ETPAN_THREAD_MESSAGE_RENDER_MIME,
              node->node_msg, NULL,
              NULL,
              filter_callback, filter_state, filter_handle_cancel);
          if (r == NO_ERROR)
            goto save_state;
          
          cond_matched = 0;
        }
      }
      else {
        cond_matched = 0;
      }
      break;
      
    case INFO_TYPE_MESSAGE:
      if (!filter_state->error_fetch) {
        if (filter_state->message != NULL) {
          cond_matched = value_regex_match(elt, filter_state->message);
        }
        else {
          r = etpan_subapp_thread_msg_op_add(app,
              THREAD_ID_SEARCH_FETCH,
              ETPAN_THREAD_MESSAGE_FETCH,
              msg, NULL,
              NULL,
              filter_callback, filter_state, filter_handle_cancel);
          if (r == NO_ERROR)
            goto save_state;
          
          cond_matched = 0;
        }
      }
      else {
        cond_matched = 0;
      }
      break;
      
    default:
      got_cond = 0;
      cond_matched = 0;
      break;
    }
    
    if (got_cond) {
      if (elt->regex_init) {
        regfree(&elt->value_regex);
        elt->regex_init = 0;
      }
      
      if (elt->condition_not) {
        switch (elt->type) {
        case INFO_TYPE_DATE_EQUAL:
        case INFO_TYPE_DATE_SINCE:
        case INFO_TYPE_DATE_BEFORE:
        case -1:
          break;
      
        default:
          cond_matched = !cond_matched;
          break;
        }
      }
      
      if (cond == INFO_TYPE_OR) {
        matched = matched || cond_matched;
        if (matched)
          break;
      }
      else {
        matched = matched && cond_matched;
        if (!matched)
          break;
      }
    }
  }
  
  if (matched)
    return MSG_MATCHED;
  else
    return MSG_NOT_MATCHED;
  
 save_state:
  filter_state->resume = 1;
  filter_state->matched = matched;
  filter_state->current_filter = j;
  return MSG_RECEIVING;
}


static void clean_filter(struct etpan_subapp * app,
    struct filter_state * filter_state)
{
  unsigned int i;
  unsigned int j;
  struct etpan_folder_free_msg_list_arg * free_msg_list_arg;
  
  /* free matched msg, not matched msg and env tree if != NULL */
  
  if (filter_state->not_matched != NULL) {
    free_msg_list_arg = malloc(sizeof(* free_msg_list_arg));
    if (free_msg_list_arg != NULL) {
      free_msg_list_arg->env_tree = filter_state->not_matched;
      free_msg_list_arg->node_params = NULL;
    
      etpan_subapp_thread_folder_op_add(app,
          THREAD_ID_SEARCH_FREE_MSG_LIST,
          ETPAN_THREAD_FOLDER_FREE_MSG_LIST,
          filter_state->folder, NULL, NULL,
          free_msg_list_arg,
          NULL, NULL, NULL);
    }
  }
  if (filter_state->matched_msg != NULL) {
    free_msg_list_arg = malloc(sizeof(* free_msg_list_arg));
    if (free_msg_list_arg != NULL) {
      free_msg_list_arg->env_tree = filter_state->matched_msg;
      free_msg_list_arg->node_params = NULL;
    
      etpan_subapp_thread_folder_op_add(app,
          THREAD_ID_SEARCH_FREE_MSG_LIST,
          ETPAN_THREAD_FOLDER_FREE_MSG_LIST,
          filter_state->folder, NULL, NULL,
          free_msg_list_arg,
          NULL, NULL, NULL);
    }
  }
  
  /* remove first childs that are NULL */
  j = 0;
  for(i = 0 ; i < carray_count(filter_state->env_tree->node_children) ; i ++) {
    struct mailmessage_tree * node;
    
    node = carray_get(filter_state->env_tree->node_children, i);
    if (node != NULL) {
      carray_set(filter_state->env_tree->node_children, j, node);
      j ++;
    }
  }
  carray_set_size(filter_state->env_tree->node_children, j);
  
  if (filter_state->env_tree != NULL) {
    free_msg_list_arg = malloc(sizeof(* free_msg_list_arg));
    if (free_msg_list_arg != NULL) {
      free_msg_list_arg->env_tree = filter_state->env_tree;
      free_msg_list_arg->node_params = NULL;
    
      etpan_subapp_thread_folder_op_add(app,
          THREAD_ID_SEARCH_FREE_MSG_LIST,
          ETPAN_THREAD_FOLDER_FREE_MSG_LIST,
          filter_state->folder, NULL, NULL,
          free_msg_list_arg,
          NULL, NULL, NULL);
    }
  }
  
  handle_finish_search(app, filter_state->rec_search_state);
  
  if (filter_state->header != NULL)
    mmap_string_unref(filter_state->header);
  if (filter_state->body != NULL)
    mmap_string_unref(filter_state->body);
  if (filter_state->message != NULL)
    mmap_string_unref(filter_state->message);
  free(filter_state);
}


static void filter_handle_cancel(struct etpan_subapp * app,
    struct etpan_thread_op * op, int app_op_type, void * data)
{
  clean_filter(app, data);
}


static void filter_callback(struct etpan_subapp * app,
    struct etpan_thread_op * op, int app_op_type, void * data)
{
  struct filter_state * filter_state;
  unsigned int i;
  struct app_state * state;
  int cond;
  int r;
  unsigned int current_node;
  int found;
  
  state = app->data;
  
  filter_state = data;
  
  if (op != NULL) {
    free(op->arg);
    
    if (op->err == NO_ERROR) {
      filter_state->error_fetch = 0;
      r = get_filter_fetch_result(app, filter_state, op);
      if (r != NO_ERROR)
        filter_state->error_fetch = 1;
    }
    else {
      filter_state->error_fetch = 1;
    }
  }
  
  found = 0;
  cond = INFO_TYPE_AND;
  for(i = filter_state->current_filter ;
      (i < carray_count(state->info_type_tab)) && found ;
      i ++) {
    struct search_cond * elt;
    
    elt = carray_get(state->info_type_tab, i);
    switch (elt->type) {
    case INFO_TYPE_OR:
      cond = INFO_TYPE_OR;
      found = 1;
      break;
    case INFO_TYPE_AND:
      cond = INFO_TYPE_AND;
      found = 1;
      break;
    }
    
    if (found)
      break;
  }
  
  if (filter_state->resume) {
    current_node = filter_state->current_node;
  }
  else {
    current_node = 0;
  }
  
  filter_state->cond = cond;
  for(i = current_node ;
      i < carray_count(filter_state->env_tree->node_children) ; i ++) {
    struct mailmessage_tree * node;
    int matched;
    
    ETPAN_APP_DEBUG((app->app, "matching %i", i));
    node = carray_get(filter_state->env_tree->node_children, i);
    
    r = match_message(app, filter_state, node);
    switch (r) {
    case MSG_RECEIVING:
      filter_state->current_node = i;
      return;
    case MSG_MATCHED:
      matched = 1;
      break;
    case MSG_NOT_MATCHED:
      matched = 0;
      break;
    default:
      matched = 0;
      break;
    }
    
    if (filter_state->header != NULL) {
      mmap_string_unref(filter_state->header);
      filter_state->header = NULL;
    }
    if (filter_state->body != NULL) {
      mmap_string_unref(filter_state->body);
      filter_state->body = NULL;
    }
    if (filter_state->message != NULL) {
      mmap_string_unref(filter_state->message);
      filter_state->message = NULL;
    }
    
    if (matched) {
      r = carray_add(filter_state->matched_msg->node_children, node, NULL);
      if (r < 0) {
        goto finish;
      }
      
      carray_set(filter_state->env_tree->node_children, i, NULL);
      node->node_parent = filter_state->matched_msg;
    }
    else {
      r = carray_add(filter_state->not_matched->node_children, node, NULL);
      if (r < 0) {
        goto finish;
      }
      
      carray_set(filter_state->env_tree->node_children, i, NULL);
      node->node_parent = filter_state->not_matched;
    }
  }
  
  r = merge_nodes(app, filter_state->rec_search_state,
      filter_state->folder, filter_state->matched_msg);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "search - not enough memory"));
    goto finish;
  }
  filter_state->matched_msg = NULL;
  
 finish:
  clean_filter(app, filter_state);
}

static int do_filter_msg(struct etpan_subapp * app,
    struct mailfolder * folder,
    struct mailmessage_tree * env_tree,
    struct recursive_search_state * rec_search_state)
{
  struct filter_state * filter_state;
  int r;
  
  /* reparent all nodes to root */
  r = reparent(env_tree, env_tree);
  if (r != NO_ERROR)
    goto err;
    
  filter_state = malloc(sizeof(* filter_state));
  if (filter_state == NULL)
    goto err;
  
  filter_state->folder = folder;
  filter_state->env_tree = env_tree;
  filter_state->current_node = 0;
  filter_state->resume = 0;
  filter_state->matched = 0;
  filter_state->current_filter = 0;
  filter_state->error_fetch = 0;
  filter_state->header = NULL;
  filter_state->body = NULL;
  filter_state->message = NULL;
  filter_state->rec_search_state = rec_search_state;
  filter_state->cond = 0;
  
  /* result */
  filter_state->matched_msg = 
    mailmessage_tree_new(NULL, (time_t) -1, NULL);
  if (filter_state->matched_msg == NULL)
    goto free_state;
  
  filter_state->not_matched = mailmessage_tree_new(NULL, (time_t) -1, NULL);
  if (filter_state->not_matched == NULL)
    goto free_matched;
  
  filter_callback(app, NULL, -1, filter_state);
  
  return NO_ERROR;
  
 free_matched:
  mailmessage_tree_free(filter_state->matched_msg);
 free_state:
  free(filter_state);
 err:
  return ERROR_MEMORY;
}


static void get_msg_list_callback(struct etpan_subapp * app,
    struct etpan_thread_op * op, int app_op_type, void * data)
{
  struct etpan_folder_get_msg_list_result * result;
  struct mailmessage_tree * env_tree;
  struct etpan_node_msg_params * node_params;
  struct recursive_search_state * rec_search_state;
  int r;
  
  rec_search_state = data;
  
  /* if error ? */
  if (op->err != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "search - could not fetch"));
    rec_search_state->remaining --;
    goto test_finished;
  }
  
  result = op->result;
  env_tree = result->env_tree;
  node_params = result->node_params;
  free(result);

  etpan_node_msg_params_free(node_params);
  
  /* filter messages */
  
  r = do_filter_msg(app, op->data.mailaccess.folder,
      env_tree, rec_search_state);
  return;
  
 test_finished:
  handle_finish_search(app, rec_search_state);
}

static int do_recursive_search(struct etpan_subapp * app,
    struct mailfolder * folder,
    int recursive, struct recursive_search_state * search_state,
    int * pcount)
{
  int r;
  int do_poll;
  
  do_poll = etpan_vfolder_do_poll(app->app->config.vfolder_config, folder);
  if (do_poll || (!recursive)) {
    if (folder->fld_storage != NULL) {
      r = etpan_subapp_thread_folder_op_add(app, THREAD_ID_SEARCH_GET_MSG_LIST,
          ETPAN_THREAD_FOLDER_GET_MSG_LIST,
          folder, NULL, NULL,
          NULL,
          get_msg_list_callback, search_state,
          NULL);
      if (r != NO_ERROR) {
        ETPAN_APP_LOG((app->app, "search - not enough memory"));
        return r;
      }
      
      (* pcount) ++;
    }
  }
  
  if (recursive) {
    unsigned int i;
    
    for(i = 0 ; i < carray_count(folder->fld_children) ; i ++) {
      struct mailfolder * child;
      
      child = carray_get(folder->fld_children, i);
      r = do_recursive_search(app, child, recursive,
          search_state, pcount);
      if (r != NO_ERROR)
        return r;
    }
  }
  
  return NO_ERROR;
}

static int do_search(struct etpan_subapp * app)
{
  struct app_state * state;
  struct mailfolder * folder;
  int recursive;
  struct recursive_search_state * search_state;
  unsigned int i;

  state = app->data;
  folder = state->folder;
  
  recursive = 0;
  for(i = 0 ; i < carray_count(state->info_type_tab) ; i ++) {
    struct search_cond * elt;
      
    elt = carray_get(state->info_type_tab, i);
    switch (elt->type) {
    case INFO_TYPE_RECURSIVE:
      recursive = 1;
      break;
    }
  }
  
  if ((!recursive) && (folder->fld_storage == NULL)) {
    ETPAN_APP_LOG((app->app, "no folder to search, try to add recursive condition"));
    return ERROR_INVAL;
  }
  
  search_state = malloc(sizeof(* search_state));
  if (search_state == NULL)
    goto err;
  
  search_state->env_tree = NULL;
  search_state->node_params = etpan_node_msg_params_new();
  if (search_state->node_params == NULL)
    goto free;
  
  search_state->remaining = 0;
  do_recursive_search(app, folder, recursive, search_state,
      &search_state->remaining);
  
  return NO_ERROR;
  
 free:
  free(search_state);
 err:
  ETPAN_APP_LOG((app->app, "search - not enough memory"));
  return ERROR_MEMORY;
}

static void add_search_cond(struct etpan_subapp * app, int type)
{
  struct app_state * state;
  int added;
  struct search_cond * elt;
  int r;
  
  state = app->data;
  
  added = 0;
  elt = search_cond_new(type);
  if (elt != NULL) {
    r = carray_add(state->info_type_tab, elt, NULL);
    if (r == 0)
      added = 1;
    else
      free(elt);
  }
  if (!added) {
    ETPAN_APP_LOG((app->app, "edit value - not enough memory"));
  }
}



#define HELP_TEXT \
"\
Help for search\n\
---------------\n\
\n\
This application will let you search messages in a folder.\n\
\n\
- up, down\n\
  arrow keys : move cursor\n\
\n\
- left, right\n\
  arrow keys : change condition type\n\
- a          : add condition\n\
- s          : add condition on Subject header\n\
- f          : add condition on From header\n\
- t          : add condition on To header\n\
- r          : add condition on recipient\n\
- Shift-R    : do recursive match\n\
- d          : delete selected condition\n\
\n\
- !          : negative condition\n\
- i          : case sensitive\n\
- [Enter]    : edit value\n\
\n\
- y          : finished edition of properties\n\
- Ctrl-G     : cancel\n\
\n\
- ?          : help\n\
- Ctrl-L     : Console log\n\
\n\
(? or q to exit help)\n\
"

static int show_help(struct etpan_subapp * app)
{
  return etpan_show_help(app, HELP_TEXT, sizeof(HELP_TEXT) - 1);
}
