/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-storage-sel.c,v 1.9 2004/11/12 13:57:02 hoa Exp $
 */

#include "etpan-storage-sel.h"
#include "etpan-subapp.h"
#include "etpan-app.h"
#include "etpan-app-subapp.h"
#include "etpan-errors.h"
#include "etpan-tools.h"
#include "etpan-cfg-storage.h"
#include "etpan-cfg-common.h"
#include <stdlib.h>
#include "etpan-thread-manager.h"
#include "etpan-subapp-thread.h"
#include "etpan-storage-common.h"
#include "etpan-storage-edit.h"
#include "etpan-help-viewer.h"

static void handle_fd(struct etpan_subapp * app, fd_set * fds);
static void handle_key(struct etpan_subapp * app, int key);
static void display(struct etpan_subapp * app, WINDOW * w);
static void set_color(struct etpan_subapp * app);
static int init(struct etpan_subapp * subapp);
static void done(struct etpan_subapp * subapp);
static int display_init(struct etpan_subapp * app);
static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app);

static struct etpan_subapp_driver etpan_storage_sel_app_driver = {
  .name = "storage-sel",
  .always_handle_key = 0,
  .always_on_top = 1,
  .get_idle_delay = NULL,
  .idle = NULL,
  .set_fd = NULL,
  .handle_fd = handle_fd,
  .handle_key = handle_key,
  .handle_resize = NULL,
  .display = display,
  .set_color = set_color,
  .init = init,
  .done = done,
  .enter = NULL,
  .leave = leave,
  .display_init = display_init,
  .display_done = NULL,
};

struct app_state {
  struct etpan_storage_common_state common_state;

  /* storage to replace */
  int do_replace;
  unsigned int replace_index;
  struct mailstorage * replace_storage;
  
  /* upcall */
  void (* upcall)(struct etpan_subapp *, int, void *);
  void * upcall_data;
};

static void handle_fd(struct etpan_subapp * app, fd_set * fds)
{
  etpan_subapp_thread_handle_fd(app, fds);
}

static void display(struct etpan_subapp * app, WINDOW * w)
{
  struct app_state * state;

  state = app->data;

  etpan_storage_common_display(app, &state->common_state, w,
      "[enter]: select  ^G: cancel");
}

static int init(struct etpan_subapp * subapp)
{
  struct app_state * state;
  int r;
  
  state = malloc(sizeof(* state));
  if (state == NULL)
    goto err;
  
  r = etpan_storage_common_init(&state->common_state);
  if (r != NO_ERROR)
    goto free;
  
  state->do_replace = 0;
  state->replace_index = 0;

  state->upcall = NULL;
  state->upcall_data = NULL;
  
  subapp->data = state;
  
  return NO_ERROR;
  
 free:
  free(state);
 err:
  return ERROR_MEMORY;
}


struct etpan_subapp * etpan_storage_sel_app_new(struct etpan_app * app)
{
  return etpan_subapp_new(app, &etpan_storage_sel_app_driver);
}

void etpan_storage_sel_app_flush(struct etpan_subapp * app)
{
  struct app_state * state;

  state = app->data;

  etpan_storage_common_flush(&state->common_state);
}

static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app)
{
  etpan_storage_sel_app_flush(app);
}

int etpan_storage_sel_app_set(struct etpan_subapp * app,
    struct etpan_app_config * config,
    struct mailstorage * storage,
    void (* upcall)(struct etpan_subapp *, int, void *),
    void * upcall_data)
{
  struct app_state * state;
  int r;

  state = app->data;
  
  r = etpan_storage_common_set(&state->common_state, config, storage);
  if (r != NO_ERROR)
    return r;
  
  state->upcall = upcall;
  state->upcall_data = upcall_data;
  
  return NO_ERROR;
}

static int delete_storage(struct etpan_subapp * app);

static int add_storage(struct etpan_subapp * app);

static int edit_storage(struct etpan_subapp * app);

static int show_help(struct etpan_subapp * app);

static void handle_key(struct etpan_subapp * app, int key)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_storage_common_handle_key(app, &state->common_state, key);

  switch (key) {
  case KEY_F(1):
  case '?':
    show_help(app);
    break;

  case KEY_CTRL('G'):
    if (state->upcall != NULL)
      state->upcall(app, ETPAN_STORAGE_SEL_CANCEL, state->upcall_data);
    break;
    
  case '\n':
    if (state->upcall != NULL)
      state->upcall(app, ETPAN_STORAGE_SEL_VALID, state->upcall_data);
    break;
    
  case 'e':
    edit_storage(app);
    break;

  case 'a':
    add_storage(app);
    break;
    
  case 'd':
    delete_storage(app);
    break;
  }
}

static int display_init(struct etpan_subapp * app)
{
  etpan_subapp_set_title(app, "etPan! - select a storage");
  return etpan_app_subapp_display_init(app);
}

static void set_color(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_storage_common_set_color(app, &state->common_state);
}

static void done(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_storage_common_done(&state->common_state);
  free(state);
}



struct mailstorage *
etpan_storage_sel_app_get_storage(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;

  return etpan_storage_common_get_selected_storage(&state->common_state);
}

#if 0
static int edit_storage_conf(struct etpan_subapp * app)
{
  struct etpan_subapp * edit_app;
  struct app_state * state;
  
  state = app->data;
  
  edit_app = etpan_app_find_subapp(app->app, "storage-conf",
      0, NULL, NULL);
  if (edit_app == NULL)
    edit_app = etpan_app_find_subapp(app->app, "storage-conf",
        1, NULL, NULL);
  if (edit_app == NULL)
    edit_app = etpan_storage_conf_app_new(app->app);
  if (edit_app == NULL)
    return ERROR_MEMORY;
  
  etpan_subapp_set_parent(edit_app, app);
  
  etpan_storage_conf_app_set(edit_app, state->common_state.config);
  
  etpan_app_switch_subapp(edit_app, 0);
  
  etpan_storage_sel_app_flush(app);
  
  return NO_ERROR;
}
#endif


static void delete_storage_callback(struct etpan_subapp * app,
    struct etpan_thread_op * op,
    int app_op_type, void * data)
{
  struct mailstorage * storage;
  unsigned int i;
  int r;
  struct app_state * state;
  struct etpan_storage_config * storage_config;
  
  state = app->data;
  
  storage = op->arg;
  
  ETPAN_APP_DEBUG((app->app, "callback delete storage"));
  
  etpan_storage_config_delete(state->common_state.config->storage_config,
      storage);
  
  r = etpan_storage_common_set(&state->common_state,
      state->common_state.config, NULL);
}


static int delete_storage(struct etpan_subapp * app)
{
  struct app_state * state;
  struct mailstorage * storage;
  int count;
  int r;

  state = app->data;
  
  storage = etpan_storage_common_get_selected_storage(&state->common_state);
  if (storage == NULL) {
    ETPAN_APP_LOG((app->app, "no storage selected"));
    return ERROR_INVAL;
  }
  
  count = etpan_storage_common_get_storage_ref_count(&state->common_state,
      storage);
  
  if (count != 0) {
    ETPAN_APP_LOG((app->app, "remaining folders are using the storage"));
    return ERROR_INVAL;
  }
  
  ETPAN_APP_DEBUG((app->app, "stopping thread"));
  r = etpan_subapp_thread_op_add(app,
      THREAD_ID_STORAGESEL_STOP_STORAGE_THREAD,
      ETPAN_THREAD_STOP_STORAGE_THREAD,
      NULL, NULL,
      NULL, NULL, storage,
      delete_storage_callback, NULL,
      NULL);
  
  return NO_ERROR;
}


static int open_storage_editor(struct etpan_subapp * app,
    struct mailstorage * storage);

static int add_storage(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;

  state->do_replace = 0;
  
  return open_storage_editor(app, NULL);
}


static int edit_storage(struct etpan_subapp * app)
{
  struct app_state * state;
  unsigned int i;
  struct mailstorage * storage;
  struct etpan_storage_config * storage_config;
  
  state = app->data;
  
  storage = etpan_storage_common_get_selected_storage(&state->common_state);
  if (storage == NULL)
    return ERROR_INVAL;
  
  storage_config = state->common_state.config->storage_config;
  for(i = 0 ; i < carray_count(storage_config->storage_tab) ; i ++) {
    if (carray_get(storage_config->storage_tab, i) == storage) {
      state->do_replace = 1;
      state->replace_index = i;
      break;
    }
  }
  
  return open_storage_editor(app, storage);
}



static void
recursive_folder_replace_storage(struct mailfolder * folder,
    struct mailstorage * old_storage,
    struct mailstorage * new_storage)
{
  unsigned int i;
  
  if (folder->fld_storage != NULL) {
    if (folder->fld_storage == old_storage)
      folder->fld_storage = new_storage;
  }
  
  for(i = 0 ; i < carray_count(folder->fld_children) ; i ++) {
    struct mailfolder * child;
    
    child = carray_get(folder->fld_children, i);
    recursive_folder_replace_storage(child, old_storage, new_storage);
  }
}

static void edit_storage_callback(struct etpan_subapp * app,
    struct etpan_thread_op * op,
    int app_op_type, void * data)
{
  struct app_state * state;
  struct mailstorage * old_storage;
  struct mailstorage * storage;
  clist * old_folders;
  int r;
  struct etpan_subapp * storage_editor_app;
  
  state = app->data;
  storage = state->replace_storage;
  old_storage = op->arg;
  storage_editor_app = data;
  
  recursive_folder_replace_storage(state->common_state.config->
      vfolder_config->root,
      old_storage, storage);
  
  /* exchange folders */
  old_folders = old_storage->sto_shared_folders;
  old_storage->sto_shared_folders = storage->sto_shared_folders;
  storage->sto_shared_folders = old_folders;
  
  r = etpan_storage_config_replace(state->common_state.config->storage_config,
      state->replace_index, storage, 0);
  
  r = etpan_storage_common_set(&state->common_state,
      state->common_state.config, storage);
  /* ignore errors */
  
  etpan_app_quit_subapp(storage_editor_app);
}

static void
storage_editor_upcall(struct etpan_subapp * storage_editor_app,
    int valid, void * data)
{
  struct mailstorage * storage;
  struct etpan_subapp * app;
  struct app_state * state;
  int r;
  
  app = data;
  state = app->data;
  
  if (valid == ETPAN_STORAGE_EDIT_CANCEL) {
    etpan_app_quit_subapp(storage_editor_app);
    return;
  }
  
  storage = etpan_storage_editor_app_get_storage(storage_editor_app);
  if (storage == NULL)
    goto err;
  
  if (state->do_replace) {
    struct mailstorage * old_storage;
#if 0
    clist * old_folders;
#endif
    
    old_storage =
      carray_get(state->common_state.config->storage_config->storage_tab,
          state->replace_index);
    
    state->replace_storage = storage;
    ETPAN_APP_DEBUG((app->app, "stopping thread"));
    r = etpan_subapp_thread_op_add(app,
        THREAD_ID_STORAGESEL_STOP_STORAGE_THREAD,
        ETPAN_THREAD_STOP_STORAGE_THREAD,
        NULL, NULL,
        NULL, NULL, old_storage,
        edit_storage_callback, storage_editor_app,
        NULL);
    
    return;
  }
  
  r = etpan_storage_config_add(state->common_state.config->storage_config,
      storage, 0);
  if (r < 0) {
      mailstorage_free(storage);
      goto err;
  }
  
  r = etpan_storage_common_set(&state->common_state,
      state->common_state.config, storage);
  /* ignore errors */
  
  etpan_app_quit_subapp(storage_editor_app);
  return;
  
 err:
  ETPAN_APP_LOG((app->app, "storage editor - could not create storage"));
  etpan_app_quit_subapp(storage_editor_app);
  return;
}


static int open_storage_editor(struct etpan_subapp * app,
    struct mailstorage * storage)
{
  struct etpan_subapp * storage_app;
  
  storage_app = etpan_app_find_subapp(app->app, "storage-edit",
      0, NULL, NULL);
  if (storage_app == NULL)
    storage_app = etpan_storage_editor_app_new(app->app);
  if (storage_app == NULL)
    return ERROR_MEMORY;
  
  etpan_storage_editor_app_set(storage_app, storage,
      storage_editor_upcall, app);
  
  etpan_subapp_set_parent(storage_app, app);
  etpan_app_switch_subapp(storage_app, 0);
  
  return NO_ERROR;
}


#define HELP_TEXT \
"\
Help for storage selection dialog\n\
---------------------------------\n\
\n\
This application will let you select an storage.\n\
\n\
- arrow keys : scroll\n\
\n\
- a          : add a storage\n\
- e          : edit storage properties\n\
- d          : delete a storage\n\
\n\
- [Enter]    : select account\n\
- Ctrl-G     : cancel\n\
\n\
- ?          : help\n\
- Ctrl-L     : Console log\n\
\n\
(? or q to exit help)\n\
"

static int show_help(struct etpan_subapp * app)
{
  return etpan_show_help(app, HELP_TEXT, sizeof(HELP_TEXT) - 1);
}
