/*************************************************************************
 *
 *	GTK Euler
 *
 *************************************************************************/

#include <gtk/gtk.h>

#include "pixmaps/new.xpm"
#include "pixmaps/open.xpm"
#include "pixmaps/save.xpm"
#include "pixmaps/copy.xpm"
#include "pixmaps/paste.xpm"
#include "pixmaps/dlg_warning.xpm"
#include "pixmaps/dlg_question.xpm"
#include "pixmaps/logo.xpm"
#include "pixmaps/icon.xpm"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/times.h>
#include <time.h>
#include <dirent.h>
#include <fnmatch.h>

#ifdef RS6000
#include <sys/time.h>
void gettimer (int, struct timestruc_t *);
#endif

#include "sysdep.h"
#include "mainloop.h"
#include "help.h"
#include "rc.h"
#include "term.h"
#include "metagtk.h"
#include "metaps.h"
#include "colbut.h"

/*-----------------------------------------------------------------------
 *	Global variables
 *----------------------------------------------------------------------*/

static int is_demo = 0;
static int in_text = 1;

static EulerPrefs prefs;
static GtkWidget *term_window = NULL;
static GtkWidget *term;
static GtkWidget *meta_window = NULL;
static GtkWidget *meta;


/*-----------------------------------------------------------------------
 *	Euler callbacks
 *----------------------------------------------------------------------*/

/* text_mode
 *	Switch to text. Graphics should not be deleted.
 *	On a window system make text visible.
 */
void text_mode (void)
{
	if (!in_text)
	{
		in_text=1;
		gdk_window_raise(term_window->window);
	}
}


/* graphic_mode
 *	Switch to graphics. Text must not be deleted.
 *	On a window system make graphics visible.
 */
void graphic_mode (void)
{
	if (in_text)
	{
		in_text=0;
		gdk_window_raise(meta_window->window);
	}
}

/************************************************************************
 *
 *	Graphic window
 *
 *		The graphic screen has coordinates from 0.0 to 1024.0 (double).
 *		There should be a function, which computes the correct screen
 *		coordinates from these internal ones.
 *
 ************************************************************************/
int usecolors=1;

// getpixelsize
//	Compute the size of pixel in screen coordinates.
//
void getpixelsize (double *x, double *y)
{
	*x=1024.0/meta->allocation.width;
	*y=1024.0/meta->allocation.height;
}


// gflush
//	Flush out remaining graphic commands (for multitasking).
//	This serves to synchronize the graphics on multitasking systems.
//
void gflush (void)
{
}


//
//			S O U N D   W A V E S
//
#ifdef WAVES
void sys_playwav (char *file)
{
}
#endif

//
//			G R A P H I C   I N P U T
//

// mouse
//	wait, until the user marked a screen point with the mouse.
//	Return screen coordinates.
//
void mouse (double *x, double *y)
{
	while (1)
	{
		if (gdk_events_pending())
		{
			GtkMeta *m = GTK_META(meta);
			
			gtk_main_iteration();
			
			if (m->x>0.0 && m->y>0.0)
			{	
				*x=m->x;
				*y=m->y;
				m->x = m->y = -1.0;
				return;
			}
			else
			{
				GtkTerm *t = GTK_TERM(term);
				if (t->scan==escape)
				{
					*x=-1.0; *y=-1.0; return;
				}
				t->scan = t->code = 0;
			}
		}
		else usleep(1);
	}
}

static int notekey=0;

// test_key
//	see, if user pressed the keyboard.
//	return the scancode, if he did.
//
int test_key ()
{
	while (gtk_events_pending())
	{
		GtkTerm *t = GTK_TERM(term);

		gtk_main_iteration();

		if (t->scan==escape) {
			t->scan = t->code = 0;
			return escape;
		} else {
			if (t->scan) notekey=t->scan;
			else notekey=t->code;
			t->scan = t->code = 0;
			return 0;
		}
	}
	return 0;
}

int test_code (void)
{
	int key=0;
	
	if (notekey)
	{	key=notekey;
		notekey=0;
		return key;
	}
	while (gtk_events_pending())
	{
		GtkTerm *t=GTK_TERM(term);
	
		gtk_main_iteration();
		if (t->scan) {
			key = t->scan;
			t->scan = t->code = 0;
			return key;
		} else if (t->code) {
			key = t->code;
			t->scan = t->code = 0;
			return key;
		}
	}
	return 0;
}

/************************************************************************
 *
 *	Text Window
 *
 *		The following text screen commands should be emulated on a graphic
 *		work station. This can be done by a standard emulator (e.g. VT100)
 *		or within a window displaying the text. Additional features may be
 *		added, such as viewing old text. But the input line should be
 *		visible as soon as a key is pressed by the user.
 *
 ************************************************************************/

// Clear the text screen
//
void clear_screen (void)
{
	gtk_term_clear(term);
}

//	Print a line onto the text screen, parse tabs and '\n'.
//	Printing should be done at the cursor position. There is no need
//	to clear the line at a '\n'.
//	The cursor should move forward with the print.
//	Think of the function as a simple emulator.
//	If you have a line buffered input with echo then do not print,
//	when the command line is on.
void gprint (char *s) /* print an output text (no newline) */
{
	gtk_term_print(term,s);
}

//
//	wait for a keystroke. return the scancode and the ascii code.
//	scancode should be a code from scantyp. Do at least generate
//	'enter'.
//
int wait_key (int *scan)
{
	int key;
	GtkTerm *t = GTK_TERM(term);
	
	t->scan = t->code = 0;
	*scan=0;
	while (1) {
		if (gtk_events_pending())
		{
			gtk_main_iteration();
			if (t->scan || t->code) break;
			t->scan = t->code = 0;
		}
		else usleep(1);
	}
	*scan = t->scan;
	key = t->code;
	t->scan = t->code = 0;
	return key;
}

//
// the command line is active (edit.c)
//
void edit_on (void)
{
	gtk_term_edit_on(term);
}

// the command line is no longer in use (graphics or computing) (edit.c)
//
void edit_off (void)
{
	gtk_term_edit_off(term);
}

/************************************************************************
 *
 *	System
 *
 ************************************************************************/

static char path[256];

// sets the path if dir!=0 and returns the path
//
char *cd (char *dir)
{
	chdir(dir);
	if (getcwd(path,256)) return path;
	return dir;
}


static void shellsort(char **buf, int n)
{
	int h, i, j;
	int seq[28];
	int p1=1, p2=1, p3=1, s=-1;

	/*
	 * establish increment sequence (see Knuth, Vol 3)
	 */
	do {
		if (++s % 2) {
			seq[s] = 8*p1 - 6*p2 + 1;
		} else {
			seq[s] = 9*p1 - 9*p3 + 1;
			p2 *= 2;
			p3 *= 2;
		}
		p1 *= 2;
	} while(3*seq[s] < n);

	s = s > 0 ? --s : 0;
	
	/*
	 *	sort
	 */
	while (s >= 0) {
		/* sort-by-insertion in increments of h */
		h = seq[s--];
		for (i = h; i<n; i++) {
			char *t = buf[i];
			for (j = i-h; j >= 0 && strcmp(buf[j], t)>0; j -= h)
				buf[j+h] = buf[j];
			buf[j+h] = t;
		}
	}
}

static int match (char *pat, char *s)
{	if (*pat==0) return *s==0;
	if (*pat=='*')
	{	pat++;
		if (!*pat) return 1;
		while (*s)
		{	if (match(pat,s)) return 1;
			s++;
		}
		return 0;
	}
	if (*s==0) return 0;
	if (*pat=='?') return match(pat+1,s+1);
	if (*pat!=*s) return 0;
	return match(pat+1,s+1);
}

/*
 *	scan a directory and get :
 *		files : an array of entries matching the pattern
 *		files_count : number of files entries
 *	the function returns the max length of a file entry
 */
int scan_dir(char *dir_name, char *pat, char ** files[], int *files_count)
{
	DIR *dir;
	struct dirent *entry;
	int entry_count=0, len=0;
	char **buf = NULL;

	dir = opendir(dir_name);
	
	if (dir) {
		while((entry = readdir(dir)) != NULL) {
			if (match(pat,entry->d_name)) {
				int l=strlen(entry->d_name);
				len = MAX(len,l);
				entry_count ++;
				buf = (char**)realloc(buf,entry_count*sizeof(char *));
				buf[entry_count-1]=(char*)malloc(l+1);
				strcpy(buf[entry_count-1],entry->d_name);
			}
		}

		closedir(dir);
		
		shellsort(buf, entry_count);
	}	

	*files = buf;
	*files_count = entry_count;
	
	return len;
}


// execute
//	Call an external program, return 0, if there was no error.
//	No need to support this on multitasking systems.
//
int execute (char *dir, char *progname)
{
	return 0;
}

// shrink
//	allows shrinking of memory for single task systems.
//	simply return 1 if you do not support this or set NOSHRINK in funcs.c
//
int shrink (size_t size)
{
	return 1;
}


// myclock
//	define a timer in seconds.
//
double myclock (void)
{
#ifdef RS6000
	struct timestruc_t t;
	gettimer(TIMEOFDAY,&t);
	return (t.tv_sec+t.tv_nsec/1000000000.0);
#else
	return ((double)(times(NULL)))/CLK_TCK;
#endif
}

// sys_wait
//	Wait for time seconds or until a key press.
//	Return the scan code or 0 (time exceeded).
//
void sys_wait (double delay, int *scan)
{
	double now;
	GtkTerm *t = GTK_TERM(term);
	now=myclock();
	*scan=0;
	t->scan = t->code = 0;
	while (myclock()<now+delay)
	{
		while (gtk_events_pending())
		{
			gtk_main_iteration();
			if (t->scan) {
				*scan = notekey = t->scan;
				t->scan = t->code = 0;
			} else if (t->code) {
				*scan = notekey = t->code;
				t->scan = t->code = 0;
			}
		}
		if (*scan==switch_screen)
		{	if (in_text)
			{	graphic_mode();
				wait_key(scan);
				text_mode();
			}
			else
			{	text_mode();
				wait_key(scan);
				graphic_mode();
			}
			*scan=0;
		}
		if (*scan) return;
		usleep(1);
	}
	*scan=0;
}


/***** stack_init
 *	get memory for stack.
 *****/
static int stack_init (long size)
{
	ramstart = (char *)malloc(size*1024l);
	if (!ramstart) return 0;
	ramend = ramstart+size*1024l;
	return 1;
}

/*-----------------------------------------------------------------------
 *	GTK things
 *----------------------------------------------------------------------*/

static GtkWidget * new_gtk_pixmap(GtkWidget *win, gchar **xpm_data)
{
	GdkPixmap *pixmap;
	GdkBitmap *mask;
	GdkColor   alpha;
	GtkWidget *icon;
	
	pixmap = gdk_pixmap_create_from_xpm_d(win->window,&mask,&alpha,xpm_data);
	icon = gtk_pixmap_new(pixmap,mask);
	gdk_pixmap_unref(pixmap);
	gdk_pixmap_unref(mask);
	return icon;
}

static void file_select(gchar *title, char *pattern, GtkSignalFunc func)
{
	GtkFileSelection *file_selector = (GtkFileSelection*)gtk_file_selection_new(title);
	
	if (path) gtk_file_selection_complete(file_selector,pattern);
	gtk_file_selection_hide_fileop_buttons(file_selector);
	gtk_signal_connect_object(GTK_OBJECT(file_selector->ok_button),"clicked",GTK_SIGNAL_FUNC(func),GTK_OBJECT(file_selector));
	gtk_signal_connect_object(GTK_OBJECT(file_selector->ok_button),"clicked",GTK_SIGNAL_FUNC(gtk_widget_destroy),GTK_OBJECT(file_selector));
	gtk_signal_connect_object(GTK_OBJECT(file_selector->cancel_button),"clicked",GTK_SIGNAL_FUNC(gtk_widget_destroy),GTK_OBJECT(file_selector));
	gtk_window_set_transient_for(GTK_WINDOW(file_selector),GTK_WINDOW(term_window));
	gtk_window_set_modal(GTK_WINDOW(file_selector),TRUE);
	gtk_window_set_position(GTK_WINDOW(file_selector),GTK_WIN_POS_MOUSE);
	gtk_widget_show(GTK_WIDGET(file_selector));
}

static void infobox(char *text)
{
	GtkWidget *	dialog;
	GtkWidget *	hbox;
	GtkWidget * icon;
	GtkWidget * label;
	GtkWidget *	button;
	
	dialog = gtk_dialog_new();
	gtk_window_set_title(GTK_WINDOW(dialog),"");
	gtk_window_set_transient_for(GTK_WINDOW(dialog),GTK_WINDOW(term_window));
	gtk_window_set_modal(GTK_WINDOW(dialog),TRUE);
	gtk_window_set_position(GTK_WINDOW(dialog),GTK_WIN_POS_MOUSE);
	gtk_signal_connect(GTK_OBJECT(dialog),"delete_event",GTK_SIGNAL_FUNC(gtk_widget_destroy),NULL);
	gtk_signal_connect(GTK_OBJECT(dialog),"destroy",GTK_SIGNAL_FUNC(gtk_widget_destroy),NULL);
	gtk_widget_realize(dialog);
	
	hbox = gtk_hbox_new(FALSE,0);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),hbox,TRUE,TRUE,0);

	icon = new_gtk_pixmap(dialog,dlg_warning_xpm);
	gtk_box_pack_start(GTK_BOX(hbox),icon,FALSE,FALSE,5);
	
	label = gtk_label_new(text);
	gtk_label_set_justify(GTK_LABEL(label),GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(hbox),label,TRUE,TRUE,20);
	
	/*
	 *	setup the buttons (Yes, No and Cancel)
	 */
	button = gtk_button_new_with_label("OK");
	GTK_WIDGET_SET_FLAGS(button,GTK_CAN_DEFAULT);
	gtk_signal_connect_object(GTK_OBJECT(button),"clicked",GTK_SIGNAL_FUNC(gtk_widget_destroy),GTK_OBJECT(dialog));
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area),button,TRUE,TRUE,0);
	gtk_widget_grab_default(button);

	gtk_widget_show_all(dialog);
}


static void yesnoquestion(char *text, GtkSignalFunc yes_cb, GtkSignalFunc no_cb)
{
	GtkWidget *	dialog;
	GtkWidget *	hbox;
	GtkWidget * icon;
	GtkWidget * label;
	GtkWidget *	button;
	
	dialog = gtk_dialog_new();
	gtk_window_set_title(GTK_WINDOW(dialog),"");
	gtk_window_set_transient_for(GTK_WINDOW(dialog),GTK_WINDOW(term_window));
	gtk_window_set_modal(GTK_WINDOW(dialog),TRUE);
	gtk_window_set_position(GTK_WINDOW(dialog),GTK_WIN_POS_MOUSE);
	gtk_signal_connect(GTK_OBJECT(dialog),"delete_event",GTK_SIGNAL_FUNC(gtk_widget_destroy),NULL);
	gtk_signal_connect(GTK_OBJECT(dialog),"destroy",GTK_SIGNAL_FUNC(gtk_widget_destroy),NULL);
	gtk_widget_realize(dialog);
	
	hbox = gtk_hbox_new(FALSE,0);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),hbox,TRUE,TRUE,0);

	icon = new_gtk_pixmap(dialog,dlg_question_xpm);
	gtk_box_pack_start(GTK_BOX(hbox),icon,FALSE,FALSE,5);
	
	label = gtk_label_new(text);
	gtk_label_set_justify(GTK_LABEL(label),GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(hbox),label,TRUE,TRUE,20);
	
	/*
	 *	setup the buttons (Yes, No and Cancel)
	 */
	button = gtk_button_new_with_label("Yes");
	GTK_WIDGET_SET_FLAGS(button,GTK_CAN_DEFAULT);
	gtk_signal_connect(GTK_OBJECT(button),"clicked",GTK_SIGNAL_FUNC(yes_cb),NULL);
	gtk_signal_connect_object(GTK_OBJECT(button),"clicked",GTK_SIGNAL_FUNC(gtk_widget_destroy),GTK_OBJECT(dialog));
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area),button,TRUE,TRUE,0);
	gtk_widget_grab_default(button);

	button = gtk_button_new_with_label("No");
	GTK_WIDGET_SET_FLAGS(button,GTK_CAN_DEFAULT);
	if (no_cb)
		gtk_signal_connect(GTK_OBJECT(button),"clicked",GTK_SIGNAL_FUNC(no_cb),NULL);
	gtk_signal_connect_object(GTK_OBJECT(button),"clicked",GTK_SIGNAL_FUNC(gtk_widget_destroy),GTK_OBJECT(dialog));
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area),button,TRUE,TRUE,0);
	
	button = gtk_button_new_with_label("Cancel");
	GTK_WIDGET_SET_FLAGS(button,GTK_CAN_DEFAULT);
	gtk_signal_connect_object(GTK_OBJECT(button),"clicked",GTK_SIGNAL_FUNC(gtk_widget_destroy),GTK_OBJECT(dialog));
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area),button,TRUE,TRUE,0);
	
	gtk_widget_show_all(dialog);
}

/*-----------------------------------------------------------------------
 *	About box
 *----------------------------------------------------------------------*/
static char text[]="\n\
Euler - 1.60\n\
\n\
EULER is a program for quickly and interactively\n\
computing with real and complex numbers and matrices,\n\
or with intervals. It can draw your functions in\n\
two and three dimensions.\n\
\n\
Copyright (C) 2001, 2002 Rene Grothmann, Eric Bouchar\n\
\n\
This program is free software; you can reditribute it\n\
and/or modifiy it under the terms of the GNU General\n\
Public License.\n\
\n\
This program is distributed in the hope that it will be\n\
usefull, but  WITHOUT ANY WARRANTY.\n\
\n\
Mail :\n\
    grothm@ku-eichstaett.de\n\
    bouchare.eric@wanadoo.fr\n\
\n\
Homepage :\n\
    http://euler.sourceforge.net\n";

static void about()
{
	GtkWidget *	dialog;
	GtkWidget *	hbox;
	GtkWidget * icon;
	GtkWidget * label;
	GtkWidget *	button;
	
	dialog = gtk_dialog_new();
	gtk_window_set_title(GTK_WINDOW(dialog),"");
	gtk_window_set_transient_for(GTK_WINDOW(dialog),GTK_WINDOW(term_window));
	gtk_window_set_modal(GTK_WINDOW(dialog),TRUE);
	gtk_window_set_position(GTK_WINDOW(dialog),GTK_WIN_POS_MOUSE);
	gtk_signal_connect(GTK_OBJECT(dialog),"delete_event",GTK_SIGNAL_FUNC(gtk_widget_destroy),NULL);
	gtk_signal_connect(GTK_OBJECT(dialog),"destroy",GTK_SIGNAL_FUNC(gtk_widget_destroy),NULL);
	gtk_widget_realize(dialog);
	
	hbox = gtk_hbox_new(FALSE,0);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),hbox,TRUE,TRUE,0);

	icon = new_gtk_pixmap(dialog,logo_xpm);
	gtk_box_pack_start(GTK_BOX(hbox),icon,FALSE,FALSE,5);
	
	label = gtk_label_new(text);
	gtk_label_set_justify(GTK_LABEL(label),GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(hbox),label,TRUE,TRUE,20);
	
	/*
	 *	setup the buttons (Yes, No and Cancel)
	 */
	button = gtk_button_new_with_label("OK");
	GTK_WIDGET_SET_FLAGS(button,GTK_CAN_DEFAULT);
	gtk_signal_connect_object(GTK_OBJECT(button),"clicked",GTK_SIGNAL_FUNC(gtk_widget_destroy),GTK_OBJECT(dialog));
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area),button,TRUE,TRUE,0);
	gtk_widget_grab_default(button);

	gtk_widget_show_all(dialog);
}

/*---------------------------------------------------------------------------
 *	preference box
 *---------------------------------------------------------------------------*/

static GtkWidget * prefs_dialog = NULL;
static GtkWidget * estack;
static GtkWidget * gstack;
static GtkWidget * glines;
static GtkWidget * colbut[16];
static GtkWidget * atexitbut;
static GtkWidget * browser;

static short def_colors[3][MAX_COLORS] = {
	{255,0,100,0  ,0  ,0  ,100,150,100,50,220 ,80  ,80  ,80  ,140 ,190},
	{255,0,0  ,100,0  ,100,100,150,100,50,80  ,220 ,80  ,140 ,140 ,190},
	{255,0,0  ,0  ,100,100,  0,150,100,50,80  ,80  ,220 ,140 , 80,190}
};

static void prefs_apply_cb(GtkWidget *widget, gpointer data)
{
	int val, i, j, changed = 0;
	
	val = atol(gtk_entry_get_text(GTK_ENTRY(estack)));
	if (val) prefs.estack = val;
	
	val = atol(gtk_entry_get_text(GTK_ENTRY(gstack)));
	if (val) prefs.gstack = val;

	val = atol(gtk_entry_get_text(GTK_ENTRY(glines)));
	if (val) {
		prefs.glines = val;
		setmetalines(prefs.glines);
		gtk_meta_update_lines(meta);
	}
	
	for (i=0 ; i<16 ; i++) {
		gushort color[3];
		
		col_but_get_color(colbut[i],color);
		
		for (j=0 ; j<3 ; j++)
			if (prefs.colors[j][i]!=color[j]) {
				prefs.colors[j][i]=color[j];
				changed =1;
			}
	}
	
	if (changed) {
		setmetacolors(prefs.colors);
		gtk_meta_update_colors(meta);
	}
	
	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(atexitbut)))
		prefs.saveatexit = 1;
	else
		prefs.saveatexit = 0;
	
	if (strlen(gtk_entry_get_text(GTK_ENTRY(browser))))
		strcpy(prefs.browser,gtk_entry_get_text(GTK_ENTRY(browser)));
}

static void prefs_ok_cb(GtkWidget *widget, gpointer data)
{
	prefs_apply_cb(widget,data);
	prefs_dialog = NULL;
}

static void prefs_cancel_cb(GtkWidget *widget, gpointer data)
{
	prefs_dialog = NULL;
}

static void prefs_reset_cb(GtkWidget *widget, gpointer data)
{
	char *s;
	int i;

	gtk_entry_set_text(GTK_ENTRY(browser),E_BROWSER_DEFAULT);
	
	s = g_strdup_printf("%ld",E_ESTACK_DEFAULT);
	gtk_entry_set_text(GTK_ENTRY(estack),s);
	g_free(s);
	s = g_strdup_printf("%ld",E_GSTACK_DEFAULT);
	gtk_entry_set_text(GTK_ENTRY(gstack),s);
	g_free(s);
	s = g_strdup_printf("%d",E_GLINES_DEFAULT);
	gtk_entry_set_text(GTK_ENTRY(glines),s);
	g_free(s);
	
	for (i=0 ; i<16 ; i++)
		col_but_set_color(colbut[i],def_colors[0][i],def_colors[1][i],def_colors[2][i]);
	
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(atexitbut),TRUE);
}

static void prefs_box()
{
	GtkWidget *	dialog;
	GtkWidget *	notebook;
	GtkWidget * vbox;
	GtkWidget *	table;
	GtkWidget * label;
	GtkWidget *	button;
	GtkWidget * frame;
	char *s;
	int i;
	
	prefs_dialog = dialog = gtk_dialog_new();
	gtk_window_set_title(GTK_WINDOW(dialog),"Preferences ...");
	gtk_window_set_position(GTK_WINDOW(dialog),GTK_WIN_POS_MOUSE);
	gtk_container_set_border_width(GTK_CONTAINER(dialog),8);
	gtk_signal_connect(GTK_OBJECT(dialog),"delete_event",GTK_SIGNAL_FUNC(gtk_widget_destroy),NULL);
	gtk_signal_connect(GTK_OBJECT(dialog),"destroy",GTK_SIGNAL_FUNC(gtk_widget_destroy),NULL);
	
	notebook = gtk_notebook_new();
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),notebook,TRUE,TRUE,0);
	
	/*
	 *	first page : memory
	 */
	frame = gtk_frame_new(NULL);
	table = gtk_table_new(4,2,FALSE);
	gtk_container_add(GTK_CONTAINER(frame),table);

	label = gtk_label_new("You can setup the memory size of the stack used for calculation. This will be effective on Euler restart.");
	gtk_label_set_line_wrap(GTK_LABEL(label),TRUE);
	gtk_label_set_justify(GTK_LABEL(label),GTK_JUSTIFY_FILL);
	gtk_table_attach(GTK_TABLE(table),label,0,2,0,1,GTK_FILL|GTK_EXPAND,0,2,2);
	label = gtk_label_new("stack memory (kb) :");
	gtk_label_set_justify(GTK_LABEL(label),GTK_JUSTIFY_RIGHT);
	gtk_table_attach(GTK_TABLE(table),label,0,1,1,2,GTK_FILL|GTK_EXPAND,0,2,2);
	estack = gtk_entry_new();
	s = g_strdup_printf("%d",prefs.estack);
	gtk_entry_set_text(GTK_ENTRY(estack),s);
	g_free(s);
	gtk_table_attach(GTK_TABLE(table),estack,1,2,1,2,GTK_FILL|GTK_EXPAND,0,2,2);

	label = gtk_label_new("You can setup the memory size of the buffer used for graphics. This will be effective on Euler restart.");
	gtk_label_set_line_wrap(GTK_LABEL(label),TRUE);
	gtk_label_set_justify(GTK_LABEL(label),GTK_JUSTIFY_FILL);
	gtk_table_attach(GTK_TABLE(table),label,0,2,2,3,GTK_FILL|GTK_EXPAND,0,2,2);
	label = gtk_label_new("graphic buffer (kb) :");
	gtk_label_set_justify(GTK_LABEL(label),GTK_JUSTIFY_RIGHT);
	gtk_table_attach(GTK_TABLE(table),label,0,1,3,4,GTK_FILL|GTK_EXPAND,0,2,2);
	gstack = gtk_entry_new();
	s = g_strdup_printf("%d",prefs.gstack);
	gtk_entry_set_text(GTK_ENTRY(gstack),s);
	g_free(s);
	gtk_table_attach(GTK_TABLE(table),gstack,1,2,3,4,GTK_FILL|GTK_EXPAND,0,2,2);
	
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook),frame,gtk_label_new("Memory"));

	/*
	 *	second page : font
	 */
	frame = gtk_frame_new(NULL);
	table = gtk_table_new(2,2,FALSE);
	gtk_container_add(GTK_CONTAINER(frame),table);
	
	label = gtk_label_new("You can set up the height of the graphics window in \
lines of text. This will not change the effective height of the graphics \
window, but the font size.");
	gtk_label_set_line_wrap(GTK_LABEL(label),TRUE);
	gtk_label_set_justify(GTK_LABEL(label),GTK_JUSTIFY_FILL);
	gtk_table_attach(GTK_TABLE(table),label,0,2,0,1,GTK_FILL|GTK_EXPAND,0,2,2);
	label = gtk_label_new("number of lines :");
	gtk_label_set_justify(GTK_LABEL(label),GTK_JUSTIFY_RIGHT);
	gtk_table_attach(GTK_TABLE(table),label,0,1,1,2,GTK_FILL|GTK_EXPAND,0,2,2);
	glines = gtk_entry_new();
	s = g_strdup_printf("%d",prefs.glines);
	gtk_entry_set_text(GTK_ENTRY(glines),s);
	g_free(s);
	gtk_table_attach(GTK_TABLE(table),glines,1,2,1,2,GTK_FILL|GTK_EXPAND,0,2,2);

	gtk_notebook_append_page(GTK_NOTEBOOK(notebook),frame,gtk_label_new("Font"));

	/*
	 *	third page : colors
	 */
	
	frame = gtk_frame_new(NULL);
	vbox = gtk_vbox_new(FALSE,0);
	gtk_container_add(GTK_CONTAINER(frame),vbox);
	label = gtk_label_new("You can set up the colors used for the graphics. \
The change will be set when you click on Apply or OK.");
	gtk_label_set_line_wrap(GTK_LABEL(label),TRUE);
	gtk_label_set_justify(GTK_LABEL(label),GTK_JUSTIFY_FILL);
	gtk_box_pack_start_defaults(GTK_BOX(vbox),label);
	
	table = gtk_table_new(8,4,TRUE);
	for (i=0 ; i<8 ; i++) {
		char *s = g_strdup_printf("%d",i);
		label = gtk_label_new(s);
		gtk_table_attach(GTK_TABLE(table),label,0,1,i,i+1,GTK_FILL|GTK_EXPAND,GTK_FILL|GTK_EXPAND,2,2);
		colbut[i] = col_but_new(10,15,prefs.colors[0][i],prefs.colors[1][i],prefs.colors[2][i]);
		gtk_table_attach(GTK_TABLE(table),colbut[i],1,2,i,i+1,GTK_FILL|GTK_EXPAND,GTK_FILL|GTK_EXPAND,2,2);
		g_free(s);

		s = g_strdup_printf("%d",i+8);
		colbut[i+8] = col_but_new(10,15,prefs.colors[0][i+8],prefs.colors[1][i+8],prefs.colors[2][i+8]);
		gtk_table_attach(GTK_TABLE(table),colbut[i+8],2,3,i,i+1,GTK_FILL|GTK_EXPAND,GTK_FILL|GTK_EXPAND,2,2);
		label = gtk_label_new(s);
		gtk_table_attach(GTK_TABLE(table),label,3,4,i,i+1,GTK_FILL|GTK_EXPAND,GTK_FILL|GTK_EXPAND,2,2);
		g_free(s);
	}
	gtk_box_pack_start_defaults(GTK_BOX(vbox),table);
	
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook),frame,gtk_label_new("Colors"));

	/*
	 *	fourth page : miscellaneous options
	 */
	frame = gtk_frame_new(NULL);
	vbox = gtk_vbox_new(FALSE,0);
	gtk_container_add(GTK_CONTAINER(frame),vbox);
	atexitbut = gtk_check_button_new_with_label("Save preferences at Euler exit");
	if (prefs.saveatexit)
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(atexitbut),TRUE);
	gtk_box_pack_start(GTK_BOX(vbox),atexitbut,FALSE,FALSE,0);
	label = gtk_label_new("Browser used to view documentation");
	gtk_label_set_line_wrap(GTK_LABEL(label),TRUE);
	gtk_label_set_justify(GTK_LABEL(label),GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(vbox),label,FALSE,FALSE,0);
	browser = gtk_entry_new();
	s = g_strdup_printf("%s",prefs.browser);
	gtk_entry_set_text(GTK_ENTRY(browser),s);
	g_free(s);
	gtk_box_pack_start(GTK_BOX(vbox),browser,FALSE,FALSE,0);
	button = gtk_button_new_with_label("Reset preferences to default values");
	gtk_signal_connect(GTK_OBJECT(button),"clicked",(GtkSignalFunc)prefs_reset_cb,NULL);
	gtk_box_pack_start(GTK_BOX(vbox),button,FALSE,FALSE,0);
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook),frame,gtk_label_new("Misc"));
	
	/*
	 *	setup the buttons (Yes, No and Cancel)
	 */
	button = gtk_button_new_with_label("OK");
	GTK_WIDGET_SET_FLAGS(button,GTK_CAN_DEFAULT);
	gtk_signal_connect(GTK_OBJECT(button),"clicked",GTK_SIGNAL_FUNC(prefs_ok_cb),NULL);
	gtk_signal_connect_object(GTK_OBJECT(button),"clicked",GTK_SIGNAL_FUNC(gtk_widget_destroy),GTK_OBJECT(dialog));
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area),button,TRUE,TRUE,0);

	button = gtk_button_new_with_label("Apply");
	GTK_WIDGET_SET_FLAGS(button,GTK_CAN_DEFAULT);
	gtk_signal_connect(GTK_OBJECT(button),"clicked",GTK_SIGNAL_FUNC(prefs_apply_cb),NULL);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area),button,TRUE,TRUE,0);
	gtk_widget_grab_default(button);
	
	button = gtk_button_new_with_label("Cancel");
	GTK_WIDGET_SET_FLAGS(button,GTK_CAN_DEFAULT);
	gtk_signal_connect(GTK_OBJECT(button),"clicked",GTK_SIGNAL_FUNC(prefs_cancel_cb),NULL);
	gtk_signal_connect_object(GTK_OBJECT(button),"clicked",GTK_SIGNAL_FUNC(gtk_widget_destroy),GTK_OBJECT(dialog));
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area),button,TRUE,TRUE,0);
	
	gtk_widget_show_all(dialog);	
}

/*---------------------------------------------------------------------------
 *	comment editor
 *---------------------------------------------------------------------------*/
static void editcomment(char *buffer, GtkSignalFunc cb)
{
	GtkWidget *window;
	GtkWidget *vbox;
	GtkWidget *scr;
	GtkWidget *text;
	GtkWidget *button;
	
	window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_signal_connect(GTK_OBJECT(window),"delete_event",(GtkSignalFunc)gtk_widget_destroy,NULL);
	gtk_signal_connect(GTK_OBJECT(window),"destroy",(GtkSignalFunc)gtk_widget_destroy,NULL);
	gtk_window_set_transient_for(GTK_WINDOW(window),GTK_WINDOW(term_window));
	gtk_window_set_modal(GTK_WINDOW(window),TRUE);
	gtk_window_set_position(GTK_WINDOW(window),GTK_WIN_POS_MOUSE);

	vbox = gtk_vbox_new(FALSE,0);
	gtk_container_add(GTK_CONTAINER(window),vbox);
	
	scr = gtk_scrolled_window_new(NULL,NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scr),GTK_POLICY_NEVER,GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start_defaults(GTK_BOX(vbox),scr);
	
	text = gtk_text_new(NULL,NULL);
	gtk_widget_set_usize(text,400,200);
	gtk_container_add(GTK_CONTAINER(scr),text);
	gtk_text_set_editable(GTK_TEXT(text),TRUE);
	gtk_window_set_focus(GTK_WINDOW(window),text);
	if (buffer)
		gtk_text_insert(GTK_TEXT(text),NULL,NULL,NULL,buffer,strlen(buffer));
	
	button = gtk_button_new_with_label("Back to the notebook");
	GTK_WIDGET_SET_FLAGS(button,GTK_CAN_DEFAULT);
	gtk_signal_connect_object(GTK_OBJECT(button),"clicked",GTK_SIGNAL_FUNC(cb),GTK_OBJECT(text));
	gtk_signal_connect_object(GTK_OBJECT(button),"clicked",GTK_SIGNAL_FUNC(gtk_widget_destroy),GTK_OBJECT(window));
	gtk_box_pack_start(GTK_BOX(vbox),button,TRUE,TRUE,0);
	gtk_widget_grab_default(button);

	gtk_widget_show_all(window);
}

static void edit_comment_cb(GtkWidget *widget, gpointer data)
{
	gtk_term_set_comment(term,gtk_editable_get_chars(GTK_EDITABLE(widget),0,-1));
}
/*---------------------------------------------------------------------------
 *	menu
 *---------------------------------------------------------------------------*/

static void menu_cb(GtkWidget *w, guint op);
static void open_cb(GtkFileSelection *fs);
static void save_as_cb(GtkFileSelection *fs);
static void postscript_cb(GtkFileSelection *fs);
static void yes_new_cb(GtkWidget *widget, gpointer data);
static void no_new_cb(GtkWidget *widget, gpointer data);
static void yes_open_cb(GtkWidget *widget, gpointer data);
static void no_open_cb(GtkWidget *widget, gpointer data);
static void yes_quit_cb(GtkWidget *widget, gpointer data);
static void no_quit_cb(GtkWidget *widget, gpointer data);

#define	E_NEW			1
#define	E_OPEN			2
#define	E_SAVE			3
#define	E_SAVEAS		4
#define E_POSTSCRIPT	5
#define	E_QUIT			6
#define	E_CUT			7
#define	E_COPY			8
#define	E_PASTE			9
#define E_INSCMD		10
#define	E_DELCMD		11
#define E_DELOUT		12
#define	E_EDITCMT		13
#define E_PREFS			14
#define E_DOC			15
#define	E_ABOUT			16

static GtkItemFactoryEntry menu_items[] = {
	{ "/_File",				NULL,			NULL,	0, "<Branch>" },
	{ "/File/_New",			"<control>N",	menu_cb,	E_NEW, NULL },
	{ "/File/_Open ...",	"<control>O",	menu_cb,	E_OPEN, NULL },
	{ "/File/_Save",		"<control>S",	menu_cb,	E_SAVE, NULL },
	{ "/File/Save _As ...",	NULL,			menu_cb,	E_SAVEAS, NULL },
	{ "/File/sep1",			NULL,			NULL,	0, "<Separator>" },
	{ "/File/_Export Graphics To", NULL,		NULL,	0, "<Branch>" },
	{ "/File/Export Graphics To/_Postscript","<control>P",menu_cb,E_POSTSCRIPT,NULL},
	{ "/File/sep2",			NULL,			NULL,	0, "<Separator>" },
	{ "/File/Quit",			"<control>Q",	menu_cb,	E_QUIT, NULL },
	{ "/_Edit",				NULL,			NULL,	0,	"<Branch>" },
	{ "/Edit/C_ut",			"<control>X",	menu_cb,	E_CUT,	NULL },
	{ "/Edit/_Copy",		"<control>C",	menu_cb,	E_COPY,	NULL },
	{ "/Edit/_Paste",		"<control>V",	menu_cb,	E_PASTE,	NULL },
	{ "/Edit/sep3",			NULL,			NULL,	0, "<Separator>" },
	{ "/Edit/Insert Command",	"<control>I",	menu_cb,	E_INSCMD,	NULL },
	{ "/Edit/Delete Command",	"<control>D",	menu_cb,	E_DELCMD,	NULL },
	{ "/Edit/sep4",			NULL,			NULL,	0, "<Separator>" },
	{ "/Edit/Delete Outputs",NULL,			menu_cb,	E_DELOUT,		NULL },
	{ "/Edit/sep5",			NULL,			NULL,	0, "<Separator>" },
	{ "/Edit/Edit Comment",	"<control>E",	menu_cb,	E_EDITCMT,	NULL },
	{ "/_Misc",				NULL,			NULL,	0, "<Branch>" },
	{ "/Misc/Demo ...",		NULL,			NULL,	0,	"<Branch>"},
	{ "/Misc/Preferences ...",NULL,			menu_cb,	E_PREFS, NULL },
	{ "/_Help",				NULL,			NULL,	0, "<LastBranch>" },
	{ "/Help/Documentation",NULL,			menu_cb,	E_DOC,	NULL },
	{ "/Help/About",		NULL,			menu_cb,	E_ABOUT,NULL },
};

static void menu_cb(GtkWidget *w, guint op)
{
	gchar *buffer;
	
	switch (op)
	{
		case E_NEW:
			if (!is_demo && gtk_term_is_changed(term))
				yesnoquestion("\nThe current notebook has been modified ...\nDo you want to save it ?\n",yes_new_cb,no_new_cb);
			else {
				gtk_term_clear_new(term);
				is_demo = 0;
			}
			break;
		case E_OPEN:
			if (!is_demo && gtk_term_is_changed(term))
				yesnoquestion("\nThe current notebook has been modified ...\nDo you want to save it ?\n",yes_open_cb,no_open_cb);
			else
				file_select("Open a notebook...","*.en",open_cb);
			break;
		case E_SAVE:
			if (!gtk_term_is_named(term))
				file_select("Save As ...","*.en",save_as_cb);
			else if (!is_demo)
				gtk_term_save(term,NULL);
			break;
		case E_SAVEAS:
			file_select("Save As ...","*.en",save_as_cb);
			break;
		case E_POSTSCRIPT:
			file_select("Export Postscript Graphics As ...","*.eps",postscript_cb);
			break;
		case E_QUIT:
			if (!is_demo && gtk_term_is_changed(term))
				yesnoquestion("\nThe current notebook has been modified ...\nDo you want to save it ?\n",yes_quit_cb,no_quit_cb);
			else
				gtk_exit(0);
			break;
		case E_CUT:
			gtk_term_cut(term);
			break;
		case E_COPY:
			gtk_term_copy(term);
			break;
		case E_PASTE:
			gtk_term_paste(term);
			break;
		case E_INSCMD:
			gtk_term_insert_command(term,NULL);
			break;
		case E_DELCMD:
			gtk_term_delete_command(term);
			break;
		case E_DELOUT:
			gtk_term_delete_outputs(term);
			break;
		case E_EDITCMT:
			buffer = gtk_term_get_comment(term);
			editcomment(buffer,edit_comment_cb);
			g_free(buffer);
			break;
		case E_PREFS:
			if (prefs_dialog)
				gdk_window_raise(prefs_dialog->window);
			else
				prefs_box();
			break;
		case E_DOC: {
				char *browser;
				browser = g_strconcat(prefs.browser," ",INSTALL_DIR,"/share/doc/euler/index.html &",NULL);
				system(browser);
				g_free(browser);
			}
			break;
		case E_ABOUT:
			about();
			break;
	}
}

static void open_cb(GtkFileSelection *fs)
{
	char *filename = gtk_file_selection_get_filename(fs);
	if ((!filename) || (strcmp(filename,"") == 0))
		return;
		
	if (!gtk_term_load(term,filename)) {
		char *txt = g_strdup_printf("\nI could not open the file\n%s !\n",filename);
		infobox(txt);
		g_free(txt);
	} else is_demo = 0;
	chdir(g_dirname(filename));
}

static void postscript_cb(GtkFileSelection *fs)
{
	char *filename = gtk_file_selection_get_filename(fs);
	if ((!filename) || (strcmp(filename,"") == 0))
		return;
	
	if (!dump_postscript(filename)) {
		char *txt = g_strdup_printf("\nI could not save the postscript graphics to the file\n%s\n",filename);
		infobox(txt);
		g_free(txt);
	}
}

static void save_as_cb(GtkFileSelection *fs)
{
	char *filename = gtk_file_selection_get_filename(fs);
	if ((!filename) || (strcmp(filename,"") == 0))
		return;
	
	if (!gtk_term_save(term,filename)) {
		char *txt = g_strdup_printf("\nI could not save the notebook to the file\n%s\n",filename);
		infobox(txt);
		g_free(txt);
	}
	chdir(g_dirname(filename));
}

static void save_as_open_cb(GtkFileSelection *fs)
{
	char *filename = gtk_file_selection_get_filename(fs);
	if ((!filename) || (strcmp(filename,"") == 0))
		return;
	
	if (gtk_term_save(term,filename)) {
		chdir(g_dirname(filename));
		file_select("Open a notebook...","*.en",open_cb);
	} else {
		char *txt = g_strdup_printf("\nI could not save the notebook to the file\n%s\nThe notebook open command is aborted ...\n",filename);
		infobox(txt);
		g_free(txt);
		chdir(g_dirname(filename));
	}
}

static void save_as_new_cb(GtkFileSelection *fs)
{
	char *filename = gtk_file_selection_get_filename(fs);
	if ((!filename) || (strcmp(filename,"") == 0))
		return;
	
	if (gtk_term_save(term,filename)) {
		gtk_term_clear_new(term);
		is_demo = 0;
	} else {
		char *txt = g_strdup_printf("\nI could not save the notebook to the file\n%s\nThe new notebook command is aborted ...\n",filename);
		infobox(txt);
		g_free(txt);
	}
	chdir(g_dirname(filename));
}

static void save_as_quit_cb(GtkFileSelection *fs)
{
	char *filename = gtk_file_selection_get_filename(fs);
	if ((!filename) || (strcmp(filename,"") == 0))
		return;
	
	if (gtk_term_save(term,filename))
		gtk_exit(0);
	else {
		char *txt = g_strdup_printf("\nI could not save the notebook to the file\n%s\nThe quit notebook command is aborted ...\n",filename);
		infobox(txt);
		g_free(txt);
	}
	chdir(g_dirname(filename));
}

static void yes_quit_cb(GtkWidget *widget, gpointer data)
{
	if (!gtk_term_is_named(term))
		file_select("Save As ...","*.en",save_as_quit_cb);
	else {
		gtk_term_save(term,NULL);
		gtk_exit(0);
	}
}

static void no_quit_cb(GtkWidget *widget, gpointer data)
{
	gtk_exit(0);
}

/*---------------------------------------------------------------------------
 *	toolbar callbacks
 *---------------------------------------------------------------------------*/
static void yes_new_cb(GtkWidget *widget, gpointer data)
{
	if (!gtk_term_is_named(term))
		file_select("Save As ...","*.en",save_as_new_cb);
	else {
		gtk_term_save(term,NULL);
		gtk_term_clear_new(term);
		is_demo = 0;
	}
}

static void no_new_cb(GtkWidget *widget, gpointer data)
{
	gtk_term_clear_new(term);
	is_demo = 0;
}

static void yes_open_cb(GtkWidget *widget, gpointer data)
{
	if (!gtk_term_is_named(term))
		file_select("Save As ...","*.en",save_as_open_cb);
	else {
		gtk_term_save(term,NULL);
		file_select("Open a notebook...","*.en",open_cb);
	}
}

static void no_open_cb(GtkWidget *widget, gpointer data)
{
	file_select("Open a notebook...","*.en",open_cb);
}

static void tb_new_cb(GtkWidget *widget, gpointer data)
{
	if (!is_demo && gtk_term_is_changed(term))
		yesnoquestion("\nThe current notebook has been modified ...\nDo you want to save it ?\n",yes_new_cb,no_new_cb);
	else {
		gtk_term_clear_new(term);
		is_demo = 0;
	}
}

static void tb_open_cb(GtkWidget *widget, gpointer data)
{
	if (!is_demo && gtk_term_is_changed(term))
		yesnoquestion("\nThe current notebook has been modified ...\nDo you want to save it ?\n",yes_open_cb,no_open_cb);
	else
		file_select("Open a notebook...","*.en",open_cb);
}

static void tb_save_cb(GtkWidget *widget, gpointer data)
{
	if (!gtk_term_is_named(term))
		file_select("Save As ...","*.en",save_as_cb);
	else if (!is_demo)
		gtk_term_save(term,NULL);
}

static void tb_copy_cb(GtkWidget *widget, gpointer data)
{
	gtk_term_copy(term);
}

static void tb_paste_cb(GtkWidget *widget, gpointer data)
{
	gtk_term_paste(term);
}

static void quit_cb(GtkWidget *widget, gpointer data)
{
	if (!is_demo && gtk_term_is_changed(term))
		yesnoquestion("\nThe current notebook has been modified ...\nDo you want to save it ?\n",yes_quit_cb,no_quit_cb);
	else
		gtk_exit(0);
}

/*---------------------------------------------------------------------------
 *	popup menu
 *---------------------------------------------------------------------------*/

static GtkItemFactoryEntry popupmenu_items[] = {
	{ "/C_ut",			"<control>X",	menu_cb,	E_CUT,	NULL },
	{ "/_Copy",			"<control>C",	menu_cb,	E_COPY,	NULL },
	{ "/_Paste",		"<control>V",	menu_cb,	E_PASTE,	NULL },
	{ "/sep3",			NULL,			NULL,	0, "<Separator>" },
	{ "/Insert Command","<control>I",	menu_cb,	E_INSCMD,	NULL },
	{ "/Delete Command","<control>D",	menu_cb,	E_DELCMD,	NULL },
	{ "/sep4",			NULL,			NULL,	0, "<Separator>" },
	{ "/Delete Outputs",NULL,			menu_cb,	E_DELOUT,		NULL },
	{ "/sep5",			NULL,			NULL,	0, "<Separator>" },
	{ "/Edit Comment",	"<control>E",	menu_cb,	E_EDITCMT,	NULL },
};

/*---------------------------------------------------------------------------
 *	term callbacks and drag & drop
 *---------------------------------------------------------------------------*/
static guint id;

static GdkAtom text_uri_list, application_octet_stream, text_plain;

enum {TARGET_RAW_DATA, TARGET_URI_LIST};

/* Targets which other apps can offer to us */
static GtkTargetEntry targets_to_accept[] =
{
	{"application/octet-stream", 0, TARGET_RAW_DATA},
	{"text/uri-list", 0, TARGET_URI_LIST}
};

#define MAX_HOST_NAME_LEN 256	/* XXX: Limit */
char    our_host_name[MAX_HOST_NAME_LEN];

static gchar *dnd_path=NULL;


static void term_changed_cb(GtkWidget *widget, gpointer data)
{
	char *title = g_strdup_printf("Euler [ *%s ]",gtk_term_get_name(term));
	gtk_window_set_title(GTK_WINDOW(term_window), title);
	g_free(title);
}

static void term_saved_cb(GtkWidget *widget, gpointer data)
{
	char *title = g_strdup_printf("Euler [ %s ]",gtk_term_get_name(term));
	gtk_window_set_title(GTK_WINDOW(term_window), title);
	g_free(title);
}

static void term_editing_cb(GtkWidget *widget, gpointer data)
{
	if (gtk_term_is_initialized(term))
		gtk_statusbar_pop(GTK_STATUSBAR(widget),id);
	else
		gtk_statusbar_push(GTK_STATUSBAR(widget),id," Editing ...");
}

static void term_interpreting_cb(GtkWidget *widget, gpointer data)
{
	gtk_statusbar_push(GTK_STATUSBAR(widget),id," Running ...");
}

static gint term_cfg_cb(GtkWidget *widget, GtkAllocation *allocation, gpointer data)
{
	GtkTerm *term = GTK_TERM(widget);
	prefs.twidth = term->twidth;
	prefs.theight = term->theight;
	return FALSE;
}

static void save_as_dndopen_cb(GtkFileSelection *fs)
{
	char *filename = gtk_file_selection_get_filename(fs);
	if ((!filename) || (strcmp(filename,"") == 0))
		return;
	
	if (gtk_term_save(term,filename)) {
		chdir(g_dirname(dnd_path));
		gtk_term_load(term, dnd_path);
	} else {
		char *txt = g_strdup_printf("\nI could not save the notebook to the file\n%s\nThe notebook open command is aborted ...\n",filename);
		infobox(txt);
		g_free(txt);
		chdir(g_dirname(filename));
	}
}

static void yes_dndopen_cb(GtkWidget *widget, gpointer data)
{
	if (!gtk_term_is_named(term))
		file_select("Save As ...","*.en",save_as_dndopen_cb);
	else {
		gtk_term_save(term,NULL);
		gtk_term_load(term, dnd_path);
		chdir(g_dirname(dnd_path));
	}
}

static void no_dndopen_cb(GtkWidget *widget, gpointer data)
{
	gtk_term_load(term, dnd_path);
	chdir(g_dirname(dnd_path));
}

static GdkAtom best_raw(GdkDragContext *context);
static gboolean provides(GdkDragContext *context, GdkAtom target);

/* Convert a list of URIs into a list of strings.
 * Lines beginning with # are skipped.
 * The text block passed in is zero terminated (after the final CRLF)
 */
GSList *uri_list_to_gslist(char *uri_list)
{
	GSList   *list = NULL;

	while (*uri_list)
	{
		char	*linebreak;
		char	*uri;
		int	length;

		linebreak = strchr(uri_list, 13);

		if (!linebreak || linebreak[1] != 10)
		{
			infobox("Incorrect or missing line break "
				"in text/uri-list data\n");
			return list;
		}

		length = linebreak - uri_list;

		if (length && uri_list[0] != '#')
		{
			uri = g_malloc(sizeof(char) * (length + 1));
			strncpy(uri, uri_list, length);
			uri[length] = 0;
			list = g_slist_append(list, uri);
		}

		uri_list = linebreak + 2;
	}

	return list;
}

/* Convert a URI to a local pathname (or NULL if it isn't local).
 * The returned pointer points inside the input string.
 * Possible formats:
 *	/path
 *	///path
 *	//host/path
 *	file://host/path
 */
char *get_local_path(char *uri)
{
	if (*uri == '/')
	{
		char    *path;

		if (uri[1] != '/')
			return uri;	/* Just a local path - no host part */

		path = strchr(uri + 2, '/');
		if (!path)
			return NULL;	    /* //something */

		if (path - uri == 2)
			return path;	/* ///path */
		if (strlen(our_host_name) == path - uri - 2 &&
			strncmp(uri + 2, our_host_name, path - uri - 2) == 0)
			return path;	/* //myhost/path */

		return NULL;	    /* From a different host */
	}
	else
	{
		if (strncasecmp(uri, "file:", 5))
			return NULL;	    /* Don't know this format */

		uri += 5;

		if (*uri == '/')
			return get_local_path(uri);

		return NULL;
	}
}

/*
 * This is called when the remote application wants to send us some
 * data. We get to decide what kind of data we'd like.
 */
static int drag_drop(GtkWidget *widget, GdkDragContext *context, gint x, gint y, guint time)
{
	guchar	*leafname = NULL;
	GdkAtom	target;

	if (gtk_drag_get_source_widget(context))
		return TRUE;		/* Drag to ourselves - ignore */

	if (provides(context, text_uri_list))
		target = text_uri_list;
	else
	{
		leafname = g_strdup("Untitled");
		target = best_raw(context);
	}

	/* Associate the leafname with the context */
	if (leafname)
		g_dataset_set_data_full(context, "uri", leafname, g_free);

	if (target)
		gtk_drag_get_data(widget, context, target, time);
	else
		gtk_drag_finish(context, FALSE, FALSE, time);

	return TRUE;
}

/* Look for the best target type for transferring data */
static GdkAtom best_raw(GdkDragContext *context)
{
	if (provides(context, text_plain))
		return text_plain;
	if (provides(context, application_octet_stream))
		return application_octet_stream;

	infobox("I can't get the data from the other application\n");
	return GDK_NONE;
}

/* Is the sended willing to supply this target type? */
static gboolean provides(GdkDragContext *context, GdkAtom target)
{
    GList	    *targets = context->targets;

    while (targets && ((GdkAtom) targets->data != target))
	targets = targets->next;

    return targets != NULL;
}

/* Called when some data arrives from the remote app (which we asked for
 * in drag_drop.
 */
static void drag_data_received(GtkWidget *widget, GdkDragContext *context, gint x, gint y, GtkSelectionData *selection_data, guint info, guint32 time)
{
	if (!selection_data->data)
	{
		/* Timeout? */
		gtk_drag_finish(context, FALSE, FALSE, time);
		return;
	}

	if (selection_data->target == text_uri_list)
	{
		GSList	*files;
		char	*localpath;

		files = uri_list_to_gslist(selection_data->data);

		if (files == NULL || files->next)
		{
			/* Only one file at a time for now */
			infobox("Sorry, I can only load one file at a time");
			gtk_drag_finish(context, FALSE, FALSE, time);
			return;
		}

		localpath = get_local_path((char *) files->data);

		/* Remember the URI for the data */
		g_dataset_set_data_full(context,"uri",files->data,g_free);

		if (localpath)
		{
			gboolean 	success = FALSE;
			
			if (gtk_term_is_editing(term)) {
				if (!is_demo && gtk_term_is_changed(term)) {
					dnd_path=localpath;
					yesnoquestion("\nThe current notebook has been modified ...\nDo you want to save it ?\n",yes_dndopen_cb,no_dndopen_cb);
					success = TRUE;
				} else {
					success = gtk_term_load(term, localpath);
					chdir(g_dirname(localpath));
				}
			} else {
				infobox("You cannot load another notebook\nwhile the interpreter is running ...");
			}
			gtk_drag_finish(context,success,FALSE,time);
		}
		else
		{
			gtk_drag_finish(context, FALSE, FALSE, time);
		}
	}
	else
	{
		gtk_drag_finish(context, FALSE, FALSE, time);
	}
}



/*---------------------------------------------------------------------------
 *	metagtk callbacks
 *---------------------------------------------------------------------------*/

static gint meta_cfg_cb(GtkWidget *widget, GdkEventConfigure *event)
{
	prefs.gwidth = event->width;
	prefs.gheight = event->height;
	return FALSE;
}

/*---------------------------------------------------------------------------
 *	demo callbacks
 *---------------------------------------------------------------------------*/
static gchar *demo_name=NULL;

static void save_as_demo_cb(GtkFileSelection *fs)
{
	char *filename = gtk_file_selection_get_filename(fs);
	if ((!filename) || (strcmp(filename,"") == 0))
		return;
	
	if (gtk_term_save(term,filename)) {
		gchar *s;
		chdir(g_dirname(filename));
		s = g_strconcat(INSTALL_DIR,"/share/euler/progs/",demo_name,NULL);
		gtk_term_load(term,s);
		g_free(s);
		is_demo = 1;
	} else {
		char *txt = g_strdup_printf("\nI could not save the notebook to the file\n%s\nThe notebook open command is aborted ...\n",filename);
		infobox(txt);
		g_free(txt);
		chdir(g_dirname(filename));
	}
}

static void yes_demo_cb(GtkWidget *widget, gpointer data)
{
	if (!gtk_term_is_named(term))
		file_select("Save As ...","*.en",save_as_demo_cb);
	else {
		gchar *s;
		gtk_term_save(term,NULL);
		s = g_strconcat(INSTALL_DIR,"/share/euler/progs/",demo_name,NULL);
		gtk_term_load(term,s);
		g_free(s);
		is_demo = 1;
	}
}

static void no_demo_cb(GtkWidget *widget, gpointer data)
{
	gchar *s;
	s = g_strconcat(INSTALL_DIR,"/share/euler/progs/",demo_name,NULL);
	gtk_term_load(term,s);
	g_free(s);
	is_demo = 1;
}

static void demo_cb(GtkMenuItem *mitem, gpointer data)
{
	GtkLabel *label = GTK_LABEL(GTK_BIN(mitem)->child);

	gtk_label_get(label,&demo_name);
	if (!is_demo) {
		if (gtk_term_is_changed(term))
			yesnoquestion("\nThe current notebook has been modified ...\nDo you want to save it ?\n",yes_demo_cb,no_demo_cb);
		else {
			gchar *s;
			s = g_strconcat(INSTALL_DIR,"/share/euler/progs/",demo_name,NULL);
			gtk_term_load(term,s);
			g_free(s);
			is_demo = 1;
		}
	} else {
		gchar *s;
		s = g_strconcat(INSTALL_DIR,"/share/euler/progs/",demo_name,NULL);
		gtk_term_load(term,s);
		g_free(s);
		is_demo = 1;
	}
}
/*---------------------------------------------------------------------------
 *	main
 *---------------------------------------------------------------------------*/
static void save_rc()
{
	if (prefs.saveatexit)
		eulerrc_save(&prefs);
}

int main( int   argc, 
          char *argv[] )
{
	GtkWidget *menubar;
	GtkItemFactory *item_factory;
	GtkAccelGroup *accel_group;
	GtkItemFactory *ppitem_factory;
	GtkAccelGroup *ppaccel_group;
	GtkWidget *vbox, *hbox;
	GtkWidget *toolbar;
	GtkWidget *scrollbar;
	GtkWidget *statusbar;
	GdkPixmap *icon_pixmap;
	GdkBitmap *icon_mask;
	GdkColor  alpha;
	GtkWidget *demo;
	GtkWidget *ditem;
	char **entries=NULL;
	int n_entries=0,i;
	char *s;
	
	gtk_init (&argc, &argv);

	prefs = eulerrc_init();
	atexit(save_rc);
	
	// allocate stack space
	if (!stack_init(prefs.estack)) gtk_exit(1);

	// allocate graphic stack space
	if (!openmeta(prefs.gstack,NULL)) gtk_exit(1);
	setmetalines(prefs.glines);
	setmetacolors(prefs.colors);

	/*
	 *	setup atoms needed for drag & drop
	 */
	text_plain = gdk_atom_intern("text/plain", FALSE);
	text_uri_list = gdk_atom_intern("text/uri-list", FALSE);
	application_octet_stream = gdk_atom_intern("application/octet-stream", FALSE);

	/*
	 * create the graphic window
	 */
	
	meta_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(meta_window), "Euler graphics window");
	gtk_signal_connect(GTK_OBJECT(meta_window),"delete_event",(GtkSignalFunc)quit_cb,NULL);
	gtk_signal_connect(GTK_OBJECT(meta_window),"destroy",(GtkSignalFunc)quit_cb,NULL);

	meta = gtk_meta_new(prefs.gwidth,prefs.gheight);
	gtk_container_add(GTK_CONTAINER(meta_window),meta);

	/*
	 *	add a hook to collect geometry changes of the graphic window
	 */
	gtk_signal_connect(GTK_OBJECT(meta_window),"configure_event",(GtkSignalFunc)meta_cfg_cb,NULL);
	
	gtk_widget_show_all(meta_window);
	
	/*
	 * create the main window
	 */
	term_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(term_window), "Euler [  ]");
	gtk_widget_realize(term_window);
	icon_pixmap = gdk_pixmap_create_from_xpm_d(term_window->window,&icon_mask,&alpha,icon_xpm);
	gdk_window_set_icon(term_window->window,NULL,icon_pixmap,icon_mask);
	gdk_window_set_group(meta_window->window,term_window->window);

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER(term_window), vbox);

	gtk_signal_connect(GTK_OBJECT(term_window),"delete_event",(GtkSignalFunc)quit_cb,NULL);
	gtk_signal_connect(GTK_OBJECT(term_window),"destroy",GTK_SIGNAL_FUNC(quit_cb),NULL);
	gtk_widget_realize(term_window);
	
	/*
	 * create the menu bar
	 */
	accel_group = gtk_accel_group_new();
	item_factory = gtk_item_factory_new(GTK_TYPE_MENU_BAR,"<Main>",accel_group);
	gtk_item_factory_create_items(item_factory,sizeof(menu_items)/sizeof(menu_items[0]),menu_items,NULL);
	gtk_accel_group_attach(accel_group,GTK_OBJECT(term_window));
	menubar = gtk_item_factory_get_widget(item_factory,"<Main>");
	gtk_box_pack_start (GTK_BOX(vbox), menubar, FALSE, FALSE, 0);
	gtk_item_factory_parse_rc("~/.euler/shortcuts");
	/*
	 * parse the install directory for available demos
	 */
	s = g_strconcat(INSTALL_DIR,"/share/euler/progs",NULL);
		
	ditem = gtk_item_factory_get_item(item_factory,"/Misc/Demo ...");
	demo = gtk_menu_new();
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(ditem),demo);

	scan_dir(s,"*.en",&entries,&n_entries);
	if (entries) {
		for (i=0;i<n_entries;i++) {
			GtkWidget *mitem = gtk_menu_item_new_with_label(entries[i]);
			gtk_menu_append(GTK_MENU(demo),mitem);
			gtk_signal_connect(GTK_OBJECT(mitem),"activate",demo_cb,NULL);
			free(entries[i]);
		}
		free(entries);
		entries = NULL;
	}
	g_free(s);

	/*
	 * create the toolbar
	 */
	toolbar = gtk_toolbar_new(GTK_ORIENTATION_HORIZONTAL,GTK_TOOLBAR_ICONS);
	gtk_toolbar_set_button_relief(GTK_TOOLBAR(toolbar),GTK_RELIEF_NONE);
	gtk_toolbar_set_space_size(GTK_TOOLBAR(toolbar),10);
	gtk_toolbar_set_space_style(GTK_TOOLBAR(toolbar),GTK_TOOLBAR_SPACE_LINE);
	gtk_toolbar_append_item(GTK_TOOLBAR(toolbar),NULL,"Create a new notebook",
				NULL,new_gtk_pixmap(term_window,new_xpm),tb_new_cb,NULL);
	gtk_toolbar_append_item(GTK_TOOLBAR(toolbar),NULL,"Open a notebook",
				NULL,new_gtk_pixmap(term_window,open_xpm),tb_open_cb,NULL);
	gtk_toolbar_append_item(GTK_TOOLBAR(toolbar),NULL,"Save the notebook",
				NULL,new_gtk_pixmap(term_window,save_xpm),tb_save_cb,NULL);
	gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
	gtk_toolbar_append_item(GTK_TOOLBAR(toolbar),NULL,"Copy the selection",
				NULL,new_gtk_pixmap(term_window,copy_xpm),tb_copy_cb,NULL);
	gtk_toolbar_append_item(GTK_TOOLBAR(toolbar),NULL,"Paste the selection",
				NULL,new_gtk_pixmap(term_window,paste_xpm),tb_paste_cb,NULL);
	
	gtk_box_pack_start(GTK_BOX(vbox),toolbar,FALSE,FALSE,0);
	
	/*
	 *	create an euler notebook widget and a scrollbar
	 */	
	
	hbox = gtk_hbox_new(FALSE,0);
	gtk_box_pack_start(GTK_BOX(vbox),hbox,TRUE,TRUE,0);
	
	term = gtk_term_new(prefs.twidth,prefs.theight,prefs.tfont);
	gtk_box_pack_start_defaults(GTK_BOX(hbox),term);
	gtk_signal_connect(GTK_OBJECT(term),"term_changed",GTK_SIGNAL_FUNC(term_changed_cb),NULL);
	gtk_signal_connect(GTK_OBJECT(term),"term_saved",GTK_SIGNAL_FUNC(term_saved_cb),NULL);
	gtk_signal_connect_after(GTK_OBJECT(term),"size_allocate",GTK_SIGNAL_FUNC(term_cfg_cb),NULL);
	/* Set the window up to receive drops */
	gtk_drag_dest_set(term,
			GTK_DEST_DEFAULT_MOTION | GTK_DEST_DEFAULT_HIGHLIGHT,
			targets_to_accept,
			sizeof(targets_to_accept) / sizeof(*targets_to_accept),
			GDK_ACTION_COPY | GDK_ACTION_PRIVATE);
	gtk_signal_connect(GTK_OBJECT(term), "drag_drop",
			GTK_SIGNAL_FUNC(drag_drop), NULL);
	gtk_signal_connect(GTK_OBJECT(term), "drag_data_received",
			GTK_SIGNAL_FUNC(drag_data_received), NULL);

	
	scrollbar = gtk_vscrollbar_new(GTK_TERM(term)->v_adj);
	GTK_WIDGET_UNSET_FLAGS (scrollbar, GTK_CAN_FOCUS);
	gtk_box_pack_start(GTK_BOX(hbox),scrollbar,FALSE,FALSE,0);
	/*
	 * create the popup menu
	 */
	ppaccel_group = gtk_accel_group_new();
	ppitem_factory = gtk_item_factory_new(GTK_TYPE_MENU,"<Main>",ppaccel_group);
	gtk_item_factory_create_items(ppitem_factory,sizeof(popupmenu_items)/sizeof(popupmenu_items[0]),popupmenu_items,NULL);
	gtk_accel_group_attach(ppaccel_group,GTK_OBJECT(term));
	gtk_term_set_popup(term,ppitem_factory);

	/*
	 *	create a status bar
	 */
	statusbar = gtk_statusbar_new();
	gtk_box_pack_start(GTK_BOX(vbox),statusbar,FALSE,FALSE,0);
	id = gtk_statusbar_get_context_id(GTK_STATUSBAR(statusbar),"euler status messages");
	gtk_signal_connect_object(GTK_OBJECT(term),"term_editing",GTK_SIGNAL_FUNC(term_editing_cb),GTK_OBJECT(statusbar));
	gtk_signal_connect_object(GTK_OBJECT(term),"term_interpreting",GTK_SIGNAL_FUNC(term_interpreting_cb),GTK_OBJECT(statusbar));
	gtk_statusbar_push(GTK_STATUSBAR(statusbar),id," Initializing Euler ...");
	/*
	 *	show everything and lets go
	 */
	gtk_widget_show_all(term_window);

	setmetadevice(gtk_meta_get_device(meta));
	
	while (gtk_events_pending()) gtk_main_iteration_do(FALSE);

	s=g_strconcat(INSTALL_DIR,"/share/euler/help.txt",NULL);
	loadhelp(s);
	g_free(s);
	
	main_loop(argc,argv);
	
	closemeta();
	
	gtk_item_factory_dump_rc("~/.euler/shortcuts",NULL,FALSE);
	
	return 0;
}

