/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * 	Module: ecemainloop.c
 */


/* NOTE THIS IS A TEMPORARY FILE THAT MIMICS THE GLIB'S MAINLOOP  BEHAVIOUR.
 * GLIB 1.2 DOES NOT ALLOW TWO INSTANCES OF MAINLOOP RUNNING SIMULTANEOUSLY.
 */

#include <ece.h>
#include <sys/poll.h>
#include "eceinternal.h"

static IPCDispatch ipcdisp;/*dispatch callback for ipc channel */
static IPC_Channel *ipc_ch; /* ipc channel */
static GDestroyNotify  ipcdest; /* destroy callback for ipc channel */
static GSourceFunc  timedisp; /* dispatch callback for timeout */
static guint  tintval; /* timeout interval */
static gboolean initflag = FALSE; /* is this subsystem initialized? */
static gboolean runflag; /* is the loop running ? */
static struct pollfd 	ipcfd[2]; /* we have maximum 2 fds */


void
ece_main_cleanup()
{
	LOG_ENTRY();
	ipcdisp = NULL;
	ipc_ch = NULL;
	ipcdest = NULL;
	timedisp = NULL;
	tintval = 100000;/* a large default timeout value */
	initflag = FALSE;
	runflag = FALSE;
	LOG_EXIT_VOID();
}

void
ece_main_init(IPC_Channel *ch,
		IPCDispatch ipdispatch,
		GDestroyNotify ipdestroy,
		guint interval,
		GSourceFunc tdispatch)
{
	LOG_ENTRY();
	if(!initflag) {
		ece_main_cleanup();
		ipcdisp = ipdispatch;
		ipcdest = ipdestroy;
		timedisp = tdispatch;
		tintval = interval;
		ipc_ch = ch;
		initflag = TRUE;
		runflag = TRUE;
	} else {
            	LOG_SERIOUS("IPC_Channel already initialized\n");
		LOG_EXIT_VOID();
		return;
	}
	LOG_EXIT_VOID();
}



static gboolean
dispatch(IPC_Channel *ch, IPCDispatch ipcdisp, GDestroyNotify  ipcdest)
{
	LOG_ENTRY();
	if(ipcdisp && ch && ipcdisp(ch, NULL) == FALSE) {
		if(ipcdest) ipcdest(NULL);
		ch->ops->destroy(ch);
		LOG_EXIT_BOOL(FALSE);
		return FALSE;
	}
	LOG_EXIT_BOOL(TRUE);
	return TRUE;
}



/* should be called when ece_main_run terminates */
void
ece_main_destroy()
{
	LOG_ENTRY();
	if(ipc_ch) ipc_ch->ops->destroy(ipc_ch);
	ece_main_cleanup();
	LOG_EXIT_VOID();
}


/* runs the loop and dispatches callbacks when any ipc event is sensed
 * as well as dispatches timeout callbacks regularly
 * The only way out of this loop is through thread terminate signal.
 */
void
ece_main_run()
{
	LOG_ENTRY();
	if(!initflag) {
            	LOG_SERIOUS("IPC_Channel not initialized\n");
		LOG_EXIT_VOID();
		return;
	}

	while(runflag) {
		if(ipc_ch && ipc_ch->ops->is_message_pending(ipc_ch)) {
			if(!dispatch(ipc_ch,ipcdisp,ipcdest)) {
				ipcdisp = NULL;
				ipcdest = NULL;
				ipc_ch = NULL;
			}
		}

		if(ipc_ch) {
			int count = 0, ret;
			int rfd = ipc_ch->ops->get_recv_select_fd(ipc_ch);
			int wfd = ipc_ch->ops->get_send_select_fd(ipc_ch);
			ipcfd[count].fd = rfd;
			ipcfd[count].events =
			   POLLIN|POLLPRI|POLLERR|POLLHUP|POLLNVAL;
			if(ipc_ch->ops->is_sending_blocked(ipc_ch)) {
				ipcfd[count].events |= POLLOUT;
			}
			ipcfd[count++].revents = 0;
			if(rfd!=wfd) {
				ipcfd[count].fd = wfd;
				ipcfd[count].events =
				POLLIN|POLLPRI|POLLERR|POLLHUP|POLLNVAL;
				if(ipc_ch->ops->is_sending_blocked(ipc_ch)) {
					ipcfd[count].events |= POLLOUT;
				}
				ipcfd[count++].revents = 0;
			}
			if((ret = poll(ipcfd, count, tintval)) >0) {
				int i;
				for(i=0; i<count; i++) {
					if(!dispatch(ipc_ch,ipcdisp,ipcdest)) {
						ipcdisp = NULL;
						ipcdest = NULL;
						ipc_ch = NULL;
					}
				}
			 }  else if (ret < 0) {
				 continue;
			 }
		} else {
			/* sleep timeout interval */
			poll(NULL,0,tintval);
		}
		if(timedisp) timedisp(NULL);
		pthread_testcancel();/* check if we have to terminate */
	}
	LOG_EXIT_VOID();
}
